%%----------------------------------------------------------------------
%% %CopyrightBegin%
%% 
%% Copyright Ericsson AB 2010-2022. All Rights Reserved.
%% 
%% Licensed under the Apache License, Version 2.0 (the "License");
%% you may not use this file except in compliance with the License.
%% You may obtain a copy of the License at
%%
%%     http://www.apache.org/licenses/LICENSE-2.0
%%
%% Unless required by applicable law or agreed to in writing, software
%% distributed under the License is distributed on an "AS IS" BASIS,
%% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%% See the License for the specific language governing permissions and
%% limitations under the License.
%% 
%% %CopyrightEnd%
%%----------------------------------------------------------------------
%% File    : xmerl_sax_std_SUITE.erl
%% Created : 2019-09-12
%%----------------------------------------------------------------------
-module(xmerl_sax_std_SUITE).
-compile(export_all).

%%----------------------------------------------------------------------
%% Include files
%%----------------------------------------------------------------------
-include_lib("common_test/include/ct.hrl").
-include_lib("kernel/include/file.hrl").

%%======================================================================
%% External functions
%%======================================================================

%%----------------------------------------------------------------------
%% Initializations
%%----------------------------------------------------------------------

%-define(dont_rm_test_dirs, true).

-ifndef(dont_rm_test_dirs).

init_per_suite(Config) ->
    file:set_cwd(datadir(Config)),
    ok=erl_tar:extract("ibm.tgz",[compressed]),
    ok=erl_tar:extract("japanese.tgz",[compressed]),
    ok=erl_tar:extract("oasis.tgz",[compressed]),
    ok=erl_tar:extract("sun.tgz",[compressed]),
    ok=erl_tar:extract("xmltest.tgz",[compressed]),
    ok = change_mode(["ibm","japanese","oasis",
                      "sun","xmltest"]),
    [{timetrap,{seconds, 1}}|Config].

end_per_suite(Config) ->
    file:set_cwd(datadir(Config)),
    ok=rm_files(["ibm","japanese","oasis","sun","xmltest"]),
    Config.

-else.

init_per_suite(Config) ->
    file:set_cwd(datadir(Config)),
    [{timetrap,{seconds, 1}}|Config].

end_per_suite(Config) ->
    Config.

-endif.

%% initialization before each testcase
init_per_testcase(_TestCase,Config) ->
    io:format("Config:\n~p\n",[Config]),
    {ok, _} = file:read_file_info(filename:join([privdir(Config)])),
    code:add_patha(privdir(Config)),
    Config.
 
%% clean up after each testcase
end_per_testcase(_Func,_Config) ->
    ok.

%%----------------------------------------------------------------------
%% Tests
%%----------------------------------------------------------------------

%%----------------------------------------------------------------------
%% Test Cases
%% Profile: James Clark  XML 1.0 Tests
%%----------------------------------------------------------------------
 
%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/001.xml
%% ID: not-wf-sa-001
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   Attribute values must start with attribute names, not "?".
'not-wf-sa-001'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/001.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/002.xml
%% ID: not-wf-sa-002
%% Type: not-wf
%% Sections: 2.3 [4]
%% Description:
%%   Names may not start with "."; it's not a Letter.
'not-wf-sa-002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/002.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/003.xml
%% ID: not-wf-sa-003
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   Processing Instruction target name is required.
'not-wf-sa-003'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/003.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/004.xml
%% ID: not-wf-sa-004
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   SGML-ism: processing instructions end in '?>' not '>'.
'not-wf-sa-004'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/004.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/005.xml
%% ID: not-wf-sa-005
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   Processing instructions end in '?>' not '?'.
'not-wf-sa-005'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/005.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/006.xml
%% ID: not-wf-sa-006
%% Type: not-wf
%% Sections: 2.5 [16]
%% Description:
%%   XML comments may not contain "--"
'not-wf-sa-006'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/006.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/007.xml
%% ID: not-wf-sa-007
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   General entity references have no whitespace after the entity name and
%%   before the semicolon.
'not-wf-sa-007'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/007.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/008.xml
%% ID: not-wf-sa-008
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   Entity references must include names, which don't begin with '.' (it's not
%%   a Letter or other name start character).
'not-wf-sa-008'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/008.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/009.xml
%% ID: not-wf-sa-009
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   Character references may have only decimal or numeric strings.
'not-wf-sa-009'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/009.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/010.xml
%% ID: not-wf-sa-010
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Ampersand may only appear as part of a general entity reference.
'not-wf-sa-010'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/010.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/011.xml
%% ID: not-wf-sa-011
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   SGML-ism: attribute values must be explicitly assigned a value, it can't
%%   act as a boolean toggle.
'not-wf-sa-011'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/011.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/012.xml
%% ID: not-wf-sa-012
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   SGML-ism: attribute values must be quoted in all cases.
'not-wf-sa-012'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/012.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/013.xml
%% ID: not-wf-sa-013
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   The quotes on both ends of an attribute value must match.
'not-wf-sa-013'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/013.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/014.xml
%% ID: not-wf-sa-014
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   Attribute values may not contain literal '<' characters.
'not-wf-sa-014'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/014.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/015.xml
%% ID: not-wf-sa-015
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   Attribute values need a value, not just an equals sign.
'not-wf-sa-015'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/015.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/016.xml
%% ID: not-wf-sa-016
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   Attribute values need an associated name.
'not-wf-sa-016'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/016.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/017.xml
%% ID: not-wf-sa-017
%% Type: not-wf
%% Sections: 2.7 [18]
%% Description:
%%   CDATA sections need a terminating ']]>'.
'not-wf-sa-017'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/017.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/018.xml
%% ID: not-wf-sa-018
%% Type: not-wf
%% Sections: 2.7 [19]
%% Description:
%%   CDATA sections begin with a literal '<![CDATA[', no space.
'not-wf-sa-018'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/018.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/019.xml
%% ID: not-wf-sa-019
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   End tags may not be abbreviated as '</>'.
'not-wf-sa-019'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/019.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/020.xml
%% ID: not-wf-sa-020
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   Attribute values may not contain literal '&' characters except as part of
%%   an entity reference.
'not-wf-sa-020'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/020.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/021.xml
%% ID: not-wf-sa-021
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   Attribute values may not contain literal '&' characters except as part of
%%   an entity reference.
'not-wf-sa-021'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/021.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/022.xml
%% ID: not-wf-sa-022
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   Character references end with semicolons, always!
'not-wf-sa-022'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/022.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/023.xml
%% ID: not-wf-sa-023
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   Digits are not valid name start characters.
'not-wf-sa-023'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/023.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/024.xml
%% ID: not-wf-sa-024
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   Digits are not valid name start characters.
'not-wf-sa-024'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/024.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/025.xml
%% ID: not-wf-sa-025
%% Type: not-wf
%% Sections: 2.4 [14]
%% Description:
%%   Text may not contain a literal ']]>' sequence.
'not-wf-sa-025'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/025.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/026.xml
%% ID: not-wf-sa-026
%% Type: not-wf
%% Sections: 2.4 [14]
%% Description:
%%   Text may not contain a literal ']]>' sequence.
'not-wf-sa-026'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/026.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/027.xml
%% ID: not-wf-sa-027
%% Type: not-wf
%% Sections: 2.5 [15]
%% Description:
%%   Comments must be terminated with "-->".
'not-wf-sa-027'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/027.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/028.xml
%% ID: not-wf-sa-028
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   Processing instructions must end with '?>'.
'not-wf-sa-028'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/028.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/029.xml
%% ID: not-wf-sa-029
%% Type: not-wf
%% Sections: 2.4 [14]
%% Description:
%%   Text may not contain a literal ']]>' sequence.
'not-wf-sa-029'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/029.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/030.xml
%% ID: not-wf-sa-030
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   A form feed is not a legal XML character.
'not-wf-sa-030'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/030.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/031.xml
%% ID: not-wf-sa-031
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   A form feed is not a legal XML character.
'not-wf-sa-031'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/031.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/032.xml
%% ID: not-wf-sa-032
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   A form feed is not a legal XML character.
'not-wf-sa-032'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/032.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/033.xml
%% ID: not-wf-sa-033
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   An ESC (octal 033) is not a legal XML character.
'not-wf-sa-033'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/033.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/034.xml
%% ID: not-wf-sa-034
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   A form feed is not a legal XML character.
'not-wf-sa-034'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/034.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/035.xml
%% ID: not-wf-sa-035
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   The '<' character is a markup delimiter and must start an element, CDATA
%%   section, PI, or comment.
'not-wf-sa-035'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/035.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/036.xml
%% ID: not-wf-sa-036
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Text may not appear after the root element.
'not-wf-sa-036'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/036.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/037.xml
%% ID: not-wf-sa-037
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Character references may not appear after the root element.
'not-wf-sa-037'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/037.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/038.xml
%% ID: not-wf-sa-038
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests the "Unique Att Spec" WF constraint by providing multiple values for
%%   an attribute.
'not-wf-sa-038'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/038.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/039.xml
%% ID: not-wf-sa-039
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests the Element Type Match WFC - end tag name must match start tag name.
'not-wf-sa-039'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/039.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/040.xml
%% ID: not-wf-sa-040
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Provides two document elements.
'not-wf-sa-040'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/040.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/041.xml
%% ID: not-wf-sa-041
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Provides two document elements.
'not-wf-sa-041'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/041.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/042.xml
%% ID: not-wf-sa-042
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   Invalid End Tag
'not-wf-sa-042'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/042.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/043.xml
%% ID: not-wf-sa-043
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Provides #PCDATA text after the document element.
'not-wf-sa-043'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/043.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/044.xml
%% ID: not-wf-sa-044
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Provides two document elements.
'not-wf-sa-044'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/044.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/045.xml
%% ID: not-wf-sa-045
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Invalid Empty Element Tag
'not-wf-sa-045'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/045.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/046.xml
%% ID: not-wf-sa-046
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   This start (or empty element) tag was not terminated correctly.
'not-wf-sa-046'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/046.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/047.xml
%% ID: not-wf-sa-047
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Invalid empty element tag invalid whitespace
'not-wf-sa-047'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/047.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/048.xml
%% ID: not-wf-sa-048
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   Provides a CDATA section after the root element.
'not-wf-sa-048'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/048.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/049.xml
%% ID: not-wf-sa-049
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   Missing start tag
'not-wf-sa-049'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/049.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/050.xml
%% ID: not-wf-sa-050
%% Type: not-wf
%% Sections: 2.1 [1]
%% Description:
%%   Empty document, with no root element.
'not-wf-sa-050'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/050.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/051.xml
%% ID: not-wf-sa-051
%% Type: not-wf
%% Sections: 2.7 [18]
%% Description:
%%   CDATA is invalid at top level of document.
'not-wf-sa-051'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/051.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/052.xml
%% ID: not-wf-sa-052
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   Invalid character reference.
'not-wf-sa-052'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/052.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/053.xml
%% ID: not-wf-sa-053
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   End tag does not match start tag.
'not-wf-sa-053'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/053.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/054.xml
%% ID: not-wf-sa-054
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   PUBLIC requires two literals.
'not-wf-sa-054'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/054.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/055.xml
%% ID: not-wf-sa-055
%% Type: not-wf
%% Sections: 2.8 [28]
%% Description:
%%   Invalid Document Type Definition format.
'not-wf-sa-055'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/055.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/056.xml
%% ID: not-wf-sa-056
%% Type: not-wf
%% Sections: 2.8 [28]
%% Description:
%%   Invalid Document Type Definition format - misplaced comment.
'not-wf-sa-056'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/056.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/057.xml
%% ID: not-wf-sa-057
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   This isn't SGML; comments can't exist in declarations.
'not-wf-sa-057'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/057.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/058.xml
%% ID: not-wf-sa-058
%% Type: not-wf
%% Sections: 3.3.1 [54]
%% Description:
%%   Invalid character , in ATTLIST enumeration
'not-wf-sa-058'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/058.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/059.xml
%% ID: not-wf-sa-059
%% Type: not-wf
%% Sections: 3.3.1 [59]
%% Description:
%%   String literal must be in quotes.
'not-wf-sa-059'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/059.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/060.xml
%% ID: not-wf-sa-060
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   Invalid type NAME defined in ATTLIST.
'not-wf-sa-060'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/060.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/061.xml
%% ID: not-wf-sa-061
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   External entity declarations require whitespace between public and system
%%   IDs.
'not-wf-sa-061'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/061.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/062.xml
%% ID: not-wf-sa-062
%% Type: not-wf
%% Sections: 4.2 [71]
%% Description:
%%   Entity declarations need space after the entity name.
'not-wf-sa-062'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/062.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/063.xml
%% ID: not-wf-sa-063
%% Type: not-wf
%% Sections: 2.8 [29]
%% Description:
%%   Conditional sections may only appear in the external DTD subset.
'not-wf-sa-063'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/063.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/064.xml
%% ID: not-wf-sa-064
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   Space is required between attribute type and default values in
%%   <!ATTLIST...> declarations.
'not-wf-sa-064'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/064.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/065.xml
%% ID: not-wf-sa-065
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   Space is required between attribute name and type in <!ATTLIST...>
%%   declarations.
'not-wf-sa-065'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/065.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/066.xml
%% ID: not-wf-sa-066
%% Type: not-wf
%% Sections: 3.3 [52]
%% Description:
%%   Required whitespace is missing.
'not-wf-sa-066'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/066.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/067.xml
%% ID: not-wf-sa-067
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   Space is required between attribute type and default values in
%%   <!ATTLIST...> declarations.
'not-wf-sa-067'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/067.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/068.xml
%% ID: not-wf-sa-068
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   Space is required between NOTATION keyword and list of enumerated choices
%%   in <!ATTLIST...> declarations.
'not-wf-sa-068'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/068.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/069.xml
%% ID: not-wf-sa-069
%% Type: not-wf
%% Sections: 4.2.2 [76]
%% Description:
%%   Space is required before an NDATA entity annotation.
'not-wf-sa-069'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/069.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/070.xml
%% ID: not-wf-sa-070
%% Type: not-wf
%% Sections: 2.5 [16]
%% Description:
%%   XML comments may not contain "--"
'not-wf-sa-070'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/070.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/071.xml
%% ID: not-wf-sa-071
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   ENTITY can't reference itself directly or indirectly.
'not-wf-sa-071'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/071.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/072.xml
%% ID: not-wf-sa-072
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Undefined ENTITY foo.
'not-wf-sa-072'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/072.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/073.xml
%% ID: not-wf-sa-073
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Undefined ENTITY f.
'not-wf-sa-073'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/073.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/074.xml
%% ID: not-wf-sa-074
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal general parsed entities are only well formed if they match the
%%   "content" production.
'not-wf-sa-074'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/074.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/075.xml
%% ID: not-wf-sa-075
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   ENTITY can't reference itself directly or indirectly.
'not-wf-sa-075'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/075.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/076.xml
%% ID: not-wf-sa-076
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Undefined ENTITY foo.
'not-wf-sa-076'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/076.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/077.xml
%% ID: not-wf-sa-077
%% Type: not-wf
%% Sections: 41. [68]
%% Description:
%%   Undefined ENTITY bar.
'not-wf-sa-077'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/077.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/078.xml
%% ID: not-wf-sa-078
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Undefined ENTITY foo.
'not-wf-sa-078'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/078.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/079.xml
%% ID: not-wf-sa-079
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   ENTITY can't reference itself directly or indirectly.
'not-wf-sa-079'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/079.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/080.xml
%% ID: not-wf-sa-080
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   ENTITY can't reference itself directly or indirectly.
'not-wf-sa-080'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/080.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/081.xml
%% ID: not-wf-sa-081
%% Type: not-wf
%% Sections: 3.1
%% Entities: general
%% Description:
%%   This tests the WFC, since the entity is referred to within an attribute.
'not-wf-sa-081'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/081.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/082.xml
%% ID: not-wf-sa-082
%% Type: not-wf
%% Sections: 3.1
%% Entities: general
%% Description:
%%   This tests the WFC, since the entity is referred to within an attribute.
'not-wf-sa-082'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/082.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/083.xml
%% ID: not-wf-sa-083
%% Type: not-wf
%% Sections: 4.2.2 [76]
%% Description:
%%   Undefined NOTATION n.
'not-wf-sa-083'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/083.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/084.xml
%% ID: not-wf-sa-084
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests the WFC by referring to an unparsed entity. (This precedes the error
%%   of not declaring that entity's notation, which may be detected any time
%%   before the DTD parsing is completed.)
'not-wf-sa-084'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/084.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/085.xml
%% ID: not-wf-sa-085
%% Type: not-wf
%% Sections: 2.3 [13]
%% Description:
%%   Public IDs may not contain "[".
'not-wf-sa-085'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/085.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/086.xml
%% ID: not-wf-sa-086
%% Type: not-wf
%% Sections: 2.3 [13]
%% Description:
%%   Public IDs may not contain "[".
'not-wf-sa-086'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/086.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/087.xml
%% ID: not-wf-sa-087
%% Type: not-wf
%% Sections: 2.3 [13]
%% Description:
%%   Public IDs may not contain "[".
'not-wf-sa-087'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/087.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/088.xml
%% ID: not-wf-sa-088
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   Attribute values are terminated by literal quote characters, and any entity
%%   expansion is done afterwards.
'not-wf-sa-088'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/088.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/089.xml
%% ID: not-wf-sa-089
%% Type: not-wf
%% Sections: 4.2 [74]
%% Description:
%%   Parameter entities "are" always parsed; NDATA annotations are not
%%   permitted.
'not-wf-sa-089'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/089.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/090.xml
%% ID: not-wf-sa-090
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   Attributes may not contain a literal "<" character; this one has one
%%   because of reference expansion.
'not-wf-sa-090'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/090.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/091.xml
%% ID: not-wf-sa-091
%% Type: not-wf
%% Sections: 4.2 [74]
%% Description:
%%   Parameter entities "are" always parsed; NDATA annotations are not
%%   permitted.
'not-wf-sa-091'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/091.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/092.xml
%% ID: not-wf-sa-092
%% Type: not-wf
%% Sections: 4.5
%% Description:
%%   The replacement text of this entity has an illegal reference, because the
%%   character reference is expanded immediately.
'not-wf-sa-092'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/092.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/093.xml
%% ID: not-wf-sa-093
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   Hexadecimal character references may not use the uppercase 'X'.
'not-wf-sa-093'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/093.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/094.xml
%% ID: not-wf-sa-094
%% Type: not-wf
%% Sections: 2.8 [24]
%% Description:
%%   Prolog VERSION must be lowercase.
'not-wf-sa-094'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/094.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/095.xml
%% ID: not-wf-sa-095
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   VersionInfo must come before EncodingDecl.
'not-wf-sa-095'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/095.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/096.xml
%% ID: not-wf-sa-096
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   Space is required before the standalone declaration.
'not-wf-sa-096'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/096.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/097.xml
%% ID: not-wf-sa-097
%% Type: not-wf
%% Sections: 2.8 [24]
%% Description:
%%   Both quotes surrounding VersionNum must be the same.
'not-wf-sa-097'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/097.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/098.xml
%% ID: not-wf-sa-098
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   Only one "version=..." string may appear in an XML declaration.
'not-wf-sa-098'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/098.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/099.xml
%% ID: not-wf-sa-099
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   Only three pseudo-attributes are in the XML declaration, and "valid=..." is
%%   not one of them.
'not-wf-sa-099'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/099.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/100.xml
%% ID: not-wf-sa-100
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   Only "yes" and "no" are permitted as values of "standalone".
'not-wf-sa-100'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/100.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/101.xml
%% ID: not-wf-sa-101
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Space is not permitted in an encoding name.
'not-wf-sa-101'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/101.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/102.xml
%% ID: not-wf-sa-102
%% Type: not-wf
%% Sections: 2.8 [26]
%% Description:
%%   Provides an illegal XML version number; spaces are illegal.
'not-wf-sa-102'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/102.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/103.xml
%% ID: not-wf-sa-103
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   End-tag required for element foo.
'not-wf-sa-103'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/103.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/104.xml
%% ID: not-wf-sa-104
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal general parsed entities are only well formed if they match the
%%   "content" production.
'not-wf-sa-104'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/104.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/105.xml
%% ID: not-wf-sa-105
%% Type: not-wf
%% Sections: 2.7 
%% Description:
%%   Invalid placement of CDATA section.
'not-wf-sa-105'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/105.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/106.xml
%% ID: not-wf-sa-106
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Invalid placement of entity declaration.
'not-wf-sa-106'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/106.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/107.xml
%% ID: not-wf-sa-107
%% Type: not-wf
%% Sections: 2.8 [28]
%% Description:
%%   Invalid document type declaration. CDATA alone is invalid.
'not-wf-sa-107'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/107.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/108.xml
%% ID: not-wf-sa-108
%% Type: not-wf
%% Sections: 2.7 [19]
%% Description:
%%   No space in '<![CDATA['.
'not-wf-sa-108'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/108.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/109.xml
%% ID: not-wf-sa-109
%% Type: not-wf
%% Sections: 4.2 [70]
%% Description:
%%   Tags invalid within EntityDecl.
'not-wf-sa-109'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/109.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/110.xml
%% ID: not-wf-sa-110
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Entity reference must be in content of element.
'not-wf-sa-110'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/110.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/111.xml
%% ID: not-wf-sa-111
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   Entiry reference must be in content of element not Start-tag.
'not-wf-sa-111'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/111.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/112.xml
%% ID: not-wf-sa-112
%% Type: not-wf
%% Sections: 2.7 [19]
%% Description:
%%   CDATA sections start '<![CDATA[', not '<!cdata['.
'not-wf-sa-112'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/112.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/113.xml
%% ID: not-wf-sa-113
%% Type: not-wf
%% Sections: 2.3 [9]
%% Description:
%%   Parameter entity values must use valid reference syntax; this reference is
%%   malformed.
'not-wf-sa-113'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/113.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/114.xml
%% ID: not-wf-sa-114
%% Type: not-wf
%% Sections: 2.3 [9]
%% Description:
%%   General entity values must use valid reference syntax; this reference is
%%   malformed.
'not-wf-sa-114'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/114.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/115.xml
%% ID: not-wf-sa-115
%% Type: not-wf
%% Sections: 4.5
%% Description:
%%   The replacement text of this entity is an illegal character reference,
%%   which must be rejected when it is parsed in the context of an attribute
%%   value.
'not-wf-sa-115'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/115.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/116.xml
%% ID: not-wf-sa-116
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal general parsed entities are only well formed if they match the
%%   "content" production. This is a partial character reference, not a full
%%   one.
'not-wf-sa-116'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/116.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/117.xml
%% ID: not-wf-sa-117
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal general parsed entities are only well formed if they match the
%%   "content" production. This is a partial character reference, not a full
%%   one.
'not-wf-sa-117'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/117.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/118.xml
%% ID: not-wf-sa-118
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   Entity reference expansion is not recursive.
'not-wf-sa-118'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/118.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/119.xml
%% ID: not-wf-sa-119
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal general parsed entities are only well formed if they match the
%%   "content" production. This is a partial character reference, not a full
%%   one.
'not-wf-sa-119'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/119.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/120.xml
%% ID: not-wf-sa-120
%% Type: not-wf
%% Sections: 4.5
%% Description:
%%   Character references are expanded in the replacement text of an internal
%%   entity, which is then parsed as usual. Accordingly, & must be doubly quoted
%%   - encoded either as or as .
'not-wf-sa-120'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/120.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/121.xml
%% ID: not-wf-sa-121
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   A name of an ENTITY was started with an invalid character.
'not-wf-sa-121'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/121.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/122.xml
%% ID: not-wf-sa-122
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   Invalid syntax mixed connectors are used.
'not-wf-sa-122'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/122.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/123.xml
%% ID: not-wf-sa-123
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   Invalid syntax mismatched parenthesis.
'not-wf-sa-123'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/123.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/124.xml
%% ID: not-wf-sa-124
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   Invalid format of Mixed-content declaration.
'not-wf-sa-124'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/124.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/125.xml
%% ID: not-wf-sa-125
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   Invalid syntax extra set of parenthesis not necessary.
'not-wf-sa-125'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/125.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/126.xml
%% ID: not-wf-sa-126
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   Invalid syntax Mixed-content must be defined as zero or more.
'not-wf-sa-126'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/126.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/127.xml
%% ID: not-wf-sa-127
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   Invalid syntax Mixed-content must be defined as zero or more.
'not-wf-sa-127'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/127.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/128.xml
%% ID: not-wf-sa-128
%% Type: not-wf
%% Sections: 2.7 [18]
%% Description:
%%   Invalid CDATA syntax.
'not-wf-sa-128'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/128.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/129.xml
%% ID: not-wf-sa-129
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   Invalid syntax for Element Type Declaration.
'not-wf-sa-129'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/129.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/130.xml
%% ID: not-wf-sa-130
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   Invalid syntax for Element Type Declaration.
'not-wf-sa-130'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/130.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/131.xml
%% ID: not-wf-sa-131
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   Invalid syntax for Element Type Declaration.
'not-wf-sa-131'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/131.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/132.xml
%% ID: not-wf-sa-132
%% Type: not-wf
%% Sections: 3.2.1 [50]
%% Description:
%%   Invalid syntax mixed connectors used.
'not-wf-sa-132'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/132.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/133.xml
%% ID: not-wf-sa-133
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Illegal whitespace before optional character causes syntax error.
'not-wf-sa-133'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/133.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/134.xml
%% ID: not-wf-sa-134
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Illegal whitespace before optional character causes syntax error.
'not-wf-sa-134'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/134.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/135.xml
%% ID: not-wf-sa-135
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   Invalid character used as connector.
'not-wf-sa-135'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/135.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/136.xml
%% ID: not-wf-sa-136
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   Tag omission is invalid in XML.
'not-wf-sa-136'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/136.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/137.xml
%% ID: not-wf-sa-137
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   Space is required before a content model.
'not-wf-sa-137'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/137.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/138.xml
%% ID: not-wf-sa-138
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   Invalid syntax for content particle.
'not-wf-sa-138'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/138.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/139.xml
%% ID: not-wf-sa-139
%% Type: not-wf
%% Sections: 3.2.1 [46]
%% Description:
%%   The element-content model should not be empty.
'not-wf-sa-139'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/139.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/140.xml
%% ID: not-wf-sa-140
%% Type: not-wf
%% Sections: 2.3 [4]
%% Edition: 1 2 3 4
%% Description:
%%   Character '&#x309a;' is a CombiningChar, not a Letter, and so may not begin
%%   a name.
'not-wf-sa-140'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/140.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/141.xml
%% ID: not-wf-sa-141
%% Type: not-wf
%% Sections: 2.3 [5]
%% Edition: 1 2 3 4
%% Description:
%%   Character #x0E5C is not legal in XML names.
'not-wf-sa-141'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/141.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/142.xml
%% ID: not-wf-sa-142
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character #x0000 is not legal anywhere in an XML document.
'not-wf-sa-142'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/142.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/143.xml
%% ID: not-wf-sa-143
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character #x001F is not legal anywhere in an XML document.
'not-wf-sa-143'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/143.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/144.xml
%% ID: not-wf-sa-144
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character #xFFFF is not legal anywhere in an XML document.
'not-wf-sa-144'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/144.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/145.xml
%% ID: not-wf-sa-145
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character #xD800 is not legal anywhere in an XML document. (If it appeared
%%   in a UTF-16 surrogate pair, it'd represent half of a UCS-4 character and so
%%   wouldn't really be in the document.)
'not-wf-sa-145'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/145.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/146.xml
%% ID: not-wf-sa-146
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character references must also refer to legal XML characters; #x00110000 is
%%   one more than the largest legal character.
'not-wf-sa-146'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/146.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/147.xml
%% ID: not-wf-sa-147
%% Type: not-wf
%% Sections: 2.8 [22]
%% Description:
%%   XML Declaration may not be preceded by whitespace.
'not-wf-sa-147'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/147.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/148.xml
%% ID: not-wf-sa-148
%% Type: not-wf
%% Sections: 2.8 [22]
%% Description:
%%   XML Declaration may not be preceded by comments or whitespace.
'not-wf-sa-148'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/148.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/149.xml
%% ID: not-wf-sa-149
%% Type: not-wf
%% Sections: 2.8 [28]
%% Description:
%%   XML Declaration may not be within a DTD.
'not-wf-sa-149'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/149.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/150.xml
%% ID: not-wf-sa-150
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   XML declarations may not be within element content.
'not-wf-sa-150'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/150.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/151.xml
%% ID: not-wf-sa-151
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   XML declarations may not follow document content.
'not-wf-sa-151'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/151.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/152.xml
%% ID: not-wf-sa-152
%% Type: not-wf
%% Sections: 2.8 [22]
%% Description:
%%   XML declarations must include the "version=..." string.
'not-wf-sa-152'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/152.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/153.xml
%% ID: not-wf-sa-153
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Text declarations may not begin internal parsed entities; they may only
%%   appear at the beginning of external parsed (parameter or general) entities.
'not-wf-sa-153'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/153.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/154.xml
%% ID: not-wf-sa-154
%% Type: not-wf
%% Sections: 2.8 2.6 [23, 17]
%% Description:
%%   '<?XML ...?>' is neither an XML declaration nor a legal processing
%%   instruction target name.
'not-wf-sa-154'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/154.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/155.xml
%% ID: not-wf-sa-155
%% Type: not-wf
%% Sections: 2.8 2.6 [23, 17]
%% Description:
%%   '<?xmL ...?>' is neither an XML declaration nor a legal processing
%%   instruction target name.
'not-wf-sa-155'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/155.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/156.xml
%% ID: not-wf-sa-156
%% Type: not-wf
%% Sections: 2.8 2.6 [23, 17]
%% Description:
%%   '<?xMl ...?>' is neither an XML declaration nor a legal processing
%%   instruction target name.
'not-wf-sa-156'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/156.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/157.xml
%% ID: not-wf-sa-157
%% Type: not-wf
%% Sections: 2.6 [17]
%% Description:
%%   '<?xmL ...?>' is not a legal processing instruction target name.
'not-wf-sa-157'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/157.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/158.xml
%% ID: not-wf-sa-158
%% Type: not-wf
%% Sections: 3.3 [52]
%% Description:
%%   SGML-ism: "#NOTATION gif" can't have attributes.
'not-wf-sa-158'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/158.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/159.xml
%% ID: not-wf-sa-159
%% Type: not-wf
%% Sections: 2.3 [9]
%% Description:
%%   Uses '&' unquoted in an entity declaration, which is illegal syntax for an
%%   entity reference.
'not-wf-sa-159'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/159.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/160.xml
%% ID: not-wf-sa-160
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Violates the WFC by using a PE reference within a declaration.
'not-wf-sa-160'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/160.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/161.xml
%% ID: not-wf-sa-161
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Violates the WFC by using a PE reference within a declaration.
'not-wf-sa-161'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/161.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/162.xml
%% ID: not-wf-sa-162
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Violates the WFC by using a PE reference within a declaration.
'not-wf-sa-162'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/162.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/163.xml
%% ID: not-wf-sa-163
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   Invalid placement of Parameter entity reference.
'not-wf-sa-163'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/163.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/164.xml
%% ID: not-wf-sa-164
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   Invalid placement of Parameter entity reference.
'not-wf-sa-164'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/164.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/165.xml
%% ID: not-wf-sa-165
%% Type: not-wf
%% Sections: 4.2 [72]
%% Description:
%%   Parameter entity declarations must have a space before the '%'.
'not-wf-sa-165'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/165.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/166.xml
%% ID: not-wf-sa-166
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-166'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/166.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/167.xml
%% ID: not-wf-sa-167
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFE is not legal anywhere in an XML document.
'not-wf-sa-167'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/167.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/168.xml
%% ID: not-wf-sa-168
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   An unpaired surrogate (D800) is not legal anywhere in an XML document.
'not-wf-sa-168'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/168.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/169.xml
%% ID: not-wf-sa-169
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   An unpaired surrogate (DC00) is not legal anywhere in an XML document.
'not-wf-sa-169'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/169.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/170.xml
%% ID: not-wf-sa-170
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Four byte UTF-8 encodings can encode UCS-4 characters which are beyond the
%%   range of legal XML characters (and can't be expressed in Unicode surrogate
%%   pairs). This document holds such a character.
'not-wf-sa-170'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/170.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/171.xml
%% ID: not-wf-sa-171
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-171'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/171.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/172.xml
%% ID: not-wf-sa-172
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-172'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/172.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/173.xml
%% ID: not-wf-sa-173
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-173'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/173.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/174.xml
%% ID: not-wf-sa-174
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-174'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/174.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/175.xml
%% ID: not-wf-sa-175
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-175'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/175.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/176.xml
%% ID: not-wf-sa-176
%% Type: not-wf
%% Sections: 3 [39]
%% Description:
%%   Start tags must have matching end tags.
'not-wf-sa-176'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/176.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/177.xml
%% ID: not-wf-sa-177
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Character FFFF is not legal anywhere in an XML document.
'not-wf-sa-177'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/177.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/178.xml
%% ID: not-wf-sa-178
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   Invalid syntax matching double quote is missing.
'not-wf-sa-178'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/178.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/179.xml
%% ID: not-wf-sa-179
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   Invalid syntax matching double quote is missing.
'not-wf-sa-179'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/179.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/180.xml
%% ID: not-wf-sa-180
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   The WFC requires entities to be declared before they are used in an
%%   attribute list declaration.
'not-wf-sa-180'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/180.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/181.xml
%% ID: not-wf-sa-181
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal parsed entities must match the production to be well formed.
'not-wf-sa-181'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/181.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/182.xml
%% ID: not-wf-sa-182
%% Type: not-wf
%% Sections: 4.3.2
%% Description:
%%   Internal parsed entities must match the production to be well formed.
'not-wf-sa-182'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/182.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/183.xml
%% ID: not-wf-sa-183
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   Mixed content declarations may not include content particles.
'not-wf-sa-183'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/183.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/184.xml
%% ID: not-wf-sa-184
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   In mixed content models, element names must not be parenthesized.
'not-wf-sa-184'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/184.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/185.xml
%% ID: not-wf-sa-185
%% Type: not-wf
%% Sections: 4.1
%% Entities: parameter
%% Description:
%%   Tests the WFC. a nonvalidating parser is permitted not to report this WFC
%%   violation, since it would need to read an external parameter entity to
%%   distinguish it from a violation of the VC.
'not-wf-sa-185'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/185.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sa/186.xml
%% ID: not-wf-sa-186
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Whitespace is required between attribute/value pairs.
'not-wf-sa-186'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/sa/186.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/001.xml
%% ID: not-wf-not-sa-001
%% Type: not-wf
%% Sections: 3.4 [62]
%% Entities: both
%% Description:
%%   Conditional sections must be properly terminated ("]>" used instead of
%%   "]]>").
'not-wf-not-sa-001'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/001.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/002.xml
%% ID: not-wf-not-sa-002
%% Type: not-wf
%% Sections: 2.6 [17]
%% Entities: both
%% Description:
%%   Processing instruction target names may not be "XML" in any combination of
%%   cases.
'not-wf-not-sa-002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/002.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/003.xml
%% ID: not-wf-not-sa-003
%% Type: not-wf
%% Sections: 3.4 [62]
%% Entities: both
%% Description:
%%   Conditional sections must be properly terminated ("]]>" omitted).
'not-wf-not-sa-003'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/003.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/004.xml
%% ID: not-wf-not-sa-004
%% Type: not-wf
%% Sections: 3.4 [62]
%% Entities: both
%% Description:
%%   Conditional sections must be properly terminated ("]]>" omitted).
'not-wf-not-sa-004'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/004.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/005.xml
%% ID: not-wf-not-sa-005
%% Type: error
%% Sections: 4.1
%% Entities: both
%% Description:
%%   Tests the VC by referring to an undefined parameter entity within an
%%   external entity.
'not-wf-not-sa-005'(_Config) -> {skip, "unknown parameter reference in external (VC test not WFC)"}.
%%    file:set_cwd(datadir(Config)),
%%    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/005.xml"]),
%%    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
%%    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/006.xml
%% ID: not-wf-not-sa-006
%% Type: not-wf
%% Sections: 3.4 [62]
%% Entities: both
%% Description:
%%   Conditional sections need a '[' after the INCLUDE or IGNORE.
'not-wf-not-sa-006'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/006.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/007.xml
%% ID: not-wf-not-sa-007
%% Type: not-wf
%% Sections: 4.3.2 [79]
%% Entities: both
%% Description:
%%   A <!DOCTYPE ...> declaration may not begin any external entity; it's only
%%   found once, in the document entity.
'not-wf-not-sa-007'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/007.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/008.xml
%% ID: not-wf-not-sa-008
%% Type: not-wf
%% Sections: 4.1 [69]
%% Entities: both
%% Description:
%%   In DTDs, the '%' character must be part of a parameter entity reference.
'not-wf-not-sa-008'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/008.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa/009.xml
%% ID: not-wf-not-sa-009
%% Type: not-wf
%% Sections: 2.8
%% Entities: both
%% Description:
%%   This test violates WFC:PE Between Declarations in Production 28a. The last
%%   character of a markup declaration is not contained in the same
%%   parameter-entity text replacement.
'not-wf-not-sa-009'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/not-sa/009.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/ext-sa/001.xml
%% ID: not-wf-ext-sa-001
%% Type: not-wf
%% Sections: 4.1
%% Entities: both
%% Description:
%%   Tests the WFC by having an external general entity be self-recursive.
'not-wf-ext-sa-001'(_Config) -> {skip, "recursive external reference"}.
%%    file:set_cwd(datadir(Config)),
%%    Path = filename:join([datadir(Config),"xmltest","not-wf/ext-sa/001.xml"]),
%%    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
%%    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/ext-sa/002.xml
%% ID: not-wf-ext-sa-002
%% Type: not-wf
%% Sections: 4.3.1 4.3.2 [77, 78]
%% Entities: both
%% Description:
%%   External entities have "text declarations", which do not permit the
%%   "standalone=..." attribute that's allowed in XML declarations.
'not-wf-ext-sa-002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/ext-sa/002.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/ext-sa/003.xml
%% ID: not-wf-ext-sa-003
%% Type: not-wf
%% Sections: 2.6 [17]
%% Entities: both
%% Description:
%%   Only one text declaration is permitted; a second one looks like an illegal
%%   processing instruction (target names of "xml" in any case are not allowed).
'not-wf-ext-sa-003'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","not-wf/ext-sa/003.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/002.xml
%% ID: invalid--002
%% Type: invalid
%% Sections: 3.2.1
%% Entities: both
%% Description:
%%   Tests the "Proper Group/PE Nesting" validity constraint by fragmenting a
%%   content model between two parameter entities.
'invalid--002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","invalid/002.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/005.xml
%% ID: invalid--005
%% Type: invalid
%% Sections: 2.8
%% Entities: both
%% Description:
%%   Tests the "Proper Declaration/PE Nesting" validity constraint by
%%   fragmenting an element declaration between two parameter entities.
'invalid--005'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","invalid/005.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/006.xml
%% ID: invalid--006
%% Type: invalid
%% Sections: 2.8
%% Entities: both
%% Description:
%%   Tests the "Proper Declaration/PE Nesting" validity constraint by
%%   fragmenting an element declaration between two parameter entities.
'invalid--006'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","invalid/006.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa/022.xml
%% ID: invalid-not-sa-022
%% Type: invalid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: invalid/not-sa/out/022.xml
%% Description:
%%   Test the "Proper Conditional Section/ PE Nesting" validity constraint.
'invalid-not-sa-022'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","invalid/not-sa/022.xml"]),
    Out = filename:join([datadir(Config),"xmltest","invalid/not-sa/out/022.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/001.xml
%% ID: valid-sa-001
%% Type: valid
%% Sections: 3.2.2 [51]
%% Output: valid/sa/out/001.xml
%% Description:
%%   Test demonstrates an Element Type Declaration with Mixed Content.
'valid-sa-001'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/001.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/001.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/002.xml
%% ID: valid-sa-002
%% Type: valid
%% Sections: 3.1 [40]
%% Output: valid/sa/out/002.xml
%% Description:
%%   Test demonstrates that whitespace is permitted after the tag name in a
%%   Start-tag.
'valid-sa-002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/002.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/002.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/003.xml
%% ID: valid-sa-003
%% Type: valid
%% Sections: 3.1 [42]
%% Output: valid/sa/out/003.xml
%% Description:
%%   Test demonstrates that whitespace is permitted after the tag name in an
%%   End-tag.
'valid-sa-003'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/003.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/003.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/004.xml
%% ID: valid-sa-004
%% Type: valid
%% Sections: 3.1 [41]
%% Output: valid/sa/out/004.xml
%% Description:
%%   Test demonstrates a valid attribute specification within a Start-tag.
'valid-sa-004'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/004.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/004.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/005.xml
%% ID: valid-sa-005
%% Type: valid
%% Sections: 3.1 [40]
%% Output: valid/sa/out/005.xml
%% Description:
%%   Test demonstrates a valid attribute specification within a Start-tag that
%%   contains whitespace on both sides of the equal sign.
'valid-sa-005'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/005.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/005.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/006.xml
%% ID: valid-sa-006
%% Type: valid
%% Sections: 3.1 [41]
%% Output: valid/sa/out/006.xml
%% Description:
%%   Test demonstrates that the AttValue within a Start-tag can use a single
%%   quote as a delimter.
'valid-sa-006'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/006.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/006.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/007.xml
%% ID: valid-sa-007
%% Type: valid
%% Sections: 3.1 4.6 [43]
%% Output: valid/sa/out/007.xml
%% Description:
%%   Test demonstrates numeric character references can be used for element
%%   content.
'valid-sa-007'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/007.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/007.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/008.xml
%% ID: valid-sa-008
%% Type: valid
%% Sections: 2.4 3.1 [43]
%% Output: valid/sa/out/008.xml
%% Description:
%%   Test demonstrates character references can be used for element content.
'valid-sa-008'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/008.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/008.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/009.xml
%% ID: valid-sa-009
%% Type: valid
%% Sections: 2.3 3.1 [43]
%% Output: valid/sa/out/009.xml
%% Description:
%%   Test demonstrates that PubidChar can be used for element content.
'valid-sa-009'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/009.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/009.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/010.xml
%% ID: valid-sa-010
%% Type: valid
%% Sections: 3.1 [40]
%% Output: valid/sa/out/010.xml
%% Description:
%%   Test demonstrates that whitespace is valid after the Attribute in a
%%   Start-tag.
'valid-sa-010'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/010.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/010.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/011.xml
%% ID: valid-sa-011
%% Type: valid
%% Sections: 3.1 [40]
%% Output: valid/sa/out/011.xml
%% Description:
%%   Test demonstrates multiple Attributes within the Start-tag.
'valid-sa-011'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/011.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/011.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/012.xml
%% ID: valid-sa-012
%% Type: valid
%% Sections: 2.3 [4]
%% Output: valid/sa/out/012.xml
%% Description:
%%   Uses a legal XML 1.0 name consisting of a single colon character
%%   (disallowed by the latest XML Namespaces draft).
'valid-sa-012'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/012.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/012.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/013.xml
%% ID: valid-sa-013
%% Type: valid
%% Sections: 2.3 3.1 [13] [40]
%% Output: valid/sa/out/013.xml
%% Description:
%%   Test demonstrates that the Attribute in a Start-tag can consist of numerals
%%   along with special characters.
'valid-sa-013'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/013.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/013.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/014.xml
%% ID: valid-sa-014
%% Type: valid
%% Sections: 2.3 3.1 [13] [40]
%% Output: valid/sa/out/014.xml
%% Description:
%%   Test demonstrates that all lower case letters are valid for the Attribute
%%   in a Start-tag.
'valid-sa-014'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/014.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/014.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/015.xml
%% ID: valid-sa-015
%% Type: valid
%% Sections: 2.3 3.1 [13] [40]
%% Output: valid/sa/out/015.xml
%% Description:
%%   Test demonstrates that all upper case letters are valid for the Attribute
%%   in a Start-tag.
'valid-sa-015'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/015.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/015.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/016.xml
%% ID: valid-sa-016
%% Type: valid
%% Sections: 2.6 3.1 [16] [43]
%% Output: valid/sa/out/016.xml
%% Description:
%%   Test demonstrates that Processing Instructions are valid element content.
'valid-sa-016'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/016.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/016.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/017.xml
%% ID: valid-sa-017
%% Type: valid
%% Sections: 2.6 3.1 [16] [43]
%% Output: valid/sa/out/017.xml
%% Description:
%%   Test demonstrates that Processing Instructions are valid element content
%%   and there can be more than one.
'valid-sa-017'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/017.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/017.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/018.xml
%% ID: valid-sa-018
%% Type: valid
%% Sections: 2.7 3.1 [18] [43]
%% Output: valid/sa/out/018.xml
%% Description:
%%   Test demonstrates that CDATA sections are valid element content.
'valid-sa-018'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/018.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/018.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/019.xml
%% ID: valid-sa-019
%% Type: valid
%% Sections: 2.7 3.1 [18] [43]
%% Output: valid/sa/out/019.xml
%% Description:
%%   Test demonstrates that CDATA sections are valid element content and that
%%   ampersands may occur in their literal form.
'valid-sa-019'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/019.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/019.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/020.xml
%% ID: valid-sa-020
%% Type: valid
%% Sections: 2.7 3.1 [18] [43]
%% Output: valid/sa/out/020.xml
%% Description:
%%   Test demonstractes that CDATA sections are valid element content and that
%%   everything between the CDStart and CDEnd is recognized as character data not
%%   markup.
'valid-sa-020'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/020.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/020.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/021.xml
%% ID: valid-sa-021
%% Type: valid
%% Sections: 2.5 3.1 [15] [43]
%% Output: valid/sa/out/021.xml
%% Description:
%%   Test demonstrates that comments are valid element content.
'valid-sa-021'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/021.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/021.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/022.xml
%% ID: valid-sa-022
%% Type: valid
%% Sections: 2.5 3.1 [15] [43]
%% Output: valid/sa/out/022.xml
%% Description:
%%   Test demonstrates that comments are valid element content and that all
%%   characters before the double-hypen right angle combination are considered
%%   part of thecomment.
'valid-sa-022'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/022.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/022.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/023.xml
%% ID: valid-sa-023
%% Type: valid
%% Sections: 3.1 [43]
%% Output: valid/sa/out/023.xml
%% Description:
%%   Test demonstrates that Entity References are valid element content.
'valid-sa-023'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/023.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/023.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/024.xml
%% ID: valid-sa-024
%% Type: valid
%% Sections: 3.1 4.1 [43] [66]
%% Output: valid/sa/out/024.xml
%% Description:
%%   Test demonstrates that Entity References are valid element content and also
%%   demonstrates a valid Entity Declaration.
'valid-sa-024'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/024.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/024.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/025.xml
%% ID: valid-sa-025
%% Type: valid
%% Sections: 3.2 [46]
%% Output: valid/sa/out/025.xml
%% Description:
%%   Test demonstrates an Element Type Declaration and that the contentspec can
%%   be of mixed content.
'valid-sa-025'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/025.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/025.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/026.xml
%% ID: valid-sa-026
%% Type: valid
%% Sections: 3.2 [46]
%% Output: valid/sa/out/026.xml
%% Description:
%%   Test demonstrates an Element Type Declaration and that EMPTY is a valid
%%   contentspec.
'valid-sa-026'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/026.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/026.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/027.xml
%% ID: valid-sa-027
%% Type: valid
%% Sections: 3.2 [46]
%% Output: valid/sa/out/027.xml
%% Description:
%%   Test demonstrates an Element Type Declaration and that ANY is a valid
%%   contenspec.
'valid-sa-027'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/027.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/027.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/028.xml
%% ID: valid-sa-028
%% Type: valid
%% Sections: 2.8 [24]
%% Output: valid/sa/out/028.xml
%% Description:
%%   Test demonstrates a valid prolog that uses double quotes as delimiters
%%   around the VersionNum.
'valid-sa-028'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/028.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/028.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/029.xml
%% ID: valid-sa-029
%% Type: valid
%% Sections: 2.8 [24]
%% Output: valid/sa/out/029.xml
%% Description:
%%   Test demonstrates a valid prolog that uses single quotes as delimiter
%%   around the VersionNum.
'valid-sa-029'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/029.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/029.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/030.xml
%% ID: valid-sa-030
%% Type: valid
%% Sections: 2.8 [25]
%% Output: valid/sa/out/030.xml
%% Description:
%%   Test demonstrates a valid prolog that contains whitespace on both sides of
%%   the equal sign in the VersionInfo.
'valid-sa-030'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/030.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/030.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/031.xml
%% ID: valid-sa-031
%% Type: valid
%% Sections: 4.3.3 [80]
%% Output: valid/sa/out/031.xml
%% Description:
%%   Test demonstrates a valid EncodingDecl within the prolog.
'valid-sa-031'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/031.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/031.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/032.xml
%% ID: valid-sa-032
%% Type: valid
%% Sections: 2.9 [32]
%% Output: valid/sa/out/032.xml
%% Description:
%%   Test demonstrates a valid SDDecl within the prolog.
'valid-sa-032'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/032.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/032.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/033.xml
%% ID: valid-sa-033
%% Type: valid
%% Sections: 2.8 [23]
%% Output: valid/sa/out/033.xml
%% Description:
%%   Test demonstrates that both a EncodingDecl and SDDecl are valid within the
%%   prolog.
'valid-sa-033'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/033.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/033.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/034.xml
%% ID: valid-sa-034
%% Type: valid
%% Sections: 3.1 [44]
%% Output: valid/sa/out/034.xml
%% Description:
%%   Test demonstrates the correct syntax for an Empty element tag.
'valid-sa-034'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/034.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/034.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/035.xml
%% ID: valid-sa-035
%% Type: valid
%% Sections: 3.1 [44]
%% Output: valid/sa/out/035.xml
%% Description:
%%   Test demonstrates that whitespace is permissible after the name in an Empty
%%   element tag.
'valid-sa-035'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/035.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/035.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/036.xml
%% ID: valid-sa-036
%% Type: valid
%% Sections: 2.6 [16]
%% Output: valid/sa/out/036.xml
%% Description:
%%   Test demonstrates a valid processing instruction.
'valid-sa-036'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/036.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/036.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/017a.xml
%% ID: valid-sa-017a
%% Type: valid
%% Sections: 2.6 3.1 [16] [43]
%% Output: valid/sa/out/017a.xml
%% Description:
%%   Test demonstrates that two apparently wrong Processing Instructions make a
%%   right one, with very odd content "some data ? > <?".
'valid-sa-017a'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/017a.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/017a.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/037.xml
%% ID: valid-sa-037
%% Type: valid
%% Sections: 2.6 [15]
%% Output: valid/sa/out/037.xml
%% Description:
%%   Test demonstrates a valid comment and that it may appear anywhere in the
%%   document including at the end.
'valid-sa-037'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/037.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/037.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/038.xml
%% ID: valid-sa-038
%% Type: valid
%% Sections: 2.6 [15]
%% Output: valid/sa/out/038.xml
%% Description:
%%   Test demonstrates a valid comment and that it may appear anywhere in the
%%   document including the beginning.
'valid-sa-038'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/038.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/038.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/039.xml
%% ID: valid-sa-039
%% Type: valid
%% Sections: 2.6 [16]
%% Output: valid/sa/out/039.xml
%% Description:
%%   Test demonstrates a valid processing instruction and that it may appear at
%%   the beginning of the document.
'valid-sa-039'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/039.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/039.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/040.xml
%% ID: valid-sa-040
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [54]
%% Output: valid/sa/out/040.xml
%% Description:
%%   Test demonstrates an Attribute List declaration that uses a StringType as
%%   the AttType.
'valid-sa-040'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/040.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/040.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/041.xml
%% ID: valid-sa-041
%% Type: valid
%% Sections: 3.3.1 4.1 [54] [66]
%% Output: valid/sa/out/041.xml
%% Description:
%%   Test demonstrates an Attribute List declaration that uses a StringType as
%%   the AttType and also expands the CDATA attribute with a character
%%   reference.
'valid-sa-041'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/041.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/041.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/042.xml
%% ID: valid-sa-042
%% Type: valid
%% Sections: 3.3.1 4.1 [54] [66]
%% Output: valid/sa/out/042.xml
%% Description:
%%   Test demonstrates an Attribute List declaration that uses a StringType as
%%   the AttType and also expands the CDATA attribute with a character
%%   reference. The test also shows that the leading zeros in the character
%%   reference are ignored.
'valid-sa-042'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/042.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/042.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/043.xml
%% ID: valid-sa-043
%% Type: valid
%% Sections: 3.3
%% Output: valid/sa/out/043.xml
%% Description:
%%   An element's attributes may be declared before its content model; and
%%   attribute values may contain newlines.
'valid-sa-043'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/043.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/043.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/044.xml
%% ID: valid-sa-044
%% Type: valid
%% Sections: 3.1 [44]
%% Output: valid/sa/out/044.xml
%% Description:
%%   Test demonstrates that the empty-element tag must be use for an elements
%%   that are declared EMPTY.
'valid-sa-044'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/044.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/044.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/045.xml
%% ID: valid-sa-045
%% Type: valid
%% Sections: 3.3 [52]
%% Output: valid/sa/out/045.xml
%% Description:
%%   Tests whether more than one definition can be provided for the same
%%   attribute of a given element type with the first declaration being binding.
'valid-sa-045'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/045.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/045.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/046.xml
%% ID: valid-sa-046
%% Type: valid
%% Sections: 3.3 [52]
%% Output: valid/sa/out/046.xml
%% Description:
%%   Test demonstrates that when more than one AttlistDecl is provided for a
%%   given element type, the contents of all those provided are merged.
'valid-sa-046'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/046.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/046.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/047.xml
%% ID: valid-sa-047
%% Type: valid
%% Sections: 3.1 [43]
%% Output: valid/sa/out/047.xml
%% Description:
%%   Test demonstrates that extra whitespace is normalized into single space
%%   character.
'valid-sa-047'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/047.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/047.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/048.xml
%% ID: valid-sa-048
%% Type: valid
%% Sections: 2.4 3.1 [14] [43]
%% Output: valid/sa/out/048.xml
%% Description:
%%   Test demonstrates that character data is valid element content.
'valid-sa-048'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/048.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/048.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/049.xml
%% ID: valid-sa-049
%% Type: valid
%% Sections: 2.2 [2]
%% Output: valid/sa/out/049.xml
%% Description:
%%   Test demonstrates that characters outside of normal ascii range can be used
%%   as element content.
'valid-sa-049'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/049.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/049.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/050.xml
%% ID: valid-sa-050
%% Type: valid
%% Sections: 2.2 [2]
%% Output: valid/sa/out/050.xml
%% Description:
%%   Test demonstrates that characters outside of normal ascii range can be used
%%   as element content.
'valid-sa-050'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/050.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/050.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/051.xml
%% ID: valid-sa-051
%% Type: valid
%% Sections: 2.2 [2]
%% Output: valid/sa/out/051.xml
%% Description:
%%   The document is encoded in UTF-16 and uses some name characters well
%%   outside of the normal ASCII range.
'valid-sa-051'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/051.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/051.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/052.xml
%% ID: valid-sa-052
%% Type: valid
%% Sections: 2.2 [2]
%% Output: valid/sa/out/052.xml
%% Description:
%%   The document is encoded in UTF-8 and the text inside the root element uses
%%   two non-ASCII characters, encoded in UTF-8 and each of which expands to a
%%   Unicode surrogate pair.
'valid-sa-052'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/052.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/052.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/053.xml
%% ID: valid-sa-053
%% Type: valid
%% Sections: 4.4.2
%% Output: valid/sa/out/053.xml
%% Description:
%%   Tests inclusion of a well-formed internal entity, which holds an element
%%   required by the content model.
'valid-sa-053'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/053.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/053.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/054.xml
%% ID: valid-sa-054
%% Type: valid
%% Sections: 3.1 [40] [42]
%% Output: valid/sa/out/054.xml
%% Description:
%%   Test demonstrates that extra whitespace within Start-tags and End-tags are
%%   nomalized into single spaces.
'valid-sa-054'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/054.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/054.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/055.xml
%% ID: valid-sa-055
%% Type: valid
%% Sections: 2.6 2.10 [16]
%% Output: valid/sa/out/055.xml
%% Description:
%%   Test demonstrates that extra whitespace within a processing instruction
%%   willnormalized into s single space character.
'valid-sa-055'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/055.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/055.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/056.xml
%% ID: valid-sa-056
%% Type: valid
%% Sections: 3.3.1 4.1 [54] [66]
%% Output: valid/sa/out/056.xml
%% Description:
%%   Test demonstrates an Attribute List declaration that uses a StringType as
%%   the AttType and also expands the CDATA attribute with a character
%%   reference. The test also shows that the leading zeros in the character
%%   reference are ignored.
'valid-sa-056'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/056.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/056.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/057.xml
%% ID: valid-sa-057
%% Type: valid
%% Sections: 3.2.1 [47]
%% Output: valid/sa/out/057.xml
%% Description:
%%   Test demonstrates an element content model whose element can occur zero or
%%   more times.
'valid-sa-057'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/057.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/057.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/058.xml
%% ID: valid-sa-058
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/058.xml
%% Description:
%%   Test demonstrates that extra whitespace be normalized into a single space
%%   character in an attribute of type NMTOKENS.
'valid-sa-058'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/058.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/058.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/059.xml
%% ID: valid-sa-059
%% Type: valid
%% Sections: 3.2 3.3 [46] [53]
%% Output: valid/sa/out/059.xml
%% Description:
%%   Test demonstrates an Element Type Declaration that uses the contentspec of
%%   EMPTY. The element cannot have any contents and must always appear as an
%%   empty element in the document. The test also shows an Attribute-list
%%   declaration with multiple AttDef's.
'valid-sa-059'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/059.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/059.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/060.xml
%% ID: valid-sa-060
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/060.xml
%% Description:
%%   Test demonstrates the use of decimal Character References within element
%%   content.
'valid-sa-060'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/060.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/060.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/061.xml
%% ID: valid-sa-061
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/061.xml
%% Description:
%%   Test demonstrates the use of decimal Character References within element
%%   content.
'valid-sa-061'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/061.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/061.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/062.xml
%% ID: valid-sa-062
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/062.xml
%% Description:
%%   Test demonstrates the use of hexadecimal Character References within
%%   element.
'valid-sa-062'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/062.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/062.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/063.xml
%% ID: valid-sa-063
%% Type: valid
%% Sections: 2.3 [5]
%% Output: valid/sa/out/063.xml
%% Description:
%%   The document is encoded in UTF-8 and the name of the root element type uses
%%   non-ASCII characters.
'valid-sa-063'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/063.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/063.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/064.xml
%% ID: valid-sa-064
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/064.xml
%% Description:
%%   Tests in-line handling of two legal character references, which each expand
%%   to a Unicode surrogate pair.
'valid-sa-064'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/064.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/064.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/065.xml
%% ID: valid-sa-065
%% Type: valid
%% Sections: 4.5
%% Output: valid/sa/out/065.xml
%% Description:
%%   Tests ability to define an internal entity which can't legally be expanded
%%   (contains an unquoted ).
'valid-sa-065'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/065.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/065.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/066.xml
%% ID: valid-sa-066
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/066.xml
%% Description:
%%   Expands a CDATA attribute with a character reference.
'valid-sa-066'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/066.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/066.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/067.xml
%% ID: valid-sa-067
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/067.xml
%% Description:
%%   Test demonstrates the use of decimal character references within element
%%   content.
'valid-sa-067'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/067.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/067.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/068.xml
%% ID: valid-sa-068
%% Type: valid
%% Sections: 2.11, 4.5
%% Output: valid/sa/out/068.xml
%% Description:
%%   Tests definition of an internal entity holding a carriage return character
%%   reference, which must not be normalized before reporting to the
%%   application. Line break normalization only occurs when parsing external
%%   parsed entities.
'valid-sa-068'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/068.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/068.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/069.xml
%% ID: valid-sa-069
%% Type: valid
%% Sections: 4.7
%% Output: valid/sa/out/069.xml
%% Description:
%%   Verifies that an XML parser will parse a NOTATION declaration; the output
%%   phase of this test ensures that it's reported to the application.
'valid-sa-069'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/069.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/069.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/070.xml
%% ID: valid-sa-070
%% Type: valid
%% Sections: 4.4.8
%% Entities: parameter
%% Output: valid/sa/out/070.xml
%% Description:
%%   Verifies that internal parameter entities are correctly expanded within the
%%   internal subset.
'valid-sa-070'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/070.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/070.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/071.xml
%% ID: valid-sa-071
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [56]
%% Output: valid/sa/out/071.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use ID as the TokenizedType
%%   within the Attribute type. The test also shows that IMPLIED is a valid
%%   DefaultDecl.
'valid-sa-071'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/071.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/071.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/072.xml
%% ID: valid-sa-072
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [56]
%% Output: valid/sa/out/072.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use IDREF as the TokenizedType
%%   within the Attribute type. The test also shows that IMPLIED is a valid
%%   DefaultDecl.
'valid-sa-072'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/072.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/072.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/073.xml
%% ID: valid-sa-073
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [56]
%% Output: valid/sa/out/073.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use IDREFS as the TokenizedType
%%   within the Attribute type. The test also shows that IMPLIED is a valid
%%   DefaultDecl.
'valid-sa-073'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/073.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/073.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/074.xml
%% ID: valid-sa-074
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [56]
%% Output: valid/sa/out/074.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use ENTITY as the TokenizedType
%%   within the Attribute type. The test also shows that IMPLIED is a valid
%%   DefaultDecl.
'valid-sa-074'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/074.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/074.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/075.xml
%% ID: valid-sa-075
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [56]
%% Output: valid/sa/out/075.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use ENTITIES as the TokenizedType
%%   within the Attribute type. The test also shows that IMPLIED is a valid
%%   DefaultDecl.
'valid-sa-075'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/075.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/075.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/076.xml
%% ID: valid-sa-076
%% Type: valid
%% Sections: 3.3.1
%% Output: valid/sa/out/076.xml
%% Description:
%%   Verifies that an XML parser will parse a NOTATION attribute; the output
%%   phase of this test ensures that both notations are reported to the
%%   application.
'valid-sa-076'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/076.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/076.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/077.xml
%% ID: valid-sa-077
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [54]
%% Output: valid/sa/out/077.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use an EnumeratedType within the
%%   Attribute type. The test also shows that IMPLIED is a valid DefaultDecl.
'valid-sa-077'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/077.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/077.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/078.xml
%% ID: valid-sa-078
%% Type: valid
%% Sections: 3.3 3.3.1 [52] [54]
%% Output: valid/sa/out/078.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use an StringType of CDATA within
%%   the Attribute type. The test also shows that REQUIRED is a valid
%%   DefaultDecl.
'valid-sa-078'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/078.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/078.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/079.xml
%% ID: valid-sa-079
%% Type: valid
%% Sections: 3.3 3.3.2 [52] [60]
%% Output: valid/sa/out/079.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use an StringType of CDATA within
%%   the Attribute type. The test also shows that FIXED is a valid DefaultDecl
%%   and that a value can be given to the attribute in the Start-tag as well as
%%   the AttListDecl.
'valid-sa-079'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/079.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/079.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/080.xml
%% ID: valid-sa-080
%% Type: valid
%% Sections: 3.3 3.3.2 [52] [60]
%% Output: valid/sa/out/080.xml
%% Description:
%%   Test demonstrates that an AttlistDecl can use an StringType of CDATA within
%%   the Attribute type. The test also shows that FIXED is a valid DefaultDecl
%%   and that an value can be given to the attribute.
'valid-sa-080'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/080.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/080.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/081.xml
%% ID: valid-sa-081
%% Type: valid
%% Sections: 3.2.1 [50]
%% Output: valid/sa/out/081.xml
%% Description:
%%   Test demonstrates the use of the optional character following a name or
%%   list to govern the number of times an element or content particles in the
%%   list occur.
'valid-sa-081'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/081.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/081.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/082.xml
%% ID: valid-sa-082
%% Type: valid
%% Sections: 4.2 [72]
%% Output: valid/sa/out/082.xml
%% Description:
%%   Tests that an external PE may be defined (but not referenced).
'valid-sa-082'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/082.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/082.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/083.xml
%% ID: valid-sa-083
%% Type: valid
%% Sections: 4.2 [72]
%% Output: valid/sa/out/083.xml
%% Description:
%%   Tests that an external PE may be defined (but not referenced).
'valid-sa-083'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/083.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/083.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/084.xml
%% ID: valid-sa-084
%% Type: valid
%% Sections: 2.10
%% Output: valid/sa/out/084.xml
%% Description:
%%   Test demonstrates that although whitespace can be used to set apart markup
%%   for greater readability it is not necessary.
'valid-sa-084'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/084.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/084.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/085.xml
%% ID: valid-sa-085
%% Type: valid
%% Sections: 4
%% Output: valid/sa/out/085.xml
%% Description:
%%   Parameter and General entities use different namespaces, so there can be an
%%   entity of each type with a given name.
'valid-sa-085'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/085.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/085.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/086.xml
%% ID: valid-sa-086
%% Type: valid
%% Sections: 4.2
%% Output: valid/sa/out/086.xml
%% Description:
%%   Tests whether entities may be declared more than once, with the first
%%   declaration being the binding one.
'valid-sa-086'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/086.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/086.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/087.xml
%% ID: valid-sa-087
%% Type: valid
%% Sections: 4.5
%% Output: valid/sa/out/087.xml
%% Description:
%%   Tests whether character references in internal entities are expanded early
%%   enough, by relying on correct handling to make the entity be well formed.
'valid-sa-087'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/087.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/087.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/088.xml
%% ID: valid-sa-088
%% Type: valid
%% Sections: 4.5
%% Output: valid/sa/out/088.xml
%% Description:
%%   Tests whether entity references in internal entities are expanded late
%%   enough, by relying on correct handling to make the expanded text be valid.
%%   (If it's expanded too early, the entity will parse as an element that's not
%%   valid in that context.)
'valid-sa-088'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/088.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/088.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/089.xml
%% ID: valid-sa-089
%% Type: valid
%% Sections: 4.1 [66]
%% Output: valid/sa/out/089.xml
%% Description:
%%   Tests entity expansion of three legal character references, which each
%%   expand to a Unicode surrogate pair.
'valid-sa-089'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/089.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/089.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/090.xml
%% ID: valid-sa-090
%% Type: valid
%% Sections: 3.3.1
%% Output: valid/sa/out/090.xml
%% Description:
%%   Verifies that an XML parser will parse a NOTATION attribute; the output
%%   phase of this test ensures that the notation is reported to the
%%   application.
'valid-sa-090'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/090.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/090.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/091.xml
%% ID: valid-sa-091
%% Type: valid
%% Sections: 3.3.1
%% Output: valid/sa/out/091.xml
%% Description:
%%   Verifies that an XML parser will parse an ENTITY attribute; the output
%%   phase of this test ensures that the notation is reported to the
%%   application, and for validating parsers it further tests that the entity is
%%   so reported.
'valid-sa-091'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/091.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/091.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/092.xml
%% ID: valid-sa-092
%% Type: valid
%% Sections: 2.3 2.10
%% Output: valid/sa/out/092.xml
%% Description:
%%   Test demonstrates that extra whitespace is normalized into a single space
%%   character.
'valid-sa-092'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/092.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/092.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/093.xml
%% ID: valid-sa-093
%% Type: valid
%% Sections: 2.10
%% Output: valid/sa/out/093.xml
%% Description:
%%   Test demonstrates that extra whitespace is not intended for inclusion in
%%   the delivered version of the document.
'valid-sa-093'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/093.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/093.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/094.xml
%% ID: valid-sa-094
%% Type: valid
%% Sections: 2.8
%% Output: valid/sa/out/094.xml
%% Description:
%%   Attribute defaults with a DTD have special parsing rules, different from
%%   other strings. That means that characters found there may look like an
%%   undefined parameter entity reference "within a markup declaration", but
%%   they aren't ... so they can't be violating the WFC.
'valid-sa-094'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/094.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/094.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/095.xml
%% ID: valid-sa-095
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/095.xml
%% Description:
%%   Basically an output test, this requires extra whitespace to be normalized
%%   into a single space character in an attribute of type NMTOKENS.
'valid-sa-095'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/095.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/095.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/096.xml
%% ID: valid-sa-096
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/096.xml
%% Description:
%%   Test demonstrates that extra whitespace is normalized into a single space
%%   character in an attribute of type NMTOKENS.
'valid-sa-096'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/096.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/096.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/097.xml
%% ID: valid-sa-097
%% Type: valid
%% Sections: 3.3
%% Entities: parameter
%% Output: valid/sa/out/097.xml
%% Description:
%%   Basically an output test, this tests whether an externally defined
%%   attribute declaration (with a default) takes proper precedence over a
%%   subsequent internal declaration.
'valid-sa-097'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/097.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/097.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/098.xml
%% ID: valid-sa-098
%% Type: valid
%% Sections: 2.6 2.10 [16]
%% Output: valid/sa/out/098.xml
%% Description:
%%   Test demonstrates that extra whitespace within a processing instruction is
%%   converted into a single space character.
'valid-sa-098'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/098.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/098.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/099.xml
%% ID: valid-sa-099
%% Type: valid
%% Sections: 4.3.3 [81]
%% Output: valid/sa/out/099.xml
%% Description:
%%   Test demonstrates the name of the encoding can be composed of lowercase
%%   characters.
'valid-sa-099'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/099.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/099.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/100.xml
%% ID: valid-sa-100
%% Type: valid
%% Sections: 2.3 [12]
%% Output: valid/sa/out/100.xml
%% Description:
%%   Makes sure that PUBLIC identifiers may have some strange characters.
'valid-sa-100'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/100.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/100.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/101.xml
%% ID: valid-sa-101
%% Type: valid
%% Sections: 4.5
%% Output: valid/sa/out/101.xml
%% Description:
%%   This tests whether entity expansion is (incorrectly) done while processing
%%   entity declarations; if it is, the entity value literal will terminate
%%   prematurely.
'valid-sa-101'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/101.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/101.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/102.xml
%% ID: valid-sa-102
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/102.xml
%% Description:
%%   Test demonstrates that a CDATA attribute can pass a double quote as its
%%   value.
'valid-sa-102'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/102.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/102.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/103.xml
%% ID: valid-sa-103
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/103.xml
%% Description:
%%   Test demonstrates that an attribute can pass a less than sign as its value.
'valid-sa-103'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/103.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/103.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/104.xml
%% ID: valid-sa-104
%% Type: valid
%% Sections: 3.1 [40]
%% Output: valid/sa/out/104.xml
%% Description:
%%   Test demonstrates that extra whitespace within an Attribute of a Start-tag
%%   is normalized to a single space character.
'valid-sa-104'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/104.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/104.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/105.xml
%% ID: valid-sa-105
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/105.xml
%% Description:
%%   Basically an output test, this requires a CDATA attribute with a tab
%%   character to be passed through as one space.
'valid-sa-105'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/105.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/105.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/106.xml
%% ID: valid-sa-106
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/106.xml
%% Description:
%%   Basically an output test, this requires a CDATA attribute with a newline
%%   character to be passed through as one space.
'valid-sa-106'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/106.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/106.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/107.xml
%% ID: valid-sa-107
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/107.xml
%% Description:
%%   Basically an output test, this requires a CDATA attribute with a return
%%   character to be passed through as one space.
'valid-sa-107'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/107.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/107.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/108.xml
%% ID: valid-sa-108
%% Type: valid
%% Sections: 2.11, 3.3.3
%% Output: valid/sa/out/108.xml
%% Description:
%%   This tests normalization of end-of-line characters (CRLF) within entities
%%   to LF, primarily as an output test.
'valid-sa-108'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/108.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/108.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/109.xml
%% ID: valid-sa-109
%% Type: valid
%% Sections: 2.3 3.1 [10][40][41]
%% Output: valid/sa/out/109.xml
%% Description:
%%   Test demonstrates that an attribute can have a null value.
'valid-sa-109'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/109.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/109.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/110.xml
%% ID: valid-sa-110
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/110.xml
%% Description:
%%   Basically an output test, this requires that a CDATA attribute with a CRLF
%%   be normalized to one space.
'valid-sa-110'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/110.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/110.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/111.xml
%% ID: valid-sa-111
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/111.xml
%% Description:
%%   Character references expanding to spaces doesn't affect treatment of
%%   attributes.
'valid-sa-111'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/111.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/111.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/112.xml
%% ID: valid-sa-112
%% Type: valid
%% Sections: 3.2.1 [48][49]
%% Output: valid/sa/out/112.xml
%% Description:
%%   Test demonstrates shows the use of content particles within the element
%%   content.
'valid-sa-112'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/112.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/112.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/113.xml
%% ID: valid-sa-113
%% Type: valid
%% Sections: 3.3 [52][53]
%% Output: valid/sa/out/113.xml
%% Description:
%%   Test demonstrates that it is not an error to have attributes declared for
%%   an element not itself declared.
'valid-sa-113'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/113.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/113.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/114.xml
%% ID: valid-sa-114
%% Type: valid
%% Sections: 2.7 [20]
%% Output: valid/sa/out/114.xml
%% Description:
%%   Test demonstrates that all text within a valid CDATA section is considered
%%   text and not recognized as markup.
'valid-sa-114'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/114.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/114.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/115.xml
%% ID: valid-sa-115
%% Type: valid
%% Sections: 3.3.3
%% Output: valid/sa/out/115.xml
%% Description:
%%   Test demonstrates that an entity reference is processed by recursively
%%   processing the replacement text of the entity.
'valid-sa-115'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/115.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/115.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/116.xml
%% ID: valid-sa-116
%% Type: valid
%% Sections: 2.11
%% Output: valid/sa/out/116.xml
%% Description:
%%   Test demonstrates that a line break within CDATA will be normalized.
'valid-sa-116'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/116.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/116.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/117.xml
%% ID: valid-sa-117
%% Type: valid
%% Sections: 4.5
%% Output: valid/sa/out/117.xml
%% Description:
%%   Test demonstrates that entity expansion is done while processing entity
%%   declarations.
'valid-sa-117'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/117.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/117.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/118.xml
%% ID: valid-sa-118
%% Type: valid
%% Sections: 4.5
%% Output: valid/sa/out/118.xml
%% Description:
%%   Test demonstrates that entity expansion is done while processing entity
%%   declarations.
'valid-sa-118'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/118.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/118.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa/119.xml
%% ID: valid-sa-119
%% Type: valid
%% Sections: 2.5
%% Output: valid/sa/out/119.xml
%% Description:
%%   Comments may contain any legal XML characters; only the string "--" is
%%   disallowed.
'valid-sa-119'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/sa/119.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/sa/out/119.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/001.xml
%% ID: valid-not-sa-001
%% Type: valid
%% Sections: 4.2.2 [75]
%% Entities: both
%% Output: valid/not-sa/out/001.xml
%% Description:
%%   Test demonstrates the use of an ExternalID within a document type
%%   definition.
'valid-not-sa-001'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/001.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/001.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/002.xml
%% ID: valid-not-sa-002
%% Type: valid
%% Sections: 4.2.2 [75]
%% Entities: both
%% Output: valid/not-sa/out/002.xml
%% Description:
%%   Test demonstrates the use of an ExternalID within a document type
%%   definition.
'valid-not-sa-002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/002.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/002.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/003.xml
%% ID: valid-not-sa-003
%% Type: valid
%% Sections: 4.1 [69]
%% Entities: both
%% Output: valid/not-sa/out/003.xml
%% Description:
%%   Test demonstrates the expansion of an external parameter entity that
%%   declares an attribute.
'valid-not-sa-003'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/003.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/003.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/004.xml
%% ID: valid-not-sa-004
%% Type: valid
%% Sections: 4.1 [69]
%% Entities: both
%% Output: valid/not-sa/out/004.xml
%% Description:
%%   Expands an external parameter entity in two different ways, with one of
%%   them declaring an attribute.
'valid-not-sa-004'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/004.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/004.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/005.xml
%% ID: valid-not-sa-005
%% Type: valid
%% Sections: 4.1 [69]
%% Entities: both
%% Output: valid/not-sa/out/005.xml
%% Description:
%%   Test demonstrates the expansion of an external parameter entity that
%%   declares an attribute.
'valid-not-sa-005'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/005.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/005.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/006.xml
%% ID: valid-not-sa-006
%% Type: valid
%% Sections: 3.3 [52]
%% Entities: both
%% Output: valid/not-sa/out/006.xml
%% Description:
%%   Test demonstrates that when more than one definition is provided for the
%%   same attribute of a given element type only the first declaration is
%%   binding.
'valid-not-sa-006'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/006.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/006.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/007.xml
%% ID: valid-not-sa-007
%% Type: valid
%% Sections: 3.3 [52]
%% Entities: both
%% Output: valid/not-sa/out/007.xml
%% Description:
%%   Test demonstrates the use of an Attribute list declaration within an
%%   external entity.
'valid-not-sa-007'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/007.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/007.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/008.xml
%% ID: valid-not-sa-008
%% Type: valid
%% Sections: 4.2.2 [75]
%% Entities: both
%% Output: valid/not-sa/out/008.xml
%% Description:
%%   Test demonstrates that an external identifier may include a public
%%   identifier.
'valid-not-sa-008'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/008.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/008.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/009.xml
%% ID: valid-not-sa-009
%% Type: valid
%% Sections: 4.2.2 [75]
%% Entities: both
%% Output: valid/not-sa/out/009.xml
%% Description:
%%   Test demonstrates that an external identifier may include a public
%%   identifier.
'valid-not-sa-009'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/009.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/009.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/010.xml
%% ID: valid-not-sa-010
%% Type: valid
%% Sections: 3.3 [52]
%% Entities: both
%% Output: valid/not-sa/out/010.xml
%% Description:
%%   Test demonstrates that when more that one definition is provided for the
%%   same attribute of a given element type only the first declaration is
%%   binding.
'valid-not-sa-010'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/010.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/010.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/011.xml
%% ID: valid-not-sa-011
%% Type: valid
%% Sections: 4.2 4.2.1 [72] [75]
%% Entities: both
%% Output: valid/not-sa/out/011.xml
%% Description:
%%   Test demonstrates a parameter entity declaration whose parameter entity
%%   definition is an ExternalID.
'valid-not-sa-011'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/011.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/011.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/012.xml
%% ID: valid-not-sa-012
%% Type: valid
%% Sections: 4.3.1 [77]
%% Entities: both
%% Output: valid/not-sa/out/012.xml
%% Description:
%%   Test demonstrates an enternal parsed entity that begins with a text
%%   declaration.
'valid-not-sa-012'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/012.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/012.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/013.xml
%% ID: valid-not-sa-013
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: valid/not-sa/out/013.xml
%% Description:
%%   Test demonstrates the use of the conditional section INCLUDE that will
%%   include its contents as part of the DTD.
'valid-not-sa-013'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/013.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/013.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/014.xml
%% ID: valid-not-sa-014
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: valid/not-sa/out/014.xml
%% Description:
%%   Test demonstrates the use of the conditional section INCLUDE that will
%%   include its contents as part of the DTD. The keyword is a parameter-entity
%%   reference.
'valid-not-sa-014'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/014.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/014.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/015.xml
%% ID: valid-not-sa-015
%% Type: valid
%% Sections: 3.4 [63]
%% Entities: both
%% Output: valid/not-sa/out/015.xml
%% Description:
%%   Test demonstrates the use of the conditional section IGNORE the will ignore
%%   its content from being part of the DTD. The keyword is a parameter-entity
%%   reference.
'valid-not-sa-015'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/015.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/015.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/016.xml
%% ID: valid-not-sa-016
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: valid/not-sa/out/016.xml
%% Description:
%%   Test demonstrates the use of the conditional section INCLUDE that will
%%   include its contents as part of the DTD. The keyword is a parameter-entity
%%   reference.
'valid-not-sa-016'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/016.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/016.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/017.xml
%% ID: valid-not-sa-017
%% Type: valid
%% Sections: 4.2 [72]
%% Entities: both
%% Output: valid/not-sa/out/017.xml
%% Description:
%%   Test demonstrates a parameter entity declaration that contains an attribute
%%   list declaration.
'valid-not-sa-017'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/017.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/017.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/018.xml
%% ID: valid-not-sa-018
%% Type: valid
%% Sections: 4.2.2 [75]
%% Entities: both
%% Output: valid/not-sa/out/018.xml
%% Description:
%%   Test demonstrates an EnternalID whose contents contain an parameter entity
%%   declaration and a attribute list definition.
'valid-not-sa-018'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/018.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/018.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/019.xml
%% ID: valid-not-sa-019
%% Type: valid
%% Sections: 4.4.8
%% Entities: both
%% Output: valid/not-sa/out/019.xml
%% Description:
%%   Test demonstrates that a parameter entity will be expanded with spaces on
%%   either side.
'valid-not-sa-019'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/019.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/019.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/020.xml
%% ID: valid-not-sa-020
%% Type: valid
%% Sections: 4.4.8
%% Entities: both
%% Output: valid/not-sa/out/020.xml
%% Description:
%%   Parameter entities expand with spaces on either side.
'valid-not-sa-020'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/020.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/020.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/021.xml
%% ID: valid-not-sa-021
%% Type: valid
%% Sections: 4.2 [72]
%% Entities: both
%% Output: valid/not-sa/out/021.xml
%% Description:
%%   Test demonstrates a parameter entity declaration that contains a partial
%%   attribute list declaration.
'valid-not-sa-021'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/021.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/021.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/023.xml
%% ID: valid-not-sa-023
%% Type: valid
%% Sections: 2.3 4.1 [10] [69]
%% Entities: both
%% Output: valid/not-sa/out/023.xml
%% Description:
%%   Test demonstrates the use of a parameter entity reference within an
%%   attribute list declaration.
'valid-not-sa-023'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/023.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/023.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/024.xml
%% ID: valid-not-sa-024
%% Type: valid
%% Sections: 2.8, 4.1 [69]
%% Entities: both
%% Output: valid/not-sa/out/024.xml
%% Description:
%%   Constructs an <!ATTLIST...> declaration from several PEs.
'valid-not-sa-024'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/024.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/024.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/025.xml
%% ID: valid-not-sa-025
%% Type: valid
%% Sections: 4.2
%% Entities: both
%% Output: valid/not-sa/out/025.xml
%% Description:
%%   Test demonstrates that when more that one definition is provided for the
%%   same entity only the first declaration is binding.
'valid-not-sa-025'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/025.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/025.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/026.xml
%% ID: valid-not-sa-026
%% Type: valid
%% Sections: 3.3 [52]
%% Entities: both
%% Output: valid/not-sa/out/026.xml
%% Description:
%%   Test demonstrates that when more that one definition is provided for the
%%   same attribute of a given element type only the first declaration is
%%   binding.
'valid-not-sa-026'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/026.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/026.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/027.xml
%% ID: valid-not-sa-027
%% Type: valid
%% Sections: 4.1 [69]
%% Entities: both
%% Output: valid/not-sa/out/027.xml
%% Description:
%%   Test demonstrates a parameter entity reference whose value is NULL.
'valid-not-sa-027'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/027.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/027.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/028.xml
%% ID: valid-not-sa-028
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: valid/not-sa/out/028.xml
%% Description:
%%   Test demonstrates the use of the conditional section INCLUDE that will
%%   include its contents.
'valid-not-sa-028'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/028.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/028.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/029.xml
%% ID: valid-not-sa-029
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: valid/not-sa/out/029.xml
%% Description:
%%   Test demonstrates the use of the conditional section IGNORE the will ignore
%%   its content from being used.
'valid-not-sa-029'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/029.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/029.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/030.xml
%% ID: valid-not-sa-030
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: both
%% Output: valid/not-sa/out/030.xml
%% Description:
%%   Test demonstrates the use of the conditional section IGNORE the will ignore
%%   its content from being used.
'valid-not-sa-030'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/030.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/030.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa/031.xml
%% ID: valid-not-sa-031
%% Type: valid
%% Sections: 2.7
%% Entities: both
%% Output: valid/not-sa/out/031.xml
%% Description:
%%   Expands a general entity which contains a CDATA section with what looks
%%   like a markup declaration (but is just text since it's in a CDATA section).
'valid-not-sa-031'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/not-sa/031.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/not-sa/out/031.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/001.xml
%% ID: valid-ext-sa-001
%% Type: valid
%% Sections: 2.11
%% Entities: both
%% Output: valid/ext-sa/out/001.xml
%% Description:
%%   A combination of carriage return line feed in an external entity must be
%%   normalized to a single newline.
'valid-ext-sa-001'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/001.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/001.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/002.xml
%% ID: valid-ext-sa-002
%% Type: valid
%% Sections: 2.11
%% Entities: both
%% Output: valid/ext-sa/out/002.xml
%% Description:
%%   A carriage return (also CRLF) in an external entity must be normalized to a
%%   single newline.
'valid-ext-sa-002'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/002.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/002.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/003.xml
%% ID: valid-ext-sa-003
%% Type: valid
%% Sections: 3.1 4.1 [43] [68]
%% Entities: both
%% Output: valid/ext-sa/out/003.xml
%% Description:
%%   Test demonstrates that the content of an element can be empty. In this case
%%   the external entity is an empty file.
'valid-ext-sa-003'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/003.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/003.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/004.xml
%% ID: valid-ext-sa-004
%% Type: valid
%% Sections: 2.11
%% Entities: both
%% Output: valid/ext-sa/out/004.xml
%% Description:
%%   A carriage return (also CRLF) in an external entity must be normalized to a
%%   single newline.
'valid-ext-sa-004'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/004.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/004.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/005.xml
%% ID: valid-ext-sa-005
%% Type: valid
%% Sections: 3.2.1 4.2.2 [48] [75]
%% Entities: both
%% Output: valid/ext-sa/out/005.xml
%% Description:
%%   Test demonstrates the use of optional character and content particles
%%   within an element content. The test also show the use of external entity.
'valid-ext-sa-005'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/005.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/005.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/006.xml
%% ID: valid-ext-sa-006
%% Type: valid
%% Sections: 2.11 3.2.1 3.2.2 4.2.2 [48] [51] [75]
%% Entities: both
%% Output: valid/ext-sa/out/006.xml
%% Description:
%%   Test demonstrates the use of optional character and content particles
%%   within mixed element content. The test also shows the use of an external
%%   entity and that a carriage control line feed in an external entity must be
%%   normalized to a single newline.
'valid-ext-sa-006'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/006.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/006.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/007.xml
%% ID: valid-ext-sa-007
%% Type: valid
%% Sections: 4.2.2 4.4.3 [75]
%% Entities: both
%% Output: valid/ext-sa/out/007.xml
%% Description:
%%   Test demonstrates the use of external entity and how replacement text is
%%   retrieved and processed.
'valid-ext-sa-007'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/007.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/007.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/008.xml
%% ID: valid-ext-sa-008
%% Type: valid
%% Sections: 4.2.2 4.3.3. 4.4.3 [75] [80]
%% Entities: both
%% Output: valid/ext-sa/out/008.xml
%% Description:
%%   Test demonstrates the use of external entity and how replacement text is
%%   retrieved and processed. Also tests the use of an EncodingDecl of UTF-16.
'valid-ext-sa-008'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/008.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/008.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/009.xml
%% ID: valid-ext-sa-009
%% Type: valid
%% Sections: 2.11
%% Entities: both
%% Output: valid/ext-sa/out/009.xml
%% Description:
%%   A carriage return (also CRLF) in an external entity must be normalized to a
%%   single newline.
'valid-ext-sa-009'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/009.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/009.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/011.xml
%% ID: valid-ext-sa-011
%% Type: valid
%% Sections: 2.11 4.2.2 [75]
%% Entities: both
%% Output: valid/ext-sa/out/011.xml
%% Description:
%%   Test demonstrates the use of a public identifier with and external entity.
%%   The test also show that a carriage control line feed combination in an
%%   external entity must be normalized to a single newline.
'valid-ext-sa-011'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/011.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/011.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/012.xml
%% ID: valid-ext-sa-012
%% Type: valid
%% Sections: 4.2.1 4.2.2
%% Entities: both
%% Output: valid/ext-sa/out/012.xml
%% Description:
%%   Test demonstrates both internal and external entities and that processing
%%   of entity references may be required to produce the correct replacement
%%   text.
'valid-ext-sa-012'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/012.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/012.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/013.xml
%% ID: valid-ext-sa-013
%% Type: valid
%% Sections: 3.3.3
%% Entities: both
%% Output: valid/ext-sa/out/013.xml
%% Description:
%%   Test demonstrates that whitespace is handled by adding a single whitespace
%%   to the normalized value in the attribute list.
'valid-ext-sa-013'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/013.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/013.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext-sa/014.xml
%% ID: valid-ext-sa-014
%% Type: valid
%% Sections: 4.1 4.4.3 [68]
%% Entities: both
%% Output: valid/ext-sa/out/014.xml
%% Description:
%%   Test demonstrates use of characters outside of normal ASCII range.
'valid-ext-sa-014'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"xmltest","valid/ext-sa/014.xml"]),
    Out = filename:join([datadir(Config),"xmltest","valid/ext-sa/out/014.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Cases
%% Profile: Fuji Xerox Japanese Text Tests XML 1.0 Tests
%%----------------------------------------------------------------------
 
%%----------------------------------------------------------------------
%% Test Case 
%% Uri: pr-xml-euc-jp.xml
%% ID: pr-xml-euc-jp
%% Type: error
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for the EUC-JP encoding, and for text which relies on Japanese
%%   characters. (Also requires ability to process a moderately complex DTD.)
'pr-xml-euc-jp'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","pr-xml-euc-jp.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: pr-xml-iso-2022-jp.xml
%% ID: pr-xml-iso-2022-jp
%% Type: error
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for the ISO-2022-JP encoding, and for text which relies on
%%   Japanese characters. (Also requires ability to process a moderately complex
%%   DTD.)
'pr-xml-iso-2022-jp'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","pr-xml-iso-2022-jp.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: pr-xml-little-endian.xml
%% ID: pr-xml-little
%% Type: valid
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for little-endian UTF-16 text which relies on Japanese
%%   characters. (Also requires ability to process a moderately complex DTD.)
'pr-xml-little'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","pr-xml-little-endian.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: pr-xml-shift_jis.xml
%% ID: pr-xml-shift_jis
%% Type: error
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for the Shift_JIS encoding, and for text which relies on
%%   Japanese characters. (Also requires ability to process a moderately complex
%%   DTD.)
'pr-xml-shift_jis'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","pr-xml-shift_jis.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: pr-xml-utf-16.xml
%% ID: pr-xml-utf-16
%% Type: valid
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support UTF-16 text which relies on Japanese characters. (Also
%%   requires ability to process a moderately complex DTD.)
'pr-xml-utf-16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","pr-xml-utf-16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: pr-xml-utf-8.xml
%% ID: pr-xml-utf-8
%% Type: valid
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for UTF-8 text which relies on Japanese characters. (Also
%%   requires ability to process a moderately complex DTD.)
'pr-xml-utf-8'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","pr-xml-utf-8.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: weekly-euc-jp.xml
%% ID: weekly-euc-jp
%% Type: error
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for EUC-JP encoding, and XML names which contain Japanese
%%   characters.
'weekly-euc-jp'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","weekly-euc-jp.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: weekly-iso-2022-jp.xml
%% ID: weekly-iso-2022-jp
%% Type: error
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for ISO-2022-JP encoding, and XML names which contain Japanese
%%   characters.
'weekly-iso-2022-jp'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","weekly-iso-2022-jp.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: weekly-little-endian.xml
%% ID: weekly-little
%% Type: valid
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for little-endian UTF-16 encoding, and XML names which contain
%%   Japanese characters.
'weekly-little'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","weekly-little-endian.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: weekly-shift_jis.xml
%% ID: weekly-shift_jis
%% Type: error
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for Shift_JIS encoding, and XML names which contain Japanese
%%   characters.
'weekly-shift_jis'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","weekly-shift_jis.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: weekly-utf-16.xml
%% ID: weekly-utf-16
%% Type: valid
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for UTF-16 encoding, and XML names which contain Japanese
%%   characters.
'weekly-utf-16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","weekly-utf-16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: weekly-utf-8.xml
%% ID: weekly-utf-8
%% Type: valid
%% Sections: 4.3.3 [4,84]
%% Entities: parameter
%% Description:
%%   Test support for UTF-8 encoding and XML names which contain Japanese
%%   characters.
'weekly-utf-8'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"japanese","weekly-utf-8.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Cases
%% Profile: Sun Microsystems XML Tests
%%----------------------------------------------------------------------
 
%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/pe01.xml
%% ID: pe01
%% Type: valid
%% Sections: 2.8
%% Entities: parameter
%% Description:
%%   Parameter entities references are NOT RECOGNIZED in default attribute
%%   values.
pe01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/pe01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/dtd00.xml
%% ID: dtd00
%% Type: valid
%% Sections: 3.2.2 [51]
%% Output: valid/out/dtd00.xml
%% Description:
%%   Tests parsing of alternative forms of text-only mixed content declaration.
dtd00(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/dtd00.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/dtd00.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/dtd01.xml
%% ID: dtd01
%% Type: valid
%% Sections: 2.5 [15]
%% Output: valid/out/dtd01.xml
%% Description:
%%   Comments don't get parameter entity expansion
dtd01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/dtd01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/dtd01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/element.xml
%% ID: element
%% Type: valid
%% Sections: 3
%% Output: valid/out/element.xml
%% Description:
%%   Tests clauses 1, 3, and 4 of the Element Valid validity constraint.
element(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/element.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/element.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext01.xml
%% ID: ext01
%% Type: valid
%% Sections: 4.3.1 4.3.2 [77] [78]
%% Entities: general
%% Output: valid/out/ext01.xml
%% Description:
%%   Tests use of external parsed entities with and without content.
ext01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/ext01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/ext01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/ext02.xml
%% ID: ext02
%% Type: valid
%% Sections: 4.3.2 [78]
%% Entities: general
%% Output: valid/out/ext02.xml
%% Description:
%%   Tests use of external parsed entities with different encodings than the
%%   base document.
ext02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/ext02.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/ext02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa01.xml
%% ID: not-sa01
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/out/not-sa01.xml
%% Description:
%%   A non-standalone document is valid if declared as such.
'not-sa01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/not-sa01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/not-sa01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa02.xml
%% ID: not-sa02
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/out/not-sa02.xml
%% Description:
%%   A non-standalone document is valid if declared as such.
'not-sa02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/not-sa02.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/not-sa02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa03.xml
%% ID: not-sa03
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/out/not-sa03.xml
%% Description:
%%   A non-standalone document is valid if declared as such.
'not-sa03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/not-sa03.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/not-sa03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/not-sa04.xml
%% ID: not-sa04
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/out/not-sa04.xml
%% Description:
%%   A non-standalone document is valid if declared as such.
'not-sa04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/not-sa04.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/not-sa04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/notation01.xml
%% ID: notation01
%% Type: valid
%% Sections: 4.7 [82]
%% Entities: parameter
%% Output: valid/out/notation01.xml
%% Description:
%%   NOTATION declarations don't need SYSTEM IDs; and externally declared
%%   notations may be used to declare unparsed entities in the internal DTD
%%   subset. The notation must be reported to the application.
notation01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/notation01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/notation01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/optional.xml
%% ID: optional
%% Type: valid
%% Sections: 3 3.2.1 [47]
%% Entities: parameter
%% Output: valid/out/optional.xml
%% Description:
%%   Tests declarations of "children" content models, and the validity
%%   constraints associated with them.
optional(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/optional.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/optional.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/required00.xml
%% ID: required00
%% Type: valid
%% Sections: 3.3.2 [60]
%% Output: valid/out/required00.xml
%% Description:
%%   Tests the #REQUIRED attribute declaration syntax, and the associated
%%   validity constraint.
required00(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/required00.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/required00.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa01.xml
%% ID: sa01
%% Type: valid
%% Sections: 2.9 [32]
%% Output: valid/out/sa01.xml
%% Description:
%%   A document may be marked 'standalone' if any optional whitespace is defined
%%   within the internal DTD subset.
sa01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/sa01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/sa01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa02.xml
%% ID: sa02
%% Type: valid
%% Sections: 2.9 [32]
%% Output: valid/out/sa02.xml
%% Description:
%%   A document may be marked 'standalone' if any attributes that need
%%   normalization are defined within the internal DTD subset.
sa02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/sa02.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/sa02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa03.xml
%% ID: sa03
%% Type: valid
%% Sections: 2.9 [32]
%% Entities: parameter
%% Output: valid/out/sa03.xml
%% Description:
%%   A document may be marked 'standalone' if any the defined entities need
%%   expanding are internal, and no attributes need defaulting or normalization.
%%   On output, requires notations to be correctly reported.
sa03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/sa03.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/sa03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa04.xml
%% ID: sa04
%% Type: valid
%% Sections: 2.9 [32]
%% Entities: parameter
%% Output: valid/out/sa04.xml
%% Description:
%%   Like sa03 but relies on attribute defaulting defined in the internal
%%   subset. On output, requires notations to be correctly reported.
sa04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/sa04.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/sa04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sa05.xml
%% ID: sa05
%% Type: valid
%% Sections: 2.9 [32]
%% Entities: parameter
%% Output: valid/out/sa05.xml
%% Description:
%%   Like sa01 but this document is standalone since it has no optional
%%   whitespace. On output, requires notations to be correctly reported.
sa05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/sa05.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/sa05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/sgml01.xml
%% ID: v-sgml01
%% Type: valid
%% Sections: 3.3.1 [59]
%% Output: valid/out/sgml01.xml
%% Description:
%%   XML permits token reuse, while SGML does not.
'v-sgml01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/sgml01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/sgml01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/v-lang01.xml
%% ID: v-lang01
%% Type: valid
%% Sections: 2.12 [35]
%% Output: valid/out/v-lang01.xml
%% Description:
%%   Tests a lowercase ISO language code.
'v-lang01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/v-lang01.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/v-lang01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/v-lang02.xml
%% ID: v-lang02
%% Type: valid
%% Sections: 2.12 [35]
%% Output: valid/out/v-lang02.xml
%% Description:
%%   Tests a ISO language code with a subcode.
'v-lang02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/v-lang02.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/v-lang02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/v-lang03.xml
%% ID: v-lang03
%% Type: valid
%% Sections: 2.12 [36]
%% Output: valid/out/v-lang03.xml
%% Description:
%%   Tests a IANA language code with a subcode.
'v-lang03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/v-lang03.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/v-lang03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/v-lang04.xml
%% ID: v-lang04
%% Type: valid
%% Sections: 2.12 [37]
%% Output: valid/out/v-lang04.xml
%% Description:
%%   Tests a user language code with a subcode.
'v-lang04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/v-lang04.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/v-lang04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/v-lang05.xml
%% ID: v-lang05
%% Type: valid
%% Sections: 2.12 [35]
%% Output: valid/out/v-lang05.xml
%% Description:
%%   Tests an uppercase ISO language code.
'v-lang05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/v-lang05.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/v-lang05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/v-lang06.xml
%% ID: v-lang06
%% Type: valid
%% Sections: 2.12 [37]
%% Output: valid/out/v-lang06.xml
%% Description:
%%   Tests a user language code.
'v-lang06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/v-lang06.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/v-lang06.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/pe00.xml
%% ID: v-pe00
%% Type: valid
%% Sections: 4.5
%% Entities: parameter
%% Output: valid/out/pe00.xml
%% Description:
%%   Tests construction of internal entity replacement text, using an example in
%%   the XML specification.
'v-pe00'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/pe00.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/pe00.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/pe03.xml
%% ID: v-pe03
%% Type: valid
%% Sections: 4.5
%% Output: valid/out/pe03.xml
%% Description:
%%   Tests construction of internal entity replacement text, using an example in
%%   the XML specification.
'v-pe03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/pe03.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/pe03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/pe02.xml
%% ID: v-pe02
%% Type: valid
%% Sections: 4.5
%% Entities: parameter
%% Output: valid/out/pe02.xml
%% Description:
%%   Tests construction of internal entity replacement text, using a complex
%%   example in the XML specification.
'v-pe02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","valid/pe02.xml"]),
    Out = filename:join([datadir(Config),"sun","valid/out/pe02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/dtd01.xml
%% ID: inv-dtd01
%% Type: invalid
%% Sections: 3.2.2
%% Description:
%%   Tests the No Duplicate Types VC
'inv-dtd01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/dtd01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/dtd02.xml
%% ID: inv-dtd02
%% Type: invalid
%% Sections: 4.2.2
%% Description:
%%   Tests the "Notation Declared" VC by using an undeclared notation name.
'inv-dtd02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/dtd02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/dtd03.xml
%% ID: inv-dtd03
%% Type: invalid
%% Sections: 3
%% Description:
%%   Tests the "Element Valid" VC (clause 2) by omitting a required element.
'inv-dtd03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/dtd03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/el01.xml
%% ID: el01
%% Type: invalid
%% Sections: 3
%% Description:
%%   Tests the Element Valid VC (clause 4) by including an undeclared child
%%   element.
el01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/el01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/el02.xml
%% ID: el02
%% Type: invalid
%% Sections: 3
%% Description:
%%   Tests the Element Valid VC (clause 1) by including elements in an EMPTY
%%   content model.
el02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/el02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/el03.xml
%% ID: el03
%% Type: invalid
%% Sections: 3
%% Description:
%%   Tests the Element Valid VC (clause 3) by including a child element not
%%   permitted by a mixed content model.
el03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/el03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/el04.xml
%% ID: el04
%% Type: invalid
%% Sections: 3.2
%% Description:
%%   Tests the Unique Element Type Declaration VC.
el04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/el04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/el05.xml
%% ID: el05
%% Type: invalid
%% Sections: 3.2.2
%% Description:
%%   Tests the No Duplicate Types VC.
el05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/el05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/el06.xml
%% ID: el06
%% Type: invalid
%% Sections: 3
%% Description:
%%   Tests the Element Valid VC (clause 1), using one of the predefined internal
%%   entities inside an EMPTY content model.
el06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/el06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id01.xml
%% ID: id01
%% Type: invalid
%% Sections: 3.3.1
%% Entities: parameter
%% Description:
%%   Tests the ID (is a Name) VC
id01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id02.xml
%% ID: id02
%% Type: invalid
%% Sections: 3.3.1
%% Entities: parameter
%% Description:
%%   Tests the ID (appears once) VC
id02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id03.xml
%% ID: id03
%% Type: invalid
%% Sections: 3.3.1
%% Entities: parameter
%% Description:
%%   Tests the One ID per Element Type VC
id03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id04.xml
%% ID: id04
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the ID Attribute Default VC
id04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id05.xml
%% ID: id05
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the ID Attribute Default VC
id05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id06.xml
%% ID: id06
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the IDREF (is a Name) VC
id06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id07.xml
%% ID: id07
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the IDREFS (is a Names) VC
id07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id08.xml
%% ID: id08
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the IDREF (matches an ID) VC
id08(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/id09.xml
%% ID: id09
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the IDREF (IDREFS matches an ID) VC
id09(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/id09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa01.xml
%% ID: inv-not-sa01
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that optional
%%   whitespace causes a validity error.
'inv-not-sa01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa02.xml
%% ID: inv-not-sa02
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that attributes
%%   needing normalization cause a validity error.
'inv-not-sa02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa04.xml
%% ID: inv-not-sa04
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that attributes
%%   needing defaulting cause a validity error.
'inv-not-sa04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa05.xml
%% ID: inv-not-sa05
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that a token
%%   attribute that needs normalization causes a validity error.
'inv-not-sa05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa06.xml
%% ID: inv-not-sa06
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that a NOTATION
%%   attribute that needs normalization causes a validity error.
'inv-not-sa06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa07.xml
%% ID: inv-not-sa07
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an NMTOKEN
%%   attribute needing normalization causes a validity error.
'inv-not-sa07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa08.xml
%% ID: inv-not-sa08
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an NMTOKENS
%%   attribute needing normalization causes a validity error.
'inv-not-sa08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa09.xml
%% ID: inv-not-sa09
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an ID attribute
%%   needing normalization causes a validity error.
'inv-not-sa09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa10.xml
%% ID: inv-not-sa10
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an IDREF
%%   attribute needing normalization causes a validity error.
'inv-not-sa10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa11.xml
%% ID: inv-not-sa11
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an IDREFS
%%   attribute needing normalization causes a validity error.
'inv-not-sa11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa12.xml
%% ID: inv-not-sa12
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an ENTITY
%%   attribute needing normalization causes a validity error.
'inv-not-sa12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa13.xml
%% ID: inv-not-sa13
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Standalone Document Declaration VC, ensuring that an ENTITIES
%%   attribute needing normalization causes a validity error.
'inv-not-sa13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/not-sa14.xml
%% ID: inv-not-sa14
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   CDATA sections containing only whitespace do not match the nonterminal S,
%%   and cannot appear in these positions.
'inv-not-sa14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/not-sa14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional01.xml
%% ID: optional01
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one is required.
optional01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional02.xml
%% ID: optional02
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing two children where one is required.
optional02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional03.xml
%% ID: optional03
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where two are required.
optional03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional04.xml
%% ID: optional04
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing three children where two are required.
optional04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional05.xml
%% ID: optional05
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or two are required (one
%%   construction of that model).
optional05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional06.xml
%% ID: optional06
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or two are required (a
%%   second construction of that model).
optional06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional07.xml
%% ID: optional07
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or two are required (a third
%%   construction of that model).
optional07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional08.xml
%% ID: optional08
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or two are required (a
%%   fourth construction of that model).
optional08(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional09.xml
%% ID: optional09
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or two are required (a fifth
%%   construction of that model).
optional09(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional10.xml
%% ID: optional10
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing three children where one or two are required (a
%%   basic construction of that model).
optional10(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional11.xml
%% ID: optional11
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing three children where one or two are required (a
%%   second construction of that model).
optional11(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional12.xml
%% ID: optional12
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing three children where one or two are required (a
%%   third construction of that model).
optional12(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional13.xml
%% ID: optional13
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing three children where one or two are required (a
%%   fourth construction of that model).
optional13(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional14.xml
%% ID: optional14
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing three children where one or two are required (a
%%   fifth construction of that model).
optional14(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional20.xml
%% ID: optional20
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or more are required (a
%%   sixth construction of that model).
optional20(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional20.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional21.xml
%% ID: optional21
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or more are required (a
%%   seventh construction of that model).
optional21(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional21.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional22.xml
%% ID: optional22
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or more are required (an
%%   eighth construction of that model).
optional22(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional22.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional23.xml
%% ID: optional23
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or more are required (a
%%   ninth construction of that model).
optional23(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional23.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional24.xml
%% ID: optional24
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing no children where one or more are required (a
%%   tenth construction of that model).
optional24(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional24.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/optional25.xml
%% ID: optional25
%% Type: invalid
%% Sections: 3
%% Entities: parameter
%% Description:
%%   Tests the Element Valid VC (clause 2) for one instance of "children"
%%   content model, providing text content where one or more elements are
%%   required.
optional25(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/optional25.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/required00.xml
%% ID: inv-required00
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the Required Attribute VC.
'inv-required00'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/required00.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/required01.xml
%% ID: inv-required01
%% Type: invalid
%% Sections: 3.1 2.10
%% Description:
%%   Tests the Attribute Value Type (declared) VC for the xml:space attribute
'inv-required01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/required01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/required02.xml
%% ID: inv-required02
%% Type: invalid
%% Sections: 3.1 2.12
%% Description:
%%   Tests the Attribute Value Type (declared) VC for the xml:lang attribute
'inv-required02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/required02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/root.xml
%% ID: root
%% Type: invalid
%% Sections: 2.8
%% Entities: parameter
%% Description:
%%   Tests the Root Element Type VC
root(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/root.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr01.xml
%% ID: attr01
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Entity Name" VC for the ENTITY attribute type.
attr01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr02.xml
%% ID: attr02
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Entity Name" VC for the ENTITIES attribute type.
attr02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr03.xml
%% ID: attr03
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Notation Attributes" VC for the NOTATION attribute type, first
%%   clause: value must be one of the ones that's declared.
attr03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr04.xml
%% ID: attr04
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Notation Attributes" VC for the NOTATION attribute type, second
%%   clause: the names in the declaration must all be declared.
attr04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr05.xml
%% ID: attr05
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Name Token" VC for the NMTOKEN attribute type.
attr05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr06.xml
%% ID: attr06
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Name Token" VC for the NMTOKENS attribute type.
attr06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr07.xml
%% ID: attr07
%% Type: invalid
%% Sections: 3.3.1
%% Description:
%%   Tests the "Enumeration" VC by providing a value which wasn't one of the
%%   choices.
attr07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr08.xml
%% ID: attr08
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Fixed Attribute Default" VC by providing the wrong value.
attr08(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr09.xml
%% ID: attr09
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal IDREF value.
attr09(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr10.xml
%% ID: attr10
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal IDREFS
%%   value.
attr10(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr11.xml
%% ID: attr11
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal ENTITY
%%   value.
attr11(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr12.xml
%% ID: attr12
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal ENTITIES
%%   value.
attr12(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr13.xml
%% ID: attr13
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal NMTOKEN
%%   value.
attr13(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr14.xml
%% ID: attr14
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal NMTOKENS
%%   value.
attr14(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr15.xml
%% ID: attr15
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal NOTATIONS
%%   value.
attr15(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/attr16.xml
%% ID: attr16
%% Type: invalid
%% Sections: 3.3.2
%% Description:
%%   Tests the "Attribute Default Legal" VC by providing an illegal enumeration
%%   value.
attr16(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/attr16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/utf16b.xml
%% ID: utf16b
%% Type: invalid
%% Sections: 4.3.3 2.8
%% Description:
%%   Tests reading an invalid "big endian" UTF-16 document
utf16b(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/utf16b.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/utf16l.xml
%% ID: utf16l
%% Type: invalid
%% Sections: 4.3.3 2.8
%% Description:
%%   Tests reading an invalid "little endian" UTF-16 document
utf16l(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/utf16l.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/empty.xml
%% ID: empty
%% Type: invalid
%% Sections: 2.4 2.7 [18] 3
%% Description:
%%   CDATA section containing only white space does not match the nonterminal S,
%%   and cannot appear in these positions.
empty(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","invalid/empty.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/not-sa03.xml
%% ID: not-wf-sa03
%% Type: not-wf
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   Tests the Entity Declared WFC, ensuring that a reference to externally
%%   defined entity causes a well-formedness error.
'not-wf-sa03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/not-sa03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist01.xml
%% ID: attlist01
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's NUTOKEN is not allowed.
attlist01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist02.xml
%% ID: attlist02
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's NUTOKENS attribute type is not allowed.
attlist02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist03.xml
%% ID: attlist03
%% Type: not-wf
%% Sections: 3.3.1 [59]
%% Description:
%%   Comma doesn't separate enumerations, unlike in SGML.
attlist03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist04.xml
%% ID: attlist04
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's NUMBER attribute type is not allowed.
attlist04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist05.xml
%% ID: attlist05
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's NUMBERS attribute type is not allowed.
attlist05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist06.xml
%% ID: attlist06
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's NAME attribute type is not allowed.
attlist06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist07.xml
%% ID: attlist07
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's NAMES attribute type is not allowed.
attlist07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist08.xml
%% ID: attlist08
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's #CURRENT is not allowed.
attlist08(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist09.xml
%% ID: attlist09
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   SGML's #CONREF is not allowed.
attlist09(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist10.xml
%% ID: attlist10
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   Whitespace required between attributes
attlist10(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/attlist11.xml
%% ID: attlist11
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Whitespace required between attributes
attlist11(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/attlist11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/cond01.xml
%% ID: cond01
%% Type: not-wf
%% Sections: 3.4 [61]
%% Entities: parameter
%% Description:
%%   Only INCLUDE and IGNORE are conditional section keywords
cond01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/cond01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/cond02.xml
%% ID: cond02
%% Type: not-wf
%% Sections: 3.4 [61]
%% Entities: parameter
%% Description:
%%   Must have keyword in conditional sections
cond02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/cond02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/content01.xml
%% ID: content01
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   No whitespace before "?" in content model
content01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/content01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/content02.xml
%% ID: content02
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   No whitespace before "*" in content model
content02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/content02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/content03.xml
%% ID: content03
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   No whitespace before "+" in content model
content03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/content03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/decl01.xml
%% ID: decl01
%% Type: not-wf
%% Sections: 4.3.1 [77]
%% Entities: parameter
%% Description:
%%   External entities may not have standalone decls.
decl01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/decl01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd00.xml
%% ID: nwf-dtd00
%% Type: not-wf
%% Sections: 3.2.1 [55]
%% Description:
%%   Comma mandatory in content model
'nwf-dtd00'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd00.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd01.xml
%% ID: nwf-dtd01
%% Type: not-wf
%% Sections: 3.2.1 [55]
%% Description:
%%   Can't mix comma and vertical bar in content models
'nwf-dtd01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd02.xml
%% ID: dtd02
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   PE name immediately after "%"
dtd02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd03.xml
%% ID: dtd03
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   PE name immediately followed by ";"
dtd03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd04.xml
%% ID: dtd04
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   PUBLIC literal must be quoted
dtd04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd05.xml
%% ID: dtd05
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   SYSTEM identifier must be quoted
dtd05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/dtd07.xml
%% ID: dtd07
%% Type: not-wf
%% Sections: 4.3.1 [77]
%% Entities: parameter
%% Description:
%%   Text declarations (which optionally begin any external entity) are required
%%   to have "encoding=...".
dtd07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/dtd07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/element00.xml
%% ID: element00
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   EOF in middle of incomplete ETAG
element00(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/element00.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/element01.xml
%% ID: element01
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   EOF in middle of incomplete ETAG
element01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/element01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/element02.xml
%% ID: element02
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   Illegal markup (<%@ ... %>)
element02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/element02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/element03.xml
%% ID: element03
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   Illegal markup (<% ... %>)
element03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/element03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/element04.xml
%% ID: element04
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   Illegal markup (<!ELEMENT ... >)
element04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/element04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding01.xml
%% ID: encoding01
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Illegal character " " in encoding name
encoding01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding02.xml
%% ID: encoding02
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Illegal character "/" in encoding name
encoding02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding03.xml
%% ID: encoding03
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Illegal character reference in encoding name
encoding03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding04.xml
%% ID: encoding04
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Illegal character ":" in encoding name
encoding04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding05.xml
%% ID: encoding05
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Illegal character "@" in encoding name
encoding05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding06.xml
%% ID: encoding06
%% Type: not-wf
%% Sections: 4.3.3 [81]
%% Description:
%%   Illegal character "+" in encoding name
encoding06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/encoding07.xml
%% ID: encoding07
%% Type: not-wf
%% Sections: 4.3.1 [77]
%% Entities: general
%% Description:
%%   Text declarations (which optionally begin any external entity) are required
%%   to have "encoding=...".
encoding07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/encoding07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/pi.xml
%% ID: pi
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   No space between PI target name and data
pi(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/pi.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/pubid01.xml
%% ID: pubid01
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   Illegal entity ref in public ID
pubid01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/pubid01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/pubid02.xml
%% ID: pubid02
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   Illegal characters in public ID
pubid02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/pubid02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/pubid03.xml
%% ID: pubid03
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   Illegal characters in public ID
pubid03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/pubid03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/pubid04.xml
%% ID: pubid04
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   Illegal characters in public ID
pubid04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/pubid04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/pubid05.xml
%% ID: pubid05
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   SGML-ism: public ID without system ID
pubid05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/pubid05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml01.xml
%% ID: sgml01
%% Type: not-wf
%% Sections: 3 [39]
%% Description:
%%   SGML-ism: omitted end tag for EMPTY content
sgml01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml02.xml
%% ID: sgml02
%% Type: not-wf
%% Sections: 2.8 
%% Description:
%%   XML declaration must be at the very beginning of a document; it"s not a
%%   processing instruction
sgml02(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml03.xml
%% ID: sgml03
%% Type: not-wf
%% Sections: 2.5 [15]
%% Description:
%%   Comments may not contain "--"
sgml03(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml04.xml
%% ID: sgml04
%% Type: not-wf
%% Sections: 3.3 [52]
%% Description:
%%   ATTLIST declarations apply to only one element, unlike SGML
sgml04(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml05.xml
%% ID: sgml05
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   ELEMENT declarations apply to only one element, unlike SGML
sgml05(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml06.xml
%% ID: sgml06
%% Type: not-wf
%% Sections: 3.3 [52]
%% Description:
%%   ATTLIST declarations are never global, unlike in SGML
sgml06(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml07.xml
%% ID: sgml07
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   SGML Tag minimization specifications are not allowed
sgml07(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml08.xml
%% ID: sgml08
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   SGML Tag minimization specifications are not allowed
sgml08(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml09.xml
%% ID: sgml09
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   SGML Content model exception specifications are not allowed
sgml09(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml10.xml
%% ID: sgml10
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   SGML Content model exception specifications are not allowed
sgml10(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml11.xml
%% ID: sgml11
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   CDATA is not a valid content model spec
sgml11(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml12.xml
%% ID: sgml12
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   RCDATA is not a valid content model spec
sgml12(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/sgml13.xml
%% ID: sgml13
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   SGML Unordered content models not allowed
sgml13(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/sgml13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/uri01.xml
%% ID: uri01
%% Type: error
%% Sections: 4.2.2 [75]
%% Description:
%%   SYSTEM ids may not have URI fragments
uri01(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"sun","not-wf/uri01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Cases
%% Profile: OASIS/NIST XML 1.0 Tests
%%----------------------------------------------------------------------
 
%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01pass2.xml
%% ID: o-p01pass2
%% Type: valid
%% Sections: 2.2 [1]
%% Description:
%%   various Misc items where they can occur
'o-p01pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p06pass1.xml
%% ID: o-p06pass1
%% Type: valid
%% Sections: 2.3 [6]
%% Description:
%%   various satisfactions of the Names production in a NAMES attribute
'o-p06pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p06pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p07pass1.xml
%% ID: o-p07pass1
%% Type: valid
%% Sections: 2.3 [7]
%% Description:
%%   various valid Nmtoken 's in an attribute list declaration.
'o-p07pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p07pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p08pass1.xml
%% ID: o-p08pass1
%% Type: valid
%% Sections: 2.3 [8]
%% Description:
%%   various satisfaction of an NMTOKENS attribute value.
'o-p08pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p08pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p09pass1.xml
%% ID: o-p09pass1
%% Type: valid
%% Sections: 2.3 [9]
%% Entities: parameter
%% Description:
%%   valid EntityValue's. Except for entity references, markup is not
%%   recognized.
'o-p09pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p09pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12pass1.xml
%% ID: o-p12pass1
%% Type: valid
%% Sections: 2.3 [12]
%% Description:
%%   valid public IDs.
'o-p12pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22pass4.xml
%% ID: o-p22pass4
%% Type: valid
%% Sections: 2.8 [22]
%% Description:
%%   XML decl and doctypedecl
'o-p22pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22pass5.xml
%% ID: o-p22pass5
%% Type: valid
%% Sections: 2.8 [22]
%% Description:
%%   just doctypedecl
'o-p22pass5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22pass5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22pass6.xml
%% ID: o-p22pass6
%% Type: valid
%% Sections: 2.8 [22]
%% Description:
%%   S between decls is not required
'o-p22pass6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22pass6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p28pass1.xml
%% ID: o-p28pass1
%% Type: valid
%% Sections: 3.1 [43] [44]
%% Description:
%%   Empty-element tag must be used for element which are declared EMPTY.
'o-p28pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p28pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p28pass3.xml
%% ID: o-p28pass3
%% Type: valid
%% Sections: 2.8 4.1 [28] [69]
%% Entities: parameter
%% Description:
%%   Valid doctypedecl with Parameter entity reference. The declaration of a
%%   parameter entity must precede any reference to it.
'o-p28pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p28pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p28pass4.xml
%% ID: o-p28pass4
%% Type: valid
%% Sections: 2.8 4.2.2 [28] [75]
%% Entities: parameter
%% Description:
%%   Valid doctypedecl with ExternalID as an External Entity declaration.
'o-p28pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p28pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p28pass5.xml
%% ID: o-p28pass5
%% Type: valid
%% Sections: 2.8 4.1 [28] [69]
%% Entities: parameter
%% Description:
%%   Valid doctypedecl with ExternalID as an External Entity. A parameter entity
%%   reference is also used.
'o-p28pass5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p28pass5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p29pass1.xml
%% ID: o-p29pass1
%% Type: valid
%% Sections: 2.8 [29]
%% Description:
%%   Valid types of markupdecl.
'o-p29pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p29pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p30pass1.xml
%% ID: o-p30pass1
%% Type: valid
%% Sections: 2.8 4.2.2 [30] [75]
%% Entities: parameter
%% Description:
%%   Valid doctypedecl with ExternalID as an External Entity. The external
%%   entity has an element declaration.
'o-p30pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p30pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p30pass2.xml
%% ID: o-p30pass2
%% Type: valid
%% Sections: 2.8 4.2.2 4.3.1 [30] [75] [77]
%% Entities: parameter
%% Description:
%%   Valid doctypedecl with ExternalID as an Enternal Entity. The external
%%   entity begins with a Text Declaration.
'o-p30pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p30pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p31pass1.xml
%% ID: o-p31pass1
%% Type: valid
%% Sections: 2.8 [31]
%% Entities: parameter
%% Description:
%%   external subset can be empty
'o-p31pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p31pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p31pass2.xml
%% ID: o-p31pass2
%% Type: valid
%% Sections: 2.8 3.4 4.2.2 [31] [62] [63] [75]
%% Entities: parameter
%% Description:
%%   Valid doctypedecl with EXternalID as Enternal Entity. The external entity
%%   contains a parameter entity reference and conditional sections.
'o-p31pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p31pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p43pass1.xml
%% ID: o-p43pass1
%% Type: valid
%% Sections: 2.4 2.5 2.6 2.7 [15] [16] [18]
%% Description:
%%   Valid use of character data, comments, processing instructions and CDATA
%%   sections within the start and end tag.
'o-p43pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p43pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p45pass1.xml
%% ID: o-p45pass1
%% Type: valid
%% Sections: 3.2 [45]
%% Description:
%%   valid element declarations
'o-p45pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p45pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46pass1.xml
%% ID: o-p46pass1
%% Type: valid
%% Sections: 3.2 3.2.1 3.2.2 [45] [46] [47] [51]
%% Description:
%%   Valid use of contentspec, element content models, and mixed content within
%%   an element type declaration.
'o-p46pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p47pass1.xml
%% ID: o-p47pass1
%% Type: valid
%% Sections: 3.2 3.2.1 [45] [46] [47] 
%% Description:
%%   Valid use of contentspec, element content models, choices, sequences and
%%   content particles within an element type declaration. The optional
%%   character following a name or list governs the number of times the element
%%   or content particle may appear.
'o-p47pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p47pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p48pass1.xml
%% ID: o-p48pass1
%% Type: valid
%% Sections: 3.2 3.2.1 [45] [46] [47]
%% Description:
%%   Valid use of contentspec, element content models, choices, sequences and
%%   content particles within an element type declaration. The optional
%%   character following a name or list governs the number of times the element
%%   or content particle may appear.
'o-p48pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p48pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p49pass1.xml
%% ID: o-p49pass1
%% Type: valid
%% Sections: 3.2 3.2.1 [45] [46] [47]
%% Description:
%%   Valid use of contentspec, element content models, choices, and content
%%   particles within an element type declaration. The optional character
%%   following a name or list governs the number of times the element or content
%%   particle may appear. Whitespace is also valid between choices.
'o-p49pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p49pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p50pass1.xml
%% ID: o-p50pass1
%% Type: valid
%% Sections: 3.2 3.2.1 [45] [46] [47]
%% Description:
%%   Valid use of contentspec, element content models, sequences and content
%%   particles within an element type declaration. The optional character
%%   following a name or list governs the number of times the element or content
%%   particle may appear. Whitespace is also valid between sequences.
'o-p50pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p50pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51pass1.xml
%% ID: o-p51pass1
%% Type: valid
%% Sections: 3.2.2 [51]
%% Description:
%%   valid Mixed contentspec's.
'o-p51pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p52pass1.xml
%% ID: o-p52pass1
%% Type: valid
%% Sections: 3.3 [52]
%% Description:
%%   valid AttlistDecls: No AttDef's are required, and the terminating S is
%%   optional, multiple ATTLISTS per element are OK, and multiple declarations
%%   of the same attribute are OK.
'o-p52pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p52pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p53pass1.xml
%% ID: o-p53pass1
%% Type: valid
%% Sections: 3.3 [53]
%% Description:
%%   a valid AttDef
'o-p53pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p53pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p54pass1.xml
%% ID: o-p54pass1
%% Type: valid
%% Sections: 3.3.1 [54]
%% Description:
%%   the three kinds of attribute types
'o-p54pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p54pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p55pass1.xml
%% ID: o-p55pass1
%% Type: valid
%% Sections: 3.3.1 [55]
%% Description:
%%   StringType = "CDATA"
'o-p55pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p55pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p56pass1.xml
%% ID: o-p56pass1
%% Type: valid
%% Sections: 3.3.1 [56]
%% Description:
%%   the 7 tokenized attribute types
'o-p56pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p56pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p57pass1.xml
%% ID: o-p57pass1
%% Type: valid
%% Sections: 3.3.1 [57]
%% Description:
%%   enumerated types are NMTOKEN or NOTATION lists
'o-p57pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p57pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58pass1.xml
%% ID: o-p58pass1
%% Type: valid
%% Sections: 3.3.1 [58]
%% Description:
%%   NOTATION enumeration has on or more items
'o-p58pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p59pass1.xml
%% ID: o-p59pass1
%% Type: valid
%% Sections: 3.3.1 [59]
%% Description:
%%   NMTOKEN enumerations haveon or more items
'o-p59pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p59pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p60pass1.xml
%% ID: o-p60pass1
%% Type: valid
%% Sections: 3.3.2 [60]
%% Description:
%%   the four types of default values
'o-p60pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p60pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p61pass1.xml
%% ID: o-p61pass1
%% Type: valid
%% Sections: 3.4 [61]
%% Entities: parameter
%% Description:
%%   valid conditional sections are INCLUDE and IGNORE
'o-p61pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p61pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p62pass1.xml
%% ID: o-p62pass1
%% Type: valid
%% Sections: 3.4 [62]
%% Entities: parameter
%% Description:
%%   valid INCLUDE sections -- options S before and after keyword, sections can
%%   nest
'o-p62pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p62pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p63pass1.xml
%% ID: o-p63pass1
%% Type: valid
%% Sections: 3.4 [63]
%% Entities: parameter
%% Description:
%%   valid IGNORE sections
'o-p63pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p63pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p64pass1.xml
%% ID: o-p64pass1
%% Type: valid
%% Sections: 3.4 [64]
%% Entities: parameter
%% Description:
%%   IGNOREd sections ignore everything except section delimiters
'o-p64pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p64pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p68pass1.xml
%% ID: o-p68pass1
%% Type: valid
%% Sections: 4.1 [68]
%% Description:
%%   Valid entity references. Also ensures that a charref to '&' isn't
%%   interpreted as an entity reference open delimiter
'o-p68pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p68pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p69pass1.xml
%% ID: o-p69pass1
%% Type: valid
%% Sections: 4.1 [69]
%% Entities: parameter
%% Description:
%%   Valid PEReferences.
'o-p69pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p69pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p70pass1.xml
%% ID: o-p70pass1
%% Type: valid
%% Sections: 4.2 [70]
%% Description:
%%   An EntityDecl is either a GEDecl or a PEDecl
'o-p70pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p70pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p71pass1.xml
%% ID: o-p71pass1
%% Type: valid
%% Sections: 4.2 [71]
%% Description:
%%   Valid GEDecls
'o-p71pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p71pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p72pass1.xml
%% ID: o-p72pass1
%% Type: valid
%% Sections: 4.2 [72]
%% Description:
%%   Valid PEDecls
'o-p72pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p72pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p73pass1.xml
%% ID: o-p73pass1
%% Type: valid
%% Sections: 4.2 [73]
%% Description:
%%   EntityDef is either Entity value or an external id, with an optional
%%   NDataDecl
'o-p73pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p73pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p76pass1.xml
%% ID: o-p76pass1
%% Type: valid
%% Sections: 4.2.2 [76]
%% Description:
%%   valid NDataDecls
'o-p76pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p76pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01pass1.xml
%% ID: o-p01pass1
%% Type: invalid
%% Sections: 2.1 [1]
%% Description:
%%   no prolog
'o-p01pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01pass3.xml
%% ID: o-p01pass3
%% Type: invalid
%% Sections: 2.1 [1]
%% Description:
%%   Misc items after the document
'o-p01pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03pass1.xml
%% ID: o-p03pass1
%% Type: invalid
%% Sections: 2.3 [3]
%% Description:
%%   all valid S characters
'o-p03pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p04pass1.xml
%% ID: o-p04pass1
%% Type: invalid
%% Sections: 2.3 [4]
%% Description:
%%   names with all valid ASCII characters, and one from each other class in
%%   NameChar
'o-p04pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p04pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p05pass1.xml
%% ID: o-p05pass1
%% Type: invalid
%% Sections: 2.3 [5]
%% Description:
%%   various valid Name constructions
'o-p05pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p05pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p06fail1.xml
%% ID: o-p06fail1
%% Type: invalid
%% Sections: 2.3 [6]
%% Description:
%%   Requires at least one name.
'o-p06fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p06fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p08fail1.xml
%% ID: o-p08fail1
%% Type: invalid
%% Sections: 2.3 [8]
%% Description:
%%   at least one Nmtoken is required.
'o-p08fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p08fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p08fail2.xml
%% ID: o-p08fail2
%% Type: invalid
%% Sections: 2.3 [8]
%% Description:
%%   an invalid Nmtoken character.
'o-p08fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p08fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p10pass1.xml
%% ID: o-p10pass1
%% Type: invalid
%% Sections: 2.3 [10]
%% Description:
%%   valid attribute values
'o-p10pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p10pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p14pass1.xml
%% ID: o-p14pass1
%% Type: invalid
%% Sections: 2.4 [14]
%% Description:
%%   valid CharData
'o-p14pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p14pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p15pass1.xml
%% ID: o-p15pass1
%% Type: invalid
%% Sections: 2.5 [15]
%% Description:
%%   valid comments
'o-p15pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p15pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p16pass1.xml
%% ID: o-p16pass1
%% Type: invalid
%% Sections: 2.6 [16] [17]
%% Description:
%%   Valid form of Processing Instruction. Shows that whitespace character data
%%   is valid before end of processing instruction.
'o-p16pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p16pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p16pass2.xml
%% ID: o-p16pass2
%% Type: invalid
%% Sections: 2.6 [16]
%% Description:
%%   Valid form of Processing Instruction. Shows that whitespace character data
%%   is valid before end of processing instruction.
'o-p16pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p16pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p16pass3.xml
%% ID: o-p16pass3
%% Type: invalid
%% Sections: 2.6 [16]
%% Description:
%%   Valid form of Processing Instruction. Shows that whitespace character data
%%   is valid before end of processing instruction.
'o-p16pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p16pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p18pass1.xml
%% ID: o-p18pass1
%% Type: invalid
%% Sections: 2.7 [18]
%% Description:
%%   valid CDSect's. Note that a CDStart in a CDSect is not recognized as such
'o-p18pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p18pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22pass1.xml
%% ID: o-p22pass1
%% Type: invalid
%% Sections: 2.8 [22]
%% Description:
%%   prolog can be empty
'o-p22pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22pass2.xml
%% ID: o-p22pass2
%% Type: invalid
%% Sections: 2.8 [22]
%% Description:
%%   XML declaration only
'o-p22pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22pass3.xml
%% ID: o-p22pass3
%% Type: invalid
%% Sections: 2.8 [22]
%% Description:
%%   XML decl and Misc
'o-p22pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23pass1.xml
%% ID: o-p23pass1
%% Type: invalid
%% Sections: 2.8 [23]
%% Description:
%%   Test shows a valid XML declaration along with version info.
'o-p23pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23pass2.xml
%% ID: o-p23pass2
%% Type: invalid
%% Sections: 2.8 [23]
%% Description:
%%   Test shows a valid XML declaration along with encoding declaration.
'o-p23pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23pass3.xml
%% ID: o-p23pass3
%% Type: invalid
%% Sections: 2.8 [23]
%% Description:
%%   Test shows a valid XML declaration along with Standalone Document
%%   Declaration.
'o-p23pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23pass4.xml
%% ID: o-p23pass4
%% Type: invalid
%% Sections: 2.8 [23]
%% Description:
%%   Test shows a valid XML declaration, encoding declarationand Standalone
%%   Document Declaration.
'o-p23pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p24pass1.xml
%% ID: o-p24pass1
%% Type: invalid
%% Sections: 2.8 [24]
%% Description:
%%   Test shows a prolog that has the VersionInfo delimited by double quotes.
'o-p24pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p24pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p24pass2.xml
%% ID: o-p24pass2
%% Type: invalid
%% Sections: 2.8 [24]
%% Description:
%%   Test shows a prolog that has the VersionInfo delimited by single quotes.
'o-p24pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p24pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p24pass3.xml
%% ID: o-p24pass3
%% Type: invalid
%% Sections: 2.8 [24]
%% Description:
%%   Test shows whitespace is allowed in prolog before version info.
'o-p24pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p24pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p24pass4.xml
%% ID: o-p24pass4
%% Type: invalid
%% Sections: 2.8 [24]
%% Description:
%%   Test shows whitespace is allowed in prolog on both sides of equal sign.
'o-p24pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p24pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p25pass1.xml
%% ID: o-p25pass1
%% Type: invalid
%% Sections: 2.8 [25]
%% Description:
%%   Test shows whitespace is NOT necessary before or after equal sign of
%%   versioninfo.
'o-p25pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p25pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p25pass2.xml
%% ID: o-p25pass2
%% Type: invalid
%% Sections: 2.8 [25]
%% Description:
%%   Test shows whitespace can be used on both sides of equal sign of
%%   versioninfo.
'o-p25pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p25pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p26pass1.xml
%% ID: o-p26pass1
%% Type: invalid
%% Sections: 2.8 [26]
%% Description:
%%   The valid version number. We cannot test others because a 1.0 processor is
%%   allowed to fail them.
'o-p26pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p26pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p27pass1.xml
%% ID: o-p27pass1
%% Type: invalid
%% Sections: 2.8 [27]
%% Description:
%%   Comments are valid as the Misc part of the prolog.
'o-p27pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p27pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p27pass2.xml
%% ID: o-p27pass2
%% Type: invalid
%% Sections: 2.8 [27]
%% Description:
%%   Processing Instructions are valid as the Misc part of the prolog.
'o-p27pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p27pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p27pass3.xml
%% ID: o-p27pass3
%% Type: invalid
%% Sections: 2.8 [27]
%% Description:
%%   Whitespace is valid as the Misc part of the prolog.
'o-p27pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p27pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p27pass4.xml
%% ID: o-p27pass4
%% Type: invalid
%% Sections: 2.8 [27]
%% Description:
%%   A combination of comments, whitespaces and processing instructions are
%%   valid as the Misc part of the prolog.
'o-p27pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p27pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32pass1.xml
%% ID: o-p32pass1
%% Type: invalid
%% Sections: 2.9 [32]
%% Description:
%%   Double quotes can be used as delimiters for the value of a Standalone
%%   Document Declaration.
'o-p32pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32pass2.xml
%% ID: o-p32pass2
%% Type: invalid
%% Sections: 2.9 [32]
%% Description:
%%   Single quotes can be used as delimiters for the value of a Standalone
%%   Document Declaration.
'o-p32pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39pass1.xml
%% ID: o-p39pass1
%% Type: invalid
%% Sections: 3 3.1 [39] [44]
%% Description:
%%   Empty element tag may be used for any element which has no content.
'o-p39pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39pass2.xml
%% ID: o-p39pass2
%% Type: invalid
%% Sections: 3 3.1 [39] [43]
%% Description:
%%   Character data is valid element content.
'o-p39pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40pass1.xml
%% ID: o-p40pass1
%% Type: invalid
%% Sections: 3.1 [40]
%% Description:
%%   Elements content can be empty.
'o-p40pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40pass2.xml
%% ID: o-p40pass2
%% Type: invalid
%% Sections: 3.1 [40]
%% Description:
%%   Whitespace is valid within a Start-tag.
'o-p40pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40pass3.xml
%% ID: o-p40pass3
%% Type: invalid
%% Sections: 3.1 [40] [41]
%% Description:
%%   Attributes are valid within a Start-tag.
'o-p40pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40pass4.xml
%% ID: o-p40pass4
%% Type: invalid
%% Sections: 3.1 [40]
%% Description:
%%   Whitespace and Multiple Attributes are valid within a Start-tag.
'o-p40pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p41pass1.xml
%% ID: o-p41pass1
%% Type: invalid
%% Sections: 3.1 [41]
%% Description:
%%   Attributes are valid within a Start-tag.
'o-p41pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p41pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p41pass2.xml
%% ID: o-p41pass2
%% Type: invalid
%% Sections: 3.1 [41]
%% Description:
%%   Whitespace is valid within a Start-tags Attribute.
'o-p41pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p41pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p42pass1.xml
%% ID: o-p42pass1
%% Type: invalid
%% Sections: 3.1 [42]
%% Description:
%%   Test shows proper syntax for an End-tag.
'o-p42pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p42pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p42pass2.xml
%% ID: o-p42pass2
%% Type: invalid
%% Sections: 3.1 [42]
%% Description:
%%   Whitespace is valid after name in End-tag.
'o-p42pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p42pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44pass1.xml
%% ID: o-p44pass1
%% Type: invalid
%% Sections: 3.1 [44]
%% Description:
%%   Valid display of an Empty Element Tag.
'o-p44pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44pass2.xml
%% ID: o-p44pass2
%% Type: invalid
%% Sections: 3.1 [44]
%% Description:
%%   Empty Element Tags can contain an Attribute.
'o-p44pass2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44pass2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44pass3.xml
%% ID: o-p44pass3
%% Type: invalid
%% Sections: 3.1 [44]
%% Description:
%%   Whitespace is valid in an Empty Element Tag following the end of the
%%   attribute value.
'o-p44pass3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44pass3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44pass4.xml
%% ID: o-p44pass4
%% Type: invalid
%% Sections: 3.1 [44]
%% Description:
%%   Whitespace is valid after the name in an Empty Element Tag.
'o-p44pass4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44pass4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44pass5.xml
%% ID: o-p44pass5
%% Type: invalid
%% Sections: 3.1 [44]
%% Description:
%%   Whitespace and Multiple Attributes are valid in an Empty Element Tag.
'o-p44pass5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44pass5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66pass1.xml
%% ID: o-p66pass1
%% Type: invalid
%% Sections: 4.1 [66]
%% Description:
%%   valid character references
'o-p66pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p74pass1.xml
%% ID: o-p74pass1
%% Type: invalid
%% Sections: 4.2 [74]
%% Description:
%%   PEDef is either an entity value or an external id
'o-p74pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p74pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75pass1.xml
%% ID: o-p75pass1
%% Type: invalid
%% Sections: 4.2.2 [75]
%% Description:
%%   valid external identifiers
'o-p75pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: e2.xml
%% ID: o-e2
%% Type: invalid
%% Sections: 3.3.1 [58] [59] Errata [E2]
%% Description:
%%   Validity Constraint: No duplicate tokens
'o-e2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","e2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01fail1.xml
%% ID: o-p01fail1
%% Type: not-wf
%% Sections: 2.1 [1]
%% Description:
%%   S cannot occur before the prolog
'o-p01fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01fail2.xml
%% ID: o-p01fail2
%% Type: not-wf
%% Sections: 2.1 [1]
%% Description:
%%   comments cannot occur before the prolog
'o-p01fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01fail3.xml
%% ID: o-p01fail3
%% Type: not-wf
%% Sections: 2.1 [1]
%% Description:
%%   only one document element
'o-p01fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p01fail4.xml
%% ID: o-p01fail4
%% Type: not-wf
%% Sections: 2.1 [1]
%% Description:
%%   document element must be complete.
'o-p01fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p01fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail1.xml
%% ID: o-p02fail1
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail10.xml
%% ID: o-p02fail10
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail11.xml
%% ID: o-p02fail11
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail12.xml
%% ID: o-p02fail12
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail13.xml
%% ID: o-p02fail13
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail14.xml
%% ID: o-p02fail14
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail15.xml
%% ID: o-p02fail15
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail16.xml
%% ID: o-p02fail16
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail17.xml
%% ID: o-p02fail17
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail17.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail18.xml
%% ID: o-p02fail18
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail18.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail19.xml
%% ID: o-p02fail19
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail19'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail19.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail2.xml
%% ID: o-p02fail2
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail20.xml
%% ID: o-p02fail20
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail20'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail20.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail21.xml
%% ID: o-p02fail21
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail21'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail21.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail22.xml
%% ID: o-p02fail22
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail22'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail22.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail23.xml
%% ID: o-p02fail23
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail23'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail23.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail24.xml
%% ID: o-p02fail24
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail24'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail24.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail25.xml
%% ID: o-p02fail25
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail25'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail25.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail26.xml
%% ID: o-p02fail26
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail26'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail26.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail27.xml
%% ID: o-p02fail27
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail27'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail27.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail28.xml
%% ID: o-p02fail28
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail28'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail28.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail29.xml
%% ID: o-p02fail29
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail29'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail29.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail3.xml
%% ID: o-p02fail3
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail30.xml
%% ID: o-p02fail30
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail30'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail30.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail31.xml
%% ID: o-p02fail31
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail31'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail31.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail4.xml
%% ID: o-p02fail4
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail5.xml
%% ID: o-p02fail5
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail6.xml
%% ID: o-p02fail6
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail7.xml
%% ID: o-p02fail7
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail7'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail7.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail8.xml
%% ID: o-p02fail8
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail8'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail8.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p02fail9.xml
%% ID: o-p02fail9
%% Type: not-wf
%% Sections: 2.2 [2]
%% Description:
%%   Use of illegal character within XML document.
'o-p02fail9'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p02fail9.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail1.xml
%% ID: o-p03fail1
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail10.xml
%% ID: o-p03fail10
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail11.xml
%% ID: o-p03fail11
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail12.xml
%% ID: o-p03fail12
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail13.xml
%% ID: o-p03fail13
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail14.xml
%% ID: o-p03fail14
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail15.xml
%% ID: o-p03fail15
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail16.xml
%% ID: o-p03fail16
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail17.xml
%% ID: o-p03fail17
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail17.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail18.xml
%% ID: o-p03fail18
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail18.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail19.xml
%% ID: o-p03fail19
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail19'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail19.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail2.xml
%% ID: o-p03fail2
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail20.xml
%% ID: o-p03fail20
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail20'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail20.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail21.xml
%% ID: o-p03fail21
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail21'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail21.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail22.xml
%% ID: o-p03fail22
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail22'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail22.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail23.xml
%% ID: o-p03fail23
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail23'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail23.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail24.xml
%% ID: o-p03fail24
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail24'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail24.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail25.xml
%% ID: o-p03fail25
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail25'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail25.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail26.xml
%% ID: o-p03fail26
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail26'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail26.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail27.xml
%% ID: o-p03fail27
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail27'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail27.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail28.xml
%% ID: o-p03fail28
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail28'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail28.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail29.xml
%% ID: o-p03fail29
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail29'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail29.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail3.xml
%% ID: o-p03fail3
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail4.xml
%% ID: o-p03fail4
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail5.xml
%% ID: o-p03fail5
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail7.xml
%% ID: o-p03fail7
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail7'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail7.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail8.xml
%% ID: o-p03fail8
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail8'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail8.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p03fail9.xml
%% ID: o-p03fail9
%% Type: not-wf
%% Sections: 2.3 [3]
%% Description:
%%   Use of illegal character within XML document.
'o-p03fail9'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p03fail9.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p04fail1.xml
%% ID: o-p04fail1
%% Type: not-wf
%% Sections: 2.3 [4]
%% Description:
%%   Name contains invalid character.
'o-p04fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p04fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p04fail2.xml
%% ID: o-p04fail2
%% Type: not-wf
%% Sections: 2.3 [4]
%% Description:
%%   Name contains invalid character.
'o-p04fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p04fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p04fail3.xml
%% ID: o-p04fail3
%% Type: not-wf
%% Sections: 2.3 [4]
%% Description:
%%   Name contains invalid character.
'o-p04fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p04fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p05fail1.xml
%% ID: o-p05fail1
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   a Name cannot start with a digit
'o-p05fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p05fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p05fail2.xml
%% ID: o-p05fail2
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   a Name cannot start with a '.'
'o-p05fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p05fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p05fail3.xml
%% ID: o-p05fail3
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   a Name cannot start with a "-"
'o-p05fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p05fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p05fail4.xml
%% ID: o-p05fail4
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   a Name cannot start with a CombiningChar
'o-p05fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p05fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p05fail5.xml
%% ID: o-p05fail5
%% Type: not-wf
%% Sections: 2.3 [5]
%% Description:
%%   a Name cannot start with an Extender
'o-p05fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p05fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p09fail1.xml
%% ID: o-p09fail1
%% Type: not-wf
%% Sections: 2.3 [9]
%% Entities: parameter
%% Description:
%%   EntityValue excludes '%'
'o-p09fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p09fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p09fail2.xml
%% ID: o-p09fail2
%% Type: not-wf
%% Sections: 2.3 [9]
%% Entities: parameter
%% Description:
%%   EntityValue excludes '&'
'o-p09fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p09fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p09fail3.xml
%% ID: o-p09fail3
%% Type: not-wf
%% Sections: 2.3 [9]
%% Description:
%%   incomplete character reference
'o-p09fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p09fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p09fail4.xml
%% ID: o-p09fail4
%% Type: not-wf
%% Sections: 2.3 [9]
%% Description:
%%   quote types must match
'o-p09fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p09fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p09fail5.xml
%% ID: o-p09fail5
%% Type: not-wf
%% Sections: 2.3 [9]
%% Description:
%%   quote types must match
'o-p09fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p09fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p10fail1.xml
%% ID: o-p10fail1
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   attribute values exclude '<'
'o-p10fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p10fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p10fail2.xml
%% ID: o-p10fail2
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   attribute values exclude '&'
'o-p10fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p10fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p10fail3.xml
%% ID: o-p10fail3
%% Type: not-wf
%% Sections: 2.3 [10]
%% Description:
%%   quote types must match
'o-p10fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p10fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p11fail1.xml
%% ID: o-p11fail1
%% Type: not-wf
%% Sections: 2.3 [11]
%% Description:
%%   quote types must match
'o-p11fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p11fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p11fail2.xml
%% ID: o-p11fail2
%% Type: not-wf
%% Sections: 2.3 [11]
%% Description:
%%   cannot contain delimiting quotes
'o-p11fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p11fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail1.xml
%% ID: o-p12fail1
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   '"' excluded
'o-p12fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail2.xml
%% ID: o-p12fail2
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   '\' excluded
'o-p12fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail3.xml
%% ID: o-p12fail3
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   entity references excluded
'o-p12fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail4.xml
%% ID: o-p12fail4
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   '>' excluded
'o-p12fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail5.xml
%% ID: o-p12fail5
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   '<' excluded
'o-p12fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail6.xml
%% ID: o-p12fail6
%% Type: not-wf
%% Sections: 2.3 [12]
%% Description:
%%   built-in entity refs excluded
'o-p12fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p12fail7.xml
%% ID: o-p12fail7
%% Type: not-wf
%% Sections: 2.3 [13]
%% Description:
%%   The public ID has a tab character, which is disallowed
'o-p12fail7'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p12fail7.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p14fail1.xml
%% ID: o-p14fail1
%% Type: not-wf
%% Sections: 2.4 [14]
%% Description:
%%   '<' excluded
'o-p14fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p14fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p14fail2.xml
%% ID: o-p14fail2
%% Type: not-wf
%% Sections: 2.4 [14]
%% Description:
%%   '&' excluded
'o-p14fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p14fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p14fail3.xml
%% ID: o-p14fail3
%% Type: not-wf
%% Sections: 2.4 [14]
%% Description:
%%   "]]>" excluded
'o-p14fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p14fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p15fail1.xml
%% ID: o-p15fail1
%% Type: not-wf
%% Sections: 2.5 [15]
%% Description:
%%   comments can't end in '-'
'o-p15fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p15fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p15fail2.xml
%% ID: o-p15fail2
%% Type: not-wf
%% Sections: 2.5 [15]
%% Description:
%%   one comment per comment (contrasted with SGML)
'o-p15fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p15fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p15fail3.xml
%% ID: o-p15fail3
%% Type: not-wf
%% Sections: 2.5 [15]
%% Description:
%%   can't include 2 or more adjacent '-'s
'o-p15fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p15fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p16fail1.xml
%% ID: o-p16fail1
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   "xml" is an invalid PITarget
'o-p16fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p16fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p16fail2.xml
%% ID: o-p16fail2
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   a PITarget must be present
'o-p16fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p16fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p16fail3.xml
%% ID: o-p16fail3
%% Type: not-wf
%% Sections: 2.6 [16]
%% Description:
%%   S after PITarget is required
'o-p16fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p16fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p18fail1.xml
%% ID: o-p18fail1
%% Type: not-wf
%% Sections: 2.7 [18]
%% Description:
%%   no space before "CDATA"
'o-p18fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p18fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p18fail2.xml
%% ID: o-p18fail2
%% Type: not-wf
%% Sections: 2.7 [18]
%% Description:
%%   no space after "CDATA"
'o-p18fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p18fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p18fail3.xml
%% ID: o-p18fail3
%% Type: not-wf
%% Sections: 2.7 [18]
%% Description:
%%   CDSect's can't nest
'o-p18fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p18fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22fail1.xml
%% ID: o-p22fail1
%% Type: not-wf
%% Sections: 2.8 [22]
%% Description:
%%   prolog must start with XML decl
'o-p22fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p22fail2.xml
%% ID: o-p22fail2
%% Type: not-wf
%% Sections: 2.8 [22]
%% Description:
%%   prolog must start with XML decl
'o-p22fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p22fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23fail1.xml
%% ID: o-p23fail1
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   "xml" must be lower-case
'o-p23fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23fail2.xml
%% ID: o-p23fail2
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   VersionInfo must be supplied
'o-p23fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23fail3.xml
%% ID: o-p23fail3
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   VersionInfo must come first
'o-p23fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23fail4.xml
%% ID: o-p23fail4
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   SDDecl must come last
'o-p23fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p23fail5.xml
%% ID: o-p23fail5
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   no SGML-type PIs
'o-p23fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p23fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p24fail1.xml
%% ID: o-p24fail1
%% Type: not-wf
%% Sections: 2.8 [24]
%% Description:
%%   quote types must match
'o-p24fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p24fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p24fail2.xml
%% ID: o-p24fail2
%% Type: not-wf
%% Sections: 2.8 [24]
%% Description:
%%   quote types must match
'o-p24fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p24fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p25fail1.xml
%% ID: o-p25fail1
%% Type: not-wf
%% Sections: 2.8 [25]
%% Description:
%%   Comment is illegal in VersionInfo.
'o-p25fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p25fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p26fail1.xml
%% ID: o-p26fail1
%% Type: not-wf
%% Sections: 2.8 [26]
%% Description:
%%   Illegal character in VersionNum.
'o-p26fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p26fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p26fail2.xml
%% ID: o-p26fail2
%% Type: not-wf
%% Sections: 2.8 [26]
%% Description:
%%   Illegal character in VersionNum.
'o-p26fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p26fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p27fail1.xml
%% ID: o-p27fail1
%% Type: not-wf
%% Sections: 2.8 [27]
%% Description:
%%   References aren't allowed in Misc, even if they would resolve to valid
%%   Misc.
'o-p27fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p27fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p28fail1.xml
%% ID: o-p28fail1
%% Type: not-wf
%% Sections: 2.8 [28]
%% Description:
%%   only declarations in DTD.
'o-p28fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p28fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p29fail1.xml
%% ID: o-p29fail1
%% Type: not-wf
%% Sections: 2.8 [29]
%% Description:
%%   A processor must not pass unknown declaration types.
'o-p29fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p29fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p30fail1.xml
%% ID: o-p30fail1
%% Type: not-wf
%% Sections: 2.8 [30]
%% Entities: parameter
%% Description:
%%   An XML declaration is not the same as a TextDecl
'o-p30fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p30fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p31fail1.xml
%% ID: o-p31fail1
%% Type: not-wf
%% Sections: 2.8 [31]
%% Entities: parameter
%% Description:
%%   external subset excludes doctypedecl
'o-p31fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p31fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32fail1.xml
%% ID: o-p32fail1
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   quote types must match
'o-p32fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32fail2.xml
%% ID: o-p32fail2
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   quote types must match
'o-p32fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32fail3.xml
%% ID: o-p32fail3
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   initial S is required
'o-p32fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32fail4.xml
%% ID: o-p32fail4
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   quotes are required
'o-p32fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p32fail5.xml
%% ID: o-p32fail5
%% Type: not-wf
%% Sections: 2.9 [32]
%% Description:
%%   yes or no must be lower case
'o-p32fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p32fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39fail1.xml
%% ID: o-p39fail1
%% Type: not-wf
%% Sections: 3 [39]
%% Description:
%%   start-tag requires end-tag
'o-p39fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39fail2.xml
%% ID: o-p39fail2
%% Type: not-wf
%% Sections: 3 [39]
%% Description:
%%   end-tag requires start-tag
'o-p39fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39fail3.xml
%% ID: o-p39fail3
%% Type: not-wf
%% Sections: 3 [39]
%% Description:
%%   XML documents contain one or more elements
'o-p39fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39fail4.xml
%% ID: o-p39fail4
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   XML declarations must be correctly terminated
'o-p39fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p39fail5.xml
%% ID: o-p39fail5
%% Type: not-wf
%% Sections: 2.8 [23]
%% Description:
%%   XML declarations must be correctly terminated
'o-p39fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p39fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40fail1.xml
%% ID: o-p40fail1
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   S is required between attributes
'o-p40fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40fail2.xml
%% ID: o-p40fail2
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   tags start with names, not nmtokens
'o-p40fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40fail3.xml
%% ID: o-p40fail3
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   tags start with names, not nmtokens
'o-p40fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p40fail4.xml
%% ID: o-p40fail4
%% Type: not-wf
%% Sections: 3.1 [40]
%% Description:
%%   no space before name
'o-p40fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p40fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p41fail1.xml
%% ID: o-p41fail1
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   quotes are required (contrast with SGML)
'o-p41fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p41fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p41fail2.xml
%% ID: o-p41fail2
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   attribute name is required (contrast with SGML)
'o-p41fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p41fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p41fail3.xml
%% ID: o-p41fail3
%% Type: not-wf
%% Sections: 3.1 [41]
%% Description:
%%   Eq required
'o-p41fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p41fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p42fail1.xml
%% ID: o-p42fail1
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   no space before name
'o-p42fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p42fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p42fail2.xml
%% ID: o-p42fail2
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   cannot end with "/>"
'o-p42fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p42fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p42fail3.xml
%% ID: o-p42fail3
%% Type: not-wf
%% Sections: 3.1 [42]
%% Description:
%%   no NET (contrast with SGML)
'o-p42fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p42fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p43fail1.xml
%% ID: o-p43fail1
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   no non-comment declarations
'o-p43fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p43fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p43fail2.xml
%% ID: o-p43fail2
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   no conditional sections
'o-p43fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p43fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p43fail3.xml
%% ID: o-p43fail3
%% Type: not-wf
%% Sections: 3.1 [43]
%% Description:
%%   no conditional sections
'o-p43fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p43fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44fail1.xml
%% ID: o-p44fail1
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Illegal space before Empty element tag.
'o-p44fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44fail2.xml
%% ID: o-p44fail2
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Illegal space after Empty element tag.
'o-p44fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44fail3.xml
%% ID: o-p44fail3
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Illegal comment in Empty element tag.
'o-p44fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44fail4.xml
%% ID: o-p44fail4
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Whitespace required between attributes.
'o-p44fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p44fail5.xml
%% ID: o-p44fail5
%% Type: not-wf
%% Sections: 3.1 [44]
%% Description:
%%   Duplicate attribute name is illegal.
'o-p44fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p44fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p45fail1.xml
%% ID: o-p45fail1
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   ELEMENT must be upper case.
'o-p45fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p45fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p45fail2.xml
%% ID: o-p45fail2
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   S before contentspec is required.
'o-p45fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p45fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p45fail3.xml
%% ID: o-p45fail3
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   only one content spec
'o-p45fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p45fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p45fail4.xml
%% ID: o-p45fail4
%% Type: not-wf
%% Sections: 3.2 [45]
%% Description:
%%   no comments in declarations (contrast with SGML)
'o-p45fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p45fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46fail1.xml
%% ID: o-p46fail1
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   no parens on declared content
'o-p46fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46fail2.xml
%% ID: o-p46fail2
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   no inclusions (contrast with SGML)
'o-p46fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46fail3.xml
%% ID: o-p46fail3
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   no exclusions (contrast with SGML)
'o-p46fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46fail4.xml
%% ID: o-p46fail4
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   no space before occurrence
'o-p46fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46fail5.xml
%% ID: o-p46fail5
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   single group
'o-p46fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p46fail6.xml
%% ID: o-p46fail6
%% Type: not-wf
%% Sections: 3.2 [46]
%% Description:
%%   can't be both declared and modeled
'o-p46fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p46fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p47fail1.xml
%% ID: o-p47fail1
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   Invalid operator '|' must match previous operator ','
'o-p47fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p47fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p47fail2.xml
%% ID: o-p47fail2
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   Illegal character '-' in Element-content model
'o-p47fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p47fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p47fail3.xml
%% ID: o-p47fail3
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   Optional character must follow a name or list
'o-p47fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p47fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p47fail4.xml
%% ID: o-p47fail4
%% Type: not-wf
%% Sections: 3.2.1 [47]
%% Description:
%%   Illegal space before optional character
'o-p47fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p47fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p48fail1.xml
%% ID: o-p48fail1
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   Illegal space before optional character
'o-p48fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p48fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p48fail2.xml
%% ID: o-p48fail2
%% Type: not-wf
%% Sections: 3.2.1 [48]
%% Description:
%%   Illegal space before optional character
'o-p48fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p48fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p49fail1.xml
%% ID: o-p49fail1
%% Type: not-wf
%% Sections: 3.2.1 [49]
%% Description:
%%   connectors must match
'o-p49fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p49fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p50fail1.xml
%% ID: o-p50fail1
%% Type: not-wf
%% Sections: 3.2.1 [50]
%% Description:
%%   connectors must match
'o-p50fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p50fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail1.xml
%% ID: o-p51fail1
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   occurrence on #PCDATA group must be *
'o-p51fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail2.xml
%% ID: o-p51fail2
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   occurrence on #PCDATA group must be *
'o-p51fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail3.xml
%% ID: o-p51fail3
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   #PCDATA must come first
'o-p51fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail4.xml
%% ID: o-p51fail4
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   occurrence on #PCDATA group must be *
'o-p51fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail5.xml
%% ID: o-p51fail5
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   only '|' connectors
'o-p51fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail6.xml
%% ID: o-p51fail6
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   Only '|' connectors and occurrence on #PCDATA group must be *
'o-p51fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p51fail7.xml
%% ID: o-p51fail7
%% Type: not-wf
%% Sections: 3.2.2 [51]
%% Description:
%%   no nested groups
'o-p51fail7'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p51fail7.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p52fail1.xml
%% ID: o-p52fail1
%% Type: not-wf
%% Sections: 3.3 [52]
%% Description:
%%   A name is required
'o-p52fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p52fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p52fail2.xml
%% ID: o-p52fail2
%% Type: not-wf
%% Sections: 3.3 [52]
%% Description:
%%   A name is required
'o-p52fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p52fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p53fail1.xml
%% ID: o-p53fail1
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   S is required before default
'o-p53fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p53fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p53fail2.xml
%% ID: o-p53fail2
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   S is required before type
'o-p53fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p53fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p53fail3.xml
%% ID: o-p53fail3
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   type is required
'o-p53fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p53fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p53fail4.xml
%% ID: o-p53fail4
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   default is required
'o-p53fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p53fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p53fail5.xml
%% ID: o-p53fail5
%% Type: not-wf
%% Sections: 3.3 [53]
%% Description:
%%   name is required
'o-p53fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p53fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p54fail1.xml
%% ID: o-p54fail1
%% Type: not-wf
%% Sections: 3.3.1 [54]
%% Description:
%%   don't pass unknown attribute types
'o-p54fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p54fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p55fail1.xml
%% ID: o-p55fail1
%% Type: not-wf
%% Sections: 3.3.1 [55]
%% Description:
%%   must be upper case
'o-p55fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p55fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p56fail1.xml
%% ID: o-p56fail1
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   no IDS type
'o-p56fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p56fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p56fail2.xml
%% ID: o-p56fail2
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   no NUMBER type
'o-p56fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p56fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p56fail3.xml
%% ID: o-p56fail3
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   no NAME type
'o-p56fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p56fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p56fail4.xml
%% ID: o-p56fail4
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   no ENTITIES type - types must be upper case
'o-p56fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p56fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p56fail5.xml
%% ID: o-p56fail5
%% Type: not-wf
%% Sections: 3.3.1 [56]
%% Description:
%%   types must be upper case
'o-p56fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p56fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p57fail1.xml
%% ID: o-p57fail1
%% Type: not-wf
%% Sections: 3.3.1 [57]
%% Description:
%%   no keyword for NMTOKEN enumeration
'o-p57fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p57fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail1.xml
%% ID: o-p58fail1
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   at least one value required
'o-p58fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail2.xml
%% ID: o-p58fail2
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   separator must be '|'
'o-p58fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail3.xml
%% ID: o-p58fail3
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   notations are NAMEs, not NMTOKENs -- note: Leaving the invalid notation
%%   undeclared would cause a validating parser to fail without checking the
%%   name syntax, so the notation is declared with an invalid name. A parser
%%   that reports error positions should report an error at the AttlistDecl on
%%   line 6, before reaching the notation declaration.
'o-p58fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail4.xml
%% ID: o-p58fail4
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   NOTATION must be upper case
'o-p58fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail5.xml
%% ID: o-p58fail5
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   S after keyword is required
'o-p58fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail6.xml
%% ID: o-p58fail6
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   parentheses are require
'o-p58fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail7.xml
%% ID: o-p58fail7
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   values are unquoted
'o-p58fail7'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail7.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p58fail8.xml
%% ID: o-p58fail8
%% Type: not-wf
%% Sections: 3.3.1 [58]
%% Description:
%%   values are unquoted
'o-p58fail8'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p58fail8.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p59fail1.xml
%% ID: o-p59fail1
%% Type: not-wf
%% Sections: 3.3.1 [59]
%% Description:
%%   at least one required
'o-p59fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p59fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p59fail2.xml
%% ID: o-p59fail2
%% Type: not-wf
%% Sections: 3.3.1 [59]
%% Description:
%%   separator must be ","
'o-p59fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p59fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p59fail3.xml
%% ID: o-p59fail3
%% Type: not-wf
%% Sections: 3.3.1 [59]
%% Description:
%%   values are unquoted
'o-p59fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p59fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p60fail1.xml
%% ID: o-p60fail1
%% Type: not-wf
%% Sections: 3.3.2 [60]
%% Description:
%%   keywords must be upper case
'o-p60fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p60fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p60fail2.xml
%% ID: o-p60fail2
%% Type: not-wf
%% Sections: 3.3.2 [60]
%% Description:
%%   S is required after #FIXED
'o-p60fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p60fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p60fail3.xml
%% ID: o-p60fail3
%% Type: not-wf
%% Sections: 3.3.2 [60]
%% Description:
%%   only #FIXED has both keyword and value
'o-p60fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p60fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p60fail4.xml
%% ID: o-p60fail4
%% Type: not-wf
%% Sections: 3.3.2 [60]
%% Description:
%%   #FIXED required value
'o-p60fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p60fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p60fail5.xml
%% ID: o-p60fail5
%% Type: not-wf
%% Sections: 3.3.2 [60]
%% Description:
%%   only one default type
'o-p60fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p60fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p61fail1.xml
%% ID: o-p61fail1
%% Type: not-wf
%% Sections: 3.4 [61]
%% Entities: parameter
%% Description:
%%   no other types, including TEMP, which is valid in SGML
'o-p61fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p61fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p62fail1.xml
%% ID: o-p62fail1
%% Type: not-wf
%% Sections: 3.4 [62]
%% Entities: parameter
%% Description:
%%   INCLUDE must be upper case
'o-p62fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p62fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p62fail2.xml
%% ID: o-p62fail2
%% Type: not-wf
%% Sections: 3.4 [62]
%% Entities: parameter
%% Description:
%%   no spaces in terminating delimiter
'o-p62fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p62fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p63fail1.xml
%% ID: o-p63fail1
%% Type: not-wf
%% Sections: 3.4 [63]
%% Entities: parameter
%% Description:
%%   IGNORE must be upper case
'o-p63fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p63fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p63fail2.xml
%% ID: o-p63fail2
%% Type: not-wf
%% Sections: 3.4 [63]
%% Entities: parameter
%% Description:
%%   delimiters must be balanced
'o-p63fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p63fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p64fail1.xml
%% ID: o-p64fail1
%% Type: not-wf
%% Sections: 3.4 [64]
%% Entities: parameter
%% Description:
%%   section delimiters must balance
'o-p64fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p64fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p64fail2.xml
%% ID: o-p64fail2
%% Type: not-wf
%% Sections: 3.4 [64]
%% Entities: parameter
%% Description:
%%   section delimiters must balance
'o-p64fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p64fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66fail1.xml
%% ID: o-p66fail1
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   terminating ';' is required
'o-p66fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66fail2.xml
%% ID: o-p66fail2
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   no S after '&#'
'o-p66fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66fail3.xml
%% ID: o-p66fail3
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   no hex digits in numeric reference
'o-p66fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66fail4.xml
%% ID: o-p66fail4
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   only hex digits in hex references
'o-p66fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66fail5.xml
%% ID: o-p66fail5
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   no references to non-characters
'o-p66fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p66fail6.xml
%% ID: o-p66fail6
%% Type: not-wf
%% Sections: 4.1 [66]
%% Description:
%%   no references to non-characters
'o-p66fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p66fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p68fail1.xml
%% ID: o-p68fail1
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   terminating ';' is required
'o-p68fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p68fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p68fail2.xml
%% ID: o-p68fail2
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   no S after '&'
'o-p68fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p68fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p68fail3.xml
%% ID: o-p68fail3
%% Type: not-wf
%% Sections: 4.1 [68]
%% Description:
%%   no S before ';'
'o-p68fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p68fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p69fail1.xml
%% ID: o-p69fail1
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   terminating ';' is required
'o-p69fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p69fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p69fail2.xml
%% ID: o-p69fail2
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   no S after '%'
'o-p69fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p69fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p69fail3.xml
%% ID: o-p69fail3
%% Type: not-wf
%% Sections: 4.1 [69]
%% Description:
%%   no S before ';'
'o-p69fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p69fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p70fail1.xml
%% ID: o-p70fail1
%% Type: not-wf
%% Sections: 4.2 [70]
%% Description:
%%   This is neither
'o-p70fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p70fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p71fail1.xml
%% ID: o-p71fail1
%% Type: not-wf
%% Sections: 4.2 [71]
%% Description:
%%   S is required before EntityDef
'o-p71fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p71fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p71fail2.xml
%% ID: o-p71fail2
%% Type: not-wf
%% Sections: 4.2 [71]
%% Description:
%%   Entity name is a Name, not an NMToken
'o-p71fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p71fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p71fail3.xml
%% ID: o-p71fail3
%% Type: not-wf
%% Sections: 4.2 [71]
%% Description:
%%   no S after "<!"
'o-p71fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p71fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p71fail4.xml
%% ID: o-p71fail4
%% Type: not-wf
%% Sections: 4.2 [71]
%% Description:
%%   S is required after "<!ENTITY"
'o-p71fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p71fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p72fail1.xml
%% ID: o-p72fail1
%% Type: not-wf
%% Sections: 4.2 [72]
%% Description:
%%   S is required after "<!ENTITY"
'o-p72fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p72fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p72fail2.xml
%% ID: o-p72fail2
%% Type: not-wf
%% Sections: 4.2 [72]
%% Description:
%%   S is required after '%'
'o-p72fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p72fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p72fail3.xml
%% ID: o-p72fail3
%% Type: not-wf
%% Sections: 4.2 [72]
%% Description:
%%   S is required after name
'o-p72fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p72fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p72fail4.xml
%% ID: o-p72fail4
%% Type: not-wf
%% Sections: 4.2 [72]
%% Description:
%%   Entity name is a name, not an NMToken
'o-p72fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p72fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p73fail1.xml
%% ID: o-p73fail1
%% Type: not-wf
%% Sections: 4.2 [73]
%% Description:
%%   No typed replacement text
'o-p73fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p73fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p73fail2.xml
%% ID: o-p73fail2
%% Type: not-wf
%% Sections: 4.2 [73]
%% Description:
%%   Only one replacement value
'o-p73fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p73fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p73fail3.xml
%% ID: o-p73fail3
%% Type: not-wf
%% Sections: 4.2 [73]
%% Description:
%%   No NDataDecl on replacement text
'o-p73fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p73fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p73fail4.xml
%% ID: o-p73fail4
%% Type: not-wf
%% Sections: 4.2 [73]
%% Description:
%%   Value is required
'o-p73fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p73fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p73fail5.xml
%% ID: o-p73fail5
%% Type: not-wf
%% Sections: 4.2 [73]
%% Description:
%%   No NDataDecl without value
'o-p73fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p73fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p74fail1.xml
%% ID: o-p74fail1
%% Type: not-wf
%% Sections: 4.2 [74]
%% Description:
%%   no NDataDecls on parameter entities
'o-p74fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p74fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p74fail2.xml
%% ID: o-p74fail2
%% Type: not-wf
%% Sections: 4.2 [74]
%% Description:
%%   value is required
'o-p74fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p74fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p74fail3.xml
%% ID: o-p74fail3
%% Type: not-wf
%% Sections: 4.2 [74]
%% Description:
%%   only one value
'o-p74fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p74fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75fail1.xml
%% ID: o-p75fail1
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   S required after "PUBLIC"
'o-p75fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75fail2.xml
%% ID: o-p75fail2
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   S required after "SYSTEM"
'o-p75fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75fail3.xml
%% ID: o-p75fail3
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   S required between literals
'o-p75fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75fail4.xml
%% ID: o-p75fail4
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   "SYSTEM" implies only one literal
'o-p75fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75fail5.xml
%% ID: o-p75fail5
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   only one keyword
'o-p75fail5'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75fail5.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p75fail6.xml
%% ID: o-p75fail6
%% Type: not-wf
%% Sections: 4.2.2 [75]
%% Description:
%%   "PUBLIC" requires two literals (contrast with SGML)
'o-p75fail6'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p75fail6.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p76fail1.xml
%% ID: o-p76fail1
%% Type: not-wf
%% Sections: 4.2.2 [76]
%% Description:
%%   S is required before "NDATA"
'o-p76fail1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p76fail1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p76fail2.xml
%% ID: o-p76fail2
%% Type: not-wf
%% Sections: 4.2.2 [76]
%% Description:
%%   "NDATA" is upper-case
'o-p76fail2'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p76fail2.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p76fail3.xml
%% ID: o-p76fail3
%% Type: not-wf
%% Sections: 4.2.2 [76]
%% Description:
%%   notation name is required
'o-p76fail3'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p76fail3.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p76fail4.xml
%% ID: o-p76fail4
%% Type: not-wf
%% Sections: 4.2.2 [76]
%% Description:
%%   notation names are Names
'o-p76fail4'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p76fail4.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: p11pass1.xml
%% ID: o-p11pass1
%% Type: error
%% Sections: 2.3, 4.2.2 [11]
%% Description:
%%   system literals may not contain URI fragments
'o-p11pass1'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"oasis","p11pass1.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Cases
%% Profile: IBM XML 1.0 Tests
%%----------------------------------------------------------------------
 
%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P28/ibm28i01.xml
%% ID: ibm-invalid-P28-ibm28i01.xml
%% Type: invalid
%% Sections: 2.8
%% Output: invalid/P28/out/ibm28i01.xml
%% Description:
%%   The test violates VC:Root Element Type in P28. The Name in the document
%%   type declaration does not match the element type of the root element.
'ibm-invalid-P28-ibm28i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P28/ibm28i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P28/out/ibm28i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P32/ibm32i01.xml
%% ID: ibm-invalid-P32-ibm32i01.xml
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Output: invalid/P32/out/ibm32i01.xml
%% Description:
%%   This test violates VC: Standalone Document Declaration in P32. The
%%   standalone document declaration has the value yes, BUT there is an external
%%   markup declaration of attributes with default values, and the associated
%%   element appears in the document with specified values for those attributes.
'ibm-invalid-P32-ibm32i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P32/ibm32i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P32/out/ibm32i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P32/ibm32i03.xml
%% ID: ibm-invalid-P32-ibm32i03.xml
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Output: invalid/P32/out/ibm32i03.xml
%% Description:
%%   This test violates VC: Standalone Document Declaration in P32. The
%%   standalone document declaration has the value yes, BUT there is an external
%%   markup declaration of attributes with values that will change if
%%   normalized.
'ibm-invalid-P32-ibm32i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P32/ibm32i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P32/out/ibm32i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P32/ibm32i04.xml
%% ID: ibm-invalid-P32-ibm32i04.xml
%% Type: invalid
%% Sections: 2.9
%% Entities: parameter
%% Output: invalid/P32/out/ibm32i04.xml
%% Description:
%%   This test violates VC: Standalone Document Declaration in P32. The
%%   standalone document declaration has the value yes, BUT there is an external
%%   markup declaration of element with element content, and white space occurs
%%   directly within the mixed content.
'ibm-invalid-P32-ibm32i04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P32/ibm32i04.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P32/out/ibm32i04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P39/ibm39i01.xml
%% ID: ibm-invalid-P39-ibm39i01.xml
%% Type: invalid
%% Sections: 3
%% Output: invalid/P39/out/ibm39i01.xml
%% Description:
%%   This test violates VC: Element Valid in P39. Element a is declared empty in
%%   DTD, but has content in the document.
'ibm-invalid-P39-ibm39i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P39/ibm39i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P39/out/ibm39i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P39/ibm39i02.xml
%% ID: ibm-invalid-P39-ibm39i02.xml
%% Type: invalid
%% Sections: 3
%% Output: invalid/P39/out/ibm39i02.xml
%% Description:
%%   This test violates VC: Element Valid in P39. root is declared only having
%%   element children in DTD, but have text content in the document.
'ibm-invalid-P39-ibm39i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P39/ibm39i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P39/out/ibm39i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P39/ibm39i03.xml
%% ID: ibm-invalid-P39-ibm39i03.xml
%% Type: invalid
%% Sections: 3
%% Output: invalid/P39/out/ibm39i03.xml
%% Description:
%%   This test violates VC: Element Valid in P39. Illegal elements are inserted
%%   in b's content of Mixed type.
'ibm-invalid-P39-ibm39i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P39/ibm39i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P39/out/ibm39i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P39/ibm39i04.xml
%% ID: ibm-invalid-P39-ibm39i04.xml
%% Type: invalid
%% Sections: 3
%% Output: invalid/P39/out/ibm39i04.xml
%% Description:
%%   This test violates VC: Element Valid in P39. Element c has undeclared
%%   element as its content of ANY type
'ibm-invalid-P39-ibm39i04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P39/ibm39i04.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P39/out/ibm39i04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P41/ibm41i01.xml
%% ID: ibm-invalid-P41-ibm41i01.xml
%% Type: invalid
%% Sections: 3.1
%% Output: invalid/P41/out/ibm41i01.xml
%% Description:
%%   This test violates VC: Attribute Value Type in P41. attr1 for Element b is
%%   not declared.
'ibm-invalid-P41-ibm41i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P41/ibm41i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P41/out/ibm41i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P41/ibm41i02.xml
%% ID: ibm-invalid-P41-ibm41i02.xml
%% Type: invalid
%% Sections: 3.1
%% Output: invalid/P41/out/ibm41i02.xml
%% Description:
%%   This test violates VC: Attribute Value Type in P41. attr3 for Element b is
%%   given a value that does not match the declaration in the DTD.
'ibm-invalid-P41-ibm41i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P41/ibm41i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P41/out/ibm41i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P45/ibm45i01.xml
%% ID: ibm-invalid-P45-ibm45i01.xml
%% Type: invalid
%% Sections: 3.2
%% Output: invalid/P45/out/ibm45i01.xml
%% Description:
%%   This test violates VC: Unique Element Type Declaration. Element not_unique
%%   has been declared 3 time in the DTD.
'ibm-invalid-P45-ibm45i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P45/ibm45i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P45/out/ibm45i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P49/ibm49i01.xml
%% ID: ibm-invalid-P49-ibm49i01.xml
%% Type: invalid
%% Sections: 3.2.1
%% Entities: parameter
%% Output: invalid/P49/out/ibm49i01.xml
%% Description:
%%   Violates VC:Proper Group/PE Nesting in P49. Open and close parenthesis for
%%   a choice content model are in different PE replace Texts.
'ibm-invalid-P49-ibm49i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P49/ibm49i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P49/out/ibm49i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P50/ibm50i01.xml
%% ID: ibm-invalid-P50-ibm50i01.xml
%% Type: invalid
%% Sections: 3.2.1
%% Entities: parameter
%% Output: invalid/P50/out/ibm50i01.xml
%% Description:
%%   Violates VC:Proper Group/PE Nesting in P50. Open and close parenthesis for
%%   a seq content model are in different PE replace Texts.
'ibm-invalid-P50-ibm50i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P50/ibm50i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P50/out/ibm50i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P51/ibm51i01.xml
%% ID: ibm-invalid-P51-ibm51i01.xml
%% Type: invalid
%% Sections: 3.2.2
%% Entities: parameter
%% Output: invalid/P51/out/ibm51i01.xml
%% Description:
%%   Violates VC:Proper Group/PE Nesting in P51. Open and close parenthesis for
%%   a Mixed content model are in different PE replace Texts.
'ibm-invalid-P51-ibm51i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P51/ibm51i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P51/out/ibm51i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P51/ibm51i03.xml
%% ID: ibm-invalid-P51-ibm51i03.xml
%% Type: invalid
%% Sections: 3.2.2
%% Output: invalid/P51/out/ibm51i03.xml
%% Description:
%%   Violates VC:No Duplicate Types in P51. Element a appears twice in the Mixed
%%   content model of Element e.
'ibm-invalid-P51-ibm51i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P51/ibm51i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P51/out/ibm51i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i01.xml
%% ID: ibm-invalid-P56-ibm56i01.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i01.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: ID. The value of the
%%   ID attribute "UniqueName" is "@999" which does not meet the Name
%%   production.
'ibm-invalid-P56-ibm56i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i02.xml
%% ID: ibm-invalid-P56-ibm56i02.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i02.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: ID. The two ID
%%   attributes "attr" and "UniqueName" have the same value "Ac999" for the
%%   element "b" and the element "tokenizer".
'ibm-invalid-P56-ibm56i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i03.xml
%% ID: ibm-invalid-P56-ibm56i03.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i03.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: ID Attribute Default.
%%   The "#FIXED" occurs in the DefaultDecl for the ID attribute "UniqueName".
'ibm-invalid-P56-ibm56i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i05.xml
%% ID: ibm-invalid-P56-ibm56i05.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i05.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: ID Attribute Default.
%%   The constant string "BOGUS" occurs in the DefaultDecl for the ID attribute
%%   "UniqueName".
'ibm-invalid-P56-ibm56i05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i05.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i06.xml
%% ID: ibm-invalid-P56-ibm56i06.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i06.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: One ID per Element
%%   Type. The element "a" has two ID attributes "first" and "second".
'ibm-invalid-P56-ibm56i06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i06.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i06.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i07.xml
%% ID: ibm-invalid-P56-ibm56i07.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i07.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: IDREF. The value of
%%   the IDREF attribute "reference" is "@456" which does not meet the Name
%%   production.
'ibm-invalid-P56-ibm56i07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i07.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i07.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i08.xml
%% ID: ibm-invalid-P56-ibm56i08.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i08.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: IDREF. The value of
%%   the IDREF attribute "reference" is "BC456" which does not match the value
%%   assigned to any ID attributes.
'ibm-invalid-P56-ibm56i08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i08.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i08.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i09.xml
%% ID: ibm-invalid-P56-ibm56i09.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i09.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: IDREFS. The value of
%%   the IDREFS attribute "reference" is "AC456 #567" which does not meet the
%%   Names production.
'ibm-invalid-P56-ibm56i09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i09.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i09.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i10.xml
%% ID: ibm-invalid-P56-ibm56i10.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i10.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: IDREFS. The value of
%%   the IDREFS attribute "reference" is "EF456 DE355" which does not match the
%%   values assigned to two ID attributes.
'ibm-invalid-P56-ibm56i10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i10.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i10.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i11.xml
%% ID: ibm-invalid-P56-ibm56i11.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i11.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: Entity Name. The value
%%   of the ENTITY attribute "sun" is "ima ge" which does not meet the Name
%%   production.
'ibm-invalid-P56-ibm56i11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i11.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i11.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i12.xml
%% ID: ibm-invalid-P56-ibm56i12.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i12.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: Entity Name. The value
%%   of the ENTITY attribute "sun" is "notimage" which does not match the name
%%   of any unparsed entity declared.
'ibm-invalid-P56-ibm56i12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i12.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i12.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i13.xml
%% ID: ibm-invalid-P56-ibm56i13.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i13.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: Entity Name. The value
%%   of the ENTITY attribute "sun" is "parsedentity" which matches the name of a
%%   parsed entity instead of an unparsed entity declared.
'ibm-invalid-P56-ibm56i13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i13.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i13.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i14.xml
%% ID: ibm-invalid-P56-ibm56i14.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i14.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: Entity Name. The value
%%   of the ENTITIES attribute "sun" is "#image1 @image" which does not meet the
%%   Names production.
'ibm-invalid-P56-ibm56i14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i14.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i14.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i15.xml
%% ID: ibm-invalid-P56-ibm56i15.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i15.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: ENTITIES. The value of
%%   the ENTITIES attribute "sun" is "image3 image4" which does not match the
%%   names of two unparsed entities declared.
'ibm-invalid-P56-ibm56i15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i15.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i15.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i16.xml
%% ID: ibm-invalid-P56-ibm56i16.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i16.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: ENTITIES. The value of
%%   the ENTITIES attribute "sun" is "parsedentity1 parsedentity2" which matches
%%   the names of two parsed entities instead of two unparsed entities declared.
'ibm-invalid-P56-ibm56i16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i16.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i16.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i17.xml
%% ID: ibm-invalid-P56-ibm56i17.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i17.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: Name Token. The value
%%   of the NMTOKEN attribute "thistoken" is "x : image" which does not meet the
%%   Nmtoken production.
'ibm-invalid-P56-ibm56i17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i17.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i17.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P56/ibm56i18.xml
%% ID: ibm-invalid-P56-ibm56i18.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P56/out/ibm56i18.xml
%% Description:
%%   Tests invalid TokenizedType which is against P56 VC: Name Token. The value
%%   of the NMTOKENS attribute "thistoken" is "@lang y: #country" which does not
%%   meet the Nmtokens production.
'ibm-invalid-P56-ibm56i18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P56/ibm56i18.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P56/out/ibm56i18.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P58/ibm58i01.xml
%% ID: ibm-invalid-P58-ibm58i01.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P58/out/ibm58i01.xml
%% Description:
%%   Tests invalid NotationType which is against P58 VC: Notation Attributes.
%%   The attribute "content-encoding" with value "raw" is not a value from the
%%   list "(base64|uuencode)".
'ibm-invalid-P58-ibm58i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P58/ibm58i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P58/out/ibm58i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P58/ibm58i02.xml
%% ID: ibm-invalid-P58-ibm58i02.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P58/out/ibm58i02.xml
%% Description:
%%   Tests invalid NotationType which is against P58 VC: Notation Attributes.
%%   The attribute "content-encoding" with value "raw" is a value from the list
%%   "(base64|uuencode|raw|ascii)", but "raw" is not a declared notation.
'ibm-invalid-P58-ibm58i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P58/ibm58i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P58/out/ibm58i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P59/ibm59i01.xml
%% ID: ibm-invalid-P59-ibm59i01.xml
%% Type: invalid
%% Sections: 3.3.1
%% Output: invalid/P59/out/ibm59i01.xml
%% Description:
%%   Tests invalid Enumeration which is against P59 VC: Enumeration. The value
%%   of the attribute is "ONE" which matches neither "one" nor "two" as declared
%%   in the Enumeration in the AttDef in the AttlistDecl.
'ibm-invalid-P59-ibm59i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P59/ibm59i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P59/out/ibm59i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P60/ibm60i01.xml
%% ID: ibm-invalid-P60-ibm60i01.xml
%% Type: invalid
%% Sections: 3.3.2
%% Output: invalid/P60/out/ibm60i01.xml
%% Description:
%%   Tests invalid DefaultDecl which is against P60 VC: Required Attribute. The
%%   attribute "chapter" for the element "two" is declared as #REQUIRED in the
%%   DefaultDecl in the AttlistDecl, but the value of this attribute is not
%%   given.
'ibm-invalid-P60-ibm60i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P60/ibm60i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P60/out/ibm60i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P60/ibm60i02.xml
%% ID: ibm-invalid-P60-ibm60i02.xml
%% Type: invalid
%% Sections: 3.3.2
%% Output: invalid/P60/out/ibm60i02.xml
%% Description:
%%   Tests invalid DefaultDecl which is against P60 VC: Fixed Attribute
%%   Default.. The attribute "chapter" for the element "one" is declared as
%%   #FIXED with the given value "Introduction" in the DefaultDecl in the
%%   AttlistDecl, but the value of a instance of this attribute is assigned to
%%   "JavaBeans".
'ibm-invalid-P60-ibm60i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P60/ibm60i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P60/out/ibm60i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P60/ibm60i03.xml
%% ID: ibm-invalid-P60-ibm60i03.xml
%% Type: invalid
%% Sections: 3.3.2
%% Output: invalid/P60/out/ibm60i03.xml
%% Description:
%%   Tests invalid DefaultDecl which is against P60 VC: Attribute Default Legal.
%%   The declared default value "c" is not legal for the type (a|b) in the
%%   AttDef in the AttlistDecl.
'ibm-invalid-P60-ibm60i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P60/ibm60i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P60/out/ibm60i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P60/ibm60i04.xml
%% ID: ibm-invalid-P60-ibm60i04.xml
%% Type: invalid
%% Sections: 3.3.2
%% Output: invalid/P60/out/ibm60i04.xml
%% Description:
%%   Tests invalid DefaultDecl which is against P60 VC: Attribute Default Legal.
%%   The declared default value "@#$" is not legal for the type NMTOKEN the
%%   AttDef in the AttlistDecl.
'ibm-invalid-P60-ibm60i04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P60/ibm60i04.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P60/out/ibm60i04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P68/ibm68i01.xml
%% ID: ibm-invalid-P68-ibm68i01.xml
%% Type: error
%% Sections: 4.1
%% Entities: parameter
%% Output: invalid/P68/out/ibm68i01.xml
%% Description:
%%   Tests invalid EntityRef which is against P68 VC: Entity Declared. The GE
%%   with the name "ge2" is referred in the file ibm68i01.dtd", but not
%%   declared.
'ibm-invalid-P68-ibm68i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P68/ibm68i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P68/out/ibm68i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P68/ibm68i02.xml
%% ID: ibm-invalid-P68-ibm68i02.xml
%% Type: error
%% Sections: 4.1
%% Entities: parameter
%% Output: invalid/P68/out/ibm68i02.xml
%% Description:
%%   Tests invalid EntityRef which is against P68 VC: Entity Declared. The GE
%%   with the name "ge1" is referred before declared in the file ibm68i01.dtd".
'ibm-invalid-P68-ibm68i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P68/ibm68i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P68/out/ibm68i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P68/ibm68i03.xml
%% ID: ibm-invalid-P68-ibm68i03.xml
%% Type: error
%% Sections: 4.1
%% Entities: both
%% Output: invalid/P68/out/ibm68i03.xml
%% Description:
%%   Tests invalid EntityRef which is against P68 VC: Entity Declared. The GE
%%   with the name "ge2" is referred in the file ibm68i03.ent", but not
%%   declared.
'ibm-invalid-P68-ibm68i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P68/ibm68i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P68/out/ibm68i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P68/ibm68i04.xml
%% ID: ibm-invalid-P68-ibm68i04.xml
%% Type: error
%% Sections: 4.1
%% Entities: both
%% Output: invalid/P68/out/ibm68i04.xml
%% Description:
%%   Tests invalid EntityRef which is against P68 VC: Entity Declared. The GE
%%   with the name "ge1" is referred before declared in the file ibm68i04.ent".
'ibm-invalid-P68-ibm68i04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P68/ibm68i04.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P68/out/ibm68i04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P69/ibm69i01.xml
%% ID: ibm-invalid-P69-ibm69i01.xml
%% Type: error
%% Sections: 4.1
%% Entities: parameter
%% Output: invalid/P69/out/ibm69i01.xml
%% Description:
%%   Tests invalid PEReference which is against P69 VC: Entity Declared. The
%%   Name "pe2" in the PEReference in the file ibm69i01.dtd does not match the
%%   Name of any declared PE.
'ibm-invalid-P69-ibm69i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P69/ibm69i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P69/out/ibm69i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P69/ibm69i02.xml
%% ID: ibm-invalid-P69-ibm69i02.xml
%% Type: error
%% Sections: 4.1
%% Entities: parameter
%% Output: invalid/P69/out/ibm69i02.xml
%% Description:
%%   Tests invalid PEReference which is against P69 VC: Entity Declared. The PE
%%   with the name "pe1" is referred before declared in the file ibm69i02.dtd
'ibm-invalid-P69-ibm69i02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P69/ibm69i02.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P69/out/ibm69i02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P69/ibm69i03.xml
%% ID: ibm-invalid-P69-ibm69i03.xml
%% Type: error
%% Sections: 4.1
%% Entities: both
%% Output: invalid/P69/out/ibm69i03.xml
%% Description:
%%   Tests invalid PEReference which is against P69 VC: Entity Declared. The
%%   Name "pe3" in the PEReference in the file ibm69i03.ent does not match the
%%   Name of any declared PE.
'ibm-invalid-P69-ibm69i03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P69/ibm69i03.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P69/out/ibm69i03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P69/ibm69i04.xml
%% ID: ibm-invalid-P69-ibm69i04.xml
%% Type: error
%% Sections: 4.1
%% Entities: both
%% Output: invalid/P69/out/ibm69i04.xml
%% Description:
%%   Tests invalid PEReference which is against P69 VC: Entity Declared. The PE
%%   with the name "pe2" is referred before declared in the file ibm69i04.ent.
'ibm-invalid-P69-ibm69i04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P69/ibm69i04.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P69/out/ibm69i04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: invalid/P76/ibm76i01.xml
%% ID: ibm-invalid-P76-ibm76i01.xml
%% Type: invalid
%% Sections: 4.2.2
%% Output: invalid/P76/out/ibm76i01.xml
%% Description:
%%   Tests invalid NDataDecl which is against P76 VC: Notation declared. The
%%   Name "JPGformat" in the NDataDecl in the EntityDecl for "ge2" does not
%%   match the Name of any declared notation.
'ibm-invalid-P76-ibm76i01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","invalid/P76/ibm76i01.xml"]),
    Out = filename:join([datadir(Config),"ibm","invalid/P76/out/ibm76i01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "invalid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P01/ibm01n01.xml
%% ID: ibm-not-wf-P01-ibm01n01.xml
%% Type: not-wf
%% Sections: 2.1
%% Description:
%%   Tests a document with no element. A well-formed document should have at
%%   lease one elements.
'ibm-not-wf-P01-ibm01n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P01/ibm01n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P01/ibm01n02.xml
%% ID: ibm-not-wf-P01-ibm01n02.xml
%% Type: not-wf
%% Sections: 2.1
%% Description:
%%   Tests a document with wrong ordering of its prolog and element. The element
%%   occurs before the xml declaration and the DTD.
'ibm-not-wf-P01-ibm01n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P01/ibm01n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P01/ibm01n03.xml
%% ID: ibm-not-wf-P01-ibm01n03.xml
%% Type: not-wf
%% Sections: 2.1
%% Description:
%%   Tests a document with wrong combination of misc and element. One PI occurs
%%   between two elements.
'ibm-not-wf-P01-ibm01n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P01/ibm01n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n01.xml
%% ID: ibm-not-wf-P02-ibm02n01.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x00
'ibm-not-wf-P02-ibm02n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n02.xml
%% ID: ibm-not-wf-P02-ibm02n02.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x01
'ibm-not-wf-P02-ibm02n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n03.xml
%% ID: ibm-not-wf-P02-ibm02n03.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x02
'ibm-not-wf-P02-ibm02n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n04.xml
%% ID: ibm-not-wf-P02-ibm02n04.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x03
'ibm-not-wf-P02-ibm02n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n05.xml
%% ID: ibm-not-wf-P02-ibm02n05.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x04
'ibm-not-wf-P02-ibm02n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n06.xml
%% ID: ibm-not-wf-P02-ibm02n06.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x05
'ibm-not-wf-P02-ibm02n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n07.xml
%% ID: ibm-not-wf-P02-ibm02n07.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x06
'ibm-not-wf-P02-ibm02n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n08.xml
%% ID: ibm-not-wf-P02-ibm02n08.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x07
'ibm-not-wf-P02-ibm02n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n09.xml
%% ID: ibm-not-wf-P02-ibm02n09.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x08
'ibm-not-wf-P02-ibm02n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n10.xml
%% ID: ibm-not-wf-P02-ibm02n10.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x0B
'ibm-not-wf-P02-ibm02n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n11.xml
%% ID: ibm-not-wf-P02-ibm02n11.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x0C
'ibm-not-wf-P02-ibm02n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n12.xml
%% ID: ibm-not-wf-P02-ibm02n12.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x0E
'ibm-not-wf-P02-ibm02n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n13.xml
%% ID: ibm-not-wf-P02-ibm02n13.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x0F
'ibm-not-wf-P02-ibm02n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n14.xml
%% ID: ibm-not-wf-P02-ibm02n14.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x10
'ibm-not-wf-P02-ibm02n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n15.xml
%% ID: ibm-not-wf-P02-ibm02n15.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x11
'ibm-not-wf-P02-ibm02n15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n16.xml
%% ID: ibm-not-wf-P02-ibm02n16.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x12
'ibm-not-wf-P02-ibm02n16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n17.xml
%% ID: ibm-not-wf-P02-ibm02n17.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x13
'ibm-not-wf-P02-ibm02n17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n17.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n18.xml
%% ID: ibm-not-wf-P02-ibm02n18.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x14
'ibm-not-wf-P02-ibm02n18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n18.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n19.xml
%% ID: ibm-not-wf-P02-ibm02n19.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x15
'ibm-not-wf-P02-ibm02n19'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n19.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n20.xml
%% ID: ibm-not-wf-P02-ibm02n20.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x16
'ibm-not-wf-P02-ibm02n20'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n20.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n21.xml
%% ID: ibm-not-wf-P02-ibm02n21.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x17
'ibm-not-wf-P02-ibm02n21'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n21.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n22.xml
%% ID: ibm-not-wf-P02-ibm02n22.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x18
'ibm-not-wf-P02-ibm02n22'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n22.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n23.xml
%% ID: ibm-not-wf-P02-ibm02n23.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x19
'ibm-not-wf-P02-ibm02n23'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n23.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n24.xml
%% ID: ibm-not-wf-P02-ibm02n24.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x1A
'ibm-not-wf-P02-ibm02n24'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n24.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n25.xml
%% ID: ibm-not-wf-P02-ibm02n25.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x1B
'ibm-not-wf-P02-ibm02n25'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n25.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n26.xml
%% ID: ibm-not-wf-P02-ibm02n26.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x1C
'ibm-not-wf-P02-ibm02n26'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n26.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n27.xml
%% ID: ibm-not-wf-P02-ibm02n27.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x1D
'ibm-not-wf-P02-ibm02n27'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n27.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n28.xml
%% ID: ibm-not-wf-P02-ibm02n28.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x1E
'ibm-not-wf-P02-ibm02n28'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n28.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n29.xml
%% ID: ibm-not-wf-P02-ibm02n29.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #x1F
'ibm-not-wf-P02-ibm02n29'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n29.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n30.xml
%% ID: ibm-not-wf-P02-ibm02n30.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #xD800
'ibm-not-wf-P02-ibm02n30'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n30.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n31.xml
%% ID: ibm-not-wf-P02-ibm02n31.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #xDFFF
'ibm-not-wf-P02-ibm02n31'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n31.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n32.xml
%% ID: ibm-not-wf-P02-ibm02n32.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #xFFFE
'ibm-not-wf-P02-ibm02n32'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n32.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P02/ibm02n33.xml
%% ID: ibm-not-wf-P02-ibm02n33.xml
%% Type: not-wf
%% Sections: 2.2
%% Description:
%%   Tests a comment which contains an illegal Char: #xFFFF
'ibm-not-wf-P02-ibm02n33'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P02/ibm02n33.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P03/ibm03n01.xml
%% ID: ibm-not-wf-P03-ibm03n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an end tag which contains an illegal space character #x3000 which
%%   follows the element name "book".
'ibm-not-wf-P03-ibm03n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P03/ibm03n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n01.xml
%% ID: ibm-not-wf-P04-ibm04n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x21
'ibm-not-wf-P04-ibm04n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n02.xml
%% ID: ibm-not-wf-P04-ibm04n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x28
'ibm-not-wf-P04-ibm04n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n03.xml
%% ID: ibm-not-wf-P04-ibm04n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x29
'ibm-not-wf-P04-ibm04n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n04.xml
%% ID: ibm-not-wf-P04-ibm04n04.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x2B
'ibm-not-wf-P04-ibm04n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n05.xml
%% ID: ibm-not-wf-P04-ibm04n05.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x2C
'ibm-not-wf-P04-ibm04n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n06.xml
%% ID: ibm-not-wf-P04-ibm04n06.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x2F
'ibm-not-wf-P04-ibm04n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n07.xml
%% ID: ibm-not-wf-P04-ibm04n07.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x3B
'ibm-not-wf-P04-ibm04n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n08.xml
%% ID: ibm-not-wf-P04-ibm04n08.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x3C
'ibm-not-wf-P04-ibm04n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n09.xml
%% ID: ibm-not-wf-P04-ibm04n09.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x3D
'ibm-not-wf-P04-ibm04n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n10.xml
%% ID: ibm-not-wf-P04-ibm04n10.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x3F
'ibm-not-wf-P04-ibm04n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n11.xml
%% ID: ibm-not-wf-P04-ibm04n11.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x5B
'ibm-not-wf-P04-ibm04n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n12.xml
%% ID: ibm-not-wf-P04-ibm04n12.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x5C
'ibm-not-wf-P04-ibm04n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n13.xml
%% ID: ibm-not-wf-P04-ibm04n13.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x5D
'ibm-not-wf-P04-ibm04n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n14.xml
%% ID: ibm-not-wf-P04-ibm04n14.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x5E
'ibm-not-wf-P04-ibm04n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n15.xml
%% ID: ibm-not-wf-P04-ibm04n15.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x60
'ibm-not-wf-P04-ibm04n15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n16.xml
%% ID: ibm-not-wf-P04-ibm04n16.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x7B
'ibm-not-wf-P04-ibm04n16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n17.xml
%% ID: ibm-not-wf-P04-ibm04n17.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x7C
'ibm-not-wf-P04-ibm04n17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n17.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P04/ibm04n18.xml
%% ID: ibm-not-wf-P04-ibm04n18.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which contains an illegal ASCII NameChar.
%%   "IllegalNameChar" is followed by #x7D
'ibm-not-wf-P04-ibm04n18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P04/ibm04n18.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P05/ibm05n01.xml
%% ID: ibm-not-wf-P05-ibm05n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which has an illegal first character. An illegal
%%   first character "." is followed by "A_name-starts_with.".
'ibm-not-wf-P05-ibm05n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P05/ibm05n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P05/ibm05n02.xml
%% ID: ibm-not-wf-P05-ibm05n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which has an illegal first character. An illegal
%%   first character "-" is followed by "A_name-starts_with-".
'ibm-not-wf-P05-ibm05n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P05/ibm05n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P05/ibm05n03.xml
%% ID: ibm-not-wf-P05-ibm05n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an element name which has an illegal first character. An illegal
%%   first character "5" is followed by "A_name-starts_with_digit".
'ibm-not-wf-P05-ibm05n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P05/ibm05n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P09/ibm09n01.xml
%% ID: ibm-not-wf-P09-ibm09n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an internal general entity with an invalid value. The entity
%%   "Fullname" contains "%".
'ibm-not-wf-P09-ibm09n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P09/ibm09n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P09/ibm09n02.xml
%% ID: ibm-not-wf-P09-ibm09n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an internal general entity with an invalid value. The entity
%%   "Fullname" contains the ampersand character.
'ibm-not-wf-P09-ibm09n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P09/ibm09n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P09/ibm09n03.xml
%% ID: ibm-not-wf-P09-ibm09n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an internal general entity with an invalid value. The entity
%%   "Fullname" contains the double quote character in the middle.
'ibm-not-wf-P09-ibm09n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P09/ibm09n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P09/ibm09n04.xml
%% ID: ibm-not-wf-P09-ibm09n04.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an internal general entity with an invalid value. The closing bracket
%%   (double quote) is missing with the value of the entity "FullName".
'ibm-not-wf-P09-ibm09n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P09/ibm09n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n01.xml
%% ID: ibm-not-wf-P10-ibm10n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The value of the attribute
%%   "first" contains the character "less than".
'ibm-not-wf-P10-ibm10n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n02.xml
%% ID: ibm-not-wf-P10-ibm10n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The value of the attribute
%%   "first" contains the character ampersand.
'ibm-not-wf-P10-ibm10n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n03.xml
%% ID: ibm-not-wf-P10-ibm10n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The value of the attribute
%%   "first" contains the double quote character in the middle.
'ibm-not-wf-P10-ibm10n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n04.xml
%% ID: ibm-not-wf-P10-ibm10n04.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The closing bracket (double
%%   quote) is missing with The value of the attribute "first".
'ibm-not-wf-P10-ibm10n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n05.xml
%% ID: ibm-not-wf-P10-ibm10n05.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The value of the attribute
%%   "first" contains the character "less than".
'ibm-not-wf-P10-ibm10n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n06.xml
%% ID: ibm-not-wf-P10-ibm10n06.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The value of the attribute
%%   "first" contains the character ampersand.
'ibm-not-wf-P10-ibm10n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n07.xml
%% ID: ibm-not-wf-P10-ibm10n07.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The value of the attribute
%%   "first" contains the double quote character in the middle.
'ibm-not-wf-P10-ibm10n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P10/ibm10n08.xml
%% ID: ibm-not-wf-P10-ibm10n08.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests an attribute with an invalid value. The closing bracket (single
%%   quote) is missing with the value of the attribute "first".
'ibm-not-wf-P10-ibm10n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P10/ibm10n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P11/ibm11n01.xml
%% ID: ibm-not-wf-P11-ibm11n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests SystemLiteral. The systemLiteral for the element "student" has a
%%   double quote character in the middle.
'ibm-not-wf-P11-ibm11n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P11/ibm11n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P11/ibm11n02.xml
%% ID: ibm-not-wf-P11-ibm11n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests SystemLiteral. The systemLiteral for the element "student" has a
%%   single quote character in the middle.
'ibm-not-wf-P11-ibm11n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P11/ibm11n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P11/ibm11n03.xml
%% ID: ibm-not-wf-P11-ibm11n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests SystemLiteral. The closing bracket (double quote) is missing with the
%%   systemLiteral for the element "student".
'ibm-not-wf-P11-ibm11n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P11/ibm11n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P11/ibm11n04.xml
%% ID: ibm-not-wf-P11-ibm11n04.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests SystemLiteral. The closing bracket (single quote) is missing with the
%%   systemLiteral for the element "student".
'ibm-not-wf-P11-ibm11n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P11/ibm11n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P12/ibm12n01.xml
%% ID: ibm-not-wf-P12-ibm12n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests PubidLiteral. The closing bracket (double quote) is missing with the
%%   value of the PubidLiteral for the entity "info".
'ibm-not-wf-P12-ibm12n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P12/ibm12n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P12/ibm12n02.xml
%% ID: ibm-not-wf-P12-ibm12n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests PubidLiteral. The value of the PubidLiteral for the entity "info" has
%%   a single quote character in the middle..
'ibm-not-wf-P12-ibm12n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P12/ibm12n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P12/ibm12n03.xml
%% ID: ibm-not-wf-P12-ibm12n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests PubidLiteral. The closing bracket (single quote) is missing with the
%%   value of the PubidLiteral for the entity "info".
'ibm-not-wf-P12-ibm12n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P12/ibm12n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P13/ibm13n01.xml
%% ID: ibm-not-wf-P13-ibm13n01.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests PubidChar. The pubidChar of the PubidLiteral for the entity "info"
%%   contains the character "{".
'ibm-not-wf-P13-ibm13n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P13/ibm13n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P13/ibm13n02.xml
%% ID: ibm-not-wf-P13-ibm13n02.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests PubidChar. The pubidChar of the PubidLiteral for the entity "info"
%%   contains the character "~".
'ibm-not-wf-P13-ibm13n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P13/ibm13n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P13/ibm13n03.xml
%% ID: ibm-not-wf-P13-ibm13n03.xml
%% Type: not-wf
%% Sections: 2.3
%% Description:
%%   Tests PubidChar. The pubidChar of the PubidLiteral for the entity "info"
%%   contains the character double quote in the middle.
'ibm-not-wf-P13-ibm13n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P13/ibm13n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P14/ibm14n01.xml
%% ID: ibm-not-wf-P14-ibm14n01.xml
%% Type: not-wf
%% Sections: 2.4
%% Description:
%%   Tests CharData. The content of the element "student" contains the sequence
%%   close-bracket close-bracket greater-than.
'ibm-not-wf-P14-ibm14n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P14/ibm14n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P14/ibm14n02.xml
%% ID: ibm-not-wf-P14-ibm14n02.xml
%% Type: not-wf
%% Sections: 2.4
%% Description:
%%   Tests CharData. The content of the element "student" contains the character
%%   "less than".
'ibm-not-wf-P14-ibm14n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P14/ibm14n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P14/ibm14n03.xml
%% ID: ibm-not-wf-P14-ibm14n03.xml
%% Type: not-wf
%% Sections: 2.4
%% Description:
%%   Tests CharData. The content of the element "student" contains the character
%%   ampersand.
'ibm-not-wf-P14-ibm14n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P14/ibm14n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P15/ibm15n01.xml
%% ID: ibm-not-wf-P15-ibm15n01.xml
%% Type: not-wf
%% Sections: 2.5
%% Description:
%%   Tests comment. The text of the second comment contains the character "-".
'ibm-not-wf-P15-ibm15n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P15/ibm15n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P15/ibm15n02.xml
%% ID: ibm-not-wf-P15-ibm15n02.xml
%% Type: not-wf
%% Sections: 2.5
%% Description:
%%   Tests comment. The second comment has a wrong closing sequence "-(greater
%%   than)".
'ibm-not-wf-P15-ibm15n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P15/ibm15n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P15/ibm15n03.xml
%% ID: ibm-not-wf-P15-ibm15n03.xml
%% Type: not-wf
%% Sections: 2.5
%% Description:
%%   Tests comment. The second comment has a wrong beginning sequence "(less
%%   than)!-".
'ibm-not-wf-P15-ibm15n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P15/ibm15n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P15/ibm15n04.xml
%% ID: ibm-not-wf-P15-ibm15n04.xml
%% Type: not-wf
%% Sections: 2.5
%% Description:
%%   Tests comment. The closing sequence is missing with the second comment.
'ibm-not-wf-P15-ibm15n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P15/ibm15n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P16/ibm16n01.xml
%% ID: ibm-not-wf-P16-ibm16n01.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PI. The content of the PI includes the sequence "?(greater than)?".
'ibm-not-wf-P16-ibm16n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P16/ibm16n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P16/ibm16n02.xml
%% ID: ibm-not-wf-P16-ibm16n02.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PI. The PITarget is missing in the PI.
'ibm-not-wf-P16-ibm16n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P16/ibm16n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P16/ibm16n03.xml
%% ID: ibm-not-wf-P16-ibm16n03.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PI. The PI has a wrong closing sequence ">".
'ibm-not-wf-P16-ibm16n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P16/ibm16n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P16/ibm16n04.xml
%% ID: ibm-not-wf-P16-ibm16n04.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PI. The closing sequence is missing in the PI.
'ibm-not-wf-P16-ibm16n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P16/ibm16n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P17/ibm17n01.xml
%% ID: ibm-not-wf-P17-ibm17n01.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PITarget. The PITarget contains the string "XML".
'ibm-not-wf-P17-ibm17n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P17/ibm17n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P17/ibm17n02.xml
%% ID: ibm-not-wf-P17-ibm17n02.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PITarget. The PITarget contains the string "xML".
'ibm-not-wf-P17-ibm17n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P17/ibm17n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P17/ibm17n03.xml
%% ID: ibm-not-wf-P17-ibm17n03.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PITarget. The PITarget contains the string "xml".
'ibm-not-wf-P17-ibm17n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P17/ibm17n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P17/ibm17n04.xml
%% ID: ibm-not-wf-P17-ibm17n04.xml
%% Type: not-wf
%% Sections: 2.6
%% Description:
%%   Tests PITarget. The PITarget contains the string "xmL".
'ibm-not-wf-P17-ibm17n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P17/ibm17n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P18/ibm18n01.xml
%% ID: ibm-not-wf-P18-ibm18n01.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDSect. The CDStart is missing in the CDSect in the content of
%%   element "student".
'ibm-not-wf-P18-ibm18n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P18/ibm18n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P18/ibm18n02.xml
%% ID: ibm-not-wf-P18-ibm18n02.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDSect. The CDEnd is missing in the CDSect in the content of element
%%   "student".
'ibm-not-wf-P18-ibm18n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P18/ibm18n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P19/ibm19n01.xml
%% ID: ibm-not-wf-P19-ibm19n01.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDStart. The CDStart contains a lower case string "cdata".
'ibm-not-wf-P19-ibm19n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P19/ibm19n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P19/ibm19n02.xml
%% ID: ibm-not-wf-P19-ibm19n02.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDStart. The CDStart contains an extra character "[".
'ibm-not-wf-P19-ibm19n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P19/ibm19n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P19/ibm19n03.xml
%% ID: ibm-not-wf-P19-ibm19n03.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDStart. The CDStart contains a wrong character "?".
'ibm-not-wf-P19-ibm19n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P19/ibm19n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P20/ibm20n01.xml
%% ID: ibm-not-wf-P20-ibm20n01.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDATA with an illegal sequence. The CDATA contains the sequence
%%   close-bracket close-bracket greater-than.
'ibm-not-wf-P20-ibm20n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P20/ibm20n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P21/ibm21n01.xml
%% ID: ibm-not-wf-P21-ibm21n01.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDEnd. One "]" is missing in the CDEnd.
'ibm-not-wf-P21-ibm21n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P21/ibm21n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P21/ibm21n02.xml
%% ID: ibm-not-wf-P21-ibm21n02.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDEnd. An extra "]" is placed in the CDEnd.
'ibm-not-wf-P21-ibm21n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P21/ibm21n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P21/ibm21n03.xml
%% ID: ibm-not-wf-P21-ibm21n03.xml
%% Type: not-wf
%% Sections: 2.7
%% Description:
%%   Tests CDEnd. A wrong character ")" is placed in the CDEnd.
'ibm-not-wf-P21-ibm21n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P21/ibm21n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P22/ibm22n01.xml
%% ID: ibm-not-wf-P22-ibm22n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests prolog with wrong field ordering. The XMLDecl occurs after the DTD.
'ibm-not-wf-P22-ibm22n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P22/ibm22n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P22/ibm22n02.xml
%% ID: ibm-not-wf-P22-ibm22n02.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests prolog with wrong field ordering. The Misc (comment) occurs before
%%   the XMLDecl.
'ibm-not-wf-P22-ibm22n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P22/ibm22n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P22/ibm22n03.xml
%% ID: ibm-not-wf-P22-ibm22n03.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests prolog with wrong field ordering. The XMLDecl occurs after the DTD
%%   and a comment. The other comment occurs before the DTD.
'ibm-not-wf-P22-ibm22n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P22/ibm22n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P23/ibm23n01.xml
%% ID: ibm-not-wf-P23-ibm23n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests XMLDecl with a required field missing. The Versioninfo is missing in
%%   the XMLDecl.
'ibm-not-wf-P23-ibm23n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P23/ibm23n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P23/ibm23n02.xml
%% ID: ibm-not-wf-P23-ibm23n02.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests XMLDecl with wrong field ordering. The VersionInfo occurs after the
%%   EncodingDecl.
'ibm-not-wf-P23-ibm23n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P23/ibm23n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P23/ibm23n03.xml
%% ID: ibm-not-wf-P23-ibm23n03.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests XMLDecl with wrong field ordering. The VersionInfo occurs after the
%%   SDDecl and the SDDecl occurs after the VersionInfo.
'ibm-not-wf-P23-ibm23n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P23/ibm23n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P23/ibm23n04.xml
%% ID: ibm-not-wf-P23-ibm23n04.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests XMLDecl with wrong key word. An upper case string "XML" is used as
%%   the key word in the XMLDecl.
'ibm-not-wf-P23-ibm23n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P23/ibm23n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P23/ibm23n05.xml
%% ID: ibm-not-wf-P23-ibm23n05.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests XMLDecl with a wrong closing sequence ">".
'ibm-not-wf-P23-ibm23n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P23/ibm23n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P23/ibm23n06.xml
%% ID: ibm-not-wf-P23-ibm23n06.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests XMLDecl with a wrong opening sequence "(less than)!".
'ibm-not-wf-P23-ibm23n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P23/ibm23n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n01.xml
%% ID: ibm-not-wf-P24-ibm24n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with a required field missing. The VersionNum is missing
%%   in the VersionInfo in the XMLDecl.
'ibm-not-wf-P24-ibm24n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n02.xml
%% ID: ibm-not-wf-P24-ibm24n02.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with a required field missing. The white space is missing
%%   between the key word "xml" and the VersionInfo in the XMLDecl.
'ibm-not-wf-P24-ibm24n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n03.xml
%% ID: ibm-not-wf-P24-ibm24n03.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with a required field missing. The "=" (equal sign) is
%%   missing between the key word "version" and the VersionNum.
'ibm-not-wf-P24-ibm24n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n04.xml
%% ID: ibm-not-wf-P24-ibm24n04.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with wrong field ordering. The VersionNum occurs before
%%   "=" and "version".
'ibm-not-wf-P24-ibm24n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n05.xml
%% ID: ibm-not-wf-P24-ibm24n05.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with wrong field ordering. The "=" occurs after "version"
%%   and the VersionNum.
'ibm-not-wf-P24-ibm24n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n06.xml
%% ID: ibm-not-wf-P24-ibm24n06.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with the wrong key word "Version".
'ibm-not-wf-P24-ibm24n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n07.xml
%% ID: ibm-not-wf-P24-ibm24n07.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with the wrong key word "versioN".
'ibm-not-wf-P24-ibm24n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n08.xml
%% ID: ibm-not-wf-P24-ibm24n08.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with mismatched quotes around the VersionNum. version =
%%   '1.0" is used as the VersionInfo.
'ibm-not-wf-P24-ibm24n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P24/ibm24n09.xml
%% ID: ibm-not-wf-P24-ibm24n09.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionInfo with mismatched quotes around the VersionNum. The closing
%%   bracket for the VersionNum is missing.
'ibm-not-wf-P24-ibm24n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P24/ibm24n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P25/ibm25n01.xml
%% ID: ibm-not-wf-P25-ibm25n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests eq with a wrong key word "==".
'ibm-not-wf-P25-ibm25n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P25/ibm25n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P25/ibm25n02.xml
%% ID: ibm-not-wf-P25-ibm25n02.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests eq with a wrong key word "eq".
'ibm-not-wf-P25-ibm25n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P25/ibm25n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P26/ibm26n01.xml
%% ID: ibm-not-wf-P26-ibm26n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests VersionNum with an illegal character "#".
'ibm-not-wf-P26-ibm26n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P26/ibm26n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P27/ibm27n01.xml
%% ID: ibm-not-wf-P27-ibm27n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests type of Misc. An element declaration is used as a type of Misc After
%%   the element "animal".
'ibm-not-wf-P27-ibm27n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P27/ibm27n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n01.xml
%% ID: ibm-not-wf-P28-ibm28n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with a required field missing. The Name "animal" is
%%   missing in the doctypedecl.
'ibm-not-wf-P28-ibm28n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n02.xml
%% ID: ibm-not-wf-P28-ibm28n02.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with wrong field ordering. The Name "animal" occurs after
%%   the markup declarations inside the "[]".
'ibm-not-wf-P28-ibm28n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n03.xml
%% ID: ibm-not-wf-P28-ibm28n03.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with wrong field ordering. The Name "animal" occurs after
%%   the markup declarations inside the "[]".
'ibm-not-wf-P28-ibm28n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n04.xml
%% ID: ibm-not-wf-P28-ibm28n04.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with general entity reference.The "(ampersand)generalE"
%%   occurs in the DTD.
'ibm-not-wf-P28-ibm28n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n05.xml
%% ID: ibm-not-wf-P28-ibm28n05.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with wrong key word. A wrong key word "DOCtYPE" occurs on
%%   line 2.
'ibm-not-wf-P28-ibm28n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n06.xml
%% ID: ibm-not-wf-P28-ibm28n06.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with mismatched brackets. The closing bracket "]" of the
%%   DTD is missing.
'ibm-not-wf-P28-ibm28n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n07.xml
%% ID: ibm-not-wf-P28-ibm28n07.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with wrong bracket. The opening bracket "{" occurs in the
%%   DTD.
'ibm-not-wf-P28-ibm28n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P28/ibm28n08.xml
%% ID: ibm-not-wf-P28-ibm28n08.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests doctypedecl with wrong opening sequence. The opening sequence "(less
%%   than)?DOCTYPE" occurs in the DTD.
'ibm-not-wf-P28-ibm28n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P28/ibm28n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/p28a/ibm28an01.xml
%% ID: ibm-not-wf-p28a-ibm28an01.xml
%% Type: not-wf
%% Sections: 2.8
%% Entities: parameter
%% Description:
%%   This test violates WFC:PE Between Declarations in Production 28a. The last
%%   character of a markup declaration is not contained in the same
%%   parameter-entity text replacement.
'ibm-not-wf-p28a-ibm28an01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/p28a/ibm28an01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n01.xml
%% ID: ibm-not-wf-P29-ibm29n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests markupdecl with an illegal markup declaration. A XMLDecl occurs
%%   inside the DTD.
'ibm-not-wf-P29-ibm29n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n02.xml
%% ID: ibm-not-wf-P29-ibm29n02.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests WFC "PEs in Internal Subset". A PE reference occurs inside an
%%   elementdecl in the DTD.
'ibm-not-wf-P29-ibm29n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n03.xml
%% ID: ibm-not-wf-P29-ibm29n03.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests WFC "PEs in Internal Subset". A PE reference occurs inside an
%%   ATTlistDecl in the DTD.
'ibm-not-wf-P29-ibm29n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n04.xml
%% ID: ibm-not-wf-P29-ibm29n04.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests WFC "PEs in Internal Subset". A PE reference occurs inside an
%%   EntityDecl in the DTD.
'ibm-not-wf-P29-ibm29n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n05.xml
%% ID: ibm-not-wf-P29-ibm29n05.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests WFC "PEs in Internal Subset". A PE reference occurs inside a PI in
%%   the DTD.
'ibm-not-wf-P29-ibm29n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n06.xml
%% ID: ibm-not-wf-P29-ibm29n06.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests WFC "PEs in Internal Subset". A PE reference occurs inside a comment
%%   in the DTD.
'ibm-not-wf-P29-ibm29n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P29/ibm29n07.xml
%% ID: ibm-not-wf-P29-ibm29n07.xml
%% Type: not-wf
%% Sections: 2.8
%% Description:
%%   Tests WFC "PEs in Internal Subset". A PE reference occurs inside a
%%   NotationDecl in the DTD.
'ibm-not-wf-P29-ibm29n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P29/ibm29n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P30/ibm30n01.xml
%% ID: ibm-not-wf-P30-ibm30n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Entities: parameter
%% Description:
%%   Tests extSubset with wrong field ordering. In the file "ibm30n01.dtd", the
%%   TextDecl occurs after the extSubsetDecl (the element declaration).
'ibm-not-wf-P30-ibm30n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P30/ibm30n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P31/ibm31n01.xml
%% ID: ibm-not-wf-P31-ibm31n01.xml
%% Type: not-wf
%% Sections: 2.8
%% Entities: parameter
%% Description:
%%   Tests extSubsetDecl with an illegal field. A general entity reference
%%   occurs in file "ibm31n01.dtd".
'ibm-not-wf-P31-ibm31n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P31/ibm31n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n01.xml
%% ID: ibm-not-wf-P32-ibm32n01.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with a required field missing. The leading white space is
%%   missing with the SDDecl in the XMLDecl.
'ibm-not-wf-P32-ibm32n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n02.xml
%% ID: ibm-not-wf-P32-ibm32n02.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with a required field missing. The "=" sign is missing in the
%%   SDDecl in the XMLDecl.
'ibm-not-wf-P32-ibm32n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n03.xml
%% ID: ibm-not-wf-P32-ibm32n03.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with wrong key word. The word "Standalone" occurs in the
%%   SDDecl in the XMLDecl.
'ibm-not-wf-P32-ibm32n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n04.xml
%% ID: ibm-not-wf-P32-ibm32n04.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with wrong key word. The word "Yes" occurs in the SDDecl in
%%   the XMLDecl.
'ibm-not-wf-P32-ibm32n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n05.xml
%% ID: ibm-not-wf-P32-ibm32n05.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with wrong key word. The word "YES" occurs in the SDDecl in
%%   the XMLDecl.
'ibm-not-wf-P32-ibm32n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n06.xml
%% ID: ibm-not-wf-P32-ibm32n06.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with wrong key word. The word "No" occurs in the SDDecl in the
%%   XMLDecl.
'ibm-not-wf-P32-ibm32n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n07.xml
%% ID: ibm-not-wf-P32-ibm32n07.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with wrong key word. The word "NO" occurs in the SDDecl in the
%%   XMLDecl.
'ibm-not-wf-P32-ibm32n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n08.xml
%% ID: ibm-not-wf-P32-ibm32n08.xml
%% Type: not-wf
%% Sections: 2.9
%% Description:
%%   Tests SDDecl with wrong field ordering. The "=" sign occurs after the key
%%   word "yes" in the SDDecl in the XMLDecl.
'ibm-not-wf-P32-ibm32n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P32/ibm32n09.xml
%% ID: ibm-not-wf-P32-ibm32n09.xml
%% Type: not-wf
%% Sections: 2.9
%% Entities: parameter
%% Description:
%%   This is test violates WFC: Entity Declared in P68. The standalone document
%%   declaration has the value yes, BUT there is an external markup declaration
%%   of an entity (other than amp, lt, gt, apos, quot), and references to this
%%   entity appear in the document.
'ibm-not-wf-P32-ibm32n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P32/ibm32n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P39/ibm39n01.xml
%% ID: ibm-not-wf-P39-ibm39n01.xml
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests element with a required field missing. The ETag is missing for the
%%   element "root".
'ibm-not-wf-P39-ibm39n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P39/ibm39n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P39/ibm39n02.xml
%% ID: ibm-not-wf-P39-ibm39n02.xml
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests element with a required field missing. The STag is missing for the
%%   element "root".
'ibm-not-wf-P39-ibm39n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P39/ibm39n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P39/ibm39n03.xml
%% ID: ibm-not-wf-P39-ibm39n03.xml
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests element with required fields missing. Both the content and the ETag
%%   are missing in the element "root".
'ibm-not-wf-P39-ibm39n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P39/ibm39n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P39/ibm39n04.xml
%% ID: ibm-not-wf-P39-ibm39n04.xml
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests element with required fields missing. Both the content and the STag
%%   are missing in the element "root".
'ibm-not-wf-P39-ibm39n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P39/ibm39n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P39/ibm39n05.xml
%% ID: ibm-not-wf-P39-ibm39n05.xml
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests element with wrong field ordering. The STag and the ETag are swapped
%%   in the element "root".
'ibm-not-wf-P39-ibm39n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P39/ibm39n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P39/ibm39n06.xml
%% ID: ibm-not-wf-P39-ibm39n06.xml
%% Type: not-wf
%% Sections: 3
%% Description:
%%   Tests element with wrong field ordering. The content occurs after the ETag
%%   of the element "root".
'ibm-not-wf-P39-ibm39n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P39/ibm39n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P40/ibm40n01.xml
%% ID: ibm-not-wf-P40-ibm40n01.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests STag with a required field missing. The Name "root" is in the STag of
%%   the element "root".
'ibm-not-wf-P40-ibm40n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P40/ibm40n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P40/ibm40n02.xml
%% ID: ibm-not-wf-P40-ibm40n02.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests STag with a required field missing. The white space between the Name
%%   "root" and the attribute "attr1" is missing in the STag of the element
%%   "root".
'ibm-not-wf-P40-ibm40n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P40/ibm40n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P40/ibm40n03.xml
%% ID: ibm-not-wf-P40-ibm40n03.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests STag with wrong field ordering. The Name "root" occurs after the
%%   attribute "attr1" in the STag of the element "root".
'ibm-not-wf-P40-ibm40n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P40/ibm40n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P40/ibm40n04.xml
%% ID: ibm-not-wf-P40-ibm40n04.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests STag with a wrong opening sequence. The string "(less than)!" is used
%%   as the opening sequence for the STag of the element "root".
'ibm-not-wf-P40-ibm40n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P40/ibm40n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P40/ibm40n05.xml
%% ID: ibm-not-wf-P40-ibm40n05.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests STag with duplicate attribute names. The attribute name "attr1"
%%   occurs twice in the STag of the element "root".
'ibm-not-wf-P40-ibm40n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P40/ibm40n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n01.xml
%% ID: ibm-not-wf-P41-ibm41n01.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with a required field missing. The attribute name is
%%   missing in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n02.xml
%% ID: ibm-not-wf-P41-ibm41n02.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with a required field missing. The "=" is missing between
%%   the attribute name and the attribute value in the Attribute in the STag of
%%   the element "root".
'ibm-not-wf-P41-ibm41n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n03.xml
%% ID: ibm-not-wf-P41-ibm41n03.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with a required field missing. The AttValue is missing in
%%   the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n04.xml
%% ID: ibm-not-wf-P41-ibm41n04.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with a required field missing. The Name and the "=" are
%%   missing in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n05.xml
%% ID: ibm-not-wf-P41-ibm41n05.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with a required field missing. The "=" and the AttValue are
%%   missing in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n06.xml
%% ID: ibm-not-wf-P41-ibm41n06.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with a required field missing. The Name and the AttValue
%%   are missing in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n07.xml
%% ID: ibm-not-wf-P41-ibm41n07.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with wrong field ordering. The "=" occurs after the Name
%%   and the AttValue in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n08.xml
%% ID: ibm-not-wf-P41-ibm41n08.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with wrong field ordering. The Name and the AttValue are
%%   swapped in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n09.xml
%% ID: ibm-not-wf-P41-ibm41n09.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute with wrong field ordering. The "=" occurs before the Name
%%   and the AttValue in the Attribute in the STag of the element "root".
'ibm-not-wf-P41-ibm41n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n10.xml
%% ID: ibm-not-wf-P41-ibm41n10.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute against WFC "no external entity references". A direct
%%   reference to the external entity "aExternal" is contained in the value of
%%   the attribute "attr1".
'ibm-not-wf-P41-ibm41n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n11.xml
%% ID: ibm-not-wf-P41-ibm41n11.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute against WFC "no external entity references". A indirect
%%   reference to the external entity "aExternal" is contained in the value of
%%   the attribute "attr1".
'ibm-not-wf-P41-ibm41n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n12.xml
%% ID: ibm-not-wf-P41-ibm41n12.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute against WFC "no external entity references". A direct
%%   reference to the external unparsed entity "aImage" is contained in the
%%   value of the attribute "attr1".
'ibm-not-wf-P41-ibm41n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n13.xml
%% ID: ibm-not-wf-P41-ibm41n13.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute against WFC "No (less than) character in Attribute Values".
%%   The character "less than" is contained in the value of the attribute
%%   "attr1".
'ibm-not-wf-P41-ibm41n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P41/ibm41n14.xml
%% ID: ibm-not-wf-P41-ibm41n14.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests Attribute against WFC "No (less than) in Attribute Values". The
%%   character "less than" is contained in the value of the attribute "attr1"
%%   through indirect internal entity reference.
'ibm-not-wf-P41-ibm41n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P41/ibm41n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P42/ibm42n01.xml
%% ID: ibm-not-wf-P42-ibm42n01.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests ETag with a required field missing. The Name is missing in the ETag
%%   of the element "root".
'ibm-not-wf-P42-ibm42n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P42/ibm42n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P42/ibm42n02.xml
%% ID: ibm-not-wf-P42-ibm42n02.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests ETag with a wrong beginning sequence. The string "(less than)\" is
%%   used as a beginning sequence of the ETag of the element "root".
'ibm-not-wf-P42-ibm42n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P42/ibm42n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P42/ibm42n03.xml
%% ID: ibm-not-wf-P42-ibm42n03.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests ETag with a wrong beginning sequence. The string "less than" is used
%%   as a beginning sequence of the ETag of the element "root".
'ibm-not-wf-P42-ibm42n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P42/ibm42n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P42/ibm42n04.xml
%% ID: ibm-not-wf-P42-ibm42n04.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests ETag with a wrong structure. An white space occurs between The
%%   beginning sequence and the Name of the ETag of the element "root".
'ibm-not-wf-P42-ibm42n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P42/ibm42n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P42/ibm42n05.xml
%% ID: ibm-not-wf-P42-ibm42n05.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests ETag with a wrong structure. The ETag of the element "root" contains
%%   an Attribute (attr1="any").
'ibm-not-wf-P42-ibm42n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P42/ibm42n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P43/ibm43n01.xml
%% ID: ibm-not-wf-P43-ibm43n01.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests element content with a wrong option. A NotationDecl is used as the
%%   content of the element "root".
'ibm-not-wf-P43-ibm43n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P43/ibm43n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P43/ibm43n02.xml
%% ID: ibm-not-wf-P43-ibm43n02.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests element content with a wrong option. An elementdecl is used as the
%%   content of the element "root".
'ibm-not-wf-P43-ibm43n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P43/ibm43n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P43/ibm43n04.xml
%% ID: ibm-not-wf-P43-ibm43n04.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests element content with a wrong option. An entitydecl is used as the
%%   content of the element "root".
'ibm-not-wf-P43-ibm43n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P43/ibm43n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P43/ibm43n05.xml
%% ID: ibm-not-wf-P43-ibm43n05.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests element content with a wrong option. An AttlistDecl is used as the
%%   content of the element "root".
'ibm-not-wf-P43-ibm43n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P43/ibm43n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P44/ibm44n01.xml
%% ID: ibm-not-wf-P44-ibm44n01.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests EmptyElemTag with a required field missing. The Name "root" is
%%   missing in the EmptyElemTag.
'ibm-not-wf-P44-ibm44n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P44/ibm44n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P44/ibm44n02.xml
%% ID: ibm-not-wf-P44-ibm44n02.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests EmptyElemTag with wrong field ordering. The Attribute (attri1 =
%%   "any") occurs before the name of the element "root" in the EmptyElemTag.
'ibm-not-wf-P44-ibm44n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P44/ibm44n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P44/ibm44n03.xml
%% ID: ibm-not-wf-P44-ibm44n03.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests EmptyElemTag with wrong closing sequence. The string "\>" is used as
%%   the closing sequence in the EmptyElemtag of the element "root".
'ibm-not-wf-P44-ibm44n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P44/ibm44n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P44/ibm44n04.xml
%% ID: ibm-not-wf-P44-ibm44n04.xml
%% Type: not-wf
%% Sections: 3.1
%% Description:
%%   Tests EmptyElemTag which against the WFC "Unique Att Spec". The attribute
%%   name "attr1" occurs twice in the EmptyElemTag of the element "root".
'ibm-not-wf-P44-ibm44n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P44/ibm44n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n01.xml
%% ID: ibm-not-wf-P45-ibm45n01.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with a required field missing. The Name is missing in the
%%   second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n02.xml
%% ID: ibm-not-wf-P45-ibm45n02.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with a required field missing. The white space is missing
%%   between "aEle" and "(#PCDATA)" in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n03.xml
%% ID: ibm-not-wf-P45-ibm45n03.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with a required field missing. The contentspec is missing
%%   in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n04.xml
%% ID: ibm-not-wf-P45-ibm45n04.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with a required field missing. The contentspec and the
%%   white space is missing in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n05.xml
%% ID: ibm-not-wf-P45-ibm45n05.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with a required field missing. The Name, the white space,
%%   and the contentspec are missing in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n06.xml
%% ID: ibm-not-wf-P45-ibm45n06.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with wrong field ordering. The Name occurs after the
%%   contentspec in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n07.xml
%% ID: ibm-not-wf-P45-ibm45n07.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with wrong beginning sequence. The string "(less
%%   than)ELEMENT" is used as the beginning sequence in the second elementdecl
%%   in the DTD.
'ibm-not-wf-P45-ibm45n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n08.xml
%% ID: ibm-not-wf-P45-ibm45n08.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with wrong key word. The string "Element" is used as the
%%   key word in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P45/ibm45n09.xml
%% ID: ibm-not-wf-P45-ibm45n09.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests elementdecl with wrong key word. The string "element" is used as the
%%   key word in the second elementdecl in the DTD.
'ibm-not-wf-P45-ibm45n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P45/ibm45n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P46/ibm46n01.xml
%% ID: ibm-not-wf-P46-ibm46n01.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests contentspec with wrong key word. the string "empty" is used as the
%%   key word in the contentspec of the second elementdecl in the DTD.
'ibm-not-wf-P46-ibm46n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P46/ibm46n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P46/ibm46n02.xml
%% ID: ibm-not-wf-P46-ibm46n02.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests contentspec with wrong key word. the string "Empty" is used as the
%%   key word in the contentspec of the second elementdecl in the DTD.
'ibm-not-wf-P46-ibm46n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P46/ibm46n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P46/ibm46n03.xml
%% ID: ibm-not-wf-P46-ibm46n03.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests contentspec with wrong key word. the string "Any" is used as the key
%%   word in the contentspec of the second elementdecl in the DTD.
'ibm-not-wf-P46-ibm46n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P46/ibm46n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P46/ibm46n04.xml
%% ID: ibm-not-wf-P46-ibm46n04.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests contentspec with wrong key word. the string "any" is used as the key
%%   word in the contentspec of the second elementdecl in the DTD.
'ibm-not-wf-P46-ibm46n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P46/ibm46n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P46/ibm46n05.xml
%% ID: ibm-not-wf-P46-ibm46n05.xml
%% Type: not-wf
%% Sections: 3.2
%% Description:
%%   Tests contentspec with a wrong option. The string "#CDATA" is used as the
%%   contentspec in the second elementdecl in the DTD.
'ibm-not-wf-P46-ibm46n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P46/ibm46n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P47/ibm47n01.xml
%% ID: ibm-not-wf-P47-ibm47n01.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests children with a required field missing. The "+" is used as the choice
%%   or seq field in the second elementdecl in the DTD.
'ibm-not-wf-P47-ibm47n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P47/ibm47n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P47/ibm47n02.xml
%% ID: ibm-not-wf-P47-ibm47n02.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests children with a required field missing. The "*" is used as the choice
%%   or seq field in the second elementdecl in the DTD.
'ibm-not-wf-P47-ibm47n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P47/ibm47n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P47/ibm47n03.xml
%% ID: ibm-not-wf-P47-ibm47n03.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests children with a required field missing. The "?" is used as the choice
%%   or seq field in the second elementdecl in the DTD.
'ibm-not-wf-P47-ibm47n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P47/ibm47n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P47/ibm47n04.xml
%% ID: ibm-not-wf-P47-ibm47n04.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests children with wrong field ordering. The "*" occurs before the seq
%%   field (a,a) in the second elementdecl in the DTD.
'ibm-not-wf-P47-ibm47n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P47/ibm47n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P47/ibm47n05.xml
%% ID: ibm-not-wf-P47-ibm47n05.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests children with wrong field ordering. The "+" occurs before the choice
%%   field (a|a) in the second elementdecl in the DTD.
'ibm-not-wf-P47-ibm47n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P47/ibm47n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P47/ibm47n06.xml
%% ID: ibm-not-wf-P47-ibm47n06.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests children with wrong key word. The "^" occurs after the seq field in
%%   the second elementdecl in the DTD.
'ibm-not-wf-P47-ibm47n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P47/ibm47n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n01.xml
%% ID: ibm-not-wf-P48-ibm48n01.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with a required fields missing. The field Name|choice|seq is
%%   missing in the second cp in the choice field in the third elementdecl in
%%   the DTD.
'ibm-not-wf-P48-ibm48n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n02.xml
%% ID: ibm-not-wf-P48-ibm48n02.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with a required fields missing. The field Name|choice|seq is
%%   missing in the cp in the third elementdecl in the DTD.
'ibm-not-wf-P48-ibm48n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n03.xml
%% ID: ibm-not-wf-P48-ibm48n03.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with a required fields missing. The field Name|choice|seq is
%%   missing in the first cp in the choice field in the third elementdecl in the
%%   DTD.
'ibm-not-wf-P48-ibm48n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n04.xml
%% ID: ibm-not-wf-P48-ibm48n04.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with wrong field ordering. The "+" occurs before the seq (a,a) in
%%   the first cp in the choice field in the third elementdecl in the DTD.
'ibm-not-wf-P48-ibm48n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n05.xml
%% ID: ibm-not-wf-P48-ibm48n05.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with wrong field ordering. The "*" occurs before the choice (a|b)
%%   in the first cp in the seq field in the third elementdecl in the DTD.
'ibm-not-wf-P48-ibm48n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n06.xml
%% ID: ibm-not-wf-P48-ibm48n06.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with wrong field ordering. The "?" occurs before the Name "a" in
%%   the second cp in the seq field in the third elementdecl in the DTD.
'ibm-not-wf-P48-ibm48n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P48/ibm48n07.xml
%% ID: ibm-not-wf-P48-ibm48n07.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests cp with wrong key word. The "^" occurs after the Name "a" in the
%%   first cp in the choice field in the third elementdecl in the DTD.
'ibm-not-wf-P48-ibm48n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P48/ibm48n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P49/ibm49n01.xml
%% ID: ibm-not-wf-P49-ibm49n01.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests choice with a required field missing. The two cps are missing in the
%%   choice field in the third elementdecl in the DTD.
'ibm-not-wf-P49-ibm49n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P49/ibm49n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P49/ibm49n02.xml
%% ID: ibm-not-wf-P49-ibm49n02.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests choice with a required field missing. The third cp is missing in the
%%   choice field in the fourth elementdecl in the DTD.
'ibm-not-wf-P49-ibm49n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P49/ibm49n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P49/ibm49n03.xml
%% ID: ibm-not-wf-P49-ibm49n03.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests choice with a wrong separator. The "!" is used as the separator in
%%   the choice field in the fourth elementdecl in the DTD.
'ibm-not-wf-P49-ibm49n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P49/ibm49n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P49/ibm49n04.xml
%% ID: ibm-not-wf-P49-ibm49n04.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests choice with a required field missing. The separator "|" is missing in
%%   the choice field (a b)+ in the fourth elementdecl in the DTD.
'ibm-not-wf-P49-ibm49n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P49/ibm49n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P49/ibm49n05.xml
%% ID: ibm-not-wf-P49-ibm49n05.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests choice with an extra separator. An extra "|" occurs between a and b
%%   in the choice field in the fourth elementdecl in the DTD.
'ibm-not-wf-P49-ibm49n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P49/ibm49n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P49/ibm49n06.xml
%% ID: ibm-not-wf-P49-ibm49n06.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests choice with a required field missing. The closing bracket ")" is
%%   missing in the choice field (a |b * in the fourth elementdecl in the DTD.
'ibm-not-wf-P49-ibm49n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P49/ibm49n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n01.xml
%% ID: ibm-not-wf-P50-ibm50n01.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with a required field missing. The two cps are missing in the seq
%%   field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n02.xml
%% ID: ibm-not-wf-P50-ibm50n02.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with a required field missing. The third cp is missing in the seq
%%   field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n03.xml
%% ID: ibm-not-wf-P50-ibm50n03.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with a wrong separator. The "|" is used as the separator between
%%   a and b in the seq field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n04.xml
%% ID: ibm-not-wf-P50-ibm50n04.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with a wrong separator. The "." is used as the separator between
%%   a and b in the seq field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n05.xml
%% ID: ibm-not-wf-P50-ibm50n05.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with an extra separator. An extra "," occurs between (a|b) and a
%%   in the seq field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n06.xml
%% ID: ibm-not-wf-P50-ibm50n06.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with a required field missing. The separator between (a|b) and
%%   (b|a) is missing in the seq field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P50/ibm50n07.xml
%% ID: ibm-not-wf-P50-ibm50n07.xml
%% Type: not-wf
%% Sections: 3.2.1
%% Description:
%%   Tests seq with wrong closing bracket. The "]" is used as the closing
%%   bracket in the seq field in the fourth elementdecl in the DTD.
'ibm-not-wf-P50-ibm50n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P50/ibm50n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n01.xml
%% ID: ibm-not-wf-P51-ibm51n01.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with a wrong key word. The string "#pcdata" is used as the key
%%   word in the Mixed field in the fourth elementdecl in the DTD.
'ibm-not-wf-P51-ibm51n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n02.xml
%% ID: ibm-not-wf-P51-ibm51n02.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with wrong field ordering. The field #PCDATA does not occur as
%%   the first component in the Mixed field in the fourth elementdecl in the
%%   DTD.
'ibm-not-wf-P51-ibm51n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n03.xml
%% ID: ibm-not-wf-P51-ibm51n03.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with a separator missing. The separator "|" is missing in
%%   between #PCDATA and a in the Mixed field in the fourth elementdecl in the
%%   DTD.
'ibm-not-wf-P51-ibm51n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n04.xml
%% ID: ibm-not-wf-P51-ibm51n04.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with a wrong key word. The string "#CDATA" is used as the key
%%   word in the Mixed field in the fourth elementdecl in the DTD.
'ibm-not-wf-P51-ibm51n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n05.xml
%% ID: ibm-not-wf-P51-ibm51n05.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with a required field missing. The "*" is missing after the ")"
%%   in the Mixed field in the fourth elementdecl in the DTD.
'ibm-not-wf-P51-ibm51n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n06.xml
%% ID: ibm-not-wf-P51-ibm51n06.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with wrong closing bracket. The "]" is used as the closing
%%   bracket in the Mixed field in the fourth elementdecl in the DTD.
'ibm-not-wf-P51-ibm51n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P51/ibm51n07.xml
%% ID: ibm-not-wf-P51-ibm51n07.xml
%% Type: not-wf
%% Sections: 3.2.2
%% Description:
%%   Tests Mixed with a required field missing. The closing bracket ")" is
%%   missing after (#PCDATA in the Mixed field in the fourth elementdecl in the
%%   DTD.
'ibm-not-wf-P51-ibm51n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P51/ibm51n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P52/ibm52n01.xml
%% ID: ibm-not-wf-P52-ibm52n01.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttlistDecl with a required field missing. The Name is missing in the
%%   AttlistDecl in the DTD.
'ibm-not-wf-P52-ibm52n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P52/ibm52n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P52/ibm52n02.xml
%% ID: ibm-not-wf-P52-ibm52n02.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttlistDecl with a required field missing. The white space is missing
%%   between the beginning sequence and the name in the AttlistDecl in the DTD.
'ibm-not-wf-P52-ibm52n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P52/ibm52n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P52/ibm52n03.xml
%% ID: ibm-not-wf-P52-ibm52n03.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttlistDecl with wrong field ordering. The Name "a" occurs after the
%%   first AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P52-ibm52n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P52/ibm52n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P52/ibm52n04.xml
%% ID: ibm-not-wf-P52-ibm52n04.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttlistDecl with wrong key word. The string "Attlist" is used as the
%%   key word in the beginning sequence in the AttlistDecl in the DTD.
'ibm-not-wf-P52-ibm52n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P52/ibm52n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P52/ibm52n05.xml
%% ID: ibm-not-wf-P52-ibm52n05.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttlistDecl with a required field missing. The closing bracket
%%   "greater than" is missing in the AttlistDecl in the DTD.
'ibm-not-wf-P52-ibm52n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P52/ibm52n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P52/ibm52n06.xml
%% ID: ibm-not-wf-P52-ibm52n06.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttlistDecl with wrong beginning sequence. The string "(less
%%   than)ATTLIST" is used as the beginning sequence in the AttlistDecl in the
%%   DTD.
'ibm-not-wf-P52-ibm52n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P52/ibm52n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n01.xml
%% ID: ibm-not-wf-P53-ibm53n01.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with a required field missing. The DefaultDecl is missing in
%%   the AttDef for the name "attr1" in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n02.xml
%% ID: ibm-not-wf-P53-ibm53n02.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with a required field missing. The white space is missing
%%   between (abc|def) and "def" in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n03.xml
%% ID: ibm-not-wf-P53-ibm53n03.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with a required field missing. The AttType is missing for
%%   "attr1" in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n04.xml
%% ID: ibm-not-wf-P53-ibm53n04.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with a required field missing. The white space is missing
%%   between "attr1" and (abc|def) in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n05.xml
%% ID: ibm-not-wf-P53-ibm53n05.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with a required field missing. The Name is missing in the
%%   AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n06.xml
%% ID: ibm-not-wf-P53-ibm53n06.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with a required field missing. The white space before the name
%%   "attr2" is missing in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n07.xml
%% ID: ibm-not-wf-P53-ibm53n07.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with wrong field ordering. The Name "attr1" occurs after the
%%   AttType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P53/ibm53n08.xml
%% ID: ibm-not-wf-P53-ibm53n08.xml
%% Type: not-wf
%% Sections: 3.3
%% Description:
%%   Tests AttDef with wrong field ordering. The Name "attr1" occurs after the
%%   AttType and "default" occurs before the AttType in the AttDef in the
%%   AttlistDecl in the DTD.
'ibm-not-wf-P53-ibm53n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P53/ibm53n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P54/ibm54n01.xml
%% ID: ibm-not-wf-P54-ibm54n01.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests AttType with a wrong option. The string "BOGUSATTR" is used as the
%%   AttType in the AttlistDecl in the DTD.
'ibm-not-wf-P54-ibm54n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P54/ibm54n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P54/ibm54n02.xml
%% ID: ibm-not-wf-P54-ibm54n02.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests AttType with a wrong option. The string "PCDATA" is used as the
%%   AttType in the AttlistDecl in the DTD.
'ibm-not-wf-P54-ibm54n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P54/ibm54n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P55/ibm55n01.xml
%% ID: ibm-not-wf-P55-ibm55n01.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests StringType with a wrong key word. The lower case string "cdata" is
%%   used as the StringType in the AttType in the AttlistDecl in the DTD.
'ibm-not-wf-P55-ibm55n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P55/ibm55n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P55/ibm55n02.xml
%% ID: ibm-not-wf-P55-ibm55n02.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests StringType with a wrong key word. The string "#CDATA" is used as the
%%   StringType in the AttType in the AttlistDecl in the DTD.
'ibm-not-wf-P55-ibm55n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P55/ibm55n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P55/ibm55n03.xml
%% ID: ibm-not-wf-P55-ibm55n03.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests StringType with a wrong key word. The string "CData" is used as the
%%   StringType in the AttType in the AttlistDecl in the DTD.
'ibm-not-wf-P55-ibm55n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P55/ibm55n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n01.xml
%% ID: ibm-not-wf-P56-ibm56n01.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type "id" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n02.xml
%% ID: ibm-not-wf-P56-ibm56n02.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type "Idref" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n03.xml
%% ID: ibm-not-wf-P56-ibm56n03.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type"Idrefs" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n04.xml
%% ID: ibm-not-wf-P56-ibm56n04.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type "EntitY" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n05.xml
%% ID: ibm-not-wf-P56-ibm56n05.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type "nmTOKEN" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n06.xml
%% ID: ibm-not-wf-P56-ibm56n06.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type "NMtokens" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P56/ibm56n07.xml
%% ID: ibm-not-wf-P56-ibm56n07.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests TokenizedType with wrong key word. The type "#ID" is used in the
%%   TokenizedType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P56-ibm56n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P56/ibm56n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P57/ibm57n01.xml
%% ID: ibm-not-wf-P57-ibm57n01.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests EnumeratedType with an illegal option. The string "NMTOKEN (a|b)" is
%%   used in the EnumeratedType in the AttlistDecl in the DTD.
'ibm-not-wf-P57-ibm57n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P57/ibm57n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n01.xml
%% ID: ibm-not-wf-P58-ibm58n01.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with wrong key word. The lower case "notation" is used
%%   as the key word in the NotationType in the AttDef in the AttlistDecl in the
%%   DTD.
'ibm-not-wf-P58-ibm58n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n02.xml
%% ID: ibm-not-wf-P58-ibm58n02.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with a required field missing. The beginning bracket "("
%%   is missing in the NotationType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P58-ibm58n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n03.xml
%% ID: ibm-not-wf-P58-ibm58n03.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with a required field missing. The Name is missing in
%%   the "()" in the NotationType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P58-ibm58n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n04.xml
%% ID: ibm-not-wf-P58-ibm58n04.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with a required field missing. The closing bracket is
%%   missing in the NotationType in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P58-ibm58n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n05.xml
%% ID: ibm-not-wf-P58-ibm58n05.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with wrong field ordering. The key word "NOTATION"
%%   occurs after "(this)" in the NotationType in the AttDef in the AttlistDecl
%%   in the DTD.
'ibm-not-wf-P58-ibm58n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n06.xml
%% ID: ibm-not-wf-P58-ibm58n06.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with wrong separator. The "," is used as a separator
%%   between "this" and "that" in the NotationType in the AttDef in the
%%   AttlistDecl in the DTD.
'ibm-not-wf-P58-ibm58n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n07.xml
%% ID: ibm-not-wf-P58-ibm58n07.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with a required field missing. The white space is
%%   missing between "NOTATION" and "(this)" in the NotationType in the AttDef
%%   in the AttlistDecl in the DTD.
'ibm-not-wf-P58-ibm58n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P58/ibm58n08.xml
%% ID: ibm-not-wf-P58-ibm58n08.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests NotationType with extra wrong characters. The double quote character
%%   occurs after "(" and before ")" in the NotationType in the AttDef in the
%%   AttlistDecl in the DTD.
'ibm-not-wf-P58-ibm58n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P58/ibm58n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P59/ibm59n01.xml
%% ID: ibm-not-wf-P59-ibm59n01.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests Enumeration with required fields missing. The Nmtokens and "|"s are
%%   missing in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P59-ibm59n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P59/ibm59n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P59/ibm59n02.xml
%% ID: ibm-not-wf-P59-ibm59n02.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests Enumeration with a required field missing. The closing bracket ")" is
%%   missing in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P59-ibm59n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P59/ibm59n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P59/ibm59n03.xml
%% ID: ibm-not-wf-P59-ibm59n03.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests Enumeration with wrong separator. The "," is used as the separator in
%%   the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P59-ibm59n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P59/ibm59n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P59/ibm59n04.xml
%% ID: ibm-not-wf-P59-ibm59n04.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests Enumeration with illegal presence. The double quotes occur around the
%%   Enumeration value in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P59-ibm59n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P59/ibm59n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P59/ibm59n05.xml
%% ID: ibm-not-wf-P59-ibm59n05.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests Enumeration with a required field missing. The white space is missing
%%   between in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P59-ibm59n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P59/ibm59n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P59/ibm59n06.xml
%% ID: ibm-not-wf-P59-ibm59n06.xml
%% Type: not-wf
%% Sections: 3.3.1
%% Description:
%%   Tests Enumeration with a required field missing. The beginning bracket "("
%%   is missing in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P59-ibm59n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P59/ibm59n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n01.xml
%% ID: ibm-not-wf-P60-ibm60n01.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with wrong key word. The string "#required" is used as
%%   the key word in the DefaultDecl in the AttDef in the AttlistDecl in the
%%   DTD.
'ibm-not-wf-P60-ibm60n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n02.xml
%% ID: ibm-not-wf-P60-ibm60n02.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with wrong key word. The string "Implied" is used as the
%%   key word in the DefaultDecl in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n03.xml
%% ID: ibm-not-wf-P60-ibm60n03.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with wrong key word. The string "!IMPLIED" is used as the
%%   key word in the DefaultDecl in the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n04.xml
%% ID: ibm-not-wf-P60-ibm60n04.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with a required field missing. There is no attribute
%%   value specified after the key word "#FIXED" in the DefaultDecl in the
%%   AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n05.xml
%% ID: ibm-not-wf-P60-ibm60n05.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with a required field missing. The white space is missing
%%   between the key word "#FIXED" and the attribute value in the DefaultDecl in
%%   the AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n06.xml
%% ID: ibm-not-wf-P60-ibm60n06.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with wrong field ordering. The key word "#FIXED" occurs
%%   after the attribute value "introduction" in the DefaultDecl in the AttDef
%%   in the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n07.xml
%% ID: ibm-not-wf-P60-ibm60n07.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl against WFC of P60. The text replacement of the entity
%%   "avalue" contains the "less than" character in the DefaultDecl in the
%%   AttDef in the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P60/ibm60n08.xml
%% ID: ibm-not-wf-P60-ibm60n08.xml
%% Type: not-wf
%% Sections: 3.3.2
%% Description:
%%   Tests DefaultDecl with more than one key word. The "#REQUIRED" and the
%%   "#IMPLIED" are used as the key words in the DefaultDecl in the AttDef in
%%   the AttlistDecl in the DTD.
'ibm-not-wf-P60-ibm60n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P60/ibm60n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P61/ibm61n01.xml
%% ID: ibm-not-wf-P61-ibm61n01.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests conditionalSect with a wrong option. The word "NOTINCLUDE" is used as
%%   part of an option which is wrong in the coditionalSect.
'ibm-not-wf-P61-ibm61n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P61/ibm61n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n01.xml
%% ID: ibm-not-wf-P62-ibm62n01.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with wrong key word. The string "include" is used as a
%%   key word in the beginning sequence in the includeSect in the file
%%   ibm62n01.dtd.
'ibm-not-wf-P62-ibm62n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n02.xml
%% ID: ibm-not-wf-P62-ibm62n02.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with wrong beginning sequence. An extra "[" occurs in the
%%   beginning sequence in the includeSect in the file ibm62n02.dtd.
'ibm-not-wf-P62-ibm62n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n03.xml
%% ID: ibm-not-wf-P62-ibm62n03.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with wrong beginning sequence. A wrong character "?"
%%   occurs in the beginning sequence in the includeSect in the file
%%   ibm62n03.dtd.
'ibm-not-wf-P62-ibm62n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n04.xml
%% ID: ibm-not-wf-P62-ibm62n04.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with a required field missing. The key word "INCLUDE" is
%%   missing in the includeSect in the file ibm62n04.dtd.
'ibm-not-wf-P62-ibm62n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n05.xml
%% ID: ibm-not-wf-P62-ibm62n05.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with a required field missing. The "[" is missing after
%%   the key word "INCLUDE" in the includeSect in the file ibm62n05.dtd.
'ibm-not-wf-P62-ibm62n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n06.xml
%% ID: ibm-not-wf-P62-ibm62n06.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with wrong field ordering. The two external subset
%%   declarations occur before the key word "INCLUDE" in the includeSect in the
%%   file ibm62n06.dtd.
'ibm-not-wf-P62-ibm62n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n07.xml
%% ID: ibm-not-wf-P62-ibm62n07.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with a required field missing. The closing sequence
%%   "]](greater than)" is missing in the includeSect in the file ibm62n07.dtd.
'ibm-not-wf-P62-ibm62n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P62/ibm62n08.xml
%% ID: ibm-not-wf-P62-ibm62n08.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with a required field missing. One "]" is missing in the
%%   closing sequence in the includeSect in the file ibm62n08.dtd.
'ibm-not-wf-P62-ibm62n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P62/ibm62n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n01.xml
%% ID: ibm-not-wf-P63-ibm63n01.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSect with wrong key word. The string "ignore" is used as a key
%%   word in the beginning sequence in the ignoreSect in the file ibm63n01.dtd.
'ibm-not-wf-P63-ibm63n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n02.xml
%% ID: ibm-not-wf-P63-ibm63n02.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSect with wrong beginning sequence. An extra "[" occurs in the
%%   beginning sequence in the ignoreSect in the file ibm63n02.dtd.
'ibm-not-wf-P63-ibm63n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n03.xml
%% ID: ibm-not-wf-P63-ibm63n03.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSect with wrong beginning sequence. A wrong character "?"
%%   occurs in the beginning sequence in the ignoreSect in the file
%%   ibm63n03.dtd.
'ibm-not-wf-P63-ibm63n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n04.xml
%% ID: ibm-not-wf-P63-ibm63n04.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSect with a required field missing. The key word "IGNORE" is
%%   missing in the ignoreSect in the file ibm63n04.dtd.
'ibm-not-wf-P63-ibm63n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n05.xml
%% ID: ibm-not-wf-P63-ibm63n05.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSect with a required field missing. The "[" is missing after
%%   the key word "IGNORE" in the ignoreSect in the file ibm63n05.dtd.
'ibm-not-wf-P63-ibm63n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n06.xml
%% ID: ibm-not-wf-P63-ibm63n06.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests includeSect with wrong field ordering. The two external subset
%%   declarations occur before the key word "IGNORE" in the ignoreSect in the
%%   file ibm63n06.dtd.
'ibm-not-wf-P63-ibm63n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P63/ibm63n07.xml
%% ID: ibm-not-wf-P63-ibm63n07.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSect with a required field missing. The closing sequence
%%   "]](greater than)" is missing in the ignoreSect in the file ibm63n07.dtd.
'ibm-not-wf-P63-ibm63n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P63/ibm63n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P64/ibm64n01.xml
%% ID: ibm-not-wf-P64-ibm64n01.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSectContents with wrong beginning sequence. The "?" occurs in
%%   beginning sequence the ignoreSectContents in the file ibm64n01.dtd.
'ibm-not-wf-P64-ibm64n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P64/ibm64n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P64/ibm64n02.xml
%% ID: ibm-not-wf-P64-ibm64n02.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSectContents with a required field missing.The closing sequence
%%   is missing in the ignoreSectContents in the file ibm64n02.dtd.
'ibm-not-wf-P64-ibm64n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P64/ibm64n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P64/ibm64n03.xml
%% ID: ibm-not-wf-P64-ibm64n03.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests ignoreSectContents with a required field missing.The beginning
%%   sequence is missing in the ignoreSectContents in the file ibm64n03.dtd.
'ibm-not-wf-P64-ibm64n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P64/ibm64n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P65/ibm65n01.xml
%% ID: ibm-not-wf-P65-ibm65n01.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests Ignore with illegal string included. The string "]](greater than)" is
%%   contained before "this" in the Ignore in the ignoreSectContents in the file
%%   ibm65n01.dtd
'ibm-not-wf-P65-ibm65n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P65/ibm65n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P65/ibm65n02.xml
%% ID: ibm-not-wf-P65-ibm65n02.xml
%% Type: not-wf
%% Sections: 3.4
%% Entities: parameter
%% Description:
%%   Tests Ignore with illegal string included. The string "(less than)![" is
%%   contained before "this" in the Ignore in the ignoreSectContents in the file
%%   ibm65n02.dtd
'ibm-not-wf-P65-ibm65n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P65/ibm65n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n01.xml
%% ID: ibm-not-wf-P66-ibm66n01.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#002f" is used as
%%   the referred character in the CharRef in the EntityDecl in the DTD.
'ibm-not-wf-P66-ibm66n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n02.xml
%% ID: ibm-not-wf-P66-ibm66n02.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with the semicolon character missing. The semicolon character
%%   is missing at the end of the CharRef in the attribute value in the STag of
%%   element "root".
'ibm-not-wf-P66-ibm66n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n03.xml
%% ID: ibm-not-wf-P66-ibm66n03.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "49" is used as
%%   the referred character in the CharRef in the EntityDecl in the DTD.
'ibm-not-wf-P66-ibm66n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n04.xml
%% ID: ibm-not-wf-P66-ibm66n04.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#5~0" is used as
%%   the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n05.xml
%% ID: ibm-not-wf-P66-ibm66n05.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#x002g" is used
%%   as the referred character in the CharRef in the EntityDecl in the DTD.
'ibm-not-wf-P66-ibm66n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n06.xml
%% ID: ibm-not-wf-P66-ibm66n06.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#x006G" is used
%%   as the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n07.xml
%% ID: ibm-not-wf-P66-ibm66n07.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#0=2f" is used as
%%   the referred character in the CharRef in the EntityDecl in the DTD.
'ibm-not-wf-P66-ibm66n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n08.xml
%% ID: ibm-not-wf-P66-ibm66n08.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#56.0" is used as
%%   the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n09.xml
%% ID: ibm-not-wf-P66-ibm66n09.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#x00/2f" is used
%%   as the referred character in the CharRef in the EntityDecl in the DTD.
'ibm-not-wf-P66-ibm66n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n10.xml
%% ID: ibm-not-wf-P66-ibm66n10.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#51)" is used as
%%   the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n11.xml
%% ID: ibm-not-wf-P66-ibm66n11.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#00 2f" is used
%%   as the referred character in the CharRef in the EntityDecl in the DTD.
'ibm-not-wf-P66-ibm66n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n12.xml
%% ID: ibm-not-wf-P66-ibm66n12.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#x0000" is used
%%   as the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n13.xml
%% ID: ibm-not-wf-P66-ibm66n13.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#x001f" is used
%%   as the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n14.xml
%% ID: ibm-not-wf-P66-ibm66n14.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#xfffe" is used
%%   as the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P66/ibm66n15.xml
%% ID: ibm-not-wf-P66-ibm66n15.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests CharRef with an illegal character referred to. The "#xffff" is used
%%   as the referred character in the attribute value in the EmptyElemTag of the
%%   element "root".
'ibm-not-wf-P66-ibm66n15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P66/ibm66n15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n01.xml
%% ID: ibm-not-wf-P68-ibm68n01.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef with a required field missing. The Name is missing in the
%%   EntityRef in the content of the element "root".
'ibm-not-wf-P68-ibm68n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n02.xml
%% ID: ibm-not-wf-P68-ibm68n02.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef with a required field missing. The semicolon is missing in
%%   the EntityRef in the attribute value in the element "root".
'ibm-not-wf-P68-ibm68n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n03.xml
%% ID: ibm-not-wf-P68-ibm68n03.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef with an extra white space. A white space occurs after the
%%   ampersand in the EntityRef in the content of the element "root".
'ibm-not-wf-P68-ibm68n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n04.xml
%% ID: ibm-not-wf-P68-ibm68n04.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef which is against P68 WFC: Entity Declared. The name "aAa"
%%   in the EntityRef in the AttValue in the STage of the element "root" does
%%   not match the Name of any declared entity in the DTD.
'ibm-not-wf-P68-ibm68n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n05.xml
%% ID: ibm-not-wf-P68-ibm68n05.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef which is against P68 WFC: Entity Declared. The entity with
%%   the name "aaa" in the EntityRef in the AttValue in the STag of the element
%%   "root" is not declared.
'ibm-not-wf-P68-ibm68n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n06.xml
%% ID: ibm-not-wf-P68-ibm68n06.xml
%% Type: not-wf
%% Sections: 4.1
%% Entities: parameter
%% Description:
%%   Tests EntityRef which is against P68 WFC: Entity Declared. The entity with
%%   the name "aaa" in the EntityRef in the AttValue in the STag of the element
%%   "root" is externally declared, but standalone is "yes".
'ibm-not-wf-P68-ibm68n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n07.xml
%% ID: ibm-not-wf-P68-ibm68n07.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef which is against P68 WFC: Entity Declared. The entity with
%%   the name "aaa" in the EntityRef in the AttValue in the STag of the element
%%   "root" is referred before declared.
'ibm-not-wf-P68-ibm68n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n08.xml
%% ID: ibm-not-wf-P68-ibm68n08.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef which is against P68 WFC: Parsed Entity. The EntityRef in
%%   the AttValue in the STag of the element "root" contains the name "aImage"
%%   of an unparsed entity.
'ibm-not-wf-P68-ibm68n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n09.xml
%% ID: ibm-not-wf-P68-ibm68n09.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef which is against P68 WFC: No Recursion. The recursive
%%   entity reference occurs with the entity declarations for "aaa" and "bbb" in
%%   the DTD.
'ibm-not-wf-P68-ibm68n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P68/ibm68n10.xml
%% ID: ibm-not-wf-P68-ibm68n10.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests EntityRef which is against P68 WFC: No Recursion. The indirect
%%   recursive entity reference occurs with the entity declarations for "aaa",
%%   "bbb", "ccc", "ddd", and "eee" in the DTD.
'ibm-not-wf-P68-ibm68n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P68/ibm68n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n01.xml
%% ID: ibm-not-wf-P69-ibm69n01.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests PEReference with a required field missing. The Name "paaa" is missing
%%   in the PEReference in the DTD.
'ibm-not-wf-P69-ibm69n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n02.xml
%% ID: ibm-not-wf-P69-ibm69n02.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests PEReference with a required field missing. The semicolon is missing
%%   in the PEReference "%paaa" in the DTD.
'ibm-not-wf-P69-ibm69n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n03.xml
%% ID: ibm-not-wf-P69-ibm69n03.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests PEReference with an extra white space. There is an extra white space
%%   occurs before ";" in the PEReference in the DTD.
'ibm-not-wf-P69-ibm69n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n04.xml
%% ID: ibm-not-wf-P69-ibm69n04.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests PEReference with an extra white space. There is an extra white space
%%   occurs after "%" in the PEReference in the DTD.
'ibm-not-wf-P69-ibm69n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n05.xml
%% ID: ibm-not-wf-P69-ibm69n05.xml
%% Type: error
%% Sections: 4.1
%% Description:
%%   Based on E29 substantial source: minutes XML-Syntax 1999-02-24 E38 in XML
%%   1.0 Errata, this WFC does not apply to P69, but the VC Entity declared
%%   still apply. Tests PEReference which is against P69 WFC: Entity Declared.
%%   The PE with the name "paaa" is referred before declared in the DTD.
'ibm-not-wf-P69-ibm69n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "error").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n06.xml
%% ID: ibm-not-wf-P69-ibm69n06.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests PEReference which is against P69 WFC: No Recursion. The recursive PE
%%   reference occurs with the entity declarations for "paaa" and "bbb" in the
%%   DTD.
'ibm-not-wf-P69-ibm69n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P69/ibm69n07.xml
%% ID: ibm-not-wf-P69-ibm69n07.xml
%% Type: not-wf
%% Sections: 4.1
%% Description:
%%   Tests PEReference which is against P69 WFC: No Recursion. The indirect
%%   recursive PE reference occurs with the entity declarations for "paaa",
%%   "bbb", "ccc", "ddd", and "eee" in the DTD.
'ibm-not-wf-P69-ibm69n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P69/ibm69n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm70n01.xml
%% ID: ibm-not-wf-P71-ibm70n01.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests
'ibm-not-wf-P71-ibm70n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm70n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n01.xml
%% ID: ibm-not-wf-P71-ibm71n01.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with a required field missing. The white space is missing
%%   between the beginning sequence and the Name "aaa" in the EntityDecl in the
%%   DTD.
'ibm-not-wf-P71-ibm71n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n02.xml
%% ID: ibm-not-wf-P71-ibm71n02.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with a required field missing. The white space is missing
%%   between the Name "aaa" and the EntityDef "aString" in the EntityDecl in the
%%   DTD.
'ibm-not-wf-P71-ibm71n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n03.xml
%% ID: ibm-not-wf-P71-ibm71n03.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with a required field missing. The EntityDef is missing in
%%   the EntityDecl with the Name "aaa" in the DTD.
'ibm-not-wf-P71-ibm71n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n04.xml
%% ID: ibm-not-wf-P71-ibm71n04.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with a required field missing. The Name is missing in the
%%   EntityDecl with the EntityDef "aString" in the DTD.
'ibm-not-wf-P71-ibm71n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n05.xml
%% ID: ibm-not-wf-P71-ibm71n05.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with wrong ordering. The Name "aaa" occurs after the
%%   EntityDef in the EntityDecl in the DTD.
'ibm-not-wf-P71-ibm71n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n06.xml
%% ID: ibm-not-wf-P71-ibm71n06.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with wrong key word. The string "entity" is used as the
%%   key word in the beginning sequence in the EntityDecl in the DTD.
'ibm-not-wf-P71-ibm71n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n07.xml
%% ID: ibm-not-wf-P71-ibm71n07.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with a required field missing. The closing bracket
%%   (greater than) is missing in the EntityDecl in the DTD.
'ibm-not-wf-P71-ibm71n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P71/ibm71n08.xml
%% ID: ibm-not-wf-P71-ibm71n08.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDecl with a required field missing. The exclamation mark is
%%   missing in the beginning sequence in the EntityDecl in the DTD.
'ibm-not-wf-P71-ibm71n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P71/ibm71n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n01.xml
%% ID: ibm-not-wf-P72-ibm72n01.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with a required field missing. The white space is missing
%%   between the beginning sequence and the "%" in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n02.xml
%% ID: ibm-not-wf-P72-ibm72n02.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with a required field missing. The Name is missing in the
%%   PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n03.xml
%% ID: ibm-not-wf-P72-ibm72n03.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with a required field missing. The white space is missing
%%   between the Name and the PEDef in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n04.xml
%% ID: ibm-not-wf-P72-ibm72n04.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with a required field missing. The PEDef is missing after the
%%   Name "paaa" in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n05.xml
%% ID: ibm-not-wf-P72-ibm72n05.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with wrong field ordering. The Name "paaa" occurs after the
%%   PEDef in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n06.xml
%% ID: ibm-not-wf-P72-ibm72n06.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with wrong field ordering. The "%" and the Name "paaa" occurs
%%   after the PEDef in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n07.xml
%% ID: ibm-not-wf-P72-ibm72n07.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with wrong key word. The string "entity" is used as the key
%%   word in the beginning sequence in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n08.xml
%% ID: ibm-not-wf-P72-ibm72n08.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with a required field missing. The closing bracket (greater
%%   than) is missing in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P72/ibm72n09.xml
%% ID: ibm-not-wf-P72-ibm72n09.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEdecl with wrong closing sequence. The string "!(greater than)" is
%%   used as the closing sequence in the PEDecl in the DTD.
'ibm-not-wf-P72-ibm72n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P72/ibm72n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P73/ibm73n01.xml
%% ID: ibm-not-wf-P73-ibm73n01.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDef with wrong field ordering. The NDataDecl "NDATA JPGformat"
%%   occurs before the ExternalID in the EntityDef in the EntityDecl.
'ibm-not-wf-P73-ibm73n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P73/ibm73n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P73/ibm73n03.xml
%% ID: ibm-not-wf-P73-ibm73n03.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests EntityDef with a required field missing. The ExternalID is missing
%%   before the NDataDecl in the EntityDef in the EntityDecl.
'ibm-not-wf-P73-ibm73n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P73/ibm73n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P74/ibm74n01.xml
%% ID: ibm-not-wf-P74-ibm74n01.xml
%% Type: not-wf
%% Sections: 4.2
%% Description:
%%   Tests PEDef with extra fields. The NDataDecl occurs after the ExternalID in
%%   the PEDef in the PEDecl in the DTD.
'ibm-not-wf-P74-ibm74n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P74/ibm74n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n01.xml
%% ID: ibm-not-wf-P75-ibm75n01.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with wrong key word. The string "system" is used as the
%%   key word in the ExternalID in the EntityDef in the EntityDecl.
'ibm-not-wf-P75-ibm75n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n02.xml
%% ID: ibm-not-wf-P75-ibm75n02.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with wrong key word. The string "public" is used as the
%%   key word in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n03.xml
%% ID: ibm-not-wf-P75-ibm75n03.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with wrong key word. The string "Public" is used as the
%%   key word in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n04.xml
%% ID: ibm-not-wf-P75-ibm75n04.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with wrong field ordering. The key word "PUBLIC" occurs
%%   after the PublicLiteral and the SystemLiteral in the ExternalID in the
%%   doctypedecl.
'ibm-not-wf-P75-ibm75n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n05.xml
%% ID: ibm-not-wf-P75-ibm75n05.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The white space between
%%   "SYSTEM" and the Systemliteral is missing in the ExternalID in the
%%   EntityDef in the EntityDecl in the DTD.
'ibm-not-wf-P75-ibm75n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n06.xml
%% ID: ibm-not-wf-P75-ibm75n06.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The Systemliteral is
%%   missing after "SYSTEM" in the ExternalID in the EntityDef in the EntityDecl
%%   in the DTD.
'ibm-not-wf-P75-ibm75n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n07.xml
%% ID: ibm-not-wf-P75-ibm75n07.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The white space between the
%%   PublicLiteral and the Systemliteral is missing in the ExternalID in the
%%   doctypedecl.
'ibm-not-wf-P75-ibm75n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n08.xml
%% ID: ibm-not-wf-P75-ibm75n08.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The key word "PUBLIC" is
%%   missing in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n09.xml
%% ID: ibm-not-wf-P75-ibm75n09.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The white space between
%%   "PUBLIC" and the PublicLiteral is missing in the ExternalID in the
%%   doctypedecl.
'ibm-not-wf-P75-ibm75n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n10.xml
%% ID: ibm-not-wf-P75-ibm75n10.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The PublicLiteral is
%%   missing in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n11.xml
%% ID: ibm-not-wf-P75-ibm75n11.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The PublicLiteral is
%%   missing in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n12.xml
%% ID: ibm-not-wf-P75-ibm75n12.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with a required field missing. The SystemLiteral is
%%   missing in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P75/ibm75n13.xml
%% ID: ibm-not-wf-P75-ibm75n13.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests ExternalID with wrong field ordering. The key word "PUBLIC" occurs
%%   after the PublicLiteral in the ExternalID in the doctypedecl.
'ibm-not-wf-P75-ibm75n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P75/ibm75n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n01.xml
%% ID: ibm-not-wf-P76-ibm76n01.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with wrong key word. The string "ndata" is used as the key
%%   word in the NDataDecl in the EntityDef in the GEDecl.
'ibm-not-wf-P76-ibm76n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n02.xml
%% ID: ibm-not-wf-P76-ibm76n02.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with wrong key word. The string "NData" is used as the key
%%   word in the NDataDecl in the EntityDef in the GEDecl.
'ibm-not-wf-P76-ibm76n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n03.xml
%% ID: ibm-not-wf-P76-ibm76n03.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with a required field missing. The leading white space is
%%   missing in the NDataDecl in the EntityDef in the GEDecl.
'ibm-not-wf-P76-ibm76n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n04.xml
%% ID: ibm-not-wf-P76-ibm76n04.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with a required field missing. The key word "NDATA" is
%%   missing in the NDataDecl in the EntityDef in the GEDecl.
'ibm-not-wf-P76-ibm76n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n05.xml
%% ID: ibm-not-wf-P76-ibm76n05.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with a required field missing. The Name after the key word
%%   "NDATA" is missing in the NDataDecl in the EntityDef in the GEDecl.
'ibm-not-wf-P76-ibm76n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n06.xml
%% ID: ibm-not-wf-P76-ibm76n06.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with a required field missing. The white space between
%%   "NDATA" and the Name is missing in the NDataDecl in the EntityDef in the
%%   GEDecl.
'ibm-not-wf-P76-ibm76n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P76/ibm76n07.xml
%% ID: ibm-not-wf-P76-ibm76n07.xml
%% Type: not-wf
%% Sections: 4.2.2
%% Description:
%%   Tests NDataDecl with wrong field ordering. The key word "NDATA" occurs
%%   after the Name in the NDataDecl in the EntityDef in the GEDecl.
'ibm-not-wf-P76-ibm76n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P76/ibm76n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P77/ibm77n01.xml
%% ID: ibm-not-wf-P77-ibm77n01.xml
%% Type: not-wf
%% Sections: 4.3.1
%% Entities: general
%% Description:
%%   Tests TextDecl with wrong field ordering. The VersionInfo occurs after the
%%   EncodingDecl in the TextDecl in the file "ibm77n01.ent".
'ibm-not-wf-P77-ibm77n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P77/ibm77n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P77/ibm77n02.xml
%% ID: ibm-not-wf-P77-ibm77n02.xml
%% Type: not-wf
%% Sections: 4.3.1
%% Entities: general
%% Description:
%%   Tests TextDecl with wrong key word. The string "XML" is used in the
%%   beginning sequence in the TextDecl in the file "ibm77n02.ent".
'ibm-not-wf-P77-ibm77n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P77/ibm77n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P77/ibm77n03.xml
%% ID: ibm-not-wf-P77-ibm77n03.xml
%% Type: not-wf
%% Sections: 4.3.1
%% Entities: parameter
%% Description:
%%   Tests TextDecl with wrong closing sequence. The character "greater than" is
%%   used as the closing sequence in the TextDecl in the file "ibm77n03.ent".
'ibm-not-wf-P77-ibm77n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P77/ibm77n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P77/ibm77n04.xml
%% ID: ibm-not-wf-P77-ibm77n04.xml
%% Type: not-wf
%% Sections: 4.3.1
%% Entities: parameter
%% Description:
%%   Tests TextDecl with a required field missing. The closing sequence is
%%   missing in the TextDecl in the file "ibm77n04.ent".
'ibm-not-wf-P77-ibm77n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P77/ibm77n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P78/ibm78n01.xml
%% ID: ibm-not-wf-P78-ibm78n01.xml
%% Type: not-wf
%% Sections: 4.3.2
%% Entities: general
%% Description:
%%   Tests extParsedEnt with wrong field ordering. The TextDecl occurs after the
%%   content in the file ibm78n01.ent.
'ibm-not-wf-P78-ibm78n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P78/ibm78n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P78/ibm78n02.xml
%% ID: ibm-not-wf-P78-ibm78n02.xml
%% Type: not-wf
%% Sections: 4.3.2
%% Entities: general
%% Description:
%%   Tests extParsedEnt with extra field. A blank line occurs before the
%%   TextDecl in the file ibm78n02.ent.
'ibm-not-wf-P78-ibm78n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P78/ibm78n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P79/ibm79n01.xml
%% ID: ibm-not-wf-P79-ibm79n01.xml
%% Type: not-wf
%% Sections: 4.3.2
%% Entities: parameter
%% Description:
%%   Tests extPE with wrong field ordering. The TextDecl occurs after the
%%   extSubsetDecl (the white space and the comment) in the file ibm79n01.ent.
'ibm-not-wf-P79-ibm79n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P79/ibm79n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P79/ibm79n02.xml
%% ID: ibm-not-wf-P79-ibm79n02.xml
%% Type: not-wf
%% Sections: 4.3.2
%% Entities: parameter
%% Description:
%%   Tests extPE with extra field. A blank line occurs before the TextDecl in
%%   the file ibm78n02.ent.
'ibm-not-wf-P79-ibm79n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P79/ibm79n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P80/ibm80n01.xml
%% ID: ibm-not-wf-P80-ibm80n01.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncodingDecl with a required field missing. The leading white space
%%   is missing in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P80-ibm80n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P80/ibm80n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P80/ibm80n02.xml
%% ID: ibm-not-wf-P80-ibm80n02.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncodingDecl with a required field missing. The "=" sign is missing
%%   in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P80-ibm80n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P80/ibm80n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P80/ibm80n03.xml
%% ID: ibm-not-wf-P80-ibm80n03.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncodingDecl with a required field missing. The double quoted EncName
%%   are missing in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P80-ibm80n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P80/ibm80n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P80/ibm80n04.xml
%% ID: ibm-not-wf-P80-ibm80n04.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncodingDecl with wrong field ordering. The string "encoding=" occurs
%%   after the double quoted EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P80-ibm80n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P80/ibm80n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P80/ibm80n05.xml
%% ID: ibm-not-wf-P80-ibm80n05.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncodingDecl with wrong field ordering. The "encoding" occurs after
%%   the double quoted EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P80-ibm80n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P80/ibm80n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P80/ibm80n06.xml
%% ID: ibm-not-wf-P80-ibm80n06.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncodingDecl with wrong key word. The string "Encoding" is used as
%%   the key word in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P80-ibm80n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P80/ibm80n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n01.xml
%% ID: ibm-not-wf-P81-ibm81n01.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The "_" is used as the first
%%   character in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n02.xml
%% ID: ibm-not-wf-P81-ibm81n02.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The "-" is used as the first
%%   character in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n03.xml
%% ID: ibm-not-wf-P81-ibm81n03.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The "." is used as the first
%%   character in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n04.xml
%% ID: ibm-not-wf-P81-ibm81n04.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with illegal characters. The "8-" is used as the initial
%%   characters in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n05.xml
%% ID: ibm-not-wf-P81-ibm81n05.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The "~" is used as one character
%%   in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n06.xml
%% ID: ibm-not-wf-P81-ibm81n06.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The "#" is used as one character
%%   in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n07.xml
%% ID: ibm-not-wf-P81-ibm81n07.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The ":" is used as one character
%%   in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n08.xml
%% ID: ibm-not-wf-P81-ibm81n08.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The "/" is used as one character
%%   in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P81/ibm81n09.xml
%% ID: ibm-not-wf-P81-ibm81n09.xml
%% Type: not-wf
%% Sections: 4.3.3
%% Description:
%%   Tests EncName with an illegal character. The ";" is used as one character
%%   in the EncName in the EncodingDecl in the XMLDecl.
'ibm-not-wf-P81-ibm81n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P81/ibm81n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n01.xml
%% ID: ibm-not-wf-P82-ibm82n01.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with a required field missing. The white space after the
%%   beginning sequence of the NotationDecl is missing in the DTD.
'ibm-not-wf-P82-ibm82n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n02.xml
%% ID: ibm-not-wf-P82-ibm82n02.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with a required field missing. The Name in the
%%   NotationDecl is missing in the DTD.
'ibm-not-wf-P82-ibm82n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n03.xml
%% ID: ibm-not-wf-P82-ibm82n03.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with a required field missing. The externalID or the
%%   PublicID is missing in the NotationDecl in the DTD.
'ibm-not-wf-P82-ibm82n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n04.xml
%% ID: ibm-not-wf-P82-ibm82n04.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with wrong field ordering. The Name occurs after the
%%   "SYSTEM" and the externalID in the NotationDecl in the DTD.
'ibm-not-wf-P82-ibm82n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n05.xml
%% ID: ibm-not-wf-P82-ibm82n05.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with wrong key word. The string "notation" is used as a
%%   key word in the NotationDecl in the DTD.
'ibm-not-wf-P82-ibm82n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n06.xml
%% ID: ibm-not-wf-P82-ibm82n06.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with a required field missing. The closing bracket (the
%%   greater than character) is missing in the NotationDecl.
'ibm-not-wf-P82-ibm82n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n07.xml
%% ID: ibm-not-wf-P82-ibm82n07.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with wrong beginning sequence. The "!" is missing in the
%%   beginning sequence in the NotationDecl in the DTD.
'ibm-not-wf-P82-ibm82n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P82/ibm82n08.xml
%% ID: ibm-not-wf-P82-ibm82n08.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests NotationDecl with wrong closing sequence. The extra "!" occurs in the
%%   closing sequence in the NotationDecl in the DTD.
'ibm-not-wf-P82-ibm82n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P82/ibm82n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P83/ibm83n01.xml
%% ID: ibm-not-wf-P83-ibm83n01.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests PublicID with wrong key word. The string "public" is used as the key
%%   word in the PublicID in the NotationDecl in the DTD.
'ibm-not-wf-P83-ibm83n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P83/ibm83n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P83/ibm83n02.xml
%% ID: ibm-not-wf-P83-ibm83n02.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests PublicID with wrong key word. The string "Public" is used as the key
%%   word in the PublicID in the NotationDecl in the DTD.
'ibm-not-wf-P83-ibm83n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P83/ibm83n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P83/ibm83n03.xml
%% ID: ibm-not-wf-P83-ibm83n03.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests PublicID with a required field missing. The key word "PUBLIC" is
%%   missing in the PublicID in the NotationDecl in the DTD.
'ibm-not-wf-P83-ibm83n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P83/ibm83n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P83/ibm83n04.xml
%% ID: ibm-not-wf-P83-ibm83n04.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests PublicID with a required field missing. The white space between the
%%   "PUBLIC" and the PubidLiteral is missing in the PublicID in the
%%   NotationDecl in the DTD.
'ibm-not-wf-P83-ibm83n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P83/ibm83n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P83/ibm83n05.xml
%% ID: ibm-not-wf-P83-ibm83n05.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests PublicID with a required field missing. The PubidLiteral is missing
%%   in the PublicID in the NotationDecl in the DTD.
'ibm-not-wf-P83-ibm83n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P83/ibm83n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P83/ibm83n06.xml
%% ID: ibm-not-wf-P83-ibm83n06.xml
%% Type: not-wf
%% Sections: 4.7
%% Description:
%%   Tests PublicID with wrong field ordering. The key word "PUBLIC" occurs
%%   after the PubidLiteral in the PublicID in the NotationDecl.
'ibm-not-wf-P83-ibm83n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P83/ibm83n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n01.xml
%% ID: ibm-not-wf-P85-ibm85n01.xml
%% Type: not-wf
%% Sections: B.
%% Description:
%%   Tests BaseChar with an illegal character. The character #x00D7 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n02.xml
%% ID: ibm-not-wf-P85-ibm85n02.xml
%% Type: not-wf
%% Sections: B.
%% Description:
%%   Tests BaseChar with an illegal character. The character #x00F7 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n03.xml
%% ID: ibm-not-wf-P85-ibm85n03.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0132 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n04.xml
%% ID: ibm-not-wf-P85-ibm85n04.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0133 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n05.xml
%% ID: ibm-not-wf-P85-ibm85n05.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x013F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n06.xml
%% ID: ibm-not-wf-P85-ibm85n06.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0140 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n07.xml
%% ID: ibm-not-wf-P85-ibm85n07.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0149 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n08.xml
%% ID: ibm-not-wf-P85-ibm85n08.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x017F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n09.xml
%% ID: ibm-not-wf-P85-ibm85n09.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01c4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n10.xml
%% ID: ibm-not-wf-P85-ibm85n10.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01CC occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n100.xml
%% ID: ibm-not-wf-P85-ibm85n100.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0BB6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n100'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n100.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n101.xml
%% ID: ibm-not-wf-P85-ibm85n101.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0BBA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n101'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n101.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n102.xml
%% ID: ibm-not-wf-P85-ibm85n102.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C0D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n102'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n102.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n103.xml
%% ID: ibm-not-wf-P85-ibm85n103.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C11 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n103'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n103.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n104.xml
%% ID: ibm-not-wf-P85-ibm85n104.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C29 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n104'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n104.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n105.xml
%% ID: ibm-not-wf-P85-ibm85n105.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C34 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n105'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n105.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n106.xml
%% ID: ibm-not-wf-P85-ibm85n106.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C5F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n106'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n106.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n107.xml
%% ID: ibm-not-wf-P85-ibm85n107.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C62 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n107'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n107.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n108.xml
%% ID: ibm-not-wf-P85-ibm85n108.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C8D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n108'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n108.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n109.xml
%% ID: ibm-not-wf-P85-ibm85n109.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0C91 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n109'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n109.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n11.xml
%% ID: ibm-not-wf-P85-ibm85n11.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01F1 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n110.xml
%% ID: ibm-not-wf-P85-ibm85n110.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0CA9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n110'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n110.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n111.xml
%% ID: ibm-not-wf-P85-ibm85n111.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0CB4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n111'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n111.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n112.xml
%% ID: ibm-not-wf-P85-ibm85n112.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0CBA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n112'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n112.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n113.xml
%% ID: ibm-not-wf-P85-ibm85n113.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0CDF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n113'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n113.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n114.xml
%% ID: ibm-not-wf-P85-ibm85n114.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0CE2 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n114'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n114.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n115.xml
%% ID: ibm-not-wf-P85-ibm85n115.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0D0D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n115'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n115.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n116.xml
%% ID: ibm-not-wf-P85-ibm85n116.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0D11 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n116'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n116.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n117.xml
%% ID: ibm-not-wf-P85-ibm85n117.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0D29 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n117'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n117.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n118.xml
%% ID: ibm-not-wf-P85-ibm85n118.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0D3A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n118'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n118.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n119.xml
%% ID: ibm-not-wf-P85-ibm85n119.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0D62 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n119'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n119.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n12.xml
%% ID: ibm-not-wf-P85-ibm85n12.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01F3 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n120.xml
%% ID: ibm-not-wf-P85-ibm85n120.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E2F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n120'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n120.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n121.xml
%% ID: ibm-not-wf-P85-ibm85n121.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E31 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n121'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n121.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n122.xml
%% ID: ibm-not-wf-P85-ibm85n122.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E34 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n122'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n122.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n123.xml
%% ID: ibm-not-wf-P85-ibm85n123.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E46 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n123'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n123.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n124.xml
%% ID: ibm-not-wf-P85-ibm85n124.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E83 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n124'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n124.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n125.xml
%% ID: ibm-not-wf-P85-ibm85n125.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E85 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n125'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n125.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n126.xml
%% ID: ibm-not-wf-P85-ibm85n126.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E89 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n126'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n126.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n127.xml
%% ID: ibm-not-wf-P85-ibm85n127.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E8B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n127'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n127.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n128.xml
%% ID: ibm-not-wf-P85-ibm85n128.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E8E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n128'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n128.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n129.xml
%% ID: ibm-not-wf-P85-ibm85n129.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0E98 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n129'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n129.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n13.xml
%% ID: ibm-not-wf-P85-ibm85n13.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01F6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n130.xml
%% ID: ibm-not-wf-P85-ibm85n130.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EA0 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n130'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n130.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n131.xml
%% ID: ibm-not-wf-P85-ibm85n131.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EA4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n131'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n131.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n132.xml
%% ID: ibm-not-wf-P85-ibm85n132.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EA6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n132'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n132.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n133.xml
%% ID: ibm-not-wf-P85-ibm85n133.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EA8 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n133'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n133.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n134.xml
%% ID: ibm-not-wf-P85-ibm85n134.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EAC occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n134'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n134.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n135.xml
%% ID: ibm-not-wf-P85-ibm85n135.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EAF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n135'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n135.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n136.xml
%% ID: ibm-not-wf-P85-ibm85n136.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EB1 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n136'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n136.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n137.xml
%% ID: ibm-not-wf-P85-ibm85n137.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EB4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n137'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n137.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n138.xml
%% ID: ibm-not-wf-P85-ibm85n138.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EBE occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n138'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n138.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n139.xml
%% ID: ibm-not-wf-P85-ibm85n139.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0EC5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n139'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n139.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n14.xml
%% ID: ibm-not-wf-P85-ibm85n14.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01F9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n140.xml
%% ID: ibm-not-wf-P85-ibm85n140.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0F48 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n140'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n140.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n141.xml
%% ID: ibm-not-wf-P85-ibm85n141.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0F6A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n141'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n141.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n142.xml
%% ID: ibm-not-wf-P85-ibm85n142.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x10C6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n142'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n142.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n143.xml
%% ID: ibm-not-wf-P85-ibm85n143.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x10F7 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n143'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n143.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n144.xml
%% ID: ibm-not-wf-P85-ibm85n144.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1011 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n144'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n144.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n145.xml
%% ID: ibm-not-wf-P85-ibm85n145.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1104 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n145'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n145.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n146.xml
%% ID: ibm-not-wf-P85-ibm85n146.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1108 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n146'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n146.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n147.xml
%% ID: ibm-not-wf-P85-ibm85n147.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x110A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n147'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n147.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n148.xml
%% ID: ibm-not-wf-P85-ibm85n148.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x110D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n148'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n148.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n149.xml
%% ID: ibm-not-wf-P85-ibm85n149.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x113B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n149'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n149.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n15.xml
%% ID: ibm-not-wf-P85-ibm85n15.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x01F9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n150.xml
%% ID: ibm-not-wf-P85-ibm85n150.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x113F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n150'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n150.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n151.xml
%% ID: ibm-not-wf-P85-ibm85n151.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1141 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n151'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n151.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n152.xml
%% ID: ibm-not-wf-P85-ibm85n152.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x114D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n152'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n152.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n153.xml
%% ID: ibm-not-wf-P85-ibm85n153.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x114f occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n153'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n153.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n154.xml
%% ID: ibm-not-wf-P85-ibm85n154.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1151 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n154'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n154.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n155.xml
%% ID: ibm-not-wf-P85-ibm85n155.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1156 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n155'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n155.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n156.xml
%% ID: ibm-not-wf-P85-ibm85n156.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x115A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n156'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n156.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n157.xml
%% ID: ibm-not-wf-P85-ibm85n157.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1162 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n157'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n157.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n158.xml
%% ID: ibm-not-wf-P85-ibm85n158.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1164 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n158'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n158.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n159.xml
%% ID: ibm-not-wf-P85-ibm85n159.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1166 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n159'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n159.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n16.xml
%% ID: ibm-not-wf-P85-ibm85n16.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0230 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n160.xml
%% ID: ibm-not-wf-P85-ibm85n160.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x116B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n160'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n160.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n161.xml
%% ID: ibm-not-wf-P85-ibm85n161.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x116F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n161'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n161.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n162.xml
%% ID: ibm-not-wf-P85-ibm85n162.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1174 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n162'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n162.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n163.xml
%% ID: ibm-not-wf-P85-ibm85n163.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x119F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n163'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n163.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n164.xml
%% ID: ibm-not-wf-P85-ibm85n164.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11AC occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n164'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n164.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n165.xml
%% ID: ibm-not-wf-P85-ibm85n165.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11B6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n165'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n165.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n166.xml
%% ID: ibm-not-wf-P85-ibm85n166.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11B9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n166'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n166.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n167.xml
%% ID: ibm-not-wf-P85-ibm85n167.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11BB occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n167'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n167.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n168.xml
%% ID: ibm-not-wf-P85-ibm85n168.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11C3 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n168'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n168.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n169.xml
%% ID: ibm-not-wf-P85-ibm85n169.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11F1 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n169'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n169.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n17.xml
%% ID: ibm-not-wf-P85-ibm85n17.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x02AF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n17.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n170.xml
%% ID: ibm-not-wf-P85-ibm85n170.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x11FA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n170'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n170.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n171.xml
%% ID: ibm-not-wf-P85-ibm85n171.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1E9C occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n171'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n171.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n172.xml
%% ID: ibm-not-wf-P85-ibm85n172.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1EFA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n172'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n172.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n173.xml
%% ID: ibm-not-wf-P85-ibm85n173.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F16 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n173'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n173.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n174.xml
%% ID: ibm-not-wf-P85-ibm85n174.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F1E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n174'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n174.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n175.xml
%% ID: ibm-not-wf-P85-ibm85n175.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F46 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n175'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n175.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n176.xml
%% ID: ibm-not-wf-P85-ibm85n176.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F4F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n176'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n176.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n177.xml
%% ID: ibm-not-wf-P85-ibm85n177.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F58 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n177'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n177.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n178.xml
%% ID: ibm-not-wf-P85-ibm85n178.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F5A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n178'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n178.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n179.xml
%% ID: ibm-not-wf-P85-ibm85n179.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F5C occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n179'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n179.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n18.xml
%% ID: ibm-not-wf-P85-ibm85n18.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x02CF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n18.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n180.xml
%% ID: ibm-not-wf-P85-ibm85n180.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F5E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n180'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n180.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n181.xml
%% ID: ibm-not-wf-P85-ibm85n181.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1F7E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n181'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n181.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n182.xml
%% ID: ibm-not-wf-P85-ibm85n182.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FB5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n182'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n182.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n183.xml
%% ID: ibm-not-wf-P85-ibm85n183.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FBD occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n183'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n183.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n184.xml
%% ID: ibm-not-wf-P85-ibm85n184.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FBF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n184'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n184.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n185.xml
%% ID: ibm-not-wf-P85-ibm85n185.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FC5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n185'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n185.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n186.xml
%% ID: ibm-not-wf-P85-ibm85n186.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FCD occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n186'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n186.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n187.xml
%% ID: ibm-not-wf-P85-ibm85n187.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FD5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n187'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n187.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n188.xml
%% ID: ibm-not-wf-P85-ibm85n188.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FDC occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n188'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n188.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n189.xml
%% ID: ibm-not-wf-P85-ibm85n189.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FED occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n189'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n189.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n19.xml
%% ID: ibm-not-wf-P85-ibm85n19.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0387 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n19'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n19.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n190.xml
%% ID: ibm-not-wf-P85-ibm85n190.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FF5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n190'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n190.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n191.xml
%% ID: ibm-not-wf-P85-ibm85n191.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x1FFD occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n191'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n191.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n192.xml
%% ID: ibm-not-wf-P85-ibm85n192.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x2127 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n192'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n192.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n193.xml
%% ID: ibm-not-wf-P85-ibm85n193.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x212F occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n193'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n193.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n194.xml
%% ID: ibm-not-wf-P85-ibm85n194.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x2183 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n194'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n194.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n195.xml
%% ID: ibm-not-wf-P85-ibm85n195.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x3095 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n195'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n195.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n196.xml
%% ID: ibm-not-wf-P85-ibm85n196.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x30FB occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n196'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n196.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n197.xml
%% ID: ibm-not-wf-P85-ibm85n197.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x312D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n197'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n197.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n198.xml
%% ID: ibm-not-wf-P85-ibm85n198.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #xD7A4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n198'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n198.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n20.xml
%% ID: ibm-not-wf-P85-ibm85n20.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x038B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n20'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n20.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n21.xml
%% ID: ibm-not-wf-P85-ibm85n21.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x03A2 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n21'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n21.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n22.xml
%% ID: ibm-not-wf-P85-ibm85n22.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x03CF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n22'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n22.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n23.xml
%% ID: ibm-not-wf-P85-ibm85n23.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x03D7 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n23'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n23.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n24.xml
%% ID: ibm-not-wf-P85-ibm85n24.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x03DD occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n24'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n24.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n25.xml
%% ID: ibm-not-wf-P85-ibm85n25.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x03E1 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n25'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n25.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n26.xml
%% ID: ibm-not-wf-P85-ibm85n26.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x03F4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n26'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n26.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n27.xml
%% ID: ibm-not-wf-P85-ibm85n27.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x040D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n27'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n27.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n28.xml
%% ID: ibm-not-wf-P85-ibm85n28.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0450 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n28'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n28.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n29.xml
%% ID: ibm-not-wf-P85-ibm85n29.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x045D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n29'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n29.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n30.xml
%% ID: ibm-not-wf-P85-ibm85n30.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0482 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n30'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n30.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n31.xml
%% ID: ibm-not-wf-P85-ibm85n31.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04C5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n31'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n31.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n32.xml
%% ID: ibm-not-wf-P85-ibm85n32.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04C6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n32'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n32.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n33.xml
%% ID: ibm-not-wf-P85-ibm85n33.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04C9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n33'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n33.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n34.xml
%% ID: ibm-not-wf-P85-ibm85n34.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04EC occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n34'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n34.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n35.xml
%% ID: ibm-not-wf-P85-ibm85n35.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04ED occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n35'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n35.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n36.xml
%% ID: ibm-not-wf-P85-ibm85n36.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04F6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n36'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n36.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n37.xml
%% ID: ibm-not-wf-P85-ibm85n37.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x04FA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n37'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n37.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n38.xml
%% ID: ibm-not-wf-P85-ibm85n38.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0557 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n38'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n38.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n39.xml
%% ID: ibm-not-wf-P85-ibm85n39.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0558 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n39'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n39.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n40.xml
%% ID: ibm-not-wf-P85-ibm85n40.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0587 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n40'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n40.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n41.xml
%% ID: ibm-not-wf-P85-ibm85n41.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x05EB occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n41'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n41.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n42.xml
%% ID: ibm-not-wf-P85-ibm85n42.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x05F3 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n42'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n42.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n43.xml
%% ID: ibm-not-wf-P85-ibm85n43.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0620 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n43'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n43.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n44.xml
%% ID: ibm-not-wf-P85-ibm85n44.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x063B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n44'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n44.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n45.xml
%% ID: ibm-not-wf-P85-ibm85n45.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x064B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n45'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n45.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n46.xml
%% ID: ibm-not-wf-P85-ibm85n46.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x06B8 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n46'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n46.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n47.xml
%% ID: ibm-not-wf-P85-ibm85n47.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x06BF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n47'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n47.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n48.xml
%% ID: ibm-not-wf-P85-ibm85n48.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x06CF occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n48'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n48.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n49.xml
%% ID: ibm-not-wf-P85-ibm85n49.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x06D4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n49'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n49.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n50.xml
%% ID: ibm-not-wf-P85-ibm85n50.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x06D6 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n50'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n50.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n51.xml
%% ID: ibm-not-wf-P85-ibm85n51.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x06E7 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n51'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n51.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n52.xml
%% ID: ibm-not-wf-P85-ibm85n52.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x093A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n52'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n52.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n53.xml
%% ID: ibm-not-wf-P85-ibm85n53.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x093E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n53'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n53.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n54.xml
%% ID: ibm-not-wf-P85-ibm85n54.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0962 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n54'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n54.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n55.xml
%% ID: ibm-not-wf-P85-ibm85n55.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x098D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n55'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n55.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n56.xml
%% ID: ibm-not-wf-P85-ibm85n56.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0991 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n56'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n56.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n57.xml
%% ID: ibm-not-wf-P85-ibm85n57.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0992 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n57'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n57.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n58.xml
%% ID: ibm-not-wf-P85-ibm85n58.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09A9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n58'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n58.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n59.xml
%% ID: ibm-not-wf-P85-ibm85n59.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09B1 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n59'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n59.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n60.xml
%% ID: ibm-not-wf-P85-ibm85n60.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09B5 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n60'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n60.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n61.xml
%% ID: ibm-not-wf-P85-ibm85n61.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09BA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n61'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n61.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n62.xml
%% ID: ibm-not-wf-P85-ibm85n62.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09DE occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n62'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n62.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n63.xml
%% ID: ibm-not-wf-P85-ibm85n63.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09E2 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n63'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n63.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n64.xml
%% ID: ibm-not-wf-P85-ibm85n64.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x09F2 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n64'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n64.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n65.xml
%% ID: ibm-not-wf-P85-ibm85n65.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A0B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n65'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n65.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n66.xml
%% ID: ibm-not-wf-P85-ibm85n66.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A11 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n66'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n66.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n67.xml
%% ID: ibm-not-wf-P85-ibm85n67.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A29 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n67'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n67.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n68.xml
%% ID: ibm-not-wf-P85-ibm85n68.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A31 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n68'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n68.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n69.xml
%% ID: ibm-not-wf-P85-ibm85n69.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A34 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n69'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n69.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n70.xml
%% ID: ibm-not-wf-P85-ibm85n70.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A37 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n70'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n70.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n71.xml
%% ID: ibm-not-wf-P85-ibm85n71.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A3A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n71'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n71.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n72.xml
%% ID: ibm-not-wf-P85-ibm85n72.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A5D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n72'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n72.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n73.xml
%% ID: ibm-not-wf-P85-ibm85n73.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A70 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n73'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n73.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n74.xml
%% ID: ibm-not-wf-P85-ibm85n74.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A75 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n74'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n74.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n75.xml
%% ID: ibm-not-wf-P85-ibm85n75.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #xA84 occurs as the
%%   first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n75'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n75.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n76.xml
%% ID: ibm-not-wf-P85-ibm85n76.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0ABC occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n76'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n76.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n77.xml
%% ID: ibm-not-wf-P85-ibm85n77.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0A92 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n77'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n77.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n78.xml
%% ID: ibm-not-wf-P85-ibm85n78.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0AA9 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n78'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n78.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n79.xml
%% ID: ibm-not-wf-P85-ibm85n79.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0AB1 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n79'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n79.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n80.xml
%% ID: ibm-not-wf-P85-ibm85n80.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0AB4 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n80'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n80.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n81.xml
%% ID: ibm-not-wf-P85-ibm85n81.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0ABA occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n81'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n81.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n82.xml
%% ID: ibm-not-wf-P85-ibm85n82.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B04 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n82'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n82.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n83.xml
%% ID: ibm-not-wf-P85-ibm85n83.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B0D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n83'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n83.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n84.xml
%% ID: ibm-not-wf-P85-ibm85n84.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B11 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n84'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n84.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n85.xml
%% ID: ibm-not-wf-P85-ibm85n85.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B29 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n85'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n85.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n86.xml
%% ID: ibm-not-wf-P85-ibm85n86.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B31 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n86'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n86.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n87.xml
%% ID: ibm-not-wf-P85-ibm85n87.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B34 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n87'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n87.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n88.xml
%% ID: ibm-not-wf-P85-ibm85n88.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B3A occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n88'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n88.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n89.xml
%% ID: ibm-not-wf-P85-ibm85n89.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B3E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n89'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n89.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n90.xml
%% ID: ibm-not-wf-P85-ibm85n90.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B5E occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n90'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n90.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n91.xml
%% ID: ibm-not-wf-P85-ibm85n91.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B62 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n91'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n91.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n92.xml
%% ID: ibm-not-wf-P85-ibm85n92.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B8B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n92'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n92.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n93.xml
%% ID: ibm-not-wf-P85-ibm85n93.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B91 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n93'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n93.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n94.xml
%% ID: ibm-not-wf-P85-ibm85n94.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B98 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n94'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n94.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n95.xml
%% ID: ibm-not-wf-P85-ibm85n95.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B9B occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n95'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n95.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n96.xml
%% ID: ibm-not-wf-P85-ibm85n96.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0B9D occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n96'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n96.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n97.xml
%% ID: ibm-not-wf-P85-ibm85n97.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0BA0 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n97'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n97.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n98.xml
%% ID: ibm-not-wf-P85-ibm85n98.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0BA7 occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n98'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n98.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P85/ibm85n99.xml
%% ID: ibm-not-wf-P85-ibm85n99.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests BaseChar with an illegal character. The character #x0BAB occurs as
%%   the first character of the PITarget in the PI in the DTD.
'ibm-not-wf-P85-ibm85n99'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P85/ibm85n99.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P86/ibm86n01.xml
%% ID: ibm-not-wf-P86-ibm86n01.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Ideographic with an illegal character. The character #x4CFF occurs as
%%   the first character in the PITarget in the PI in the DTD.
'ibm-not-wf-P86-ibm86n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P86/ibm86n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P86/ibm86n02.xml
%% ID: ibm-not-wf-P86-ibm86n02.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Ideographic with an illegal character. The character #x9FA6 occurs as
%%   the first character in the PITarget in the PI in the DTD.
'ibm-not-wf-P86-ibm86n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P86/ibm86n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P86/ibm86n03.xml
%% ID: ibm-not-wf-P86-ibm86n03.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Ideographic with an illegal character. The character #x3008 occurs as
%%   the first character in the PITarget in the PI in the DTD.
'ibm-not-wf-P86-ibm86n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P86/ibm86n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P86/ibm86n04.xml
%% ID: ibm-not-wf-P86-ibm86n04.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Ideographic with an illegal character. The character #x302A occurs as
%%   the first character in the PITarget in the PI in the DTD.
'ibm-not-wf-P86-ibm86n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P86/ibm86n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n01.xml
%% ID: ibm-not-wf-P87-ibm87n01.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x02FF occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n02.xml
%% ID: ibm-not-wf-P87-ibm87n02.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0346 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n03.xml
%% ID: ibm-not-wf-P87-ibm87n03.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0362 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n04.xml
%% ID: ibm-not-wf-P87-ibm87n04.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0487 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n05.xml
%% ID: ibm-not-wf-P87-ibm87n05.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x05A2 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n06.xml
%% ID: ibm-not-wf-P87-ibm87n06.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x05BA occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n07.xml
%% ID: ibm-not-wf-P87-ibm87n07.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x05BE occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n08.xml
%% ID: ibm-not-wf-P87-ibm87n08.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x05C0 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n09.xml
%% ID: ibm-not-wf-P87-ibm87n09.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x05C3 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n10.xml
%% ID: ibm-not-wf-P87-ibm87n10.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0653 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n11.xml
%% ID: ibm-not-wf-P87-ibm87n11.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x06B8 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n12.xml
%% ID: ibm-not-wf-P87-ibm87n12.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x06B9 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n13.xml
%% ID: ibm-not-wf-P87-ibm87n13.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x06E9 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n14.xml
%% ID: ibm-not-wf-P87-ibm87n14.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x06EE occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n15.xml
%% ID: ibm-not-wf-P87-ibm87n15.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0904 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n16.xml
%% ID: ibm-not-wf-P87-ibm87n16.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x093B occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n17.xml
%% ID: ibm-not-wf-P87-ibm87n17.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x094E occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n17'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n17.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n18.xml
%% ID: ibm-not-wf-P87-ibm87n18.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0955 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n18'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n18.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n19.xml
%% ID: ibm-not-wf-P87-ibm87n19.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0964 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n19'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n19.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n20.xml
%% ID: ibm-not-wf-P87-ibm87n20.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0984 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n20'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n20.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n21.xml
%% ID: ibm-not-wf-P87-ibm87n21.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x09C5 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n21'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n21.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n22.xml
%% ID: ibm-not-wf-P87-ibm87n22.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x09C9 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n22'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n22.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n23.xml
%% ID: ibm-not-wf-P87-ibm87n23.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x09CE occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n23'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n23.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n24.xml
%% ID: ibm-not-wf-P87-ibm87n24.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x09D8 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n24'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n24.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n25.xml
%% ID: ibm-not-wf-P87-ibm87n25.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x09E4 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n25'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n25.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n26.xml
%% ID: ibm-not-wf-P87-ibm87n26.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A03 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n26'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n26.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n27.xml
%% ID: ibm-not-wf-P87-ibm87n27.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A3D occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n27'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n27.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n28.xml
%% ID: ibm-not-wf-P87-ibm87n28.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A46 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n28'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n28.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n29.xml
%% ID: ibm-not-wf-P87-ibm87n29.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A49 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n29'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n29.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n30.xml
%% ID: ibm-not-wf-P87-ibm87n30.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A4E occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n30'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n30.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n31.xml
%% ID: ibm-not-wf-P87-ibm87n31.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A80 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n31'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n31.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n32.xml
%% ID: ibm-not-wf-P87-ibm87n32.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0A84 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n32'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n32.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n33.xml
%% ID: ibm-not-wf-P87-ibm87n33.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0ABB occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n33'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n33.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n34.xml
%% ID: ibm-not-wf-P87-ibm87n34.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0AC6 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n34'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n34.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n35.xml
%% ID: ibm-not-wf-P87-ibm87n35.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0ACA occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n35'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n35.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n36.xml
%% ID: ibm-not-wf-P87-ibm87n36.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0ACE occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n36'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n36.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n37.xml
%% ID: ibm-not-wf-P87-ibm87n37.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B04 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n37'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n37.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n38.xml
%% ID: ibm-not-wf-P87-ibm87n38.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B3B occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n38'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n38.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n39.xml
%% ID: ibm-not-wf-P87-ibm87n39.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B44 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n39'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n39.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n40.xml
%% ID: ibm-not-wf-P87-ibm87n40.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B4A occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n40'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n40.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n41.xml
%% ID: ibm-not-wf-P87-ibm87n41.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B4E occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n41'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n41.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n42.xml
%% ID: ibm-not-wf-P87-ibm87n42.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B58 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n42'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n42.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n43.xml
%% ID: ibm-not-wf-P87-ibm87n43.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0B84 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n43'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n43.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n44.xml
%% ID: ibm-not-wf-P87-ibm87n44.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0BC3 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n44'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n44.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n45.xml
%% ID: ibm-not-wf-P87-ibm87n45.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0BC9 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n45'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n45.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n46.xml
%% ID: ibm-not-wf-P87-ibm87n46.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0BD6 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n46'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n46.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n47.xml
%% ID: ibm-not-wf-P87-ibm87n47.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0C0D occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n47'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n47.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n48.xml
%% ID: ibm-not-wf-P87-ibm87n48.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0C45 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n48'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n48.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n49.xml
%% ID: ibm-not-wf-P87-ibm87n49.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0C49 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n49'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n49.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n50.xml
%% ID: ibm-not-wf-P87-ibm87n50.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0C54 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n50'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n50.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n51.xml
%% ID: ibm-not-wf-P87-ibm87n51.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0C81 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n51'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n51.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n52.xml
%% ID: ibm-not-wf-P87-ibm87n52.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0C84 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n52'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n52.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n53.xml
%% ID: ibm-not-wf-P87-ibm87n53.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0CC5 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n53'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n53.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n54.xml
%% ID: ibm-not-wf-P87-ibm87n54.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0CC9 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n54'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n54.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n55.xml
%% ID: ibm-not-wf-P87-ibm87n55.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0CD4 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n55'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n55.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n56.xml
%% ID: ibm-not-wf-P87-ibm87n56.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0CD7 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n56'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n56.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n57.xml
%% ID: ibm-not-wf-P87-ibm87n57.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0D04 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n57'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n57.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n58.xml
%% ID: ibm-not-wf-P87-ibm87n58.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0D45 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n58'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n58.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n59.xml
%% ID: ibm-not-wf-P87-ibm87n59.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0D49 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n59'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n59.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n60.xml
%% ID: ibm-not-wf-P87-ibm87n60.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0D4E occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n60'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n60.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n61.xml
%% ID: ibm-not-wf-P87-ibm87n61.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0D58 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n61'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n61.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n62.xml
%% ID: ibm-not-wf-P87-ibm87n62.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0E3F occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n62'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n62.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n63.xml
%% ID: ibm-not-wf-P87-ibm87n63.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0E3B occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n63'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n63.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n64.xml
%% ID: ibm-not-wf-P87-ibm87n64.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0E4F occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n64'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n64.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n66.xml
%% ID: ibm-not-wf-P87-ibm87n66.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0EBA occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n66'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n66.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n67.xml
%% ID: ibm-not-wf-P87-ibm87n67.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0EBE occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n67'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n67.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n68.xml
%% ID: ibm-not-wf-P87-ibm87n68.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0ECE occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n68'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n68.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n69.xml
%% ID: ibm-not-wf-P87-ibm87n69.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F1A occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n69'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n69.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n70.xml
%% ID: ibm-not-wf-P87-ibm87n70.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F36 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n70'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n70.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n71.xml
%% ID: ibm-not-wf-P87-ibm87n71.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F38 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n71'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n71.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n72.xml
%% ID: ibm-not-wf-P87-ibm87n72.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F3B occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n72'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n72.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n73.xml
%% ID: ibm-not-wf-P87-ibm87n73.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F3A occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n73'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n73.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n74.xml
%% ID: ibm-not-wf-P87-ibm87n74.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F70 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n74'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n74.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n75.xml
%% ID: ibm-not-wf-P87-ibm87n75.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F85 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n75'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n75.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n76.xml
%% ID: ibm-not-wf-P87-ibm87n76.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F8C occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n76'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n76.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n77.xml
%% ID: ibm-not-wf-P87-ibm87n77.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F96 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n77'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n77.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n78.xml
%% ID: ibm-not-wf-P87-ibm87n78.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0F98 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n78'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n78.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n79.xml
%% ID: ibm-not-wf-P87-ibm87n79.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0FB0 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n79'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n79.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n80.xml
%% ID: ibm-not-wf-P87-ibm87n80.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0FB8 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n80'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n80.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n81.xml
%% ID: ibm-not-wf-P87-ibm87n81.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x0FBA occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n81'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n81.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n82.xml
%% ID: ibm-not-wf-P87-ibm87n82.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x20DD occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n82'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n82.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n83.xml
%% ID: ibm-not-wf-P87-ibm87n83.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x20E2 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n83'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n83.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n84.xml
%% ID: ibm-not-wf-P87-ibm87n84.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x3030 occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n84'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n84.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P87/ibm87n85.xml
%% ID: ibm-not-wf-P87-ibm87n85.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests CombiningChar with an illegal character. The character #x309B occurs
%%   as the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P87-ibm87n85'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P87/ibm87n85.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n01.xml
%% ID: ibm-not-wf-P88-ibm88n01.xml
%% Type: not-wf
%% Sections: B.
%% Description:
%%   Tests Digit with an illegal character. The character #x0029 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n02.xml
%% ID: ibm-not-wf-P88-ibm88n02.xml
%% Type: not-wf
%% Sections: B.
%% Description:
%%   Tests Digit with an illegal character. The character #x003B occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n03.xml
%% ID: ibm-not-wf-P88-ibm88n03.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x066A occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n04.xml
%% ID: ibm-not-wf-P88-ibm88n04.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x06FA occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n05.xml
%% ID: ibm-not-wf-P88-ibm88n05.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0970 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n06.xml
%% ID: ibm-not-wf-P88-ibm88n06.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x09F2 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n08.xml
%% ID: ibm-not-wf-P88-ibm88n08.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0AF0 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n09.xml
%% ID: ibm-not-wf-P88-ibm88n09.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0B70 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n10.xml
%% ID: ibm-not-wf-P88-ibm88n10.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0C65 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n11.xml
%% ID: ibm-not-wf-P88-ibm88n11.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0CE5 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n12.xml
%% ID: ibm-not-wf-P88-ibm88n12.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0CF0 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n13.xml
%% ID: ibm-not-wf-P88-ibm88n13.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0D70 occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n13'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n13.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n14.xml
%% ID: ibm-not-wf-P88-ibm88n14.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0E5A occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n14'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n14.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n15.xml
%% ID: ibm-not-wf-P88-ibm88n15.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0EDA occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n15'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n15.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P88/ibm88n16.xml
%% ID: ibm-not-wf-P88-ibm88n16.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Digit with an illegal character. The character #x0F2A occurs as the
%%   second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P88-ibm88n16'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P88/ibm88n16.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n01.xml
%% ID: ibm-not-wf-P89-ibm89n01.xml
%% Type: not-wf
%% Sections: B.
%% Description:
%%   Tests Extender with an illegal character. The character #x00B6 occurs as
%%   the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P89-ibm89n01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n02.xml
%% ID: ibm-not-wf-P89-ibm89n02.xml
%% Type: not-wf
%% Sections: B.
%% Description:
%%   Tests Extender with an illegal character. The character #x00B8 occurs as
%%   the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P89-ibm89n02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n02.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n03.xml
%% ID: ibm-not-wf-P89-ibm89n03.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x02D2 occurs as
%%   the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P89-ibm89n03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n03.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n04.xml
%% ID: ibm-not-wf-P89-ibm89n04.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x03FE occurs as
%%   the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P89-ibm89n04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n04.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n05.xml
%% ID: ibm-not-wf-P89-ibm89n05.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x065F occurs as
%%   the second character in the PITarget in the PI in the DTD.
'ibm-not-wf-P89-ibm89n05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n05.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n06.xml
%% ID: ibm-not-wf-P89-ibm89n06.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x0EC7 occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n06.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n07.xml
%% ID: ibm-not-wf-P89-ibm89n07.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x3006 occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n07.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n08.xml
%% ID: ibm-not-wf-P89-ibm89n08.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x3030 occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n08.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n09.xml
%% ID: ibm-not-wf-P89-ibm89n09.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x3036 occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n09'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n09.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n10.xml
%% ID: ibm-not-wf-P89-ibm89n10.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x309C occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n10'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n10.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n11.xml
%% ID: ibm-not-wf-P89-ibm89n11.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x309F occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n11'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n11.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: not-wf/P89/ibm89n12.xml
%% ID: ibm-not-wf-P89-ibm89n12.xml
%% Type: not-wf
%% Sections: B.
%% Edition: 1 2 3 4
%% Description:
%%   Tests Extender with an illegal character. The character #x30FF occurs as
%%   the second character in the PITarget in the PI in the DTD. [Also contains
%%   two top-level elements -- one should be removed]
'ibm-not-wf-P89-ibm89n12'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","not-wf/P89/ibm89n12.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "not-wf").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P01/ibm01v01.xml
%% ID: ibm-valid-P01-ibm01v01.xml
%% Type: valid
%% Sections: 2.1
%% Output: valid/P01/out/ibm01v01.xml
%% Description:
%%   Tests with a xml document consisting of followed by then
'ibm-valid-P01-ibm01v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P01/ibm01v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P01/out/ibm01v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P02/ibm02v01.xml
%% ID: ibm-valid-P02-ibm02v01.xml
%% Type: valid
%% Sections: 2.2
%% Description:
%%   This test case covers legal character ranges plus discrete legal characters
%%   for production 02.
'ibm-valid-P02-ibm02v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P02/ibm02v01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P03/ibm03v01.xml
%% ID: ibm-valid-P03-ibm03v01.xml
%% Type: valid
%% Sections: 2.3
%% Description:
%%   Tests all 4 legal white space characters - #x20 #x9 #xD #xA
'ibm-valid-P03-ibm03v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P03/ibm03v01.xml"]),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid").

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P09/ibm09v01.xml
%% ID: ibm-valid-P09-ibm09v01.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P09/out/ibm09v01.xml
%% Description:
%%   Empty EntityValue is legal
'ibm-valid-P09-ibm09v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P09/ibm09v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P09/out/ibm09v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P09/ibm09v02.xml
%% ID: ibm-valid-P09-ibm09v02.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P09/out/ibm09v02.xml
%% Description:
%%   Tests a normal EnitityValue
'ibm-valid-P09-ibm09v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P09/ibm09v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P09/out/ibm09v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P09/ibm09v03.xml
%% ID: ibm-valid-P09-ibm09v03.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P09/out/ibm09v03.xml
%% Description:
%%   Tests EnitityValue referencing a Parameter Entity
'ibm-valid-P09-ibm09v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P09/ibm09v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P09/out/ibm09v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P09/ibm09v04.xml
%% ID: ibm-valid-P09-ibm09v04.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P09/out/ibm09v04.xml
%% Description:
%%   Tests EnitityValue referencing a General Entity
'ibm-valid-P09-ibm09v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P09/ibm09v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P09/out/ibm09v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P09/ibm09v05.xml
%% ID: ibm-valid-P09-ibm09v05.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P09/out/ibm09v05.xml
%% Description:
%%   Tests EnitityValue with combination of GE, PE and text, the GE used is
%%   declared in the student.dtd
'ibm-valid-P09-ibm09v05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P09/ibm09v05.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P09/out/ibm09v05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v01.xml
%% ID: ibm-valid-P10-ibm10v01.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v01.xml
%% Description:
%%   Tests empty AttValue with double quotes as the delimiters
'ibm-valid-P10-ibm10v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v02.xml
%% ID: ibm-valid-P10-ibm10v02.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v02.xml
%% Description:
%%   Tests empty AttValue with single quotes as the delimiters
'ibm-valid-P10-ibm10v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v03.xml
%% ID: ibm-valid-P10-ibm10v03.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v03.xml
%% Description:
%%   Test AttValue with double quotes as the delimiters and single quote inside
'ibm-valid-P10-ibm10v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v04.xml
%% ID: ibm-valid-P10-ibm10v04.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v04.xml
%% Description:
%%   Test AttValue with single quotes as the delimiters and double quote inside
'ibm-valid-P10-ibm10v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v05.xml
%% ID: ibm-valid-P10-ibm10v05.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v05.xml
%% Description:
%%   Test AttValue with a GE reference and double quotes as the delimiters
'ibm-valid-P10-ibm10v05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v05.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v06.xml
%% ID: ibm-valid-P10-ibm10v06.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v06.xml
%% Description:
%%   Test AttValue with a GE reference and single quotes as the delimiters
'ibm-valid-P10-ibm10v06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v06.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v06.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v07.xml
%% ID: ibm-valid-P10-ibm10v07.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v07.xml
%% Description:
%%   testing AttValue with mixed references and text content in double quotes
'ibm-valid-P10-ibm10v07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v07.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v07.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P10/ibm10v08.xml
%% ID: ibm-valid-P10-ibm10v08.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P10/out/ibm10v08.xml
%% Description:
%%   testing AttValue with mixed references and text content in single quotes
'ibm-valid-P10-ibm10v08'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P10/ibm10v08.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P10/out/ibm10v08.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P11/ibm11v01.xml
%% ID: ibm-valid-P11-ibm11v01.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P11/out/ibm11v01.xml
%% Description:
%%   Tests empty systemliteral using the double quotes
'ibm-valid-P11-ibm11v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P11/ibm11v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P11/out/ibm11v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P11/ibm11v02.xml
%% ID: ibm-valid-P11-ibm11v02.xml
%% Type: valid
%% Sections: 2.3
%% Output: valid/P11/out/ibm11v02.xml
%% Description:
%%   Tests empty systemliteral using the single quotes
'ibm-valid-P11-ibm11v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P11/ibm11v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P11/out/ibm11v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P11/ibm11v03.xml
%% ID: ibm-valid-P11-ibm11v03.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P11/out/ibm11v03.xml
%% Description:
%%   Tests regular systemliteral using the single quotes
'ibm-valid-P11-ibm11v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P11/ibm11v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P11/out/ibm11v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P11/ibm11v04.xml
%% ID: ibm-valid-P11-ibm11v04.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P11/out/ibm11v04.xml
%% Description:
%%   Tests regular systemliteral using the double quotes
'ibm-valid-P11-ibm11v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P11/ibm11v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P11/out/ibm11v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P12/ibm12v01.xml
%% ID: ibm-valid-P12-ibm12v01.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P12/out/ibm12v01.xml
%% Description:
%%   Tests empty systemliteral using the double quotes
'ibm-valid-P12-ibm12v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P12/ibm12v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P12/out/ibm12v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P12/ibm12v02.xml
%% ID: ibm-valid-P12-ibm12v02.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P12/out/ibm12v02.xml
%% Description:
%%   Tests empty systemliteral using the single quotes
'ibm-valid-P12-ibm12v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P12/ibm12v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P12/out/ibm12v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P12/ibm12v03.xml
%% ID: ibm-valid-P12-ibm12v03.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P12/out/ibm12v03.xml
%% Description:
%%   Tests regular systemliteral using the double quotes
'ibm-valid-P12-ibm12v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P12/ibm12v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P12/out/ibm12v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P12/ibm12v04.xml
%% ID: ibm-valid-P12-ibm12v04.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P12/out/ibm12v04.xml
%% Description:
%%   Tests regular systemliteral using the single quotes
'ibm-valid-P12-ibm12v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P12/ibm12v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P12/out/ibm12v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P13/ibm13v01.xml
%% ID: ibm-valid-P13-ibm13v01.xml
%% Type: valid
%% Sections: 2.3
%% Entities: parameter
%% Output: valid/P13/out/ibm13v01.xml
%% Description:
%%   Testing PubidChar with all legal PubidChar in a PubidLiteral
'ibm-valid-P13-ibm13v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P13/ibm13v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P13/out/ibm13v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P14/ibm14v01.xml
%% ID: ibm-valid-P14-ibm14v01.xml
%% Type: valid
%% Sections: 2.4
%% Output: valid/P14/out/ibm14v01.xml
%% Description:
%%   Testing CharData with empty string
'ibm-valid-P14-ibm14v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P14/ibm14v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P14/out/ibm14v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P14/ibm14v02.xml
%% ID: ibm-valid-P14-ibm14v02.xml
%% Type: valid
%% Sections: 2.4
%% Output: valid/P14/out/ibm14v02.xml
%% Description:
%%   Testing CharData with white space character
'ibm-valid-P14-ibm14v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P14/ibm14v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P14/out/ibm14v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P14/ibm14v03.xml
%% ID: ibm-valid-P14-ibm14v03.xml
%% Type: valid
%% Sections: 2.4
%% Output: valid/P14/out/ibm14v03.xml
%% Description:
%%   Testing CharData with a general text string
'ibm-valid-P14-ibm14v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P14/ibm14v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P14/out/ibm14v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P15/ibm15v01.xml
%% ID: ibm-valid-P15-ibm15v01.xml
%% Type: valid
%% Sections: 2.5
%% Output: valid/P15/out/ibm15v01.xml
%% Description:
%%   Tests empty comment
'ibm-valid-P15-ibm15v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P15/ibm15v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P15/out/ibm15v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P15/ibm15v02.xml
%% ID: ibm-valid-P15-ibm15v02.xml
%% Type: valid
%% Sections: 2.5
%% Output: valid/P15/out/ibm15v02.xml
%% Description:
%%   Tests comment with regular text
'ibm-valid-P15-ibm15v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P15/ibm15v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P15/out/ibm15v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P15/ibm15v03.xml
%% ID: ibm-valid-P15-ibm15v03.xml
%% Type: valid
%% Sections: 2.5
%% Output: valid/P15/out/ibm15v03.xml
%% Description:
%%   Tests comment with one dash inside
'ibm-valid-P15-ibm15v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P15/ibm15v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P15/out/ibm15v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P15/ibm15v04.xml
%% ID: ibm-valid-P15-ibm15v04.xml
%% Type: valid
%% Sections: 2.5
%% Output: valid/P15/out/ibm15v04.xml
%% Description:
%%   Tests comment with more comprehensive content
'ibm-valid-P15-ibm15v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P15/ibm15v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P15/out/ibm15v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P16/ibm16v01.xml
%% ID: ibm-valid-P16-ibm16v01.xml
%% Type: valid
%% Sections: 2.6
%% Output: valid/P16/out/ibm16v01.xml
%% Description:
%%   Tests PI definition with only PItarget name and nothing else
'ibm-valid-P16-ibm16v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P16/ibm16v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P16/out/ibm16v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P16/ibm16v02.xml
%% ID: ibm-valid-P16-ibm16v02.xml
%% Type: valid
%% Sections: 2.6
%% Output: valid/P16/out/ibm16v02.xml
%% Description:
%%   Tests PI definition with only PItarget name and a white space
'ibm-valid-P16-ibm16v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P16/ibm16v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P16/out/ibm16v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P16/ibm16v03.xml
%% ID: ibm-valid-P16-ibm16v03.xml
%% Type: valid
%% Sections: 2.6
%% Output: valid/P16/out/ibm16v03.xml
%% Description:
%%   Tests PI definition with PItarget name and text that contains question mark
%%   and right angle
'ibm-valid-P16-ibm16v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P16/ibm16v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P16/out/ibm16v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P17/ibm17v01.xml
%% ID: ibm-valid-P17-ibm17v01.xml
%% Type: valid
%% Sections: 2.6
%% Output: valid/P17/out/ibm17v01.xml
%% Description:
%%   Tests PITarget name
'ibm-valid-P17-ibm17v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P17/ibm17v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P17/out/ibm17v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P18/ibm18v01.xml
%% ID: ibm-valid-P18-ibm18v01.xml
%% Type: valid
%% Sections: 2.7
%% Output: valid/P18/out/ibm18v01.xml
%% Description:
%%   Tests CDSect with CDStart CData CDEnd
'ibm-valid-P18-ibm18v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P18/ibm18v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P18/out/ibm18v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P19/ibm19v01.xml
%% ID: ibm-valid-P19-ibm19v01.xml
%% Type: valid
%% Sections: 2.7
%% Output: valid/P19/out/ibm19v01.xml
%% Description:
%%   Tests CDStart
'ibm-valid-P19-ibm19v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P19/ibm19v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P19/out/ibm19v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P20/ibm20v01.xml
%% ID: ibm-valid-P20-ibm20v01.xml
%% Type: valid
%% Sections: 2.7
%% Output: valid/P20/out/ibm20v01.xml
%% Description:
%%   Tests CDATA with empty string
'ibm-valid-P20-ibm20v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P20/ibm20v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P20/out/ibm20v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P20/ibm20v02.xml
%% ID: ibm-valid-P20-ibm20v02.xml
%% Type: valid
%% Sections: 2.7
%% Output: valid/P20/out/ibm20v02.xml
%% Description:
%%   Tests CDATA with regular content
'ibm-valid-P20-ibm20v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P20/ibm20v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P20/out/ibm20v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P21/ibm21v01.xml
%% ID: ibm-valid-P21-ibm21v01.xml
%% Type: valid
%% Sections: 2.7
%% Output: valid/P21/out/ibm21v01.xml
%% Description:
%%   Tests CDEnd
'ibm-valid-P21-ibm21v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P21/ibm21v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P21/out/ibm21v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v01.xml
%% ID: ibm-valid-P22-ibm22v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v01.xml
%% Description:
%%   Tests prolog with XMLDecl and doctypedecl
'ibm-valid-P22-ibm22v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v02.xml
%% ID: ibm-valid-P22-ibm22v02.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v02.xml
%% Description:
%%   Tests prolog with doctypedecl
'ibm-valid-P22-ibm22v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v03.xml
%% ID: ibm-valid-P22-ibm22v03.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v03.xml
%% Description:
%%   Tests prolog with Misc doctypedecl
'ibm-valid-P22-ibm22v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v04.xml
%% ID: ibm-valid-P22-ibm22v04.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v04.xml
%% Description:
%%   Tests prolog with doctypedecl Misc
'ibm-valid-P22-ibm22v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v05.xml
%% ID: ibm-valid-P22-ibm22v05.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v05.xml
%% Description:
%%   Tests prolog with XMLDecl Misc doctypedecl
'ibm-valid-P22-ibm22v05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v05.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v06.xml
%% ID: ibm-valid-P22-ibm22v06.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v06.xml
%% Description:
%%   Tests prolog with XMLDecl doctypedecl Misc
'ibm-valid-P22-ibm22v06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v06.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v06.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P22/ibm22v07.xml
%% ID: ibm-valid-P22-ibm22v07.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P22/out/ibm22v07.xml
%% Description:
%%   Tests prolog with XMLDecl Misc doctypedecl Misc
'ibm-valid-P22-ibm22v07'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P22/ibm22v07.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P22/out/ibm22v07.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P23/ibm23v01.xml
%% ID: ibm-valid-P23-ibm23v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P23/out/ibm23v01.xml
%% Description:
%%   Tests XMLDecl with VersionInfo only
'ibm-valid-P23-ibm23v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P23/ibm23v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P23/out/ibm23v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P23/ibm23v02.xml
%% ID: ibm-valid-P23-ibm23v02.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P23/out/ibm23v02.xml
%% Description:
%%   Tests XMLDecl with VersionInfo EncodingDecl
'ibm-valid-P23-ibm23v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P23/ibm23v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P23/out/ibm23v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P23/ibm23v03.xml
%% ID: ibm-valid-P23-ibm23v03.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P23/out/ibm23v03.xml
%% Description:
%%   Tests XMLDecl with VersionInfo SDDecl
'ibm-valid-P23-ibm23v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P23/ibm23v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P23/out/ibm23v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P23/ibm23v04.xml
%% ID: ibm-valid-P23-ibm23v04.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P23/out/ibm23v04.xml
%% Description:
%%   Tests XMLDecl with VerstionInfo and a trailing whitespace char
'ibm-valid-P23-ibm23v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P23/ibm23v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P23/out/ibm23v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P23/ibm23v05.xml
%% ID: ibm-valid-P23-ibm23v05.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P23/out/ibm23v05.xml
%% Description:
%%   Tests XMLDecl with VersionInfo EncodingDecl SDDecl
'ibm-valid-P23-ibm23v05'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P23/ibm23v05.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P23/out/ibm23v05.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P23/ibm23v06.xml
%% ID: ibm-valid-P23-ibm23v06.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P23/out/ibm23v06.xml
%% Description:
%%   Tests XMLDecl with VersionInfo EncodingDecl SDDecl and a trailing
%%   whitespace
'ibm-valid-P23-ibm23v06'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P23/ibm23v06.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P23/out/ibm23v06.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P24/ibm24v01.xml
%% ID: ibm-valid-P24-ibm24v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P24/out/ibm24v01.xml
%% Description:
%%   Tests VersionInfo with single quote
'ibm-valid-P24-ibm24v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P24/ibm24v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P24/out/ibm24v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P24/ibm24v02.xml
%% ID: ibm-valid-P24-ibm24v02.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P24/out/ibm24v02.xml
%% Description:
%%   Tests VersionInfo with double quote
'ibm-valid-P24-ibm24v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P24/ibm24v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P24/out/ibm24v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P25/ibm25v01.xml
%% ID: ibm-valid-P25-ibm25v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P25/out/ibm25v01.xml
%% Description:
%%   Tests EQ with =
'ibm-valid-P25-ibm25v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P25/ibm25v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P25/out/ibm25v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P25/ibm25v02.xml
%% ID: ibm-valid-P25-ibm25v02.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P25/out/ibm25v02.xml
%% Description:
%%   Tests EQ with = and spaces on both sides
'ibm-valid-P25-ibm25v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P25/ibm25v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P25/out/ibm25v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P25/ibm25v03.xml
%% ID: ibm-valid-P25-ibm25v03.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P25/out/ibm25v03.xml
%% Description:
%%   Tests EQ with = and space in front of it
'ibm-valid-P25-ibm25v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P25/ibm25v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P25/out/ibm25v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P25/ibm25v04.xml
%% ID: ibm-valid-P25-ibm25v04.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P25/out/ibm25v04.xml
%% Description:
%%   Tests EQ with = and space after it
'ibm-valid-P25-ibm25v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P25/ibm25v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P25/out/ibm25v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P26/ibm26v01.xml
%% ID: ibm-valid-P26-ibm26v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P26/out/ibm26v01.xml
%% Description:
%%   Tests VersionNum 1.0
'ibm-valid-P26-ibm26v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P26/ibm26v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P26/out/ibm26v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P27/ibm27v01.xml
%% ID: ibm-valid-P27-ibm27v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P27/out/ibm27v01.xml
%% Description:
%%   Tests Misc with comment
'ibm-valid-P27-ibm27v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P27/ibm27v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P27/out/ibm27v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P27/ibm27v02.xml
%% ID: ibm-valid-P27-ibm27v02.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P27/out/ibm27v02.xml
%% Description:
%%   Tests Misc with PI
'ibm-valid-P27-ibm27v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P27/ibm27v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P27/out/ibm27v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P27/ibm27v03.xml
%% ID: ibm-valid-P27-ibm27v03.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P27/out/ibm27v03.xml
%% Description:
%%   Tests Misc with white spaces
'ibm-valid-P27-ibm27v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P27/ibm27v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P27/out/ibm27v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P28/ibm28v01.xml
%% ID: ibm-valid-P28-ibm28v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P28/out/ibm28v01.xml
%% Description:
%%   Tests doctypedecl with internal DTD only
'ibm-valid-P28-ibm28v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P28/ibm28v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P28/out/ibm28v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P28/ibm28v02.xml
%% ID: ibm-valid-P28-ibm28v02.xml
%% Type: valid
%% Sections: 2.8
%% Entities: parameter
%% Output: valid/P28/out/ibm28v02.xml
%% Description:
%%   Tests doctypedecl with external subset and combinations of different markup
%%   declarations and PEReferences
'ibm-valid-P28-ibm28v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P28/ibm28v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P28/out/ibm28v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P29/ibm29v01.xml
%% ID: ibm-valid-P29-ibm29v01.xml
%% Type: valid
%% Sections: 2.8
%% Output: valid/P29/out/ibm29v01.xml
%% Description:
%%   Tests markupdecl with combinations of elementdecl, AttlistDecl,EntityDecl,
%%   NotationDecl, PI and comment
'ibm-valid-P29-ibm29v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P29/ibm29v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P29/out/ibm29v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P29/ibm29v02.xml
%% ID: ibm-valid-P29-ibm29v02.xml
%% Type: valid
%% Sections: 2.8
%% Entities: parameter
%% Output: valid/P29/out/ibm29v02.xml
%% Description:
%%   Tests WFC: PE in internal subset as a positive test
'ibm-valid-P29-ibm29v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P29/ibm29v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P29/out/ibm29v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P30/ibm30v01.xml
%% ID: ibm-valid-P30-ibm30v01.xml
%% Type: valid
%% Sections: 2.8
%% Entities: parameter
%% Output: valid/P30/out/ibm30v01.xml
%% Description:
%%   Tests extSubset with extSubsetDecl only in the dtd file
'ibm-valid-P30-ibm30v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P30/ibm30v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P30/out/ibm30v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P30/ibm30v02.xml
%% ID: ibm-valid-P30-ibm30v02.xml
%% Type: valid
%% Sections: 2.8
%% Entities: parameter
%% Output: valid/P30/out/ibm30v02.xml
%% Description:
%%   Tests extSubset with TextDecl and extSubsetDecl in the dtd file
'ibm-valid-P30-ibm30v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P30/ibm30v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P30/out/ibm30v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P31/ibm31v01.xml
%% ID: ibm-valid-P31-ibm31v01.xml
%% Type: valid
%% Sections: 2.8
%% Entities: parameter
%% Output: valid/P31/out/ibm31v01.xml
%% Description:
%%   Tests extSubsetDecl with combinations of markupdecls, conditionalSects,
%%   PEReferences and white spaces
'ibm-valid-P31-ibm31v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P31/ibm31v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P31/out/ibm31v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P32/ibm32v01.xml
%% ID: ibm-valid-P32-ibm32v01.xml
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/P32/out/ibm32v01.xml
%% Description:
%%   Tests VC: Standalone Document Declaration with absent attribute that has
%%   default value and standalone is no
'ibm-valid-P32-ibm32v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P32/ibm32v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P32/out/ibm32v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P32/ibm32v02.xml
%% ID: ibm-valid-P32-ibm32v02.xml
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/P32/out/ibm32v02.xml
%% Description:
%%   Tests VC: Standalone Document Declaration with external entity reference
%%   and standalone is no
'ibm-valid-P32-ibm32v02'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P32/ibm32v02.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P32/out/ibm32v02.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P32/ibm32v03.xml
%% ID: ibm-valid-P32-ibm32v03.xml
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/P32/out/ibm32v03.xml
%% Description:
%%   Tests VC: Standalone Document Declaration with attribute values that need
%%   to be normalized and standalone is no
'ibm-valid-P32-ibm32v03'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P32/ibm32v03.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P32/out/ibm32v03.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P32/ibm32v04.xml
%% ID: ibm-valid-P32-ibm32v04.xml
%% Type: valid
%% Sections: 2.9
%% Entities: parameter
%% Output: valid/P32/out/ibm32v04.xml
%% Description:
%%   Tests VC: Standalone Document Declaration with whitespace in mixed content
%%   and standalone is no
'ibm-valid-P32-ibm32v04'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P32/ibm32v04.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P32/out/ibm32v04.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P33/ibm33v01.xml
%% ID: ibm-valid-P33-ibm33v01.xml
%% Type: valid
%% Sections: 2.12
%% Output: valid/P33/out/ibm33v01.xml
%% Description:
%%   Tests LanguageID with Langcode - Subcode
'ibm-valid-P33-ibm33v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P33/ibm33v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P33/out/ibm33v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P34/ibm34v01.xml
%% ID: ibm-valid-P34-ibm34v01.xml
%% Type: valid
%% Sections: 2.12
%% Output: valid/P34/out/ibm34v01.xml
%% Description:
%%   Duplicate Test as ibm33v01.xml
'ibm-valid-P34-ibm34v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P34/ibm34v01.xml"]),
    Out = filename:join([datadir(Config),"ibm","valid/P34/out/ibm34v01.xml"]),
    {ok, O} = file:read_file(Out),
    R = xmerl_sax_parser:file(Path, [{event_fun, fun sax_canon/3}]),
    check_result(R, "valid", O).

%%----------------------------------------------------------------------
%% Test Case 
%% Uri: valid/P35/ibm35v01.xml
%% ID: ibm-valid-P35-ibm35v01.xml
%% Type: valid
%% Sections: 2.12
%% Output: valid/P35/out/ibm35v01.xml
%% Description:
%%   Tests ISO639Code
'ibm-valid-P35-ibm35v01'(Config) -> 
    file:set_cwd(datadir(Config)),
    Path = filename:join([datadir(Config),"ibm","valid/P35/ibm35v01.xml"]),
    Out = filename:join([datadir(Confi