// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __ACClassDatabase_h__
#define __ACClassDatabase_h__

/** \file 
 *  AspectC++ specific class database. This is a database of all
 *  aspect and pointcut info objects. */

#include "Puma/Array.h"

#include <iostream>
using std::ostream;
using std::endl;

namespace Puma {


class ACAspectInfo;
class ACPointcutInfo;
class ACSliceInfo;
class CObjectInfo;
class CFunctionInfo;
class CClassInfo;


/** \class ACClassDatabase ACClassDatabase.h Puma/ACClassDatabase.h
 *  AspectC++ specific database of all aspect, slice, and pointcut 
 *  info objects. */
class ACClassDatabase {
  Array<ACAspectInfo*>   _aspect;   // index of aspects
  Array<ACPointcutInfo*> _pointcut; // index of pointcuts
  Array<ACSliceInfo*>    _slice;    // index of slices

  void clear (); // release all allocated heap objects
  
public:
  /** Destruct the database. Destroys all aspect and pointcut infos
   *  in the database. */
  virtual ~ACClassDatabase ();

  // access to the database entries /////////////////////////////
  
  /** Get the number of aspect infos in the database. */
  int AspectInfos () const { return _aspect.length (); }
  /** Get the n-th aspect info. 
   *  \param n The number of the aspect info.
   *  \return The aspect info or \c NULL if \e n is invalid. */
  ACAspectInfo *AspectInfo (int n) const { return _aspect.lookup (n); }
  /** Get the info of the aspect with the given name.
   *  \param n The name of the aspect. 
   *  \return The aspect info or \c NULL if there is no aspect with that name. */
  ACAspectInfo *AspectInfo (const char *n) const;
  /** Get the aspect info according to the given class info. 
   *  \param ci The class info of the aspect. 
   *  \return The aspect info or \c NULL if no matching aspect was found. */
  ACAspectInfo *AspectInfo (CClassInfo *ci) const;
  /** Get the number of pointcut infos in the database. */
  int PointcutInfos () const { return _pointcut.length (); }
  /** Get the n-th pointcut info. 
   *  \param n The number of the pointcut info.
   *  \return The pointcut info or \c NULL if \e n is invalid. */
  ACPointcutInfo *PointcutInfo (int n) const { return _pointcut.lookup (n); }
  /** Get the pointcut info according to the given function info. 
   *  \param fi The function info of the pointcut. 
   *  \return The pointcut info or \c NULL if no matching pointcut was found. */
  ACPointcutInfo *PointcutInfo (CFunctionInfo* fi) const;
  /** Get the number of slice infos in the database. */
  int SliceInfos () const { return _slice.length (); }
  /** Get the n-th slice info. 
   *  \param n The number of the slice info.
   *  \return The slice info or \c NULL if \e n is invalid. */
  ACSliceInfo *SliceInfo (int n) const { return _slice.lookup (n); }
  /** Get the slice info that is represented by the given object info.
   *  \param oi The object info of the slice. 
   *  \return The slice info or \c NULL if no matching slice was found. */
  ACSliceInfo *SliceInfo (CObjectInfo* oi) const;
  
public:
  // Administration /////////////////////////////////////////////

  /** Create a new aspect info from the given class info.
   *  The info is added to the database.
   *  \param ci The class info. */
  ACAspectInfo *new_aspect (CClassInfo *ci);
  /** Create a new pointcut info from the given function info.
   *  The info is added to the database.
   *  \param fi The function info. */
  ACPointcutInfo *new_pointcut (CFunctionInfo *fi);
  /** Create a new slice info from the given object info.
   *  The info is added to the database.
   *  \param oi The object info. */
  ACSliceInfo *new_slice (CObjectInfo *oi);

  /** Dump the contents of the database. 
   *  \param out The output stream to dump to. 
   *  \param level The output level (not yet used). */
  void Dump (ostream &out, int level = 0) const;
};

inline ACClassDatabase::~ACClassDatabase () {
  clear ();
}

} // namespace Puma

#endif /* __ACClassDatabase_h__ */
