// Note: Some of the longer tests are commented-out.
var count = (function sqrt(BigNumber) {
    var start = +new Date(),
        log,
        error,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function T(root, value) {
        assert(root, new BigNumber(value).sqrt().toString())
    }

    function isMinusZero(n) {
        return n.toString() === '0' && n.s == -1;
    }

    log('\n Testing sqrt...');

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    // Test the special cases specified by ES 15.8.2.17
    T('NaN', NaN);
    T('2', 4);
    T('0.1', 0.01);

    assert(false, isMinusZero(new BigNumber(0).sqrt()));
    assert(false, isMinusZero(new BigNumber('0').sqrt()));
    assert(true, isMinusZero(new BigNumber(-0).sqrt()));
    assert(true, isMinusZero(new BigNumber('-0').sqrt()));

    T('Infinity', Infinity);
    T('NaN', -Infinity);

    BigNumber.config({ERRORS : false});

    T('NaN', undefined);
    T('NaN', null);
    T('NaN', []);

    BigNumber.config({ERRORS : true});

    T('NaN', -1);
    T('NaN', -35.999);
    T('NaN', '-0.00000000000001');

    // Test against Math.sqrt of squared integers.
    for (var i = 0; i < 1e4; i++) {
        var j = Math.floor(Math.random() * Math.pow(2, Math.floor(Math.random() * 26) + 1));
        j = Math.pow(j, 2);
        T(Math.sqrt(j).toString(), j.toString());
    }

    // Initial rounding tests.
    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 0});
    T('11', '101');             // 10.04...
    T('11', '111');             // 10.53...
    T('1000', 999000.25);       // 999.5
    T('1', 0.25);               // 0.5

    BigNumber.config({ROUNDING_MODE : 1});
    T('10', '101');
    T('10', '111');
    T('999', 999000.25);
    T('0', 0.25);

    BigNumber.config({ROUNDING_MODE : 2});
    T('11', '101');
    T('11', '111');
    T('1000', 999000.25);
    T('1', 0.25);

    BigNumber.config({ROUNDING_MODE : 3});
    T('10', '101');
    T('10', '111');
    T('999', 999000.25);
    T('0', 0.25);

    BigNumber.config({ROUNDING_MODE : 4});
    T('10', '101');
    T('11', '111');
    T('1000', 999000.25);
    T('1', 0.25);

    BigNumber.config({ROUNDING_MODE : 5});
    T('10', '101');
    T('11', '111');
    T('999', 999000.25);
    T('0', 0.25);

    BigNumber.config({ROUNDING_MODE : 6});
    T('10', '101');
    T('11', '111');
    T('1000', 999000.25);
    T('0', 0.25);

    BigNumber.config({ROUNDING_MODE : 7});
    T('10', '101');
    T('11', '111');
    T('1000', 999000.25);
    T('1', 0.25);

    BigNumber.config({ROUNDING_MODE : 8});
    T('10', '101');
    T('11', '111');
    T('999', 999000.25);
    T('0', 0.25);

    T('1850249442642802149156602', '3423.423e45');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 0});
    T('0.01', '0.0000001');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 1});
    T('0',    '0.0000001');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 4});
    T('2.6', '7');
    T('30490054086086012186828338', '929643398172450327852909723850478224408368639961857.22035621170');
    T('10.6', '112.439121106');
    T('1007.1', '1014216.88354115504848');
    T('14496268662.7', '210141805141682355705.090');
    T('220983929785.7', '48833897223529771949253.378778868849049559624562513592321569');
    T('1180052939140.7', '1392524939174533146207410.12619352287');
    T('120468836204612.6', '14512740496493785900839846763.82328768609126547');

    BigNumber.config({DECIMAL_PLACES : 10});
    T('1.0998820819', '1.20974059415009437172900');
    T('4372247436270791523.3453396636', '19116547643976509183347719022510178407.6659485696744517151189447');
    T('645.6559215353', '416871.569013637');
    T('9.3832989934', '88.0463');
    T('20195394139.0634492424', '407853944432118316238.060');
    T('203574.8431903851', '41442716779.9898655080652537763202594320363763900771333411614469322118841');
    T('434835511792648.9460364767', '189081922315974940615057855618.468666216438821105981678672603');
    T('1870631871126475697594310.8568506983', '3499263597274139582669747936294437345027253159338.2893957584');
    T('76418328144.477890944', '5839760876397101738682.29836547353');

    BigNumber.config({DECIMAL_PLACES : 20});
    T('346.00153309920463414559', '119717.060907');
    T('263359760985014.62241224437474495709', '69358363706084030080212405554.3002793975414740876419285502');
    T('4.14645859760929707104', '17.193118901688058566035426');
    T('140510858365741634743828657.01509424473320174184', '19743301318677505942122892989445395975381650127203766.2408172612421');
    T('492974271191746.46231483998877250102', '243023632057033585989741137703.800100153395469328387');
    T('164.78781194238374484766', '27155.022964758430758944789724435');
    T('7366.78400597487165781255', '54269506.59068717789735301787194313247');
    T('23762326490698.34975320239967323205', '564648160250544549780100380.5514166040523');
    T('213634053277525452.46978666514569063831', '45639508719784583520579147409415270.488869');
    T('5.291502622129181181', '28');
    T('11.47156968462642430546', '131.59691102924');
    T('316358493873927157.1740897995245240711', '100082696646179606634487051294887359.5035240594546');
    T('500048686.48939798981543930512', '250048688859772239.7160140065116884182948058142834537');
    T('6022.37550162532363706717', '36269006.68257686850553015065182301017562876');
    T('6.8460154244205197188', '46.86792719140366873824501484885278');
    T('45189.48536114270306146252', '2042089587.204930656134881238462876156607');
    T('856160326.88774167756915629603', '733010505336524684.93087677');
    T('522130.17609423801622137924', '272619920788.2');
    T('52292561.72399606203863014237', '2734512011657938');

    BigNumber.config({DECIMAL_PLACES : 73});
    T('26887623635.3707912464629567820776000467035000476700672842803068855485151227900078069', '722944304757350004189.2345');
    T('114762757369717325768.5268370835012953651076368760452481091507997658004031794079990971409285748', '13170490479100608368417468277815894123421.4484497151693769463078329436779');
    T('196.9765983083095309326276839454825311907994129311587833723243476358522465777', '38799.78028111312916322430363229763959902004100996');
    T('120016703223.6151811849913819241621570830640571538995217287681670128389432142005932891', '14404009052665322623144.3467403');
    T('91537950494500774895026544968.0163490860552716188715682357057804200505025482975033902996558184955299368', '8379196380733674659135406615691779599176042234171010036822.5636818392');
    T('15.1179067962466285341140019013436699388546686629569704817549228235938583883', '228.5511059');
    T('88.8125725884151110485104354919995585100283886785637267479492612921303631122', '7887.6730497725031780');
    T('3218172063439012577521546329.9090241863260351466688419917169048515469580394951005515857084741579831863', '10356631429899311993763656402168917661960022178561764021');
    T('19107253881509743.3779260316306131671181869224843248420235662435238492667857670998535738552', '365087150892469154433642273521263.83276841');
    T('400.3068500117233817006410985991134968372763336206911625632292309118077689013', '160245.5741663084');

    BigNumber.config({DECIMAL_PLACES : 100});
    T('1.7320508075688772935274463415058723669428052538103806280558069794519330169088000370811461867572485757', 3);
    T('1.0488088481701515469914535136799375984752718576815039848757557635800059255011006914193852889331944178', 1.1);
    T('0.3162277660168379331998893544432718533719555139325216826857504852792594438639238221344248108379300295', 0.1);
    T('0.9486832980505137995996680633298155601158665417975650480572514558377783315917714664032744325137900886', 0.9);
    T('0.99999999999999999999999999999999999999994999999999999999999999999999999999999999875', '0.9999999999999999999999999999999999999999');
    T('11111.1110609999998870049994903925471271162312358157188236047401971668110785559489100264545730662845742184','123456789.00987654321');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 2});
    T('4949554350723.461435143921800511302655747834828959172999243354749297834846074272605873894779037895263058987767', '24498088270765545887271078.495701603537518591752268455746');

    BigNumber.config({DECIMAL_PLACES : 92, ROUNDING_MODE : 2});
    T('34024250164506590698940450635557.52953427667455604566896038084827726129860253810440168217368794648838157166948213114635075589', '1157649599256926764239267638462457331268783688179593190844781949.0026');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 4});
    T('44.40502294864850084', '1971.80606307');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 3});
    T('53661161093118917.324727467936347226946388973481645944102836490011463645342312303037845424027327131', '2879520209861659438086912639838761.96836512');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 1});
    T('14192376.643123461322847366137186892667087869800484670508360531721790645551341594437545247745705539930735', '201423554780276.3686384009255372');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 3});
    T('4855847283088595702176944.8112532951528228009352578372167502504010846351042776625505257853252286361', '23579252836678896488410638492374748088831131237827.786414707871968195569');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 4});
    T('2553252336.0331', '6519097491458426345.3554782053764871036657698339691144086');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 5});
    T('313415140.8429479671745256541153927986977746351343070479080617690487091', '98229050509604911');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 0});
    T('1.537808174192351787259717', '2.3648539806128145775');

    BigNumber.config({DECIMAL_PLACES : 20, ROUNDING_MODE : 4});
    T('62.30949119560079364913', '3882.4726930546528212066');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 1});
    T('20416269776260474006.927083982432268806569265768750039002235065141713481228428261', '416824071577046905366185925400975939799.8');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 4});
    T('174228234.709795821707519878032640346805381350977820681861118903584344693715322225768054125', '30355477770091701.53717021425249799086667828628302277998976752563');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 0});
    T('8346080245081.12842427016192329441165264650792513611706834892396914269233387534464159247031782538653', '69657055457333468703224935.763141948224298');

    BigNumber.config({DECIMAL_PLACES : 92, ROUNDING_MODE : 3});
    T('272.7796965259930339886508185007545884578607374328807903817278380417881107290650631082406652753', '74408.76283681285609575234724616');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 0});
    T('2576.47265', '6638211.30041');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 6});
    T('163256493.009563679477426538597267934928790668772138046453329418', '26652682509781714.54341292267818515442604');

    BigNumber.config({DECIMAL_PLACES : 57, ROUNDING_MODE : 6});
    T('6413898755.503103237879757203310804711689478588298288050385897319721', '41138097245844256487.4');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 4});
    T('651.83586891179899229526795359574620017366743', '424890');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 3});
    T('9.840172762711028743800750259911398446280416492345', '96.829');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 4});
    T('936008582670169403.24586736956499382709796268807769178807', '876112066832219350113028056940121643.4989897905396802');

    BigNumber.config({DECIMAL_PLACES : 67, ROUNDING_MODE : 1});
    T('3.1691816831920187703897673787206897581382652839389771357354751032132', '10.043712541079797228859738302727066114683041689841812835450076520');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 2});
    T('201959414.0905242545521797507742344926679143', '40787604939787846.812484750440288725205988');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 2});
    T('3.235364585328831221179338900994013430389718885202447735097146763445925265374153439191445', '10.4675840');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 2});
    T('1391.926691686756208298398279775140528179063627807536777764086771937670002756456', '1937459.9150300380745526266059714001651697913188367053087');

    BigNumber.config({DECIMAL_PLACES : 45, ROUNDING_MODE : 2});
    T('869.8354526359820716456765000567449871995222312', '756613.7146624438099121');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 6});
    T('118594.4273063315320244062844564170472198563498805200012242127200252960812599679770627273052218882949925', '14064638188.1167541187722341530354253926613230053009933342');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 2});
    T('15459965429.4176806583122774869069153370447277862964242', '239010531078789811116.914');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 4});
    T('5248173.56787', '27543325798538.753780506094441280733789361778933753426582596791104');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 6});
    T('6.05855328019481326674235', '36.7060678489593315124567');

    BigNumber.config({DECIMAL_PLACES : 27, ROUNDING_MODE : 0});
    T('907424493602245756936.066129375330541537462629768', '823419211589292150660922033473258691103594.23268549807171826865266573694');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 2});
    T('11439280.9959241551974895751378888566402750717305622394543095198836564669553210222', '130857149703711.532');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 6});
    T('164207.20075782859923311531097410558094817321169039986325465336651402772907533144002814794168144323', '26964004780.7218252942882108');

    BigNumber.config({DECIMAL_PLACES : 14, ROUNDING_MODE : 5});
    T('67866.91300857901339', '4605917881.314031310494203432328648');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 6});
    T('2.723397260313830449769700482256736114679660269490922765', '7.4168926374848775741132294');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 6});
    T('145626737347.699863159799647089039014085200730659988131836086866937945539247414906654868', '21207146630535962175511.4592');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 2});
    T('3198723.293675674761595756274163847124886137068077187912', '10231830709503.357046273671271695065871403474525');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 0});
    T('19699.655043109373052433709', '388076408.8175045546569');

    BigNumber.config({DECIMAL_PLACES : 69, ROUNDING_MODE : 1});
    T('15625591.477694858909123645800863577029584072458683168542082897370551849161553', '244159109027810.20442572274363387');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 1});
    T('85762713.1327157666976061', '7355242963884497.437336977648000579022368327415840055129120741942567239');

    BigNumber.config({DECIMAL_PLACES : 49, ROUNDING_MODE : 0});
    T('41.0148460976104358708576256643898030638770464836414', '1682.21760041067');

    BigNumber.config({DECIMAL_PLACES : 92, ROUNDING_MODE : 3});
    T('627.06679575379477143240692835573346249172484449442579029111338455375888695749139590976427077308', '393212.7663369313683879605008384834086623');

    BigNumber.config({DECIMAL_PLACES : 39, ROUNDING_MODE : 5});
    T('5.880136052847757025272423749873232616784', '34.576');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 2});
    T('74542661576900.1739714442102128431140459355738187797819043083309413315', '5556608394968269531202589612.9');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 5});
    T('7719200214.3705019090215357285448226659983404326837819', '59586051949537602626.95016591737228440200846895047666959869445081113');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 1});
    T('74629.64', '5569583803.0907402');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 6});
    T('44.05501263636205450407476438041804283847324427155286698836103844', '1940.8441383900203');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 1});
    T('1119.912102354466031064620923', '1254203.117');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 1});
    T('92093628036.6480124503328403126750210810151658188780763586760022178450462311', '8481236324952480852423.718960437545027');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 0});
    T('7.55149177315316950216695542333911', '57.025028');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 4});
    T('1947605507655512.1474872179815308192647511525219417', '3793167213450085186132700081381');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 2});
    T('897186472920280057757.758955997894245720068203481844645803566957566661460082190511', '804943567191132421390195361422932115503582.03422');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 0});
    T('8775397.48306750827108348789044450160735393428', '77007600985827.5591133');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 6});
    T('557716086077.5080956743653150675842959387446784555727262266063122593621607958161147001199667165261203780317', '311047232669614419511648.6130023664452767398438696');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 1});
    T('3.6766253682232922013', '13.517574098263058968167681315043395710343763189956497798543');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 3});
    T('364.437601989709154496727744417233984148798720533510776338306467381487', '132814.76574400966188457585354574350011547665650395112601058');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 5});
    T('29784617918.0127174734394784458305404701929975', '887123464522004224898.5545193466489');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 4});
    T('374863740368.0156985152529892838083189109926642299794', '140522823842699082383715.919571886197001394994582');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 5});
    T('3321962129060763825931501.6171393711588977731624802456157919679857160686287774881', '11035432386913922898118885283020330437836330418535.765864');

    BigNumber.config({DECIMAL_PLACES : 63, ROUNDING_MODE : 1});
    T('3039991938095044282.834999396730197828406224765246402870200714765906547758763203772', '9241550983682863551151813316288949279.9625949439074666');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 5});
    T('5.5727910421977962406401022785010286010813965565340383771288278905072628945', '31.056');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 2});
    T('1809373906.386837382140423787863676125946315451309506008234344142479881132815536036260566', '3273833933113563766.3701096193056864');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 4});
    T('948722.35086770259132293410301310241', '900074099035.9401838348750892');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 1});
    T('23603002885797536.790280728725364655699852314300322649', '557101745226966849549415425061219.969394346783798760227');

    BigNumber.config({DECIMAL_PLACES : 71, ROUNDING_MODE : 0});
    T('1.3760699520736582089086174843894761768111119574016193232032195760251607', '1.893568513');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 0});
    T('14865553.721249212888439807434', '220984687439346.321003197181614764009809253917');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 1});
    T('3293155557597199373259839225.706003736695865575939572351947834646050410715323588632698181660823328571895043', '10844873526533321118731760593542551681937887331657167954.4066926866');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 0});
    T('1.4602844538376761066216387158903803538206186160126991802', '2.13243068612');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 0});
    T('52497683.2535666615551547765238857555777973908683364642662996875860780953333112481757317373', '2756006746991813.49967765680');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 2});
    T('229.108483256294986105584052287445152310628239217404673001762961439710445106676980785785353195978212', '52490.6971');

    BigNumber.config({DECIMAL_PLACES : 18, ROUNDING_MODE : 1});
    T('247788926631.668855265587728589', '61399352161274570866824.398764295216033428917127');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 3});
    T('21424.038976941299431640698751966107735138702410669278306', '458989446.0855');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 0});
    T('719485.015161442678887199799035233897302403555352773104081712230052841078911919691996286833814524', '517658687041.861401507658');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 1});
    T('5.8309518948453004', '34');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 1});
    T('756731558.25262830674557663857258007802853198', '572642651255450988.3807853');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 5});
    T('56599816.250503709163296236449780271198341341082422352679019828047161', '3203539199590783.75467227017766942');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 1});
    T('315.35186696767786854256245741213', '99446.8');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 1});
    T('21743622919.6152043904934006640996008653', '472785137670415625131.3819412384141603569725008109774351813892945');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 1});
    T('9931.93848832625073050157085444478223645326098496189725178399175', '98643402.13589633051811977376659116574508796');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 6});
    T('12491945070243173325.00123703924746770276608629177675', '156048691637972720536868554818649758614');

    BigNumber.config({DECIMAL_PLACES : 13, ROUNDING_MODE : 3});
    T('31142.7911241108894', '969873439');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 5});
    T('323.97605471', '104960.4840234371387');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 5});
    T('5715183064.0940040641822561', '32663317456106928967.132035290253014719696309540999261182915037018712');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 2});
    T('135.35229191926174097438753947924689763303130529240391689831443968668652723708826191007753307865150657', '18320.242927797047184106');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 4});
    T('1.986294948124507296707366989944167364007457', '3.94536762094493913289346223994950919328456012507449654266826');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 6});
    T('1854467167302463060.67001538852537388423828924920448451675904798181668319798201139', '3439048474602821519577229021879982239.2082');

    BigNumber.config({DECIMAL_PLACES : 47, ROUNDING_MODE : 2});
    T('81193654.78972507709037814751156680233229599082421344592', '6592409578113045.952464677461833939906456779681791713165521338857097972');

    BigNumber.config({DECIMAL_PLACES : 53, ROUNDING_MODE : 2});
    T('243658571.81955213863332506453273200107839362175562305528292871', '59369499621143847.5904454643435133899921413');

    BigNumber.config({DECIMAL_PLACES : 15, ROUNDING_MODE : 2});
    T('1.972442648088913', '3.89053');

    BigNumber.config({DECIMAL_PLACES : 44, ROUNDING_MODE : 5});
    T('29173231718.778879604378348144121047338664684120880285', '851077448917566101882.514088939153719024901521');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 6});
    T('2663045169970861026925375.635420072098834063612997764970291415141689017483375036375781735588621', '7091809577305132096990572152434051531398473618829.89422557768661926');

    BigNumber.config({DECIMAL_PLACES : 88, ROUNDING_MODE : 5});
    T('220761.160604830397059610011998293759083892046977144231182486245187301213113818936381230594214', '48735490031.5917204821');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 0});
    T('161677855435317442406.2546035610007912236205334351', '26139728938163405854159841438358732967516.30006485122550144217058909');

    BigNumber.config({DECIMAL_PLACES : 27, ROUNDING_MODE : 6});
    T('244041886899114069999190717.069960065813984399980837008', '59556442561279983551883153005400521473838950253623262');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 3});
    T('1107466938973440740730667939506694.2638801341959912170851419737317922658410730057816703901502302', '1226483020919202717870476287034767333832211470320394554682503342877');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 1});
    T('2088927.797138422758659686170900239830365738839263869523289', '4363619341657.58350562244267149496471405124');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 6});
    T('90061141477839967.2682846126818450599184988316687847204716382695549492605718861', '8111009204291506563499764980695382.1759246268375');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 3});
    T('5429621071395721.253939502546988879109480438494987866198414850987001042506015656799576011128802517427918324', '29480784978944419958421524887597.42349107833570174813');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 6});
    T('1476747291384.9099630858', '2180782562612668171679967.63324');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 6});
    T('9210807993.639856342437524083401141956511342156180578284602614905116773271331676341823693120447', '84838983895699875876');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 2});
    T('2.388242031285774017289777195752330791381535517849', '5.7037');

    BigNumber.config({DECIMAL_PLACES : 26, ROUNDING_MODE : 6});
    T('3.37200237247840619012635848', '11.3704');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 6});
    T('113893455834773270196180930512.9915840908366637690221083458', '12971719281987449465429560596505161586489324719573129843872.171');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 2});
    T('90231.00398976', '8141634081');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 5});
    T('1597579508606622211.845652947503721264640830072971865988482347827730509123712404166521804347520039996034940695234128954', '2552260286319776493852490523528001935.48994776557');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 2});
    T('1299424183315225375029397575.07015635096795459', '1688503208184440439916254126997862516329616872207524279.05763620');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 4});
    T('2693874.2163861651089144', '7256958293710.1751161960');

    BigNumber.config({DECIMAL_PLACES : 74, ROUNDING_MODE : 6});
    T('5732941347.55256238286493674720703619067740838113000509955135586822000095629002988639', '32866616494477789872.3518055761631860371');

    BigNumber.config({DECIMAL_PLACES : 15, ROUNDING_MODE : 2});
    T('68038.789982024004217', '4629276942.217969996065357856482669982723691716245127238');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 6});
    T('16288419988283588.674', '265312625714716342994884296792766');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 3});
    T('3.0446674695276658405945560356807623150745767505665219956760952761834211468101552516', '9.27');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 2});
    T('79.6969259131116047841695097300091923882482217290138206', '6351.60');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 0});
    T('126.370571227352967491262772615456540499132900009240252107883825190012068734134048217720997378442783801', '15969.52127232748969252');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 0});
    T('88796173.2384202907121064495468439111066566843', '7884760381787547.81840923');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 4});
    T('365543004.8061649872764474380277913079084143042', '133621688362719958.9872107366');

    BigNumber.config({DECIMAL_PLACES : 100, ROUNDING_MODE : 0});
    T('143498617762600.4140327037226405237893689515984193391783866884659658377021608257588943250828403643110208744660318261', '20591853299776899056200145422.1846447111664776');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 4});
    T('12782418240925459479.2031797451385115697726882124289981424172567856909899486963994', '163390216085943917855551726128161143915.02664633');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 4});
    T('12.07324881148295353275508188921102928327377', '145.76333686397455005144032806105');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 0});
    T('211449127401352679264090.403777356355051577835784756039917073614638523204498613898150412748', '44710733478793476460089809808062830465327473341.01106');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 0});
    T('8434.69565989858107121020381907194564097406849153317203553829228632792387361758032492018751318', '71144090.8751119600030');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 3});
    T('431.790690033956151299657456473777574071370518767442408773272403696713563734140114626948052851', '186443.2');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 3});
    T('1.108018699799390988378447956003009', '1.22770543910513292750985');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 2});
    T('9581559.366976805376', '91806279902900.9593482950972');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 5});
    T('2418839541.0987386261807014430009304412452809170287221944718963714320359391426925314537124868517232384773', '5850784725582756467.469546239192868802578236912006511');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 6});
    T('11040265120287567759934269.263029933332634568404407042737837112821286414662597953247828811089203658709412952839364191', '121887453926238263019360388091188570494838624535471');

    BigNumber.config({DECIMAL_PLACES : 49, ROUNDING_MODE : 3});
    T('233.2570187785530519308734053376321961140912819913698', '54408.8368094582511025342');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 4});
    T('29.6490992649274348544759045400715775766925', '879.06908722152055781935911221883768070437');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 3});
    T('7505560981.6906822826179406598', '56333445649877598342.8475019094481329264104906070647412405735303985455');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 6});
    T('4.666365343768356140513396346961342877525930132228624', '21.7749655215223685799362730400310696');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 5});
    T('209228165501008616561872.1477049513127309490959', '43776425238917452235867870727072879650986013517.690781514934992668');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 1});
    T('17.122151356727858418696401754969183602519558', '293.168067082697642757847');

    BigNumber.config({DECIMAL_PLACES : 68, ROUNDING_MODE : 2});
    T('32739585325.26765210723377428518717434262024156135815886665844964799952569026833', '1071880447270480993629.4640435138594357807725479611653996305');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 5});
    T('1.6928161743', '2.86562659997131621752114765980023438405067892440724267945937722378861');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 0});
    T('16417.554697867626966836606837813475423042958767384549424626836426490725472399363532843792074346', '269536102.25747538817901717219695597174449981');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 6});
    T('109652272385643.1603769184581163062699906986', '12023620839335281581899617727.8772961473185586727206798164');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 6});
    T('13489.123629030538320414969169194877155101718802208604977345597134035561855309153297129880622467', '181956456.27927');

    BigNumber.config({DECIMAL_PLACES : 46, ROUNDING_MODE : 3});
    T('15162908032.86965932250477929505589077613516814656108803', '229913780013263241677.3791');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 4});
    T('2.372129844675455365912878765809140727257324413647', '5.627');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 0});
    T('148.820093791127547454620941738144491984235834280198020332', '22147.420316');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 1});
    T('739871537758100310.43883822919081011754036378492392360075543', '547409892384536053343836850938988791.2508666160979');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 2});
    T('63478.7126270666964166648665771468502603343618655904464016578865228572987361391362408539907916', '4029546956.789716846462443886848');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 1});
    T('77140.793792105688747751462443855549841986175279935245610391', '5950702066.8761715670568717062071879678486494595966218026');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 1});
    T('162820594803834668.752129825654424512211529624619694188452350414129430986701621110932095616631076098099130503248', '26510546092274513134180208644185957.57422670990654');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 5});
    T('204494310168.4671924285173237222417467622712661919483412527661975953571549254104320801', '41817922891277264574994.94380');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 0});
    T('4395.909427668232162220120918266434', '19324019.69626244445210257981');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 0});
    T('5.22775235102390824796354516485007398148629306999271', '27.329394643636');

    BigNumber.config({DECIMAL_PLACES : 68, ROUNDING_MODE : 3});
    T('997295.09412022112243931640765631069576943976068322455480694643995328956477', '994597504756.26070722226025041925449713707868740387903881304557');

    BigNumber.config({DECIMAL_PLACES : 67, ROUNDING_MODE : 5});
    T('203637147057.7748844672269277496593344371356921412578700817688601724867963410626', '41468087661829834285659.350412232933496521226586662970934783390865142');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 1});
    T('939333.959604513485668563847772506141997730133849933463092593953521840168273751255714815246', '882348287666.2937728893202363');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 4});
    T('4.568369512', '20.87');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 4});
    T('370.818280526354041083927163168590728498008550586626184882319882834674931772525681078764889', '137506.1971725218006486309891482928119051089854653');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 0});
    T('9677366.13442750739243118863157357061982716666842510521012211042628756371590553458718', '93651415299764.397079279228147695065210419679');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 4});
    T('459260874.20127731126343992531130965108732', '210920550572121463.814947484719373265508800614967431627085952061224891');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 2});
    T('31.9734264663642204389422311887643391814535306082202464098568847002787713669652098830099427401843', '1022.3');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 0});
    T('2973063742', '8839108013968540509.06630564811');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 4});
    T('34146881023818508.954756503743080316053431668549275132795068', '1166009483654816582318330803409852.007911');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 5});
    T('7.549834435271', '57');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 5});
    T('457768114718.924178666375597479663315126937742538839621311715859853903792113549247134805042232897618803407927', '209551646853318127271063.5596189');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 2});
    T('2458857125246799394445091629.502227209261508743296310153768826317308293252686594142446545917844105369673045257530483701916834102', '6045978362376954524014907890057287078792404475852907401');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 3});
    T('232.95426589783669381174298578564010084473683056750226995822363784044724761408941752678312753', '54267.69');

    BigNumber.config({DECIMAL_PLACES : 26, ROUNDING_MODE : 6});
    T('411369926586282.35173941504630680193130958', '169225216499603331644025273726.130552532747109702369176231397148094894');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 4});
    T('146.837594711181641908188613240193044745216301', '21561.27922056523889531529752');

    BigNumber.config({DECIMAL_PLACES : 80, ROUNDING_MODE : 1});
    T('2436.80241424922206290898076140846108744470830585565672963845308477988719476570192112', '5938006.00609083724509944');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 3});
    T('237641.142572063817719734051264723120416667001252355959725', '56473312642.9559622068518833017478');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 5});
    T('2.54986118578098487598754727764', '6.50179206675241026831530806973871597654152197264612412571');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 1});
    T('5968940791314695.9391845388865719907476112181090409748506', '35628254170220508537420332407309.79446040');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 6});
    T('7757883860673599094.299765297', '60184761995699906684011270143497832606.405945309948');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 6});
    T('2.2181073012818834324103200205219865836926753', '4.92');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 2});
    T('603896.1020372225715031692835372960243779222201981644175886120182699827032153603', '364690502055.7515356757459417779202315');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 4});
    T('910.9629520458008175934592151820959436914447449057963517034109551518362704397', '829853.5');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 6});
    T('4.242640687119285146405066172629094235709015626130844219530039213972197435386321117', '18');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 4});
    T('1259229966702.622212273094', '1585660109041887045436663.9');

    BigNumber.config({DECIMAL_PLACES : 26, ROUNDING_MODE : 1});
    T('163.04833062468649267909332731', '26584.7581194970792');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 6});
    T('4204818551811.1334068235516094940662770670832347987097641633616076970760141535', '17680499053655077194352936.9862782964932');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 4});
    T('39405180.06334322', '1552768215824502');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 6});
    T('11019819.18', '121436414839837.12468');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 0});
    T('827301397472.2029918861527516536634619742535183', '684427602259459998932963.33834317372593105949228354646289796636815');

    BigNumber.config({DECIMAL_PLACES : 46, ROUNDING_MODE : 1});
    T('2.7666477911002694164386944265228742439351110337', '7.65434');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 2});
    T('423.17097521237558847890270524607668156801132098651031712232154305066696431425891584532068', '179073.674262192994730735440');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 4});
    T('79.8964796181302813829514', '6383.4474553703074736391804543616');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 4});
    T('256884.8986204117896931254014585418288760261941990857711922965396545498637125556442506103725098664222399', '65989851139.2192430115427289');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 5});
    T('208901.771039551638948558459935195746717360564690378279861596712182280143337256297483637095086136', '43639949943.461255846177212498507');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 0});
    T('16935269806518296805647.546457060582195325905464915681524239174883463381011239795201896196859153349350411010935076375029', '286803363419570270122726919163225156875554457.5998001852');

    BigNumber.config({DECIMAL_PLACES : 26, ROUNDING_MODE : 4});
    T('51576.16987553369007338491641136', '2660101299.0299088953962881552848995811');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 5});
    T('19594015.1384231226764662349318', '383925429244754.5033');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 2});
    T('13803.380831002140583034541319780513750531431940983173205661145824004926065412840500070745', '190533322.36567734512665230751076729702896');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 2});
    T('3.87298334620741689', '15');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 5});
    T('2.59229627936', '6.72');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 2});
    T('27378680.92056694834550996640150630587355668564', '749592168950216.6421');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 3});
    T('153816531.812650974480814609014611069782098313441090291614251952157913199023826783428321238042213', '23659525458872269.27721846475011520302565099737');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 0});
    T('2029498.8125968959', '4118865430332.21');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 5});
    T('7473970526006.1604681414601953537014727721512435634625424159607807892328716740512518', '55860235423608802990631308.948719255851037211256093771306843473391914');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 3});
    T('2241646555429607930993.608927689837', '5024979279469426302209565333918945376304889.403534974');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 4});
    T('83117929425.537107714299371091957543379566', '6908590191988567392806.80486418318444628');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 1});
    T('65772725.59339738278078887228018658173869609912689', '4326051431984351.09884157745017974');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 6});
    T('40227541142203473010813785626410.19087001878840388060497980542506597235802568540711096477426661', '1618255066347673101991637745640498016905970250767945408572170702');

    BigNumber.config({DECIMAL_PLACES : 46, ROUNDING_MODE : 5});
    T('5470786820.0139452521385135579833672428710925952543352056', '29929508430038295403.2021080448706293');

    BigNumber.config({DECIMAL_PLACES : 98, ROUNDING_MODE : 1});
    T('353254.68729598168174020062443678572940775163547873542847911555211434402472387360698376451692274187949171', '124788874096.581801769341529731');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 1});
    T('4710250560799.345071943574342121547002', '22186460345510544746149875.21079977475782182108767576173806601062569351');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 0});
    T('889742696550533491635.902878780831679082243711108150405478590463', '791642066065014722475930480934055487526379.2980796495');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 3});
    T('54.68423930325994760604974477176105234826564193829924891568556', '2990.36602817620');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 3});
    T('687.69813872076169812204913554689609442', '472928.73');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 1});
    T('1922328.5814452710447231491355356265501994748371717620744762610836', '3695347175041.3880723243439344');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 5});
    T('259.814919028223566794421', '67503.79214964236844196364376129941576224666');

    BigNumber.config({DECIMAL_PLACES : 15, ROUNDING_MODE : 4});
    T('89116010281422470.490792878028479', '7941663288478595468163012949770631.031610');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 3});
    T('5932381112762754622904981552.373251673627440940633090556721714745', '35193145667064258780606184878777954613913495498807552110.674');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 4});
    T('893056745827291.2797429722245393357197289561968389609411292397482876045', '797550351267631140592697284483.78129');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 3});
    T('3763501642015094266579640.1219523405439829814657228094809771542386037789129194751290098235201476058052607', '14163944609450310758114750482357005006752272738570.57225865671032927');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 1});
    T('776.5090582120990070056744810742483620827196187879468568581488531077672537925651203290541573461', '602966.31748544096431040937268');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 2});
    T('48.604526538173376182800519112210650467428825886317769747923817613773340459260266201', '2362.40');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 3});
    T('1619107165354498184885.5242', '2621508012902278327380962859916298137513959.447556874953217092');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 1});
    T('2995751.634073306004244447026435588871407905052680216908898588671550117511803869048557530431723092132', '8974527853052.88312');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 2});
    T('75467153073996458962915131484.3993555081', '5695291193094013185500551046546949646554620436546808622591.516878456685');

    BigNumber.config({DECIMAL_PLACES : 18, ROUNDING_MODE : 4});
    T('10256041108976420629.199454567907275476', '105186379229014287888491059604397048384.620569192314');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 5});
    T('1663.8453977796919070850392655525052909309077', '2768381.5077126611909711080090814986835313295705839');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 6});
    T('123529131429703819443118.54931511124012408797939563765987050611899', '15259446311777039991024049097756900338962157854');

    BigNumber.config({DECIMAL_PLACES : 57, ROUNDING_MODE : 5});
    T('5560438.142579741702366402248857210395589101314248740685263570978', '30918472337455.6479130284098891831113215775590418915104774');

    BigNumber.config({DECIMAL_PLACES : 68, ROUNDING_MODE : 0});
    T('36281644.76988135818686116202939957191270810974653912706871975492103944369636', '1316357747207859.31266167');

    BigNumber.config({DECIMAL_PLACES : 57, ROUNDING_MODE : 5});
    T('5.282991576748916225741027424914998052144656285680852715908', '27.91');

    BigNumber.config({DECIMAL_PLACES : 67, ROUNDING_MODE : 2});
    T('26826.4127493781769703743437728144843044287785972406934806238303342625783', '719656421');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 3});
    T('2', '5.37625548458271122007');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 4});
    T('1668.6764282929150311920750595338744', '2784481.0223404');

    BigNumber.config({DECIMAL_PLACES : 67, ROUNDING_MODE : 1});
    T('10.2385664037012525496880466372434589015640964461096948083431020419951', '104.828242003');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 3});
    T('66372960095452040.45994587668260499913729362119803066270517659552152382678744234600982770351253058209521', '4405369831832468935843830578406093.2565156168971');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 0});
    T('473407875637291658578.9019', '224115016715413405092267530282257875406682.49835382');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 1});
    T('68651941131739807.174750106032127662294695087292808832370099322483965602512157211426100868174677659683', '4713089021155867956380018732563084.26698833111');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 2});
    T('7348028468128439177110.54556973', '53993522368425976483646062280915010569224044');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 6});
    T('2119642072.680265413658964490823', '4492882516276291566.29881022242536487557872701343389871875');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 0});
    T('759911.9536428739189649', '577466177289.32936');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 0});
    T('106553761.162264246284252054647064491', '11353704017824852.4611643486114409049688739044');

    BigNumber.config({DECIMAL_PLACES : 100, ROUNDING_MODE : 5});
    T('14.7456978008019568614040993673761366504625708181901764989192874845947809143096249709002564722127596636', '217.4356036325756670544458007125524938');

    BigNumber.config({DECIMAL_PLACES : 98, ROUNDING_MODE : 6});
    T('295414.57241487468020923791493376733960197400575508671338810222236013093392808359746725949898886883126268', '87269769595.06323634741877845138');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 6});
    T('37516617.579099132622513300879884121967683', '1407496594576370.06261787957718');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 3});
    T('1131208352281613015.555408144638994057195008122161118805442530013839952441611311', '1279632336271681894535673114499118754.39788237415185486830');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 0});
    T('32966.69405563486904244570818399194304016536448224495617094756766158966137083438144470645864678778946', '1086802916.95783141');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 0});
    T('31309.0857947795921595950486356877036608601136216829953948182363', '980258853.30486924605479');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 4});
    T('1770073.1489207444615594580039659422', '3133158952530.2');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 3});
    T('1914337.5501650280798142520555737935991361014908729126592669', '3664688255971.8414000128737374263789123210685');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 4});
    T('2062559182194.651333939625538780888106868996091400331597659659567648164339407129332825200701125200486512', '4254150380055468916249334.9191129113190464726018607977');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 4});
    T('12.07145393', '145.72');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 0});
    T('1317013851091420.6561408685542383805866756541198653673575459876412143969639093098412316192938882', '1734525483966654741818421922573.810904219017657492124021067226646767023');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 0});
    T('20690303.52001400622355151906987833847090254985134637242281', '428088659750303.9764329');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 2});
    T('11606.7592352505839337735348351151776393253628311', '134716859.94507472');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 6});
    T('128962450782451915346224.663', '16631313711816332632714461413913179482268224163.627580');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 4});
    T('25034434221195.1626546646428355078518084227214031163523557198228259498967796293623093732280578488089414910009545', '626722896775347450122233568.221736229089870371148583168');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 5});
    T('5280981018824947175.23073432389104277713300016379472042681039852099605839769140560424539432101265', '27888760521189377071172993311260489494.9228964746031216476847321190');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 2});
    T('162673090405691503.024889494298383043507184034396478832316388582171564767570698269884558142411454657165336758', '26462534342138280932191837518704625.0840975098546');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 0});
    T('5.56776436283002192212', '31');

    BigNumber.config({DECIMAL_PLACES : 45, ROUNDING_MODE : 0});
    T('7682710.552671708235374165899366979310815641756553575', '59024041436133.2246');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 5});
    T('6391.5469', '40851871.991');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 5});
    T('1834507.975076828570245235511028215397', '3365419510620.485874651367130674850280070521899330536764207749717544');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 5});
    T('3495480.38256147827415476994571475428636224261289845004330975322527090233565328776690121708467858140792', '12218383104872.1385087696898062860761518572');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 3});
    T('1.26009423159539937638421724613120826', '1.5878374725');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 4});
    T('1369370604956.05098382437442746815592765507526977434916820834002', '1875175853717701043262789.163');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 4});
    T('61111046097989.562614478991002463202122550327564207016373610481385147575687339517642575745', '3734559955190605346508566151.747128467087991987800252614000906073017');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 6});
    T('487839154280.605258799', '237987040449216180199565.97996551897500100835854353148858639128012');

    BigNumber.config({DECIMAL_PLACES : 46, ROUNDING_MODE : 0});
    T('26026.3809206474465998782648730859947008804679763994', '677372503.826641430068000281530877034016405774877793');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 2});
    T('3.5355339059327376220042218105242451964241796884423701829416993449768311961552676', '12.5');

    BigNumber.config({DECIMAL_PLACES : 80, ROUNDING_MODE : 2});
    T('189237794910982562213.25636508486094088772004804584906546011408616284058885391779627042175154320806668', '35810943022771097721298012826864918201819.2991');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 1});
    T('3771198.65750627716579658341655388678', '14221939314377.1471847');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 2});
    T('29135124.530018624988827182698', '848855481379693.0036377');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 1});
    T('998223.9287829203446196803384884403292209921824958032442949937536171090461827984654389784946553623', '996451011994.8088280477062132');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 4});
    T('5126338484054822038.6885613203', '26279346253101490909404023029363590984.22045482623574089');

    BigNumber.config({DECIMAL_PLACES : 76, ROUNDING_MODE : 0});
    T('320594084.2355540715784886905782644237424727036120667152617154249798799923426746035027', '102780566846833539.6758377523578');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 0});
    T('2180790.7087754998512230743198519235426181143993335287947509925434061761510853', '4755848115481.5470038067258431');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 1});
    T('4435299142639732.0386665272768109416139138688675365733', '19671878484700742088824375591214.762823606521949640217195100557734');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 2});
    T('355924722776.6529947424', '126682408283637287091499.8123529985117620693781205410971392972914');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 1});
    T('1840.27953983121746818190253230579', '3386628.78472139752');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 4});
    T('18505.45444317533478496724906701013', '342451844.14843774');

    BigNumber.config({DECIMAL_PLACES : 74, ROUNDING_MODE : 0});
    T('1380998315760.07536639428018906694759131234868833646200425311534294649362527033527763552', '1907156348132164826104731.69607807136577');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 4});
    T('15478.581565616610788736047569976878217626279567162328064930393795809558245153963295159638', '239586487.28344637');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 5});
    T('20814354.1066762193292296130907', '433237336878109.19638027119868364020593141');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 3});
    T('65411343534377893951480143.0687813', '4278643862972400711315519535941613397047166597617356.53505313610762');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 3});
    T('26.161836991932925045265787369493587190351416331869263156799084171994782512198', '684.44171479247');

    BigNumber.config({DECIMAL_PLACES : 98, ROUNDING_MODE : 5});
    T('4.01130901327733662798257776541675440146186278181542626669368510258079411649993922721806882261901641', '16.0906');

    BigNumber.config({DECIMAL_PLACES : 47, ROUNDING_MODE : 3});
    T('1149242295355.33233473660241121236631015179107791864017709795', '1320757853433592920847063.41923783204587387124699776335');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 3});
    T('9474.726373832', '89770439.858993859142728578146543311908');

    BigNumber.config({DECIMAL_PLACES : 80, ROUNDING_MODE : 1});
    T('8.83176086632784685476404272695925396417463948093141782621020297255713993823544284', '78');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 0});
    T('3830962978040350811709514.1639124190557328292714921812', '14676277339115793415584423492635818739231491629096.2');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 1});
    T('904094739518026362.276858998253417589801385050899447007512426103468322262', '817387298024167938863983875545245626.80');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 2});
    T('3898018.2159', '15194546010949.81802721206885605');

    BigNumber.config({DECIMAL_PLACES : 69, ROUNDING_MODE : 3});
    T('184916754268166424.532451013579940568421105901355402407206681792452037755126944803242074', '34194206009073445584566849436943803.6680118993321070740');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 2});
    T('10728918189914538012.14436634401825539813105200873037997842267828754845567', '115109685525879046747891569700090752540.64852941266');

    BigNumber.config({DECIMAL_PLACES : 53, ROUNDING_MODE : 5});
    T('5173978485.87510754763548985924623523505620610130206120024600441', '26770053372298470472.82008752828');

    BigNumber.config({DECIMAL_PLACES : 57, ROUNDING_MODE : 5});
    T('12448083579765518960272733448.844733656410419557424433748732709492340453674614639609498', '154954784808827937239154352490553377950290394508934428937.6950287957173019');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 6});
    T('21937056544951997887838.266472241058656756241231521937385200509335828305313379527318257363340679517701971098439856037229', '481234449856421286927459543035431540893700257.7118353406076308244033');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 2});
    T('8705805512511376.661415004088266418167596075698134210732650646137955094273212000791730326866575509931229837909597', '75791049621673473659571306712104.5913470');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 2});
    T('21744792388806330.69543805888148448016988971611269397259934845377013557', '472835996032289729681394667373604.66074661819409142691068');

    BigNumber.config({DECIMAL_PLACES : 69, ROUNDING_MODE : 3});
    T('364148032238911.207459019623811970686792287639949655428263721201429401774057623001667', '132603789383471116094969998370.54564120810660443599509');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 1});
    T('6130.86992934880235087497305800565551205615083568452544051362005512506946811033000392166546', '37587566.09059338872940942140');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 1});
    T('148902754396098661216264733282.2432677199650259537991922683099806354990954412183841781360864322902131461744', '22172030266744879178523766174920227318932007998329511426386.71431306');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 0});
    T('2.485558288', '6.178');

    BigNumber.config({DECIMAL_PLACES : 71, ROUNDING_MODE : 3});
    T('48675762028559728895548.12665897790225369108328127728148623968749329776515932108839536622078517', '2369329809060977133824105060183733685306224037.8');

    BigNumber.config({DECIMAL_PLACES : 76, ROUNDING_MODE : 4});
    T('24.7386337537059632989284591358444621508831953522417226063918014385697260780257', '612.0');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 1});
    T('31.610335068913719526253073315531', '999.2132831689965253909026235101062925484544085049');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 3});
    T('381.951720361337291359017000726333760115427476940754', '145887.116686985200');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 4});
    T('10249810.4776964144606649104434449145329654563', '105058614828695.2');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 2});
    T('4565314072683631718401.4254286569265369656898689812980558791518310535475002322479184308757040096111213195929252587721798', '20842092582243208192634490035016683394003768.897');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 5});
    T('11921.919157895436980161095883273297050465674732349580111298784795959952105486977509191', '142132156.40739424522513940329925953476743994585969');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 5});
    T('42221784994101.50903525126461999032247998888873167843', '1782679128088135365358664827.18789866270490430767365565');

    BigNumber.config({DECIMAL_PLACES : 65, ROUNDING_MODE : 3});
    T('97847.44665485852461064907480959656322328400159999501985784579886747875', '9574122816.8753846778');

    BigNumber.config({DECIMAL_PLACES : 13, ROUNDING_MODE : 5});
    T('234205.612503318998', '54852268928.0548121688');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 1});
    T('25529284068924417.312616362288832310787818', '651744345071838052967175093720670.309523048811416290');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 0});
    T('90387.925862', '8169977141.632377593');

    BigNumber.config({DECIMAL_PLACES : 13, ROUNDING_MODE : 5});
    T('39727850756.6674266118418', '1578302125744040621188.10968243099467673952848');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 5});
    T('10922946652.61277809893232338572941735232941', '119310763575824694072.67899');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 3});
    T('1204691748.01962453331240668450742564575254687343257', '1451282207746578530.68');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 0});
    T('2168167871148394953234.1925145526882349014061487960355124218083724786535449969286840357060655099527403', '4700951917480162980663868559509055318620701.436239024401270');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 4});
    T('468012.3980129886634423908716', '219035604693.86811504914527871748991009824119534');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 3});
    T('152027.9034509030131743006877834012496725', '23112483427.6770883018541414');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 1});
    T('39070.768987', '1526524989.297643871802383576028102147694626812685590152575');

    BigNumber.config({DECIMAL_PLACES : 46, ROUNDING_MODE : 4});
    T('1250964.0328412267817990068707571462032258488049181937', '1564911011462.3859182790021956920389318080447013349220674');

    BigNumber.config({DECIMAL_PLACES : 78, ROUNDING_MODE : 5});
    T('1.574747598823379695455819287781216314798721967909499025165940557210793017153248', '2.47983');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 3});
    T('99.82965166080369078090076790080975783846377844493852381966581290508065966343620655848892826147126', '9965.959350717405097');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 5});
    T('647.729457744512342323271374802023230498738468582236031741072278', '419553.45043');

    BigNumber.config({DECIMAL_PLACES : 80, ROUNDING_MODE : 6});
    T('24697777055157274614045404601142553.70406771835510501104735791650848400272399441549342218507230159973062171502661103', '609980191466253139733233884276803110291097008877197723579364303731893');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 2});
    T('13.6431266042795337', '186.1349035404');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 5});
    T('173.2188929263660950499725455519748713007516020857763370494332244159518279439900397902257637451', '30004.784866635882');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 4});
    T('46143126703751828496.866649027371445895940007418220059475791917987944465104919918209683758208288829164', '2129188141998495085389252895218765450927.7063857106');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 4});
    T('64866.977655547797763275486444509839974560207365651674817907766434', '4207724790.165337269565');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 2});
    T('17584290857147344.415797971139463901060315043736480054621679443194306011937028118601468078448454466113714486390327668', '309207284948755688576114250910516.74');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 6});
    T('2.94358860030385493881670889399291911046241548916385003586188315285977498131365266043', '8.664713847838807868001451403595636253922161359756547');

    BigNumber.config({DECIMAL_PLACES : 83, ROUNDING_MODE : 6});
    T('424.99280768859670787492492363122383472286022111724084853084296368784059306715270358792', '180618.886587036545015612122512773593063212916813764199769387313');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 2});
    T('18.31090032843085055141567877490619869051620042759275170646107', '335.2890708377290305906582988852002980024');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 1});
    T('126427741490350.05876064315047378101790577982019123921425265937548', '15983973818350781697093926569.612618127837088601804542393750879971390');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 3});
    T('16.692445803416586201775384489366982', '278.6377469');

    BigNumber.config({DECIMAL_PLACES : 26, ROUNDING_MODE : 2});
    T('349210873373756.75981101873418700429959408', '121948234082461977906208449874.02898224579023209742993297127871656');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 4});
    T('10.406728592598156155682425873215240545', '108.30');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 2});
    T('154577239693381679058287.97436120925772268748839189894586354157517982123268495410171938553647222928153513557194250214191', '23894123031225172524806703141007699579590862366.2631');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 4});
    T('1183519938.0327546884078159164077132538648035819895600967199921207723839756282768130542759614616292199', '1400719443721055497.588328093817233056244752470');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 4});
    T('241257503.34869155337527048121955221501097', '58205182922043916.0906037558925342272671667138');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 0});
    T('20.3943935923576801013979387591948549133326048632573667789299', '415.93129');

    BigNumber.config({DECIMAL_PLACES : 69, ROUNDING_MODE : 6});
    T('1.486678588956157516725362331766399749551245061708835963596471115535207', '2.2102132268606715586559333071481289982224051499');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 4});
    T('5480561357126273281877150.2516417121681290163460862339136083441886517', '30036552789225778387170901122576813278108089781372.771583915452');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 3});
    T('186290.73', '34704239588.561202019698480025986856466562575839');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 3});
    T('1019119990.3697989', '1038605554771339166.6912148749760860679670771315340046767');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 0});
    T('5.51382659658824384028874579869515', '30.402283737243896279380556624294805637164470');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 0});
    T('132128402859245050100.944755960799740493288068513067396744451394670499181597586057807458829316094508487338595892', '17457914842134955530797594713735952482525.87431286629942757');

    BigNumber.config({DECIMAL_PLACES : 25, ROUNDING_MODE : 2});
    T('11194229425.7115257840132096032453966', '125310772435465796361.6');

    BigNumber.config({DECIMAL_PLACES : 65, ROUNDING_MODE : 1});
    T('2514912879934.47954895904718779302011160467785506374230597396840365067162301819', '6324786793660337947551689.6597215');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 5});
    T('23110048.3', '534074334023039.0646');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 4});
    T('2.8178317099209633246681281054844265828694464028893106', '7.94017554543610');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 0});
    T('83922723879121100060348.00424897481020168', '7043023583291202900493646259695479821033571894.5162');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 0});
    T('247358400344048909864251401578.275102984', '61186178220766775849900540163098993844125752619416049989983.33');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 6});
    T('81.917085580481926713094910295288886277402363899734148', '6710.40891');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 1});
    T('1881948645458414392373493267.44076491245651937895770407455779709686155608638313', '3541730704142760714361731732288699331881581040832341249.39126');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 4});
    T('324.26623374592211771183741217403875', '105148.590347765');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 4});
    T('1662149798843078.4069237887302461551370934948249024202080631', '2762741953794086012243855678967.8');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 3});
    T('675642835238849681736474.578123268849552096600873729347635703046100261327366969339596447149784597367597431761384272288381', '456493240809591377271607984202883887598233546223.73731084236993405421');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 5});
    T('5789496541496.6234418909569785217083510247010199083639377506967258726477304725684424188767358767048961015', '33518270204001364079261054.6956981043060432728471824679780846396880826203');

    BigNumber.config({DECIMAL_PLACES : 78, ROUNDING_MODE : 2});
    T('3898118405075.003146968384302560049987066028159380764464068397263645779570408017201600465061', '15195327099984486320266383.0779');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 1});
    T('75328512486.03568083840615998761042897028410187816426562961651629515834372852207961734642188953302492', '5674384793358833469158.7800150542809744853291679751200');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 4});
    T('5798169.528389841210757172244446608220276215346825819225777169316424526085117502743814039', '33618769879948.4736420937');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 1});
    T('29119162498.850656252014862720939576335974293913751260474542935386631954376862430745731464008616766564510391', '847925624634470395269.44447102349587477');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 6});
    T('168.23340288442161387499486814301418324124', '28302.477846072118591243169889731912227');

    BigNumber.config({DECIMAL_PLACES : 13, ROUNDING_MODE : 4});
    T('84.717176534632', '7177');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 5});
    T('446.6528853595373338635581695148843692555354568157075769833229135178150603561840120444549015301812', '199498.80');

    BigNumber.config({DECIMAL_PLACES : 27, ROUNDING_MODE : 1});
    T('13.385220437154074004622844494', '179.1641261512071');

    BigNumber.config({DECIMAL_PLACES : 18, ROUNDING_MODE : 5});
    T('25187.464910513177055609', '634408388.6183325662617834169211144795841400981156');

    BigNumber.config({DECIMAL_PLACES : 98, ROUNDING_MODE : 3});
    T('10422.15813601939556380841988779887735492607695215255973816250455181808482251151607984617041577195472527', '108621380.21219528176229683714575');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 5});
    T('2208553892153424409400475031042.6572327746873142396027723799591270120723216684906399117752180406894099292', '4877710294546039817041980082316047470007304183838484383878968.2209');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 5});
    T('14.4260892287582664239790267726919919828624026033250319230120150996965846303871869390740935534225811', '208.11205043609527416641080601664504139952660870888');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 1});
    T('50518253294056311797.1731845046869551098465', '2552093915882431397702073747003243930704.039747494091408300134971');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 1});
    T('23832299110227469789029554304.948356558865989463643235875638019404450568339468087397165994402723017298021132402151310599948', '567978480879349048201333612155955923171671328708933141291.1');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 4});
    T('97932545.8343664485428157368834804942294639641837112693533951', '9590783533600285.2329863436120504074140917839');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 4});
    T('662542878881.876031608894840844837559122854993420182484052151826850145', '438963066357084252820457.6516316337481795');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 1});
    T('2.408258174236454324395775462046657264396664516691587445008530664989455090566', '5.7997074337767003950634583826104408675441912175');

    BigNumber.config({DECIMAL_PLACES : 67, ROUNDING_MODE : 2});
    T('19840.4326250311386701679603718626113117527540565797995311111902694479563', '393642766.7484');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 0});
    T('15364571.38004578150166243985432371567701531656413692847732610913', '236070053692521.93070035438755932153027579886205373478180');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 5});
    T('557.884441439264', '311235.05');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 4});
    T('973.3437664448876512', '947398.08767712');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 4});
    T('253803.10166048', '64416014412.481521657597915615475256293');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 4});
    T('10979495135456507.342255564998728572391728604678115723240759397693', '120549313429513108511981910613966.523081580');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 4});
    T('275756032001872551178215198414.24547397537996419327607093729301388962050404448273494643547281384033137267141232586354706', '76041389185417758565246602361525083482592333735384806760948.73885982708');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 3});
    T('622715720.6863237085763126032498599017169901239797664342712392168097039974270476', '387774868789887525.14937758830956496718493909569236992782655630649990');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 0});
    T('168020.06571690822552882775092697246077349555511570099299844483404809694983876654837809362564', '28230742483.514158818734');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 3});
    T('11.7588160560738046080529419704288531653704396950661705922012083704', '138.269755040579104756364889');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 5});
    T('32912865546108258981522.757929715992045216906368802942491', '1083256718456200104722022580166775873143293905.1344841968439478203');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 3});
    T('4.166533331199931730602544326927', '17.36');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 4});
    T('13.1438229181471400406148696794413041749550438491914', '172.76008090361');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 1});
    T('140388.44002034207101031451179095355772785653', '19708914091.345183231428356292375830226197664009');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 6});
    T('242308.729399', '58713520342.9660909760390092854509');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 6});
    T('563619360203.007379904114453023253', '317666783195647379114879.695054964988341456801');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 5});
    T('59647218914478876.7858096522484430132157141971', '3557790724231766676348271355770172.83342006');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 2});
    T('1615983789784.28557364029229913482567909068840532382641403331829', '2611403608845582067513660.0062671164003418');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 3});
    T('49824048.11723328394671314846823689101965932', '2482435770788377.54686097356795579347173');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 6});
    T('3873399959291556633217.406801030839980478563896498128017913070289853996800649949943', '15003227244639832583385968352771422955089431.37902415349699763');

    BigNumber.config({DECIMAL_PLACES : 13, ROUNDING_MODE : 5});
    T('1069094409.0917779579569', '1142962855551297884.450653020529661');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 4});
    T('722662624195087368689595270662669.143097757212534864309051652910227081852406821943121855819128756801032825995791664965218054', '522241268408530075571015801012379480742910546734575022468778895822.7');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 2});
    T('30862804.01109', '952512671426425.6135331321216487473802');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 4});
    T('1102616829129019317396593.875343928415659074623558923629097274645985317064624881354801034751875382964051628994443076758824026', '1215763871878532982271799732726570299157805883536.58539');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 4});
    T('1.732050807568877293527', '3');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 2});
    T('168694642.8213633884274824334697145155532567219877', '28457882516627370.2');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 5});
    T('38.82141524469451899324885987106050839542190648554377069441850531749678620576749', '1507.102281601');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 0});
    T('3708639695.845872597235507175528689024964901895541187165232455597159738447', '13754008393603766407.7561988');

    BigNumber.config({DECIMAL_PLACES : 68, ROUNDING_MODE : 5});
    T('38154.94717636627489849459718350118176052576391544419824369739236599682079', '1455799994.031300773784046392217039165082991');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 4});
    T('53861660.242327701941421642702708321820603', '2901078444059944.6398231');

    BigNumber.config({DECIMAL_PLACES : 44, ROUNDING_MODE : 4});
    T('66519636.90005441495284260944015149981502627014844358', '4424862093315080.93581004496660187296078112048061051');

    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 4});
    T('637.68', '406636.4438334');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 2});
    T('41351116245595004128.850620728637405181715', '1709914814756711069822067981893933170323');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 1});
    T('22459732672013415497.3853699921391046229573', '504439591698306876553874829950080159407.2170874815252123');

    BigNumber.config({DECIMAL_PLACES : 71, ROUNDING_MODE : 6});
    T('11157237081923757329367254.81266156634018984033545412569365748968958548685350160685800062464993237', '124483939302254559619977058929566395439363258244559.727038073080997530');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 3});
    T('95493319361.43998295249', '9118974042665968313514');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 5});
    T('29.769596290034456365817227089424776835', '886.2288632716332683');

    BigNumber.config({DECIMAL_PLACES : 85, ROUNDING_MODE : 1});
    T('5397955557584682.9062196137312779972149743250300061250217778253358986584900136549913495459719480101042', '29137924201659364934764001707116.55318205283801807954313080327696327');

    BigNumber.config({DECIMAL_PLACES : 49, ROUNDING_MODE : 3});
    T('445909149943912057897653.3061127067661639263187209140228907079537075199638', '198834970003702246826932602696089951470569651509.383300823252946843936');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 6});
    T('138070606399.4147429423070318301311902897789', '19063492351502107366289.2467253642099');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 0});
    T('2.828427124746190097603377448419396157139343750753896146353359475981464956924214077700775068655284', '8');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 4});
    T('2064697698243932846.565435522719237034364504912661784788335223237983', '4262976585133794377599987093885187119.23');

    BigNumber.config({DECIMAL_PLACES : 29, ROUNDING_MODE : 4});
    T('1178718920.97866576388857757648973129508', '1389378294673110105.461225');

    BigNumber.config({DECIMAL_PLACES : 83, ROUNDING_MODE : 4});
    T('127622.9679188807821988228260439983645956713934866081244383838758589369960872848880087674', '16287621940.42367333133099782844035945474356067647612931678720139496987');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 5});
    T('4184557.3924000870910318746615706242234077254404466436266097920196360998523801490310121360756808685', '17510520570290.216452602485725796055410479930910699820524114');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 1});
    T('5052352182213.797005876498492011065356236496707214014862642148319', '25526262573120516662336097.11698538677217835690');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 2});
    T('42962929756.4524874753904254242420999449242014725', '1845813333257870594777.54199726872487979735452840824704377894604164');

    BigNumber.config({DECIMAL_PLACES : 15, ROUNDING_MODE : 2});
    T('168552520711834568.361558662238009', '28409952238313421147614420010576914.875572232');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 1});
    T('2657299207.955928329491651283753102211', '7061239080603204033.72779831');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 1});
    T('3457111251941.512647403949385333329309', '11951618208300612934484424.7889623678041889110925531673150994095');

    BigNumber.config({DECIMAL_PLACES : 56, ROUNDING_MODE : 5});
    T('16589542242815120252699.92265991917398936543675728311599787724806007026073243495', '275212911826147330293614584060085860179008690.253010639947');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 1});
    T('9804797931985.8334842704072729046935810090146497008995969314820034132254912985', '96134062487073676975741888.206515315310498601298549');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 4});
    T('225892.20559798985395295116237397587419516789890792906677567114604', '51027288549.924518711712074082612750594');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 4});
    T('212361356175642374.977190707414996501630173108910035758593734944529542910622890148761', '45097345596758041873283619282123892');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 5});
    T('29439920525604978702137963975872.69380730159987832221210796568237689', '866708920553937325445884613300050781747349007538519310874214296.8110223');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 4});
    T('1794661.69950729989940645822298294821075937317947091890782289675406', '3220810615678.43');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 0});
    T('70943035594.1874458255551741147170761086757974', '5032914299318146884584.65');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 0});
    T('99675095.36380397144700231541652768934274', '9935124635783415.963067816149336');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 0});
    T('338.977137872157980140325006163392742535324091920552022', '114905.5');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 4});
    T('40323.511502232159196214269860943981338317851981421978663218667848589388654136721795079462638130870006184', '1625985579.870649244041736269081');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 2});
    T('295.1603373872948769259', '87119.62476658174193358770628953642875013784920139');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 1});
    T('325.42058050006609869285755826829582236913188764341404269672712729886860377826386017746564', '105898.554213');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 5});
    T('1.4142135623730950488016887242097', '2');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 3});
    T('6983926090.66911511517713486643086264164426147359', '48775223639928789120.645609367588844026447525761410');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 3});
    T('2393.370169129531592840167459790940738107671625604470887826', '5728220.7664791226615102');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 1});
    T('5.493632677928148482954946', '30.180');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 6});
    T('285181133.07225', '81328278660373834.0857869874879');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 5});
    T('9542652739457662719.528', '91062221305878834929293220504352063866.753627');

    BigNumber.config({DECIMAL_PLACES : 78, ROUNDING_MODE : 6});
    T('12098280403299808.817491296272404617641688923237454622741160966458759689038562738845236283437224', '146368388716868184691692991958045.0');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 2});
    T('25105674989.27196694892019569379756456643245439196899838495631560703289', '630294916666955977775.754761194822998662848290817493307378901564640345');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 4});
    T('298760546437220458.455423225771015955168504223011027135', '89257864107466561972186848476830462.4');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 1});
    T('1.1', '1.393474');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 1});
    T('25453431235727183076231582.0045002428296642817772', '647877161671892234077761466199978409616134323779669.066906');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 6});
    T('1166849378.009584249870469632351477502709648', '1361537470961353636');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 0});
    T('120735117546849.465916751', '14576968609051557796737216230.279553123');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 0});
    T('180.802933832667757267699871900802757327723388473217684830187551089875783911937647806249', '32689.7008825000351504');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 6});
    T('1723788708991912690.6979822127245358278366754115883', '2971447513248005056078047539326528478.011258578775759932105013136721');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 6});
    T('41.31839758166512923717904870453658035', '1707.209978716550800090464759416961234');

    BigNumber.config({DECIMAL_PLACES : 27, ROUNDING_MODE : 5});
    T('12.2574125193918108917955657', '150.244161670543100821504558563');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 1});
    T('1971415909727091.042395951589950091275895878232674723637406777994288306294009519427713077175092202052917069713678', '3886480689125093978070206267175.340387');

    BigNumber.config({DECIMAL_PLACES : 100, ROUNDING_MODE : 2});
    T('29818.1986588727388202116378654602160649072880350624797884079437903955143243950832588490362445279127949134', '889124971.26');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 0});
    T('1104096573655.7434508163966770646809860896275509688219219768440479417337', '1219029243958352523057154.72');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 1});
    T('8.3212032754884674791676327316', '69.242423952');

    BigNumber.config({DECIMAL_PLACES : 14, ROUNDING_MODE : 3});
    T('2.83003939901903', '8.009123');

    BigNumber.config({DECIMAL_PLACES : 83, ROUNDING_MODE : 3});
    T('340.03323361794512720886720764159962236524054263630048376346443471719835370047017804718', '115622.59996467604822471094600529392511693650574987150');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 1});
    T('275469969089.0278368513030849885226902642614972231941872', '75883703869909951923056.9925972076612838068953091606028');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 3});
    T('1.7381024135533556067543983015209908994079067310222845767824051781162295524', '3.021');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 5});
    T('147188.397533864935817557107794698076331354474102470320276846588830452486', '21664424368.58705752');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 3});
    T('13205967.790168468376219309952073343238141391650648614253222880670277752337543824936186866', '174397585274967.06');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 3});
    T('476378245.2022326695078600546489393633852826695925662433781243683416850986996053', '226936232501958513.404536825694807');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 0});
    T('65441262.7320917904408200255581619450931183912735051626126918347697', '4282558867970665.8685319872056422875263001');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 2});
    T('16370078535249', '267979471250201715812551325.0');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 3});
    T('58598535224572228344606081143.86965847762937769186434110613067109010826343324179566657415332264633113223594', '3433788330465432215791468757905667403887370264585778014417.069465');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 2});
    T('318363725.1170424420783194862296940836221974113987521316095299779738219371', '101355461470399760.7');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 1});
    T('14365771163.04683583', '206375381109028038378.8968049886952690030430276282428851486182857624');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 1});
    T('2841081846208924.95006940291133134386661306911437159092547083699830973529759836', '8071746056857913481680923737647.56');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 2});
    T('131.3328926430846569131', '17248.32869');

    BigNumber.config({DECIMAL_PLACES : 26, ROUNDING_MODE : 0});
    T('2450172181.50221018709980940566629037', '6003343719007299620.145731224358202401202766492638826695870673033440');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 1});
    T('175.721010954473741615074590589138', '30877.87369086228070118');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 3});
    T('35222503818695.83554274', '1240624775258042717246577010.284467326536941313249177');

    BigNumber.config({DECIMAL_PLACES : 57, ROUNDING_MODE : 4});
    T('275332.808008853169433992082394650601012733904717450517250251808', '75808155166.04');

    BigNumber.config({DECIMAL_PLACES : 68, ROUNDING_MODE : 4});
    T('650535575.02591020903090816952476015222892057257466848913366015211767330519338', '423196534374291650.4612554166');

    BigNumber.config({DECIMAL_PLACES : 47, ROUNDING_MODE : 1});
    T('406927200.63717875393669232657792854954820044504469731078', '165589746618410732.8838911359');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 5});
    T('3.5', '12.2');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 1});
    T('37755263902.770333976947348906520419869045275', '1425459952367832590789.24209908593868009183303760839070054881073325');

    BigNumber.config({DECIMAL_PLACES : 30, ROUNDING_MODE : 4});
    T('5', '25');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 5});
    T('207981785245332.655650133225843609703074411112915893328590968293802716495668521411178574935737507656194691295', '43256422993835672341998199525.8710');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 1});
    T('5567.151876678055', '30993180.018');

    BigNumber.config({DECIMAL_PLACES : 44, ROUNDING_MODE : 4});
    T('230454239.20351432886371300553461777852426637742579329', '53109156366870600.60918351286261378516983689388197643');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 5});
    T('1', '1');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 1});
    T('93700655057314214506084.06304464429', '8779812758169783883346075569138347125163328358.876');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 5});
    T('9.852661929651296017820353932148571', '97.0749471');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 0});
    T('567906032003800411043011528704.9630913901', '322517261186301576710825760318275219851051098240516062527531');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 6});
    T('1.14593577918', '1.31316881');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 5});
    T('136701658884103.983801784031791378405', '18687343541665925641857895181.7974776695004112848393');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 0});
    T('9', '77.476');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 0});
    T('5939114.486012363442761208219053271364071174533711946796290607324372252717149892498', '35273080877961.90');

    BigNumber.config({DECIMAL_PLACES : 3, ROUNDING_MODE : 4});
    T('758230332146782649.038', '574913236587420337445176452427091337.30');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 5});
    T('9666198.978421', '93435402690422.29441946383951996502994575376130537');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 0});
    T('2.590278911712160295850740530343268793109490349355580782704975797477636584901', '6.709544840460733512694560285948463771031066891686610162271281426162621');

    BigNumber.config({DECIMAL_PLACES : 57, ROUNDING_MODE : 2});
    T('5.518061253737584236314019174438934514502897225542757968064', '30.449');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 6});
    T('2649559.3276549431175517495047876221738510158494831871465587320520938465946988', '7020164630763.31422095634109838375574');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 3});
    T('567.9550364780740745791662861971589109151899', '322572.92346081045270510896663891363296');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 2});
    T('3150748169414198847840463.18235887773465431759166732173861612818759700849401971501876448272195126779179581325159644604856019', '9927214027066925084042058637410661418578292156497');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 2});
    T('6230.779757580163866697716030430928315350835358299363369491', '38822616.38747072560206254107330849348366896430098893811980691');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 2});
    T('2.716247411411559909781469174249066399553542751307', '7.378');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 5});
    T('1003856.804189946449097794004928186526019323205855400434577744585139331157822001974710614149317739', '1007728483318.452486681');

    BigNumber.config({DECIMAL_PLACES : 72, ROUNDING_MODE : 5});
    T('58545618495785373747.978660683381606360982662056343866125407934242953992920901245175791334319', '3427589445054046448675910703464645073846.568134490845289998323');

    BigNumber.config({DECIMAL_PLACES : 39, ROUNDING_MODE : 4});
    T('152936840373282.069033230237997018443718074008255872817', '23389677143362760272547678245.15312539480334913939812559754408447298');

    BigNumber.config({DECIMAL_PLACES : 85, ROUNDING_MODE : 3});
    T('5925.3196723393751424830150929842214599276197242531387756086505181454007563045132273149778', '35109413.219412');

    BigNumber.config({DECIMAL_PLACES : 80, ROUNDING_MODE : 3});
    T('38561228.56697678575069939167914746425178007661373044851182281245001919601239245478067382', '1486968348594626.53354241638419629718421662019251450932947890');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 5});
    T('150.4509587455', '22635.4909874539178687243');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 5});
    T('1.42777813310472190178856389527377945214335953146126319604328603730379698595792254248379', '2.03855039737200497185073691500462232203792065665077273244');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 5});
    T('516549215.787659938423166339922918873941357009613190991233143989', '266823092330846471.379278');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 0});
    T('2', '1.08767122699');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 0});
    T('6324067120009086551.5468269550550503238490888524340636718633012263225092', '39993824938380022323743029539005736229.3283338711988086216840448390248');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 2});
    T('144118.390048216955510241008109108904440825', '20770110350.090');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 6});
    T('18.78297101099823345', '352.8');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 4});
    T('625521102275061592166085.7029768359919241933', '391276649391408064574867871389972320049448749231.515233113926295781');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 1});
    T('15882418.01726333951134210284351839021652515057337109745443', '252251202075091.1486881249278033607224037158843540912');

    BigNumber.config({DECIMAL_PLACES : 98, ROUNDING_MODE : 1});
    T('9246102.56100069809273634059139333855094718263186671420242936944261575893351481020001402555139465129260003', '85490412568543.6679950745889670338728');

    BigNumber.config({DECIMAL_PLACES : 15, ROUNDING_MODE : 1});
    T('60961608.064554172663083', '3716317657816316.3414696396181849068766696759170');

    BigNumber.config({DECIMAL_PLACES : 78, ROUNDING_MODE : 6});
    T('5246979273996.215099692426428798175290449427111226766305846934068259604146954605057636536678', '27530791501745848489064024.8188180632265280334699492709026');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 0});
    T('549012.663441058051793486405383817407015996732587541151595200978084268869699408419983990553', '301414904618.6444803');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 4});
    T('143455.7859397685383562199471977165574195449037114065988291931607472028', '20579562519.59669265955319');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 5});
    T('7323.414075814701603113726796738850491925289154116673', '53632393.7258409');

    BigNumber.config({DECIMAL_PLACES : 45, ROUNDING_MODE : 0});
    T('314146057801896.072108842090558464557933909765109713994989546', '98687745632472227996598145632.74641148442027');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 3});
    T('1980077181169544258109.45175016109951535899380651346232425215206090353014542452514108138787115693996106945620698162', '3920705643388328194332944477286896844012715');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 3});
    T('6011.1', '36133937.82872246239');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 2});
    T('31921798365272.20937299799446076643740006676603681348599', '1019001210873095498660884207.3946941973644977');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 4});
    T('4.8579831205964476879084184791886', '23.6');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 4});
    T('532180335383422496024802938009316.437231456810652157973165941886814685643110949100858986799826492779541359875188007141128020804568', '283215909368812049909044402175454534173632261029912991407795749397.57');

    BigNumber.config({DECIMAL_PLACES : 88, ROUNDING_MODE : 5});
    T('4117620393205561878616466439.3589035337592787404765061242925453323082755577389136298123562314912920363974305057593966', '16954797702542326015871361358548968955831149713211464473.1194145872');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 6});
    T('125845316.009946996954481751436769926191513940640832361751949762234621432238139835729629824687966125681673', '15837043561643421.95007453');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 2});
    T('43215736562696288.1278295788274110509465500800588178280120433', '1867599886656364788451547336030574.1042929');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 4});
    T('15621491479513403048.699533910172088405978421413948149996097276644744344976', '244030996044509850142368364708463636903.065970027514531227023');

    BigNumber.config({DECIMAL_PLACES : 100, ROUNDING_MODE : 3});
    T('178337437922.3409184119906563307617679811185757303031127956447827491012301101339529853906621960842816952844554116', '31804241764704800710353.817213026160423493644719533');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 1});
    T('887156.5304358788486852760691897281127658626051846106550248530081803993620243050311642', '787046709495.026433990039530');

    BigNumber.config({DECIMAL_PLACES : 63, ROUNDING_MODE : 1});
    T('507.236726280487857380940671340356658005564340464804099830756705102', '257289.096487746561');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 5});
    T('706928215661578.645391690986420023561721605027645326', '499747502098463447175418473761.2978027539601447630695939');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 6});
    T('517763510430698606317.0809006022190406', '268079052733520144503181249583007928315915.4142727240345924');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 4});
    T('28120380999784456871711599714211.3366834435766268345186681299717226185752325559124860129892133352608046765821533895', '790755827573038690221243071006380153898932673099784811839693348.2');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 6});
    T('15996.31328328727145297157844857483958854490192496672111354158713663533836276729421', '255882038.657072806407074465204520879667922106288826705864274522358957126');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 2});
    T('4529.2020575270077036355835603989646565391897334057016463815795973838051013483', '20513671.27790688');

    BigNumber.config({DECIMAL_PLACES : 3, ROUNDING_MODE : 1});
    T('11.957', '142.978068453');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 5});
    T('1685976945242985678.95815067126816165675418912553152194874360838240711904223001193', '2842518259890869530436269343367542397.5276842644433823844843054994');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 4});
    T('375790983.100062192270906028735048485455706166096392111303292385474398132353985937006549336350115', '141218862979311228.19925');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 6});
    T('1.3582121735950112188773107283180479288740509274225903562199127996284062', '1.8447403085016848904555');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 1});
    T('373517970953920.4727120160117475904085285980006296740379565850799987331831582', '139515674625533777927629498890.02832807275');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 0});
    T('130670434621810838954485430.74993708051', '17074762484252940770822154406823463583074092132494098.1620220080582134');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 6});
    T('410862.91578166713390158', '168808335564.6133');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 5});
    T('1.056882207249227940185147536423171386481708667802783', '1.117');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 3});
    T('152.6779992371273697633357536044552314267940328982004863405769613320275457342815692626846477563670374', '23310.5714510522657034278023418482838530446');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 5});
    T('5045832252789636.3813168206783005421277846727380224180447331191378125099729439', '25460423123292136945025493632037.90746370');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 0});
    T('241.0932155085819555358790117930184281430544398516041254679339298261706270710481942', '58125.938564267542760919843105102271');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 0});
    T('817025.371828418518631071224096112860591604381801846574731525820101525218802811325', '667530458211.36553674192');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 5});
    T('28278311609946.326736102526929532491', '799662907509225213536765867.66273');

    BigNumber.config({DECIMAL_PLACES : 56, ROUNDING_MODE : 6});
    T('1.60044993673654159696798740599151325319220173155080452821', '2.56144');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 2});
    T('57250.10343607', '3277574343.439725271285333261007');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 6});
    T('110855888.6785004', '12289028054700075.1523748772313326451842826700743417901399875438785852');

    BigNumber.config({DECIMAL_PLACES : 83, ROUNDING_MODE : 6});
    T('317340360.91013300290215375746814647227032587585124984322407432345305135787860411883975304799', '100704904662573470.40604311931228040');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 2});
    T('30794104978259578447.5324907738864954704313972589515591946319482', '948276901412071352410750952884177783211.51');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 6});
    T('904743.1002686033189575318435946062559210433293', '818560077483.643999');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 2});
    T('814.22285241569076215709135504419403461514193802087829114916346', '662958.85339594374');

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 0});
    T('143152984767.6392799439406910636477216233068934157431', '20492777047883963708442.98940241835545510');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 0});
    T('194263.56051042638141037702010101367144786906155207240685722988634', '37738330942.18809220193041257372062644812542074623487171376384576');

    BigNumber.config({DECIMAL_PLACES : 49, ROUNDING_MODE : 6});
    T('125613206310637238.9999516821420060389669668726285618243029863493137', '15778677599638715084039823531413754');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 2});
    T('2.81203323799266774393606177414945136596491179572335714244696589280445996807268708486433927395365', '7.907530931575527548477411073438');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 5});
    T('97111976.5122288174252479213412431470964858043', '9430735982111681.5109964772635493531635006539282859130');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 6});
    T('3949738515.41597806049335653753381952486682043359067751877565231', '15600434340160414359.0242581826605547758376333581');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 1});
    T('3506245338782.0109313646218819195242298503032566600274291448066584621385570357489121387', '12293756375730578609336028');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 4});
    T('131730705.496855457', '17352978770699264.525247991247407117213299374535');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 0});
    T('11.2973449', '127.63');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 5});
    T('848.3895866356', '719764.8907118');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 5});
    T('5534.674100613332409185919669221553225849120645345046040040296971736869045359760082626174208915379', '30632617.4');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 2});
    T('10928938620655456801.29603452229846959727304567882485936765429318591408', '119441699374054398699281419105567220380.0967437459074243');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 5});
    T('8725513451707327112558.6421593666547417330120647038', '76134584995925513871275158019859409214493256');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 0});
    T('15138902098810276416.34547477556955649142152996550437235286286', '229186356757362192283401407016237741007.875');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 6});
    T('77016750197767918492761921.18729300767631826805523375696015268816409374420696804911556102049294908310575896539550300168736', '5931579811025384712266992972162620186506533282884778');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 4});
    T('204726156.573360406099498017635273106', '41912799185300080.215440520642760629606577');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 3});
    T('18354.9618575334846703385169041300559086', '336904624.79150907');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 1});
    T('5.2702941094401932817072533773788105939199657130326470941562465397903460264359252040580345287167', '27.776');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 2});
    T('1506899789309989.325166093578631118018664419471205880907150775938926778426237139996792273511931625682', '2270746975022490218466170988903.9034');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 1});
    T('215.141797893854182599917347182504049580948821046088961606891404751873995447073405879627656862967', '46285.993201');

    BigNumber.config({DECIMAL_PLACES : 59, ROUNDING_MODE : 4});
    T('5619123349635224222077.73689475358389819964179507664344923519119631866482798769645', '31574547218415782317658255169091503768595218');

    BigNumber.config({DECIMAL_PLACES : 74, ROUNDING_MODE : 6});
    T('3139311096862303881033372.8895562022105500271168912428067595756172195924006540181335036545047807615', '9855274162882801500447430900223953535768965599628.60066266');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 0});
    T('5188166.045323001', '26917066913842.5');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 5});
    T('48.006249593151931618014834046061943060009707858084284198526528', '2304.6');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 2});
    T('1.099164138733610613605106295601295', '1.208161803878');

    BigNumber.config({DECIMAL_PLACES : 49, ROUNDING_MODE : 1});
    T('1215373388900855295845.5639091709273307937490329590109938808688535471517', '1477132474448349650830670930172186917647227.199322184697');

    BigNumber.config({DECIMAL_PLACES : 11, ROUNDING_MODE : 3});
    T('41886.32804080003', '1754464476.741510996586572659655625634656330393988765738834499008646065');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 6});
    T('2930.8115486762881469691621854256971849413713695023369381527515359143547703303370877901349455103678', '8589656.33385430252628284912359208432320350710697');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 1});
    T('196113185633565235.48902239272865560851203376619671611896859555577833436790431753', '38460381579345217875459335568935383.8847449697048154978487350092029369484');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 1});
    T('1.3090836489697669430592230864645135220438796259107745885986886542431248034364317796928116096', '1.7137');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 4});
    T('6494.1269986060183145522727735872976974', '42173685.4740236118');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 6});
    T('1.73205080756887729352744634150587236694280525381038062805580697945193301690880003708114619', '3');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 1});
    T('510173005.753621652801505640054349100211743000777249630455648526090128271276356538524042100252', '260276495799684872.0535671973932244012807052944');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 4});
    T('26680.140555343682781267465538767291165017590933503085728684041740432806578026822495320310875029221617', '711829900.0528947178462167211');

    BigNumber.config({DECIMAL_PLACES : 45, ROUNDING_MODE : 6});
    T('26863311087875171271392.514957801132618076235905273081446630753668319', '721637482603957117805410790743695797070814094.4');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 4});
    T('42.095644254289553432987753554042034809688485652611694000319150583822', '1772.04326518370109313123764508971031570992396357099');

    BigNumber.config({DECIMAL_PLACES : 45, ROUNDING_MODE : 1});
    T('29.162455980379039696650800234586962597340058414', '850.44883880754521733664670163746907111624452011747340966506055962305864');

    BigNumber.config({DECIMAL_PLACES : 20, ROUNDING_MODE : 2});
    T('174294956414350158.07876863178146700685', '30378731831480221313550099920962541.2005954592883112926466593');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 5});
    T('7.549834435270749697236684806946117058222194704623380138298626905710721953917811195582452074148', '57');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 0});
    T('47296675573534188016.13909410443959003286535401918757168674179806460628276222655770436524602843813149348446075', '2236975520308145512917915800636179158586.5092232275216783');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 6});
    T('1447945.6262667282803484290772440854', '2096546536624.94796979316325227120931989581');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 0});
    T('4.3292008813174583991369', '18.741980270799858523549010');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 1});
    T('2849247060619898470261078632206729.3993355880326702595070558832203445897791540572152075', '8118208812451131388164025563919715495922430919813078457425427917914');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 3});
    T('3.4829585125292549197', '12.1310');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 5});
    T('32.428290118351907948165411821421678725137118993804751360645501', '1051.594');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 3});
    T('7.2070479393438198497379099999716403294014366391234593782940686026097873630434300194405', '51.94154');

    BigNumber.config({DECIMAL_PLACES : 15, ROUNDING_MODE : 6});
    T('3.605551275463989', '13');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 2});
    T('603824436782267945465764574485423.80547609420748446504714711955721919730100747232448916748085772070372661156640640062', '364603950455423098555431241498110457415113057007663996510897977425.745632');

    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 4});
    T('4.3574355751624058321', '18.987244791690926526060867101645214646023908888');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 0});
    T('504996390701500.435239100967145881400957693439105021537528696911649198853064966592673953962003822544', '255021354621542475250452411149.0055116842377539');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 1});
    T('2909.86260112701154717384254378546693847874984', '8467300.357437657503048755260363563653074');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 5});
    T('9.3053479247151205055515881569599031024893777073506787564241255718300181642', '86.5895');

    BigNumber.config({DECIMAL_PLACES : 44, ROUNDING_MODE : 1});
    T('15184837.76261575421430103273013272422421584951851796', '230579297876961.4243352370777246641742839628933607349');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 0});
    T('50659385450.8214538694263336502871503348032787319532637210737215146773327181', '2566373334254900398902');

    BigNumber.config({DECIMAL_PLACES : 77, ROUNDING_MODE : 0});
    T('226.60228379499643935590935027196664811910610985785113122241953538017762352877784', '51348.59502110810590185955521392268');

    BigNumber.config({DECIMAL_PLACES : 22, ROUNDING_MODE : 0});
    T('9175842.8911770299558613410655', '84196092763564.036005');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 2});
    T('62524.6899557388029462095111297731720065761609804421739715387797639468988712711156019109649322575312', '3909336854.0612647523247262815157065521719556832865767218863239798');

    BigNumber.config({DECIMAL_PLACES : 80, ROUNDING_MODE : 5});
    T('652.31309350134792806991152377498486054402221136563232349988445782260141722659661051', '425512.3719532982845082007576');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 5});
    T('186832499748734.4829944850442559429780865393108169056074459010304450005', '34906382962360870651268614097.94565831926234089039');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 2});
    T('3693959583316804.51199229840601385770877601259801966369268344952147731012934598215128326668350642370497748936308', '13645337403178060015123543336999.7073728144053018493');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 5});
    T('36.94471545431091826803359482480916267400888', '1364.912');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 1});
    T('43406.38666118846617211', '1884114402.9806');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 3});
    T('31987334765.001851900823021837506682024607261426616460152499', '1023189585368296079968.1551605');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 4});
    T('18838.0008560899974069055717427261526855765440186889008088955787710407', '354870276.254047475192657981134539079181359836556514113293');

    BigNumber.config({DECIMAL_PLACES : 50, ROUNDING_MODE : 5});
    T('84947581.82669065818681578404113066561367677316674137015818', '7216091658202304.9799531382271756');

    BigNumber.config({DECIMAL_PLACES : 46, ROUNDING_MODE : 1});
    T('1101.6975972788540124782719480347417662099926636854', '1213737.59585');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 4});
    T('167205274115.103780530647', '27957603691906994337374.3099966843916195600451');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 5});
    T('113.530603646298245333104493769623840715435780348102499661940881285297460005634412373532130963', '12889.19796429286843872');

    BigNumber.config({DECIMAL_PLACES : 88, ROUNDING_MODE : 0});
    T('82445912016.2463318822900430706509122231808029943890719024634238434031416371678942728723333531287386', '6797328408190631297644.10179858770371764');

    BigNumber.config({DECIMAL_PLACES : 56, ROUNDING_MODE : 6});
    T('207651.3290995157162487299892765302290963929658888771167106618', '43119074476.795382480774731451352243170');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 0});
    T('8552153877171979936586493.5446127951', '73139335938827728891918366925379870129969607150489.41');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 2});
    T('127053.542191315698254482051137112861614287006654322252567463225598634600921108288525772345080371829053454', '16142602583.360438243472020343390659992');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 5});
    T('225.1', '50649.6');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 3});
    T('11596007050.2163502967919818632722642459840343', '134467379508667301633.785635590876603169506');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 1});
    T('69437193.18414365787090023169347820662309913940132765325216924168874836547917544180140358', '4821523797292086.456524223677764683');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 3});
    T('5233576575683568887.6', '27390323773543750860832487238432062688.6624185028799484121035355694');

    BigNumber.config({DECIMAL_PLACES : 10, ROUNDING_MODE : 1});
    T('46384.4029667643', '2151512838.58318');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 1});
    T('195576.992166358873062953962788281277982207581311538326506522826301', '38250359864.84');

    BigNumber.config({DECIMAL_PLACES : 68, ROUNDING_MODE : 4});
    T('139866932291876214.19345627298588358215061823765973088810201696524681681453253341518481', '19562758748740285291218891169284396.9939195499727764062935729066980524');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 5});
    T('3666563896676760573917601692322.77212311488832397132809756567872530081505667435009858623027596887637272536145349022150950454966', '13443690808413470589583040162367634914502883969219377974038547.0');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 2});
    T('6961365655.56399610188128596984179', '48460611790465945211.698621678317269105061971842437574643735599856953314');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 6});
    T('1846705955.64084', '3410322886599335678.3442689068158599617426109752302');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 3});
    T('20383600747812483.68227625101786038548655392179182361804236021798711869594207098', '415491179446221643995603131550002.1039032281795321692412');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 0});
    T('646712979.5996517630443612769041002926810614019248010215062306018646711436498243', '418237677982659597.441624218');

    BigNumber.config({DECIMAL_PLACES : 41, ROUNDING_MODE : 4});
    T('1.19496644978306365000998087438061877340396', '1.42794481610713918');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 5});
    T('23.211007210632864978030233482478479', '538.750855732050851236433');

    BigNumber.config({DECIMAL_PLACES : 42, ROUNDING_MODE : 2});
    T('13144319811979.322618006430364389461341957559744602233547', '172773143319592135100403688.71821166');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 3});
    T('187740161098222151.1246717298374850292596863378927231390131419626237794062025199113420744', '35246368089186405941471994235233738.419206808');

    BigNumber.config({DECIMAL_PLACES : 91, ROUNDING_MODE : 5});
    T('10178.8922729849318155784011624865183762792234558462424939879226978772185670426454331737205675199', '103609847.9050323516770459242872382420478556123172737547274829818796745411');

    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 1});
    T('118915484534858138581.323615778120248', '14140892462160085127547570668448789567436.5212757788');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 3});
    T('1381256041.58235', '1907868252407743284.86488650760');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 0});
    T('332932692.06871668', '110844177448122917.8860635946844763529034613752768046936282087236524');

    BigNumber.config({DECIMAL_PLACES : 25, ROUNDING_MODE : 5});
    T('4066.8947213589552554383213699', '16539632.674617334307963480316101638407048800069358878699410140');

    BigNumber.config({DECIMAL_PLACES : 65, ROUNDING_MODE : 4});
    T('22189652304.16065994787886588015884847951186574095167035995581747786105688131', '492380669379542485181.2474040');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 2});
    T('158353311059733655489.651998750507515328606317455210195317', '25075771123580765253407550077977888331941.700640558633406504581615363');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 2});
    T('1143117392445205.276951994811536971745352', '1306717372910725454386211774949.25170879486190843648039398');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 4});
    T('8040.191011551416394620773267580733826', '64644671.5022321884');

    BigNumber.config({DECIMAL_PLACES : 71, ROUNDING_MODE : 0});
    T('561890.07913644314118462700408674766996898826917350839637746190609856169025443', '315720461031.95833577709317260');

    BigNumber.config({DECIMAL_PLACES : 64, ROUNDING_MODE : 1});
    T('4.8039462944541750986312884225369797018444226801916417173216151656', '23.0779');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 6});
    T('9', '80');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 2});
    T('1326108751496903140357139441.556612', '1758564420796675206900880488447739278881655552577114313.23383192739240437');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 5});
    T('14151144510912448.473766149283465377648372312085324275310001302203928077203283587712858416110305268972089', '200254890968727520521220007130381.2005848441498962923460336400353828714');

    BigNumber.config({DECIMAL_PLACES : 9, ROUNDING_MODE : 6});
    T('1563587460831009.71743044', '2444805747667964347315337272292.5');

    BigNumber.config({DECIMAL_PLACES : 72, ROUNDING_MODE : 5});
    T('52.135400641023178004669430567271849421048932266185106328569016760444478187', '2718.1');

    BigNumber.config({DECIMAL_PLACES : 13, ROUNDING_MODE : 6});
    T('1351268503014631893144505099260337.724034725277', '1825926567239404241691076828603758142583205695412586217329125535956.66');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 5});
    T('11695015362959112818180.313719', '136773384339849669329939661060606953014336914.9830023');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 0});
    T('7.14759622880307922056248', '51.08813185');

    BigNumber.config({DECIMAL_PLACES : 33, ROUNDING_MODE : 6});
    T('77955097085870883368715827.917274179795960111351375993042359', '6076997161667555092341666871976429897795152678135287.8510389381');

    BigNumber.config({DECIMAL_PLACES : 62, ROUNDING_MODE : 3});
    T('302769879390478259.67043697315910757236727400949100809491703248976382488199775657', '91669599866124752017570835756271700.413528435509');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 2});
    T('2282.2870108730847351310392476188091352', '5208834.0');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 4});
    T('17078631911484768.4490028733180418062149995467371729', '291679667967985875726407897209547.1387');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 1});
    T('10', '111.670519788005621906773268955014426');

    BigNumber.config({DECIMAL_PLACES : 69, ROUNDING_MODE : 6});
    T('30174323609673224480.488459745551217192738545389494419174874058939462629781457993432294265', '910489805301282971552972612705593407831.71268096');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 5});
    T('14.624254852812159605249622627938622344', '213.868830');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 3});
    T('29466625146755603.86185507', '868281997539409712828877902435825');

    BigNumber.config({DECIMAL_PLACES : 8, ROUNDING_MODE : 4});
    T('6927129100625732.43372701', '47985117576735868701359921592072.70372575612465393845774834727674');

    BigNumber.config({DECIMAL_PLACES : 94, ROUNDING_MODE : 5});
    T('9642346281.31563248643896664520776898744585192834730152101992432409131888949068606645072468415514441421', '92974841808801406424.6547297963');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 2});
    T('3088984.5336327982458333177572065032974592687507455458620037359367363260748579302628662882', '9541825449022.6360771777341161');

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 5});
    T('255.021597053934205856842576268995125858655341587771945346677594354254384952531873852274081014203538', '65036.01496393918362398639291683514922');

    BigNumber.config({DECIMAL_PLACES : 28, ROUNDING_MODE : 6});
    T('1.6646941094584521805350551637', '2.7712064780656691696');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 5});
    T('19884686303023658767.08797961545126816028858276270712259956194324382612814640205061878988423416501222805073703', '395400749369656702132720990647874201569.1939430970209654');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 2});
    T('9.6166664915127248990027449407750613964822315991471685061583', '92.480274408983661791945');

    BigNumber.config({DECIMAL_PLACES : 88, ROUNDING_MODE : 6});
    T('36686.0099950360970023090500579037458831558057639869287390847137799628072928071825825683327205', '1345863329.35588841');

    BigNumber.config({DECIMAL_PLACES : 14, ROUNDING_MODE : 5});
    T('82540126.57450483338603', '6812872494935279.00063912455569141793822346070304115108');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 4});
    T('5968822849220.00895', '35626846205370865688433017.83222');

    BigNumber.config({DECIMAL_PLACES : 98, ROUNDING_MODE : 4});
    T('4816036.61099593028962176834662170900210233124809910321599137332324694451063302849228481281532867578835396', '23194208638453.16557264455596232474012945340603152059440');

    BigNumber.config({DECIMAL_PLACES : 35, ROUNDING_MODE : 6});
    T('462306278297631226.13050497526669222098203681114573462', '213727094953406852826524541520000863.7');

    BigNumber.config({DECIMAL_PLACES : 76, ROUNDING_MODE : 6});
    T('1152247133339717980785389844.0669369155655882911665420542531405497051805584818042141420118135205037703317', '1327673456289597827892437378695792745399331158560266522');

    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 2});
    T('8440.29450824835258603622655681205756', '71238571.3859673');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 0});
    T('372.901628755987177132359193081806236478345155487248214227646402521533643140930822086713180088', '139055.62472886808277107884115049680');

    BigNumber.config({DECIMAL_PLACES : 76, ROUNDING_MODE : 4});
    T('33636.1271745309846040761853747749832841960229355764342958756202876527205256997547', '1131389051.3012217576165671647512263221');

    BigNumber.config({DECIMAL_PLACES : 67, ROUNDING_MODE : 5});
    T('730642441472613374.3703419923164107610452588300422615171223528468371378585782084037021', '533838377281061260221750424175452238.799786368085667707972');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 0});
    T('4.5788382806122340708247474338099311669807185394701277113796016932695447461', '20.96576');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 2});
    T('1864801744998620.013461802873958666465697174610674741137094975087863374090471208443', '3477485548149898222390988884771.11817149815840898839369209152890');

    BigNumber.config({DECIMAL_PLACES : 100, ROUNDING_MODE : 1});
    T('563874215863580.2177451764691813196377888522148073482708829811406591604089329763209930727281872382057875471855029327', '317954131315767460486951695672.98301031763037726866477256860609405919575');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 3});
    T('421945583.1988849379362665405151159104233', '178038075181047131.2144288231');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 6});
    T('107729651834007850.9577', '11605677884276551125438030363905372.108250181197');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 5});
    T('40416496508882.880295518466489909876887230188810128', '1633493190052542050826387695.49412988761528497243381704242064847014101');

    BigNumber.config({DECIMAL_PLACES : 20, ROUNDING_MODE : 1});
    T('152.2434280471558132361', '23178.0613835495093166083287924848173635972');

    BigNumber.config({DECIMAL_PLACES : 43, ROUNDING_MODE : 3});
    T('2.6360766301456412706748188538578683408291152', '6.9489');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 0});
    T('5585761094110.8119307548005902880792769404176694786028672201071469437384', '31200727000482014779134653.665916381675162519321008239592797688');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 6});
    T('705554.1228818517008122812577645421096429956370245502968178545245876570204000353369717718', '497806620315.5790897792940015037475985377832001974452467227');

    BigNumber.config({DECIMAL_PLACES : 6, ROUNDING_MODE : 3});
    T('12261723754.180635', '150349869423837649402.99929324676906802471173415');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 5});
    T('281477675403235918.2357151530419555292357090950366728', '79229681750409442645557612236437995.9476523401612199126960867');

    BigNumber.config({DECIMAL_PLACES : 70, ROUNDING_MODE : 6});
    T('9952285418.2729610572401721588784084577508583224943467082894527238719302981248716', '99047985046768607423.38096514');

    BigNumber.config({DECIMAL_PLACES : 3, ROUNDING_MODE : 6});
    T('13999.647', '195990129.039656321401845946648205546325683328673485939369440496168');

    BigNumber.config({DECIMAL_PLACES : 84, ROUNDING_MODE : 1});
    T('14047571.047709437493635361374607055104972689228218006265314138529512357717625916281663520003', '197334252340444.4234');

    BigNumber.config({DECIMAL_PLACES : 25, ROUNDING_MODE : 5});
    T('2205837.7005821761318559775242478', '4865719961309.6621197151014351797289406864523486396');

    BigNumber.config({DECIMAL_PLACES : 21, ROUNDING_MODE : 5});
    T('758.30644993960060605864', '575028.67202');

    BigNumber.config({DECIMAL_PLACES : 71, ROUNDING_MODE : 0});
    T('3.05122926047847148656588738870173096139590274766271145582324089671169013', '9.31');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 4});
    T('9987.0540005269950793629951267710033859502301660583410487260054767889497417917580627', '99741247.6094422566309427596792705224314859453123429533');

    BigNumber.config({DECIMAL_PLACES : 55, ROUNDING_MODE : 1});
    T('153736.2249720002675176784618619275585698111135968831038733952', '23634826868.6414786551');

    BigNumber.config({DECIMAL_PLACES : 99, ROUNDING_MODE : 2});
    T('124306362204054.700418459258259880707339315715942679895841027323391732672342072298858769481551839030444032097998078', '15452071684405638957675416841.87559955466718280375315');

    BigNumber.config({DECIMAL_PLACES : 89, ROUNDING_MODE : 4});
    T('8354004408580181649765187.31783156135039411717464270711151376270906610358555385926351601775037854447124260858388496', '69789389658577110583494784782945375526722091196870.2723607941331503389');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 4});
    T('69612192477.44215090986896764071456', '4845857341516453584012.55400593651071611869');

    BigNumber.config({DECIMAL_PLACES : 76, ROUNDING_MODE : 1});
    T('26894442745048233575.436901864421884016296340079004943902924557895292772056665868939809658762881', '723311050566677565290951117583555651705.50647920351246054353644335');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 6});
    T('718470177.271028322610907514636281025794525522162254026399567933688109176809379897895939350045484', '516199395627862862.91');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 1});
    T('624214653401002320394878594.342626363311637934460517', '389643933520533457715770790451497422021845205674084966.57462249521');

    BigNumber.config({DECIMAL_PLACES : 83, ROUNDING_MODE : 5});
    T('14961.11077476503262776978477227891095270825535559491495043352456073683918164489113356909', '223834835.614790354856161392017958644437815');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 0});
    T('7', '43.66395760675375468584137073521451');

    BigNumber.config({DECIMAL_PLACES : 23, ROUNDING_MODE : 1});
    T('1345.51506346178747090783855', '1810410.786002577965235589781993065428275582786284340392554');

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 4});
    T('52557016.286119501062009568005557119407270734435843448903725', '2762239960899430.472320476603615601541');

    BigNumber.config({DECIMAL_PLACES : 75, ROUNDING_MODE : 6});
    T('304128108642.66638543798285540424121063889329767580106958078233889375947456592753506621', '92493906466565488169925.03727298085107554398631873900964');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 1});
    T('130147293691930.7913013', '16938318055333688341214239428.8400763989063');

    BigNumber.config({DECIMAL_PLACES : 63, ROUNDING_MODE : 5});
    T('380.30847711766825669015284843032870317433236412578443415674498103', '144634.53776756');

    BigNumber.config({DECIMAL_PLACES : 78, ROUNDING_MODE : 3});
    T('72756584205966672305485804984603.352401313503269431826525164286447688973624911557811334526750753359694240719487', '5293520545319919032011373362935198316496639737287994460100006286.544679');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 3});
    T('76576.797797984240571846486238988876843143519708396246972607938595543560509008391915715743283102567', '5864005960.9933640357177527847992571777392026372877736596182');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 0});
    T('12178753964826187888951303.3249458', '148322048135769591351831540771148782675389671073614.2772432575');

    BigNumber.config({DECIMAL_PLACES : 92, ROUNDING_MODE : 4});
    T('40.80297039432300938521588341960090555158418429841606635380853913866422389482874612382890570235', '1664.882393');

    BigNumber.config({DECIMAL_PLACES : 66, ROUNDING_MODE : 0});
    T('8003.231428514446299367379006843513173610973656001982693440541673067266', '64051713.29836138476649526336604765435988757860210340327039402774');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 4});
    T('105.20351606648271594', '11067.779792750686945021773996405472111');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 3});
    T('19979049.779364305047782515786241164454568396483520587153712', '399162430086316.886209908431086170916983676744304561893703431115837965');

    BigNumber.config({DECIMAL_PLACES : 18, ROUNDING_MODE : 0});
    T('36166928277228.730988109424899314', '1308046701010207183212414138.1873209010791907983');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 6});
    T('686799335895162784002331424287.2818926431403101047781274303105132057580624450031766616563460450210013819541610091966', '471693327786036635340416131163020770161082699744963321356476.24648141');

    BigNumber.config({DECIMAL_PLACES : 93, ROUNDING_MODE : 0});
    T('427059.834923060848086980328496831458550638128577698933740189975808665668389059481658485647685358764', '182380102604.5119819638914025693');

    BigNumber.config({DECIMAL_PLACES : 5, ROUNDING_MODE : 6});
    T('1355182862282090304776488.03585', '1836520590223078937218481593506430138446303673212.5687201599');

    BigNumber.config({DECIMAL_PLACES : 48, ROUNDING_MODE : 3});
    T('13217886.656445116298311129371851749281920448872987595652', '174712527662629.85589582771304');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 6});
    T('86179288451453008625398100282.4917767047410320423825986277768734685720245779702133', '7426869757998741901399152994719251895695045225784410813119.4');

    BigNumber.config({DECIMAL_PLACES : 87, ROUNDING_MODE : 2});
    T('145756.030733419668905069100873071310401624486529447576932195647882358629038395814721231824791', '21244820495.161579064939048774678');

    BigNumber.config({DECIMAL_PLACES : 31, ROUNDING_MODE : 6});
    T('1.2553560373690457964806915022022', '1.57591878055891310621921366234841349460228');

    BigNumber.config({DECIMAL_PLACES : 88, ROUNDING_MODE : 5});
    T('1980389898829432.2712605078486037862306810847539694712458886073478584925281882450847097667064103718895545', '3921944151385648986846968832057.6679856140253370947697275533733897527');

    BigNumber.config({DECIMAL_PLACES : 34, ROUNDING_MODE : 6});
    T('2501.6649809177087331569494731239885899', '6258327.67675');

    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 6});
    T('2196196.716556668638', '4823280017814.292327');

    BigNumber.config({DECIMAL_PLACES : 14, ROUNDING_MODE : 4});
    T('874813882040530524727754.39838037065528', '765299328210823255354810360214120241836069380387.27355428536');

    BigNumber.config({DECIMAL_PLACES : 52, ROUNDING_MODE : 2});
    T('7165.9194827881318885066411753304818478310534939735256546', '51350402.0338025276330915356430492');

    BigNumber.config({DECIMAL_PLACES : 7, ROUNDING_MODE : 3});
    T('3.4641016', '12');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 1});
    T('2581628832.3443659411183261224916944470874517039666191989033706133244', '6664807427991734308.81835798420895458431799580591091528807175336');

    BigNumber.config({DECIMAL_PLACES : 63, ROUNDING_MODE : 1});
    T('36252.485247610266422965983738050016935932785999111018508294599446694', '1314242686.6282');

    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 4});
    T('1.113858159731300796830271790717030399', '1.24068');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 4});
    T('6.350335424211858118059869', '40.32676');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 6});
    T('157154.7', '24697591891.264796990694652886539719395624528788627616727785895955829');

    BigNumber.config({DECIMAL_PLACES : 53, ROUNDING_MODE : 1});
    T('2496808231197550137.09602614853116445176036893767527178453680678846727346', '6234051343375838977712099106754338652.9448320');

    BigNumber.config({DECIMAL_PLACES : 79, ROUNDING_MODE : 2});
    T('10324.2514732464973649724678228525764035909757738486664404504563868912514925606104179', '106590168.4828324712958760');

    BigNumber.config({DECIMAL_PLACES : 27, ROUNDING_MODE : 6});
    T('386531949115297351728.142898431057482793224359928', '149406947686870821169328664677654478117516');

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 1});
    T('4774', '22799696.79785731491262');

    BigNumber.config({DECIMAL_PLACES : 83, ROUNDING_MODE : 5});
    T('251.45257784284141788311293466206354643346413863260081033788120457331898535528741326412', '63228.39890381022276846692010483321');

    BigNumber.config({DECIMAL_PLACES : 36, ROUNDING_MODE : 5});
    T('1531518860296701459127.524799663297047232327147517392658905', '2345550019444507360974678507452524424491068.1160833888702371276524');

    BigNumber.config({DECIMAL_PLACES : 54, ROUNDING_MODE : 3});
    T('29135561.909785745541160117186613533017688130999291885339151706', '848880967798958');

    BigNumber.config({DECIMAL_PLACES : 58, ROUNDING_MODE : 2});
    T('1.6531787562148262737395863536342116867328098612496157055434', '2.733');

    BigNumber.config({DECIMAL_PLACES : 1, ROUNDING_MODE : 2});
    T('8.5', '71.266173581517898007439610813');

    BigNumber.config({DECIMAL_PLACES : 95, ROUNDING_MODE : 1});
    T('1293.62699352182397997386824883628486696727271220130303818496237941488924320676792238233671136280435', '1673470.7983683132214500750687162859347');

    BigNumber.config({DECIMAL_PLACES : 38, ROUNDING_MODE : 5});
    T('71022849.96143668902502364469073874242232416113', '5044245216644747.49972893008');

    BigNumber.config({DECIMAL_PLACES : 90, ROUNDING_MODE : 6});
    T('1169773.951601496071390789369787040569431646053789571885643406476450096643513202530182542812519318', '1368371097845.37927324536101610772653318635120755');

    BigNumber.config({DECIMAL_PLACES : 65, ROUNDING_MODE : 3});
    T('762442649.06995637756037230561717976386559245889049246647051943766363369227', '581318793120812652.6480427310033696');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 3});
    T('20.02456159330336', '400.983067004');

    /*
    BigNumber.config({ROUNDING_MODE : 4});

    BigNumber.config({DECIMAL_PLACES : 162});
    T('56956.133334288074055649096270448425887257859236300516446822612657765367909481348792658918577056212964517487314659149646507878360034362047738124408378251380663712880296', '3244001124.393201124203315112854063456225');
    T('8.654478609367522146986967465937732834299195635917473015566911783193688697212875063970818876014535054471159717340710683348537384638848599775961897310476836944774203', '74.9');
    T('2.090157855342270554221573582468214310613536016537697179653818592200887519700347392246743415543531332923171565100303846336657331022146874248351314711045809920119035', '4.368759860249');
    T('4.898979485566356196394568149411782783931894961313340256865385134501920754914630053079718866209280469637189202453228378249717730919675514683251567902474557105657825', '24');
    T('5.744562646538028659850611468218929318220264457982792367699877470565900721457404627027125365596788122419557476945650837869934822615203403401618532019562307772727978', '33');
    T('2761776864220467592773690125.261636709953648799225108854949552005927418667927585431550152627344259326832390700476831904159140150409161169310381661652182261480548883652702710905192306201153399', '7627411447743439090016907911169126086985950772934507394.445820');
    T('1016542.132684851351405985352421550427715149810639767584056678930969543043928600815438993175050555365846506655226675294372370499294878158021265695180065201424027976920607', '1033357907523.46593019166446894535290441918452466583813147208529350643');
    T('67416893557691730594377.689393229996593018164745728497408716814685783382213046224003027235822262763354330687209958508471208061724779548394876757652983131645878622964693470107017790602867', '4545037536969136767952040548173187304914752143.582107466151');
    T('931291993003973764424957570.943051255559335880882830118695138743017644168272132602760327188450198488698421979593757322045111416715065737343620373909805293078217072698779281254073818550714084', '867304776233313518982078261171691244956483307888374339.8');
    T('7949156432.018787509478522996179666181753678431098013792417245447999791336046537264938248517945929921988256639266549619633490785427215813297039897879841661734306307344111428', '63189087980705660327.62527705239541530402411702633104087462882');

    BigNumber.config({DECIMAL_PLACES : 370});
    T('1809865.4646006170616739546078520411523015790932463877848464678333257743018277689819262142781464857679567195196165283927863805894754890115198595353876516167856225104865664069164914054910658837497837052107088079809127635832553363804364513985303062345471350909037103500826046492068332088151173009073561164990623048692985267110502955089488933796659145847257595817786679297018780916', '3275612999954.007450386437800884');
    T('15.467269449647536285076994289048952792417346402263779331615370527324047804918952006389255250887444861529354673321542742996531338786891880304667201838005095317222412003012101386939275591127834757079082739161518869565123311068788189396146419765711123314785783579057845157770289130253714499563733678728735780353326475011925555877160399757945810318742227262034456726810253364', '239.236424228');
    T('21515312700788.8445283321238112651368817617351069674058374989844530588669193691975020463789803512804134095628531146877959931040403076115365864125573069051677734320206655238129119512589782592335983752466615626386174785231997318261202374171285170556291668296113625567950158566625308039499284336455793814655198232231567816284588561787228106085162970376394887992755493290799749825112551493', '462908680612725763398121582.1988499361251936376406501063453105637');
    T('74521299.538521976899190350443219558946674884841824374949238074926728684231529951608824829578859140723045085427094885285460459390853176343907992141008355659931966765663861028960423841901422569042236880047925960465288786047115089356108425087999854467423897118283417884004208867563212081515310360937085763989389534476061158207010096784634714697440008094764047817723635160212611201', '5553424084910115.80715723353');
    T('1195846909.8136451949628200680969918871468342637166442528552430459926616316029985160802054349155727577926247240133710269402224145037039410361654704934859568454043181682255773043960693809830780634720080576036391829106533152077104572023692346636965236517235893617548662539881565621455421628698536701895553668525060605712336259944095451585719721016583710402332219225707779232668654387', '1430049831710844464.5');
    T('63.363458803814609930869562561468877004673338900304322840644175341400833525250548003632643676779722860896524366210225631392441735539731411621657232048800481726194808203828100942514894307898059274946673100310375760980610689938987663753721124115742622475359805871962619958182552829213760795254063144063662267166941269648405943630007682837690708592188638914255746685886657562', '4014.9279115827111984');
    T('1373.2346174982060050121030325620899957248769145184187327677463246552753848965766950735510496743337066151954054529870656708088991835100431884436734596808270643667093234708089558732282842724926267168039341016233235612898755425310217129820000361621133178074632555226320411936053443770992225078255443638854008065800708779022149941454885905152491670161465306374888415427437669047', '1885773.314695444154208');
    T('99039087799528681.3905138833923030877513778499442375472507806052812263260160068742079767622207118750338964200879423680059646588427359069818605754805574948873861241916211290058134240391945041506590998012465268591014107092271256564136984223735037018803536800185626368799784882325385180065015284472329021977308861285931031378317690220446799508322901392419207606647824465676961912238391025575', '9808740912162750909706883310050182.630527570227310645154');
    T('3.7549966711037175032676936665539744167725993728876755783348100730830605055312952529891445353338440126820518274797006625702148474699248805782797465540022583593955664081518990293569512283879169129175449735381912776937984799578622377264197905942888539479665214673739023726144209253217799902885022479674447157064859576103449224248517314930440882218595889166920009969275046849', '14.1');
    T('206071.9099345662880392539413524097442128237359131885717624301620021107001424182134616945091475733036367971785943678559801220356724867730180036789665163630501875994159091513263234064759811698332280709521477933310954189936526370887672959806770806072945194934626710227301991226774310750612896243496085684922345254659244357871503291966086952146902972859397840015475313905910082084', '42465632064.08');

    BigNumber.config({DECIMAL_PLACES : 500});
    T('89817010695289.72578371683807498008284714528555535180774576243832060526469276583706655412471844057215592302486816618503186850287965389458117650040638501953754812900903677140548024632204429469967898609400125991154018500677065159619195677725360191990304667964228249993084865619207470251438905759438499669605919588917712329966593460182177256512515083462907857531175748343129201000221666928295926394404306298592439213726996025312323143449177688154398992378748586802380541165612583584316700420134596203072442651698983252', '8067095410237788990654508945.4938889620226978171380538868');
    T('186.91379248627563760392813675540099210378050749661641999000996750119358911115159296630333873386370549699551207601328525759106230341322609045770969551608159859115697029049776748016235872091886303045181663791045868196184184442196533834203793388185938722188507323334759581974443773297252169512928459122350262968374432650606625865642232545046019710829844705222543050921091232570671006274142816262680418047451611346801167563231483515250455068357839726457106935322253398592250219377292043183684947137995080635', '34936.765821602511000');
    T('292.92251535175646173232431728499234372421154418946156867421786152176227096294653007831356181142938132889371346675529071648577515728469391396601893486617422900836680345530254508647264899909100464612083986261265346005164216654675922351728655350260050928208855804233241090095534108831210571537049578731309192576632259878664884788879135261573200114727909644940386409994830780435119468243089463108989041078103645175912033463261798424149302644555980788660713206241347966250504060797821189239505007200036079705', '85803.6');
    T('87.99026650715407396765197446961570650402634143914903620761528887814661859660279310794320854730578987027187472233291266405007186843204281271710739248189201489451091841274017441508638469723085782460443376671758758899784113134866643875281588211922668056326695185017497400934539461041472568010022163207379422457239669042993592416973790381056817762649457565899290348718808685202005765019872456068078605894916581511896996069884199563861272506806598271531781334519497271503568279500098078388331561457291680397', '7742.287');

    BigNumber.config({DECIMAL_PLACES : 546});
    T('9.964115615547623702713334497987282609345101391172570364849116418075816800113640413848859226514077074748449697076758383273301113543709109909215564170600206464958142454266490605912308358603735813671079804922222224887150495344623713603270259563840115497313009052230649783355536482654846683965583855936794259084015703511152134740957173368048269478700424790749310707209710341986660107034655477210204159808351721182045528640956247532335115997324595227811783575481533603875179445069446441375208735891932273626153407869944580993315321965826502787138270543', '99.2836');
    T('1.307100287867682515567807786801412047161737137612032679611924993481917045781734946943323223902612735068918688582083522849332454081551075797942682446349695592044164001986554222994267637605078971511529704949687080403427315218684552520061137105759437079989266294002711708038550421771209354048361232627247521144464653931131639432007805143883006541670565652328321108764304216848523001514451272005371748881596518362518185934727350138270219612720116871348545672028299730589386444001708243439232746226873799802838450712779197269114313367331740420673483739', '1.7085111625437785');
    T('2179715624641.104045451948581834441551779587152634672576510540355644708264438654277476265980691695066448533110871133836814682961495657626427294887178053496311230025672990968595927778033477830265241856037103266547607039668793835170188641774636594856220333273217160195552886973287073504687530341285661755863078114639819058730667005812528346789548066256665974549595181496892972941783341809790771728591496793467248475138025398858570306396853160195272474728879369794382569618121601470222156680464345972678848920184128051561184284016452602903346229119622194482045757', '4751160204304558385373451.3272332038165539477606810406144617');
    T('39700.822577241580199678797939419163898886687420779218247258787086313113076859180773033077810860147036816212863235098843384590784069398316497506974294775336733386252013568969235999557522683195367974539653911477425701468208578424416631114715708576208486734127431256111817135113173806314284300643116495170122929525074255758154469988956319381763329406201302760736651128326823314686997919541126017908099065125010684846684750979030402294608072194301572266690808347512619218310009474588358252587183377754733236309560561175738877889452782714224060764008968002', '1576155313.3096147862201867397559036886508');
    T('1.25794522328842464889802954979839811250322044212082313174176990936228267186851635174161106107130430165025307524401690618283690880517619898891441753062149884262103288048373107482275796232219249109268011616848844350466917383546494632887628802000805894961164041424123794218062370303230048277719411087505573740519315205969736920463416188404939520455876144966180409180303192875736712339448801403919111160506886246944037198526719945050982972666268955585211897123758272243907184437446082156997384618356833205776554862542099528216674650656658010050062231', '1.58242618479416454763664575065542265574614542');
    T('1.624807680927192072091976713653208069700840817345062565530315118906583360568134538999831520589587375487595902303174139599422604662135717284255618657549630378160122248467747111071314750502291444441102163105431957570130226305138621706071808286505297736384595648509872968772996129393038122816853427310850020078925936656695761188275301963596778663712697756996198695154835116221911512411787284450370861055864406526637410758400234987941681870570312002436208851096849359131928628121324230998619957793873321252762427349655447578019631964607840051797535418', '2.64');
    T('18717636231873067799.216510948844550745146581264323539180043356582323609699016487098997305478787718252655554063999956808135384833559712709262538242436026132212417100611618928564814091772415407036455749502462185745609885577739497207736237382361239184683282552834573776377544551919252603573173778169991084849083487588143333148865799493939248463005741865268503183685175205052842601336209845292039862576097287476259672443140291459083371484939336391166547726508221768343629133487223197760680035244638718987232096565446767519310698558767017657627306105991008804977618723991', '350349906108727416303231044881030663104.72849777316356826833515716205');
    T('1406647840009650.118863996881143742849154676375102411920435563067539895576725213355050743434710662527010561053318071130411761077372296190615719469716323240116449116038768804389193494725211826193028915331814451122261159238787939478074609835120057229806915907338375241900963530017435537812109185293173117802944350037251407192941405132508530625153810048779676085394020972072535926554304846973136440349018882339256154370074386916139960439011412622403829257294020556809081837690809972797912280848596346835691547349767211626827238693605471315108662202888476421988937787', '1978658145803814237711662058049.2544850135714954117875755770241265');
    T('16224378173956061607697862.633217842546427767370100855088171518705366007862250561754707243455931128283904259226113595005700826232879759488413300163002142315099475508032176198042904834129778691907469882290201099517369886984799723525714860875048918708175179481025780980785512861833918274288833606546216800173441803028496920438426672142753847581458627155975653306204761742838982103620921662180680218361989958267237075686446982867517247246816010479297470319222022775438943899994549052934785516690930630661752088290413832045273976791735833819617572894548667319311757989955998909', '263230447131541828089867036367977560519472832588035');

    BigNumber.config({DECIMAL_PLACES : 632});
    T('3.86684626019706141613723825656699191004871243725435371952851102513193897300170783147869803597806359975114434638267696014418502590333002587206926390370265920607601489345602853935200922900139573459186086120455990493891388555877391847755178584266457035830275159909609535167181434973137025675857847818998237738460415581254272837772822208258242097068355115922762128443987625212248757154527289970249951218201075262114449980122023773692545156022247353209795532545041431858554008616906673871136025390102854614065356679474908404595273532216911231835866021312311702806567650395727590121314352963554602357180371548166632447396668297987703182598', '14.9525');
    T('6136762.26717905744611296480849548677883172137201795887253092440892121660504456966433584211515954953762017772580407928362904974338214424385785116489885777383448138309647350406214299077359122447734595685831178247253060071791025543499094904828078657524955810034216582881357125704944563248190767043385107964306199859119157271835974466792774248278303577397200213978694185732540975537354775071603950395990745619635709671790827074205226736000604510720564275623625311465188763715973066573171788415579221561790250264794581356922531739753608438715435448872139277604185884704559289000914307078242295952209310265904281185435952775847430570132829431484', '37659851123872.6452468949180815564659044706870951617310978456174564869');
    T('55584781983.24490696445791805408404378469879379761888397244664120330644772739351913528775184610233179267381874891789954760338007940124374026003041155925024622975618046085564412007615044940190982877876574993585348546104467074110944013210078722838691327953750473674565370730974387207982723611177965946904757209441712191363169109505165807521321899672463583311642877639945439852157230367246653362314539530980304123767170725630541393623958472950450526635784974484049786565210736734671468997713205366127380246276456634498806056301931312061784211915142498400899530080598267700949248388709352054923014180086178893193927288901542298741856815720408810426', '3089667988124867612740.083389049922345');
    T('1805064995.270685679369819389640432247172036506554563452359428083087328761361845652635553731357289990951745475326934025307848780337675451754210392546185393830016044551272507825821935983758954014040138643670810605163035656676595367710188204749719310806964242566460755804623103957354726598643871684791346043973374359934258681039001336439023622338991932038561462612292035112751670372206620090017313856898980333410467539643215942489844251168601645977120694640664003642300053879055949999123327998888486028967684255707894113041249429918747443321228571191289316571869868803697397454910198466063233810801875443876639885268362424418465737511676419844', '3258259637151560514.0297800164403204384');
    T('125608085.77042703662921929440576851258415350669039591021196510765322333997277721479170585932128266456989640874478239906029432251376275054410584089545239517498359890440064005754383089916535639818817512288635165379914355624246231855413853842721091980292116104830063838444932363335270696391961823972078627647993680508983028621966220040659701503608235895398573773969030627362781943814154637432716276122770088096908725956005011070226435272443863093204149560949439672185351571599604932488380558361146700216684520038630554153765869107472500236645669202075005111635063065232469777930428976428832531849390576765836025904341256201817451204276318678703', '15777391210910955');
    T('400136410988877553453582.37101163421699743482259029089762021744122736955079765363264351177743971980656087133988131526954003823716158204437583618123747540964307177407397030906491059469918005937208664667811006296696880134895202857597704648341649853175892159884095827375237964722988879265135211319838716432321482642548617255100570921567626849693162463365611013775481030077269042675179237005903008278699962449590513074693187300493182718210865710393790428022783178966907764188174129568850369221656184631847400442208130426552208194145273689796897187359838715920990228135785300048009159741600170382195085666098719445640520314294446983533576482734809632684663212616', '160109147399059929314877918876109427110612023604.056303');
    T('64425.39708677991402075085097430222723408726402575346205033002867968020486619151730572708995063516358815252500577128885851092173171595206567522039205583159322846821419081246805971111195118357652973497110646963488156724407496612917479150812759668015947988612070186616711130012713409715318722886985358698149495882008505150340412263757850467789892597196854389378140820673566768146692006440247742979249196409269612046734062701334694240456008673406876828634709710253057778438021730083759189755501262144599075931168939668888175040990882526418711846468669816130425272415774060047616063598593294412963338255219828974342389552617191145782209745534', '4150631789.7892698323562331016634257014646926299287710080');
    T('2348735572822814115892.26408027183370740113506236192664551809826630380561100793615743475343501311869598689198309313317112398706024097995873626836363321037439405890327574217792370459096267386514893897401545965801783674703449797383205640533682179139228566108876766321414465683099263110272385159768745386268309348310852289538873334789073960542826136816561584906159230939562245879068384254598940764221550586208646296806811463864282431442235899226384157724906001227498116520670108267462634019231455786976410894467414188400388964899956986375533986475289820788610178735962555060048758466647342357027384051113897165096975205112706486602166905905775844738974189124', '5516558791043312750956805399216206701389822');
    T('19103802792.01564371961775338869391426446448029286247977641509368293241906066424287682005366039677603554915595198800053308245870218297432073530945933295939680395210865727953275117534560571377076828346246298768896265547871244418699953520469341265486692862736845896173671301154158095469340988225077329614333663129688855004957099781613220058661676296497307424000643112303165084920380937145312220630345213927929149505102622408340154771991875263250438238123325276964888634956357905768104652580983969312174465484750475804611861727043684792253730740484442336788006549604123329617407383758260482554791627340811587637419676122565011048819721900614226661', '364955281116224704333.02204944536');
    T('786.66034601980542438553763403049983609539851928329220147388661995860830168191247746764643770838112443216591486730544324590358255064211791028661132434467570268525552578636466675861637897309153324177405101531421741979722830892607461959298447068614984201744211147154166256831083543381185024040849431152799003153201087012609166537695511250190586392519199228602942589459773862505985356810151934653735972412239324596989225867188473011661629585794534658478043644553227272995034111771271753677915199703704761728445313825636390237745870611713634838750078939991244597310336345791337461711284642126303822583081171392986526905442664045593489255837', '618834.5');
    T('484123309.97874763869513775027582083760255026422629367680383349228461771885785050876393607959363763562763934512375018110967693444089318521616498831171036375527664923123605952647435477649669320835658085436446422080237311136711591389890017124515694523108269386574671466685756523731039177122366440515798582799809939132135784826352950211725429348489639356434994276274903713471535999323126607571342955569317639976321722080574385327905662904910228854922313780512285515691527026937933195761382827883549497100022950886879913145746098441721130133524826420689100661999466875481778061645781459258894902002955465824742017940875767981672057269065469536219', '234375379264778573');
    T('60481.17814179171662585426325768194943112296504919133512658120011357361699298422464207199076657876167621169341810823582081006427726259877303866169887391022736822586491147183342565113223464521871784706899966618623012338886757864618596318826063555227727089240080604262744311440712982849273211826938920661172972338303097711545177717262755702859402796454231559971416344991871436949784360654196445387214315983072589147867091579864546816048090705979976439302638898419268726958146631934559321370862789950979122360237770183583617309004258400281803258965980586023033093791875491145215922928234063783157425179949277559990637828701834500469899317547', '3657972909.4191441244525931');

    BigNumber.config({DECIMAL_PLACES : 736});
    T('2597183854328.5679091905600850947966546165753359857320959797758353429783579407079182939565903056335948426093650795717332704791479071528177519723191124947077728435798063428371161778782597345429371741476675515164182960273091825852969638261871905188816333787724804370693852910065686936635346834748197083291357776431587087848800111892221988002406355426392615087567342248786343354283235703582188835390882988811316410094207611114574124785705884308466258369161492987882300555227716609880303616554951040594534188187971605746361153720086319724295669667359878510806390096282081082192285001657224319276565070534065340503324866753603713794833032825245475870135528342060466443304262980556284100001276572690814262354674742134248012676070023441506227778586365371105', '6745363973184995853492478.5954');
    T('6774655005740.164037907899356400277125449228462111588353929941351964991999943642694024106002045654190469894640190593584143456920968883709637122260941740850619226234998910249288941064461839175935885418202698405604592790954509193961851585542921730413008144762195532966076043993178536807834225605865732009222042249139153172302609878922537826024024765317187183190793601555449704914268951048131236504332674153119659391120390475083522595748414072828021281006933882134638700653092554277006524993846324403138880161051477517078245698926520722382184525492883300322699107657083780701388937788565713142369080803278939033122119335669992754410484005485791854368115176247813286850021744435931737469007811125083823541974831435625884357393099194117719985023995309226', '45895950446800262033415363.0107589586980934');
    T('9.8994949366116653416118210694678865499877031276386365122367581659351273492347492719527127402934910091450969236163860791744738523909551604885088504796995508198925586614925391492905996535080694744240002907669198012417919217421780341030996219304862243902668663046133901610002501558783382253959667598559278107737758243226618074843443068084667653832202520143197321088490265423627229990794806222238594919301227082997948930337052138081945322151797347614794505584303510640492417811766809910911093299829106606470195648540287464840079528339429999802421027773045400050984119089599487743576041585398129111011611345309324458600582014599091265650208792257610596486142602179717725277099578410704164010928022305362395364280948034487315508458169196178987', '98');
    T('87181.6234142115525219620524691647002880373681637662664358089648865241053931751056720336743475573704667754469828869574971770860976622090504059965985957160672759992061409022462794221981547073047121333832741614179766660169320255090772060280986675121181149188392777041782943372931223028492877312925787086980314820336807572760207027632594701362292053256504201751684547275915772868234710319808033440202969911746099931970130960400927941767359496945988333067403190717036520674748830967881080827391448011943102976676893692584203783154822448864894042034636583043861494867962494678297073555456340277748807852218397069734889863567431741539523977203341337841540435143868785323742066976229168363835362478774511412078654288725506862167769261569623256490877', '7600635461.13740');
    T('4.5825756949558400065880471937280084889844565767679719026072421239068684255477708866043615594934450326776009053975857408733118991707556019635738633623447553702817228244277981919405969049143449200907589407063891768364965598654748231477641506627774586518102302361748154008139424109257393659219939175422411846714980804218526078649141055769578125547677613025186352569361961122418978100629104930136394500794769783448686170875778719949754100133598746930349723271067563482654740784196320965532158989549437375426406190866502980945808909769566178106179115952200007681424851342306651436695554446605650313382719148849087683899267628798139136482473790088818917079648337517673741700029719068290493626817167652340083439423393091048762655243512703702846', '21');

    BigNumber.config({DECIMAL_PLACES : 848});
    T('379533758647.33202998025355894771830486671460954826351562848093352359089756978384860789725657951059073867429383079637341943322428876224868137020890589117272041215996572802315226952651591422629979697849613357409838019360130112971147443970457701882865210579938015503729060796093840510498092974918957475137648847670773105840478476180774814337983644367224768670934474545884964455308420763849854108455051079039927823221492636701262715567129335694288707429643301669699497029554944442286919275175462196928260780423046910979090241979424076642169272255969335731465485030674055509703481930565243394712596916609989415808863418419835631677700360441697210083653827363223966242987581200002526377610727629936341627933125420316464144353438091881740224503276222975587368345923672238451009927592206329287360888148821167396976641547561007365328906795045416630208044281045956444564', '144045873952971280443387.35507896');
    T('29280910699687273104284697299144.07732565567508749484602637961092769263535159046426239841768965764616197975075043919815786449076328689684216110222824295072121081997345891375218925808047569533170997772123590787790622766592641361527147653442819434809931915905219637858026995741191495013690327145578287454531481294854702286205399948005290251392013711014077536227775356988930049595580907927661449806384364243569981109141658673685934994029666133875734164693622860720260710554408636555912010467046231604717399110220057952965339477795911201244340638839241074392352601833736408661548059322837128092912263901889749977388312010067566997456646543486536129645809758335017030111183330787782491931288652088515560703734327816427347518899939257817316435042124357010083771290898500690085513377138936923205163016022860153854624965235876950834393909963448127891696108571214858469890656907205330389064', '857371731403060633386241816096841559023596513404769917799492267');
    T('168053.04565201524955446843348960783836388810418113245166824850488690908142778357772728901345436007400982876927959300542146773837124920813399432364711787980877197182718826668919533357860217297872518296792465703667850144785269168793146280553383246166503410178151205910689122332208239404835749221852981903152552178180222439382939101404418973062745842861363866266832597644550779142636812629347441140406958185996959809516768124071903548716934343215775054973484772291998064110162394410530239517557259046875734667137531966463173455499642893757346141380481602264381153514663075350700048417551424923591583912256618460494552096382446465611749335306415219871948883932584670263698301372130930285595038191486637767642010763257747895310053515488933059515998388551181750336022799922855572438292341639496128201699781316142673719517556328928078948111316144339893593051947', '28241826152.9183215732505128601928949420068589166172608979274924914576295');
    T('1.53638009381417855214702667564342109510652564612190733749841594063585238492794950136881162227066019813049456848975091045178675462617786165883956111632075830748654795458628807912771681561582305470619962729037215106906417797478639043759073671526233150625502278985764285240647638298730753064584097255132097117503260610789389208952460787617822842317892961282136931991642143510463540230197185978192801359575147064297679386685692745377174344652745131673928629270650815492384956061132688378461069498104923826104838586490333978324236045359018410309925372259501184209171535596493166826553453125808213238306178803667493845876133602921479334100929846229655991832177474753899009595435568508601133899514000384344943496815277376139253215613276576713699227714772914534633123209445267405754874955789973243763421775941905041695031197207082735610030092448816414167544', '2.36046379266846408899539510197326890965893678862224859555792838223');
    T('714.61241243068259484311371189785768955751306153276228425541721886979377905758882133446197375671353721565433127023956674702853562936178184807139172505791453268944641741010891530662590385073593152193455805953240947347607646479943534513966413454182559305296106776895316113963327870900366698599660151420070929801570405751345702353410644051682554211976059908915814146604831899204551744694140994594366464823764389570912629592676719346123171637726003544266618213687267325838123794874975527306940385282149798065123421458819840361280851912003779857428730583255104381030891533142101378117306978641247346815845252044894839537420273047299982385917944987149055257937383992606691257056568881397021059046488147159677518654506427704867625959014109859319284280932956098825423372240493017931941597922444987378795761902915931399027545696337420679858404395579037783661674', '510670.9');
    T('773.59472663687912451159535229389853939842605059162703128538789346863286440830494881075568422219621192333688675281822184873387170452981843472591508432370385971110199119193528015500760543414524536242455544083490104908083974792351201506935765119161643924169037264164099952814746396926161440416117565279304931119669840952024989118295794278477494347926021963207180850887861862534960178708980838985387368839444500154547698864030192542133112686175071539548217199903944950274902992241082394258689568848798217431438259861325777938469444568799288405922158163958958223163833937391698968594686798212843588496878387116515526389464647971296312524748306815742536620947543414457504426763294470951946122437335497099494910632585762600740409680286609238779306759977660554002182801075912581235512217070538480899809469699327979364989178344899103109616009308707414801940872', '598448.80108038774004895');
    T('19008878036934380567.84396002161007307494234445138537376816800048343616085674206626936199749636474720092004342883045275441459628576839920971330702661877884977464784187807450210290565316696580997145384562781682128000995898273298139743308228182662902555052057691717197189289590092259236658987055552383019917577612145454574579719020972478133587710937217336044281243302964649388576337469548420171603649754413233412982255987047582379037245718253216484000312226525699478281161585924999301376693452172203272239337852180725955560197800243914977419579018956725334811006235730357909615062778332245395221460477491786215707869006840640344479485584853728869293288919876710562517254695116835080137005037450411798398558498522188066156712112948975309706291905861236209197663377395628993573005290776392002172440861220587153770661110412322002802191380189193046537735515344097800652278112', '361337444223046269803581586105656008038');
    T('46167547682276.69659839142168613300984418968797481977026434015151042113254784992976400662854539508545353029755222531760805168040819029032876000018647210639969596871042153723516022413007597033768299999552605409537870966847134283383166186477065225427587170669800959221329223140225378864660505062217758873720294200147677451805234547441880387069084927910307749688653503693156999242010683739605876996616940672435261250941033196533134836202233289888667726447834984240822254637787801181963951894038647953211128853346780059900073416876543354577813540692889219448228728777909901391731624805683971528990295277436840161294579496431876765112237134229826779443182941938910283177882124027553167373791516105803517046194645899736931109009785536544701916275081569242392295709503599284148920065992874897610987646342569478342044371048498987276792566426432612227674148724386753297859', '2131442458995292379923442891.39619052405363168659553832');
    T('357.42144219667737476940723134311437237901070981997683604955545480813878989092658709890042422907764330003065939163827123844540310894482334918652641803484529491350116195509236941247526441637135860986047310858645732381561361417002489911900180310969665452837050841076667945444753611474726298122460938050651055487683172415032060683745296386129046642462594028333292429079902699978203730134661132884803238529838974477445191759967198471721764551979289888303016458787572583353512731118148130889821591351857798041593003937898405797601412782467072190053744174243225780844049835772183482949546683341934096818885756007124382254659796426711715080646604861045965994466209388552589568508290975087418855481674502829846363820364296679009102756062772768505880054488205022390142319706776092235958701849286052672927602214663014895377779429627657631859807708535799000791569', '127750.0873419527858363938899699529356795979');

    BigNumber.config({DECIMAL_PLACES : 900});
    T('7962389669.802619226612420082472137241175608666661735561414607900997891208674399029037690473846801874885757356935098345580426591903055562834097030473393461243383271678522047266370481726950449902079177413029320835311617540338949723112537110203362427858192330773112242226738687601815379027664862463277214286567895961535965292898558369885348154321255856088316638660454457806259636801735717647353100345104124424309537274722443977557763052770155305495894443403910135805323387075986979884756306279940582254155475425470071419924272589017637263436788853746084045571609754415271337660969860121360062716460814804823355709978626612344057738355406928320053347312100043083244150469170748731787389950929048830984414804237666515908525165972258804621273737876286813438283880399279278335982619019392983109710369604460059859022468994373480045434248040713896862456845920018164581498873236394584793215632879036361684025776086529569', '63399649253779463637.88320468645646644051');
    T('36985.537435306622236467270937252825254026014721851309188254305389713697015230264185263378799026382334714397309571564625416241215217867149723572425854181806966554642360423829060014881901773199386994460664634062101904797359294551321318937235734678512961188607801986220680153014134029493832615546969623974916827167078654634358574333678389616366410278915565703581941672170730893983545040190300249274221426700113848290541507663318853535927775893362185694382544176373482231304831101837902849800951258220726268990677590958157914974175471404517996778205179950329995483136931650616286373118261630043285103533033040476431824360766396365422877272153761572905701234187642833656097211644081252236980059582418494584853215946576617549259994456993750322107749520964217795499374138097379418654677424839627787482713659455961703181646841698674573783883478313655188817893687086751450042662911182317161526490882315503431203481', '1367929979.37846755563562136');
    T('138.653825283302247079520109639909554830495824478735118824960659298570304260365235201452949658158554711469364399658798667165750129626981635907513515727752104322402093210187651015212649867280391248459370998127185412494794237194604573645718383791562171900088214128456904501530127980235949384533924069927959103110766006405141214543337550010111094563431666164054892460728841085337947430066655077091396441886965426495467949136348081716343859215231704926659844712690080814553677280196114278268613368774166884194655930307728924841441917187821902525242579401284440544702546289065296084983476636972703960454064127793719473325530264873714830294999070781526696237151909302447919905808564848939241449791664413964130684658760296224947984073972311661335447223880763498816716066907256956648196113869920459515551692886783038790708833001987620466496959254546687493058038109403699935644806603442158642646086655023039259769', '19224.883265692505457601247932291125354259868778652089681716374870935');
    T('50950.440174742357175572071690890164127102751485515727366550390683608820436132956072025681803295779371667169337127406084175358445300796083953408843705969018873810510032889756321223393470759559987454753284749000495070038043049225296587093286942372337838245436787768377078265062338033176995223379224008076742040545070485158584805287828654557110412928438688654162185555625503306254982210625622689237909265654674405693567429263013187828668914075564173639899430670900405284455694427199638591118702956228653269921902023423130819746958603054104765246190233159809281895358531923339811813637075650854380095230369039753422001934387105096606677337450258221260413504059391491686285381140987958601361542484696758005412715512502019501470056625769917175363496998761143393423591945956802212685363531078784887861348308753434147834816578247850672086197881232957501680862454026927989538997998521459117388396046360043657658205', '2595947354.0');
    T('10046454623254826961875382317790.784871452244144975743064060498244021304220015969466939024480223499960901641559066665153037197032593546502649980935727654208439388853861419936425552530271530035494928115227726482675679211892969869213891378152728772302193716001319517533416123919336946085430093922543201768660204280287527989092149328624358683559115176981914090302709863561780525393374380243154381915249441910602391360612168161991933174297686912644986231916957310635449019093633359598148150548731899729849141307548127201093974033939034042561257126753456557941211256135898280856479300825592342274773764495952079469431439741906942077077162075390603294007186657196505030105092603376243537890291750177091215762681318887536194545745370727643575228977262734270257261824137515623111996150864948935760969962110266201746190014377183448005770929873280926905656965694617633851807904103533078781099132125392433972079961527137877338300842675298415396', '100931250497118287147460895653422009190477535915527140997436145.5');
    T('50312147.895978789945318850814716769716556529351439472968676011531432412685095617354326688006158268280144448102851948493093105193248333706042845584818818717006618051899381392239777663697210274767993016068817609233147463673391458914578371028438213381481915193432915606333241411830951892952556850603476911011964242221281045401447599028993820263011805445545067496904941492500519891679173682065830498079066798970609902316454882840175774548136572667012712115571877757306654257623667462477526405934096044262931622005593806759836229355795092155596007885177190326989920383365286112949649812553854003818096593686636208662967380298108969324741181800273045253898252610330097550632621999520072653397155741940824966908033268285600350252107768305144976490584881544838157210518233799995832553321591285254897707500335337126009624774906497706411815347441932376790317896998001657977732263675649024815291900604673316858390505885', '2531312225906842.98');
    T('5.75006173879898440869241532260800496489142124887111833188883261579283008763735295296179041343250976501937101036585472900854336327489380123282821037481044202916203441849004689471000360958945402609580784572353735408452226943576599178952237206876788896067842553596176879865752033951176088455975165207593727159176177085095190710656229140725205998687018442446680477310130738465615556577329602100580746699396762667032579047007129320587668015965987773825003138735379716404415157863124832466280028377682906939092007140554677511356063954505216670858009340392141492467369216371170692156583092566735501630865322582512619490429963282482788624823392158158604152325793299574006823928013422664018092614581197981204160781788183807939164255353777192327520682879326632498396112963534952155920134048996062606067862129834413685924986614062985203382638465654985330058303916447123647081510424589324268892533850475930056739', '33.06321');
    T('2.028430427695266062116855907049682671555242569377976031112753398044812309192685964100965099265530959360284365757870326033053601551523629075378378856745863785169732838994844919825336134776071475304944281214520910583421287945003559393430216592924156944793703503429916620090936360805054797721801359569417127688057382996969292921041738603630480342967692998233877581650620914181744888433057925936957891749666196052068806893889255339075884495080504767342571830767185504079538214007613797919720564247610464573496500053065413926974012933910366107634249863611197215314558918759158057008769626766485169836568960326396402057910902853473776179915408801623228568357317462913587164892064775658027833999904990053550155044417555290139236050616243301785245316354941879867361509091783046268854502291693337916935351747382119202947309008531765489919678841207661025090634628632028473270852667224552478918030875120109633343', '4.11453');
    T('14733978519855.986227038024852091926922255130476070894431360177088165996112823755229640976682373714559133174204321582457357379198592714332912375824678967479352499320917276441508273646332050526613540155298794906758075083423887078200491328803728379571737163423282193094119893971969158039986280336661223358044916730250644983211811708805361569086764053956417999765164201297583468149451593886541587337271459070510498401319438130775438554793347084598691604611470750783139289554584374151809770130348507437456446993653835198138094762369827003035065371494357530678660768629856933317557149699487339377901600994041273525198654016009799518759093883882149332746270829795417341182432885267251862546055608965944281933745525115737980560603226764663919494161818265452113376337284183674111018389097369572010360312518568654815367830169145059770161568817087559064704425726854477168276312695713542835478594810618373570380952296080331906', '217090123023577598725208942.7547');
    */

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];;
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;