#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <fstream>
#include <cstring>
#include <stdexcept>
#include <algorithm>

#include <openssl/asn1t.h>
#include <openssl/ssl.h>
#include <openssl/ui.h>
#include <openssl/safestack.h>

#include "opensslgenericutil.h"
#include "opensslextutil.h"

#ifndef X509_EXTENSIONS
typedef STACK_OF(X509_EXTENSION) X509_EXTENSIONS;

DECLARE_ASN1_ENCODE_FUNCTIONS(X509_EXTENSIONS, X509_EXTENSIONS, X509_EXTENSIONS)

ASN1_ITEM_TEMPLATE(X509_EXTENSIONS) =
  ASN1_EX_TEMPLATE_TYPE(ASN1_TFLG_SEQUENCE_OF, 0, X509_EXTENSIONS, X509_EXTENSION)
ASN1_ITEM_TEMPLATE_END(X509_EXTENSIONS)

IMPLEMENT_ASN1_ENCODE_FUNCTIONS_fname(X509_EXTENSIONS, X509_EXTENSIONS, X509_EXTENSIONS)
#endif

namespace AuthN {
namespace OpenSSL {

  /*
  static const char DigitalSignature_id[]    = "KeyUsage.digitalSignature";
  static const char NonRepudiation_id[]      = "KeyUsage.nonRepudiation";
  static const char KeyEncipherment_id[]     = "KeyUsage.keyEncipherment";
  static const char DataEncipherment_id[]    = "KeyUsage.dataEncipherment";
  static const char KeyAgreement_id[]        = "KeyUsage.keyAgreement";
  static const char KeyCertificateSign_id[]  = "KeyUsage.keyCertSign";
  static const char CRLSign_id[]             = "KeyUsage.crlSign";
  static const char EncipherOnly_id[]        = "KeyUsage.encipherOnly";
  static const char DecipherOnly_id[]        = "KeyUsage.decipherOnly";
  static const char ServerAuth_id[]          = "1.3.6.1.5.5.7.3.1";
  static const char ClientAuth_id[]          = "1.3.6.1.5.5.7.3.2";
  static const char CodeSigning_id[]         = "1.3.6.1.5.5.7.3.3";
  static const char EmailProtection_id[]     = "1.3.6.1.5.5.7.3.4";
  static const char IPSecEndSystem_id[]      = "1.3.6.1.5.5.7.3.5";
  static const char IPSecTunnel_id[]         = "1.3.6.1.5.5.7.3.6";
  static const char IPSecUser_id[]           = "1.3.6.1.5.5.7.3.7";
  static const char TimeStamping_id[]        = "1.3.6.1.5.5.7.3.8";
  static const char OCSPSigning_id[]         = "1.3.6.1.5.5.7.3.9";
  */
  static const char DigitalSignature_id[]    = "DigitalSignature";
  static const char NonRepudiation_id[]      = "NonRepudiation";
  static const char KeyEncipherment_id[]     = "KeyEncipherment";
  static const char DataEncipherment_id[]    = "DataEncipherment";
  static const char KeyAgreement_id[]        = "KeyAgreement";
  static const char KeyCertificateSign_id[]  = "KeyCertSign";
  static const char CRLSign_id[]             = "CRLSign";
  static const char EncipherOnly_id[]        = "EncipherOnly";
  static const char DecipherOnly_id[]        = "DecipherOnly";
  static const char ServerAuth_id[]          = "TLS Web Server Authentication";
  static const char ClientAuth_id[]          = "TLS Web Client Authentication";
  static const char CodeSigning_id[]         = "Code Signing";
  static const char EmailProtection_id[]     = "E-mail Protection";
  static const char IPSecEndSystem_id[]      = "IPSec End System";
  static const char IPSecTunnel_id[]         = "IPSec Tunnel";
  static const char IPSecUser_id[]           = "IPSec User";
  static const char TimeStamping_id[]        = "Time Stamping";
  static const char OCSPSigning_id[]         = "OCSP Signing";


  static bool KeyUsageTypeExtended(KeyUsageType k) {
    switch(k) {
      case DigitalSignature:
      case NonRepudiation:
      case KeyEncipherment:
      case DataEncipherment:
      case KeyAgreement:
      case KeyCertificateSign:
      case CRLSign:
      case EncipherOnly:
      case DecipherOnly:
        return false;
      default:
        break;
    }
    return true;
  }

  static int KeyUsageIdToType(const std::string& id) {
    if(id == DigitalSignature_id)
      return DigitalSignature;
    else if(id == NonRepudiation_id)
      return NonRepudiation;
    else if(id == KeyEncipherment_id)
      return KeyEncipherment;
    else if(id == DataEncipherment_id)
      return DataEncipherment;
    else if(id == KeyAgreement_id)
      return KeyAgreement;
    else if(id == KeyCertificateSign_id)
      return KeyCertificateSign;
    else if(id == CRLSign_id)
      return CRLSign;
    else if(id == EncipherOnly_id)
      return EncipherOnly;
    else if(id == DecipherOnly_id)
      return DecipherOnly;
    else if(id == ServerAuth_id)
      return ServerAuth;
    else if(id == ClientAuth_id)
      return ClientAuth;
    else if(id == CodeSigning_id)
      return CodeSigning;
    else if(id == EmailProtection_id)
      return EmailProtection;
    else if(id == IPSecEndSystem_id)
      return IPSecEndSystem;
    else if(id == IPSecTunnel_id)
      return IPSecTunnel;
    else if(id == IPSecUser_id)
      return IPSecUser;
    else if(id == TimeStamping_id)
      return TimeStamping;
    else if(id == OCSPSigning_id)
      return OCSPSigning;
    else
      return -1;
  }

  static std::string KeyUsageTypeToId(KeyUsageType k) {
    std::string out;
    switch(k) {
      case DigitalSignature:   out = DigitalSignature_id; break;
      case NonRepudiation:     out = NonRepudiation_id; break;
      case KeyEncipherment:    out = KeyEncipherment_id; break;
      case DataEncipherment:   out = DataEncipherment_id; break;
      case KeyAgreement:       out = KeyAgreement_id; break;
      case KeyCertificateSign: out = KeyCertificateSign_id; break;
      case CRLSign:            out = CRLSign_id; break;
      case EncipherOnly:       out = EncipherOnly_id; break;
      case DecipherOnly:       out = DecipherOnly_id; break;
      case ServerAuth:         out = ServerAuth_id; break;
      case ClientAuth:         out = ClientAuth_id; break;
      case CodeSigning:        out = CodeSigning_id; break;
      case EmailProtection:    out = EmailProtection_id; break;
      case IPSecEndSystem:     out = IPSecEndSystem_id; break;
      case IPSecTunnel:        out = IPSecTunnel_id; break;
      case IPSecUser:          out = IPSecUser_id; break;
      case TimeStamping:       out = TimeStamping_id; break;
      case OCSPSigning:        out = OCSPSigning_id; break;
    }
    return out;
  }

  KeyUsage::KeyUsage(KeyUsageType type) {
    extended_ = KeyUsageTypeExtended(type);
    type_ = type;
    id_ = KeyUsageTypeToId(type);
  }

  KeyUsage::KeyUsage(const std::string& id, bool is_extended) {
    extended_ = is_extended;
    type_ = KeyUsageIdToType(id);
    id_ = id;
  }

  bool KeyUsage::operator<(const KeyUsage &other) const {
    // sort by knowns (in enum order), then by ids (in string order)
    if(type_ != -1) {
      if(other.type_ == -1) return true;
      else if(type_ < other.type_) return true;
      else return false;
    }
    else {
      if(other.type_ != -1) return false;
      else if(id_ < other.id_) return true;
      else return false;
    }
  }

  bool KeyUsage::operator==(const KeyUsage &other) const {
    if(type_!= -1 && other.type_ != -1) {
      if(type_ != other.type_) return false;
    }
    else {
      if(id_ != other.id_) return false;
    }
    if(extended_ != other.extended_) return false;
    return true;
  }


  KeyUsages& KeyUsages::operator+= ( const KeyUsages & other ) {
    //for(std::list<KeyUsage>::const_iterator i=other.keyusages.begin(); i!=other.keyusages.end(); i++) {
    for(int i = 0; i<other.Size(); i++) {
      int j = 0;
      for(; j<(*this).Size(); j++) {
        if((*this)[j] == other[i]) break;
      }
      if(j == (*this).Size()) keyusages.push_back(other[i]);
    }
    return *this;
  }

  KeyUsages& KeyUsages::operator+= ( const KeyUsage& value ) {
    //keyusages.push_back(value); return *this;
    int j = 0;
    for(; j<(*this).Size(); j++) {
      if((*this)[j] == value) break;
    }
    if(j == (*this).Size()) keyusages.push_back(value);
    return *this;
  }

  KeyUsages& KeyUsages::Intersection( const KeyUsages & other ) {
    std::list<KeyUsage> tmp_kus;
    for(int i = 0; i<other.Size(); i++) {
      int j = 0;
      for(; j<(*this).Size(); j++) {
        if((*this)[j] == other[i]) break;
      }
      if(j != (*this).Size()) tmp_kus.push_back(other[i]);
    }
    keyusages.clear();
    keyusages = tmp_kus;
    return *this;
  }

  KeyUsages::operator std::string(void) const {
    std::string ret;
    for(int i = 0; i<(*this).Size(); i++) {
      if(i==0) ret.append((*this)[i].id());
      else ret.append(", ").append((*this)[i].id());
    }
    return ret;
  }


  void X509ExtUtil::add_ext2req(X509_EXTENSION* ext) {
    if(!ext) return;
    int ext_nid;
    ext_nid = OBJ_obj2nid(X509_EXTENSION_get_object(ext));

    // The existing extensions
    STACK_OF(X509_EXTENSION)* req_exts = x509_req_get_extensions(req);

    // Create a new stack to include the extensions
    STACK_OF(X509_EXTENSION)* exts = NULL;
    exts = sk_X509_EXTENSION_new_null();

    if(req_exts) {
      int num = sk_X509_EXTENSION_num(req_exts);
      for(int i = 0; i < num; i++) {
        int nid;
        X509_EXTENSION* x509_ext = sk_X509_EXTENSION_value(req_exts, i);
        if(!x509_ext) continue;
        nid = OBJ_obj2nid(X509_EXTENSION_get_object(x509_ext));
        if(nid != ext_nid) {
          // Skip the existing extension with the same oid as the one to add
          X509_EXTENSION* x509_ext_dup = X509_EXTENSION_dup(x509_ext);
          sk_X509_EXTENSION_push(exts, x509_ext_dup);
        }
      }
    }
    // Add the input extension
    sk_X509_EXTENSION_push(exts, ext);
    while(X509_REQ_get_attr_count(req) > 0) X509_REQ_delete_attr(req, 0);
    X509_REQ_add_extensions(req, exts);

    if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
    if(exts) sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
  }

  bool X509ExtUtil::set_subject_key_id(const std::string& sub_kid_conf) {
    bool ret = false;
    X509V3_CTX ctx;
    X509V3_set_ctx_nodb(&ctx);
    X509V3_set_ctx(&ctx, NULL, cert, NULL, NULL, 0);
    X509_EXTENSION* ex = X509V3_EXT_conf_nid(NULL, &ctx, NID_subject_key_identifier, (char*)(sub_kid_conf.c_str()));
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
      else if(req != NULL) { add_ext2req(ex); ret = true; }
    }
    return ret;
  }

  std::string X509ExtUtil::get_subject_key_id() {
    std::string ret;
    int index;
    X509_EXTENSION* ex = NULL;
    STACK_OF(X509_EXTENSION)* req_exts = NULL;
    if(cert) {
      index = X509_get_ext_by_NID(cert, NID_subject_key_identifier, -1);
      if(index != -1) {
        ex = X509_get_ext(cert, index);
      }
    }
    else if(req) {
      req_exts = x509_req_get_extensions(req);
      if(!req_exts) return ret;
      index = X509v3_get_ext_by_NID(req_exts, NID_subject_key_identifier, -1);
      if(index != -1) {
        ex = X509v3_get_ext(req_exts, index);
      }
    }
    else return ret;
    if(!ex) {
      if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
      return ret;
    }

    ASN1_OCTET_STRING* skid = (ASN1_OCTET_STRING *)X509V3_EXT_d2i(ex);
    ret.assign((const char*)ASN1_STRING_data(skid), ASN1_STRING_length(skid));
    ASN1_OCTET_STRING_free(skid);
    if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
    return ret;
  }

  bool X509ExtUtil::set_authority_key_id(const std::string& auth_kid_conf) {
    bool ret = false;
    X509V3_CTX ctx;
    X509V3_set_ctx_nodb(&ctx);
    X509V3_set_ctx(&ctx, NULL, cert, NULL, NULL, 0);
    X509_EXTENSION* ex = X509V3_EXT_conf_nid(NULL, &ctx,
      NID_authority_key_identifier, (char*)(auth_kid_conf.c_str()));
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
      else if(req != NULL) { add_ext2req(ex); ret = true; }
    }
    return ret;
  }

  IssuerId X509ExtUtil::get_authority_key_id() {
    IssuerId ret;
    int index;
    X509_EXTENSION* ex = NULL;
    STACK_OF(X509_EXTENSION)* req_exts = NULL;
    if(cert) {
      index = X509_get_ext_by_NID(cert, NID_authority_key_identifier, -1);
      if(index != -1) {
        ex = X509_get_ext(cert, index);
      }
    }
    else if(req) {
      req_exts = x509_req_get_extensions(req);
      if(!req_exts) return ret;
      index = X509v3_get_ext_by_NID(req_exts, NID_authority_key_identifier, -1);
      if(index != -1) {
        ex = X509v3_get_ext(req_exts, index);
      }
    }
    else return ret;
    if(!ex) {
      if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
      return ret;
    }

    AUTHORITY_KEYID *akid = (AUTHORITY_KEYID *)X509V3_EXT_d2i(ex);
    if(akid->keyid) ret.keyid.assign((const char*)ASN1_STRING_data(akid->keyid),
      ASN1_STRING_length(akid->keyid));
    if(akid->issuer) {
      GENERAL_NAMES* gens;
      GENERAL_NAME* gen;
      X509_NAME* nm = NULL;
      gens = akid->issuer;
      for(int i = 0; i < sk_GENERAL_NAME_num(gens); i++) {
        gen = sk_GENERAL_NAME_value(gens, i);
        if(gen->type == GEN_DIRNAME) {
          nm = gen->d.dirn; break;
        }
      }
      if(nm) {
        char buf[256];
        X509_NAME_oneline(nm,buf,sizeof(buf)-1);
    	buf[sizeof(buf)-1] = 0;
        std::string str(buf);
        ret.issuer.assign(str);
      }
    }
    if(akid->keyid) ret.serial = ASN1_INTEGER_get(akid->serial);

    AUTHORITY_KEYID_free(akid);
    if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
    return ret;
  }

  bool X509ExtUtil::set_subject_alt_name(const std::string& alt_name) {
    bool ret = false;
    X509_EXTENSION* ex = X509V3_EXT_conf_nid(NULL, NULL,
      NID_subject_alt_name, (char*)(alt_name.c_str()));
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
      else if(req != NULL) { add_ext2req(ex); ret = true; }
    }
    return ret;
  }

  void X509ExtUtil::get_subject_alt_name(std::string& alt_name) {

  }

  bool X509ExtUtil::set_basic_constraints(bool ca, int pathlen) {
    bool ret = false;
    BASIC_CONSTRAINTS *bs = BASIC_CONSTRAINTS_new();
    bs->ca = (ca ? 1: 0);
    // According to RFC 5280, section 4.2.1.9
    // A pathLenConstraint of zero indicates that no non-self-issued
    // intermediate CA certificates may follow in a valid certification path.
    // Where it appears, the pathLenConstraint field MUST be greater than or
    // equal to zero. Where pathLenConstraint does not appear, no limit is imposed.
    // if pathlen = -1, then it means it will not absent in the extension
    if(pathlen != -1) {
      bs->pathlen = ASN1_INTEGER_new();
      ASN1_INTEGER_set(bs->pathlen, pathlen);
    }
    X509_EXTENSION* ex = X509V3_EXT_i2d(NID_basic_constraints, 1, bs);
    BASIC_CONSTRAINTS_free(bs);
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
      else if(req != NULL) { add_ext2req(ex); ret = true; }
    }
    return ret;
  }

  void X509ExtUtil::get_basic_constraints(bool* is_ca, int* pathlen) {
    int index;
    X509_EXTENSION* ex = NULL;
    STACK_OF(X509_EXTENSION)* req_exts = NULL;
    if(cert) {
      index = X509_get_ext_by_NID(cert, NID_basic_constraints, -1);
      if(index != -1) {
        ex = X509_get_ext(cert, index);
      }
    }
    else if(req) {
      req_exts = x509_req_get_extensions(req);
      if(!req_exts) return;
      index = X509v3_get_ext_by_NID(req_exts, NID_basic_constraints, -1);
      if(index != -1) {
        ex = X509v3_get_ext(req_exts, index);
      }
    }
    else return;
    if(!ex) {
      if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
      return;
    }

    BASIC_CONSTRAINTS *bs = (BASIC_CONSTRAINTS *)X509V3_EXT_d2i(ex);
    *is_ca = (bs->ca ? true: false);
    if(bs->pathlen) *pathlen = ASN1_INTEGER_get(bs->pathlen);
    else *pathlen = -1;

    BASIC_CONSTRAINTS_free(bs);
    if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
  }

  bool X509ExtUtil::set_key_usage(KeyUsages& keyusages, bool issuer_is_ca) {
    bool ret = false;
    ASN1_BIT_STRING* keyusage = NULL;
    for(int n = 0; n < keyusages.Size(); ++n) {
      int bit = -1;
      switch(keyusages[n].type()) {
        case DigitalSignature: bit = Bit_DigitalSignature; break;
        case NonRepudiation: if(issuer_is_ca) bit = Bit_NonRepudiation; break;
        case KeyEncipherment: bit = Bit_KeyEncipherment; break;
        case DataEncipherment: bit = Bit_DataEncipherment; break;
        case KeyAgreement: bit = Bit_KeyAgreement; break;
        case KeyCertificateSign: if(issuer_is_ca) bit = Bit_KeyCertificateSign; break;
        case CRLSign: if(issuer_is_ca) bit = Bit_CRLSign; break;
        case EncipherOnly: bit = Bit_EncipherOnly; break;
        case DecipherOnly: bit = Bit_DecipherOnly; break;
        default: break;
      }
      if(bit != -1) {
        if(!keyusage)
          keyusage = ASN1_BIT_STRING_new();
        ASN1_BIT_STRING_set_bit(keyusage, bit, 1);
      }
    }
    if(!keyusage) return false;

    X509_EXTENSION* ex = X509V3_EXT_i2d(NID_key_usage, 1, keyusage);
    ASN1_BIT_STRING_free(keyusage);
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
      else if(req != NULL) { add_ext2req(ex); ret = true; }
    }
    return ret;
  }

  KeyUsages X509ExtUtil::get_key_usage() {
    KeyUsages keyusages;
    int index;
    X509_EXTENSION* ex = NULL;
    STACK_OF(X509_EXTENSION)* req_exts = NULL;
    if(cert) {
      index = X509_get_ext_by_NID(cert, NID_key_usage, -1);
      if(index != -1) {
         ex = X509_get_ext(cert, index);
      }
    }
    else if(req) {
      req_exts = x509_req_get_extensions(req);
      if(!req_exts) return keyusages;
      index = X509v3_get_ext_by_NID(req_exts, NID_key_usage, -1);
      if(index != -1) {
        ex = X509v3_get_ext(req_exts, index);
      }
    }
    else return keyusages;
    if(!ex) {
      if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
      return keyusages;
    }

    int bit_table[9] = {
                DigitalSignature,
                NonRepudiation,
                KeyEncipherment,
                DataEncipherment,
                KeyAgreement,
                KeyCertificateSign,
                CRLSign,
                EncipherOnly,
                DecipherOnly
    };
    ASN1_BIT_STRING* keyusage = (ASN1_BIT_STRING *)X509V3_EXT_d2i(ex);
    for(int n = 0; n < 9; ++n) {
      if(ASN1_BIT_STRING_get_bit(keyusage, n))
        keyusages += KeyUsage((KeyUsageType)bit_table[n]);
      }
    ASN1_BIT_STRING_free(keyusage);
    if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
    return keyusages;
  }

  bool X509ExtUtil::set_ext_key_usage(KeyUsages& keyusages) {
    bool ret = false;
    EXTENDED_KEY_USAGE* extkeyusage = NULL;
    for(int n = 0; n < keyusages.Size(); ++n) {
      int nid = -1;
      switch(keyusages[n].type()) {
        case ServerAuth: nid = NID_server_auth; break;
        case ClientAuth: nid = NID_client_auth; break;
        case CodeSigning: nid = NID_code_sign; break;
        case EmailProtection: nid = NID_email_protect; break;
        case IPSecEndSystem: nid = NID_ipsecEndSystem; break;
        case IPSecTunnel: nid = NID_ipsecTunnel; break;
        case IPSecUser: nid = NID_ipsecUser; break;
        case TimeStamping: nid = NID_time_stamp; break;
        case OCSPSigning: nid = NID_OCSP_sign; break;
        default: break;
      }
      if(nid != -1) {
        if(!extkeyusage)
          extkeyusage = sk_ASN1_OBJECT_new_null();
        ASN1_OBJECT *obj = OBJ_nid2obj(nid);
        sk_ASN1_OBJECT_push(extkeyusage, obj);
      }
    }
    if(!extkeyusage) return false;

    X509_EXTENSION* ex = X509V3_EXT_i2d(NID_ext_key_usage, 0, extkeyusage);
    sk_ASN1_OBJECT_pop_free(extkeyusage, ASN1_OBJECT_free);
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
      else if(req != NULL) { add_ext2req(ex); ret = true; }
    }
    return ret;
  }

  KeyUsages X509ExtUtil::get_ext_key_usage() {
    KeyUsages keyusages;
    int index;
    X509_EXTENSION* ex = NULL;
    STACK_OF(X509_EXTENSION)* req_exts = NULL;
    if(cert) {
      index = X509_get_ext_by_NID(cert, NID_ext_key_usage, -1);
      if(index != -1) {
        ex = X509_get_ext(cert, index);
      }
    }
    else if(req) {
      req_exts = x509_req_get_extensions(req);
      // if(!req_exts) keyusages;
      index = X509v3_get_ext_by_NID(req_exts, NID_ext_key_usage, -1);
      if(index != -1) {
        ex = X509v3_get_ext(req_exts, index);
      }
    }
        else return keyusages;
        if(!ex) {
          if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
          return keyusages;
        }

        EXTENDED_KEY_USAGE *extkeyusage = (EXTENDED_KEY_USAGE *)X509V3_EXT_d2i(ex);
        for(int n = 0; n < sk_ASN1_OBJECT_num(extkeyusage); ++n) {
          ASN1_OBJECT *obj = sk_ASN1_OBJECT_value(extkeyusage, n);
          int nid = OBJ_obj2nid(obj);
          if(nid == NID_undef) continue;
          int t = -1;
          switch(nid) {
            case NID_server_auth: t = ServerAuth; break;
            case NID_client_auth: t = ClientAuth; break;
            case NID_code_sign: t = CodeSigning; break;
            case NID_email_protect: t = EmailProtection; break;
            case NID_ipsecEndSystem: t = IPSecEndSystem; break;
            case NID_ipsecTunnel: t = IPSecTunnel; break;
            case NID_ipsecUser: t = IPSecUser; break;
            case NID_time_stamp: t = TimeStamping; break;
            case NID_OCSP_sign: t = OCSPSigning; break;
          };
          if(t == -1) continue;
          keyusages += KeyUsage((KeyUsageType)t);
        }
        sk_ASN1_OBJECT_pop_free(extkeyusage, ASN1_OBJECT_free);
        if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
        return keyusages;
      }

      bool X509ExtUtil::set_cert_policies(const std::vector<std::string>& policies) {
        bool ret = false;
        STACK_OF(POLICYINFO)* pls = NULL;
        for(unsigned int i = 0; i < policies.size(); i++) {
          std::string policy = policies[i];
          ASN1_OBJECT* obj = OBJ_txt2obj(policy.c_str(), 1); //only accept dotted input?
          if(!obj) continue;
          if(!pls) pls = sk_POLICYINFO_new_null();
          POLICYINFO* pol = POLICYINFO_new();
          pol->policyid = obj;
          sk_POLICYINFO_push(pls, pol);
        }
        if(!pls) return false;
        X509_EXTENSION* ex = X509V3_EXT_i2d(NID_certificate_policies, 0, pls);
        sk_POLICYINFO_pop_free(pls, POLICYINFO_free);
        if(ex != NULL) {
          if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); ret = true; }
          else if(req != NULL) { add_ext2req(ex); ret = true; }
        }
        return ret;
      }

      void X509ExtUtil::get_cert_policies(std::vector<std::string>& policies) {
        int index;
        X509_EXTENSION* ex = NULL;
        STACK_OF(X509_EXTENSION)* req_exts = NULL;
        if(cert) {
          index = X509_get_ext_by_NID(cert, NID_certificate_policies, -1);
          if(index != -1) ex = X509_get_ext(cert, index);
        }
        else if(req) {
          req_exts = x509_req_get_extensions(req);
          if(!req_exts) return;
          index = X509v3_get_ext_by_NID(req_exts, NID_certificate_policies, -1);
          if(index != -1) ex = X509v3_get_ext(req_exts, index);
        }
        else return;

        if(ex) {
          STACK_OF(POLICYINFO)* pls = (STACK_OF(POLICYINFO) *)X509V3_EXT_d2i(ex);
          for(int i = 0; i < sk_POLICYINFO_num(pls); i++) {
            POLICYINFO* pol = sk_POLICYINFO_value(pls, i);
            char buf[256];
            int buflen;
            buflen = OBJ_obj2txt(buf, sizeof(buf), pol->policyid, 1); //only accept dotted input
            std::string str(buf, buflen);
            policies.push_back(str);
          }
          sk_POLICYINFO_pop_free(pls, POLICYINFO_free);
        }
        if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
      }

#ifdef HAVE_OPENSSL_PROXY
      Status X509ExtUtil::set_proxy_policy(PROXY_CERT_INFO_EXTENSION* proxy_info) {
        if(!proxy_info) return Status(-1, "PROXY_POLICY is empty");
        Status status = Status(-1);
        if(cert != NULL) {
          if(X509_add1_ext_i2d(cert,NID_proxyCertInfo,proxy_info,1,X509V3_ADD_REPLACE) != 1) {
            logctx->LogFormat(Context::LogError,
              "Failed to add proxy policy into X509: %s" + GetOpenSSLError());
            status = Status(-1,"Failed to add proxy policy into X509: "+GetOpenSSLError());
          }
        }
        else if(req !=NULL) {
          X509_EXTENSION* ex = X509V3_EXT_i2d(NID_proxyCertInfo, 0, proxy_info);
          if(ex != NULL) {
            add_ext2req(ex); status = Status(0);
          }
        }
        return status;
      }
#endif

      Status X509ExtUtil::set_proxy_policy(const std::string& pci_policy,
          const std::string& pci_language, const std::string& pci_pathlen) {
#ifdef HAVE_OPENSSL_PROXY
        //proxy cert info is only supported since openssl version >=097g
        Status status;
        ASN1_OBJECT* pl_obj= NULL;
        ASN1_OCTET_STRING* policy_string = NULL;
        ASN1_INTEGER* policy_pathlen = NULL;

        PROXY_CERT_INFO_EXTENSION proxy_info;
        PROXY_POLICY proxy_policy;
        memset(&proxy_info,0,sizeof(proxy_info));
        memset(&proxy_policy,0,sizeof(proxy_policy));
        proxy_info.pcPathLengthConstraint = NULL;
        proxy_info.proxyPolicy = &proxy_policy;
        proxy_policy.policyLanguage = NULL;
        proxy_policy.policy = NULL;

        if(!pci_policy.empty()) { //anyLanguage
          if(!pci_pathlen.empty()) {
            if(!(policy_pathlen = s2i_ASN1_INTEGER(NULL, (char*)pci_pathlen.c_str()))) goto err;
            proxy_info.pcPathLengthConstraint = policy_pathlen;
          }
          pl_obj = OBJ_nid2obj(NID_id_ppl_anyLanguage);  // Proxy with policy
          if(!pl_obj) {
            logctx->LogFormat(Context::LogError, "Failed to obtain proxy policy object: %s" + GetOpenSSLError());
            status = Status(-1,"Failed to obtain proxy policy object: "+GetOpenSSLError());
            goto err;
          }
          policy_string = ASN1_OCTET_STRING_new();
          if(!policy_string) {
            logctx->LogFormat(Context::LogError, "Failed to allocate policy: %s" + GetOpenSSLError());
            status = Status(-1,"Failed to allocate policy: "+GetOpenSSLError());
            goto err;
          }
          ASN1_OCTET_STRING_set(policy_string,
            (const unsigned char*)(pci_policy.c_str()),pci_policy.length());
          proxy_policy.policyLanguage = pl_obj;
          proxy_policy.policy = policy_string;
          if(!set_proxy_policy(&proxy_info)) goto err;
        }
        else if(!pci_language.empty()) { //inheritAll and independent
          if(!pci_pathlen.empty()) {
            if(!(policy_pathlen = s2i_ASN1_INTEGER(NULL, (char*)pci_pathlen.c_str()))) goto err;
            proxy_info.pcPathLengthConstraint = policy_pathlen;
          }
          pl_obj = OBJ_txt2obj(pci_language.c_str(),0);
          if(!pl_obj) goto err;
          proxy_policy.policyLanguage = pl_obj;
          if(!set_proxy_policy(&proxy_info)) goto err;
        }
        else {
          pl_obj = OBJ_nid2obj(NID_id_ppl_inheritAll);  // Unrestricted proxy
          if(!pl_obj) goto err;
          proxy_policy.policyLanguage = pl_obj;
          if(!set_proxy_policy(&proxy_info)) goto err;
        }

        status = Status(0);

        //TODO : Add into X509 request
      err:
        if(pl_obj) ASN1_OBJECT_free(pl_obj);
        if(policy_string) ASN1_OCTET_STRING_free(policy_string);
        if(policy_pathlen) ASN1_INTEGER_free(policy_pathlen);

        return status;
#else
        return Status(-1, "proxy cert info is not supported by openssl");
#endif
      }

      Status X509ExtUtil::get_proxy_policy(Credentials::Extension& policy) const {
#ifdef HAVE_OPENSSL_PROXY
        if(cert) {
          int pos = X509_get_ext_by_NID((X509*)cert, NID_proxyCertInfo, -1);
          if(pos < 0) {
            logctx->Log(Context::LogError, "Failed to get proxy cert info extension from X509 cert");
            return Status(-1,"Failed to get proxy cert info extension from X509 cert");
          }
          X509_EXTENSION* ex = X509_get_ext((X509*)cert, pos);
          PROXY_CERT_INFO_EXTENSION* pci = (PROXY_CERT_INFO_EXTENSION *)X509V3_EXT_d2i(ex);
          if(!pci) {
            logctx->Log(Context::LogError, "Failed to parse proxy cert info extension");
            return Status(-1,"Failed to parse proxy cert info extension");
          }
          if(!(pci->proxyPolicy)) {
            logctx->Log(Context::LogError, "Failed to retrieve proxy policy");
            return Status(-1,"Failed to retrieve proxy policy");
          }
          if(!(pci->proxyPolicy->policyLanguage)) {
            logctx->Log(Context::LogError, "Failed to retrieve proxy policy language");
            return Status(-1,"Failed to retrieve proxy policy language");
          }
          std::string oid;
          char buf[256];
          int buflen = OBJ_obj2txt(buf, sizeof(buf), pci->proxyPolicy->policyLanguage, 0);
          if(buflen < 0) {
            logctx->Log(Context::LogError, "Failed to convert proxy policy language");
            return Status(-1,"Failed to convert proxy policy language");
          }
          policy.oid.assign(buf,buflen);
          policy.critical = X509_EXTENSION_get_critical(ex)?true:false;
          if(pci->proxyPolicy->policy && pci->proxyPolicy->policy->data) {
            policy.value.assign((const char*)(pci->proxyPolicy->policy->data),
              pci->proxyPolicy->policy->length);
          }
          return Status(0);
        }
        else if(req) {
          STACK_OF(X509_EXTENSION)* req_exts = NULL;
          req_exts = x509_req_get_extensions(req);
          int num = sk_X509_EXTENSION_num(req_exts);
          for(int i = 0; i < num; i++) {
            int nid;
            X509_EXTENSION* x509_ext = sk_X509_EXTENSION_value(req_exts, i);
            if(!x509_ext) continue; // or fail?
            nid = OBJ_obj2nid(X509_EXTENSION_get_object(x509_ext));
            if(nid == NID_proxyCertInfo) {
              std::string oid;
              std::string ext_val;
              if(!retrieve_extension(*logctx, x509_ext, policy)) {
                logctx->Log(Context::LogError, "Failed to retrieve policy extension");
                return Status(-1,"Failed to retrieve policy extension");
              }
              return Status(0);
            }
          }
          if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
        }
        return Status(-1, "Failed to get proxy policy from certificate extension");
#else
        return Status(-1, "proxy cert info is not supported by openssl");
#endif
      }

      // get the extension in a general way, do not understand the structure of extension
      // only get back binary in string
      Status X509ExtUtil::get_extension(int pos, Credentials::Extension& ext) const {
        X509_EXTENSION* ex = NULL;
        if(cert) {
          int num = X509_get_ext_count(cert);
          if ((num > 0) && (pos < num)) {
            ex = X509_get_ext(cert, pos);
            if(!retrieve_extension(*logctx, ex, ext))
              return Status(-1, "Failed to retrieve the extension");
          }
          else if(pos >=num) {
            logctx->LogFormat(Context::LogError,
                   "The ext position %d exceeds the ext number %d", pos, num);
            return Status(-1, "The ext position exceeds the ext number");
          }
        }
        else if(req) {
          STACK_OF(X509_EXTENSION)* req_exts = NULL;
          req_exts = x509_req_get_extensions(req);
          int num = sk_X509_EXTENSION_num(req_exts);
          if((num > 0) && (pos < num) && (pos >= 0)) {
            ex = sk_X509_EXTENSION_value(req_exts, pos);
            if(!ex) {
              if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
              return Status(-1,"Failed to obtain requested extension: "+GetOpenSSLError());
            }
            if(!retrieve_extension(*logctx, ex, ext)) {
              if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
              return Status(-1,"Failed to retrieve requested extension");
            }
          }
          else if(pos >=num) {
            if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
            return Status(-1,"There is no extension at requested position");
          }

        }
        else return Status(-1, "There is no extension exists");

        return Status(0);
      }

      static std::string get_obj_txt(X509_EXTENSION* ext) {
        ASN1_OBJECT* obj;
        obj = X509_EXTENSION_get_object(ext);
        char buf[256];
        int buflen;
        buflen = OBJ_obj2txt(buf, sizeof(buf), obj, 0);
        std::string str = std::string(buf, buflen);
        return str;
      }

      Status X509ExtUtil::get_extension(const std::string& name, Credentials::Extension& ext) const {
        std::string oid;
        std::string ext_val;
        X509_EXTENSION* ex = NULL;

        if(cert != NULL) {
          int nid;
          nid = OBJ_txt2nid(name.c_str());
          if(nid == NID_undef) nid = OBJ_sn2nid(name.c_str());
          if(nid == NID_undef) nid = OBJ_ln2nid(name.c_str());
          if(nid != NID_undef) {
            int pos = X509_get_ext_by_NID(cert, nid, -1);
            if (pos != -1) {
              ex = X509_get_ext(cert, pos);
              if(!retrieve_extension(*logctx, ex, ext))
                return Status(-1, "Failed to retrieve the extension");
            }
          }
          else { // If the 'name' argument is an oid, and this oid is not registered in openssl
            int num = X509_get_ext_count(cert);
            if(num == 0)return Status(-1, "X509 certificate does not include any extension");
            int i = 0;
            for(i = 0; i < num; i++) {
              ex = X509_get_ext(cert, i);
              std::string str = get_obj_txt(ex);
              if(str == name) {
                if(!retrieve_extension(*logctx, ex, ext))
                  return Status(-1, "Failed to retrieve the extension");
                break;
              }
            }
            if(i == num) return Status(-1, "Failed to find extension with the name "+ name);
          }
        }
        else if(req) {
          STACK_OF(X509_EXTENSION)* req_exts = NULL;
          req_exts = x509_req_get_extensions(req);
          int num = sk_X509_EXTENSION_num(req_exts);
          if(num == 0) return Status(-1, "X509 request does not include any extension");
          int i = 0;
          for(i = 0; i < num; i++) {
            ex = sk_X509_EXTENSION_value(req_exts, i);
            if(!ex) continue; // or fail?


            //bool critical; std::string oid, value;
            //parse_extension(ex, critical, oid, value);

            const char* extname_sn = OBJ_nid2sn(OBJ_obj2nid(X509_EXTENSION_get_object(ex)));
            const char* extname_txt = NULL;
            char extname_txt_string[80];
            memset(extname_txt_string, 0, 80);
            OBJ_obj2txt(extname_txt_string, sizeof extname_txt_string, X509_EXTENSION_get_object(ex), 1);
            extname_txt = extname_txt_string;
            const char* extname_ln = OBJ_nid2ln(OBJ_obj2nid(X509_EXTENSION_get_object(ex)));

            if ((name == extname_sn) || (name == extname_txt) || (name == extname_ln)) {
              if(!retrieve_extension(*logctx, ex, ext)) {
                if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
            	return Status(-1,"Failed to retrieve extension of X509 request");
              }
              break;
            }
          }
          if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
          if(i == num) return Status(-1, "Failed to find extension with the name " + name);
        }
        return Status(0);
      }

      // set the extension in a general way, the input extension
      // value should be in asn1 format
      Status X509ExtUtil::set_extension(Credentials::Extension& ext) {
        if(ext.oid.empty()) {
          return (Status(-1,"New extension is missing OID"));
        }

        X509_EXTENSION* x509_ext = NULL;
        if(!ext.value.empty()) {
          int nid;
          nid = OBJ_txt2nid((char*)(ext.oid.c_str()));
          if(nid == NID_undef) nid = OBJ_sn2nid((char*)(ext.oid.c_str()));
          if(nid == NID_undef) nid = OBJ_ln2nid((char*)(ext.oid.c_str()));
          if(nid != NID_undef) {
            ASN1_OCTET_STRING *ext_oct;
            ext_oct = M_ASN1_OCTET_STRING_new();
            ext_oct->data = (unsigned char*)(ext.value.c_str());
            ext_oct->length = ext.value.length();

            x509_ext = X509_EXTENSION_create_by_NID(NULL, nid, ext.critical? 1:0, ext_oct);
            ext_oct->data = NULL;
            M_ASN1_OCTET_STRING_free(ext_oct);

            if(!x509_ext)
              return (Status(-1,"Failed to create extensions: "+GetOpenSSLError()));
          }
          else return Status(-1,"Failed to find extension name: "+ext.oid);
        }
        else return (Status(-1,"Extension string content is empty"));

        if(cert != NULL) {
          X509_add_ext(cert, x509_ext, -1);
          X509_EXTENSION_free(x509_ext);
        }
        else if(req != NULL) {
          add_ext2req(x509_ext);
          // No need to delete x509_ext because it will
          // be deleted inside add_ext2req method.
        }
        return Status(0);
      }

      static void tokenize(const std::string& source,
          std::list<std::string>& dest, const std::string& seperator) {
        std::string::size_type pos1 = 0, pos2;
        while(true) {
          pos2 = source.find(seperator);
          if(pos2 == std::string::npos) break;
          std::string str = source.substr(pos1, pos2-pos1);
          trim_blank(str);
          dest.push_back(str);
          pos1 = pos2+1;
        }
        // the only one, or the last one
        std::string str = source.substr(pos1);
        trim_blank(str);
        dest.push_back(str);
      }

  static X509_EXTENSION* do_ext_i2d(const X509V3_EXT_METHOD* method,
      int ext_nid, int crit, void *ext_struc) {
    unsigned char* ext_der;
    int ext_len;
    ASN1_OCTET_STRING* ext_oct;
    X509_EXTENSION* ext;
    // Convert internal representation to DER
    if (method->it) {
      ext_der = NULL;
      ext_len = ASN1_item_i2d((ASN1_VALUE*)ext_struc, &ext_der, ASN1_ITEM_ptr(method->it));
      if (ext_len < 0) return NULL;
    }
    else {
      unsigned char *p;
      ext_len = method->i2d(ext_struc, NULL);
      if(!(ext_der = (unsigned char*)OPENSSL_malloc(ext_len))) return NULL;
      p = ext_der;
      method->i2d(ext_struc, &p);
    }
    if (!(ext_oct = M_ASN1_OCTET_STRING_new())) return NULL;
    ext_oct->data = ext_der;
    ext_oct->length = ext_len;
    ext = X509_EXTENSION_create_by_NID(NULL, ext_nid, crit, ext_oct);
    if (!ext) return NULL;
    M_ASN1_OCTET_STRING_free(ext_oct);

    return ext;
  }

  static X509_EXTENSION* create_ext_from_value(int ext_nid, int crit, char* value) {
#if (OPENSSL_VERSION_NUMBER >= 0x10000000L)
    const X509V3_EXT_METHOD* method;
#else
    X509V3_EXT_METHOD* method;
#endif
    X509_EXTENSION* ext = NULL;
    STACK_OF(CONF_VALUE)* nval = NULL;
    void* ext_struc = NULL;

    X509V3_CTX ctx;
    X509V3_set_ctx_nodb(&ctx);

    if (ext_nid == NID_undef) return NULL;
    if (!(method = X509V3_EXT_get_nid(ext_nid))) return NULL;
    if (method->v2i) {
      nval = X509V3_parse_list(value);
      if(sk_CONF_VALUE_num(nval) <= 0) return NULL;
      ext_struc = method->v2i(method, &ctx, nval);
      sk_CONF_VALUE_pop_free(nval, X509V3_conf_free);
      if(!ext_struc) return NULL;
    }
    else if(method->s2i) {
      if(!(ext_struc = method->s2i(method, &ctx, value))) return NULL;
    }
    else if(method->r2i) {
      if(!ctx.db || !ctx.db_meth) return NULL;
      if(!(ext_struc = method->r2i(method, &ctx, value))) return NULL;
    }
    else return NULL;

    ext  = do_ext_i2d(method, ext_nid, crit, ext_struc);
    if(method->it) ASN1_item_free((ASN1_VALUE*)ext_struc, ASN1_ITEM_ptr(method->it));
    else method->ext_free(ext_struc);
    return ext;
  }

  // set the extension correspondingly according to the oid,
  // the attributes should be in the format as that in openssl.conf
  Status X509ExtUtil::set_attributes(const std::string& name, const std::string& attrs_value) {
    Status stat;
    int nid;
    nid = OBJ_txt2nid(name.c_str());
    if(nid == NID_undef) nid = OBJ_sn2nid(name.c_str());
    if(nid == NID_undef) nid = OBJ_ln2nid(name.c_str());
    if(nid == NID_undef) return Status(-1, "The input extension name is not recognizable");

    X509_EXTENSION* ex = X509V3_EXT_conf_nid(NULL, NULL,
        nid, (char*)(attrs_value.c_str()));
    if(ex != NULL) {
      if(cert != NULL) { X509_add_ext(cert, ex, -1); X509_EXTENSION_free(ex); stat = Status(0); }
      else if(req != NULL) { add_ext2req(ex); stat = Status(0); } 
    }
    return stat;
  }

  // set the extension correspondingly according to the oid,
  // the attribute value should be in plain text format
  Status X509ExtUtil::set_attributes(const std::string& name, std::list<std::string>& attrs) {
    // parse the name
    /* The format of the input "name" should be the oid of the extension
     * or the name of the extension (e.g. X509v3 Basic Constraints)
     * The format of the input attribute should be the printed values
     * (the extention values printed by openssl's X509V3_EXT_print method,
     * i.e., the extension values output by "openssl -in ./cert.pem -text".
     * It is kind of problematic here, because the output of openssl and
     * nss's certutil does not give output with the same format, so the
     * input here is openssl specific. )
     *
     *	X509v3 Basic Constraints: critical
     * 		CA:FALSE
     * 	X509v3 Key Usage
     * 		Digital Signature, Key Encipherment
     * 	X509v3 Subject Key Identifier
     * 		81:00:9B:CF:97:07:0D:15:87:83:3A:E1:DC:91:12:1A:8C:C0:CC:73
     * 	X509v3 Authority Key Identifier
     * 		keyid:18:05:C0:FC:0B:D1:B7:3A:F4:65:92:09:FB:59:A1:5F:C7:88:C4:F0
     * 		DirName:/O=Grid/O=NorduGrid/CN=NorduGrid Certification Authority
     *      serial:00
     * 	X509v3 Subject Alternative Name
     * 		email:testuser@nordugrid.org
     */

    std::string oid_name;
    //bool critical = false;
    std::string::size_type pos = name.find(":");
    if(pos != std::string::npos) {
      oid_name = name.substr(0, pos);
      std::string crit_str = name.substr(pos+1);
      //if(crit_str.find("critical") != std::string::npos) critical = true;
    }
    else oid_name = name;
    trim_blank(oid_name);

    int nid;
    nid = OBJ_txt2nid(oid_name.c_str());
    if(nid == NID_undef) nid = OBJ_sn2nid(oid_name.c_str());
    if(nid == NID_undef) nid = OBJ_ln2nid(oid_name.c_str());
    if(nid == NID_undef) return Status(-1, "The input extension name is not recognizable");
    if(nid == NID_basic_constraints) {
      // only one item should be available in the attribute list
      std::string value = *attrs.begin();
      value = upper_string(value);
      bool ca=false;
      int pathlen=-1;

      pos = value.find("ca:");
      if(pos != std::string::npos)
        if((pos = value.find("true", pos+3)) != std::string::npos) ca = true;
      if(pos == std::string::npos) pos = 0;
      pos = value.find("pathlen:");
      if(pos != std::string::npos) {
        pos = value.find_first_of("0123456789", pos+8);
        std::string pathlen_str = value.substr(pos);
        trim_blank(pathlen_str);
        if(pos != std::string::npos) pathlen = atoi(pathlen_str.c_str());
      }
      set_basic_constraints(ca, pathlen);
    }
    else if(nid == NID_key_usage) {
      std::string ku_str;
      std::string value = *attrs.begin();
      value = lower_string(value);
      std::list<std::string> ku_str_list;
      tokenize(value, ku_str_list, ",");
      KeyUsages keyusages;
      std::list<std::string>::iterator it;
      for(it = ku_str_list.begin(); it != ku_str_list.end(); it++) {
        ku_str = *it; KeyUsage ku(ku_str, false);
        keyusages += ku;
      }
      set_key_usage(keyusages);
    }
    else if(nid == NID_ext_key_usage) {
      std::string ku_str;
      std::string value = *attrs.begin();
      value = lower_string(value);
      std::list<std::string> ku_str_list;
      tokenize(value, ku_str_list, ",");
      KeyUsages keyusages;
      std::list<std::string>::iterator it;
      for(it = ku_str_list.begin(); it != ku_str_list.end(); it++) {
        ku_str = *it; KeyUsage ku(ku_str, false);
        keyusages += ku;
      }
      set_ext_key_usage(keyusages);
    }
    else if(nid == NID_certificate_policies) {
      std::vector<std::string> policies;
      std::string value = *attrs.begin();
      policies.push_back(value);
      set_cert_policies(policies);
    }
    else if(nid == NID_subject_key_identifier) {
      std::string value = *attrs.begin();
      set_subject_key_id(value);
    }
    else if(nid == NID_authority_key_identifier) {

      //set_authority_key_id(value);
    }

    return Status(0);
  }

  Status X509ExtUtil::get_attributes(const std::string& name, std::list<std::string>& attrs) const {
    // 1. Suppose the attribute is a extension, which means the
    // name argument can be matched into an extension name
    std::string ext_val;
    int nid;
    nid = OBJ_txt2nid(name.c_str());
    if(nid == NID_undef) nid = OBJ_sn2nid(name.c_str());
    if(nid == NID_undef) nid = OBJ_ln2nid(name.c_str());
    if(nid == NID_undef) return Status(-1, "Cann't find OID for: "+ name);

    if(cert != NULL) {
      int pos = X509_get_ext_by_NID(cert, nid, -1);
      if(pos != -1) {
        X509_EXTENSION* x509_ext = NULL;
        x509_ext = X509_get_ext(cert, pos);
        bool critical; std::string oid, value;
        if(!parse_extension(x509_ext, critical, oid, value)) {
          return Status(-1, "Failed to parse exention of X509 certificate");
        }
        attrs.push_back(value);
        /*
        std::string attr;
        attr.append((char*)(x509_ext->value->data), x509_ext->value->length);
        attrs.push_back(attr);
        */
        return Status(0);
      }
    }
    else if(req != NULL) {
      STACK_OF(X509_EXTENSION)* req_exts = NULL;
      req_exts = x509_req_get_extensions(req);
      int num = sk_X509_EXTENSION_num(req_exts);
      if(num == 0) return Status(-1, "X509 request does not include any extension");
      int i = 0;
      X509_EXTENSION* ex = NULL;
      for(i = 0; i < num; i++) {
        ex = sk_X509_EXTENSION_value(req_exts, i);
        if(!ex) continue;
        const char* extname_sn = OBJ_nid2sn(OBJ_obj2nid(X509_EXTENSION_get_object(ex)));
        const char* extname_txt = NULL;
        char extname_txt_string[80];
        memset(extname_txt_string, 0, 80);
        OBJ_obj2txt(extname_txt_string, sizeof extname_txt_string, X509_EXTENSION_get_object(ex), 1);
        extname_txt = extname_txt_string;
        const char* extname_ln = OBJ_nid2ln(OBJ_obj2nid(X509_EXTENSION_get_object(ex)));

        if ((name == extname_sn) || (name == extname_txt) || (name == extname_ln)) {
          bool critical; std::string oid, value;
          if(!parse_extension(ex, critical, oid, value)) {
            if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
            return Status(-1,"Failed to parse extension of X509 request");
          }
          attrs.push_back(value);
          break;
        }
      }
      if(req_exts) sk_X509_EXTENSION_pop_free(req_exts, X509_EXTENSION_free);
      if(i == num) return Status(-1, "Failed to find attribute with the name " + name);
      return Status(0);
    }
    return Status(-1); //TODO: some 3-party attributes
  }

  bool retrieve_extension(AuthN::Context& context,
      X509_EXTENSION* x509_ext, AuthN::Credentials::Extension& extension) {
    if(!x509_ext) return false;

    if(X509_EXTENSION_get_critical(x509_ext)) extension.critical = true;

    char buf[256];
    int buflen;
    buflen = OBJ_obj2txt(buf, sizeof(buf), X509_EXTENSION_get_object(x509_ext), 0);
    if(buflen < 0) {
      context.Log(Context::LogError,"Failed to call OBJ_obj2txt to convert obj to txt");
      return false;
    }
    else extension.oid = buf;

    extension.value.clear();
    extension.value.assign((const char*)(x509_ext->value->data),
        x509_ext->value->length);

    return true;
  }

  bool parse_extension(X509_EXTENSION* x509_ext, bool& crit, std::string& oid, std::string& ext_val) {
    char buf[256];
    int buflen;
    buflen = OBJ_obj2txt(buf, sizeof(buf), X509_EXTENSION_get_object(x509_ext), 0);
    if(buflen < 0) {
      //context.Log(Context::LogError,"Failed to call OBJ_obj2txt to convert obj to txt");
      return false;
    }
    else oid = buf;

    if(X509_EXTENSION_get_critical(x509_ext)) crit=true;

    BIO* bio = NULL;
    bio = BIO_new(BIO_s_mem());
    //ASN1_OBJECT* obj;
    //obj = X509_EXTENSION_get_object(x509_ext);
    //i2a_ASN1_OBJECT(bio, obj);
    if(!X509V3_EXT_print(bio, x509_ext, 0, 0)) {
      //If the extension is not recognized by openssl, such as the voms AC extension,
      //we just output the binary itself
      BIO_printf(bio, "%4s", "");
      M_ASN1_OCTET_STRING_print(bio, x509_ext->value);
    }
    //BIO_write(bio, "\n", 1);

    ext_val.clear();
    for(;;) {
      char s[256];
      int l = BIO_read(bio,s,sizeof(s));
      if(l <= 0) break;
      ext_val.append(s,l);
    }
    BIO_free_all(bio);

    return true;
  }

  STACK_OF(X509_EXTENSION)* x509_req_get_extensions(X509_REQ *req) {
    // As an alternative for X509_REQ_get_extensions, because it seems
    // X509_REQ_get_extensions only give one extension in the condition
    // of multiple extensions exists under X509_REQ

    STACK_OF(X509_EXTENSION)* exts;
    STACK_OF(X509_EXTENSION)* ret_exts;
    X509_ATTRIBUTE *attr;
    ASN1_TYPE *ext = NULL;
    const unsigned char *p;
    ret_exts = sk_X509_EXTENSION_new_null();

    for(int i = 0;; i++) {
      attr = X509_REQ_get_attr(req, i);
      if(attr == NULL) break;
      if(attr->single) ext = attr->value.single;
      else if(sk_ASN1_TYPE_num(attr->value.set))
              ext = sk_ASN1_TYPE_value(attr->value.set, 0);
      if(!ext || (ext->type != V_ASN1_SEQUENCE)) continue;
      p = ext->value.sequence->data;
#if OPENSSL_VERSION_NUMBER >= 0x00908000L
      exts = (STACK_OF(X509_EXTENSION) *) ASN1_item_d2i(NULL, (const unsigned char**)(&p), ext->value.sequence->length,
        ASN1_ITEM_rptr(X509_EXTENSIONS));
#else
      exts = (STACK_OF(X509_EXTENSION) *) ASN1_item_d2i(NULL, (unsigned char**)(&p), ext->value.sequence->length,
        ASN1_ITEM_rptr(X509_EXTENSIONS));
#endif

      for(int j = 0; j < sk_X509_EXTENSION_num(exts); j++) {
        X509_EXTENSION* ex = sk_X509_EXTENSION_value(exts, j);
        sk_X509_EXTENSION_push(ret_exts, ex);
      }
    }
    if(sk_X509_EXTENSION_num(ret_exts) == 0) { sk_X509_EXTENSION_pop_free(ret_exts, X509_EXTENSION_free); return NULL; }
    else return ret_exts;
  }

}
}



