// -*- indent-tabs-mode: nil -*-

#include <cppunit/extensions/HelperMacros.h>

#include <string>
#include <fstream>
#include <iostream>
#include <sstream>

#include <sys/stat.h>
#include <sys/wait.h>
#include <spawn.h>
#include <signal.h>
#include <unistd.h>

#include "../canlxx.h"

// This test code will establish the three participants of
// OCSP stapling scenario: ocsp responder, server, client.
// ocsp responder and server are launched through spawn,
// and will be killed after the communication is finished.


std::string trusted_cacertpath = "trusted_certificates";
std::string certpath = "testCA/certs/cert.pem";
std::string keypath = "testCA/certs/key.pem";
std::string host = "localhost";
int port = 29999;

class OCSPStaplingTest
  : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(OCSPStaplingTest);
  CPPUNIT_TEST(TestOCSPStapling);
  CPPUNIT_TEST_SUITE_END();

public:
  OCSPStaplingTest() : cadir("testCA") {};
  void setUp();
  void tearDown();
  void TestOCSPStapling();

private:
  void launch_ocsp_responder();
  void kill_ocsp_responder();
  pid_t ocsp_pid;
  std::string cadir;
};

void OCSPStaplingTest::setUp() {
  // Setup OCSP responder which will accept
  // OCSP request from the server side of the
  // following pairing communication channel
  launch_ocsp_responder();
}

void OCSPStaplingTest::tearDown() {
  // Close the OCSP responder
  kill_ocsp_responder();
}

void OCSPStaplingTest::TestOCSPStapling() {

  // Setup the server which will accept request from client
  pid_t pid;
  char *argv[6];
  char port_str[8];
  sprintf(port_str, "%d", port);
  argv[0] = (char*)"/bin/sh";
  argv[1] = (char*)"../../test_suite/test_ionetwork_server";
  argv[2] = (char*)port_str;
  argv[3] = (char*)trusted_cacertpath.c_str();
  argv[4] = (char*)certpath.c_str();
  argv[5] = (char*)keypath.c_str();

  int i;
  if((i = posix_spawn(&pid, argv[0], NULL, NULL, argv, NULL)) != 0) {
    std::cout<<"posix_spawn for test_ionetwork_server failed"<<std::endl;
    kill_ocsp_responder();
    exit (1);
  }
  else sleep(2); // Probably need to wait for the server to be ready.

  // Create the client and send request to server
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  ctx.SetCredentials(certpath, keypath);
  ctx.SetCAPath(trusted_cacertpath);

  AuthN::IONetwork io(ctx);
  io.SetTimeout(10000);

  AuthN::Credentials cred(ctx);
  io.SetOwnCredentials(cred);

  AuthN::Status st = io.Connect(host, port);
  if(!st) {
    std::cerr<<"Connect failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
    kill_ocsp_responder();
    if(kill(pid, SIGTERM)!=0) { std::cout<<"failed to kill child"<<std::endl; exit(1); }
  }
  else std::cout<<"Connect succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  std::string message = "hello from client with ocsp stapling";
  st = io.Write(message);
  if(!st) { 
    std::cerr<<"Client: Write failed: "<<st.GetCode()<<": "<<st.GetDescription()<<std::endl;
    kill_ocsp_responder();
    if(kill(pid, SIGTERM)!=0) { std::cout<<"failed to kill child"<<std::endl; exit(1); }
  }
  else std::cout<<"Client: Write succeeded"<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  sleep(2);

  std::string str;
  st = io.Read(str);
  if(!st) {
    std::cerr<<"Client: Read failed: "<<st.GetCode()<<std::endl;
    kill_ocsp_responder();
    if(kill(pid, SIGTERM)!=0) { std::cout<<"failed to kill child"<<std::endl; exit(1); }
  }
  else std::cout<<"Client: Read succeeded: "<<str<<std::endl;
  CPPUNIT_ASSERT_EQUAL(st, AuthN::Status(0));

  io.Close();

  // Close the server
  if(kill(pid, SIGTERM)!=0) { std::cout<<"failed to kill child"<<std::endl; exit(1); }
}

void OCSPStaplingTest::launch_ocsp_responder() {
  char *argv[16];
  std::string cafile = cadir + "/cacert.pem";
  std::string cakey = cadir + "/cakey.pem";
  std::string index_file = cadir + "/ocspid.txt";

  argv[0] = (char*)"/usr/bin/openssl";
  argv[1] = (char*)"ocsp";
  argv[2] = (char*)"-port";
  argv[3] = (char*)"8888";
  argv[4] = (char*)"-rsigner";
  argv[5] = (char*)(cafile.c_str());
  argv[6] = (char*)"-rkey";
  argv[7] = (char*)(cakey.c_str());
  argv[8] = (char*)"-ndays";
  argv[9] = (char*)"10";
  argv[10] = (char*)"-CA";
  argv[11] = (char*)(cafile.c_str());
  argv[12] = (char*)"-resp_key_id";
  argv[13] = (char*)"-index";
  argv[14] = (char*)(index_file.c_str());
  argv[15] = NULL;

  std::ofstream f(index_file.c_str(), std::ifstream::trunc);
  f.close();

  if(posix_spawn(&ocsp_pid, argv[0], NULL, NULL, argv, NULL) != 0) {
    std::cout<<"posix_spawn for ocsp responder failed"<<std::endl;
    exit (1);
  }
  else sleep(2); //sleep a little time to wait the openssl ocsp server being launched

}

void OCSPStaplingTest::kill_ocsp_responder() {
  int sig = SIGTERM;
  if(kill(ocsp_pid, sig)!=0) { std::cout<<"failed to kill child"<<std::endl; exit(1); }
}

CPPUNIT_TEST_SUITE_REGISTRATION(OCSPStaplingTest);
