// -*- indent-tabs-mode: nil -*-

#include <cppunit/extensions/HelperMacros.h>

#include <string>
#include <fstream>
#include <iostream>
#include <sys/stat.h>
#include <dirent.h>
#include <cerrno>
#include <sys/wait.h>
#include <spawn.h>
#include <signal.h>

#include "../canlxx.h"
#include "../fileutil.h"
#include "../opensslutil.h"

class opensslutilTest
  : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(opensslutilTest);
  CPPUNIT_TEST(TestCACert);
  CPPUNIT_TEST(TestEECCert);
  CPPUNIT_TEST(TestProxyCert);
  CPPUNIT_TEST_SUITE_END();

public:
  opensslutilTest() : opensslcnf("openssl_opensslutil.cnf"), cadir("testCA_opensslutil"), trusted_cadir("openssl_trusted_certificates"), ca_subject("/O=EU/OU=EMI/CN=TestCA"), eec_subject("/O=EU/OU=EMI/OU=ARC/CN=Test"), ca_hash("d3c6104b") {}
  void setUp();
  void tearDown();
  void TestCACert();
  void TestEECCert();
  void TestProxyCert() {};

private:
  void createCA();
  void createCRL();
  void createEEC();
  void createOpenSSLconf();

private:
  std::string opensslcnf;
  std::string cadir;
  std::string ca_cert_str;
  std::string ca_key_str;
  std::string crl_str;
  std::string eec_cert_str;
  std::string eec_key_str;
  std::string trusted_cadir;
  std::string ca_subject;
  std::string eec_subject;
  std::string ca_hash;
};

void opensslutilTest::TestCACert() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);

  // Create CA key
  AuthN::OpenSSL::KeyContext key(ctx);  
  key.createKey(2048);

  // Options for the certificate
  AuthN::OpenSSL::CertificateOptions opts;
  opts.setSubject("/O=EU/OU=EMI/CN=TestCA");
  opts.setAsCA();
  long serial = 0; //CA serial must be 0, otherwise the openssl verification will try to find the isser of a CA;
  opts.setSerialNumber(serial);
  AuthN::Utils::Time start;
  AuthN::Utils::Time end = start + 3600*12;  
  opts.setValidityPeriod(start, end);  

  // Create CA certiticate
  AuthN::OpenSSL::CertContext cert(ctx);
  // Set the path to openssl configuration
  cert.setOSSLConf(opensslcnf);
  cert.createSelfSigned(opts, key);

  std::string cert_str;
  cert.certToPEM(cert_str);
  std::cout<<cert_str<<std::endl;
  std::string key_str;
  cert.keyToPEM(key_str);
  std::cout<<key_str<<std::endl;
  
  std::string privkey_str;
  std::string passphrase;
  key.privateToPEM(privkey_str);
  std::cout<<privkey_str<<std::endl;
  std::string pubkey_str;
  key.publicToPEM(pubkey_str);
  std::cout<<pubkey_str<<std::endl;
  
}

void opensslutilTest::TestEECCert() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  ctx.SetCAPath(trusted_cadir);

  // Create CA key and certificate  
  createCA();

  //------ EEC signing
  // Create certificate signing request, and sign EEC certificate using CA key

  // Options for the request
  AuthN::OpenSSL::CertificateOptions eec_req_opts;
  eec_req_opts.setSubject(eec_subject);
  eec_req_opts.setAsUser();
 
  // Create Key
  AuthN::OpenSSL::KeyContext eec_key(ctx);
  eec_key.createKey(2048);

  // Create CSR
  AuthN::OpenSSL::CSRContext eec_request(ctx);
  eec_request.createRequest(eec_req_opts, eec_key);  
  std::string csr_str;
  eec_request.csrToPEM(csr_str);

  // Import the CA cert/key
  AuthN::OpenSSL::CertContext ca_cert(ctx);
  ca_cert.certFromPEM(ca_cert_str);
  ca_cert.keyFromPEM(ca_key_str);

  // Set the path to openssl configuration file
  ca_cert.setOSSLConf(opensslcnf);

  // Options for signing the request
  AuthN::OpenSSL::CertificateOptions eec_sign_opts;
  eec_sign_opts.setSubject("/O=EU/OU=EMI/CN=TestEEC");
  eec_sign_opts.setAsUser();

  // Sign the certificate
  AuthN::OpenSSL::CertContext* eec = ca_cert.signRequest(eec_sign_opts, eec_request);
  //std::string eec_cert_str;
  //std::string eec_key_str;
  CPPUNIT_ASSERT(eec != NULL);

  eec->certToPEM(eec_cert_str);
  eec_key.privateToPEM(eec_key_str);
  std::cout<<eec_cert_str<<std::endl;
  std::cout<<"eec key: "<<eec_key_str<<std::endl;
  //delete eec;  

  //------ Proxy signing
  // Create certificate signing request, and sign proxy certificate using EEC key

  // Options for the request
  AuthN::OpenSSL::CertificateOptions proxy_req_opts;
  long serial = 123456;
  proxy_req_opts.setSerialNumber(serial);
  AuthN::Utils::Time start;
  AuthN::Utils::Time end = start + 3600*12;
  proxy_req_opts.setValidityPeriod(start, end);
 
  // Create Key
  AuthN::OpenSSL::KeyContext proxy_key(ctx);
  proxy_key.createKey(2048);

  // Create CSR
  AuthN::OpenSSL::CSRContext proxy_request(ctx);
  proxy_request.createRequest(proxy_req_opts, proxy_key);
  std::string proxy_csr_str;
  proxy_request.csrToPEM(proxy_csr_str);


  // Import the eec cert/key
  AuthN::OpenSSL::CertContext eec_cert(ctx);
  eec_cert.certFromPEM(eec_cert_str);
  eec_cert.keyFromPEM(eec_key_str);
  
  // Set the path to openssl configuration file
  eec_cert.setOSSLConf(opensslcnf);

  // Options for signing the request
  AuthN::OpenSSL::CertificateOptions proxy_sign_opts;

  // Sign the certificate
  AuthN::OpenSSL::CertContext* proxy = eec_cert.signRequest(proxy_sign_opts, proxy_request);

  CPPUNIT_ASSERT(proxy != NULL);

  std::string proxy_cert_str;
  std::string proxy_key_str;
  if(proxy) proxy->certToPEM(proxy_cert_str);
  std::cout<<proxy_cert_str<<std::endl;
  delete proxy;


  std::string crl_file = trusted_cadir + "/" + ca_hash + ".r0";
  remove(crl_file.c_str());
  // Validate the EEC certificate
  // 1. without CRL present
  AuthN::Validator::ValidationMode mode = AuthN::Validator::ValidationCRLIfPresent; //Mandatory;

  AuthN::Status stat = eec->validate(mode);
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(0));

  // 2. with CRL file present
  createCRL();
  stat = eec->validate(mode); // validation checking should fail
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(-1));

  // 3. with CRLContext object present
  std::vector<AuthN::OpenSSL::CertContext*> trusted;
  std::vector<AuthN::OpenSSL::CertContext*> untrusted;
  std::vector<AuthN::OpenSSL::CRLContext*> crls;
  
  AuthN::OpenSSL::CRLContext crl_ctx(&ctx, crl_file.c_str());
  crls.push_back(&crl_ctx); // the crl_file will not be used if the CRLContext list is given
  stat = eec->validate(mode, trusted, untrusted, crls); // validation checking should fail
  CPPUNIT_ASSERT_EQUAL(stat, AuthN::Status(-1));

}

void opensslutilTest::setUp() {
  AuthN::Utils::File::makedir(cadir.c_str());
  AuthN::Utils::File::makedir(trusted_cadir.c_str());
  std::string certs = cadir + "/certs";
  AuthN::Utils::File::makedir(certs.c_str());
  std::string crl = cadir + "/crl";
  AuthN::Utils::File::makedir(crl.c_str());
  std::string serial = cadir + "/serial";
  std::ofstream s_f(serial.c_str(), std::ifstream::trunc);
  s_f << "00"; s_f.close();
  std::string index = cadir + "/index.txt";
  std::ofstream i_f(index.c_str(), std::ifstream::trunc);
  i_f << ""; i_f.close();
  std::string crlnumber = cadir + "/crlnumber";
  std::ofstream c_f(crlnumber.c_str(), std::ifstream::trunc);
  c_f << "00"; c_f.close();

  createOpenSSLconf();

}

void opensslutilTest::tearDown() {

}

void opensslutilTest::createCA() {
  AuthN::Context ctx(AuthN::Context::EmptyContext);

  // Create CA key and certificate
  AuthN::OpenSSL::KeyContext key(ctx);
  key.createKey(2048);

  AuthN::OpenSSL::CertificateOptions ca_opts;
  ca_opts.setSubject(ca_subject);
  ca_opts.setAsCA();
  long serial = 0;//123456;
  ca_opts.setSerialNumber(serial);
  AuthN::Utils::Time start;
  int hours = 24*365;
  ca_opts.setValidityPeriod(start, hours);

  AuthN::OpenSSL::CertContext cert(ctx);
  cert.setOSSLConf(opensslcnf);
  cert.createSelfSigned(ca_opts, key);

  cert.certToPEM(ca_cert_str);
  cert.keyToPEM(ca_key_str);

  std::string trusted_ca_file = trusted_cadir + "/" + ca_hash + ".0";
  std::ofstream trusted_ca_file_stream(trusted_ca_file.c_str(), std::ifstream::trunc);
  trusted_ca_file_stream << ca_cert_str;
  trusted_ca_file_stream.close();

  std::string ca_file = cadir + "/" + "cacert.pem";
  std::ofstream ca_file_stream(ca_file.c_str(), std::ifstream::trunc);
  ca_file_stream << ca_cert_str;
  ca_file_stream.close();

  std::string ca_key_file = cadir + "/" + "cakey.pem";
  std::ofstream ca_key_file_stream(ca_key_file.c_str(), std::ifstream::trunc);
  ca_key_file_stream << ca_key_str;
  ca_key_file_stream.close();
}

void opensslutilTest::createCRL() {
  //Revoke the eec certificate, and set the crl
  int res;
  std::string cmd;
  std::string rev_cert = "./revoked.pem";
  std::ofstream rev_cert_stream(rev_cert.c_str(), std::ifstream::trunc);
  rev_cert_stream << eec_cert_str;
  rev_cert_stream.close();

  // Revoke a certificate
  cmd = "openssl ca -config " + opensslcnf + " -revoke " + rev_cert;
  res = system(cmd.c_str());
  CPPUNIT_ASSERT_EQUAL(res, 0);

  // Generate a CRL, directly put the CRL file into trusted_cadir
  std::string crl_file = trusted_cadir + "/" + ca_hash + ".r0";
  cmd = "openssl ca -config " + opensslcnf + " -gencrl -out " + crl_file;
  res = system(cmd.c_str());
  CPPUNIT_ASSERT_EQUAL(res, 0);

  //Get the content of the crl
  std::ifstream in(crl_file.c_str(), std::ios::in);
  CPPUNIT_ASSERT_EQUAL(!in, false);
  crl_str.clear();
  std::getline<char>(in, crl_str, 0); in.close();
}

void opensslutilTest::createEEC() {

}

void opensslutilTest::createOpenSSLconf() {
  //The configuration file does not include a typical complete information,
  //only the info that is needed in this test
  std::ofstream f(opensslcnf.c_str(), std::ifstream::trunc);
  f << "[ ca ]" << std::endl;
  f << "default_ca  = CA_default" << std::endl;
  f << "[ CA_default ]" << std::endl;
  f << "dir             = ./" << cadir << std::endl;
  f << "certs           = $dir/certs" << std::endl;
  f << "crl_dir         = $dir/crl" << std::endl;
  f << "database        = $dir/index.txt" << std::endl;
  f << "new_certs_dir   = $dir/newcerts" << std::endl;
  f << "certificate     = $dir/cacert.pem" << std::endl;
  f << "serial          = $dir/serial" << std::endl;
  f << "crlnumber       = $dir/crlnumber" << std::endl;
  f << "crl             = $dir/crl.pem" << std::endl;
  f << "private_key     = $dir/cakey.pem" << std::endl;
  //f << "private_key     = $dir/private/cakey.pem" << std::endl;
  //f << "RANDFILE        = $dir/private/.rand" << std::endl;
  f << "x509_extensions = usr_cert" << std::endl;
  f << "default_days    = 365" << std::endl;
  f << "default_crl_days= 30" << std::endl;
  f << "default_md      = sha512" << std::endl;
  f << "[ req ]" << std::endl;
  f << "default_bits    = 1024" << std::endl;
  f << "default_keyfile = privkey.pem" << std::endl;
  f << "distinguished_name  = req_distinguished_name" << std::endl;
  f << "attributes          = req_attributes" << std::endl;
  f << "x509_extensions = v3_ca" << std::endl;
  f << "req_extensions = v3_req" << std::endl;
  f << "[ req_distinguished_name ]" << std::endl;
  f << "O = EU" << std::endl;
  f << "OU = EMI" << std::endl;
  f << "CN = CA" << std::endl;
  f << "[ req_attributes ]" << std::endl;
  f << "challengePassword  = A challenge password" << std::endl;
  f << "[ usr_cert ]" << std::endl;
  f << "basicConstraints=CA:FALSE" << std::endl;
  f << "subjectKeyIdentifier=hash" << std::endl;
  f << "authorityKeyIdentifier=keyid,issuer" << std::endl;
  f << "authorityInfoAccess=OCSP;URI:http://localhost:8888" << std::endl;
  f << "[ v3_req ]" << std::endl;
  f << "basicConstraints = CA:FALSE" << std::endl;
  f << "keyUsage = nonRepudiation, digitalSignature, keyEncipherment" << std::endl;
  f << "[ v3_ca ]" << std::endl;
  f << "basicConstraints=CA:true" << std::endl;
  f << "subjectKeyIdentifier=hash" << std::endl;
  f << "authorityKeyIdentifier=keyid:always,issuer:always" << std::endl;
  f << "[ proxy_cert_ext ]" << std::endl;
  f << "basicConstraints=CA:FALSE" << std::endl;
  f << "subjectKeyIdentifier=hash" << std::endl;
  f << "authorityKeyIdentifier=keyid,issuer:always" << std::endl;
  f << "proxyCertInfo=critical,language:id-ppl-anyLanguage,pathlen:3,policy:foo" << std::endl;
  f.close();

}

CPPUNIT_TEST_SUITE_REGISTRATION(opensslutilTest);
