#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>

#include "canlxx.h"
#include "opensslutil.h"

namespace AuthN {

using namespace AuthN::OpenSSL;

#define X509_NAME_MAXLEN 256

  static std::string toxstring(unsigned int val) {
    std::stringstream ss;
    ss << std::hex << val;
    return ss.str();
  }

  Validator::Validator(const AuthN::Context& ctx) : mode_(Validator::ValidationNone) {
    context_ = &ctx.Copy();
  }

  Validator::Validator(const AuthN::Context& ctx, ValidationMode mode) : mode_(mode) {
    context_ = &ctx.Copy();
  }

  Validator::~Validator(void) {
    if(context_) delete context_;
  }

  Validator::operator bool(void) const { return true; }

  bool Validator::operator!(void) const { return false; }

  Validator& Validator::Copy(void) const { 
    Validator* validator = new Validator(*this);
    validator->SetContext(*context_);
    return *validator;
  }

  Validator::ValidationMode Validator::Mode(void) const { return mode_; }

  void Validator::SetMode(ValidationMode mode) { mode_ = mode; }

  const AuthN::Context& Validator::GetContext(void) const { return *context_; }

  void Validator::SetContext(const AuthN::Context& env) { context_ = &env.Copy(); }

  //We need an explicit checking of the valitidy of crl here, since 
  //we need to get the validity for the validate mode option ValidationCRLIfValid
  static Status verify_crl(Context& context, const std::string& crl_file, const std::string& CApath) {
    Status ret = Status(-1);
    X509_CRL* crl = NULL;
    X509_STORE* verify_store = NULL;
    X509_STORE_CTX ctx;
    X509_OBJECT obj;
    EVP_PKEY* pkey = NULL;
    int i;

    CRLContext crl_ctx(&context, crl_file.c_str());
    crl = crl_ctx.getCRL();
    if(!crl) goto end;

    verify_store = setup_verify(context, "", CApath);
    if(!verify_store) goto end;
    if(!X509_STORE_CTX_init(&ctx, verify_store, NULL, NULL)) {
      X509_STORE_free(verify_store); verify_store = NULL;
      ret = Status(-1, "Failed to initialize X509 store");
      goto end;
    }
    //loadCRLFile(context, crl_file, crl);
    //crl = crl_ctx.getCRL();
    //if(!crl) goto end;
    i = X509_STORE_get_by_subject(&ctx, X509_LU_X509, X509_CRL_get_issuer(crl), &obj);
    if(i <= 0) {
      ret = Status(-1, "Failed to get CRL issuer certificate");
      goto end;
    }
    pkey = X509_get_pubkey(obj.data.x509);
    if(!pkey) {
      ret = Status(-1, "Failed to get CRL issuer public key");
      goto end;
    }
    i = X509_CRL_verify(crl, pkey);
    if(i <= 0) {
      ret = Status(-1, "CRL verification failed");
      // TODO: extract exact error
      goto end;
    }
    ret = Status(0);

    end:
    X509_OBJECT_free_contents(&obj);
    if(pkey) EVP_PKEY_free(pkey);
    //if(crl) X509_CRL_free(crl);
    if(verify_store) {
      X509_STORE_CTX_cleanup(&ctx);
      X509_STORE_free(verify_store);
    }
    return ret;
  }

  Status Validator::Validate(Credentials& cred) {
   AuthN::OpenSSL::CertContext* cert = cred.certctx_;
   if(!cert) {
     context_->Log(Context::LogError, "Failed to get CertContext object from Cedentials");
     return (last_error_ = Status(-1, "Failed to get CertContext object from Credentials"));
   }
   last_error_ = cert->validate(mode_, context_);
   return last_error_;
  }

  Status Validator::GetStatus(void) const {
    return last_error_;
  }

}
