#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import tests
import unittest

# if this fails to import, then the config_manager module is bust
# and this test script will never be reached to be imported
import config_manager

import os
from StringIO import StringIO
import sys

class TestMain(unittest.TestCase):

    def test_imports(self):
        from config_manager import main

class TestCommandArgument(unittest.TestCase):

    def test_imports(self):
        from config_manager import CommandArgument

    def test_rule(self):
        from config_manager import CommandArgument
        self.assertEqual("build|show|update", CommandArgument().rule())

class TestUrlArgument(unittest.TestCase):

    def test_imports(self):
        from config_manager import UrlArgument

    def test_rule(self):
        from config_manager import UrlArgument
        self.assertEqual("url", UrlArgument().rule())


class TestMainProgram(unittest.TestCase):

    def setUp(self):
        self.real_stdout = sys.stdout
        self.real_stderr = sys.stderr
        self.real_stdin = sys.stdin
        self.captured_out = StringIO()
        self.captured_err = StringIO()
        sys.stdout = self.captured_out
        sys.stderr = self.captured_err
        sys.stdin = StringIO("")

    def tearDown(self):
        sys.stdout = self.real_stdout
        sys.stderr = self.real_stderr
        sys.stdin = self.real_stdin
        
    def run_config_manager(self, argv, stdin_content = None):
        from config_manager import main
        if stdin_content is not None:
            sys.stdin = StringIO(stdin_content)
        try:
            return main(argv)
        except SystemExit, e:
            return e.code

    def test_help(self):
        result = self.run_config_manager(["cm.py", "--help"])
        self.assertEqual(result, 0)
        self.assertEqual("usage: cm [options] build|show|update [url]\n"
                         "\n"
                         "options:\n"
                         "  -h, --help  show this help message and exit\n",
                         self.captured_out.getvalue())

    def test_bare_call(self):
        """If no options or arguments are given, print usage."""
        result = self.run_config_manager(["cm.py"])
        self.assertEqual(result, 0)
        self.assertEqual("usage: cm [options] build|show|update [url]\n\n",
                         self.captured_out.getvalue())
    
    def test_show_config(self):
        result = self.run_config_manager(["cm.py", "show"], 
            "./foo arch://foo@example.com/bar--gam--0\n"
            "./foo/bar http://host.example.com/something\n")
        self.assertEqual(result, 0)
        self.assertEqual("./foo\tarch://foo@example.com/bar--gam--0\n"
                         "./foo/bar\thttp://host.example.com/something\n",
                         self.captured_out.getvalue())

    def test_show_config_file(self):
        import tempfile
        config = tempfile.NamedTemporaryFile()
        config.write("./foo arch://foo@example.com/bar--gam--0\n"
                     "./foo/bar http://host.example.com/something\n")
        config.flush()
        result = self.run_config_manager(["cm.py", "show", config.name])
        self.assertEqual(result, 0)
        self.assertEqual("./foo\tarch://foo@example.com/bar--gam--0\n"
                         "./foo/bar\thttp://host.example.com/something\n",
                         self.captured_out.getvalue())

    def test_build_config(self):
        import shutil
        from config_manager.implementations.fake_vcs import fake_builds
        shutil.rmtree("foo", ignore_errors=True)
        shutil.rmtree("bar", ignore_errors=True)
        result = self.run_config_manager(["cm.py", "build"], 
            "./foo fake://foo@example.com/bar--gam--0\n"
            "./bar fake://host.example.com/something\n")
        self.assertEqual(result, 0)
        self.assertEqual("", self.captured_out.getvalue())
        self.assertEqual(fake_builds, [("fake://host.example.com/something",
                                        os.path.join(
                                            os.path.abspath(os.curdir),
                                            "bar")),
                                       ("fake://foo@example.com/bar--gam--0",
                                        os.path.join(
                                            os.path.abspath(os.curdir),
                                            "foo"))])

    def test_update_config(self):
        import shutil
        shutil.rmtree("foo")
        shutil.rmtree("bar")
        result = self.run_config_manager(["cm.py", "build"], 
            "./foo fake://foo@example.com/bar--gam--0\n"
            "./bar fake://host.example.com/something\n")
        self.assertEqual(result, 0)
        self.assertEqual("", self.captured_out.getvalue())
        result = self.run_config_manager(["cm.py", "update"], 
            "./foo fake://foo@example.com/bar--gam--0\n"
            "./bar fake://host.example.com/something\n")
        from config_manager import fake_updates
        self.assertEqual(fake_updates, [("fake://host.example.com/something",
                                         os.path.join(
                                            os.path.abspath(os.curdir),
                                            "./bar")),
                                       ("fake://foo@example.com/bar--gam--0",
                                         os.path.join(
                                            os.path.abspath(os.curdir),
                                            "./foo"))])


def test_suite():
    loader = tests.TestLoader()
    result = loader.loadTestsFromName(__name__)
    return result
