/*
 * Copyright (C) 2007-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define CONFIG_CPU	80386

/*
 * FUNCTION NAMING CONVENTIONS
 *
 *   get_VAR()
 *	returns the value of the variable specified by VAR
 *
 *   set_VAR(value)
 *	assigns value to the variable specified by VAR; immediately
 *
 *   sset_VAR(value)
 *	scheduled set; like set_VAR(value), but needs a commit
 *
 *   commit_VAR(value)
 *	makes a scheduled set persistent
 *
 *   store_VAR()
 *	set_VAR(t0)
 *
 *   load_VAR()
 *	t1 = get_VAR()
 *
 *   load_VAR1_VAR2()
 *	t1 = get_VAR1()
 *	t2 = get_VAR2()
 *
 * VAR
 *
 *   TODO
 *
 * ABBREVIATIONS
 *
 *   TODO
 */

#ifdef STATE
struct {

/**********************************************************************
 * Temporary Registers
 **********************************************************************/

uint32_t t0;	/* Result					*/
uint32_t t1;	/* Operand1					*/
uint32_t t2;	/* Operand2					*/
uint32_t t3;	/* Operand3 (for SHRD, SHLD)			*/
int32_t  tj;	/* Jump offset					*/

/**********************************************************************
 * General-Purpose Data Registers
 **********************************************************************/

uint32_t eax;
uint32_t ecx;
uint32_t edx;
uint32_t ebx;
uint32_t esp;
uint32_t ebp;
uint32_t esi;
uint32_t edi;

/**********************************************************************
 * Status and Control Registers
 **********************************************************************/

uint32_t eip;
uint32_t eip_new;	/* Instruction pointer		*/

/* EFLAGS Register */
#define EFLAG_CF    0	/* Carry Flag				*/
/*		    1      Reserved (set to 1)			*/
#define EFLAG_PF    2	/* Parity Flag				*/
/*		    3      Reserved (set to 0)			*/
#define EFLAG_AF    4	/* Auxiliary Carry Flag			*/
/*		    5      Reserved (set to 0)			*/
#define EFLAG_ZF    6	/* Zero Flag				*/
#define EFLAG_SF    7	/* Sign Flag				*/
#define EFLAG_TF    8	/* Trap Flag				*/
#define EFLAG_IF    9	/* Interrupt Enable Flag		*/
#define EFLAG_DF    10	/* Direction Flag			*/
#define EFLAG_OF    11	/* Overflow Flag			*/
#define EFLAG_IOPL  12	/* I/O Privilege Level			*/
#define EFLAG_NT    14	/* Nested Task				*/
/*		    15     Reserved (set to 0)			*/
#define EFLAG_RF    16	/* Resume Flag				*/
#define EFLAG_VM    17	/* Virtual-8086 Mode			*/
#define EFLAG_AC    18	/* Alignment Check			*/
#define EFLAG_VIF   19	/* Virtual Interrupt Flag		*/
#define EFLAG_VIP   20	/* Virtual Interrupt Pending		*/
#define EFLAG_ID    21	/* ID Flag				*/
/*		    22     Reserved (set to 0)			*/
/*		    ...						*/
/*		    31     Reserved (set to 0)			*/
bool eflag_cf,    eflag_cf_new;	  /* Carry Flag			*/
bool eflag_pf,    eflag_pf_new;	  /* Parity flag		*/
bool eflag_af,    eflag_af_new;	  /* Auxiliary Carry Flag	*/
bool eflag_zf,    eflag_zf_new;	  /* Zero Flag			*/
bool eflag_sf,    eflag_sf_new;	  /* Sign	Flag		*/
bool eflag_tf,    eflag_tf_new;	  /* Trap Flag			*/
bool eflag_if,    eflag_if_new;	  /* Interrupt Enable Flag	*/
bool eflag_df,    eflag_df_new;	  /* Direction Flag		*/
bool eflag_of,    eflag_of_new;	  /* Overflow Flag		*/
bool eflag_iopl,  eflag_iopl_new; /* I/O Privilege Level	*/
bool eflag_nt,    eflag_nt_new;	  /* Nested Task Flag		*/
bool eflag_rf,    eflag_rf_new;	  /* Resume Flag		*/
bool eflag_vm,    eflag_vm_new;	  /* Virtual-8086 Mode		*/

/**********************************************************************
 * Debug Registers
 **********************************************************************/

uint32_t dr0;
uint32_t dr1;
uint32_t dr2;
uint32_t dr3;
/* dr4 does not exists */
/* dr5 does not exists */
uint32_t dr6;
uint32_t dr7;

/**********************************************************************
 * Code- and Data-Segment Types
 **********************************************************************/

#define SEGMENT_DATA_READ_ONLY					0x0
#define SEGMENT_DATA_READ_WRITE					0x2
#define SEGMENT_DATA_READ_ONLY_EXPAND_DOWN			0x4
#define SEGMENT_DATA_READ_WRITE_EXPAND_DOWN			0x6
#define SEGMENT_CODE_EXEC_ONLY					0x8
#define SEGMENT_CODE_EXEC_READ					0xa
#define SEGMENT_CODE_EXEC_ONLY_CONFORMING			0xc
#define SEGMENT_CODE_EXEC_READ_CONFORMING			0xe

/**********************************************************************
 * System-Segment and Gate-Descriptor Types
 **********************************************************************/

/* Reserved */
#define SEGMENT_16BIT_AVAIL_TSS					0x1
#define SEGMENT__LDT						0x2
#define SEGMENT_16BIT_BUSY_TSS					0x3
#define SEGMENT_16BIT_CALL_GATE					0x4
#define SEGMENT_TASK_GATE					0x5
#define SEGMENT_16BIT_INTERRUPT_GATE				0x6
#define SEGMENT_16BIT_TRAP_GATE					0x7
/* Reserved */
#define SEGMENT_32BIT_AVAIL_TSS					0x9
/* Reserved */
#define SEGMENT_32BIT_BUSY_TSS					0xb
#define SEGMENT_32BIT_CALL_GATE					0xc
/* Reserved */
#define SEGMENT_32BIT_INTERRUPT_GATE				0xe
#define SEGMENT_32BIT_TRAP_GATE					0xf


/**********************************************************************
 * Segment Selectors
 * 13 index	Index in the GDT or LDT
 *  1 ti	Table Indicator (0 = GDT; 1 = LDT)
 *  2 rpl	Requested Privilege Level (RPL)
 *
 * Segment Descriptors
 *  4 type	Segment type
 *  1 sflag	Descriptor type (0 = system; 1 = code or data)
 *  2 dpl	Descriptor privilege level
 *  1 pflag	Segment present
 *
 * Data-/Code-Segment Descriptors
 * 32 limit	Segment limit scaled (20 bit for gflag == 0)
 * 32 base	Segment base address
 *  1 dflag:	Default operation size (0 = 16-bit segment; 1 = 32-bit segment)
 *  1 avlflag:	Available for use by system software
 *
 * System-Segment Descriptors
 * 32 limit	Segment limit scaled (20 bit for gflag == 0)
 * 32 base	Segment base address
 *
 * Call-/Trap-/Interrupt-/Task-Gate Descriptors
 * 16 selector	Segment Selectors
 * 32 offset	Offset in Segment
 *  5 paramcount
 **********************************************************************/

/* Segment Selector (filled by NAME_(sset_selector)()) */
uint16_t selector;
uint16_t selector_index;
uint8_t	selector_ti;
uint8_t	selector_rpl;

/* Segment Descriptor (filled by NAME_(sset_descriptor)()) */
uint8_t	descriptor_type;
bool	descriptor_sflag;
uint8_t	descriptor_dpl;
bool	descriptor_pflag;

/* Data-/Code-Segment Descriptor (filled by NAME_(sset_descriptor)()) */
bool	descriptor_segment_dflag;

/* System-Segment Descriptor (filled by NAME_(sset_descriptor)()) */
uint32_t descriptor_system_limit;
uint32_t descriptor_system_base;

/* Call-/Trap-/Interrupt-/Task-Gate Descriptor (filled by NAME_(sset_descriptor)()) */
uint16_t descriptor_gate_selector;
uint32_t descriptor_gate_offset;
uint8_t	descriptor_gate_paramcount;

/**********************************************************************/

/* Code Segment */
uint16_t cs_selector;
uint8_t  cs_type;
bool cs_sflag;
bool cs_pflag;
bool cs_segment_dflag;

uint16_t temp_cs_selector;
uint8_t temp_cs_selector_rpl;
uint8_t  temp_cs_type;
bool temp_cs_sflag;
uint8_t  temp_cs_dpl;
bool temp_cs_pflag;
uint32_t temp_cs_segment_limit;
uint32_t temp_cs_segment_base;
bool temp_cs_segment_dflag;

/* Stack Segment */
uint16_t ss_selector;
uint8_t ss_type;
bool ss_sflag;
uint8_t ss_dpl;
bool ss_pflag;
bool ss_segment_dflag;

/* Data Segment */
uint16_t ds_selector;
uint8_t ds_type;
bool ds_sflag;
uint8_t ds_dpl;
bool ds_pflag;
bool ds_segment_dflag;

/* Data Segment */
uint16_t es_selector;
uint8_t  es_type;
bool es_sflag;
uint8_t  es_dpl;
bool es_pflag;
bool es_segment_dflag;

/* Data Segment */
uint16_t fs_selector;
uint8_t  fs_type;
bool fs_sflag;
uint8_t  fs_dpl;
bool fs_pflag;
bool fs_segment_dflag;

/* Data Segment */
uint16_t gs_selector;
uint8_t  gs_type;
bool gs_sflag;
uint8_t  gs_dpl;
bool gs_pflag;
bool gs_segment_dflag;

/**********************************************************************
 * Memory-management Registers
 **********************************************************************/

/* Local Descriptor Table Register */
uint16_t ldtr_selector;
/*	ldtr_type	= SEGMENT__LDT					*/
/*	ldtr_sflag	= 0						*/
/*	ldtr_dpl	not used					*/
/*	ldtr_pflag	= 1						*/

/* Task Register */
uint16_t tr_selector;
uint8_t  tr_type;
/*	tr_sflag	= 0						*/
/*	tr_dpl		not used					*/
/*	tr_pflag	= 1						*/

/**********************************************************************
 * Instruction / Prefix
 **********************************************************************/

bool prefix_operand_size_override;
bool prefix_address_size_override;

#define SEGMENT_NONE -1
#define SEGMENT_ES 0
#define SEGMENT_CS 1
#define SEGMENT_SS 2
#define SEGMENT_DS 3
#define SEGMENT_FS 4
#define SEGMENT_GS 5
#define SEGMENT_GDT 6
#define SEGMENT_LDT 7
#define SEGMENT_IDT 8
#define SEGMENT_TSS 9
#define SEGMENT_SS_NEW 10
int prefix_segment_override;

#define LR_NONE  -1
#define LR_LOCK  0  /* 0xf0: LOCK prefix				    */
#define LR_REPNZ 1  /* 0xf2: REPNE/REPNZ prefix (for string instructions)   */
#define LR_REPZ  2  /* 0xf3: REP/REPE/REPZ prefix (for string instructions) */
int prefix_lock_repeat;

/**********************************************************************/

uint32_t instruction_opcode;
uint32_t instruction_opcode2;
uint32_t instruction_mod;
uint32_t instruction_reg;
uint32_t instruction_rm;
uint32_t instruction_scale;
uint32_t instruction_index;
uint32_t instruction_base;
uint32_t instruction_displacement;
uint32_t instruction_immediate;

/**********************************************************************
 * Exception/Interrupt/Halt State
 **********************************************************************/

#define EXCEPTION_DE	0  /* Fault: Divide Error			*/
#define EXCEPTION_DB	1  /* Fault/Trap: Debug				*/
#define EXCEPTION_NMI	2  /* Interrupt: NMI Interrupt			*/
#define EXCEPTION_BP	3  /* Trap: Breakpoint				*/
#define EXCEPTION_OF	4  /* Trap: Overflow				*/
#define EXCEPTION_BR	5  /* Fault: BOUND Range Exceeded		*/
#define EXCEPTION_UD	6  /* Fault: Invalid Opcode (Undefined Opcode)	*/
#define EXCEPTION_NM	7  /* Fault: Device Not Available (No Math Coprocessor) */
#define EXCEPTION_DF	8  /* Abort: Double Fault			*/
/*			9     Fault: Coprocessor Segment Overrun (reserved) */
#define EXCEPTION_TS	10 /* Fault: Invalid TSS			*/
#define EXCEPTION_NP	11 /* Fault: Segment Not Present		*/
#define EXCEPTION_SS	12 /* Fault: Stack-Segment Fault		*/
#define EXCEPTION_GP	13 /* Fault: General Protection			*/
#define EXCEPTION_PF	14 /* Fault: Page Fault				*/
/*			15    Intel reserved. Do not use.		*/
#define EXCEPTION_MF	16 /* Fault: Floating-Point Error (Math Fault)	*/
#define EXCEPTION_AC	17 /* Fault: Alignment Check			*/
#define EXCEPTION_MC	18 /* Abort: Machine Check			*/
#define EXCEPTION_XF	19 /* Fault: Streaming SIMD Extensions		*/
/*			20    Intel reserved. Do not use.		*/
/*			...						*/
/*			31    Intel reserved, Do not use.		*/
/*			32    Interrupt: User Defined (Nonreserved)	*/
/*			...						*/
/*			255   Interrupt: User Defined (Nonreserved)	*/
int exception_pending;
int exception_vector;
int exception_error_code;	/* -1 == NONE */
bool exception_is_interrupt;
bool exception_double_page_fault;

bool interrupt_delay;

bool halt_state;

jmp_buf jmp_env;

bool debug_enabled;

} NAME;

uint32_t descriptor_segment_limit;
uint32_t descriptor_segment_base;

uint8_t cs_dpl;
uint32_t cs_segment_limit;
uint32_t cs_segment_base;
uint32_t ss_segment_limit;
uint32_t ss_segment_base;
uint32_t ds_segment_limit;
uint32_t ds_segment_base;
uint32_t es_segment_limit;
uint32_t es_segment_base;
uint32_t fs_segment_limit;
uint32_t fs_segment_base;
uint32_t gs_segment_limit;
uint32_t gs_segment_base;

/* Global Descriptor Table Register */
uint32_t gdtr_limit;
uint32_t gdtr_base;

/* Interrupt Descriptor Table Register */
uint32_t idtr_limit;
uint32_t idtr_base;

uint32_t ldtr_system_limit;
uint32_t ldtr_system_base;
uint32_t tr_system_limit;
uint32_t tr_system_base;

/* Control Register 0:						*/
/* Contains system control flags that control			*/
/* operating mode and states of the processor			*/
#define CR0_PE		0  /* Protection Enable			*/
#define CR0_MP		1  /* Monitor Coprocessor		*/
#define CR0_EM		2  /* Emulation				*/
#define CR0_TS		3  /* Task Switched			*/
#define CR0_ET		4  /* Extension Type			*/
#define CR0_NE		5  /* Numeric Error			*/
/*			      Reserved				*/
#define CR0_WP		16 /* Write Protect			*/
			   /* Reserved				*/
#define CR0_AM		18 /* Alignment Mask			*/
/*			      Reserved				*/
#define CR0_NW		29 /* Not Write-through			*/
#define CR0_CD		30 /* Cache Disable			*/
#define CR0_PG		31 /* Paging				*/
bool cr0_pe;	  /* Protection Enable				*/
bool cr0_mp;	  /* Monitor Coprocessor			*/
bool cr0_em;	  /* Emulation					*/
bool cr0_ts;	  /* Task Switch				*/
bool cr0_et;	  /* Extension Type				*/
bool cr0_pg;	  /* Paging					*/

/* Control Register 1:						*/
/* Reserved...							*/

/* Control Register 2:						*/
/* Contains the page-fault linear address			*/
/* (the linear address that caused a page fault)		*/
uint32_t cr2_pfla; /* Page-Fault Linear Address			*/

/* Control Register 3:						*/
/* Contains the physical address of the base of the		*/
/* page directory and two flags (PCD and PWT)			*/
/*			      Reserved				*/
#define CR3_PWT		3  /* Page-level Writes Transparent	*/
#define CR3_PCD		4  /* Page-level Cache Disable		*/
/*			      Reserved				*/
#define CR3_PDB		12 /* 12-31: Page-Directory Base	*/
uint32_t cr3_pdb;  /* 12-31: Page-Directory Base		*/
#endif /* STATE */

#ifdef BEHAVIOR

/******************************************************************************/

#ifndef DEBUG_CONTROL_FLOW
# define DEBUG_CONTROL_FLOW 0
#endif

#define DEBUG_EXCEPTIONS 0

#define DEBUG_ENABLED 0

#if DEBUG_ENABLED
#define DEBUG(...)							\
	if (cpssp->NAME.debug_enabled) {						\
		fprintf(stderr, __VA_ARGS__);				\
	}

#else
#define DEBUG(...)
#endif

#define DEBUG_GET DEBUG
#define DEBUG_SET DEBUG

/**********************************************************************/

#define ERROR(...)							\
	{								\
		fprintf(stderr, "\n*** ERROR ***\n"			\
			"Line: %d\nFunction: %s\nFile: %s\n",		\
			__LINE__, __FUNCTION__, __FILE__);		\
		fprintf(stderr, __VA_ARGS__);				\
		fixme();						\
	}

#define ERROR_NYI() ERROR("Not yet implemented...\n")

#define DEBUG_ERROR_SWITCH() ERROR("Unhandled switch case...\n");

/**********************************************************************
 * Misc Functions for simple calculations
 **********************************************************************/

static uint32_t
NAME_(limit_scaled)(struct cpssp *cpssp, uint32_t limit, bool gflag)
{
	if (gflag) {
		return (limit << 12) | 0xfff;
	} else {
		return limit;
	}
}

/**********************************************************************
 * Modes of Operation
 **********************************************************************/

static bool
NAME_(real_mode)(struct cpssp *cpssp)
{
	return ! cpssp->cr0_pe;
}

static bool
NAME_(protected_mode)(struct cpssp *cpssp)
{
	return cpssp->cr0_pe && ! cpssp->NAME.eflag_vm;
}

static bool
NAME_(virtual8086_mode)(struct cpssp *cpssp)
{
	return cpssp->cr0_pe && cpssp->NAME.eflag_vm;
}

/**********************************************************************
 * Exception
 **********************************************************************/

static void __attribute__((__noreturn__))
NAME_(exception)(struct cpssp *cpssp, uint8_t vector, int32_t error_code)
{
	cpssp->NAME.eip_new = cpssp->NAME.eip;

	if (cpssp->NAME.exception_double_page_fault) {
		ERROR("Subsequent fault after two page faults...");
	}

	/* PREVIOUS exception was... */
	if (! cpssp->NAME.exception_pending) {
		cpssp->NAME.exception_pending = 1;
		cpssp->NAME.exception_vector = vector;
		cpssp->NAME.exception_error_code = error_code;
		cpssp->NAME.exception_is_interrupt = 0;

	} else switch (cpssp->NAME.exception_vector) {
	case EXCEPTION_DF:
		ERROR("Subsequent fault after double fault...");
		break;
	case EXCEPTION_PF:
		if (vector == EXCEPTION_PF) {
			cpssp->NAME.exception_vector = EXCEPTION_PF;
			cpssp->NAME.exception_error_code = error_code;
			cpssp->NAME.exception_is_interrupt = 0;
			cpssp->NAME.exception_double_page_fault = 1;
		} else {
			cpssp->NAME.exception_vector = EXCEPTION_DF;
			cpssp->NAME.exception_error_code = 0;
			cpssp->NAME.exception_is_interrupt = 0;
		}
		break;
	default:
		/* Double Fault */
		cpssp->NAME.exception_vector = EXCEPTION_DF;
		cpssp->NAME.exception_error_code = 0;
		cpssp->NAME.exception_is_interrupt = 0;
		break;
	}

	longjmp(cpssp->NAME.jmp_env, 1);
}

/**********************************************************************
 * Memory Read/Write Functions
 **********************************************************************/

static void
NAME_(error)(struct cpssp *cpssp, uint8_t seg, bool serr, uint8_t merr)
{
	if (serr) {
		switch (seg) {
		case SEGMENT_TSS:
			NAME_(exception)(cpssp, EXCEPTION_TS, 0);
		case SEGMENT_SS:
		case SEGMENT_SS_NEW:
			NAME_(exception)(cpssp, EXCEPTION_SS, 0);
		default:
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	}
	if (merr) {
		NAME_(exception)(cpssp, EXCEPTION_PF, merr & 0x7);
	}
}

static uint8_t
NAME_(smrb)(struct cpssp *cpssp, uint32_t off, uint8_t seg)
{
	bool serr;
	uint8_t merr;
	uint8_t val = 0;

	CHIP_(smrb)(cpssp, seg, off, &val, &serr, &merr);
	NAME_(error)(cpssp, seg, serr, merr);
	return val;
}

static uint16_t
NAME_(smrw)(struct cpssp *cpssp, uint32_t off, uint8_t seg)
{
	bool serr;
	uint8_t merr;
	uint16_t val = 0;

	CHIP_(smrw)(cpssp, seg, off, &val, &serr, &merr);
	NAME_(error)(cpssp, seg, serr, merr);
	return val;
}

static uint32_t
NAME_(smrd)(struct cpssp *cpssp, uint32_t off, uint8_t seg)
{
	bool serr;
	uint8_t merr;
	uint32_t val = 0;

	CHIP_(smrd)(cpssp, seg, off, &val, &serr, &merr);
	NAME_(error)(cpssp, seg, serr, merr);
	return val;
}

static void
NAME_(smwb)(struct cpssp *cpssp, uint32_t off, uint8_t seg, uint8_t val)
{
	bool serr;
	uint8_t merr;

	CHIP_(smwb)(cpssp, seg, off, val, &serr, &merr);
	NAME_(error)(cpssp, seg, serr, merr);
}

static void
NAME_(smww)(struct cpssp *cpssp, uint32_t off, uint8_t seg, uint16_t val)
{
	bool serr;
	uint8_t merr;

	CHIP_(smww)(cpssp, seg, off, val, &serr, &merr);
	NAME_(error)(cpssp, seg, serr, merr);
}

static void
NAME_(smwd)(struct cpssp *cpssp, uint32_t off, uint8_t seg, uint32_t val)
{
	bool serr;
	uint8_t merr;

	CHIP_(smwd)(cpssp, seg, off, val, &serr, &merr);
	NAME_(error)(cpssp, seg, serr, merr);
}

#if 0 /* FIXME */
/**********************************************************************
 * LOCK and UNLOCK
 **********************************************************************/

static void
lock(void)
{
	/* TODO? */
}

static void
unlock(void)
{
	/* TODO? */
}
#endif

/**********************************************************************
 * EFLAGS Register Accessors
 **********************************************************************/

#define DEFINE_GET_EFLAG(FLAG, flag)					\
static bool								\
NAME_(get_##FLAG)(struct cpssp *cpssp)					\
{									\
	return cpssp->NAME.eflag_##flag;						\
}

DEFINE_GET_EFLAG(CF, cf)
DEFINE_GET_EFLAG(PF, pf)
DEFINE_GET_EFLAG(AF, af)
DEFINE_GET_EFLAG(ZF, zf)
DEFINE_GET_EFLAG(SF, sf)
DEFINE_GET_EFLAG(TF, tf)
DEFINE_GET_EFLAG(IF, if)
DEFINE_GET_EFLAG(DF, df)
DEFINE_GET_EFLAG(OF, of)
DEFINE_GET_EFLAG(IOPL, iopl)
DEFINE_GET_EFLAG(NT, nt)
DEFINE_GET_EFLAG(RF, rf)
DEFINE_GET_EFLAG(VM, vm)

static uint8_t
NAME_(get_EFLAGS8)(struct cpssp *cpssp)
{
	uint8_t eflags;

	eflags = 0;
	eflags |= NAME_(get_CF)(cpssp) << EFLAG_CF;
	eflags |= 1 << 1;
	eflags |= NAME_(get_PF)(cpssp) << EFLAG_PF;
	/* eflags |= 0 << 3; */
	eflags |= NAME_(get_AF)(cpssp) << EFLAG_AF;
	/* eflags |= 0 << 5; */
	eflags |= NAME_(get_ZF)(cpssp) << EFLAG_ZF;
	eflags |= NAME_(get_SF)(cpssp) << EFLAG_SF;

	return eflags;
}

static uint16_t
NAME_(get_EFLAGS16)(struct cpssp *cpssp)
{
	uint16_t eflags;

	eflags = 0;
	eflags |= NAME_(get_EFLAGS8)(cpssp);
	eflags |= NAME_(get_TF)(cpssp) << EFLAG_TF;
	eflags |= NAME_(get_IF)(cpssp) << EFLAG_IF;
	eflags |= NAME_(get_DF)(cpssp) << EFLAG_DF;
	eflags |= NAME_(get_OF)(cpssp) << EFLAG_OF;
	eflags |= NAME_(get_IOPL)(cpssp) << EFLAG_IOPL;
	eflags |= NAME_(get_NT)(cpssp) << EFLAG_NT;
	/* eflags |= 0 << 15; */

	return eflags;
}

static uint32_t
NAME_(get_EFLAGS32)(struct cpssp *cpssp)
{
	uint32_t eflags;

	eflags = 0;
	eflags |= NAME_(get_EFLAGS16)(cpssp);
	eflags |= NAME_(get_RF)(cpssp) << EFLAG_RF;
	eflags |= NAME_(get_VM)(cpssp) << EFLAG_VM;

	return eflags;
}

/******************************************************************************/

#define DEFINE_SSET_EFLAG(FLAG, flag)					\
static void								\
NAME_(sset_##FLAG)(struct cpssp *cpssp, uint8_t value)							\
{									\
	cpssp->NAME.eflag_##flag##_new = !!value;					\
}

DEFINE_SSET_EFLAG(CF, cf)
DEFINE_SSET_EFLAG(PF, pf)
DEFINE_SSET_EFLAG(AF, af)
DEFINE_SSET_EFLAG(ZF, zf)
DEFINE_SSET_EFLAG(SF, sf)
DEFINE_SSET_EFLAG(TF, tf)
DEFINE_SSET_EFLAG(IF, if)
DEFINE_SSET_EFLAG(DF, df)
DEFINE_SSET_EFLAG(OF, of)
DEFINE_SSET_EFLAG(IOPL, iopl)
DEFINE_SSET_EFLAG(NT, nt)
DEFINE_SSET_EFLAG(RF, rf)
DEFINE_SSET_EFLAG(VM, vm)

static void
NAME_(sset_EFLAGS8)(struct cpssp *cpssp, uint8_t eflags)
{
	NAME_(sset_CF)(cpssp, (eflags >> EFLAG_CF) & 1);
	/* Reserved */
	NAME_(sset_PF)(cpssp, (eflags >> EFLAG_PF) & 1);
	/* Reserved */
	NAME_(sset_AF)(cpssp, (eflags >> EFLAG_AF) & 1);
	/* Reserved */
	NAME_(sset_ZF)(cpssp, (eflags >> EFLAG_ZF) & 1);
	NAME_(sset_SF)(cpssp, (eflags >> EFLAG_SF) & 1);
}

static void
NAME_(sset_EFLAGS16)(struct cpssp *cpssp, uint16_t eflags)
{
	NAME_(sset_EFLAGS8)(cpssp, eflags);
	NAME_(sset_TF)(cpssp, (eflags >> EFLAG_TF) & 1);
	NAME_(sset_IF)(cpssp, (eflags >> EFLAG_IF) & 1);
	NAME_(sset_DF)(cpssp, (eflags >> EFLAG_DF) & 1);
	NAME_(sset_OF)(cpssp, (eflags >> EFLAG_OF) & 1);
	NAME_(sset_IOPL)(cpssp, (eflags >> EFLAG_IOPL) & 3);
	NAME_(sset_NT)(cpssp, (eflags >> EFLAG_NT) & 1);
	/* Reserved */
}

static void
NAME_(sset_EFLAGS32)(struct cpssp *cpssp, uint32_t eflags)
{
	NAME_(sset_EFLAGS16)(cpssp, eflags);
	NAME_(sset_RF)(cpssp, (eflags >> EFLAG_RF) & 1);
	NAME_(sset_VM)(cpssp, (eflags >> EFLAG_VM) & 1);
}

/******************************************************************************/

#define DEFINE_COMMIT_EFLAG(FLAG, flag)					\
static void								\
NAME_(commit_##FLAG)(struct cpssp *cpssp)				\
{									\
	cpssp->NAME.eflag_##flag = cpssp->NAME.eflag_##flag##_new;		\
}

DEFINE_COMMIT_EFLAG(CF, cf)
DEFINE_COMMIT_EFLAG(PF, pf)
DEFINE_COMMIT_EFLAG(AF, af)
DEFINE_COMMIT_EFLAG(ZF, zf)
DEFINE_COMMIT_EFLAG(SF, sf)
DEFINE_COMMIT_EFLAG(TF, tf)
DEFINE_COMMIT_EFLAG(IF, if)
DEFINE_COMMIT_EFLAG(DF, df)
DEFINE_COMMIT_EFLAG(OF, of)
DEFINE_COMMIT_EFLAG(IOPL, iopl)
DEFINE_COMMIT_EFLAG(NT, nt)
DEFINE_COMMIT_EFLAG(RF, rf)
DEFINE_COMMIT_EFLAG(VM, vm)

static void
NAME_(commit_EFLAGS8)(struct cpssp *cpssp)
{
	NAME_(commit_CF)(cpssp);
	NAME_(commit_PF)(cpssp);
	NAME_(commit_AF)(cpssp);
	NAME_(commit_ZF)(cpssp);
	NAME_(commit_SF)(cpssp);
}

static void
NAME_(commit_EFLAGS16)(struct cpssp *cpssp)
{
	NAME_(commit_EFLAGS8)(cpssp);
	NAME_(commit_TF)(cpssp);
	NAME_(commit_IF)(cpssp);
	NAME_(commit_DF)(cpssp);
	NAME_(commit_OF)(cpssp);
	NAME_(commit_IOPL)(cpssp);
	NAME_(commit_NT)(cpssp);
}

static void
NAME_(commit_EFLAGS32)(struct cpssp *cpssp)
{
	NAME_(commit_EFLAGS16)(cpssp);
	NAME_(commit_RF)(cpssp);
	NAME_(commit_VM)(cpssp);
}

static void
NAME_(commit_OSZAP)(struct cpssp *cpssp)
{
	NAME_(commit_OF)(cpssp);
	NAME_(commit_SF)(cpssp);
	NAME_(commit_ZF)(cpssp);
	NAME_(commit_AF)(cpssp);
	NAME_(commit_PF)(cpssp);
}

static void
NAME_(commit_OSZAPC)(struct cpssp *cpssp)
{
	NAME_(commit_OF)(cpssp);
	NAME_(commit_SF)(cpssp);
	NAME_(commit_ZF)(cpssp);
	NAME_(commit_AF)(cpssp);
	NAME_(commit_PF)(cpssp);
	NAME_(commit_CF)(cpssp);
}

/******************************************************************************
 * Control Register Accessors
 ******************************************************************************/

#define DEFINE_SET_CR(no, FLAG, flag)					\
static void \
NAME_(set_##FLAG)(struct cpssp *cpssp, bool value) \
{ \
	cpssp->cr##no##_##flag = !!value; \
}

DEFINE_SET_CR(0, PE, pe)
DEFINE_SET_CR(0, MP, mp)
DEFINE_SET_CR(0, EM, em)
DEFINE_SET_CR(0, TS, ts)
DEFINE_SET_CR(0, ET, et)
DEFINE_SET_CR(0, PG, pg)

static void
NAME_(set_PFLA)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->cr2_pfla = value;
}

static void
NAME_(set_PDB)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->cr3_pdb = value & 0xfffff; /* bits 12-31 */
	CHIP_(tlb_flush)(cpssp);
}

static void
NAME_(set_CR0)(struct cpssp *cpssp, uint32_t cr0)
{
	NAME_(set_PE)(cpssp, (cr0 >> CR0_PE) & 1);
	NAME_(set_MP)(cpssp, (cr0 >> CR0_MP) & 1);
	NAME_(set_EM)(cpssp, (cr0 >> CR0_EM) & 1);
	NAME_(set_TS)(cpssp, (cr0 >> CR0_TS) & 1);
	NAME_(set_ET)(cpssp, (cr0 >> CR0_ET) & 1);
	NAME_(set_PG)(cpssp, (cr0 >> CR0_PG) & 1);
}

static void
NAME_(set_CR2)(struct cpssp *cpssp, uint32_t cr2)
{
	NAME_(set_PFLA)(cpssp, cr2);
}

static void
NAME_(set_CR3)(struct cpssp *cpssp, uint32_t cr3)
{
	NAME_(set_PDB)(cpssp, (cr3 >> CR3_PDB) & 0xfffff); /* bits 12-31 */
}

/******************************************************************************/

#define DEFINE_GET_CR(no, FLAG, flag)					\
static bool								\
NAME_(get_##FLAG)(struct cpssp *cpssp)					\
{									\
	return cpssp->cr##no##_##flag;					\
}

DEFINE_GET_CR(0, PE, pe)
DEFINE_GET_CR(0, MP, mp)
DEFINE_GET_CR(0, EM, em)
DEFINE_GET_CR(0, TS, ts)
DEFINE_GET_CR(0, ET, et)
DEFINE_GET_CR(0, PG, pg)

static uint32_t
NAME_(get_PFLA)(struct cpssp *cpssp)
{
	return cpssp->cr2_pfla;
}

static uint32_t
NAME_(get_PDB)(struct cpssp *cpssp)
{
	return cpssp->cr3_pdb;
}

static uint32_t
NAME_(get_CR0)(struct cpssp *cpssp)
{
	uint32_t cr0;

	cr0 = 0;
	cr0 |= NAME_(get_PE)(cpssp) << CR0_PE;
	cr0 |= NAME_(get_MP)(cpssp) << CR0_MP;
	cr0 |= NAME_(get_EM)(cpssp) << CR0_EM;
	cr0 |= NAME_(get_TS)(cpssp) << CR0_TS;
	cr0 |= NAME_(get_ET)(cpssp) << CR0_ET;
	/* Reserved... */
	cr0 |= NAME_(get_PG)(cpssp) << CR0_PG;

	return cr0;
}

static uint32_t
NAME_(get_CR2)(struct cpssp *cpssp)
{
	return NAME_(get_PFLA)(cpssp);
}

static uint32_t
NAME_(get_CR3)(struct cpssp *cpssp)
{
	uint32_t cr3;

	cr3 = 0;
	/* Reserved... */
	cr3 |= NAME_(get_PDB)(cpssp) << CR3_PDB;

	return cr3;
}

/******************************************************************************
 * Effective Operand- and Address-Size Attributes
 *
 * D Flag in Code Segment Descriptor	 0   0   0   0   1   1   1   1
 * Operand-Size Prefix 0x66		 N   N   Y   Y   N   N   Y   Y
 * Address-Size Prefix 0x67		 N   Y   N   Y   N   Y   N   Y
 * Effective Operand Size		16  16  32  32  32  32  16  16
 * Effective Address Size		16  32  16  32  32  16  32  16
 ******************************************************************************/

static bool
NAME_(operand_size_32)(struct cpssp *cpssp)
{
	return cpssp->NAME.cs_segment_dflag ^ cpssp->NAME.prefix_operand_size_override;
}

static bool
NAME_(address_size_32)(struct cpssp *cpssp)
{
	return cpssp->NAME.cs_segment_dflag ^ cpssp->NAME.prefix_address_size_override;
}

static bool
NAME_(stack_size_32)(struct cpssp *cpssp)
{
	return cpssp->NAME.ss_segment_dflag;
}

/******************************************************************************
 * Fetch Immediate
 ******************************************************************************/

static void
NAME_(fetch_immb)(struct cpssp *cpssp)
{
	cpssp->NAME.instruction_immediate = NAME_(smrb)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 1;
}

static void
NAME_(fetch_immw)(struct cpssp *cpssp)
{
	cpssp->NAME.instruction_immediate = NAME_(smrw)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 2;
}

static void
NAME_(fetch_immd)(struct cpssp *cpssp)
{
	cpssp->NAME.instruction_immediate = NAME_(smrd)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 4;
}

static void
NAME_(fetch_immv)(struct cpssp *cpssp, bool size_32)
{
	if (size_32) {
		NAME_(fetch_immd)(cpssp);
	} else {
		NAME_(fetch_immw)(cpssp);
	}
}

static void
NAME_(fetch_ptr)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.instruction_displacement = NAME_(smrd)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
		cpssp->NAME.eip_new += 4;
	} else {
		cpssp->NAME.instruction_displacement = NAME_(smrw)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
		cpssp->NAME.eip_new += 2;
	}

	cpssp->NAME.instruction_immediate = NAME_(smrw)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 2;
}

/******************************************************************************
 * Fetch ModR/M
 *
 * Fetch the ModR/M byte and if required: 1 SIB byte and 1, 2 or 4 bytes
 * displacement.
 *
 * Mod/RM:
 *     7 6   5 4 3   2 1 0
 *     Mod    Reg     R/M
 *
 * SIB:
 *     7 6   5 4 3   2 1 0
 *    Scale  Index   Base
 ******************************************************************************/

static void
NAME_(fetch_modrm)(struct cpssp *cpssp)
{
	uint8_t modrm;
	uint8_t sib;

	modrm = NAME_(smrb)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 1;

	cpssp->NAME.instruction_mod = (modrm >> 6) & 3;
	cpssp->NAME.instruction_reg = (modrm >> 3) & 7;
	cpssp->NAME.instruction_rm = modrm & 7;

	if (NAME_(address_size_32)(cpssp) && cpssp->NAME.instruction_mod != 3 && cpssp->NAME.instruction_rm == 4) {
		sib = NAME_(smrb)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
		cpssp->NAME.eip_new += 1;

		cpssp->NAME.instruction_scale = (sib >> 6) & 3;
		cpssp->NAME.instruction_index = (sib >> 3) & 7;
		cpssp->NAME.instruction_base = sib & 7;
	} else {
		cpssp->NAME.instruction_scale = 0;
		cpssp->NAME.instruction_index = 0;
		cpssp->NAME.instruction_base = 0;
	}

	if (cpssp->NAME.instruction_mod == 1) {
		cpssp->NAME.instruction_displacement = NAME_(smrb)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
		cpssp->NAME.eip_new += 1;
	} else {
		cpssp->NAME.instruction_displacement = 0;
		if (NAME_(address_size_32)(cpssp)) {
			switch (cpssp->NAME.instruction_mod) {
			case 0:
				if ((cpssp->NAME.instruction_rm != 5)
				 && (cpssp->NAME.instruction_base != 5)) {
					break;
				}
			case 2:
				cpssp->NAME.instruction_displacement = NAME_(smrd)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
				cpssp->NAME.eip_new += 4;
				break;
			default:
				break;
			}
		} else {
			switch (cpssp->NAME.instruction_mod) {
			case 0:
				if (cpssp->NAME.instruction_rm != 6) {
					break;
				}
			case 2:
				cpssp->NAME.instruction_displacement = NAME_(smrw)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
				cpssp->NAME.eip_new += 2;
				break;
			default:
				break;
			}
		}
	}
}

/******************************************************************************
 * Get Instruction *
 ******************************************************************************/

static uint32_t
NAME_(get_instruction_mod)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_mod;
}

static uint32_t
NAME_(get_instruction_reg)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_reg;
}

static uint32_t
NAME_(get_instruction_rm)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_rm;
}

static uint32_t
NAME_(get_instruction_scale)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_scale;
}

static uint32_t
NAME_(get_instruction_index)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_index;
}

static uint32_t
NAME_(get_instruction_base)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_base;
}

static uint32_t
NAME_(get_instruction_displacement)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_displacement;
}

static uint32_t
NAME_(get_instruction_immediate)(struct cpssp *cpssp)
{
	return cpssp->NAME.instruction_immediate;
}

/******************************************************************************
 * Get
 ******************************************************************************/

static uint8_t
NAME_(get_CPL)(struct cpssp *cpssp)
{
	return cpssp->cs_dpl;
}

static uint32_t
NAME_(get_CS)(struct cpssp *cpssp)
{
	return cpssp->NAME.cs_selector;
}

static uint32_t
NAME_(get_SS)(struct cpssp *cpssp)
{
	return cpssp->NAME.ss_selector;
}

static uint32_t
NAME_(get_DS)(struct cpssp *cpssp)
{
	return cpssp->NAME.ds_selector;
}

static uint32_t
NAME_(get_ES)(struct cpssp *cpssp)
{
	return cpssp->NAME.es_selector;
}

static uint32_t
NAME_(get_FS)(struct cpssp *cpssp)
{
	return cpssp->NAME.fs_selector;
}

static uint32_t
NAME_(get_GS)(struct cpssp *cpssp)
{
	return cpssp->NAME.gs_selector;
}

/******************************************************************************/

static uint32_t
NAME_(get_AL)(struct cpssp *cpssp)
{
	return cpssp->NAME.eax & 0xff;
}

static uint32_t
NAME_(get_CL)(struct cpssp *cpssp)
{
	return cpssp->NAME.ecx & 0xff;
}

static uint32_t
NAME_(get_DL)(struct cpssp *cpssp)
{
	return cpssp->NAME.edx & 0xff;
}

static uint32_t
NAME_(get_BL)(struct cpssp *cpssp)
{
	return cpssp->NAME.ebx & 0xff;
}

/******************************************************************************/

static uint32_t
NAME_(get_AH)(struct cpssp *cpssp)
{
	return (cpssp->NAME.eax >> 8) & 0xff;
}

static uint32_t
NAME_(get_CH)(struct cpssp *cpssp)
{
	return (cpssp->NAME.ecx >> 8) & 0xff;
}

static uint32_t
NAME_(get_DH)(struct cpssp *cpssp)
{
	return (cpssp->NAME.edx >> 8) & 0xff;
}

static uint32_t
NAME_(get_BH)(struct cpssp *cpssp)
{
	return (cpssp->NAME.ebx >> 8) & 0xff;
}

/******************************************************************************/

static uint32_t
NAME_(get_AX)(struct cpssp *cpssp)
{
	return cpssp->NAME.eax & 0xffff;
}

static uint32_t
NAME_(get_CX)(struct cpssp *cpssp)
{
	return cpssp->NAME.ecx & 0xffff;
}

static uint32_t
NAME_(get_DX)(struct cpssp *cpssp)
{
	return cpssp->NAME.edx & 0xffff;
}

static uint32_t
NAME_(get_BX)(struct cpssp *cpssp)
{
	return cpssp->NAME.ebx & 0xffff;
}

/******************************************************************************/

static uint32_t
NAME_(get_SP)(struct cpssp *cpssp)
{
	return cpssp->NAME.esp & 0xffff;
}

static uint32_t
NAME_(get_BP)(struct cpssp *cpssp)
{
	return cpssp->NAME.ebp & 0xffff;
}

static uint32_t
NAME_(get_SI)(struct cpssp *cpssp)
{
	return cpssp->NAME.esi & 0xffff;
}

static uint32_t
NAME_(get_DI)(struct cpssp *cpssp)
{
	return cpssp->NAME.edi & 0xffff;
}

/******************************************************************************/

static uint32_t
NAME_(get_EAX)(struct cpssp *cpssp)
{
	return cpssp->NAME.eax;
}

static uint32_t
NAME_(get_ECX)(struct cpssp *cpssp)
{
	return cpssp->NAME.ecx;
}

static uint32_t
NAME_(get_EDX)(struct cpssp *cpssp)
{
	return cpssp->NAME.edx;
}

static uint32_t
NAME_(get_EBX)(struct cpssp *cpssp)
{
	return cpssp->NAME.ebx;
}

/******************************************************************************/

static uint32_t
NAME_(get_ESP)(struct cpssp *cpssp)
{
	return cpssp->NAME.esp;
}

static uint32_t
NAME_(get_EBP)(struct cpssp *cpssp)
{
	return cpssp->NAME.ebp;
}

static uint32_t
NAME_(get_ESI)(struct cpssp *cpssp)
{
	return cpssp->NAME.esi;
}

static uint32_t
NAME_(get_EDI)(struct cpssp *cpssp)
{
	return cpssp->NAME.edi;
}

/******************************************************************************/

static uint32_t
NAME_(get_eAX)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_EAX)(cpssp);
	} else {
		return NAME_(get_AX)(cpssp);
	}
}

static uint32_t
NAME_(get_eCX)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_ECX)(cpssp);
	} else {
		return NAME_(get_CX)(cpssp);
	}
}

static uint32_t
NAME_(get_eDX)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_EDX)(cpssp);
	} else {
		return NAME_(get_DX)(cpssp);
	}
}

static uint32_t
NAME_(get_eBX)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_EBX)(cpssp);
	} else {
		return NAME_(get_BX)(cpssp);
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_eSP)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_ESP)(cpssp);
	} else {
		return NAME_(get_SP)(cpssp);
	}
}

static uint32_t
NAME_(get_eBP)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_EBP)(cpssp);
	} else {
		return NAME_(get_BP)(cpssp);
	}
}

static uint32_t
NAME_(get_eSI)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_ESI)(cpssp);
	} else {
		return NAME_(get_SI)(cpssp);
	}
}

static uint32_t
NAME_(get_eDI)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_EDI)(cpssp);
	} else {
		return NAME_(get_DI)(cpssp);
	}
}

/******************************************************************************/

static uint8_t
NAME_(get_segment_overwritten)(struct cpssp *cpssp, uint8_t segment)
{
	if (cpssp->NAME.prefix_segment_override == SEGMENT_NONE) {
		return segment;
	} else {
		return cpssp->NAME.prefix_segment_override;
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_address_sib)(struct cpssp *cpssp)
{
	uint32_t addr;

	addr = 0;

	switch (NAME_(get_instruction_index)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		addr += NAME_(get_EAX)(cpssp);
		break;
	case 1:
		addr += NAME_(get_ECX)(cpssp);
		break;
	case 2:
		addr += NAME_(get_EDX)(cpssp);
		break;
	case 3:
		addr += NAME_(get_EBX)(cpssp);
		break;
	case 4:
		break;
	case 5:
		addr += NAME_(get_EBP)(cpssp);
		break;
	case 6:
		addr += NAME_(get_ESI)(cpssp);
		break;
	case 7:
		addr += NAME_(get_EDI)(cpssp);
		break;
	}

	switch (NAME_(get_instruction_scale)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		break;
	case 1:
		addr *= 2;
		break;
	case 2:
		addr *= 4;
		break;
	case 3:
		addr *= 8;
		break;
	}

	switch (NAME_(get_instruction_base)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		addr += NAME_(get_EAX)(cpssp);
		break;
	case 1:
		addr += NAME_(get_ECX)(cpssp);
		break;
	case 2:
		addr += NAME_(get_EDX)(cpssp);
		break;
	case 3:
		addr += NAME_(get_EBX)(cpssp);
		break;
	case 4:
		addr += NAME_(get_ESP)(cpssp);
		break;
	case 5:
		if (NAME_(get_instruction_mod)(cpssp) == 0) {
			addr += NAME_(get_instruction_displacement)(cpssp);
		} else {
			addr += NAME_(get_EBP)(cpssp);
		}
		break;
	case 6:
		addr += NAME_(get_ESI)(cpssp);
		break;
	case 7:
		addr += NAME_(get_EDI)(cpssp);
		break;
	}

	return addr;
}

static uint8_t
NAME_(get_segment_sib)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_base)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return SEGMENT_DS;
	case 1:
		return SEGMENT_DS;
	case 2:
		return SEGMENT_DS;
	case 3:
		return SEGMENT_DS;
	case 4:
		return SEGMENT_SS;
	case 5:
		if (NAME_(get_instruction_mod)(cpssp) == 0) {
			return SEGMENT_DS;
		} else {
			return SEGMENT_SS;
		}
	case 6:
		return SEGMENT_DS;
	case 7:
		return SEGMENT_DS;
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_address_mod0)(struct cpssp *cpssp)
{
	uint32_t addr;

	addr = 0;

	if (NAME_(address_size_32)(cpssp)) {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			addr += NAME_(get_EAX)(cpssp);
			break;
		case 1:
			addr += NAME_(get_ECX)(cpssp);
			break;
		case 2:
			addr += NAME_(get_EDX)(cpssp);
			break;
		case 3:
			addr += NAME_(get_EBX)(cpssp);
			break;
		case 4:
			addr += NAME_(get_address_sib)(cpssp);
			break;
		case 5:
			addr += NAME_(get_instruction_displacement)(cpssp);
			break;
		case 6:
			addr += NAME_(get_ESI)(cpssp);
			break;
		case 7:
			addr += NAME_(get_EDI)(cpssp);
			break;
		}
	} else {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			addr += NAME_(get_BX)(cpssp) + NAME_(get_SI)(cpssp);
			break;
		case 1:
			addr += NAME_(get_BX)(cpssp) + NAME_(get_DI)(cpssp);
			break;
		case 2:
			addr += NAME_(get_BP)(cpssp) + NAME_(get_SI)(cpssp);
			break;
		case 3:
			addr += NAME_(get_BP)(cpssp) + NAME_(get_DI)(cpssp);
			break;
		case 4:
			addr += NAME_(get_SI)(cpssp);
			break;
		case 5:
			addr += NAME_(get_DI)(cpssp);
			break;
		case 6:
			addr += NAME_(get_instruction_displacement)(cpssp);
			break;
		case 7:
			addr += NAME_(get_BX)(cpssp);
			break;
		}

		addr &= 0xffff;
	}

	return addr;
}

static uint8_t
NAME_(get_segment_mod0)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			return SEGMENT_DS;
		case 1:
			return SEGMENT_DS;
		case 2:
			return SEGMENT_DS;
		case 3:
			return SEGMENT_DS;
		case 4:
			return NAME_(get_segment_sib)(cpssp);
		case 5:
			return SEGMENT_DS;
		case 6:
			return SEGMENT_DS;
		case 7:
			return SEGMENT_DS;
		}
	} else {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			return SEGMENT_DS;
		case 1:
			return SEGMENT_DS;
		case 2:
			return SEGMENT_SS;
		case 3:
			return SEGMENT_SS;
		case 4:
			return SEGMENT_DS;
		case 5:
			return SEGMENT_DS;
		case 6:
			return SEGMENT_DS;
		case 7:
			return SEGMENT_DS;
		}
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_address_mod1)(struct cpssp *cpssp)
{
	uint32_t addr;

	addr = 0;

	addr += (int8_t) NAME_(get_instruction_displacement)(cpssp);

	if (NAME_(address_size_32)(cpssp)) {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			addr += NAME_(get_EAX)(cpssp);
			break;
		case 1:
			addr += NAME_(get_ECX)(cpssp);
			break;
		case 2:
			addr += NAME_(get_EDX)(cpssp);
			break;
		case 3:
			addr += NAME_(get_EBX)(cpssp);
			break;
		case 4:
			addr += NAME_(get_address_sib)(cpssp);
			break;
		case 5:
			addr += NAME_(get_EBP)(cpssp);
			break;
		case 6:
			addr += NAME_(get_ESI)(cpssp);
			break;
		case 7:
			addr += NAME_(get_EDI)(cpssp);
			break;
		}
	} else {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			addr += NAME_(get_BX)(cpssp) + NAME_(get_SI)(cpssp);
			break;
		case 1:
			addr += NAME_(get_BX)(cpssp) + NAME_(get_DI)(cpssp);
			break;
		case 2:
			addr += NAME_(get_BP)(cpssp) + NAME_(get_SI)(cpssp);
			break;
		case 3:
			addr += NAME_(get_BP)(cpssp) + NAME_(get_DI)(cpssp);
			break;
		case 4:
			addr += NAME_(get_SI)(cpssp);
			break;
		case 5:
			addr += NAME_(get_DI)(cpssp);
			break;
		case 6:
			addr += NAME_(get_BP)(cpssp);
			break;
		case 7:
			addr += NAME_(get_BX)(cpssp);
			break;
		}

		addr &= 0xffff;
	}

	return addr;
}

static uint8_t
NAME_(get_segment_mod1)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			return SEGMENT_DS;
		case 1:
			return SEGMENT_DS;
		case 2:
			return SEGMENT_DS;
		case 3:
			return SEGMENT_DS;
		case 4:
			return NAME_(get_segment_sib)(cpssp);
		case 5:
			return SEGMENT_SS;
		case 6:
			return SEGMENT_DS;
		case 7:
			return SEGMENT_DS;
		}
	} else {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			return SEGMENT_DS;
		case 1:
			return SEGMENT_DS;
		case 2:
			return SEGMENT_SS;
		case 3:
			return SEGMENT_SS;
		case 4:
			return SEGMENT_DS;
		case 5:
			return SEGMENT_DS;
		case 6:
			return SEGMENT_SS;
		case 7:
			return SEGMENT_DS;
		}
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_address_mod2)(struct cpssp *cpssp)
{
	uint32_t addr;

	addr = 0;

	addr += NAME_(get_instruction_displacement)(cpssp);

	if (NAME_(address_size_32)(cpssp)) {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			addr += NAME_(get_EAX)(cpssp);
			break;
		case 1:
			addr += NAME_(get_ECX)(cpssp);
			break;
		case 2:
			addr += NAME_(get_EDX)(cpssp);
			break;
		case 3:
			addr += NAME_(get_EBX)(cpssp);
			break;
		case 4:
			addr += NAME_(get_address_sib)(cpssp);
			break;
		case 5:
			addr += NAME_(get_EBP)(cpssp);
			break;
		case 6:
			addr += NAME_(get_ESI)(cpssp);
			break;
		case 7:
			addr += NAME_(get_EDI)(cpssp);
			break;
		}
	} else {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			addr += NAME_(get_BX)(cpssp) + NAME_(get_SI)(cpssp);
			break;
		case 1:
			addr += NAME_(get_BX)(cpssp) + NAME_(get_DI)(cpssp);
			break;
		case 2:
			addr += NAME_(get_BP)(cpssp) + NAME_(get_SI)(cpssp);
			break;
		case 3:
			addr += NAME_(get_BP)(cpssp) + NAME_(get_DI)(cpssp);
			break;
		case 4:
			addr += NAME_(get_SI)(cpssp);
			break;
		case 5:
			addr += NAME_(get_DI)(cpssp);
			break;
		case 6:
			addr += NAME_(get_BP)(cpssp);
			break;
		case 7:
			addr += NAME_(get_BX)(cpssp);
			break;
		}

		addr &= 0xffff;
	}

	return addr;
}

static uint8_t
NAME_(get_segment_mod2)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			return SEGMENT_DS;
		case 1:
			return SEGMENT_DS;
		case 2:
			return SEGMENT_DS;
		case 3:
			return SEGMENT_DS;
		case 4:
			return NAME_(get_segment_sib)(cpssp);
		case 5:
			return SEGMENT_SS;
		case 6:
			return SEGMENT_DS;
		case 7:
			return SEGMENT_DS;
		}
	} else {
		switch (NAME_(get_instruction_rm)(cpssp)) {
		default:
			DEBUG_ERROR_SWITCH();
		case 0:
			return SEGMENT_DS;
		case 1:
			return SEGMENT_DS;
		case 2:
			return SEGMENT_SS;
		case 3:
			return SEGMENT_SS;
		case 4:
			return SEGMENT_DS;
		case 5:
			return SEGMENT_DS;
		case 6:
			return SEGMENT_SS;
		case 7:
			return SEGMENT_DS;
		}
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_DR0)(struct cpssp *cpssp)
{
	return cpssp->NAME.dr0;
}

static uint32_t
NAME_(get_DR1)(struct cpssp *cpssp)
{
	return cpssp->NAME.dr1;
}

static uint32_t
NAME_(get_DR2)(struct cpssp *cpssp)
{
	return cpssp->NAME.dr2;
}

static uint32_t
NAME_(get_DR3)(struct cpssp *cpssp)
{
	return cpssp->NAME.dr3;
}

static uint32_t
NAME_(get_DR6)(struct cpssp *cpssp)
{
	return cpssp->NAME.dr6;
}

static uint32_t
NAME_(get_DR7)(struct cpssp *cpssp)
{
	return cpssp->NAME.dr7;
}

/******************************************************************************/

static uint32_t
NAME_(get_Rb)(struct cpssp *cpssp, uint32_t reg)
{
	switch (reg) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(get_AL)(cpssp);
	case 1:
		return NAME_(get_CL)(cpssp);
	case 2:
		return NAME_(get_DL)(cpssp);
	case 3:
		return NAME_(get_BL)(cpssp);
	case 4:
		return NAME_(get_AH)(cpssp);
	case 5:
		return NAME_(get_CH)(cpssp);
	case 6:
		return NAME_(get_DH)(cpssp);
	case 7:
		return NAME_(get_BH)(cpssp);
	}
}

static uint32_t
NAME_(get_Rw)(struct cpssp *cpssp, uint32_t reg)
{
	switch (reg) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(get_AX)(cpssp);
	case 1:
		return NAME_(get_CX)(cpssp);
	case 2:
		return NAME_(get_DX)(cpssp);
	case 3:
		return NAME_(get_BX)(cpssp);
	case 4:
		return NAME_(get_SP)(cpssp);
	case 5:
		return NAME_(get_BP)(cpssp);
	case 6:
		return NAME_(get_SI)(cpssp);
	case 7:
		return NAME_(get_DI)(cpssp);
	}
}

static uint32_t
NAME_(get_Rd)(struct cpssp *cpssp, uint32_t reg)
{
	switch (reg) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(get_EAX)(cpssp);
	case 1:
		return NAME_(get_ECX)(cpssp);
	case 2:
		return NAME_(get_EDX)(cpssp);
	case 3:
		return NAME_(get_EBX)(cpssp);
	case 4:
		return NAME_(get_ESP)(cpssp);
	case 5:
		return NAME_(get_EBP)(cpssp);
	case 6:
		return NAME_(get_ESI)(cpssp);
	case 7:
		return NAME_(get_EDI)(cpssp);
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_Cd)(struct cpssp *cpssp)
{
	if (NAME_(protected_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(get_CR0)(cpssp);
	case 1:
		ERROR_NYI(); /* FIXME: Reserved */
	case 2:
		return NAME_(get_CR2)(cpssp);
	case 3:
		return NAME_(get_CR3)(cpssp);
	case 4:
	case 5:
	case 6:
	case 7:
		ERROR_NYI(); /* FIXME: Reserved */
		return 0;
	}
}

static uint32_t
NAME_(get_Dd)(struct cpssp *cpssp)
{
	if (NAME_(protected_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(get_DR0)(cpssp);
	case 1:
		return NAME_(get_DR1)(cpssp);
	case 2:
		return NAME_(get_DR2)(cpssp);
	case 3:
		return NAME_(get_DR3)(cpssp);
	case 4:
	case 5:
		return 0; /* "Intel reserved" */
	case 6:
		return NAME_(get_DR6)(cpssp);
	case 7:
		return NAME_(get_DR7)(cpssp);
	}
}

static uint32_t
NAME_(get_Td)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	case 6:
	case 7:
		ERROR_NYI();
		break;
	default:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
	}
	return 0;
}

/******************************************************************************/

static uint32_t
NAME_(get_Gb)(struct cpssp *cpssp)
{
	return NAME_(get_Rb)(cpssp, NAME_(get_instruction_reg)(cpssp));
}

static uint32_t
NAME_(get_Gw)(struct cpssp *cpssp)
{
	return NAME_(get_Rw)(cpssp, NAME_(get_instruction_reg)(cpssp));
}

static uint32_t
NAME_(get_Gd)(struct cpssp *cpssp)
{
	return NAME_(get_Rd)(cpssp, NAME_(get_instruction_reg)(cpssp));
}

static uint32_t
NAME_(get_Gv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_Gd)(cpssp);
	} else {
		return NAME_(get_Gw)(cpssp);
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_Eb)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(smrb)(cpssp, NAME_(get_address_mod0)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)));
	case 1:
		return NAME_(smrb)(cpssp, NAME_(get_address_mod1)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)));
	case 2:
		return NAME_(smrb)(cpssp, NAME_(get_address_mod2)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)));
	case 3:
		return NAME_(get_Rb)(cpssp, NAME_(get_instruction_rm)(cpssp));
	}
}

static uint32_t
NAME_(get_Ew)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(smrw)(cpssp, NAME_(get_address_mod0)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)));
	case 1:
		return NAME_(smrw)(cpssp, NAME_(get_address_mod1)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)));
	case 2:
		return NAME_(smrw)(cpssp, NAME_(get_address_mod2)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)));
	case 3:
		return NAME_(get_Rw)(cpssp, NAME_(get_instruction_rm)(cpssp));
	}
}

static uint32_t
NAME_(get_Ed)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(smrd)(cpssp, NAME_(get_address_mod0)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)));
	case 1:
		return NAME_(smrd)(cpssp, NAME_(get_address_mod1)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)));
	case 2:
		return NAME_(smrd)(cpssp, NAME_(get_address_mod2)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)));
	case 3:
		return NAME_(get_Rd)(cpssp, NAME_(get_instruction_rm)(cpssp));
	}
}

static uint32_t
NAME_(get_Ev)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_Ed)(cpssp);
	} else {
		return NAME_(get_Ew)(cpssp);
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_EwBIT)(struct cpssp *cpssp)
{
	int32_t offset;

	offset = (int16_t) (NAME_(get_Rw)(cpssp, NAME_(get_instruction_reg)(cpssp)) & 0xfff0) / 16;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(smrw)(cpssp, NAME_(get_address_mod0)(cpssp) + 2 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)));
	case 1:
		return NAME_(smrw)(cpssp, NAME_(get_address_mod1)(cpssp) + 2 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)));
	case 2:
		return NAME_(smrw)(cpssp, NAME_(get_address_mod2)(cpssp) + 2 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)));
	case 3:
		return NAME_(get_Rw)(cpssp, NAME_(get_instruction_rm)(cpssp));
	}
}

static uint32_t
NAME_(get_EdBIT)(struct cpssp *cpssp)
{
	int32_t offset;

	offset = (int32_t) (NAME_(get_Rd)(cpssp, NAME_(get_instruction_reg)(cpssp)) & 0xffffffe0) / 32;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(smrd)(cpssp, NAME_(get_address_mod0)(cpssp) + 4 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)));
	case 1:
		return NAME_(smrd)(cpssp, NAME_(get_address_mod1)(cpssp) + 4 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)));
	case 2:
		return NAME_(smrd)(cpssp, NAME_(get_address_mod2)(cpssp) + 4 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)));
	case 3:
		return NAME_(get_Rd)(cpssp, NAME_(get_instruction_rm)(cpssp));
	}
}

static uint32_t
NAME_(get_EvBIT)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_EdBIT)(cpssp);
	} else {
		return NAME_(get_EwBIT)(cpssp);
	}
}

static uint32_t
NAME_(get_effective_address)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		return NAME_(get_address_mod0)(cpssp);
	case 1:
		return NAME_(get_address_mod1)(cpssp);
	case 2:
		return NAME_(get_address_mod2)(cpssp);
	case 3:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_Sw)(struct cpssp *cpssp)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case SEGMENT_ES:
		return NAME_(get_ES)(cpssp);
	case SEGMENT_CS:
		return NAME_(get_CS)(cpssp);
	case SEGMENT_SS:
		return NAME_(get_SS)(cpssp);
	case SEGMENT_DS:
		return NAME_(get_DS)(cpssp);
	case SEGMENT_FS:
		return NAME_(get_FS)(cpssp);
	case SEGMENT_GS:
		return NAME_(get_GS)(cpssp);
	case 6:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	case 7:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}
}

/******************************************************************************
 * Get Immediates
 ******************************************************************************/

static uint32_t
NAME_(get_Ib)(struct cpssp *cpssp)
{
	return NAME_(get_instruction_immediate)(cpssp) & 0xff;
}

static uint32_t
NAME_(get_IbSE)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return (uint32_t) (int8_t) (NAME_(get_instruction_immediate)(cpssp) & 0xff);
	} else {
		return (uint16_t) (int8_t) (NAME_(get_instruction_immediate)(cpssp) & 0xff);
	}
}

static uint32_t
NAME_(get_Iw)(struct cpssp *cpssp)
{
	return NAME_(get_instruction_immediate)(cpssp) & 0xffff;
}

static uint32_t
NAME_(get_Id)(struct cpssp *cpssp)
{
	return NAME_(get_instruction_immediate)(cpssp);
}

static uint32_t
NAME_(get_Iv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_Id)(cpssp);
	} else {
		return NAME_(get_Iw)(cpssp);
	}
}

/******************************************************************************/

static int32_t
NAME_(get_Jb)(struct cpssp *cpssp)
{
	return (int8_t) NAME_(get_instruction_immediate)(cpssp);
}

static int32_t
NAME_(get_Jw)(struct cpssp *cpssp)
{
	return (int16_t) NAME_(get_instruction_immediate)(cpssp);
}

static int32_t
NAME_(get_Jd)(struct cpssp *cpssp)
{
	return (int32_t) NAME_(get_instruction_immediate)(cpssp);
}

static int32_t
NAME_(get_Jv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_Jd)(cpssp);
	} else {
		return NAME_(get_Jw)(cpssp);
	}
}

/******************************************************************************/

static uint32_t
NAME_(get_Ob)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrb)(cpssp, NAME_(get_Id)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	} else {
		return NAME_(smrb)(cpssp, NAME_(get_Iw)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	}
}

static uint32_t
NAME_(get_Ow)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrw)(cpssp, NAME_(get_Id)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	} else {
		return NAME_(smrw)(cpssp, NAME_(get_Iw)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	}
}

static uint32_t
NAME_(get_Od)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrd)(cpssp, NAME_(get_Id)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	} else {
		return NAME_(smrd)(cpssp, NAME_(get_Iw)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	}
}

static uint32_t
NAME_(get_Ov)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		return NAME_(get_Od)(cpssp);
	} else {
		return NAME_(get_Ow)(cpssp);
	}
}

/******************************************************************************
 * Get Memory addressed by the DS:eSI register pair
 ******************************************************************************/

static uint32_t
NAME_(get_Xb)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrb)(cpssp, NAME_(get_ESI)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	} else {
		return NAME_(smrb)(cpssp, NAME_(get_SI)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	}
}

static uint32_t
NAME_(get_Xw)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrw)(cpssp, NAME_(get_ESI)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	} else {
		return NAME_(smrw)(cpssp, NAME_(get_SI)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	}
}

static uint32_t
NAME_(get_Xd)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrd)(cpssp, NAME_(get_ESI)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	} else {
		return NAME_(smrd)(cpssp, NAME_(get_SI)(cpssp), NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS));
	}
}

/******************************************************************************
 * Get Memory addressed by the ES:eDI register pair
 ******************************************************************************/

static uint32_t
NAME_(get_Yb)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrb)(cpssp, NAME_(get_EDI)(cpssp), SEGMENT_ES);
	} else {
		return NAME_(smrb)(cpssp, NAME_(get_DI)(cpssp), SEGMENT_ES);
	}
}

static uint32_t
NAME_(get_Yw)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrw)(cpssp, NAME_(get_EDI)(cpssp), SEGMENT_ES);
	} else {
		return NAME_(smrw)(cpssp, NAME_(get_DI)(cpssp), SEGMENT_ES);
	}
}

static uint32_t
NAME_(get_Yd)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		return NAME_(smrd)(cpssp, NAME_(get_EDI)(cpssp), SEGMENT_ES);
	} else {
		return NAME_(smrd)(cpssp, NAME_(get_DI)(cpssp), SEGMENT_ES);
	}
}

/******************************************************************************
 * Load
 ******************************************************************************/

static void
NAME_(load_CS)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_CS)(cpssp);
}

static void
NAME_(load_SS)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_SS)(cpssp);
}

static void
NAME_(load_DS)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DS)(cpssp);
}

static void
NAME_(load_ES)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_ES)(cpssp);
}

static void
NAME_(load_FS)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_FS)(cpssp);
}

static void
NAME_(load_GS)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_GS)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_AL)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_AL)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_eAX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
}

static void
NAME_(load_eCX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eCX)(cpssp);
}

static void
NAME_(load_eDX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eDX)(cpssp);
}

static void
NAME_(load_eBX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eBX)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_eSP)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eSP)(cpssp);
}

static void
NAME_(load_eBP)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eBP)(cpssp);
}

static void
NAME_(load_eSI)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eSI)(cpssp);
}

static void
NAME_(load_eDI)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eDI)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_Cd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Cd)(cpssp);
}

static void
NAME_(load_Dd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Dd)(cpssp);
}

static void
NAME_(load_Td)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Td)(cpssp);
}

static void
NAME_(load_Rd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Rd)(cpssp, NAME_(get_instruction_rm)(cpssp));
}

/******************************************************************************/

static void
NAME_(load_0)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = 0;
}

static void
NAME_(load_Ib)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ib)(cpssp);
}

static void
NAME_(load_IbSE)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_IbSE)(cpssp);
}

static void
NAME_(load_Iw)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Iw)(cpssp);
}

static void
NAME_(load_Iv)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Iv)(cpssp);
}

static void
NAME_(load_AL_Ib)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_AL)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Ib)(cpssp);
}

static void
NAME_(load_eAX_Iv)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Iv)(cpssp);
}

static void
NAME_(load_Ib_AL)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ib)(cpssp);
	cpssp->NAME.t2 = NAME_(get_AL)(cpssp);
}

static void
NAME_(load_Ib_eAX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ib)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eAX)(cpssp);
}

static void
NAME_(load_Iw_Ib)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(smrw)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 2;

	cpssp->NAME.t2 = NAME_(smrb)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 1;
}

/******************************************************************************/

static void
NAME_(load_eAX_eAX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eAX)(cpssp);
}

static void
NAME_(load_eAX_eCX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eCX)(cpssp);
}

static void
NAME_(load_eAX_eDX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eDX)(cpssp);
}

static void
NAME_(load_eAX_eBX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eBX)(cpssp);
}

static void
NAME_(load_eAX_eSP)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eSP)(cpssp);
}

static void
NAME_(load_eAX_eBP)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eBP)(cpssp);
}

static void
NAME_(load_eAX_eSI)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eSI)(cpssp);
}

static void
NAME_(load_eAX_eDI)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_eAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eDI)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_DX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DX)(cpssp);
}

static void
NAME_(load_DX_AL)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_AL)(cpssp);
}

static void
NAME_(load_DX_eAX)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_eAX)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_Eb_Gb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Eb)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Gb)(cpssp);
}

static void
NAME_(load_Ev_Gv)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Gv)(cpssp);
}

static void
NAME_(load_Ev_Gv_Ib)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Gv)(cpssp);
	cpssp->NAME.t3 = NAME_(get_Ib)(cpssp);
}

static void
NAME_(load_Ev_Gv_CL)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Gv)(cpssp);
	cpssp->NAME.t3 = NAME_(get_CL)(cpssp);
}

static void
NAME_(load_EvBIT_Gv)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_EvBIT)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Gv)(cpssp);
}

static void
NAME_(load_Eb_Ib)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Eb)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Ib)(cpssp);
}

static void
NAME_(load_Eb_1)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Eb)(cpssp);
	cpssp->NAME.t2 = 1;
}

static void
NAME_(load_Ev_1)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = 1;
}

static void
NAME_(load_Eb_CL)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Eb)(cpssp);
	cpssp->NAME.t2 = NAME_(get_CL)(cpssp);
}

static void
NAME_(load_Ev_CL)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_CL)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_Ev_Iv)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Iv)(cpssp);
}

static void
NAME_(load_Ev_Ib)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Ib)(cpssp);
}

static void
NAME_(load_Ev_IbSE)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
	cpssp->NAME.t2 = NAME_(get_IbSE)(cpssp);
}

static void
NAME_(load_Eb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Eb)(cpssp);
}

static void
NAME_(load_Ew)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ew)(cpssp);
}

static void
NAME_(load_Ev)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ev)(cpssp);
}

static void
NAME_(load_Gb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Gb)(cpssp);
}

static void
NAME_(load_Gv)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Gv)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_Gb_Eb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Gb)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Eb)(cpssp);
}

static void
NAME_(load_Gv_Ev)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Gv)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Ev)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_Sw)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Sw)(cpssp);
}

static void
NAME_(load_Ob)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ob)(cpssp);
}

static void
NAME_(load_Ov)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Ov)(cpssp);
}

static void
NAME_(load_Ap)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_instruction_displacement)(cpssp);
	cpssp->NAME.t2 = NAME_(get_instruction_immediate)(cpssp);
}

/******************************************************************************/

static void
NAME_(load_Jb)(struct cpssp *cpssp)
{
	cpssp->NAME.tj = NAME_(get_Jb)(cpssp);
	/* TODO: debug */
}

static void
NAME_(load_Jv)(struct cpssp *cpssp)
{
	cpssp->NAME.tj = NAME_(get_Jv)(cpssp);
	/* TODO: debug */
}

/******************************************************************************/

static void
NAME_(load_effective_address)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_effective_address)(cpssp);
	/* TODO: debug */
}

static void
NAME_(load_Ms)(struct cpssp *cpssp)
{
	uint32_t addr;
	uint8_t segment;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		addr = NAME_(get_address_mod0)(cpssp);
		segment = NAME_(get_segment_mod0)(cpssp);
		break;
	case 1:
		addr = NAME_(get_address_mod1)(cpssp);
		segment = NAME_(get_segment_mod1)(cpssp);
		break;
	case 2:
		addr = NAME_(get_address_mod2)(cpssp);
		segment = NAME_(get_segment_mod2)(cpssp);
		break;
	case 3:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	segment = NAME_(get_segment_overwritten)(cpssp, segment);

	cpssp->NAME.t1 = NAME_(smrw)(cpssp, addr + 0, segment); /* limit */
	cpssp->NAME.t2 = NAME_(smrd)(cpssp, addr + 2, segment); /* base */
}

static void
NAME_(load_Mp)(struct cpssp *cpssp)
{
	uint32_t addr;
	uint8_t segment;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		addr = NAME_(get_address_mod0)(cpssp);
		segment = NAME_(get_segment_mod0)(cpssp);
		break;
	case 1:
		addr = NAME_(get_address_mod1)(cpssp);
		segment = NAME_(get_segment_mod1)(cpssp);
		break;
	case 2:
		addr = NAME_(get_address_mod2)(cpssp);
		segment = NAME_(get_segment_mod2)(cpssp);
		break;
	case 3:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	segment = NAME_(get_segment_overwritten)(cpssp, segment);

	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.t1 = NAME_(smrd)(cpssp, addr + 0, segment); /* register */
		cpssp->NAME.t2 = NAME_(smrw)(cpssp, addr + 4, segment); /* segment cpssp->NAME.selector */
	} else {
		cpssp->NAME.t1 = NAME_(smrw)(cpssp, addr + 0, segment); /* register */
		cpssp->NAME.t2 = NAME_(smrw)(cpssp, addr + 2, segment); /* segment cpssp->NAME.selector */
	}
}

static void
NAME_(load_Gv_Ma)(struct cpssp *cpssp)
{
	uint32_t addr;
	uint8_t segment;

	cpssp->NAME.t1 = NAME_(get_Gv)(cpssp);

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		addr = NAME_(get_address_mod0)(cpssp);
		segment = NAME_(get_segment_mod0)(cpssp);
		break;
	case 1:
		addr = NAME_(get_address_mod1)(cpssp);
		segment = NAME_(get_segment_mod1)(cpssp);
		break;
	case 2:
		addr = NAME_(get_address_mod2)(cpssp);
		segment = NAME_(get_segment_mod2)(cpssp);
		break;
	case 3:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	segment = NAME_(get_segment_overwritten)(cpssp, segment);

	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.t2 = NAME_(smrd)(cpssp, addr + 0, segment); /* lower bound */
		cpssp->NAME.t3 = NAME_(smrd)(cpssp, addr + 4, segment); /* upper bound */
	} else {
		cpssp->NAME.t2 = NAME_(smrw)(cpssp, addr + 0, segment); /* lower bound */
		cpssp->NAME.t3 = NAME_(smrw)(cpssp, addr + 2, segment); /* upper bound */
	}
}

static void
NAME_(load_Ep)(struct cpssp *cpssp)
{
	NAME_(load_Mp)(cpssp);
}

/******************************************************************************
 * Load Memory addressed by the ES:eDI and DS:eSI register pair
 ******************************************************************************/

static void
NAME_(load_Xb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Xb)(cpssp);
}

static void
NAME_(load_Xw)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Xw)(cpssp);
}

static void
NAME_(load_Xd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Xd)(cpssp);
}

static void
NAME_(load_DX_Xb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Xb)(cpssp);
}

static void
NAME_(load_DX_Xw)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Xw)(cpssp);
}

static void
NAME_(load_DX_Xd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_DX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Xd)(cpssp);
}

static void
NAME_(load_AL_Yb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_AL)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Yb)(cpssp);
}

static void
NAME_(load_AX_Yw)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_AX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Yw)(cpssp);
}

static void
NAME_(load_EAX_Yd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_EAX)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Yd)(cpssp);
}

static void
NAME_(load_Xb_Yb)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Xb)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Yb)(cpssp);
}

static void
NAME_(load_Xw_Yw)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Xw)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Yw)(cpssp);
}

static void
NAME_(load_Xd_Yd)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = NAME_(get_Xd)(cpssp);
	cpssp->NAME.t2 = NAME_(get_Yd)(cpssp);
}

/******************************************************************************
 * SET Segment Selector and Descriptor
 ******************************************************************************/

static void
NAME_(sset_selector)(struct cpssp *cpssp, uint16_t value)
{
	cpssp->NAME.selector = value;
	cpssp->NAME.selector_index = value >> 3;
	cpssp->NAME.selector_ti = (value >> 2) & 1;
	cpssp->NAME.selector_rpl = value & 3;
}

static void
NAME_(sset_descriptor)(struct cpssp *cpssp, uint32_t dw1, uint32_t dw2)
{
	bool gflag;

	cpssp->NAME.descriptor_type  = (dw2 >>  8) & 0xf;
	cpssp->NAME.descriptor_sflag = (dw2 >> 12) & 0x1;
	cpssp->NAME.descriptor_dpl   = (dw2 >> 13) & 0x3;
	cpssp->NAME.descriptor_pflag = (dw2 >> 15) & 0x1;

	if (cpssp->NAME.descriptor_sflag) { /* code or data segment descriptor */
		cpssp->NAME.descriptor_type &= ~1; /* Ignore Accessed flag. */
		gflag   = (dw2 & 0x00800000) > 0;
		cpssp->descriptor_segment_limit   = (dw1 & 0x0000ffff)
					   | (dw2 & 0x000f0000);
		cpssp->descriptor_segment_limit = NAME_(limit_scaled)(cpssp, 
				cpssp->descriptor_segment_limit, gflag);
		cpssp->descriptor_segment_base    = (dw1 >> 16)
					   | ((dw2 & 0xff) << 16)
					   | (dw2 & 0xff000000);
		cpssp->NAME.descriptor_segment_dflag   = (dw2 & 0x00400000) > 0;

	} else { /* system segment descriptor */
		switch (cpssp->NAME.descriptor_type) {
		default:
		case 0x0:
		case 0x8:
		case 0xa:
		case 0xd:
			/* Reserved */
			break;
		case SEGMENT_16BIT_CALL_GATE:
			cpssp->NAME.descriptor_gate_paramcount = dw2 & 0x1f;
			cpssp->NAME.descriptor_gate_selector = dw1 >> 16;
			cpssp->NAME.descriptor_gate_offset = dw1 & 0xffff;
			break;
		case SEGMENT_16BIT_INTERRUPT_GATE:
		case SEGMENT_16BIT_TRAP_GATE:
			cpssp->NAME.descriptor_gate_selector = dw1 >> 16;
			cpssp->NAME.descriptor_gate_offset = dw1 & 0xffff;
			break;
		case SEGMENT_32BIT_CALL_GATE:
			cpssp->NAME.descriptor_gate_paramcount = dw2 & 0x1f;
			cpssp->NAME.descriptor_gate_selector = dw1 >> 16;
			cpssp->NAME.descriptor_gate_offset = (dw2 & 0xffff0000)
					       | (dw1 & 0x0000ffff);
			break;
		case SEGMENT_32BIT_INTERRUPT_GATE:
		case SEGMENT_32BIT_TRAP_GATE:
			cpssp->NAME.descriptor_gate_selector = dw1 >> 16;
			cpssp->NAME.descriptor_gate_offset = (dw2 & 0xffff0000)
					       | (dw1 & 0x0000ffff);
			break;
		case SEGMENT_TASK_GATE:
			cpssp->NAME.descriptor_gate_selector = dw1 >> 16;
			break;
		case SEGMENT_16BIT_AVAIL_TSS:
		case SEGMENT_16BIT_BUSY_TSS:
			cpssp->NAME.descriptor_system_base = (dw1 >> 16)
					       | ((dw2 & 0xff) << 16);
			cpssp->NAME.descriptor_system_limit = (dw1 & 0xffff);
			break;
		case SEGMENT__LDT:
		case SEGMENT_32BIT_AVAIL_TSS:
		case SEGMENT_32BIT_BUSY_TSS:
			gflag = (dw2 & 0x00800000) > 0;
			cpssp->NAME.descriptor_system_base = (dw1 >> 16)
					       | ((dw2 & 0xff) << 16)
					       | (dw2 & 0xff000000);
			cpssp->NAME.descriptor_system_limit = (dw1 & 0x0000ffff)
						| (dw2 & 0x000f0000);
			cpssp->NAME.descriptor_system_limit = NAME_(limit_scaled)(cpssp, 
					cpssp->NAME.descriptor_system_limit, gflag);
			break;
		}
	}
}

static void
NAME_(sset_SEG)(struct cpssp *cpssp, uint8_t segment, uint16_t value)
{
	if (NAME_(protected_mode)(cpssp)) {
		NAME_(sset_selector)(cpssp, value);

		if (cpssp->NAME.selector_index == 0) {
			if (segment == SEGMENT_SS) {
				NAME_(exception)(cpssp, EXCEPTION_GP, 0);
			} else {
				NAME_(sset_selector)(cpssp, 0);
			}
			return;
		}
		if (segment == SEGMENT_SS
		 && cpssp->NAME.selector_rpl != NAME_(get_CPL)(cpssp)) {
			NAME_(exception)(cpssp, EXCEPTION_GP, value & 0xfffc);
		}

		NAME_(sset_descriptor)(cpssp, NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8,
					cpssp->NAME.selector_ti ? SEGMENT_LDT : SEGMENT_GDT),
				NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8 + 4,
					cpssp->NAME.selector_ti ? SEGMENT_LDT : SEGMENT_GDT));
	} else {
		cpssp->NAME.selector = value;

		cpssp->NAME.descriptor_type = SEGMENT_DATA_READ_WRITE;
		cpssp->NAME.descriptor_sflag = 1;
		if (NAME_(real_mode)(cpssp)) {
			cpssp->NAME.descriptor_dpl = 0;
		} else { /* virtual8086_mode */
			cpssp->NAME.descriptor_dpl = 3;
		}
		cpssp->NAME.descriptor_pflag = 1;

		cpssp->descriptor_segment_limit = 0xffff;
		cpssp->descriptor_segment_base = value << 4;
		cpssp->NAME.descriptor_segment_dflag = 0;
	}
}

static void
NAME_(check_SEG)(struct cpssp *cpssp, uint8_t segment, uint16_t value)
{
	if ((!cpssp->NAME.descriptor_sflag) /* no data/code segment */
	 || (((cpssp->NAME.descriptor_type >> 3) & 1) /* code segment */
	  && (!((cpssp->NAME.descriptor_type >> 1) & 1)))) { /* not readable */
		NAME_(exception)(cpssp, EXCEPTION_GP, value & 0xfffc);
	}

	if (segment == SEGMENT_SS) {
		if (cpssp->NAME.descriptor_dpl != NAME_(get_CPL)(cpssp)) {
			NAME_(exception)(cpssp, EXCEPTION_GP, value & 0xfffc);
		}
	} else {
		/* if data or non-conforming code, then rpl and cpl	*/
		/* must be less than or equal to dpl			*/
		if ((!((cpssp->NAME.descriptor_type >> 1) & 1)) /* data segemnt */
		 || (!((cpssp->NAME.descriptor_type >> 2) & 1))) { /* non-conformaing */
			if (((cpssp->NAME.selector_rpl) > cpssp->NAME.descriptor_dpl)
			 || (NAME_(get_CPL)(cpssp) > cpssp->NAME.descriptor_dpl)) {
				NAME_(exception)(cpssp, EXCEPTION_GP, value & 0xfffc);
			}
		}
	}
	if (!cpssp->NAME.descriptor_pflag) { /* segment not present */
		NAME_(exception)(cpssp, EXCEPTION_NP, value & 0xfffc);
	}
}

static void
NAME_(load_code_desc)(struct cpssp *cpssp, bool vm, uint16_t value)
{
	uint32_t dw1;
	uint32_t dw2;
	bool gflag;

	if (NAME_(real_mode)(cpssp)
	 || vm) {
		cpssp->NAME.temp_cs_selector = value;
		cpssp->NAME.temp_cs_selector_rpl = NAME_(real_mode)(cpssp) ? 0 : 3;

		cpssp->NAME.temp_cs_type = SEGMENT_CODE_EXEC_READ;
		cpssp->NAME.temp_cs_sflag = 1;
		cpssp->NAME.temp_cs_dpl = NAME_(real_mode)(cpssp) ? 0 : 3;
		cpssp->NAME.temp_cs_pflag = 1;

		cpssp->NAME.temp_cs_segment_limit = 0xffff;
		cpssp->NAME.temp_cs_segment_base = value << 4;
		cpssp->NAME.temp_cs_segment_dflag = 0;

	} else {
		NAME_(sset_selector)(cpssp, value);

		if (cpssp->NAME.selector_index == 0) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}

		dw1 = NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8,
				cpssp->NAME.selector_ti ? SEGMENT_LDT : SEGMENT_GDT);
		dw2 = NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8 + 4,
				cpssp->NAME.selector_ti ? SEGMENT_LDT : SEGMENT_GDT);

		cpssp->NAME.temp_cs_selector		= cpssp->NAME.selector;
		cpssp->NAME.temp_cs_selector_rpl	= cpssp->NAME.selector_rpl;

		cpssp->NAME.temp_cs_type		= (dw2 >> 8) & 0xf;
		cpssp->NAME.temp_cs_sflag		= (dw2 >> 12) & 0x1;
		cpssp->NAME.temp_cs_dpl		= (dw2 >> 13) & 0x3;;
		cpssp->NAME.temp_cs_pflag		= (dw2 >> 15) & 0x1;;

		if (cpssp->NAME.temp_cs_sflag) {
			/* Code or Data Descriptor */
			gflag = (dw2 & 0x00800000) > 0;
			cpssp->NAME.temp_cs_segment_limit = NAME_(limit_scaled)(cpssp, 
					(dw1 & 0x0000ffff) | (dw2 & 0x000f0000),
					gflag);
			cpssp->NAME.temp_cs_segment_base = (dw1 >> 16)
					| ((dw2 & 0xff) << 16)
					| (dw2 & 0xff000000);
			cpssp->NAME.temp_cs_segment_dflag = (dw2 & 0x00400000) > 0;

		} else {
			/* System Descriptor */
			switch (cpssp->NAME.temp_cs_type) {
			case 0x0:
			case /*0x1*/ SEGMENT_16BIT_AVAIL_TSS:
			case /*0x2*/ SEGMENT__LDT:
			case /*0x3*/ SEGMENT_16BIT_BUSY_TSS:
			case 0x8:
			case /*0x9*/ SEGMENT_32BIT_AVAIL_TSS:
			case 0xa:
			case /*0xb*/ SEGMENT_32BIT_BUSY_TSS:
				/* System Segment */
				gflag = (dw2 & 0x00800000) > 0;
				cpssp->NAME.descriptor_system_base = (dw1 >> 16)
						| ((dw2 & 0xff) << 16)
						| (dw2 & 0xff000000);
				cpssp->NAME.descriptor_system_limit = NAME_(limit_scaled)(cpssp, 
						(dw1 & 0x0000ffff) | (dw2 & 0x000f0000),
						gflag);
				break;
			case /*0x4*/ SEGMENT_16BIT_CALL_GATE:
			case /*0x5*/ SEGMENT_TASK_GATE:
			case /*0x6*/ SEGMENT_16BIT_INTERRUPT_GATE:
			case /*0x7*/ SEGMENT_16BIT_TRAP_GATE:
			case /*0xc*/ SEGMENT_32BIT_CALL_GATE:
			case 0xd:
			case /*0xe*/ SEGMENT_32BIT_INTERRUPT_GATE:
			case /*0xf*/ SEGMENT_32BIT_TRAP_GATE:
				/* System Gate */
				cpssp->NAME.descriptor_gate_paramcount = dw2 & 0x1f;
				cpssp->NAME.descriptor_gate_selector = dw1 >> 16;
				cpssp->NAME.descriptor_gate_offset = (dw2 & 0xffff0000)
						| (dw1 & 0x0000ffff);
				break;
			}
		}
	}
}

static void
NAME_(load_data_desc)(struct cpssp *cpssp, bool vm, uint16_t value)
{
	if (NAME_(real_mode)(cpssp)
	 || vm) {
		cpssp->NAME.selector = value;
		cpssp->NAME.selector_rpl = NAME_(real_mode)(cpssp) ? 0 : 3;

		cpssp->NAME.descriptor_type = SEGMENT_CODE_EXEC_READ;
		cpssp->NAME.descriptor_sflag = 1;
		cpssp->NAME.descriptor_dpl = NAME_(real_mode)(cpssp) ? 0 : 3;
		cpssp->NAME.descriptor_pflag = 1;

		cpssp->descriptor_segment_limit = 0xffff;
		cpssp->descriptor_segment_base = value << 4;
		cpssp->NAME.descriptor_segment_dflag = 0;

	} else {
		NAME_(sset_selector)(cpssp, value);

		if (cpssp->NAME.selector_index == 0) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}

		NAME_(sset_descriptor)(cpssp, NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8,
					cpssp->NAME.selector_ti ? SEGMENT_LDT : SEGMENT_GDT),
				NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8 + 4,
					cpssp->NAME.selector_ti ? SEGMENT_LDT : SEGMENT_GDT));
	}
}

static void
NAME_(sset_SS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_SEG)(cpssp, SEGMENT_SS, value);
	NAME_(check_SEG)(cpssp, SEGMENT_SS, value);
}

static void
NAME_(sset_DS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_SEG)(cpssp, SEGMENT_DS, value);
	NAME_(check_SEG)(cpssp, SEGMENT_DS, value);
}

static void
NAME_(sset_ES)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_SEG)(cpssp, SEGMENT_ES, value);
	NAME_(check_SEG)(cpssp, SEGMENT_ES, value);
}

static void
NAME_(sset_FS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_SEG)(cpssp, SEGMENT_FS, value);
	NAME_(check_SEG)(cpssp, SEGMENT_FS, value);
}

static void
NAME_(sset_GS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_SEG)(cpssp, SEGMENT_GS, value);
	NAME_(check_SEG)(cpssp, SEGMENT_GS, value);
}

/******************************************************************************/

static void
NAME_(commit_CS)(struct cpssp *cpssp)
{
	cpssp->NAME.cs_selector	= cpssp->NAME.temp_cs_selector;
	cpssp->NAME.cs_type		= cpssp->NAME.temp_cs_type;
	cpssp->NAME.cs_sflag		= cpssp->NAME.temp_cs_sflag;
	cpssp->cs_dpl			= cpssp->NAME.temp_cs_dpl;
	cpssp->NAME.cs_pflag		= cpssp->NAME.temp_cs_pflag;
	cpssp->cs_segment_limit	= cpssp->NAME.temp_cs_segment_limit;
	cpssp->cs_segment_base	= cpssp->NAME.temp_cs_segment_base;
	cpssp->NAME.cs_segment_dflag	= cpssp->NAME.temp_cs_segment_dflag;
}

static void
NAME_(commit_SS)(struct cpssp *cpssp)
{
	cpssp->NAME.ss_selector	= cpssp->NAME.selector;
	cpssp->NAME.ss_type		= cpssp->NAME.descriptor_type;
	cpssp->NAME.ss_sflag		= cpssp->NAME.descriptor_sflag;
	cpssp->NAME.ss_dpl		= cpssp->NAME.descriptor_dpl;
	cpssp->NAME.ss_pflag		= cpssp->NAME.descriptor_pflag;
	cpssp->ss_segment_limit		= cpssp->descriptor_segment_limit;
	cpssp->ss_segment_base		= cpssp->descriptor_segment_base;
	cpssp->NAME.ss_segment_dflag	= cpssp->NAME.descriptor_segment_dflag;
}

static void
NAME_(commit_DS)(struct cpssp *cpssp)
{
	cpssp->NAME.ds_selector	= cpssp->NAME.selector;
	cpssp->NAME.ds_type		= cpssp->NAME.descriptor_type;
	cpssp->NAME.ds_sflag		= cpssp->NAME.descriptor_sflag;
	cpssp->NAME.ds_dpl		= cpssp->NAME.descriptor_dpl;
	cpssp->NAME.ds_pflag		= cpssp->NAME.descriptor_pflag;
	cpssp->ds_segment_limit		= cpssp->descriptor_segment_limit;
	cpssp->ds_segment_base		= cpssp->descriptor_segment_base;
	cpssp->NAME.ds_segment_dflag	= cpssp->NAME.descriptor_segment_dflag;
}

static void
NAME_(commit_ES)(struct cpssp *cpssp)
{
	cpssp->NAME.es_selector	= cpssp->NAME.selector;
	cpssp->NAME.es_type		= cpssp->NAME.descriptor_type;
	cpssp->NAME.es_sflag		= cpssp->NAME.descriptor_sflag;
	cpssp->NAME.es_dpl		= cpssp->NAME.descriptor_dpl;
	cpssp->NAME.es_pflag		= cpssp->NAME.descriptor_pflag;
	cpssp->es_segment_limit		= cpssp->descriptor_segment_limit;
	cpssp->es_segment_base		= cpssp->descriptor_segment_base;
	cpssp->NAME.es_segment_dflag	= cpssp->NAME.descriptor_segment_dflag;
}

static void
NAME_(commit_FS)(struct cpssp *cpssp)
{
	cpssp->NAME.fs_selector	= cpssp->NAME.selector;
	cpssp->NAME.fs_type		= cpssp->NAME.descriptor_type;
	cpssp->NAME.fs_sflag		= cpssp->NAME.descriptor_sflag;
	cpssp->NAME.fs_dpl		= cpssp->NAME.descriptor_dpl;
	cpssp->NAME.fs_pflag		= cpssp->NAME.descriptor_pflag;
	cpssp->fs_segment_limit		= cpssp->descriptor_segment_limit;
	cpssp->fs_segment_base		= cpssp->descriptor_segment_base;
	cpssp->NAME.fs_segment_dflag	= cpssp->NAME.descriptor_segment_dflag;
}

static void
NAME_(commit_GS)(struct cpssp *cpssp)
{
	cpssp->NAME.gs_selector	= cpssp->NAME.selector;
	cpssp->NAME.gs_type		= cpssp->NAME.descriptor_type;
	cpssp->NAME.gs_sflag		= cpssp->NAME.descriptor_sflag;
	cpssp->NAME.gs_dpl		= cpssp->NAME.descriptor_dpl;
	cpssp->NAME.gs_pflag		= cpssp->NAME.descriptor_pflag;
	cpssp->gs_segment_limit		= cpssp->descriptor_segment_limit;
	cpssp->gs_segment_base		= cpssp->descriptor_segment_base;
	cpssp->NAME.gs_segment_dflag	= cpssp->NAME.descriptor_segment_dflag;
}

static void
NAME_(set_SS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_SS)(cpssp, value);
	NAME_(commit_SS)(cpssp);
}

static void
NAME_(set_DS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_DS)(cpssp, value);
	NAME_(commit_DS)(cpssp);
}

static void
NAME_(set_ES)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_ES)(cpssp, value);
	NAME_(commit_ES)(cpssp);
}

static void
NAME_(set_FS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_FS)(cpssp, value);
	NAME_(commit_FS)(cpssp);
}

static void
NAME_(set_GS)(struct cpssp *cpssp, uint16_t value)
{
	NAME_(sset_GS)(cpssp, value);
	NAME_(commit_GS)(cpssp);
}

/******************************************************************************
 * SET
 ******************************************************************************/

static void
NAME_(set_AL)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.eax = (cpssp->NAME.eax & ~0xff) | (value & 0xff);
}

static void
NAME_(set_CL)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ecx = (cpssp->NAME.ecx & ~0xff) | (value & 0xff);
}

static void
NAME_(set_DL)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.edx = (cpssp->NAME.edx & ~0xff) | (value & 0xff);
}

static void
NAME_(set_BL)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ebx = (cpssp->NAME.ebx & ~0xff) | (value & 0xff);
}

/******************************************************************************/

static void
NAME_(set_AH)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.eax = (cpssp->NAME.eax & 0xffff00ff) | ((value & 0xff) << 8);
}

static void
NAME_(set_CH)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ecx = (cpssp->NAME.ecx & 0xffff00ff) | ((value & 0xff) << 8);
}

static void
NAME_(set_DH)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.edx = (cpssp->NAME.edx & 0xffff00ff) | ((value & 0xff) << 8);
}

static void
NAME_(set_BH)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ebx = (cpssp->NAME.ebx & 0xffff00ff) | ((value & 0xff) << 8);
}

/******************************************************************************/

static void
NAME_(set_AX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.eax = (cpssp->NAME.eax & ~0xffff) | (value & 0xffff);
}

static void
NAME_(set_CX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ecx = (cpssp->NAME.ecx & ~0xffff) | (value & 0xffff);
}

static void
NAME_(set_DX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.edx = (cpssp->NAME.edx & ~0xffff) | (value & 0xffff);
}

static void
NAME_(set_BX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ebx = (cpssp->NAME.ebx & ~0xffff) | (value & 0xffff);
}

/******************************************************************************/

static void
NAME_(set_SP)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.esp = (cpssp->NAME.esp & ~0xffff) | (value & 0xffff);
}

static void
NAME_(set_BP)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ebp = (cpssp->NAME.ebp & ~0xffff) | (value & 0xffff);
}

static void
NAME_(set_SI)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.esi = (cpssp->NAME.esi & ~0xffff) | (value & 0xffff);
}

static void
NAME_(set_DI)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.edi = (cpssp->NAME.edi & ~0xffff) | (value & 0xffff);
}

/******************************************************************************/

static void
NAME_(set_EAX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.eax = value;
}

static void
NAME_(set_ECX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ecx = value;
}

static void
NAME_(set_EDX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.edx = value;
}

static void
NAME_(set_EBX)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ebx = value;
}

/******************************************************************************/

static void
NAME_(set_ESP)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.esp = value;
}

static void
NAME_(set_EBP)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.ebp = value;
}

static void
NAME_(set_ESI)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.esi = value;
}

static void
NAME_(set_EDI)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.edi = value;
}

/******************************************************************************/

static void
NAME_(set_eAX)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EAX)(cpssp, value);
	} else {
		NAME_(set_AX)(cpssp, value);
	}
}

static void
NAME_(set_eCX)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_ECX)(cpssp, value);
	} else {
		NAME_(set_CX)(cpssp, value);
	}
}

static void
NAME_(set_eDX)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EDX)(cpssp, value);
	} else {
		NAME_(set_DX)(cpssp, value);
	}
}

static void
NAME_(set_eBX)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EBX)(cpssp, value);
	} else {
		NAME_(set_BX)(cpssp, value);
	}
}

/******************************************************************************/

static void
NAME_(set_eSP)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_ESP)(cpssp, value);
	} else {
		NAME_(set_SP)(cpssp, value);
	}
}

static void
NAME_(set_eBP)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EBP)(cpssp, value);
	} else {
		NAME_(set_BP)(cpssp, value);
	}
}

static void
NAME_(set_eSI)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_ESI)(cpssp, value);
	} else {
		NAME_(set_SI)(cpssp, value);
	}
}

static void
NAME_(set_eDI)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EDI)(cpssp, value);
	} else {
		NAME_(set_DI)(cpssp, value);
	}
}

/******************************************************************************/

/* TODO: implement functionality of debugging registers */

static void
NAME_(set_DR0)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.dr0 = value;
}

static void
NAME_(set_DR1)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.dr1 = value;
}

static void
NAME_(set_DR2)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.dr2 = value;
}

static void
NAME_(set_DR3)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.dr3 = value;
}

/* DR4 and DR5 do not exist on 80386 */

static void
NAME_(set_DR6)(struct cpssp *cpssp, uint32_t value)
{
	cpssp->NAME.dr6 = value;
}

static void
NAME_(set_DR7)(struct cpssp *cpssp, uint32_t value)
{

	if (value & 0xff) {
		/* someone tries to enable breakpoints */
		ERROR_NYI();
	}
	cpssp->NAME.dr7 = value;
}

/******************************************************************************/

static void
NAME_(set_Rb)(struct cpssp *cpssp, uint32_t value, uint32_t reg)
{
	switch (reg) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_AL)(cpssp, value);
		break;
	case 1:
		NAME_(set_CL)(cpssp, value);
		break;
	case 2:
		NAME_(set_DL)(cpssp, value);
		break;
	case 3:
		NAME_(set_BL)(cpssp, value);
		break;
	case 4:
		NAME_(set_AH)(cpssp, value);
		break;
	case 5:
		NAME_(set_CH)(cpssp, value);
		break;
	case 6:
		NAME_(set_DH)(cpssp, value);
		break;
	case 7:
		NAME_(set_BH)(cpssp, value);
		break;
	}
}

static void
NAME_(set_Rw)(struct cpssp *cpssp, uint32_t value, uint32_t reg)
{
	switch (reg) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_AX)(cpssp, value);
		break;
	case 1:
		NAME_(set_CX)(cpssp, value);
		break;
	case 2:
		NAME_(set_DX)(cpssp, value);
		break;
	case 3:
		NAME_(set_BX)(cpssp, value);
		break;
	case 4:
		NAME_(set_SP)(cpssp, value);
		break;
	case 5:
		NAME_(set_BP)(cpssp, value);
		break;
	case 6:
		NAME_(set_SI)(cpssp, value);
		break;
	case 7:
		NAME_(set_DI)(cpssp, value);
		break;
	}
}

static void
NAME_(set_Rd)(struct cpssp *cpssp, uint32_t value, uint32_t reg)
{
	switch (reg) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_EAX)(cpssp, value);
		break;
	case 1:
		NAME_(set_ECX)(cpssp, value);
		break;
	case 2:
		NAME_(set_EDX)(cpssp, value);
		break;
	case 3:
		NAME_(set_EBX)(cpssp, value);
		break;
	case 4:
		NAME_(set_ESP)(cpssp, value);
		break;
	case 5:
		NAME_(set_EBP)(cpssp, value);
		break;
	case 6:
		NAME_(set_ESI)(cpssp, value);
		break;
	case 7:
		NAME_(set_EDI)(cpssp, value);
		break;
	}
}

/******************************************************************************/

static void
NAME_(set_Cd)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(protected_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	switch(NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_CR0)(cpssp, value);
		break;
	case 1:
		ERROR_NYI(); /* FIXME: Reserved */
	case 2:
		NAME_(set_CR2)(cpssp, value);
		break;
	case 3:
		NAME_(set_CR3)(cpssp, value);
		break;
	case 4:
	case 5:
	case 6:
	case 7:
		ERROR_NYI(); /* FIXME: Reserved */
	}
}

static void
NAME_(set_Dd)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(protected_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	switch(NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_DR0)(cpssp, value);
		break;
	case 1:
		NAME_(set_DR1)(cpssp, value);
		break;
	case 2:
		NAME_(set_DR2)(cpssp, value);
		break;
	case 3:
		NAME_(set_DR3)(cpssp, value);
		break;
	case 4: /* "Intel reserved" */
	case 5: /* "Intel reserved" */
		break;
	case 6:
		NAME_(set_DR6)(cpssp, value);
		break;
	case 7:
		NAME_(set_DR7)(cpssp, value);
		break;
	}
}

static void
NAME_(set_Td)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	case 6:
	case 7:
		ERROR_NYI(); /* FIXME */
		break;
	default:
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		/*NOTREACHED*/
	}
}

/******************************************************************************/

static void
NAME_(set_Eb)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(smwb)(cpssp, NAME_(get_address_mod0)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)), value);
		break;
	case 1:
		NAME_(smwb)(cpssp, NAME_(get_address_mod1)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)), value);
		break;
	case 2:
		NAME_(smwb)(cpssp, NAME_(get_address_mod2)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)), value);
		break;
	case 3:
		NAME_(set_Rb)(cpssp, value, NAME_(get_instruction_rm)(cpssp));
		break;
	}
}

static void
NAME_(set_Ew)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(smww)(cpssp, NAME_(get_address_mod0)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)), value);
		break;
	case 1:
		NAME_(smww)(cpssp, NAME_(get_address_mod1)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)), value);
		break;
	case 2:
		NAME_(smww)(cpssp, NAME_(get_address_mod2)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)), value);
		break;
	case 3:
		NAME_(set_Rw)(cpssp, value, NAME_(get_instruction_rm)(cpssp));
		break;
	}
}

static void
NAME_(set_Ed)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(smwd)(cpssp, NAME_(get_address_mod0)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)), value);
		break;
	case 1:
		NAME_(smwd)(cpssp, NAME_(get_address_mod1)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)), value);
		break;
	case 2:
		NAME_(smwd)(cpssp, NAME_(get_address_mod2)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)), value);
		break;
	case 3:
		NAME_(set_Rd)(cpssp, value, NAME_(get_instruction_rm)(cpssp));
		break;
	}
}

static void
NAME_(set_Ev)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_Ed)(cpssp, value);
	} else {
		NAME_(set_Ew)(cpssp, value);
	}
}

/******************************************************************************/

static void
NAME_(set_EwBIT)(struct cpssp *cpssp, uint32_t value)
{
	int32_t offset;

	offset = ((int16_t) (NAME_(get_Rw)(cpssp, NAME_(get_instruction_reg)(cpssp)) & 0xfff0)) / 16;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(smww)(cpssp, NAME_(get_address_mod0)(cpssp) + 2 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)), value);
		break;
	case 1:
		NAME_(smww)(cpssp, NAME_(get_address_mod1)(cpssp) + 2 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)), value);
		break;
	case 2:
		NAME_(smww)(cpssp, NAME_(get_address_mod2)(cpssp) + 2 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)), value);
		break;
	case 3: NAME_(set_Rd)(cpssp, value, NAME_(get_instruction_rm)(cpssp));
		break;
	}
}

static void
NAME_(set_EdBIT)(struct cpssp *cpssp, uint32_t value)
{
	int32_t offset;

	offset = ((int32_t) (NAME_(get_Rd)(cpssp, NAME_(get_instruction_reg)(cpssp)) & 0xffffffe0)) / 32;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(smwd)(cpssp, NAME_(get_address_mod0)(cpssp) + 4 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod0)(cpssp)), value);
		break;
	case 1:
		NAME_(smwd)(cpssp, NAME_(get_address_mod1)(cpssp) + 4 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod1)(cpssp)), value);
		break;
	case 2:
		NAME_(smwd)(cpssp, NAME_(get_address_mod2)(cpssp) + 4 * offset,
			NAME_(get_segment_overwritten)(cpssp, NAME_(get_segment_mod2)(cpssp)), value);
		break;
	case 3:
		NAME_(set_Rd)(cpssp, value, NAME_(get_instruction_rm)(cpssp));
		break;
	}
}

static void
NAME_(set_EvBIT)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EdBIT)(cpssp, value);
	} else {
		NAME_(set_EwBIT)(cpssp, value);
	}
}

/******************************************************************************/

static void
NAME_(set_Ob)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(address_size_32)(cpssp)) {
		NAME_(smwb)(cpssp, NAME_(get_Id)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS), value);
	} else {
		NAME_(smwb)(cpssp, NAME_(get_Iw)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS), value);
	}
}

static void
NAME_(set_Ow)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(address_size_32)(cpssp)) {
		NAME_(smww)(cpssp, NAME_(get_Id)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS), value);
	} else {
		NAME_(smww)(cpssp, NAME_(get_Iw)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS), value);
	}
}

static void
NAME_(set_Od)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(address_size_32)(cpssp)) {
		NAME_(smwd)(cpssp, NAME_(get_Id)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS), value);
	} else {
		NAME_(smwd)(cpssp, NAME_(get_Iw)(cpssp),
			NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS), value);
	}
}

static void
NAME_(set_Ov)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_Od)(cpssp, value);
	} else {
		NAME_(set_Ow)(cpssp, value);
	}
}

/******************************************************************************/

static void
NAME_(set_Gb)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_AL)(cpssp, value);
		break;
	case 1:
		NAME_(set_CL)(cpssp, value);
		break;
	case 2:
		NAME_(set_DL)(cpssp, value);
		break;
	case 3:
		NAME_(set_BL)(cpssp, value);
		break;
	case 4:
		NAME_(set_AH)(cpssp, value);
		break;
	case 5:
		NAME_(set_CH)(cpssp, value);
		break;
	case 6:
		NAME_(set_DH)(cpssp, value);
		break;
	case 7:
		NAME_(set_BH)(cpssp, value);
		break;
	}
}

static void
NAME_(set_Gw)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_AX)(cpssp, value);
		break;
	case 1:
		NAME_(set_CX)(cpssp, value);
		break;
	case 2:
		NAME_(set_DX)(cpssp, value);
		break;
	case 3:
		NAME_(set_BX)(cpssp, value);
		break;
	case 4:
		NAME_(set_SP)(cpssp, value);
		break;
	case 5:
		NAME_(set_BP)(cpssp, value);
		break;
	case 6:
		NAME_(set_SI)(cpssp, value);
		break;
	case 7:
		NAME_(set_DI)(cpssp, value);
		break;
	}
}

static void
NAME_(set_Gd)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		NAME_(set_EAX)(cpssp, value);
		break;
	case 1:
		NAME_(set_ECX)(cpssp, value);
		break;
	case 2:
		NAME_(set_EDX)(cpssp, value);
		break;
	case 3:
		NAME_(set_EBX)(cpssp, value);
		break;
	case 4:
		NAME_(set_ESP)(cpssp, value);
		break;
	case 5:
		NAME_(set_EBP)(cpssp, value);
		break;
	case 6:
		NAME_(set_ESI)(cpssp, value);
		break;
	case 7:
		NAME_(set_EDI)(cpssp, value);
		break;
	}
}

static void
NAME_(set_Gv)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_Gd)(cpssp, value);
	} else {
		NAME_(set_Gw)(cpssp, value);
	}
}

/******************************************************************************/

static void
NAME_(set_Sw)(struct cpssp *cpssp, uint32_t value)
{
	switch (NAME_(get_instruction_reg)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case SEGMENT_ES:
		NAME_(set_ES)(cpssp, value);
		break;
	case SEGMENT_CS:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
	case SEGMENT_SS:
		NAME_(set_SS)(cpssp, value);
		break;
	case SEGMENT_DS:
		NAME_(set_DS)(cpssp, value);
		break;
	case SEGMENT_FS:
		NAME_(set_FS)(cpssp, value);
		break;
	case SEGMENT_GS:
		NAME_(set_GS)(cpssp, value);
		break;
	case 6:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
	case 7:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
	}
}

/******************************************************************************
 * Set Memory addressed by the ES:eDI register pair
 ******************************************************************************/

static void
NAME_(set_Yb)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(address_size_32)(cpssp)) {
		NAME_(smwb)(cpssp, NAME_(get_EDI)(cpssp), SEGMENT_ES, value);
	} else {
		NAME_(smwb)(cpssp, NAME_(get_DI)(cpssp), SEGMENT_ES, value);
	}
}

static void
NAME_(set_Yw)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(address_size_32)(cpssp)) {
		NAME_(smww)(cpssp, NAME_(get_EDI)(cpssp), SEGMENT_ES, value);
	} else {
		NAME_(smww)(cpssp, NAME_(get_DI)(cpssp), SEGMENT_ES, value);
	}
}

static void
NAME_(set_Yd)(struct cpssp *cpssp, uint32_t value)
{
	if (NAME_(address_size_32)(cpssp)) {
		NAME_(smwd)(cpssp, NAME_(get_EDI)(cpssp), SEGMENT_ES, value);
	} else {
		NAME_(smwd)(cpssp, NAME_(get_DI)(cpssp), SEGMENT_ES, value);
	}
}

/******************************************************************************
 * Store
 ******************************************************************************/

static void
NAME_(store_SS)(struct cpssp *cpssp)
{
	NAME_(set_SS)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_DS)(struct cpssp *cpssp)
{
	NAME_(set_DS)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_ES)(struct cpssp *cpssp)
{
	NAME_(set_ES)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_FS)(struct cpssp *cpssp)
{
	NAME_(set_FS)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_GS)(struct cpssp *cpssp)
{
	NAME_(set_GS)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************/

static void
NAME_(store_AL)(struct cpssp *cpssp)
{
	NAME_(set_AL)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_CL)(struct cpssp *cpssp)
{
	NAME_(set_CL)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_DL)(struct cpssp *cpssp)
{
	NAME_(set_DL)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_BL)(struct cpssp *cpssp)
{
	NAME_(set_BL)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************/

static void
NAME_(store_AH)(struct cpssp *cpssp)
{
	NAME_(set_AH)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_CH)(struct cpssp *cpssp)
{
	NAME_(set_CH)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_DH)(struct cpssp *cpssp)
{
	NAME_(set_DH)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_BH)(struct cpssp *cpssp)
{
	NAME_(set_BH)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************/

static void
NAME_(store_eAX)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_eCX)(struct cpssp *cpssp)
{
	NAME_(set_eCX)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_eDX)(struct cpssp *cpssp)
{
	NAME_(set_eDX)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_eBX)(struct cpssp *cpssp)
{
	NAME_(set_eBX)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************/

static void
NAME_(store_eSP)(struct cpssp *cpssp)
{
	NAME_(set_eSP)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_eBP)(struct cpssp *cpssp)
{
	NAME_(set_eBP)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_eSI)(struct cpssp *cpssp)
{
	NAME_(set_eSI)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_eDI)(struct cpssp *cpssp)
{
	NAME_(set_eDI)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************/

static void
NAME_(store_Cd)(struct cpssp *cpssp)
{
	NAME_(set_Cd)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Dd)(struct cpssp *cpssp)
{
	NAME_(set_Dd)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Td)(struct cpssp *cpssp)
{
	NAME_(set_Td)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Rd)(struct cpssp *cpssp)
{
	NAME_(set_Rd)(cpssp, cpssp->NAME.t0, NAME_(get_instruction_rm)(cpssp));
}

/******************************************************************************/

static void
NAME_(store_Eb)(struct cpssp *cpssp)
{
	NAME_(set_Eb)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Ew)(struct cpssp *cpssp)
{
	NAME_(set_Ew)(cpssp, cpssp->NAME.t0);
}

#if 0
static void
NAME_(store_Ed)(struct cpssp *cpssp)
{
	NAME_(set_Ed)(cpssp, cpssp->NAME.t0);
}
#endif
static void
NAME_(store_Ev)(struct cpssp *cpssp)
{
	NAME_(set_Ev)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_EvBIT)(struct cpssp *cpssp)
{
	NAME_(set_EvBIT)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************/

static void
NAME_(store_Gb)(struct cpssp *cpssp)
{
	NAME_(set_Gb)(cpssp, cpssp->NAME.t0);
}
#if 0
static void
store_Gw(void)
{
	NAME_(set_Gw)(cpssp, cpssp->NAME.t0);
}

static void
store_Gd(void)
{
	NAME_(set_Gd)(cpssp, cpssp->NAME.t0);
}
#endif
static void
NAME_(store_Gv)(struct cpssp *cpssp)
{
	NAME_(set_Gv)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Sw)(struct cpssp *cpssp)
{
	NAME_(set_Sw)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Ob)(struct cpssp *cpssp)
{
	NAME_(set_Ob)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Ov)(struct cpssp *cpssp)
{
	NAME_(set_Ov)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************
 * Store Memory addressed by the ES:eDI register pair
 ******************************************************************************/

static void
NAME_(store_Yb)(struct cpssp *cpssp)
{
	NAME_(set_Yb)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Yw)(struct cpssp *cpssp)
{
	NAME_(set_Yw)(cpssp, cpssp->NAME.t0);
}

static void
NAME_(store_Yd)(struct cpssp *cpssp)
{
	NAME_(set_Yd)(cpssp, cpssp->NAME.t0);
}

/******************************************************************************
 * Store cpssp->NAME.t1 and cpssp->NAME.t2
 ******************************************************************************/

static void
NAME_(store_eAX_eAX)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eAX)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eCX)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eCX)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eDX)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eDX)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eBX)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eBX)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eSP)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eSP)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eBP)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eBP)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eSI)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eSI)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_eAX_eDI)(struct cpssp *cpssp)
{
	NAME_(set_eAX)(cpssp, cpssp->NAME.t1);
	NAME_(set_eDI)(cpssp, cpssp->NAME.t2);
}

/******************************************************************************/

static void
NAME_(store_Eb_Gb)(struct cpssp *cpssp)
{
	NAME_(set_Eb)(cpssp, cpssp->NAME.t1);
	NAME_(set_Gb)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_Ew_Gw)(struct cpssp *cpssp)
{
	NAME_(set_Ew)(cpssp, cpssp->NAME.t1);
	NAME_(set_Gw)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_Ed_Gd)(struct cpssp *cpssp)
{
	NAME_(set_Ed)(cpssp, cpssp->NAME.t1);
	NAME_(set_Gd)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(store_Ev_Gv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(store_Ed_Gd)(cpssp);
	} else {
		NAME_(store_Ew_Gw)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(store_Ms)(struct cpssp *cpssp)
{
	uint32_t addr;
	uint8_t segment;

	switch (NAME_(get_instruction_mod)(cpssp)) {
	default:
		DEBUG_ERROR_SWITCH();
	case 0:
		addr = NAME_(get_address_mod0)(cpssp);
		segment = NAME_(get_segment_mod0)(cpssp);
		break;
	case 1:
		addr = NAME_(get_address_mod1)(cpssp);
		segment = NAME_(get_segment_mod1)(cpssp);
		break;
	case 2:
		addr = NAME_(get_address_mod2)(cpssp);
		segment = NAME_(get_segment_mod2)(cpssp);
		break;
	case 3:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	segment = NAME_(get_segment_overwritten)(cpssp, segment);

	NAME_(smww)(cpssp, addr + 0, segment, cpssp->NAME.t1); /* limit */
	NAME_(smwd)(cpssp, addr + 2, segment, cpssp->NAME.t2); /* base */
}

/******************************************************************************
 * cpssp->NAME.eip
 ******************************************************************************/

static void
NAME_(commit_EIP)(struct cpssp *cpssp)
{
	if (cpssp->NAME.eip != cpssp->NAME.eip_new) {
		cpssp->NAME.eip = cpssp->NAME.eip_new;
		cpssp->NAME.interrupt_delay = 0;
	}

	cpssp->NAME.prefix_lock_repeat = LR_NONE;
	cpssp->NAME.prefix_segment_override = SEGMENT_NONE;
	cpssp->NAME.prefix_operand_size_override = 0;
	cpssp->NAME.prefix_address_size_override = 0;
	cpssp->NAME.instruction_opcode2 = 0;
}

static void
NAME_(commit_EIP_and_delay_interrupts)(struct cpssp *cpssp)
{
	if (cpssp->NAME.eip != cpssp->NAME.eip_new) {
		cpssp->NAME.eip = cpssp->NAME.eip_new;

		if (cpssp->NAME.interrupt_delay) {
			cpssp->NAME.interrupt_delay = 0;
		} else {
			cpssp->NAME.interrupt_delay = 1;
		}
	}

	cpssp->NAME.prefix_lock_repeat = LR_NONE;
	cpssp->NAME.prefix_segment_override = SEGMENT_NONE;
	cpssp->NAME.prefix_operand_size_override = 0;
	cpssp->NAME.prefix_address_size_override = 0;
	cpssp->NAME.instruction_opcode2 = 0;
}

static void
NAME_(revert_EIP)(struct cpssp *cpssp)
{
	cpssp->NAME.eip_new = cpssp->NAME.eip;

	cpssp->NAME.prefix_lock_repeat = LR_NONE;
	cpssp->NAME.prefix_segment_override = SEGMENT_NONE;
	cpssp->NAME.prefix_operand_size_override = 0;
	cpssp->NAME.prefix_address_size_override = 0;
}

/******************************************************************************
 * Parity
 ******************************************************************************/

static bool
NAME_(parity)(struct cpssp *cpssp, uint8_t value)
{
	return ((value >> 7) & 1)
	     ^ ((value >> 6) & 1)
	     ^ ((value >> 5) & 1)
	     ^ ((value >> 4) & 1)
	     ^ ((value >> 3) & 1)
	     ^ ((value >> 2) & 1)
	     ^ ((value >> 1) & 1)
	     ^ ((value >> 0) & 1);
}

/******************************************************************************
 * I/O Permission Check
 ******************************************************************************/

static void
NAME_(ioperm_check)(struct cpssp *cpssp, uint32_t port, uint8_t size)
{
	bool access_denied;
	uint32_t tss_iopb_addr;
	uint16_t value;
	uint16_t mask;

	if ((NAME_(protected_mode)(cpssp)
	  && NAME_(get_IOPL)(cpssp) < NAME_(get_CPL)(cpssp))
	 || NAME_(virtual8086_mode)(cpssp)) {
		if (cpssp->NAME.tr_type == SEGMENT_32BIT_BUSY_TSS
		 || cpssp->NAME.tr_type == SEGMENT_32BIT_AVAIL_TSS) {
			tss_iopb_addr = NAME_(smrw)(cpssp, 0x66, SEGMENT_TSS) + (port >> 3);
			value = NAME_(smrw)(cpssp, tss_iopb_addr, SEGMENT_TSS);

			mask = ((1 << size) - 1) << (port & 7);

			access_denied = value & mask;
		} else {
			access_denied = 1;
		}

		if (access_denied) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	}
}

/******************************************************************************
 * OUT - Output to Port
 ******************************************************************************/

static void
NAME_(outb)(struct cpssp *cpssp)
{
	NAME_(ioperm_check)(cpssp, cpssp->NAME.t1 & 0xffff, 1);
	CHIP_(uiowb)(cpssp, cpssp->NAME.t1 & 0xffff, cpssp->NAME.t2 & 0xff);
}

static void
NAME_(outw)(struct cpssp *cpssp)
{
	NAME_(ioperm_check)(cpssp, cpssp->NAME.t1 & 0xffff, 2);
	CHIP_(uioww)(cpssp, cpssp->NAME.t1 & 0xffff, cpssp->NAME.t2 & 0xffff);
}

static void
NAME_(outd)(struct cpssp *cpssp)
{
	NAME_(ioperm_check)(cpssp, cpssp->NAME.t1 & 0xffff, 4);
	CHIP_(uiowd)(cpssp, cpssp->NAME.t1 & 0xffff, cpssp->NAME.t2);
}

static void
NAME_(outv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(outd)(cpssp);
	} else {
		NAME_(outw)(cpssp);
	}
}

/******************************************************************************
 * IN - Input from Port
 ******************************************************************************/

static void
NAME_(inb)(struct cpssp *cpssp)
{
	uint8_t val;

	NAME_(ioperm_check)(cpssp, cpssp->NAME.t1 & 0xffff, 1);
	CHIP_(uiorb)(cpssp, cpssp->NAME.t1 & 0xffff, &val);
	cpssp->NAME.t0 = (uint32_t) val;
}

static void
NAME_(inw)(struct cpssp *cpssp)
{
	uint16_t val;

	NAME_(ioperm_check)(cpssp, cpssp->NAME.t1 & 0xffff, 2);
	CHIP_(uiorw)(cpssp, cpssp->NAME.t1 & 0xffff, &val);
	cpssp->NAME.t0 = (uint32_t) val;
}

static void
NAME_(ind)(struct cpssp *cpssp)
{
	uint32_t val;

	NAME_(ioperm_check)(cpssp, cpssp->NAME.t1 & 0xffff, 4);
	CHIP_(uiord)(cpssp, cpssp->NAME.t1 & 0xffff, &val);
	cpssp->NAME.t0 = val;
}

static void
NAME_(inv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(ind)(cpssp);
	} else {
		NAME_(inw)(cpssp);
	}
}

/******************************************************************************
 * INC - Increment by 1
 ******************************************************************************/

static void
NAME_(sset_flags_for_inc)(struct cpssp *cpssp)
{
	/* cf unchanged */
	NAME_(sset_AF)(cpssp, (cpssp->NAME.t0 & 0xf) == 0);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(incb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 + 1) & 0xff;
	NAME_(sset_flags_for_inc)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x80);
}

static void
NAME_(incw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 + 1) & 0xffff;
	NAME_(sset_flags_for_inc)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x8000);
}

static void
NAME_(incd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 + 1;
	NAME_(sset_flags_for_inc)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x80000000);
}

static void
NAME_(incv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(incd)(cpssp);
	} else {
		NAME_(incw)(cpssp);
	}
}

/******************************************************************************
 * DEC - Decrement by 1
 ******************************************************************************/

static void
NAME_(sset_flags_for_dec)(struct cpssp *cpssp)
{
	/* cf unchanged */
	NAME_(sset_AF)(cpssp, (cpssp->NAME.t0 & 0xf) == 0xf);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(decb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 - 1) & 0xff;
	NAME_(sset_flags_for_dec)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x7f);
}

static void
NAME_(decw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 - 1) & 0xffff;
	NAME_(sset_flags_for_dec)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0xf777); /* Correct? FIXME */
}

static void
NAME_(decd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 - 1;
	NAME_(sset_flags_for_dec)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0xf7777777); /* Correct? FIXME */
}

static void
NAME_(decv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(decd)(cpssp);
	} else {
		NAME_(decw)(cpssp);
	}
}

/******************************************************************************
 * ADD - Add
 ******************************************************************************/

static void
NAME_(sset_flags_for_add)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, cpssp->NAME.t0 < cpssp->NAME.t1);
	NAME_(sset_AF)(cpssp, (cpssp->NAME.t1 ^ cpssp->NAME.t2 ^ cpssp->NAME.t0) & 0x10);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(addb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 + cpssp->NAME.t2) & 0xff;
	NAME_(sset_flags_for_add)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_OF)(cpssp, ((~(cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t2 ^ cpssp->NAME.t0)) & 0x80) != 0);
}

static void
NAME_(addw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 + cpssp->NAME.t2) & 0xffff;
	NAME_(sset_flags_for_add)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_OF)(cpssp, ((~(cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t2 ^ cpssp->NAME.t0)) & 0x8000) != 0);
}

static void
NAME_(addd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 + cpssp->NAME.t2;
	NAME_(sset_flags_for_add)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_OF)(cpssp, ((~(cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t2 ^ cpssp->NAME.t0)) & 0x80000000) != 0);
}

static void
NAME_(addv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(addd)(cpssp);
	} else {
		NAME_(addw)(cpssp);
	}
}

/******************************************************************************
 * OR - Logical Inclusive OR
 ******************************************************************************/

static void
NAME_(sset_flags_for_or)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);
	NAME_(sset_AF)(cpssp, 0); /* undefined */
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(orb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 | cpssp->NAME.t2) & 0xff;
	NAME_(sset_flags_for_or)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
}

static void
NAME_(orw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 | cpssp->NAME.t2) & 0xffff;
	NAME_(sset_flags_for_or)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
}

static void
NAME_(ord)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 | cpssp->NAME.t2;
	NAME_(sset_flags_for_or)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
}

static void
NAME_(orv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(ord)(cpssp);
	} else {
		NAME_(orw)(cpssp);
	}
}

/******************************************************************************
 * ADC - Add with Carry
 ******************************************************************************/

static void
NAME_(sset_flags_for_adc)(struct cpssp *cpssp)
{
	NAME_(sset_AF)(cpssp, (cpssp->NAME.t1 ^ cpssp->NAME.t2 ^ cpssp->NAME.t0) & 0x10);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(adcb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 + cpssp->NAME.t2 + NAME_(get_CF)(cpssp)) & 0xff;

	NAME_(sset_flags_for_adc)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_OF)(cpssp, ((~(cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t2 ^ cpssp->NAME.t0)) & 0x80) != 0);

	if (NAME_(get_CF)(cpssp)) {
		NAME_(sset_CF)(cpssp, cpssp->NAME.t0 <= cpssp->NAME.t1);
	} else {
		NAME_(sset_CF)(cpssp, cpssp->NAME.t0 < cpssp->NAME.t1);
	}
}

static void
NAME_(adcw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 + cpssp->NAME.t2 + NAME_(get_CF)(cpssp)) & 0xffff;

	NAME_(sset_flags_for_adc)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_OF)(cpssp, ((~(cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t2 ^ cpssp->NAME.t0)) & 0x8000) != 0);

	if (NAME_(get_CF)(cpssp)) {
		NAME_(sset_CF)(cpssp, cpssp->NAME.t0 <= cpssp->NAME.t1);
	} else {
		NAME_(sset_CF)(cpssp, cpssp->NAME.t0 < cpssp->NAME.t1);
	}
}

static void
NAME_(adcd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 + cpssp->NAME.t2 + NAME_(get_CF)(cpssp);

	NAME_(sset_flags_for_adc)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_OF)(cpssp, ((~(cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t2 ^ cpssp->NAME.t0)) & 0x80000000) != 0);

	if (NAME_(get_CF)(cpssp)) {
		NAME_(sset_CF)(cpssp, cpssp->NAME.t0 <= cpssp->NAME.t1);
	} else {
		NAME_(sset_CF)(cpssp, cpssp->NAME.t0 < cpssp->NAME.t1);
	}
}

static void
NAME_(adcv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(adcd)(cpssp);
	} else {
		NAME_(adcw)(cpssp);
	}
}

/******************************************************************************
 * SBB - Integer Subtraction with Borrow
 ******************************************************************************/

static void
NAME_(sset_flags_for_sbb)(struct cpssp *cpssp)
{
	NAME_(sset_AF)(cpssp, !((cpssp->NAME.t1 ^ cpssp->NAME.t2 ^ cpssp->NAME.t0) & 0x10));
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(sbbb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 - (cpssp->NAME.t2 + NAME_(get_CF)(cpssp))) & 0xff;
	NAME_(sset_flags_for_sbb)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 < cpssp->NAME.t0) || (cpssp->NAME.t2 == 0xff));
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t1 ^ cpssp->NAME.t0)) & 0x80) != 0);
}

static void
NAME_(sbbw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 - (cpssp->NAME.t2 + NAME_(get_CF)(cpssp))) & 0xffff;
	NAME_(sset_flags_for_sbb)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 < cpssp->NAME.t0) || (cpssp->NAME.t2 == 0xffff));
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t1 ^ cpssp->NAME.t0)) & 0x8000) != 0);
}

static void
NAME_(sbbd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 - (cpssp->NAME.t2 + NAME_(get_CF)(cpssp));
	NAME_(sset_flags_for_sbb)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 < cpssp->NAME.t0) || (cpssp->NAME.t2 == 0xffffffff));
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t1 ^ cpssp->NAME.t0)) & 0x80000000) != 0);
}

static void
NAME_(sbbv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(sbbd)(cpssp);
	} else {
		NAME_(sbbw)(cpssp);
	}
}

/******************************************************************************
 * AND - Logical AND
 ******************************************************************************/

static void
NAME_(sset_flags_for_and)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);
	NAME_(sset_AF)(cpssp, 0); /* undefined */
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(andb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 & cpssp->NAME.t2) & 0xff;
	NAME_(sset_flags_for_and)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
}

static void
NAME_(andw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 & cpssp->NAME.t2) & 0xffff;
	NAME_(sset_flags_for_and)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
}

static void
NAME_(andd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 & cpssp->NAME.t2;
	NAME_(sset_flags_for_and)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
}

static void
NAME_(andv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(andd)(cpssp);
	} else {
		NAME_(andw)(cpssp);
	}
}

/******************************************************************************
 * SUB - Substract
 ******************************************************************************/

static void
NAME_(sset_flags_for_sub)(struct cpssp *cpssp)
{
	NAME_(sset_AF)(cpssp, !((cpssp->NAME.t1 ^ cpssp->NAME.t2 ^ cpssp->NAME.t0) & 0x10));
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(subb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 - cpssp->NAME.t2) & 0xff;
	NAME_(sset_flags_for_sub)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 & 0xff) < (cpssp->NAME.t2 & 0xff));
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t1 ^ cpssp->NAME.t0)) & 0x80) != 0);
}

static void
NAME_(subw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 - cpssp->NAME.t2) & 0xffff;
	NAME_(sset_flags_for_sub)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 & 0xffff) < (cpssp->NAME.t2 & 0xffff));
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t1 ^ cpssp->NAME.t0)) & 0x8000) != 0);
}

static void
NAME_(subd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 - cpssp->NAME.t2;
	NAME_(sset_flags_for_sub)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_CF)(cpssp, cpssp->NAME.t1 < cpssp->NAME.t2);
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 ^ cpssp->NAME.t2) & (cpssp->NAME.t1 ^ cpssp->NAME.t0)) & 0x80000000) != 0);
}

static void
NAME_(subv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(subd)(cpssp);
	} else {
		NAME_(subw)(cpssp);
	}
}

/******************************************************************************
 * XOR - Logical Exclusive OR
 ******************************************************************************/

static void
NAME_(sset_flags_for_xor)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);
	NAME_(sset_AF)(cpssp, 0); /* undefined */
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(xorb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 ^ cpssp->NAME.t2) & 0xff;
	NAME_(sset_flags_for_xor)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
}

static void
NAME_(xorw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 ^ cpssp->NAME.t2) & 0xffff;
	NAME_(sset_flags_for_xor)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
}

static void
NAME_(xord)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 ^ cpssp->NAME.t2;
	NAME_(sset_flags_for_xor)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
}

static void
NAME_(xorv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(xord)(cpssp);
	} else {
		NAME_(xorw)(cpssp);
	}
}

/**********************************************************************
 * ALU - ALU Operations
 *********************************************************************/

static void
NAME_(alub)(struct cpssp *cpssp, uint8_t op)
{
	switch (op) {
	case 0:
		NAME_(addb)(cpssp);
		break;
	case 1:
		NAME_(orb)(cpssp);
		break;
	case 2:
		NAME_(adcb)(cpssp);
		break;
	case 3:
		NAME_(sbbb)(cpssp);
		break;
	case 4:
		NAME_(andb)(cpssp);
		break;
	case 5:
		NAME_(subb)(cpssp);
		break;
	case 6:
		NAME_(xorb)(cpssp);
		break;
	case 7:
		NAME_(subb)(cpssp);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(aluv)(struct cpssp *cpssp, uint8_t op)
{
	switch (op) {
	case 0:
		NAME_(addv)(cpssp);
		break;
	case 1:
		NAME_(orv)(cpssp);
		break;
	case 2:
		NAME_(adcv)(cpssp);
		break;
	case 3:
		NAME_(sbbv)(cpssp);
		break;
	case 4:
		NAME_(andv)(cpssp);
		break;
	case 5:
		NAME_(subv)(cpssp);
		break;
	case 6:
		NAME_(xorv)(cpssp);
		break;
	case 7:
		NAME_(subv)(cpssp);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

/**********************************************************************
 * TEST - Logical Compare
 *********************************************************************/

/* TODO: merge with AND */

static void
NAME_(sset_flags_for_test)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);
	NAME_(sset_AF)(cpssp, 0); /* undefined */
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(testb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 & cpssp->NAME.t2) & 0xff;
	NAME_(sset_flags_for_test)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
}

static void
NAME_(testw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = (cpssp->NAME.t1 & cpssp->NAME.t2) & 0xffff;
	NAME_(sset_flags_for_test)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
}

static void
NAME_(testd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 & cpssp->NAME.t2;
	NAME_(sset_flags_for_test)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
}

static void
NAME_(testv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(testd)(cpssp);
	} else {
		NAME_(testw)(cpssp);
	}
}

/******************************************************************************
 * NOT - One's Complement Negation
 ******************************************************************************/

static void
NAME_(notb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ~cpssp->NAME.t1;
}

static void
NAME_(notv)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ~cpssp->NAME.t1;
}

/******************************************************************************
 * NEG - Two's Complement Negation
 ******************************************************************************/

static void
NAME_(sset_flags_for_neg)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, cpssp->NAME.t1 != 0);
	NAME_(sset_AF)(cpssp, (cpssp->NAME.t0 & 0xf) != 0);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
}

static void
NAME_(negb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = -cpssp->NAME.t1 & 0xff;
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x80);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_flags_for_neg)(cpssp);
}

static void
NAME_(negw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = -cpssp->NAME.t1 & 0xffff;
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x8000);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_flags_for_neg)(cpssp);
}

static void
NAME_(negd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = -cpssp->NAME.t1;
	NAME_(sset_OF)(cpssp, cpssp->NAME.t0 == 0x80000000);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_flags_for_neg)(cpssp);
}

static void
NAME_(negv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(negd)(cpssp);
	} else {
		NAME_(negw)(cpssp);
	}
}

/******************************************************************************
 * MUL - Unsigned Multiply
 ******************************************************************************/

static void
NAME_(mulb)(struct cpssp *cpssp)
{
	uint8_t product8l;
	uint8_t product8h;
	uint32_t product16;

	product16 = NAME_(get_AL)(cpssp) * (uint8_t) cpssp->NAME.t1;
	product8l = product16 & 0xff;
	product8h = product16 >> 8;

	NAME_(sset_CF)(cpssp, product8h != 0);
	NAME_(sset_ZF)(cpssp, product8l == 0);
	NAME_(sset_SF)(cpssp, product8l >= 0x80);
	NAME_(sset_OF)(cpssp, product8h != 0);
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product8l));

	NAME_(set_AX)(cpssp, product16);
}

static void
NAME_(mulw)(struct cpssp *cpssp)
{
	uint16_t product16l;
	uint16_t product16h;
	uint32_t product32;

	product32 = NAME_(get_AX)(cpssp) * (uint16_t) cpssp->NAME.t1;
	product16l = product32 & 0xffff;
	product16h = product32 >> 16;

	NAME_(sset_CF)(cpssp, product16h != 0);
	NAME_(sset_ZF)(cpssp, product16l == 0);
	NAME_(sset_SF)(cpssp, product16l >= 0x8000);
	NAME_(sset_OF)(cpssp, product16h != 0);
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product16l));

	NAME_(set_AX)(cpssp, product16l);
	NAME_(set_DX)(cpssp, product16h);
}

static void
NAME_(muld)(struct cpssp *cpssp)
{
	uint32_t product32l;
	uint32_t product32h;
	uint64_t product64;

	product64 = (uint64_t) NAME_(get_EAX)(cpssp) * (uint64_t) cpssp->NAME.t1;
	product32l = product64 & 0xffffffff;
	product32h = product64 >> 32;

	NAME_(sset_CF)(cpssp, product32h != 0);
	NAME_(sset_ZF)(cpssp, product32l == 0);
	NAME_(sset_SF)(cpssp, product32l >= 0x80000000);
	NAME_(sset_OF)(cpssp, product32h != 0);
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product32l));

	NAME_(set_EAX)(cpssp, product32l);
	NAME_(set_EDX)(cpssp, product32h);
}

static void
NAME_(mulv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(muld)(cpssp);
	} else {
		NAME_(mulw)(cpssp);
	}
}

/******************************************************************************
 * IMUL - Signed Multiply
 ******************************************************************************/

static void
NAME_(imul1b)(struct cpssp *cpssp)
{
	int8_t op2;
	int8_t op1;
	int16_t product16;
	uint8_t product8l;
	uint8_t product8h;
	bool cf;

	op1 = NAME_(get_AL)(cpssp);
	op2 = cpssp->NAME.t1;

	product16 = op1 * op2;
	product8l = product16 & 0xff;
	product8h = product16 >> 8;

	cf = !(((product8l < 0x80) && (product8h == 0))
	    || ((product8l & 0x80) && (product8h == 0xff)));

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf);
	NAME_(sset_AF)(cpssp, 0);			/* undefined */
	NAME_(sset_ZF)(cpssp, product8l == 0);	/* undefined */
	NAME_(sset_SF)(cpssp, product8l >= 0x80);	/* undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product8l));	/* undefined */

	NAME_(set_AX)(cpssp, product16);
}

static void
NAME_(imul1w)(struct cpssp *cpssp)
{
	int16_t op1;
	int16_t op2;
	int32_t product32;
	uint16_t product16l;
	uint16_t product16h;
	bool cf;

	op1 = NAME_(get_AX)(cpssp);
	op2 = cpssp->NAME.t1;

	product32 = op1 * op2;
	product16l = product32 & 0xffff;
	product16h = product32 >> 16;

	cf = !(((product16l < 0x8000) && (product16h == 0))
	    || ((product16l & 0x8000) && (product16h == 0xffff)));

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf);
	NAME_(sset_AF)(cpssp, 0);			/* undefined */
	NAME_(sset_ZF)(cpssp, product16l == 0);	/* undefined */
	NAME_(sset_SF)(cpssp, product16l >= 0x8000);	/* undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product16l));	/* undefined */

	NAME_(set_AX)(cpssp, product16l);
	NAME_(set_DX)(cpssp, product16h);
}

static void
NAME_(imul1d)(struct cpssp *cpssp)
{
	int32_t op1;
	int32_t op2;
	int64_t product64;
	uint32_t product32l;
	uint32_t product32h;
	bool cf;

	op1 = NAME_(get_EAX)(cpssp);
	op2 = cpssp->NAME.t1;

	product64 = ((int64_t) op1) * ((int64_t) op2);
	product32l = product64 & 0xffffffff;
	product32h = product64 >> 32;

	cf = !(((product32l < 0x80000000) && (product32h == 0))
	    || ((product32l & 0x80000000) && (product32h == 0xffffffff)));

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf);
	NAME_(sset_AF)(cpssp, 0);				/* undefined */
	NAME_(sset_ZF)(cpssp, product32l == 0);		/* undefined */
	NAME_(sset_SF)(cpssp, product32l >= 0x80000000);	/* undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product32l));		/* undefined */

	NAME_(set_EAX)(cpssp, product32l);
	NAME_(set_EDX)(cpssp, product32h);
}

static void
NAME_(imul1v)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(imul1d)(cpssp);
	} else {
		NAME_(imul1w)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(imul3ww)(struct cpssp *cpssp)
{
	int16_t op1;
	int16_t op2;
	int32_t product32;
	uint16_t product16l;
	uint16_t product16h;
	bool cf;

	op1 = cpssp->NAME.t1;
	op2 = cpssp->NAME.t2;

	product32 = op1 * op2;
	product16l = product32 & 0xffff;
	product16h = product32 >> 16;

	cpssp->NAME.t0 = product16l;

	cf = !(((product16l < 0x8000) && (product16h == 0))
	    || ((product16l & 0x8000) && (product16h == 0xffff)));

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf);
	NAME_(sset_AF)(cpssp, 0);			/* undefined */
	NAME_(sset_SF)(cpssp, product16l >= 0x8000);	/* undefined */
	NAME_(sset_ZF)(cpssp, product16l == 0);	/* undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product16l));	/* undefined */
}

static void
NAME_(imul3dd)(struct cpssp *cpssp)
{
	int32_t op1;
	int32_t op2;
	int64_t product64;
	uint32_t product32l;
	uint32_t product32h;
	bool cf;

	op1 = cpssp->NAME.t1;
	op2 = cpssp->NAME.t2;

	product64 = ((int64_t) op1) * ((int64_t) op2);
	product32l = product64 & 0xffffffff;
	product32h = product64 >> 32;

	cpssp->NAME.t0 = product32l;

	cf = !(((product32l < 0x80000000) && (product32h == 0))
	    || ((product32l & 0x80000000) && (product32h == 0xffffffff)));

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf);
	NAME_(sset_AF)(cpssp, 0);				/* undefined */
	NAME_(sset_SF)(cpssp, product32l >= 0x80000000);	/* undefined */
	NAME_(sset_ZF)(cpssp, product32l == 0);		/* undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, product32l));		/* undefined */
}

static void
NAME_(imul3vv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(imul3dd)(cpssp);
	} else {
		NAME_(imul3ww)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(imul3wb)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = (int16_t) (int8_t) cpssp->NAME.t2;

	NAME_(imul3ww)(cpssp);
}

static void
NAME_(imul3db)(struct cpssp *cpssp)
{
	cpssp->NAME.t2 = (int32_t) (int8_t) cpssp->NAME.t2;

	NAME_(imul3dd)(cpssp);
}

static void
NAME_(imul3vb)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(imul3db)(cpssp);
	} else {
		NAME_(imul3wb)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(imul2v)(struct cpssp *cpssp)
{
	NAME_(imul3vv)(cpssp);
}

/******************************************************************************
 * DIV - Unsigned Divide
 ******************************************************************************/

static void
NAME_(sset_flags_for_div)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0); /* undefined */
	NAME_(sset_OF)(cpssp, 0); /* undefined */
	NAME_(sset_SF)(cpssp, 0); /* undefined */
	NAME_(sset_ZF)(cpssp, 0); /* undefined */
	NAME_(sset_AF)(cpssp, 0); /* undefined */
	NAME_(sset_PF)(cpssp, 0); /* undefined */
}

static void
NAME_(divb)(struct cpssp *cpssp)
{
	uint8_t remainder8;
	uint8_t quotient8;
	uint16_t operand16;
	uint16_t quotient16;

	operand16 = NAME_(get_AX)(cpssp);

	if (cpssp->NAME.t1 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	quotient16 = operand16 / (uint8_t) cpssp->NAME.t1;
	remainder8 = operand16 % (uint8_t) cpssp->NAME.t1;
	quotient8 = quotient16 & 0xff;

	if (quotient16 != quotient8) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	NAME_(sset_flags_for_div)(cpssp);

	NAME_(set_AL)(cpssp, quotient8);
	NAME_(set_AH)(cpssp, remainder8);
}

static void
NAME_(divw)(struct cpssp *cpssp)
{
	uint16_t remainder16;
	uint16_t quotient16;
	uint32_t operand32;
	uint32_t quotient32;

	operand32 = (NAME_(get_DX)(cpssp) << 16) | NAME_(get_AX)(cpssp);

	if (cpssp->NAME.t1 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	quotient32 = operand32 / (uint16_t) cpssp->NAME.t1;
	remainder16 = operand32 % (uint16_t) cpssp->NAME.t1;
	quotient16 = quotient32 & 0xffff;

	if (quotient32 != quotient16) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	NAME_(sset_flags_for_div)(cpssp);

	NAME_(set_AX)(cpssp, quotient16);
	NAME_(set_DX)(cpssp, remainder16);
}

static void
NAME_(divd)(struct cpssp *cpssp)
{
	uint32_t remainder32;
	uint32_t quotient32;
	uint64_t operand64;
	uint64_t quotient64;

	operand64 = (((uint64_t) NAME_(get_EDX)(cpssp)) << 32) + (uint64_t) NAME_(get_EAX)(cpssp);

	if (cpssp->NAME.t1 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	quotient64 = operand64 / cpssp->NAME.t1;
	remainder32 = (uint32_t) (operand64 % cpssp->NAME.t1);
	quotient32 = (uint32_t) (quotient64 & 0xffffffff);

	if (quotient64 != quotient32) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	NAME_(sset_flags_for_div)(cpssp);

	NAME_(set_EAX)(cpssp, quotient32);
	NAME_(set_EDX)(cpssp, remainder32);
}

static void
NAME_(divv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(divd)(cpssp);
	} else {
		NAME_(divw)(cpssp);
	}
}

/******************************************************************************
 * IDIV - Signed Divide
 ******************************************************************************/

static void
NAME_(sset_flags_for_idiv)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0); /* undefined */
	NAME_(sset_OF)(cpssp, 0); /* undefined */
	NAME_(sset_SF)(cpssp, 0); /* undefined */
	NAME_(sset_ZF)(cpssp, 0); /* undefined */
	NAME_(sset_AF)(cpssp, 0); /* undefined */
	NAME_(sset_PF)(cpssp, 0); /* undefined */
}

static void
NAME_(idivb)(struct cpssp *cpssp)
{
	int8_t op2;
	int8_t quotient8l;
	int8_t remainder8;
	int16_t op1;
	int16_t quotient16;

	op1 = NAME_(get_AX)(cpssp);
	op2 = cpssp->NAME.t1;

	if (op2 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	if ((op1 == ((int16_t) 0x8000)) && (op2 == -1)) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	quotient16 = op1 / op2;
	remainder8 = op1 % op2;
	quotient8l = quotient16 & 0xff;

	if (quotient16 != quotient8l) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	NAME_(sset_flags_for_idiv)(cpssp);

	NAME_(set_AL)(cpssp, quotient8l);
	NAME_(set_AH)(cpssp, remainder8);
}

static void
NAME_(idivw)(struct cpssp *cpssp)
{
	int16_t op2;
	int16_t remainder16;
	int16_t quotient16l;
	int32_t op1;
	int32_t quotient32;

	op1 = ((NAME_(get_DX)(cpssp) & 0xffff) << 16) | (NAME_(get_AX)(cpssp) & 0xffff);
	op2 = cpssp->NAME.t1;

	if (op2 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	if ((op1 == ((int32_t) 0x80000000)) && (op2 == -1)) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	quotient32 = op1 / op2;
	remainder16 = op1 % op2;
	quotient16l = quotient32 & 0xffff;

	if (quotient32 != quotient16l) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	NAME_(sset_flags_for_idiv)(cpssp);

	NAME_(set_AX)(cpssp, quotient16l);
	NAME_(set_DX)(cpssp, remainder16);
}

static void
NAME_(idivd)(struct cpssp *cpssp)
{
	int32_t op2;
	int32_t remainder32;
	int32_t quotient32l;
	int64_t op1;
	int64_t quotient64;

	op1 = ((uint64_t) NAME_(get_EDX)(cpssp) << 32) | NAME_(get_EAX)(cpssp);
	op2 = cpssp->NAME.t1;

	if (op2 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	if ((op1 == ((int64_t) 0x8000000000000000ULL)) && (op2 == -1)) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	quotient64 = op1 / op2;
	remainder32 = (int32_t) (op1 % op2);
	quotient32l = (int32_t) (quotient64 & 0xffffffff);

	if (quotient64 != quotient32l) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	NAME_(sset_flags_for_idiv)(cpssp);

	NAME_(set_EAX)(cpssp, quotient32l);
	NAME_(set_EDX)(cpssp, remainder32);
}

static void
NAME_(idivv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(idivd)(cpssp);
	} else {
		NAME_(idivw)(cpssp);
	}
}

/******************************************************************************
 * XCHG - Exchange Register/Memory with Register
 ******************************************************************************/

static void
NAME_(xchg)(struct cpssp *cpssp)
{
	uint32_t t;

	t = cpssp->NAME.t1;
	cpssp->NAME.t1 = cpssp->NAME.t2;
	cpssp->NAME.t2 = t;
}

/******************************************************************************
 * MOV
 ******************************************************************************/

static void
NAME_(movb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
}

static void
NAME_(movw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
}

static void
NAME_(movd)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
}

static void
NAME_(movv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(movd)(cpssp);
	} else {
		NAME_(movw)(cpssp);
	}
}

/******************************************************************************
 * MOVS/MOVSB/MOVSW/MOVSD - Move Data from String to String
 ******************************************************************************/

#define REP_CHECK()							\
	if (cpssp->NAME.prefix_lock_repeat == LR_REPZ) {				\
		if (NAME_(address_size_32)(cpssp)) {				\
			if (NAME_(get_ECX)(cpssp) == 0) {				\
				return;					\
			}						\
		} else {						\
			if (NAME_(get_CX)(cpssp) == 0) {				\
				return;					\
			}						\
		}							\
	}

#define REP_DECREMENT()							\
	if (cpssp->NAME.prefix_lock_repeat == LR_REPZ) {				\
		if (NAME_(address_size_32)(cpssp)) {				\
			NAME_(set_ECX)(cpssp, NAME_(get_ECX)(cpssp) - 1);				\
		} else {						\
			NAME_(set_CX)(cpssp, NAME_(get_CX)(cpssp) - 1);				\
		}							\
		NAME_(revert_EIP)(cpssp);						\
	}

#define REP_DECINC_eDIb()						\
	if (NAME_(address_size_32)(cpssp)) {					\
		if (NAME_(get_DF)(cpssp)) {						\
			NAME_(set_EDI)(cpssp, NAME_(get_EDI)(cpssp) - 1);				\
		} else {						\
			NAME_(set_EDI)(cpssp, NAME_(get_EDI)(cpssp) + 1);				\
		}							\
	} else {							\
		if (NAME_(get_DF)(cpssp)) {						\
			NAME_(set_DI)(cpssp, NAME_(get_DI)(cpssp) - 1);				\
		} else {						\
			NAME_(set_DI)(cpssp, NAME_(get_DI)(cpssp) + 1);				\
		}							\
	}

#define REP_DECINC_eSIb()						\
	if (NAME_(address_size_32)(cpssp)) {					\
		if (NAME_(get_DF)(cpssp)) {						\
			NAME_(set_ESI)(cpssp, NAME_(get_ESI)(cpssp) - 1);				\
		} else {						\
			NAME_(set_ESI)(cpssp, NAME_(get_ESI)(cpssp) + 1);				\
		}							\
	} else {							\
		if (NAME_(get_DF)(cpssp)) {						\
			NAME_(set_SI)(cpssp, NAME_(get_SI)(cpssp) - 1);				\
		} else {						\
			NAME_(set_SI)(cpssp, NAME_(get_SI)(cpssp) + 1);				\
		}							\
	}

#define REP_DECINC_eSIv()						\
	if (NAME_(operand_size_32)(cpssp)) {					\
		if (NAME_(address_size_32)(cpssp)) {				\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_ESI)(cpssp, NAME_(get_ESI)(cpssp) - 4);			\
			} else {					\
				NAME_(set_ESI)(cpssp, NAME_(get_ESI)(cpssp) + 4);			\
			}						\
		} else {						\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_SI)(cpssp, NAME_(get_SI)(cpssp) - 4);			\
			} else {					\
				NAME_(set_SI)(cpssp, NAME_(get_SI)(cpssp) + 4);			\
			}						\
		}							\
	} else {							\
		if (NAME_(address_size_32)(cpssp)) {				\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_ESI)(cpssp, NAME_(get_ESI)(cpssp) - 2);			\
			} else {					\
				NAME_(set_ESI)(cpssp, NAME_(get_ESI)(cpssp) + 2);			\
			}						\
		} else {						\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_SI)(cpssp, NAME_(get_SI)(cpssp) - 2);			\
			} else {					\
				NAME_(set_SI)(cpssp, NAME_(get_SI)(cpssp) + 2);			\
			}						\
		}							\
	}

#define REP_DECINC_eDIv()						\
	if (NAME_(operand_size_32)(cpssp)) {					\
		if (NAME_(address_size_32)(cpssp)) {				\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_EDI)(cpssp, NAME_(get_EDI)(cpssp) - 4);			\
			} else {					\
				NAME_(set_EDI)(cpssp, NAME_(get_EDI)(cpssp) + 4);			\
			}						\
		} else {						\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_DI)(cpssp, NAME_(get_DI)(cpssp) - 4);			\
			} else {					\
				NAME_(set_DI)(cpssp, NAME_(get_DI)(cpssp) + 4);			\
			}						\
		}							\
	} else {							\
		if (NAME_(address_size_32)(cpssp)) {				\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_EDI)(cpssp, NAME_(get_EDI)(cpssp) - 2);			\
			} else {					\
				NAME_(set_EDI)(cpssp, NAME_(get_EDI)(cpssp) + 2);			\
			}						\
		} else {						\
			if (NAME_(get_DF)(cpssp)) {					\
				NAME_(set_DI)(cpssp, NAME_(get_DI)(cpssp) - 2);			\
			} else {					\
				NAME_(set_DI)(cpssp, NAME_(get_DI)(cpssp) + 2);			\
			}						\
		}							\
	}

static void
NAME_(movsb)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_Xb)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_Yb)(cpssp);

	REP_DECINC_eDIb();
	REP_DECINC_eSIb();

	REP_DECREMENT();
}

static void
NAME_(movsw)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_Xw)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_Yw)(cpssp);

	REP_DECINC_eDIv();
	REP_DECINC_eSIv();

	REP_DECREMENT();
}

static void
NAME_(movsd)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_Xd)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_Yd)(cpssp);

	REP_DECINC_eDIv();
	REP_DECINC_eSIv();

	REP_DECREMENT();
}

static void
NAME_(movsv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(movsd)(cpssp);
	} else {
		NAME_(movsw)(cpssp);
	}
}

/******************************************************************************
 * LODS/LODSB/LODSW/LODSD - Load String
 ******************************************************************************/

static void
NAME_(lodsb)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_Xb)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_AL)(cpssp);

	REP_DECINC_eSIb();

	REP_DECREMENT();
}

static void
NAME_(lodsw)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_Xw)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_eAX)(cpssp);

	REP_DECINC_eSIv();

	REP_DECREMENT();
}

static void
NAME_(lodsd)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_Xd)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_eAX)(cpssp);

	REP_DECINC_eSIv();

	REP_DECREMENT();
}

static void
NAME_(lodsv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(lodsd)(cpssp);
	} else {
		NAME_(lodsw)(cpssp);
	}
}

/******************************************************************************
 * OUTS/OUTSB/OUTSW/OUTSD - Output String to Port
 ******************************************************************************/

static void
NAME_(outsb)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_DX_Xb)(cpssp);

	NAME_(outb)(cpssp);

	REP_DECINC_eSIb();

	REP_DECREMENT();
}

static void
NAME_(outsw)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_DX_Xw)(cpssp);

	NAME_(outw)(cpssp);

	REP_DECINC_eSIv();

	REP_DECREMENT();
}

static void
NAME_(outsd)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_DX_Xd)(cpssp);

	NAME_(outd)(cpssp);

	REP_DECINC_eSIv();

	REP_DECREMENT();
}

static void
NAME_(outsv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(outsd)(cpssp);
	} else {
		NAME_(outsw)(cpssp);
	}
}

/******************************************************************************
 * STOS/STOSB/STOSW/STOSD - Store String
 ******************************************************************************/

static void
NAME_(stosb)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_AL)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_Yb)(cpssp);

	REP_DECINC_eDIb();

	REP_DECREMENT();
}

static void
NAME_(stosw)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_eAX)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_Yw)(cpssp);

	REP_DECINC_eDIv();

	REP_DECREMENT();
}

static void
NAME_(stosd)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_eAX)(cpssp);
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(store_Yd)(cpssp);

	REP_DECINC_eDIv();

	REP_DECREMENT();
}

static void
NAME_(stosv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(stosd)(cpssp);
	} else {
		NAME_(stosw)(cpssp);
	}
}

/******************************************************************************
 * INS/INSB/INSW/INSD - Input from Port to String
 ******************************************************************************/

static void
NAME_(insb)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_DX)(cpssp);
	NAME_(inb)(cpssp);
	NAME_(store_Yb)(cpssp);

	REP_DECINC_eDIb();

	REP_DECREMENT();
}

static void
NAME_(insw)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_DX)(cpssp);
	NAME_(inw)(cpssp);
	NAME_(store_Yw)(cpssp);

	REP_DECINC_eDIv();

	REP_DECREMENT();
}

static void
NAME_(insd)(struct cpssp *cpssp)
{
	REP_CHECK();

	NAME_(load_DX)(cpssp);
	NAME_(ind)(cpssp);
	NAME_(store_Yd)(cpssp);

	REP_DECINC_eDIv();

	REP_DECREMENT();
}

static void
NAME_(insv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(insd)(cpssp);
	} else {
		NAME_(insw)(cpssp);
	}
}

/******************************************************************************
 * CMPS/CMPSB/CMPSW/CMPSD - Compare String Operands
 ******************************************************************************/

#define REPZ_REPNZ_CHECK()						\
	if ((cpssp->NAME.prefix_lock_repeat == LR_REPZ)				\
	 || (cpssp->NAME.prefix_lock_repeat == LR_REPNZ)) {				\
		if (NAME_(address_size_32)(cpssp)) {				\
			if (NAME_(get_ECX)(cpssp) == 0) {				\
				return;					\
			}						\
		} else {						\
			if (NAME_(get_CX)(cpssp) == 0) {				\
				return;					\
			}						\
		}							\
	}

#define REPZ_REPNZ_DECREMENT()						\
	NAME_(commit_OSZAPC)(cpssp);						\
	if ((cpssp->NAME.prefix_lock_repeat == LR_REPZ)				\
	 || (cpssp->NAME.prefix_lock_repeat == LR_REPNZ)) {				\
		break_condition = NAME_(get_ZF)(cpssp);				\
		if (cpssp->NAME.prefix_lock_repeat == LR_REPZ) {			\
			break_condition = !break_condition;		\
		}							\
		if (!break_condition) {					\
			NAME_(revert_EIP)(cpssp);					\
		}							\
		if (NAME_(address_size_32)(cpssp)) {				\
			NAME_(set_ECX)(cpssp, NAME_(get_ECX)(cpssp) - 1);				\
		} else {						\
			NAME_(set_CX)(cpssp, NAME_(get_CX)(cpssp) - 1);				\
		}							\
	}

static void
NAME_(cmpsb)(struct cpssp *cpssp)
{
	bool break_condition;			\

	REPZ_REPNZ_CHECK();

	NAME_(load_Xb_Yb)(cpssp);

	NAME_(subb)(cpssp);

	REP_DECINC_eDIb();
	REP_DECINC_eSIb();

	REPZ_REPNZ_DECREMENT();
}

static void
NAME_(cmpsw)(struct cpssp *cpssp)
{
	bool break_condition;			\

	REPZ_REPNZ_CHECK();

	NAME_(load_Xw_Yw)(cpssp);

	NAME_(subw)(cpssp);

	REP_DECINC_eSIv();
	REP_DECINC_eDIv();

	REPZ_REPNZ_DECREMENT();
}

static void
NAME_(cmpsd)(struct cpssp *cpssp)
{
	bool break_condition;			\

	REPZ_REPNZ_CHECK();

	NAME_(load_Xd_Yd)(cpssp);

	NAME_(subd)(cpssp);

	REP_DECINC_eSIv();
	REP_DECINC_eDIv();

	REPZ_REPNZ_DECREMENT();
}

static void
NAME_(cmpsv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(cmpsd)(cpssp);
	} else {
		NAME_(cmpsw)(cpssp);
	}
}

/******************************************************************************
 * SCAS/SCASB/SCASW/SCASD - Scan String
 ******************************************************************************/

static void
NAME_(scasb)(struct cpssp *cpssp)
{
	bool break_condition;			\

	REPZ_REPNZ_CHECK();

	NAME_(load_AL_Yb)(cpssp);

	NAME_(subb)(cpssp);

	REP_DECINC_eDIb();

	REPZ_REPNZ_DECREMENT();
}

static void
NAME_(scasw)(struct cpssp *cpssp)
{
	bool break_condition;			\

	REPZ_REPNZ_CHECK();

	NAME_(load_AX_Yw)(cpssp);

	NAME_(subw)(cpssp);

	REP_DECINC_eDIv();

	REPZ_REPNZ_DECREMENT();
}

static void
NAME_(scasd)(struct cpssp *cpssp)
{
	bool break_condition;			\

	REPZ_REPNZ_CHECK();

	NAME_(load_EAX_Yd)(cpssp);

	NAME_(subd)(cpssp);

	REP_DECINC_eDIv();

	REPZ_REPNZ_DECREMENT();
}

static void
NAME_(scasv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(scasd)(cpssp);
	} else {
		NAME_(scasw)(cpssp);
	}
}

/******************************************************************************
 * LEA
 ******************************************************************************/

static void
NAME_(lea)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
}

/******************************************************************************
 * MOVZX - Move with Zero-Extend
 ******************************************************************************/

static void
NAME_(movzxb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 & 0xff;
}

static void
NAME_(movzxw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1 & 0xffff;
}

/******************************************************************************
 * MOVSX - Move with Sign-Extension
 ******************************************************************************/

static void
NAME_(movsxb)(struct cpssp *cpssp)
{
	int16_t t;

	t = (int8_t) cpssp->NAME.t1;
	cpssp->NAME.t0 = (uint32_t) t;
}

static void
NAME_(movsxw)(struct cpssp *cpssp)
{
	int32_t t;

	if (NAME_(operand_size_32)(cpssp)) {
		t = (int16_t) cpssp->NAME.t1;
	} else {
		t = (int8_t) cpssp->NAME.t1;
	}

	cpssp->NAME.t0 = (uint32_t) t;
}

/******************************************************************************
 * LDS/LES/LFS/LGS/LSS
 ******************************************************************************/

static void
NAME_(lds)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(set_DS)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(les)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(set_ES)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(lfs)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(set_FS)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(lgs)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(set_GS)(cpssp, cpssp->NAME.t2);
}

static void
NAME_(lss)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = cpssp->NAME.t1;
	NAME_(set_SS)(cpssp, cpssp->NAME.t2);
}

/******************************************************************************
 * PUSH - Push Word or Doubleword Onto the Stack
 ******************************************************************************/


static void
NAME_(push32)(struct cpssp *cpssp, uint32_t value)
{
	uint32_t temp_esp;
	uint16_t temp_sp;

	if (NAME_(stack_size_32)(cpssp)) {
		temp_esp = NAME_(get_ESP)(cpssp) - 4;
		NAME_(smwd)(cpssp, temp_esp, SEGMENT_SS, value);
		NAME_(set_ESP)(cpssp, temp_esp);
	} else {
		temp_sp = NAME_(get_SP)(cpssp) - 4;
		NAME_(smwd)(cpssp, temp_sp, SEGMENT_SS, value);
		NAME_(set_SP)(cpssp, temp_sp);
	}
}

static void
NAME_(push16)(struct cpssp *cpssp, uint16_t value)
{
	uint32_t temp_esp;
	uint16_t temp_sp;

	if (NAME_(stack_size_32)(cpssp)) {
		temp_esp = NAME_(get_ESP)(cpssp) - 2;
		NAME_(smww)(cpssp, temp_esp, SEGMENT_SS, value);
		NAME_(set_ESP)(cpssp, temp_esp);
	} else {
		temp_sp = NAME_(get_SP)(cpssp) - 2;
		NAME_(smww)(cpssp, temp_sp, SEGMENT_SS, value);
		NAME_(set_SP)(cpssp, temp_sp);
	}
}

static void
NAME_(pushv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(push32)(cpssp, cpssp->NAME.t1);
	} else {
		NAME_(push16)(cpssp, cpssp->NAME.t1);
	}
}

/******************************************************************************
 * PUSHA - Push All General-Purpose Registers
 ******************************************************************************/

static void
NAME_(pushav)(struct cpssp *cpssp)
{
	uint32_t temp_esp;
	uint16_t temp_sp;

	if (NAME_(operand_size_32)(cpssp)) {
		temp_esp = NAME_(get_ESP)(cpssp);

		NAME_(smwd)(cpssp, temp_esp - 1 * 4, SEGMENT_SS, NAME_(get_EAX)(cpssp));
		NAME_(smwd)(cpssp, temp_esp - 2 * 4, SEGMENT_SS, NAME_(get_ECX)(cpssp));
		NAME_(smwd)(cpssp, temp_esp - 3 * 4, SEGMENT_SS, NAME_(get_EDX)(cpssp));
		NAME_(smwd)(cpssp, temp_esp - 4 * 4, SEGMENT_SS, NAME_(get_EBX)(cpssp));
		NAME_(smwd)(cpssp, temp_esp - 5 * 4, SEGMENT_SS, temp_esp);
		NAME_(smwd)(cpssp, temp_esp - 6 * 4, SEGMENT_SS, NAME_(get_EBP)(cpssp));
		NAME_(smwd)(cpssp, temp_esp - 7 * 4, SEGMENT_SS, NAME_(get_ESI)(cpssp));
		NAME_(smwd)(cpssp, temp_esp - 8 * 4, SEGMENT_SS, NAME_(get_EDI)(cpssp));

		NAME_(set_ESP)(cpssp, temp_esp - 8 * 4);
	} else {
		temp_sp = NAME_(get_SP)(cpssp);

		NAME_(smww)(cpssp, temp_sp - 1 * 2, SEGMENT_SS, NAME_(get_AX)(cpssp));
		NAME_(smww)(cpssp, temp_sp - 2 * 2, SEGMENT_SS, NAME_(get_CX)(cpssp));
		NAME_(smww)(cpssp, temp_sp - 3 * 2, SEGMENT_SS, NAME_(get_DX)(cpssp));
		NAME_(smww)(cpssp, temp_sp - 4 * 2, SEGMENT_SS, NAME_(get_BX)(cpssp));
		NAME_(smww)(cpssp, temp_sp - 5 * 2, SEGMENT_SS, temp_sp);
		NAME_(smww)(cpssp, temp_sp - 6 * 2, SEGMENT_SS, NAME_(get_BP)(cpssp));
		NAME_(smww)(cpssp, temp_sp - 7 * 2, SEGMENT_SS, NAME_(get_SI)(cpssp));
		NAME_(smww)(cpssp, temp_sp - 8 * 2, SEGMENT_SS, NAME_(get_DI)(cpssp));

		NAME_(set_SP)(cpssp, temp_sp - 8 * 2);
	}
}

/******************************************************************************
 * POP - Pop a Value from the Stack
 ******************************************************************************/

static uint32_t
NAME_(pop32)(struct cpssp *cpssp)
{
	uint32_t value;
	uint32_t temp_esp;
	uint16_t temp_sp;

	if (NAME_(stack_size_32)(cpssp)) {
		temp_esp = NAME_(get_ESP)(cpssp);
		value = NAME_(smrd)(cpssp, temp_esp, SEGMENT_SS);
		NAME_(set_ESP)(cpssp, temp_esp + 4);
	} else {
		temp_sp = NAME_(get_SP)(cpssp);
		value = NAME_(smrd)(cpssp, temp_sp, SEGMENT_SS);
		NAME_(set_SP)(cpssp, temp_sp + 4);
	}

	return value;
}

static uint16_t
NAME_(pop16)(struct cpssp *cpssp)
{
	uint16_t value;
	uint32_t temp_esp;
	uint16_t temp_sp;

	if (NAME_(stack_size_32)(cpssp)) {
		temp_esp = NAME_(get_ESP)(cpssp);
		value = NAME_(smrw)(cpssp, temp_esp, SEGMENT_SS);
		NAME_(set_ESP)(cpssp, temp_esp + 2);
	} else {
		temp_sp = NAME_(get_SP)(cpssp);
		value = NAME_(smrw)(cpssp, temp_sp, SEGMENT_SS);
		NAME_(set_SP)(cpssp, temp_sp + 2);
	}

	return value;
}

static void
NAME_(popv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.t0 = NAME_(pop32)(cpssp);
	} else {
		cpssp->NAME.t0 = NAME_(pop16)(cpssp);
	}
}

/******************************************************************************
 * POPA/POPAD - Pop All General-Purpose Registers
 ******************************************************************************/

static void
NAME_(popav)(struct cpssp *cpssp)
{
	uint32_t temp_edi;
	uint32_t temp_esi;
	uint32_t temp_ebp;
	uint32_t temp_esp;
	uint32_t temp_ebx;
	uint32_t temp_edx;
	uint32_t temp_ecx;
	uint32_t temp_eax;
	uint16_t temp_di;
	uint16_t temp_si;
	uint16_t temp_bp;
	uint16_t temp_sp;
	uint16_t temp_bx;
	uint16_t temp_dx;
	uint16_t temp_cx;
	uint16_t temp_ax;

	if (NAME_(operand_size_32)(cpssp)) {
		temp_esp = NAME_(get_ESP)(cpssp) + 8 * 4;

		temp_eax = NAME_(smrd)(cpssp, temp_esp - 1 * 4, SEGMENT_SS);
		temp_ecx = NAME_(smrd)(cpssp, temp_esp - 2 * 4, SEGMENT_SS);
		temp_edx = NAME_(smrd)(cpssp, temp_esp - 3 * 4, SEGMENT_SS);
		temp_ebx = NAME_(smrd)(cpssp, temp_esp - 4 * 4, SEGMENT_SS);
		/* temp_esp */
		temp_ebp = NAME_(smrd)(cpssp, temp_esp - 6 * 4, SEGMENT_SS);
		temp_esi = NAME_(smrd)(cpssp, temp_esp - 7 * 4, SEGMENT_SS);
		temp_edi = NAME_(smrd)(cpssp, temp_esp - 8 * 4, SEGMENT_SS);

		NAME_(set_EAX)(cpssp, temp_eax);
		NAME_(set_ECX)(cpssp, temp_ecx);
		NAME_(set_EDX)(cpssp, temp_edx);
		NAME_(set_EBX)(cpssp, temp_ebx);
		NAME_(set_ESP)(cpssp, temp_esp);
		NAME_(set_EBP)(cpssp, temp_ebp);
		NAME_(set_ESI)(cpssp, temp_esi);
		NAME_(set_EDI)(cpssp, temp_edi);
	} else {
		temp_sp = NAME_(get_SP)(cpssp) + 8 * 2;

		temp_ax = NAME_(smrw)(cpssp, temp_sp - 1 * 2, SEGMENT_SS);
		temp_cx = NAME_(smrw)(cpssp, temp_sp - 2 * 2, SEGMENT_SS);
		temp_dx = NAME_(smrw)(cpssp, temp_sp - 3 * 2, SEGMENT_SS);
		temp_bx = NAME_(smrw)(cpssp, temp_sp - 4 * 2, SEGMENT_SS);
		/* temp_sp */
		temp_bp = NAME_(smrw)(cpssp, temp_sp - 6 * 2, SEGMENT_SS);
		temp_si = NAME_(smrw)(cpssp, temp_sp - 7 * 2, SEGMENT_SS);
		temp_di = NAME_(smrw)(cpssp, temp_sp - 8 * 2, SEGMENT_SS);

		NAME_(set_AX)(cpssp, temp_ax);
		NAME_(set_CX)(cpssp, temp_cx);
		NAME_(set_DX)(cpssp, temp_dx);
		NAME_(set_BX)(cpssp, temp_bx);
		NAME_(set_SP)(cpssp, temp_sp);
		NAME_(set_BP)(cpssp, temp_bp);
		NAME_(set_SI)(cpssp, temp_si);
		NAME_(set_DI)(cpssp, temp_di);
	}
}

/******************************************************************************
 * ENTER - Make Stack Frame for Procedure Parameters
 ******************************************************************************/

static void
NAME_(enter)(struct cpssp *cpssp)
{
	uint32_t frame_temp;

	cpssp->NAME.t2 = cpssp->NAME.t2 & 31;

	if (NAME_(stack_size_32)(cpssp)) {
		NAME_(push32)(cpssp, NAME_(get_EBP)(cpssp));
		frame_temp = NAME_(get_ESP)(cpssp);
	} else {
		NAME_(push16)(cpssp, NAME_(get_BP)(cpssp));
		frame_temp = NAME_(get_SP)(cpssp);
	}

	if (cpssp->NAME.t2 > 1) {
		/* TODO: needs a loop with memory accesses to the stack */
		ERROR_NYI();
	}

	if (cpssp->NAME.t2 != 0) {
		if (NAME_(operand_size_32)(cpssp)) {
			NAME_(push32)(cpssp, frame_temp);
		} else {
			NAME_(push16)(cpssp, frame_temp & 0xffff);
		}
	}

	if (NAME_(stack_size_32)(cpssp)) {
		NAME_(set_EBP)(cpssp, frame_temp);
		NAME_(set_ESP)(cpssp, NAME_(get_ESP)(cpssp) - cpssp->NAME.t1);
	} else {
		NAME_(set_BP)(cpssp, frame_temp & 0xffff);
		NAME_(set_SP)(cpssp, NAME_(get_SP)(cpssp) - (cpssp->NAME.t1 & 0xffff));
	}
}

/******************************************************************************
 * BOUND - Check Array Index Agains Bounds
 ******************************************************************************/

static void
NAME_(bound32)(struct cpssp *cpssp)
{
	int32_t index;
	int32_t lower;
	int32_t upper;

	index = (int32_t) cpssp->NAME.t1;
	lower = (int32_t) cpssp->NAME.t2;
	upper = (int32_t) cpssp->NAME.t3 + 4;

	if (index < lower || upper < index) {
		NAME_(exception)(cpssp, EXCEPTION_BR, -1);
	}
}

static void
NAME_(bound16)(struct cpssp *cpssp)
{
	int16_t index;
	int16_t lower;
	int16_t upper;

	index = (int16_t) cpssp->NAME.t1;
	lower = (int16_t) cpssp->NAME.t2;
	upper = (int16_t) cpssp->NAME.t3 + 2;

	if (index < lower || upper < index) {
		NAME_(exception)(cpssp, EXCEPTION_BR, -1);
	}
}

static void
NAME_(boundv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(bound32)(cpssp);
	} else {
		NAME_(bound16)(cpssp);
	}
}

#if 80686 <= CONFIG_CPU

/**********************************************************************
 * CMOVcc - Move if Condition is Met
 **********************************************************************/

static void
cmovo(void)
{
	cpssp->NAME.t0 = NAME_(get_OF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovno(void)
{
	cpssp->NAME.t0 = ! NAME_(get_OF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovb(void)
{
	cpssp->NAME.t0 = NAME_(get_CF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovnb(void)
{
	cpssp->NAME.t0 = ! NAME_(get_CF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovz(void)
{
	cpssp->NAME.t0 = NAME_(get_ZF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovnz(void)
{
	cpssp->NAME.t0 = ! NAME_(get_ZF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovbe(void)
{
	cpssp->NAME.t0 = (NAME_(get_ZF)(cpssp) || NAME_(get_ZF)(cpssp)) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovnbe(void)
{
	cpssp->NAME.t0 = ! (NAME_(get_ZF)(cpssp) || NAME_(get_ZF)(cpssp)) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovs(void)
{
	cpssp->NAME.t0 = NAME_(get_SF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovns(void)
{
	cpssp->NAME.t0 = ! NAME_(get_SF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovp(void)
{
	cpssp->NAME.t0 = NAME_(get_PF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovnp(void)
{
	cpssp->NAME.t0 = ! NAME_(get_PF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovl(void)
{
	cpssp->NAME.t0 = NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovnl(void)
{
	cpssp->NAME.t0 = NAME_(get_SF)(cpssp) == NAME_(get_OF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovle(void)
{
	cpssp->NAME.t0 = NAME_(get_ZF)(cpssp) || NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

static void
cmovnle(void)
{
	cpssp->NAME.t0 = ! NAME_(get_ZF)(cpssp) && NAME_(get_SF)(cpssp) == NAME_(get_OF)(cpssp) ? cpssp->NAME.t2 : cpssp->NAME.t1;
}

#endif /* 80686 <= CONFIG_CPU */

/**********************************************************************
 * Jcc - Jump if Condition is Met
 **********************************************************************/

static void
NAME_(jcc)(struct cpssp *cpssp)
{
	cpssp->NAME.eip_new += cpssp->NAME.tj;

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

static void
NAME_(jo)(struct cpssp *cpssp)
{
	if (NAME_(get_OF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jno)(struct cpssp *cpssp)
{
	if (! NAME_(get_OF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jb)(struct cpssp *cpssp)
{
	if (NAME_(get_CF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jnb)(struct cpssp *cpssp)
{
	if (! NAME_(get_CF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jz)(struct cpssp *cpssp)
{
	if (NAME_(get_ZF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jnz)(struct cpssp *cpssp)
{
	if (! NAME_(get_ZF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jbe)(struct cpssp *cpssp)
{
	if (NAME_(get_CF)(cpssp) || NAME_(get_ZF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jnbe)(struct cpssp *cpssp)
{
	if (! (NAME_(get_CF)(cpssp) || NAME_(get_ZF)(cpssp))) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(js)(struct cpssp *cpssp)
{
	if (NAME_(get_SF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jns)(struct cpssp *cpssp)
{
	if (! NAME_(get_SF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jp)(struct cpssp *cpssp)
{
	if (NAME_(get_PF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jnp)(struct cpssp *cpssp)
{
	if (! NAME_(get_PF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jl)(struct cpssp *cpssp)
{
	if (NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jnl)(struct cpssp *cpssp)
{
	if (NAME_(get_SF)(cpssp) == NAME_(get_OF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jle)(struct cpssp *cpssp)
{
	if (NAME_(get_ZF)(cpssp) || NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp)) {
		NAME_(jcc)(cpssp);
	}
}

static void
NAME_(jnle)(struct cpssp *cpssp)
{
	if (! (NAME_(get_ZF)(cpssp) || NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp))) {
		NAME_(jcc)(cpssp);
	}
}

/******************************************************************************
 * SETcc - Set Byte on Condition
 ******************************************************************************/

static void
NAME_(seto)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_OF)(cpssp);
}

static void
NAME_(setno)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! NAME_(get_OF)(cpssp);
}

static void
NAME_(setb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_CF)(cpssp);
}

static void
NAME_(setnb)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! NAME_(get_CF)(cpssp);
}

static void
NAME_(setz)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_ZF)(cpssp);
}

static void
NAME_(setnz)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! NAME_(get_ZF)(cpssp);
}

static void
NAME_(setbe)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_CF)(cpssp) || NAME_(get_ZF)(cpssp);
}

static void
NAME_(setnbe)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! (NAME_(get_CF)(cpssp) || NAME_(get_ZF)(cpssp));
}

static void
NAME_(sets)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_SF)(cpssp);
}

static void
NAME_(setns)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! NAME_(get_SF)(cpssp);
}

static void
NAME_(setp)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_PF)(cpssp);
}

static void
NAME_(setnp)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! NAME_(get_PF)(cpssp);
}

static void
NAME_(setl)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp);
}

static void
NAME_(setnl)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_SF)(cpssp) == NAME_(get_OF)(cpssp);
}

static void
NAME_(setle)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_ZF)(cpssp) || NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp);
}

static void
NAME_(setnle)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = ! (NAME_(get_ZF)(cpssp) || NAME_(get_SF)(cpssp) != NAME_(get_OF)(cpssp));
}

/******************************************************************************
 * LAHF - Load Status Flags into AH Register
 ******************************************************************************/

static void
NAME_(lahf)(struct cpssp *cpssp)
{
	NAME_(set_AH)(cpssp, NAME_(get_EFLAGS8)(cpssp));
}

/******************************************************************************
 * SAHF - Store AH into Flags
 ******************************************************************************/

static void
NAME_(sahf)(struct cpssp *cpssp)
{
	NAME_(sset_EFLAGS8)(cpssp, NAME_(get_AH)(cpssp));
	NAME_(commit_EFLAGS8)(cpssp);
}

/******************************************************************************
 * PUSHF/PUSHFD - Push EFLAGS Register onto the Stack
 ******************************************************************************/

static void
NAME_(pushf)(struct cpssp *cpssp)
{
	if (NAME_(virtual8086_mode)(cpssp)) {
		if (NAME_(get_IOPL)(cpssp) < 3) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	}

	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(push32)(cpssp, NAME_(get_EFLAGS32)(cpssp) & 0x00fcffff);
	} else {
		NAME_(push16)(cpssp, NAME_(get_EFLAGS16)(cpssp));
	}
}

/******************************************************************************
 * POPF/POPFD - Pop Stack into EFLAGS Register
 ******************************************************************************/

static void
NAME_(popf)(struct cpssp *cpssp)
{
	uint32_t mask;

	mask = 0;
	mask |= 1 << EFLAG_CF;
	mask |= 1 << EFLAG_PF;
	mask |= 1 << EFLAG_AF;
	mask |= 1 << EFLAG_ZF;
	mask |= 1 << EFLAG_SF;
	mask |= 1 << EFLAG_TF;
	mask |= 1 << EFLAG_DF;
	mask |= 1 << EFLAG_OF;
	mask |= 1 << EFLAG_NT;

	if (NAME_(real_mode)(cpssp)) {
		mask |= 1 << EFLAG_IF;
		mask |= 3 << EFLAG_IOPL;
	} else if (NAME_(protected_mode)(cpssp)) {
		if (NAME_(get_CPL)(cpssp) == 0) {
			mask |= 3 << EFLAG_IOPL;
		}
		if (NAME_(get_CPL)(cpssp) <= NAME_(get_IOPL)(cpssp)) {
			mask |= 1 << EFLAG_IF;
		}
	} else { /* virtual8086_mode */
		if (NAME_(get_IOPL)(cpssp) < 3) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
		mask |= 1 << EFLAG_IF;
	}

	if (NAME_(operand_size_32)(cpssp)) {
		mask |= 1 << EFLAG_RF;
		mask |= 1 << EFLAG_AC;

		NAME_(sset_EFLAGS32)(cpssp, (NAME_(pop32)(cpssp) & mask) | (NAME_(get_EFLAGS32)(cpssp) & ~mask));
		NAME_(commit_EFLAGS32)(cpssp);
	} else {
		NAME_(sset_EFLAGS16)(cpssp, (NAME_(pop16)(cpssp) & mask) | (NAME_(get_EFLAGS16)(cpssp) & ~mask));
		NAME_(commit_EFLAGS16)(cpssp);
	}
}

/******************************************************************************
 * CWD/CDQ - Convert Word to Doubleword/Convert Doubleword to Quadword
 ******************************************************************************/

static void
NAME_(cwd)(struct cpssp *cpssp)
{
	if (NAME_(get_AX)(cpssp) & 0x8000) {
		NAME_(set_DX)(cpssp, 0xffff);
	} else {
		NAME_(set_DX)(cpssp, 0x0000);
	}
}

static void
NAME_(cdq)(struct cpssp *cpssp)
{
	if (NAME_(get_EAX)(cpssp) & 0x80000000) {
		NAME_(set_EDX)(cpssp, 0xffffffff);
	} else {
		NAME_(set_EDX)(cpssp, 0x00000000);
	}
}

/******************************************************************************
 * CBW/CWDE - Convert Byte to Word/Convert Word to Doubleword
 ******************************************************************************/

static void
NAME_(cbw)(struct cpssp *cpssp)
{
	NAME_(set_AX)(cpssp, (int8_t) NAME_(get_AL)(cpssp));
}

static void
NAME_(cwde)(struct cpssp *cpssp)
{
	NAME_(set_EAX)(cpssp, (int16_t) NAME_(get_AX)(cpssp));
}

/******************************************************************************
 * AAA - ASCII Adjust After Addition
 ******************************************************************************/

static void
NAME_(aaa)(struct cpssp *cpssp)
{
	/* TODO */
	ERROR_NYI();
}

/******************************************************************************
 * AAS - ASCII Adjust AL After Substraction
 ******************************************************************************/

static void
NAME_(aas)(struct cpssp *cpssp)
{
	/* TODO */
	ERROR_NYI();
}

/******************************************************************************
 * DAA - Decimal Adjust AL after Addition
 *
 * Flags:	CF, AF, SF, ZF, PF, OF
 * Exceptions:	None
 ******************************************************************************/

static void
NAME_(daa)(struct cpssp *cpssp)
{
	uint8_t al;
	bool af;
	bool cf;

	al = NAME_(get_AL)(cpssp);
	af = NAME_(get_AF)(cpssp);
	cf = NAME_(get_CF)(cpssp);

	if (((al & 0x0f) > 9) || af) {
		af = 1;
		al += 6;
	} else {
		af = 0;
	}

	if ((al > 0x9f) || cf) {
		al += 0x60;
		cf = 1;
	} else {
		cf = 0;
	}

	NAME_(set_AL)(cpssp, al);
	NAME_(sset_AF)(cpssp, af);
	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, al));
	NAME_(sset_ZF)(cpssp, al == 0);
	NAME_(sset_SF)(cpssp, al >= 0x80);
	NAME_(sset_OF)(cpssp, 0); /* undefined */
}

/******************************************************************************
 * DAS - Decimal Adjust AL after Subtraction
 ******************************************************************************/

static void
NAME_(das)(struct cpssp *cpssp)
{
	uint8_t al;
	uint8_t al_old;
	bool af;
	bool cf;
	bool cf_old;

	al = NAME_(get_AL)(cpssp);
	af = NAME_(get_AF)(cpssp);
	cf = 0;

	al_old = al;
	cf_old = NAME_(get_CF)(cpssp);

	if (((al & 0x0f) > 9) || af) {
		af = 1;
		cf = cf_old || (al < 6);
		al = al - 6;
	} else {
		af = 0;
	}

	if ((al_old > 0x99) || cf_old) {
		al = al - 0x60;
		cf = 1;
	}

	NAME_(set_AL)(cpssp, al);
	NAME_(sset_AF)(cpssp, af);
	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, al));
	NAME_(sset_ZF)(cpssp, al == 0);
	NAME_(sset_SF)(cpssp, al >= 0x80);
	NAME_(sset_OF)(cpssp, 0); /* undefined */
}

/******************************************************************************
 * AAM - ASCII Adjust AX After Multiply
 ******************************************************************************/

static void
NAME_(aam)(struct cpssp *cpssp)
{
	uint8_t al;

	if (cpssp->NAME.t1 == 0) {
		NAME_(exception)(cpssp, EXCEPTION_DE, -1);
	}

	al = NAME_(get_AL)(cpssp);
	NAME_(set_AH)(cpssp, al / (uint8_t) cpssp->NAME.t1);
	al = al % (uint8_t) cpssp->NAME.t1;
	NAME_(set_AL)(cpssp, al);

	NAME_(sset_OF)(cpssp, 0); /* Undefined */
	NAME_(sset_AF)(cpssp, 0); /* Undefined */
	NAME_(sset_CF)(cpssp, 0); /* Undefined */
	NAME_(sset_SF)(cpssp, al >= 0x80);
	NAME_(sset_ZF)(cpssp, al == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, al));
}

/******************************************************************************
 * AAD - ASCII Adjust AX Before Division
 ******************************************************************************/

static void
NAME_(aad)(struct cpssp *cpssp)
{
	uint16_t tmp;

	tmp = NAME_(get_AH)(cpssp);
	tmp *= (uint8_t) cpssp->NAME.t1;
	tmp += NAME_(get_AL)(cpssp);
	tmp &= 0xff;

	NAME_(set_AL)(cpssp, tmp);
	NAME_(set_AH)(cpssp, 0);

	NAME_(sset_OF)(cpssp, 0); /* Undefined */
	NAME_(sset_AF)(cpssp, 0); /* Undefined */
	NAME_(sset_CF)(cpssp, 0); /* Undefined */
	NAME_(sset_SF)(cpssp, tmp >= 0x80);
	NAME_(sset_ZF)(cpssp, tmp == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, tmp));
}

/******************************************************************************
 * WAIT/FWAIT - Wait
 ******************************************************************************/

static void
NAME_(fwait)(struct cpssp *cpssp)
{
	if (NAME_(get_TS)(cpssp) && NAME_(get_MP)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_NM, -1);
	}
}

/******************************************************************************
 * LOOP/LOOPcc - Loop According to ECX Counter
 ******************************************************************************/

static void
NAME_(loop)(struct cpssp *cpssp)
{
	uint32_t count;

	if (NAME_(address_size_32)(cpssp)) {
		count = NAME_(get_ECX)(cpssp) - 1;
		NAME_(set_ECX)(cpssp, count);
	} else {
		count = NAME_(get_CX)(cpssp) - 1;
		NAME_(set_CX)(cpssp, count);
	}

	if (count != 0) {
		cpssp->NAME.eip_new += cpssp->NAME.tj;
	}

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

static void
NAME_(loope)(struct cpssp *cpssp)
{
	uint32_t count;

	if (NAME_(address_size_32)(cpssp)) {
		count = NAME_(get_ECX)(cpssp) - 1;
		NAME_(set_ECX)(cpssp, count);
	} else {
		count = NAME_(get_CX)(cpssp) - 1;
		NAME_(set_CX)(cpssp, count);
	}

	if (count != 0 && NAME_(get_ZF)(cpssp)) {
		cpssp->NAME.eip_new += cpssp->NAME.tj;
	}

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

static void
NAME_(loopne)(struct cpssp *cpssp)
{
	uint32_t count;

	if (NAME_(address_size_32)(cpssp)) {
		count = NAME_(get_ECX)(cpssp) - 1;
		NAME_(set_ECX)(cpssp, count);
	} else {
		count = NAME_(get_CX)(cpssp) - 1;
		NAME_(set_CX)(cpssp, count);
	}

	if (count != 0 && ! NAME_(get_ZF)(cpssp)) {
		cpssp->NAME.eip_new += cpssp->NAME.tj;
	}

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

static void
NAME_(jcxz)(struct cpssp *cpssp)
{
	if (NAME_(address_size_32)(cpssp)) {
		if (NAME_(get_ECX)(cpssp) == 0) {
			cpssp->NAME.eip_new += cpssp->NAME.tj;
		}
	} else {
		if (NAME_(get_CX)(cpssp) == 0) {
			cpssp->NAME.eip_new += cpssp->NAME.tj;
		}
	}

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

/******************************************************************************
 * JMP - Jump
 ******************************************************************************/

static void
NAME_(jmp_far)(struct cpssp *cpssp)
{
	NAME_(load_code_desc)(cpssp, cpssp->NAME.eflag_vm, cpssp->NAME.t2);

	/* Check descriptor. */
	/* FIXME */

	if (! cpssp->NAME.temp_cs_sflag) {
		/*
		 * Call Gate
		 * Task Gate
		 */
		cpssp->NAME.t1 = cpssp->NAME.descriptor_gate_offset;
		cpssp->NAME.t2 = cpssp->NAME.descriptor_gate_selector;

		NAME_(load_code_desc)(cpssp, cpssp->NAME.eflag_vm, cpssp->NAME.t2);

		/* Check descriptor. */
		/* FIXME */
	}

	if (! cpssp->NAME.temp_cs_sflag) {
		/*
		 * Task State Segment
		 */
		ERROR_NYI(); /* TODO */

	} else {
		/*
		 * Code Segment
		 */
		if (NAME_(operand_size_32)(cpssp)) {
			cpssp->NAME.eip_new = cpssp->NAME.t1;
		} else {
			cpssp->NAME.eip_new = cpssp->NAME.t1 & 0xffff;
		}
		NAME_(commit_CS)(cpssp);
	}
}

static void
NAME_(jmp_near)(struct cpssp *cpssp)
{
	cpssp->NAME.eip_new += cpssp->NAME.tj;

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

static void
NAME_(jmp_near_absolute)(struct cpssp *cpssp)
{
	cpssp->NAME.eip_new = cpssp->NAME.t1;

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

/**********************************************************************
 * CALL - Call Procedure
 **********************************************************************/

static void
NAME_(call_far)(struct cpssp *cpssp)
{
	NAME_(load_code_desc)(cpssp, cpssp->NAME.eflag_vm, cpssp->NAME.t2);

	/* Check descriptor. */
	/* FIXME */

	if (! cpssp->NAME.temp_cs_sflag) {
		/*
		 * Call Gate
		 * Task Gate
		 */
		cpssp->NAME.t1 = cpssp->NAME.descriptor_gate_offset;
		cpssp->NAME.t2 = cpssp->NAME.descriptor_gate_selector;

		NAME_(load_code_desc)(cpssp, cpssp->NAME.eflag_vm, cpssp->NAME.t2);

		/* Check descriptor. */
		/* FIXME */
	}

	if (! cpssp->NAME.temp_cs_sflag) {
		/*
		 * Task State Segment
		 */
		ERROR_NYI(); /* TODO */

	} else {
		/*
		 * Code Segment
		 */
		if (! ((cpssp->NAME.temp_cs_type >> 2) & 1) /* non-conforming code segment */
		 && cpssp->NAME.temp_cs_dpl < NAME_(get_CPL)(cpssp)) {
			/* Switch stack. */
			ERROR_NYI();
		}
		if (NAME_(operand_size_32)(cpssp)) {
			NAME_(push32)(cpssp, NAME_(get_CS)(cpssp));
			NAME_(push32)(cpssp, cpssp->NAME.eip_new);
			cpssp->NAME.eip_new = cpssp->NAME.t1;
		} else {
			NAME_(push16)(cpssp, NAME_(get_CS)(cpssp));
			NAME_(push16)(cpssp, cpssp->NAME.eip_new);
			cpssp->NAME.eip_new = cpssp->NAME.t1 & 0xffff;
		}
		NAME_(commit_CS)(cpssp);
	}
}

static void
NAME_(call_near)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(push32)(cpssp, cpssp->NAME.eip_new);
	} else {
		NAME_(push16)(cpssp, cpssp->NAME.eip_new);
	}

	cpssp->NAME.eip_new += cpssp->NAME.tj;

	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
}

static void
NAME_(call_near_absolute)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(push32)(cpssp, cpssp->NAME.eip_new);
	} else {
		NAME_(push16)(cpssp, cpssp->NAME.eip_new);
	}

	cpssp->NAME.eip_new = cpssp->NAME.t1;

#if 0 /* FIXME */
	if (! NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new &= 0xffff;
	}
#endif
}

/**********************************************************************
 * RET - Return from Procedure
 **********************************************************************/

static void
NAME_(ret_far)(struct cpssp *cpssp)
{
	uint16_t cs_new;

	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new = NAME_(pop32)(cpssp);
		cs_new = NAME_(pop32)(cpssp);
	} else {
		cpssp->NAME.eip_new = NAME_(pop16)(cpssp);
		cs_new = NAME_(pop16)(cpssp);
	}
	NAME_(load_code_desc)(cpssp, cpssp->NAME.eflag_vm, cs_new);

	/* Checks */
	if (cpssp->NAME.temp_cs_selector_rpl < NAME_(get_CPL)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (cpssp->NAME.temp_cs_selector_rpl != NAME_(get_CPL)(cpssp)) {
		ERROR_NYI();
	}

	if (NAME_(stack_size_32)(cpssp)) {
		NAME_(set_ESP)(cpssp, NAME_(get_ESP)(cpssp) + cpssp->NAME.t1);
	} else {
		NAME_(set_SP)(cpssp, NAME_(get_SP)(cpssp) + cpssp->NAME.t1);
	}

	NAME_(commit_CS)(cpssp);
}

static void
NAME_(ret_near)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new = NAME_(pop32)(cpssp);
	} else {
		cpssp->NAME.eip_new = NAME_(pop16)(cpssp);
	}

	if (cpssp->NAME.t1 != 0) {
		if (NAME_(stack_size_32)(cpssp)) {
			NAME_(set_ESP)(cpssp, NAME_(get_ESP)(cpssp) + cpssp->NAME.t1);
		} else {
			NAME_(set_SP)(cpssp, NAME_(get_SP)(cpssp) + cpssp->NAME.t1);
		}
	}
}

/**********************************************************************
 * LEAVE - High Level Procedure Exit
 **********************************************************************/

static void
NAME_(leave)(struct cpssp *cpssp)
{
	if (NAME_(stack_size_32)(cpssp)) {
		NAME_(set_ESP)(cpssp, NAME_(get_EBP)(cpssp));
	} else {
		NAME_(set_SP)(cpssp, NAME_(get_BP)(cpssp));
	}

	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(set_EBP)(cpssp, NAME_(pop32)(cpssp));
	} else {
		NAME_(set_BP)(cpssp, NAME_(pop16)(cpssp));
	}
}

/**********************************************************************
 * INT n/INTO/INT 3 - Call to Interrupt Procedure
 **********************************************************************/

static void
NAME_(int3)(struct cpssp *cpssp)
{
	cpssp->NAME.exception_pending = 1;
	cpssp->NAME.exception_vector = 3;
	cpssp->NAME.exception_error_code = -1;
	cpssp->NAME.exception_is_interrupt = 1;
}

static void
NAME_(intn)(struct cpssp *cpssp)
{
	if (NAME_(virtual8086_mode)(cpssp)) {
		if (NAME_(get_IOPL)(cpssp) < 3) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	}

	cpssp->NAME.exception_pending = 1;
	cpssp->NAME.exception_vector = cpssp->NAME.t1 & 0xff;
	cpssp->NAME.exception_error_code = -1;
	cpssp->NAME.exception_is_interrupt = 1;
}

static void
NAME_(into)(struct cpssp *cpssp)
{
	if (NAME_(get_OF)(cpssp)) {
		cpssp->NAME.exception_pending = 1;
		cpssp->NAME.exception_vector = 4;
		cpssp->NAME.exception_error_code = -1;
		cpssp->NAME.exception_is_interrupt = 1;
	}
}

/******************************************************************************
 * ICEBP - Undocumented Opcode
 ******************************************************************************/

static void
NAME_(icebp)(struct cpssp *cpssp)
{
	/* TODO: check IR (DR7) */
	cpssp->NAME.exception_pending = 1;
	cpssp->NAME.exception_vector = cpssp->NAME.t1;
	cpssp->NAME.exception_error_code = -1;
	cpssp->NAME.exception_is_interrupt = 1;
}

/******************************************************************************
 * RCL/RCR/ROL/ROR - Rotate
 ******************************************************************************/

static void
NAME_(rclb)(struct cpssp *cpssp)
{
	uint8_t value;
	uint8_t count;
	bool cf;

	value = cpssp->NAME.t1;
	count = (cpssp->NAME.t2 & 31) % 9;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		return;
	}

	if (count == 1) {
		cpssp->NAME.t0 = (value << 1) | NAME_(get_CF)(cpssp);
	} else {
		cpssp->NAME.t0 = (value << count) | (NAME_(get_CF)(cpssp) << (count - 1))
			| (value >> (9 - count));
	}

	cf = (value >> (8 - count)) & 1;

	cpssp->NAME.t0 &= 0xff;

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf ^ (cpssp->NAME.t0 >> 7));
}

static void
NAME_(rclw)(struct cpssp *cpssp)
{
	uint16_t value;
	uint8_t count;
	bool cf;

	value = cpssp->NAME.t1;
	count = (cpssp->NAME.t2 & 31) % 17;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		return;
	}

	if (count == 1) {
		cpssp->NAME.t0 = (value << 1) | NAME_(get_CF)(cpssp);
	} else if (count == 16) {
		cpssp->NAME.t0 = (NAME_(get_CF)(cpssp) << 15) | (value >> 1);
	} else { /* 2..15 */
		cpssp->NAME.t0 = (value << count) | (NAME_(get_CF)(cpssp) << (count - 1))
			| (value >> (17 - count));
	}

	cf = (value >> (16 - count)) & 1;

	cpssp->NAME.t0 &= 0xffff;

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf ^ (cpssp->NAME.t0 >> 15));
}

static void
NAME_(rcld)(struct cpssp *cpssp)
{
	uint32_t value;
	uint8_t count;
	bool cf;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 31;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		return;
	}

	if (count == 1) {
		cpssp->NAME.t0 = (value << 1) | NAME_(get_CF)(cpssp);
	} else {
		cpssp->NAME.t0 = (value << count) | (NAME_(get_CF)(cpssp) << (count - 1))
			| (value >> (33 - count));
	}

	cf = (value >> (32 - count)) & 1;

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf ^ (cpssp->NAME.t0 >> 31));
}

static void
NAME_(rclv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(rcld)(cpssp);
	} else {
		NAME_(rclw)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(rcrb)(struct cpssp *cpssp)
{
	uint8_t value;
	uint8_t count;

	value = cpssp->NAME.t1;
	count = (cpssp->NAME.t2 & 31) % 9;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = (value >> count) | (NAME_(get_CF)(cpssp) << (8 - count))
		| (value << (9 - count));

	cpssp->NAME.t0 &= 0xff;

	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t0 << 1) ^ cpssp->NAME.t0) & 0x80) > 0);
}

static void
NAME_(rcrw)(struct cpssp *cpssp)
{
	uint16_t value;
	uint8_t count;

	value = cpssp->NAME.t1;
	count = (cpssp->NAME.t2 & 31) % 17;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = (value >> count) | (NAME_(get_CF)(cpssp) << (16 - count))
		| (value << (17 - count));

	cpssp->NAME.t0 &= 0xffff;

	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t0 << 1) ^ cpssp->NAME.t0) & 0x8000) > 0);
}

static void
NAME_(rcrd)(struct cpssp *cpssp)
{
	uint32_t value;
	uint8_t count;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 31;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		return;
	}

	if (count == 1) {
		cpssp->NAME.t0 = (value >> 1) | (NAME_(get_CF)(cpssp) << 31);
	} else {
		cpssp->NAME.t0 = (value >> count) | (NAME_(get_CF)(cpssp) << (32 - count))
			| (value << (33 - count));
	}

	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t0 << 1) ^ cpssp->NAME.t0) & 0x80000000) > 0);
}

static void
NAME_(rcrv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(rcrd)(cpssp);
	} else {
		NAME_(rcrw)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(rolb)(struct cpssp *cpssp)
{
	uint8_t value;
	uint8_t count;
	bool cf;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 7;

	cpssp->NAME.t0 = (value << count) | (value >> (8 - count));

	cf = cpssp->NAME.t0 & 1;

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf ^ (cpssp->NAME.t0 >> 7)); /* Undefined for multi-bit rotates */
}

static void
NAME_(rolw)(struct cpssp *cpssp)
{
	uint16_t value;
	uint8_t count;
	bool cf;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 15;

	cpssp->NAME.t0 = (value << count) | (value >> (16 - count));

	cf = cpssp->NAME.t0 & 1;

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf ^ (cpssp->NAME.t0 >> 15)); /* Undefined for multi-bit rotates */
}

static void
NAME_(rold)(struct cpssp *cpssp)
{
	uint32_t value;
	uint8_t count;
	bool cf;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 31;

	cpssp->NAME.t0 = (value << count) | (value >> (32 - count));

	cf = cpssp->NAME.t0 & 1;

	NAME_(sset_CF)(cpssp, cf);
	NAME_(sset_OF)(cpssp, cf ^ (cpssp->NAME.t0 >> 31)); /* Undefined for multi-bit rotates */
}

static void
NAME_(rolv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(rold)(cpssp);
	} else {
		NAME_(rolw)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(rorb)(struct cpssp *cpssp)
{
	uint8_t value;
	uint8_t count;
	bool b6;
	bool b7;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 7;

	cpssp->NAME.t0 = (value >> count) | (value << (8 - count));

	b7 = (cpssp->NAME.t0 & 0x80) != 0;
	b6 = (cpssp->NAME.t0 & 0x40) != 0;

	NAME_(sset_CF)(cpssp, b7);
	NAME_(sset_OF)(cpssp, b7 ^ b6);
}

static void
NAME_(rorw)(struct cpssp *cpssp)
{
	uint16_t value;
	uint8_t count;
	bool b14;
	bool b15;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 15;

	cpssp->NAME.t0 = (value >> count) | (value << (16 - count));

	b15 = (cpssp->NAME.t0 & 0x8000) != 0;
	b14 = (cpssp->NAME.t0 & 0x4000) != 0;

	NAME_(sset_CF)(cpssp, b15);
	NAME_(sset_OF)(cpssp, b15 ^ b14);
}

static void
NAME_(rord)(struct cpssp *cpssp)
{
	uint32_t value;
	uint8_t count;
	bool b30;
	bool b31;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 * 31;

	cpssp->NAME.t0 = (value >> count) | (value << (32 - count));

	b31 = (cpssp->NAME.t0 & 0x80000000) != 0;
	b30 = (cpssp->NAME.t0 & 0x40000000) != 0;

	NAME_(sset_CF)(cpssp, b31);
	NAME_(sset_OF)(cpssp, b31 ^ b30);
}

static void
NAME_(rorv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(rord)(cpssp);
	} else {
		NAME_(rorw)(cpssp);
	}
}

/******************************************************************************
 * SAL/SAR/SHL/SHR - Shift
 ******************************************************************************/

static void
NAME_(sarb)(struct cpssp *cpssp)
{
	uint8_t count;
	uint8_t value;
	uint8_t result;

	count = cpssp->NAME.t2 & 31;
	value = cpssp->NAME.t1;

	if (count == 0)
	{
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	if (count < 8) {
		if (value & 0x80) {
			result = (value >> count) | (0xff << (8 - count));
		} else {
			result = value >> count;
		}
		NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);
	} else {
		if (value & 0x80) {
			result = 0xff;
		} else {
			result = 0;
		}
		NAME_(sset_CF)(cpssp, (value & 0x80) > 0);
	}
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);

	cpssp->NAME.t0 = result;

	NAME_(sset_SF)(cpssp, result >= 0x80);
	NAME_(sset_ZF)(cpssp, result == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, result));
}

static void
NAME_(sarw)(struct cpssp *cpssp)
{
	uint8_t count;
	uint16_t value;
	uint16_t result;

	count = cpssp->NAME.t2 & 31;
	value = cpssp->NAME.t1;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	if (count < 16) {
		if (value & 0x8000) {
			result = (value >> count) | (0xffff << (16 - count));
		} else {
			result = value >> count;
		}
		NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);
	} else {
		if (value & 0x8000) {
			result = 0xffff;
		} else {
			result = 0;
		}
		NAME_(sset_CF)(cpssp, (value & 0x8000) > 0);
	}
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);

	cpssp->NAME.t0 = result;

	NAME_(sset_SF)(cpssp, result >= 0x8000);
	NAME_(sset_ZF)(cpssp, result == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, result));
}

static void
NAME_(sard)(struct cpssp *cpssp)
{
	uint8_t count;
	uint32_t value;
	uint32_t result;

	count = cpssp->NAME.t2 & 31;
	value = cpssp->NAME.t1;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	if (value & 0x80000000) {
		result = (value >> count) | (0xffffffff << (32 - count));
	} else {
		result = value >> count;
	}
	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_OF)(cpssp, 0);

	cpssp->NAME.t0 = result;

	NAME_(sset_SF)(cpssp, result >= 0x80000000);
	NAME_(sset_ZF)(cpssp, result == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, result));
}

static void
NAME_(sarv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(sard)(cpssp);
	} else {
		NAME_(sarw)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(shlb)(struct cpssp *cpssp)
{
	uint8_t count;
	uint8_t value;
	uint8_t result;

	count = cpssp->NAME.t2 & 31;
	value = cpssp->NAME.t1;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	result = value << count;

	if (count <= 8) {
		NAME_(sset_CF)(cpssp, (value >> (8 - count)) & 1);
	} else {
		NAME_(sset_CF)(cpssp, 0);
	}

	if (count == 1) {
		NAME_(sset_OF)(cpssp, ((value ^ result) & 0x80) > 0);
	} else {
		NAME_(sset_OF)(cpssp, (((value << (count - 1)) ^ result) & 0x80) > 0);
	}

	NAME_(sset_AF)(cpssp, 0); /* undefined */

	cpssp->NAME.t0 = result;

	NAME_(sset_SF)(cpssp, result >= 0x80);
	NAME_(sset_ZF)(cpssp, result == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, result));
}

static void
NAME_(shlw)(struct cpssp *cpssp)
{
	uint8_t count;
	uint16_t value;
	uint16_t result;

	count = cpssp->NAME.t2 & 31;
	value = cpssp->NAME.t1;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	result = value << count;

	if (count <= 16) {
		NAME_(sset_CF)(cpssp, (value >> (16 - count)) & 1);
	} else {
		NAME_(sset_CF)(cpssp, 0);
	}

	if (count == 1) {
		NAME_(sset_OF)(cpssp, ((value ^ result) & 0x8000) > 0);
	} else {
		NAME_(sset_OF)(cpssp, (((value << (count - 1)) ^ result) & 0x8000) > 0);
	}

	NAME_(sset_AF)(cpssp, 0); /* undefined */

	cpssp->NAME.t0 = result;

	NAME_(sset_SF)(cpssp, result >= 0x8000);
	NAME_(sset_ZF)(cpssp, result == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, result));
}

static void
NAME_(shld)(struct cpssp *cpssp)
{
	uint8_t count;
	uint32_t value;
	uint32_t result;

	count = cpssp->NAME.t2 & 31;
	value = cpssp->NAME.t1;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	result = value << count;

	NAME_(sset_CF)(cpssp, (value >> (32 - count)) & 1);

	if (count == 1) {
		NAME_(sset_OF)(cpssp, ((value ^ result) & 0x80000000) > 0);
	} else {
		NAME_(sset_OF)(cpssp, (((value << (count - 1)) ^ result) & 0x80000000) > 0);
	}

	NAME_(sset_AF)(cpssp, 0); /* undefined */

	cpssp->NAME.t0 = result;

	NAME_(sset_SF)(cpssp, result >= 0x80000000);
	NAME_(sset_ZF)(cpssp, result == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, result));
}

static void
NAME_(shlv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(shld)(cpssp);
	} else {
		NAME_(shlw)(cpssp);
	}
}

/******************************************************************************/

static void
NAME_(shrb)(struct cpssp *cpssp)
{
	uint8_t value;
	uint8_t count;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 31;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = value >> count;

	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);

	if (count == 1) {
		NAME_(sset_OF)(cpssp, value >= 0x80);
	} else {
		NAME_(sset_OF)(cpssp, 0); /* undocumented, but hardware really does it */
	}

	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
	NAME_(sset_AF)(cpssp, 0); /* Undefined */
}

static void
NAME_(shrw)(struct cpssp *cpssp)
{
	uint16_t value;
	uint8_t count;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 31;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = value >> count;

	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);

	if (count == 1) {
		NAME_(sset_OF)(cpssp, value >= 0x8000);
	} else {
		NAME_(sset_OF)(cpssp, 0); /* undocumented, but hardware really does it */
	}

	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
	NAME_(sset_AF)(cpssp, 0); /* Undefined */
}

static void
NAME_(shrd)(struct cpssp *cpssp)
{
	uint32_t value;
	uint8_t count;

	value = cpssp->NAME.t1;
	count = cpssp->NAME.t2 & 31;

	if (count == 0) {
		cpssp->NAME.t0 = value;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = value >> count;

	NAME_(sset_CF)(cpssp, (value >> (count - 1)) & 1);

	if (count == 1) {
		NAME_(sset_OF)(cpssp, value >= 80000000);
	} else {
		NAME_(sset_OF)(cpssp, 0); /* undocumented, but hardware really does it */
	}

	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
	NAME_(sset_AF)(cpssp, 0); /* Undefined */
}

static void
NAME_(shrv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(shrd)(cpssp);
	} else {
		NAME_(shrw)(cpssp);
	}
}

/******************************************************************************
 * SHRD - Double Precision Shift Right
 ******************************************************************************/

static void
NAME_(shrdw)(struct cpssp *cpssp)
{
	uint8_t count;

	count = cpssp->NAME.t3 & 31;

	if (!count) {
		cpssp->NAME.t0 = cpssp->NAME.t1;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = ((cpssp->NAME.t2 << 16) | cpssp->NAME.t1) >> count;

	if (count > 16) {
		cpssp->NAME.t0 |= cpssp->NAME.t1 << (32 - count);
	}

	cpssp->NAME.t0 &= 0xffff;

	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (count - 1)) & 1);
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t0 << 1) ^ cpssp->NAME.t0) & 0x8000) > 0);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
}

static void
NAME_(shrdd)(struct cpssp *cpssp)
{
	uint8_t count;

	count = cpssp->NAME.t3 & 31;

	if (!count) {
		cpssp->NAME.t0 = cpssp->NAME.t1;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = (cpssp->NAME.t2 << (32 - count)) | (cpssp->NAME.t1 >> count);

	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (count - 1)) & 1);
	NAME_(sset_OF)(cpssp, (((cpssp->NAME.t0 << 1) ^ cpssp->NAME.t0) & 0x80000000) > 0);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
}

static void
NAME_(shrdv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(shrdd)(cpssp);
	} else {
		NAME_(shrdw)(cpssp);
	}
}

/******************************************************************************
 * SHLD - Double Precision Shift Left
 ******************************************************************************/

static void
NAME_(shldw)(struct cpssp *cpssp)
{
	uint8_t count;

	count = cpssp->NAME.t3 & 31;

	if (! count) {
		cpssp->NAME.t0 = cpssp->NAME.t1;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = ((cpssp->NAME.t1 << 16) | cpssp->NAME.t2) << count;

	if (count > 16) {
		cpssp->NAME.t0 |= cpssp->NAME.t1 << (count - 16);
	}

	cpssp->NAME.t0 >>= 16;

	if (count <= 16) {
		NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (16 - count)) & 1);
	} else {
		NAME_(sset_CF)(cpssp, 0);
	}

	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));

	if (count == 1) {
		NAME_(sset_OF)(cpssp, ((cpssp->NAME.t1 ^ cpssp->NAME.t0) & 0x8000) > 0);
	} else {
		NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 << (count - 1)) ^ cpssp->NAME.t0) & 0x8000) > 0);
	}
}

static void
NAME_(shldd)(struct cpssp *cpssp)
{
	uint8_t count;

	count = cpssp->NAME.t3 & 31;

	if (! count) {
		cpssp->NAME.t0 = cpssp->NAME.t1;
		NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
		NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
		NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
		NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
		NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
		NAME_(sset_CF)(cpssp, NAME_(get_CF)(cpssp));
		return;
	}

	cpssp->NAME.t0 = (cpssp->NAME.t1 << count) | (cpssp->NAME.t2 >> (32 - count));

	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (32 - count)) & 1);
	NAME_(sset_AF)(cpssp, 0);
	NAME_(sset_ZF)(cpssp, cpssp->NAME.t0 == 0);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000);
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));

	if (count == 1) {
		NAME_(sset_OF)(cpssp, ((cpssp->NAME.t1 ^ cpssp->NAME.t0) & 0x80000000) > 0);
	} else {
		NAME_(sset_OF)(cpssp, (((cpssp->NAME.t1 << (count - 1)) ^ cpssp->NAME.t0) & 0x80000000) > 0);
	}
}

static void
NAME_(shldv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(shldd)(cpssp);
	} else {
		NAME_(shldw)(cpssp);
	}
}

/**********************************************************************
 * SHIFT - Shift Operations
 *********************************************************************/

static void
NAME_(shiftb)(struct cpssp *cpssp, uint8_t op)
{
	switch (op) {
	case 0: NAME_(rolb)(cpssp); break;
	case 1: NAME_(rorb)(cpssp); break;
	case 2: NAME_(rclb)(cpssp); break;
	case 3: NAME_(rcrb)(cpssp); break;
	case 4: NAME_(shlb)(cpssp); break;
	case 5: NAME_(shrb)(cpssp); break;
	case 6: ERROR_NYI(); /* salv()/NAME_(shlv)(cpssp) or error? */ break;
	case 7: NAME_(sarb)(cpssp); break;
	default: assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(shiftv)(struct cpssp *cpssp, uint8_t op)
{
	switch (op) {
	case 0:
		NAME_(rolv)(cpssp);
		break;
	case 1:
		NAME_(rorv)(cpssp);
		break;
	case 2:
		NAME_(rclv)(cpssp);
		break;
	case 3:
		NAME_(rcrv)(cpssp);
		break;
	case 4:
		NAME_(shlv)(cpssp);
		break;
	case 5:
		NAME_(shrv)(cpssp);
		break;
	case 6:
		ERROR_NYI(); /* salv()/NAME_(shlv)(cpssp) or error? */ break;
	case 7:
		NAME_(sarv)(cpssp);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

/**********************************************************************
 * SALC - Set AL on Carry
 **********************************************************************/

static void
NAME_(salc)(struct cpssp *cpssp)
{
	if (NAME_(get_CF)(cpssp)) {
		NAME_(set_AL)(cpssp, 0xff);
	} else {
		NAME_(set_AL)(cpssp, 0x00);
	}
}

/**********************************************************************
 * XLAT/XLATB - Table Look-up Translation
 **********************************************************************/

static void
NAME_(xlat)(struct cpssp *cpssp)
{
	uint32_t offset;

	if (NAME_(address_size_32)(cpssp)) {
		offset = NAME_(get_EBX)(cpssp) + NAME_(get_AL)(cpssp);
	} else {
		offset = NAME_(get_BX)(cpssp) + NAME_(get_AL)(cpssp);
	}

	NAME_(set_AL)(cpssp, NAME_(smrb)(cpssp, offset, NAME_(get_segment_overwritten)(cpssp, SEGMENT_DS)));
}

/**********************************************************************
 * LMSW - Load Machine Status Word
 **********************************************************************/

static void
NAME_(lmsw)(struct cpssp *cpssp)
{
	if (! NAME_(real_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (! NAME_(get_PE)(cpssp)) {
		NAME_(set_PE)(cpssp, (cpssp->NAME.t1 >> CR0_PE) & 1);
	}

	NAME_(set_MP)(cpssp, (cpssp->NAME.t1 >> CR0_MP) & 1);
	NAME_(set_EM)(cpssp, (cpssp->NAME.t1 >> CR0_EM) & 1);
	NAME_(set_TS)(cpssp, (cpssp->NAME.t1 >> CR0_TS) & 1);
}

/**********************************************************************
 * SMSW - Store Machine Status Word
 **********************************************************************/

static void
NAME_(smsw)(struct cpssp *cpssp)
{
	cpssp->NAME.t0 = NAME_(get_CR0)(cpssp) & 0xffff;
}

/******************************************************************************
 * HLT - Halt
 ******************************************************************************/

static void
NAME_(hlt)(struct cpssp *cpssp)
{
	if (!NAME_(real_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	cpssp->NAME.halt_state = 1;
}

/******************************************************************************
 * STR - Store Task Register
 ******************************************************************************/

static void
NAME_(str)(struct cpssp *cpssp)
{
	if (NAME_(real_mode)(cpssp) || NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	cpssp->NAME.t0 = cpssp->NAME.tr_selector;
}

/******************************************************************************
 * SLDT - Store Local Descriptor Table Register
 ******************************************************************************/

static void
NAME_(sldt)(struct cpssp *cpssp)
{
	if (NAME_(real_mode)(cpssp) || NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	cpssp->NAME.t0 = cpssp->NAME.ldtr_selector;
}

/******************************************************************************
 * LTR - Load Task Register
 ******************************************************************************/

static void
NAME_(ltr)(struct cpssp *cpssp)
{
	uint32_t dw1;
	uint32_t dw2;

	if (NAME_(real_mode)(cpssp) || NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	if (NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	NAME_(sset_selector)(cpssp, cpssp->NAME.t1);

	if (cpssp->NAME.selector_index == 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (cpssp->NAME.selector_ti != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
	}

	dw1 = NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8, SEGMENT_GDT);
	dw2 = NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8 + 4, SEGMENT_GDT);

	NAME_(sset_descriptor)(cpssp, dw1, dw2);

	if (cpssp->NAME.descriptor_sflag) {
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
	}

	if (cpssp->NAME.descriptor_type != SEGMENT_32BIT_AVAIL_TSS
	 && cpssp->NAME.descriptor_type != SEGMENT_16BIT_AVAIL_TSS) {
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
	}

	if (!cpssp->NAME.descriptor_pflag) {
		NAME_(exception)(cpssp, EXCEPTION_NP, cpssp->NAME.selector & 0xfffc);
	}

	cpssp->NAME.tr_selector = cpssp->NAME.selector;
	cpssp->NAME.tr_type = cpssp->NAME.descriptor_type;
	cpssp->tr_system_limit = cpssp->NAME.descriptor_system_limit;
	cpssp->tr_system_base = cpssp->NAME.descriptor_system_base;

	/* set busy bit */
	NAME_(smwd)(cpssp, cpssp->NAME.selector_index * 8 + 4, SEGMENT_GDT, dw2 | 0x200);
}

/******************************************************************************
 * LLDT - Load Local Descriptor Table Register
 ******************************************************************************/

static void
NAME_(lldt)(struct cpssp *cpssp)
{
	if (NAME_(real_mode)(cpssp) || NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
	}

	if (NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	NAME_(sset_selector)(cpssp, cpssp->NAME.t1);

	if (cpssp->NAME.selector_index == 0) {
		cpssp->NAME.ldtr_selector = 0;
		cpssp->ldtr_system_limit = 0;
		cpssp->ldtr_system_base = 0;
		return;
	}

	if (cpssp->NAME.selector_ti != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
	}

	NAME_(sset_descriptor)(cpssp, NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8, SEGMENT_GDT),
			NAME_(smrd)(cpssp, cpssp->NAME.selector_index * 8 + 4, SEGMENT_GDT));

	if ((cpssp->NAME.descriptor_sflag)
	 || (cpssp->NAME.descriptor_type != SEGMENT__LDT)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
	}

	if (!cpssp->NAME.descriptor_pflag) {
		NAME_(exception)(cpssp, EXCEPTION_NP, cpssp->NAME.selector & 0xfffc);
	}

	cpssp->NAME.ldtr_selector = cpssp->NAME.selector;
	cpssp->ldtr_system_limit = cpssp->NAME.descriptor_system_limit;
	cpssp->ldtr_system_base = cpssp->NAME.descriptor_system_base;
}

/******************************************************************************
 * SGDT/SIDT - Store Global/Interrupt Descriptor Table Register
 ******************************************************************************/

static void
NAME_(sgdt)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = cpssp->gdtr_limit;
	cpssp->NAME.t2 = cpssp->gdtr_base;
}

static void
NAME_(sidt)(struct cpssp *cpssp)
{
	cpssp->NAME.t1 = cpssp->idtr_limit;
	cpssp->NAME.t2 = cpssp->idtr_base;
}

/******************************************************************************
 * LGDT/LIDT - Load Global/Interrupt Descriptor Table Register
 ******************************************************************************/

static void
NAME_(lgdt)(struct cpssp *cpssp)
{
	uint16_t limit;
	uint32_t base;

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (NAME_(protected_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	limit = cpssp->NAME.t1;
	base = cpssp->NAME.t2;

	if (!NAME_(operand_size_32)(cpssp)) {
		base &= 0x00ffffff;
	}

	cpssp->gdtr_limit = limit;
	cpssp->gdtr_base = base;
}

static void
NAME_(lidt)(struct cpssp *cpssp)
{
	uint16_t limit;
	uint32_t base;

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (NAME_(protected_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	limit = cpssp->NAME.t1;
	base = cpssp->NAME.t2;

	if (!NAME_(operand_size_32)(cpssp)) {
		base &= 0x00ffffff;
	}

	cpssp->idtr_limit = limit;
	cpssp->idtr_base = base;
}

/******************************************************************************
 * BSR - Bit Scan Reverse
 ******************************************************************************/

static void
NAME_(sset_flags_for_bsr)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);		/* Undefined */
	NAME_(sset_OF)(cpssp, 0);		/* Undefined */
	NAME_(sset_AF)(cpssp, 0);		/* Undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));	/* Undefined */
}

static void
NAME_(bsrw)(struct cpssp *cpssp)
{
	if (cpssp->NAME.t2 == 0) {
		cpssp->NAME.t0 = cpssp->NAME.t1; /* Undefined */

		NAME_(sset_ZF)(cpssp, 1);

	} else {
		if ((cpssp->NAME.t2 >> 15) & 1) cpssp->NAME.t0 = 15;
		else if ((cpssp->NAME.t2 >> 14) & 1) cpssp->NAME.t0 = 14;
		else if ((cpssp->NAME.t2 >> 13) & 1) cpssp->NAME.t0 = 13;
		else if ((cpssp->NAME.t2 >> 12) & 1) cpssp->NAME.t0 = 12;
		else if ((cpssp->NAME.t2 >> 11) & 1) cpssp->NAME.t0 = 11;
		else if ((cpssp->NAME.t2 >> 10) & 1) cpssp->NAME.t0 = 10;
		else if ((cpssp->NAME.t2 >> 9) & 1) cpssp->NAME.t0 = 9;
		else if ((cpssp->NAME.t2 >> 8) & 1) cpssp->NAME.t0 = 8;
		else if ((cpssp->NAME.t2 >> 7) & 1) cpssp->NAME.t0 = 7;
		else if ((cpssp->NAME.t2 >> 6) & 1) cpssp->NAME.t0 = 6;
		else if ((cpssp->NAME.t2 >> 5) & 1) cpssp->NAME.t0 = 5;
		else if ((cpssp->NAME.t2 >> 4) & 1) cpssp->NAME.t0 = 4;
		else if ((cpssp->NAME.t2 >> 3) & 1) cpssp->NAME.t0 = 3;
		else if ((cpssp->NAME.t2 >> 2) & 1) cpssp->NAME.t0 = 2;
		else if ((cpssp->NAME.t2 >> 1) & 1) cpssp->NAME.t0 = 1;
		else cpssp->NAME.t0 = 0;

		NAME_(sset_ZF)(cpssp, 0);
	}

	NAME_(sset_flags_for_bsr)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000); /* Undefined */
}

static void
NAME_(bsrd)(struct cpssp *cpssp)
{
	if (cpssp->NAME.t2 == 0) {
		cpssp->NAME.t0 = cpssp->NAME.t1; /* Undefined */

		NAME_(sset_ZF)(cpssp, 1);

	} else {
		if ((cpssp->NAME.t2 >> 31) & 1) cpssp->NAME.t0 = 31;
		else if ((cpssp->NAME.t2 >> 30) & 1) cpssp->NAME.t0 = 30;
		else if ((cpssp->NAME.t2 >> 29) & 1) cpssp->NAME.t0 = 29;
		else if ((cpssp->NAME.t2 >> 28) & 1) cpssp->NAME.t0 = 28;
		else if ((cpssp->NAME.t2 >> 27) & 1) cpssp->NAME.t0 = 27;
		else if ((cpssp->NAME.t2 >> 26) & 1) cpssp->NAME.t0 = 26;
		else if ((cpssp->NAME.t2 >> 25) & 1) cpssp->NAME.t0 = 25;
		else if ((cpssp->NAME.t2 >> 24) & 1) cpssp->NAME.t0 = 24;
		else if ((cpssp->NAME.t2 >> 23) & 1) cpssp->NAME.t0 = 23;
		else if ((cpssp->NAME.t2 >> 22) & 1) cpssp->NAME.t0 = 22;
		else if ((cpssp->NAME.t2 >> 21) & 1) cpssp->NAME.t0 = 21;
		else if ((cpssp->NAME.t2 >> 20) & 1) cpssp->NAME.t0 = 20;
		else if ((cpssp->NAME.t2 >> 19) & 1) cpssp->NAME.t0 = 19;
		else if ((cpssp->NAME.t2 >> 18) & 1) cpssp->NAME.t0 = 18;
		else if ((cpssp->NAME.t2 >> 17) & 1) cpssp->NAME.t0 = 17;
		else if ((cpssp->NAME.t2 >> 16) & 1) cpssp->NAME.t0 = 16;
		else if ((cpssp->NAME.t2 >> 15) & 1) cpssp->NAME.t0 = 15;
		else if ((cpssp->NAME.t2 >> 14) & 1) cpssp->NAME.t0 = 14;
		else if ((cpssp->NAME.t2 >> 13) & 1) cpssp->NAME.t0 = 13;
		else if ((cpssp->NAME.t2 >> 12) & 1) cpssp->NAME.t0 = 12;
		else if ((cpssp->NAME.t2 >> 11) & 1) cpssp->NAME.t0 = 11;
		else if ((cpssp->NAME.t2 >> 10) & 1) cpssp->NAME.t0 = 10;
		else if ((cpssp->NAME.t2 >> 9) & 1) cpssp->NAME.t0 = 9;
		else if ((cpssp->NAME.t2 >> 8) & 1) cpssp->NAME.t0 = 8;
		else if ((cpssp->NAME.t2 >> 7) & 1) cpssp->NAME.t0 = 7;
		else if ((cpssp->NAME.t2 >> 6) & 1) cpssp->NAME.t0 = 6;
		else if ((cpssp->NAME.t2 >> 5) & 1) cpssp->NAME.t0 = 5;
		else if ((cpssp->NAME.t2 >> 4) & 1) cpssp->NAME.t0 = 4;
		else if ((cpssp->NAME.t2 >> 3) & 1) cpssp->NAME.t0 = 3;
		else if ((cpssp->NAME.t2 >> 2) & 1) cpssp->NAME.t0 = 2;
		else if ((cpssp->NAME.t2 >> 1) & 1) cpssp->NAME.t0 = 1;
		else cpssp->NAME.t0 = 0;

		NAME_(sset_ZF)(cpssp, 0);
	}

	NAME_(sset_flags_for_bsr)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000); /* Undefined */
}

static void
NAME_(bsrv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(bsrd)(cpssp);
	} else {
		NAME_(bsrw)(cpssp);
	}
}

/******************************************************************************
 * BSF - Bit Scan Forward
 ******************************************************************************/

static void
NAME_(sset_flags_for_bsf)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);		/* Undefined */
	NAME_(sset_OF)(cpssp, 0);		/* Undefined */
	NAME_(sset_AF)(cpssp, 0);		/* Undefined */
	NAME_(sset_PF)(cpssp, NAME_(parity)(cpssp, cpssp->NAME.t0));	/* Undefined */
}

static void
NAME_(bsfw)(struct cpssp *cpssp)
{
	if (cpssp->NAME.t2 == 0) {
		cpssp->NAME.t0 = cpssp->NAME.t1; /* Undefined */

		NAME_(sset_ZF)(cpssp, 1);

	} else {
		if ((cpssp->NAME.t2 >> 0) & 1) cpssp->NAME.t0 = 0;
		else if ((cpssp->NAME.t2 >> 1) & 1) cpssp->NAME.t0 = 1;
		else if ((cpssp->NAME.t2 >> 2) & 1) cpssp->NAME.t0 = 2;
		else if ((cpssp->NAME.t2 >> 3) & 1) cpssp->NAME.t0 = 3;
		else if ((cpssp->NAME.t2 >> 4) & 1) cpssp->NAME.t0 = 4;
		else if ((cpssp->NAME.t2 >> 5) & 1) cpssp->NAME.t0 = 5;
		else if ((cpssp->NAME.t2 >> 6) & 1) cpssp->NAME.t0 = 6;
		else if ((cpssp->NAME.t2 >> 7) & 1) cpssp->NAME.t0 = 7;
		else if ((cpssp->NAME.t2 >> 8) & 1) cpssp->NAME.t0 = 8;
		else if ((cpssp->NAME.t2 >> 9) & 1) cpssp->NAME.t0 = 9;
		else if ((cpssp->NAME.t2 >> 10) & 1) cpssp->NAME.t0 = 10;
		else if ((cpssp->NAME.t2 >> 11) & 1) cpssp->NAME.t0 = 11;
		else if ((cpssp->NAME.t2 >> 12) & 1) cpssp->NAME.t0 = 12;
		else if ((cpssp->NAME.t2 >> 13) & 1) cpssp->NAME.t0 = 13;
		else if ((cpssp->NAME.t2 >> 14) & 1) cpssp->NAME.t0 = 14;
		else cpssp->NAME.t0 = 15;

		NAME_(sset_ZF)(cpssp, 0);
	}

	NAME_(sset_flags_for_bsf)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x8000); /* Undefined */
}

static void
NAME_(bsfd)(struct cpssp *cpssp)
{
	if (cpssp->NAME.t2 == 0) {
		cpssp->NAME.t0 = cpssp->NAME.t1; /* Undefined */

		NAME_(sset_ZF)(cpssp, 1);

	} else {
		if ((cpssp->NAME.t2 >> 0) & 1) cpssp->NAME.t0 = 0;
		else if ((cpssp->NAME.t2 >> 1) & 1) cpssp->NAME.t0 = 1;
		else if ((cpssp->NAME.t2 >> 2) & 1) cpssp->NAME.t0 = 2;
		else if ((cpssp->NAME.t2 >> 3) & 1) cpssp->NAME.t0 = 3;
		else if ((cpssp->NAME.t2 >> 4) & 1) cpssp->NAME.t0 = 4;
		else if ((cpssp->NAME.t2 >> 5) & 1) cpssp->NAME.t0 = 5;
		else if ((cpssp->NAME.t2 >> 6) & 1) cpssp->NAME.t0 = 6;
		else if ((cpssp->NAME.t2 >> 7) & 1) cpssp->NAME.t0 = 7;
		else if ((cpssp->NAME.t2 >> 8) & 1) cpssp->NAME.t0 = 8;
		else if ((cpssp->NAME.t2 >> 9) & 1) cpssp->NAME.t0 = 9;
		else if ((cpssp->NAME.t2 >> 10) & 1) cpssp->NAME.t0 = 10;
		else if ((cpssp->NAME.t2 >> 11) & 1) cpssp->NAME.t0 = 11;
		else if ((cpssp->NAME.t2 >> 12) & 1) cpssp->NAME.t0 = 12;
		else if ((cpssp->NAME.t2 >> 13) & 1) cpssp->NAME.t0 = 13;
		else if ((cpssp->NAME.t2 >> 14) & 1) cpssp->NAME.t0 = 14;
		else if ((cpssp->NAME.t2 >> 15) & 1) cpssp->NAME.t0 = 15;
		else if ((cpssp->NAME.t2 >> 16) & 1) cpssp->NAME.t0 = 16;
		else if ((cpssp->NAME.t2 >> 17) & 1) cpssp->NAME.t0 = 17;
		else if ((cpssp->NAME.t2 >> 18) & 1) cpssp->NAME.t0 = 18;
		else if ((cpssp->NAME.t2 >> 19) & 1) cpssp->NAME.t0 = 19;
		else if ((cpssp->NAME.t2 >> 20) & 1) cpssp->NAME.t0 = 20;
		else if ((cpssp->NAME.t2 >> 21) & 1) cpssp->NAME.t0 = 21;
		else if ((cpssp->NAME.t2 >> 22) & 1) cpssp->NAME.t0 = 22;
		else if ((cpssp->NAME.t2 >> 23) & 1) cpssp->NAME.t0 = 23;
		else if ((cpssp->NAME.t2 >> 24) & 1) cpssp->NAME.t0 = 24;
		else if ((cpssp->NAME.t2 >> 25) & 1) cpssp->NAME.t0 = 25;
		else if ((cpssp->NAME.t2 >> 26) & 1) cpssp->NAME.t0 = 26;
		else if ((cpssp->NAME.t2 >> 27) & 1) cpssp->NAME.t0 = 27;
		else if ((cpssp->NAME.t2 >> 28) & 1) cpssp->NAME.t0 = 28;
		else if ((cpssp->NAME.t2 >> 29) & 1) cpssp->NAME.t0 = 29;
		else if ((cpssp->NAME.t2 >> 30) & 1) cpssp->NAME.t0 = 30;
		else cpssp->NAME.t0 = 31;

		NAME_(sset_ZF)(cpssp, 0);
	}

	NAME_(sset_flags_for_bsf)(cpssp);
	NAME_(sset_SF)(cpssp, cpssp->NAME.t0 >= 0x80000000); /* Undefined */
}

static void
NAME_(bsfv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(bsfd)(cpssp);
	} else {
		NAME_(bsfw)(cpssp);
	}
}

/******************************************************************************
 * BT - Bit Test
 ******************************************************************************/

static void
NAME_(btww)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 15)) & 1);
}

static void
NAME_(btdd)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 31)) & 1);
}

static void
NAME_(btvv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btdd)(cpssp);
	} else {
		NAME_(btww)(cpssp);
	}
}

/******************************************************************************
 * BTS - Bit Test and Set
 ******************************************************************************/

static void
NAME_(btsww)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 15)) & 1);
	cpssp->NAME.t0 = cpssp->NAME.t1 | (1 << (cpssp->NAME.t2 & 15));

	/* OF, SF, ZF, AF, PF undefined */
	NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
	NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
	NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
	NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
	NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
}

static void
NAME_(btsdd)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 31)) & 1);
	cpssp->NAME.t0 = cpssp->NAME.t1 | (1 << (cpssp->NAME.t2 & 31));

	/* OF, SF, ZF, AF, PF undefined */
	NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
	NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
	NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
	NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
	NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
}

static void
NAME_(btsvv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btsdd)(cpssp);
	} else {
		NAME_(btsww)(cpssp);
	}
}

static void
NAME_(btsvb)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btsdd)(cpssp);
	} else {
		NAME_(btsww)(cpssp);
	}
}

/******************************************************************************
 * BTR - Bit Test and Reset
 ******************************************************************************/

static void
NAME_(btrww)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 15)) & 1);
	cpssp->NAME.t0 = cpssp->NAME.t1 & ~(1 << (cpssp->NAME.t2 & 15));

	/* OF, SF, ZF, AF, PF undefined */
	NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
	NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
	NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
	NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
	NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
}

static void
NAME_(btrdd)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 31)) & 1);
	cpssp->NAME.t0 = cpssp->NAME.t1 & ~(1 << (cpssp->NAME.t2 & 31));

	/* OF, SF, ZF, AF, PF undefined */
	NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
	NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
	NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
	NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
	NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
}

static void
NAME_(btrvv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btrdd)(cpssp);
	} else {
		NAME_(btrww)(cpssp);
	}
}

static void
NAME_(btrvb)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btrdd)(cpssp);
	} else {
		NAME_(btrww)(cpssp);
	}
}

/******************************************************************************
 * BTC - Bit Test and Complement
 ******************************************************************************/

static void
NAME_(btcww)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 15)) & 1);
	cpssp->NAME.t0 = cpssp->NAME.t1 ^ (1 << (cpssp->NAME.t2 & 15));

	/* OF, SF, ZF, AF, PF undefined */
	NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
	NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
	NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
	NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
	NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
}

static void
NAME_(btcdd)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, (cpssp->NAME.t1 >> (cpssp->NAME.t2 & 31)) & 1);
	cpssp->NAME.t0 = cpssp->NAME.t1 ^ (1 << (cpssp->NAME.t2 & 31));

	/* OF, SF, ZF, AF, PF undefined */
	NAME_(sset_OF)(cpssp, NAME_(get_OF)(cpssp));
	NAME_(sset_SF)(cpssp, NAME_(get_SF)(cpssp));
	NAME_(sset_ZF)(cpssp, NAME_(get_ZF)(cpssp));
	NAME_(sset_AF)(cpssp, NAME_(get_AF)(cpssp));
	NAME_(sset_PF)(cpssp, NAME_(get_PF)(cpssp));
}

static void
NAME_(btcvv)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btcdd)(cpssp);
	} else {
		NAME_(btcww)(cpssp);
	}
}

static void
NAME_(btcvb)(struct cpssp *cpssp)
{
	if (NAME_(operand_size_32)(cpssp)) {
		NAME_(btcdd)(cpssp);
	} else {
		NAME_(btcww)(cpssp);
	}
}

/******************************************************************************
 * CMC - Complement Carry Flag
 ******************************************************************************/

static void
NAME_(cmc)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, ! NAME_(get_CF)(cpssp));
	NAME_(commit_CF)(cpssp);
}

/******************************************************************************
 * STD - Set Direction Flag
 ******************************************************************************/

static void
NAME_(std)(struct cpssp *cpssp)
{
	NAME_(sset_DF)(cpssp, 1);
}

/******************************************************************************
 * CLD - Clear Direction Flag
 ******************************************************************************/

static void
NAME_(cld)(struct cpssp *cpssp)
{
	NAME_(sset_DF)(cpssp, 0);
}

/******************************************************************************
 * STC - Set Carry Flag
 ******************************************************************************/

static void
NAME_(stc)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 1);
}

/******************************************************************************
 * CLC - Clear Carry Flag
 ******************************************************************************/

static void
NAME_(clc)(struct cpssp *cpssp)
{
	NAME_(sset_CF)(cpssp, 0);
}

/******************************************************************************
 * CLTS - Clear Task-Switched Flag in CR0
 ******************************************************************************/

static void
NAME_(clts)(struct cpssp *cpssp)
{
	if (! NAME_(real_mode)(cpssp) && NAME_(get_CPL)(cpssp) != 0) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	NAME_(set_TS)(cpssp, 0);
}

/******************************************************************************
 * STI - Set Interrupt Flag
 ******************************************************************************/

static void
NAME_(sti)(struct cpssp *cpssp)
{
	if (NAME_(protected_mode)(cpssp)) {
		if (NAME_(get_IOPL)(cpssp) < NAME_(get_CPL)(cpssp)) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	} else if (NAME_(virtual8086_mode)(cpssp)) {
		if (NAME_(get_IOPL)(cpssp) != 3) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	}

	NAME_(sset_IF)(cpssp, 1);
}

/******************************************************************************
 * CLI - Clear Interrupt Flag
 ******************************************************************************/

static void
NAME_(cli)(struct cpssp *cpssp)
{
	if (NAME_(protected_mode)(cpssp)) {
		if (NAME_(get_IOPL)(cpssp) < NAME_(get_CPL)(cpssp)) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	} else if (NAME_(virtual8086_mode)(cpssp)) {
		if (NAME_(get_IOPL)(cpssp) != 3) {
			NAME_(exception)(cpssp, EXCEPTION_GP, 0);
		}
	}

	NAME_(sset_IF)(cpssp, 0);
}

/******************************************************************************
 * IRET/IRETD - Interrupt Return
 ******************************************************************************/

static void
NAME_(iret)(struct cpssp *cpssp)
{
	uint32_t esp_cur;
	uint16_t cs_new;
	uint32_t ss_new;
	uint32_t esp_new;
	uint32_t eflags_new;
	uint32_t es_new;
	uint32_t ds_new;
	uint32_t fs_new;
	uint32_t gs_new;

	if (NAME_(get_NT)(cpssp)) {
		ERROR_NYI();
	}

	if (NAME_(stack_size_32)(cpssp)) {
		esp_cur = NAME_(get_ESP)(cpssp);
	} else {
		esp_cur = NAME_(get_SP)(cpssp);
	}

	if (NAME_(operand_size_32)(cpssp)) {
		cpssp->NAME.eip_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;
		cs_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;
		eflags_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;

		/*
		 * In real-mode/virtual 8086 mode VIF/VIP/VM/IOPL bits
		 * mustn't be changed.
		 * In virtual 8086 mode IOPL must be set to 3.
		 * FIXME
		 */

		NAME_(sset_EFLAGS32)(cpssp, eflags_new);
	} else {
		cpssp->NAME.eip_new = NAME_(smrw)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 2;
		cs_new = NAME_(smrw)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 2;
		eflags_new = NAME_(smrw)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 2;

		/*
		 * In virtual 8086 mode IOPL must be set to 3.
		 * FIXME
		 */

		NAME_(sset_EFLAGS16)(cpssp, eflags_new);
	}

	/* Read code descriptor. */
	NAME_(load_code_desc)(cpssp, cpssp->NAME.eflag_vm_new, cs_new);

	/* Check descriptor. */
	if (cpssp->NAME.temp_cs_selector_rpl < NAME_(get_CPL)(cpssp)
	 || ! cpssp->NAME.temp_cs_sflag /* no code or data segment */
	 || ! ((cpssp->NAME.temp_cs_type >> 3) & 1)) { /* no code segment */
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.temp_cs_selector & 0xfffc);
	}
	if ((cpssp->NAME.temp_cs_type >> 2) & 1) { /* conforming */
		if (cpssp->NAME.temp_cs_dpl > cpssp->NAME.temp_cs_selector_rpl) {
			NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.temp_cs_selector & 0xfffc);
		}
	} else { /* non-conforming */
		if (cpssp->NAME.temp_cs_dpl != cpssp->NAME.temp_cs_selector_rpl) {
			NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.temp_cs_selector & 0xfffc);
		}
	}
	if (! cpssp->NAME.temp_cs_pflag) {
		NAME_(exception)(cpssp, EXCEPTION_NP, cpssp->NAME.temp_cs_selector & 0xfffc);
	}

	if (cpssp->NAME.temp_cs_selector_rpl != NAME_(get_CPL)(cpssp)) {
		/*
		 * Switch stack.
		 */
		if (NAME_(operand_size_32)(cpssp)) {
			esp_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
			esp_cur += 4;
			ss_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
			esp_cur += 4;
		} else {
			esp_new = NAME_(smrw)(cpssp, esp_cur, SEGMENT_SS);
			esp_cur += 2;
			ss_new = NAME_(smrw)(cpssp, esp_cur, SEGMENT_SS);
			esp_cur += 2;
		}

		/* Read stack descriptor. */
		NAME_(load_data_desc)(cpssp, cpssp->NAME.eflag_vm_new, ss_new);

		/* Check stack descriptor. */
		if (! cpssp->NAME.descriptor_sflag /* no data/code segment */
		 || ((cpssp->NAME.descriptor_type >> 3) & 1) /* code segment */
		 || ! ((cpssp->NAME.descriptor_type >> 1) & 1) /* not writeable */
		 || cpssp->NAME.descriptor_dpl != cpssp->NAME.temp_cs_selector_rpl) {
			NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
		}
		if (! cpssp->NAME.descriptor_pflag) {
			NAME_(exception)(cpssp, EXCEPTION_NP, cpssp->NAME.selector & 0xfffc);
		}

	} else {
		/*
		 * Use same stack.
		 */
		esp_new = esp_cur;

		cpssp->NAME.selector = cpssp->NAME.ss_selector;
		cpssp->NAME.descriptor_type = cpssp->NAME.ss_type;
		cpssp->NAME.descriptor_sflag = cpssp->NAME.ss_sflag;
		cpssp->NAME.descriptor_dpl = cpssp->NAME.ss_dpl;
		cpssp->NAME.descriptor_pflag = cpssp->NAME.ss_pflag;
		cpssp->descriptor_segment_limit = cpssp->ss_segment_limit;
		cpssp->descriptor_segment_base = cpssp->ss_segment_base;
		cpssp->NAME.descriptor_segment_dflag = cpssp->NAME.ss_segment_dflag;
	}

	if (cpssp->NAME.eflag_vm_new
	 && NAME_(get_CPL)(cpssp) == 0) {
		es_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;
		ds_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;
		fs_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;
		gs_new = NAME_(smrd)(cpssp, esp_cur, SEGMENT_SS);
		esp_cur += 4;
	} else {
		/* Avoid warnings from gcc. */
		es_new = 0;
		ds_new = 0;
		fs_new = 0;
		gs_new = 0;
	}

	/* Commit stack. */
	NAME_(commit_SS)(cpssp);
	if (NAME_(stack_size_32)(cpssp)) {
		NAME_(set_ESP)(cpssp, esp_new);
	} else {
		NAME_(set_SP)(cpssp, esp_new);
	}

	/* Commit eflags. */
	if (NAME_(get_CPL)(cpssp) <= NAME_(get_IOPL)(cpssp)) {
		NAME_(commit_IF)(cpssp);
	}
	if (cpssp->NAME.temp_cs_segment_dflag) { /* new operand size == 32 */
		NAME_(commit_RF)(cpssp);
	}
	if (NAME_(get_CPL)(cpssp) == 0) {
		NAME_(commit_IOPL)(cpssp);
	}
	NAME_(commit_CF)(cpssp);
	NAME_(commit_PF)(cpssp);
	NAME_(commit_AF)(cpssp);
	NAME_(commit_ZF)(cpssp);
	NAME_(commit_SF)(cpssp);
	NAME_(commit_TF)(cpssp);
	NAME_(commit_OF)(cpssp);
	NAME_(commit_DF)(cpssp);
	NAME_(commit_NT)(cpssp);

	/* Commit segments. Must be done after setting cpssp->NAME.eflag_vm. */
	if (cpssp->NAME.eflag_vm_new
	 && NAME_(get_CPL)(cpssp) == 0) {
		NAME_(set_ES)(cpssp, es_new);
		NAME_(set_DS)(cpssp, ds_new);
		NAME_(set_FS)(cpssp, fs_new);
		NAME_(set_GS)(cpssp, gs_new);
	}

	/* Commit instruction pointer. */
	NAME_(commit_CS)(cpssp);
}

/******************************************************************************
 * Interrupt
 ******************************************************************************/

static void
interrupt_task_gate(int error_code)
{
	ERROR_NYI(); /* TODO */
}

static void
interrupt_gate(int error_code)
{
	uint16_t gate_selector;
	uint32_t gate_offset;
	uint8_t  gate_type;
	bool inner;
	uint16_t ss_new;
	uint32_t esp_new;
	uint32_t offset;

	gate_selector = cpssp->NAME.descriptor_gate_selector;
	gate_offset = cpssp->NAME.descriptor_gate_offset;
	gate_type = cpssp->NAME.descriptor_type;

	/* Load segment descriptor. */
	NAME_(load_code_desc)(cpssp, 0, gate_selector);

	/* Checks segment descriptor. */
	if (! cpssp->NAME.temp_cs_sflag /* no data/code segment */
	 || ! ((cpssp->NAME.temp_cs_type >> 3) & 1) /* no code segemnt */
	 || cpssp->NAME.temp_cs_dpl > NAME_(get_CPL)(cpssp)) {
		NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.temp_cs_selector & 0xfffc);
	}
	/* Some more checks missing... FIXME */
	if (! cpssp->NAME.temp_cs_pflag) {
		NAME_(exception)(cpssp, EXCEPTION_NP, cpssp->NAME.selector & 0xfffc);
	}

	inner = ! ((cpssp->NAME.temp_cs_type >> 2) & 1) /* non-conforming code segment */
		&& cpssp->NAME.temp_cs_dpl < NAME_(get_CPL)(cpssp);

	if (inner) {
		if (NAME_(virtual8086_mode)(cpssp)
		 && cpssp->NAME.temp_cs_dpl != 0) {
			NAME_(exception)(cpssp, EXCEPTION_GP, cpssp->NAME.selector & 0xfffc);
		}

		if (cpssp->NAME.tr_type == SEGMENT_32BIT_AVAIL_TSS
		 || cpssp->NAME.tr_type == SEGMENT_32BIT_BUSY_TSS) {
			offset = cpssp->NAME.temp_cs_dpl * 8 + 4;

			esp_new = NAME_(smrd)(cpssp, offset, SEGMENT_TSS);
			ss_new = NAME_(smrw)(cpssp, offset + 4, SEGMENT_TSS);
		} else {
			/* SEGMENT_16BIT_AVAIL_TSS */
			/* SEGMENT_16BIT_BUSY_TSS */
			offset = cpssp->NAME.temp_cs_dpl * 4 + 2;

			esp_new = NAME_(smrw)(cpssp, offset, SEGMENT_TSS);
			ss_new = NAME_(smrw)(cpssp, offset + 2, SEGMENT_TSS);
		}

		/* Load Stack Descriptor */
		NAME_(load_data_desc)(cpssp, 0, ss_new);

		/* Checks */
		if (cpssp->NAME.selector_rpl != cpssp->NAME.temp_cs_dpl
		 || cpssp->NAME.descriptor_dpl != cpssp->NAME.temp_cs_dpl
		 || ! cpssp->NAME.descriptor_sflag /* no data/code segment */
		 || ((cpssp->NAME.descriptor_type >> 3) & 1) /* code segment */
		 || ! ((cpssp->NAME.descriptor_type >> 1) & 1)) { /* not writeable */
			NAME_(exception)(cpssp, EXCEPTION_TS, cpssp->NAME.selector & 0xfffc);
		}
		if (! cpssp->NAME.descriptor_pflag) {
			NAME_(exception)(cpssp, EXCEPTION_SS, cpssp->NAME.selector & 0xfffc);
		}

		if (! cpssp->NAME.descriptor_segment_dflag) {
			esp_new &= 0xffff;
		}
	} else {
		if (NAME_(stack_size_32)(cpssp)) {
			esp_new = NAME_(get_ESP)(cpssp);
		} else {
			esp_new = NAME_(get_SP)(cpssp);
		}

		cpssp->NAME.selector = cpssp->NAME.ss_selector;
		cpssp->NAME.descriptor_type = cpssp->NAME.ss_type;
		cpssp->NAME.descriptor_sflag = cpssp->NAME.ss_sflag;
		cpssp->NAME.descriptor_dpl = cpssp->NAME.ss_dpl;
		cpssp->NAME.descriptor_pflag = cpssp->NAME.ss_pflag;
		cpssp->descriptor_segment_limit = cpssp->ss_segment_limit;
		cpssp->descriptor_segment_base = cpssp->ss_segment_base;
		cpssp->NAME.descriptor_segment_dflag = cpssp->NAME.ss_segment_dflag;
	}

	if (gate_offset > cpssp->NAME.temp_cs_segment_limit) {
		NAME_(exception)(cpssp, EXCEPTION_GP, 0);
	}

	if (inner) {
		if (NAME_(virtual8086_mode)(cpssp)) {
			/*
			 * Push GS, FS, DS, ES.
			 */
			switch (gate_type) {
			default: DEBUG_ERROR_SWITCH();
			case SEGMENT_32BIT_INTERRUPT_GATE:
			case SEGMENT_32BIT_TRAP_GATE:
				esp_new -= 4;
				NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_GS)(cpssp));
				esp_new -= 4;
				NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_FS)(cpssp));
				esp_new -= 4;
				NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_DS)(cpssp));
				esp_new -= 4;
				NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_ES)(cpssp));
				break;
			case SEGMENT_16BIT_INTERRUPT_GATE:
			case SEGMENT_16BIT_TRAP_GATE:
				esp_new -= 2;
				NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_GS)(cpssp));
				esp_new -= 2;
				NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_FS)(cpssp));
				esp_new -= 2;
				NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_DS)(cpssp));
				esp_new -= 2;
				NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_ES)(cpssp));
				break;
			}
		}
		/*
		 * Push SS, SP.
		 */
		switch (gate_type) {
		default: DEBUG_ERROR_SWITCH();
		case SEGMENT_32BIT_INTERRUPT_GATE:
		case SEGMENT_32BIT_TRAP_GATE:
			esp_new -= 4;
			NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_SS)(cpssp));
			esp_new -= 4;
			NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_ESP)(cpssp));
			break;
		case SEGMENT_16BIT_INTERRUPT_GATE:
		case SEGMENT_16BIT_TRAP_GATE:
			esp_new -= 2;
			NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_SS)(cpssp));
			esp_new -= 2;
			NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_ESP)(cpssp));
			break;
		}
	}

	/*
	 * Push FLAGS, CS, IP.
	 */
	switch (gate_type) {
	default: DEBUG_ERROR_SWITCH();
	case SEGMENT_32BIT_INTERRUPT_GATE:
	case SEGMENT_32BIT_TRAP_GATE:
		esp_new -= 4;
		NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_EFLAGS32)(cpssp));
		esp_new -= 4;
		NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_CS)(cpssp));
		esp_new -= 4;
		NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, cpssp->NAME.eip_new);
		break;
	case SEGMENT_16BIT_INTERRUPT_GATE:
	case SEGMENT_16BIT_TRAP_GATE:
		esp_new -= 2;
		NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_EFLAGS16)(cpssp));
		esp_new -= 2;
		NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, NAME_(get_CS)(cpssp));
		esp_new -= 2;
		NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, cpssp->NAME.eip_new);
		break;
	}
	if (error_code >= 0) {
		/*
		 * Push error code.
		 */
		switch (gate_type) {
		default: DEBUG_ERROR_SWITCH();
		case SEGMENT_32BIT_INTERRUPT_GATE:
		case SEGMENT_32BIT_TRAP_GATE:
			esp_new -= 4;
			NAME_(smwd)(cpssp, esp_new, SEGMENT_SS_NEW, error_code);
			break;
		case SEGMENT_16BIT_INTERRUPT_GATE:
		case SEGMENT_16BIT_TRAP_GATE:
			esp_new -= 2;
			NAME_(smww)(cpssp, esp_new, SEGMENT_SS_NEW, error_code);
			break;
		}
	}

	/* Commit Stack */
	NAME_(commit_SS)(cpssp);
	if (NAME_(stack_size_32)(cpssp)) {
		NAME_(set_ESP)(cpssp, esp_new);
	} else {
		NAME_(set_SP)(cpssp, esp_new);
	}

	/* Commit Instruction Pointer */
	NAME_(commit_CS)(cpssp);
	cpssp->NAME.eip_new = gate_offset;

	/* Commit Flags */
	if (! (gate_type & 1)) {
		NAME_(sset_IF)(cpssp, 0); NAME_(commit_IF)(cpssp);
	}

	NAME_(sset_TF)(cpssp, 0); NAME_(commit_TF)(cpssp);
	NAME_(sset_NT)(cpssp, 0); NAME_(commit_NT)(cpssp);
	NAME_(sset_RF)(cpssp, 0); NAME_(commit_RF)(cpssp);
	NAME_(sset_VM)(cpssp, 0); NAME_(commit_VM)(cpssp);

	if (NAME_(virtual8086_mode)(cpssp)) {
		NAME_(set_DS)(cpssp, 0);
		NAME_(set_ES)(cpssp, 0);
		NAME_(set_FS)(cpssp, 0);
		NAME_(set_GS)(cpssp, 0);
	}
}

static void
interrupt(uint8_t vector, bool is_soft_int, int error_code)
{
	if (NAME_(real_mode)(cpssp)) {
		NAME_(sset_descriptor)(cpssp, NAME_(smrd)(cpssp, vector * 4, SEGMENT_IDT),
				(1 << 15) /* present */
				| (0 << 13) /* dpl */
				| (0 << 12) /* system */
				| (SEGMENT_16BIT_INTERRUPT_GATE << 8) /* type */
				| (0 << 0)); /* base (23..16) */
	} else {
		NAME_(sset_descriptor)(cpssp, NAME_(smrd)(cpssp, vector * 8, SEGMENT_IDT),
				NAME_(smrd)(cpssp, vector * 8 + 4, SEGMENT_IDT));
	}

	/* Checks */
	if (cpssp->NAME.descriptor_sflag
	 || (cpssp->NAME.descriptor_type != SEGMENT_TASK_GATE
	  && cpssp->NAME.descriptor_type != SEGMENT_16BIT_INTERRUPT_GATE
	  && cpssp->NAME.descriptor_type != SEGMENT_32BIT_INTERRUPT_GATE
	  && cpssp->NAME.descriptor_type != SEGMENT_16BIT_TRAP_GATE
	  && cpssp->NAME.descriptor_type != SEGMENT_32BIT_TRAP_GATE)
	 || (is_soft_int
	  && cpssp->NAME.descriptor_dpl < NAME_(get_CPL)(cpssp))) {
		NAME_(exception)(cpssp, EXCEPTION_GP, vector * 8 + 2);
	}
	if (! cpssp->NAME.descriptor_pflag) {
		NAME_(exception)(cpssp, EXCEPTION_NP, vector * 8 + 2);
	}

	if (cpssp->NAME.descriptor_type == SEGMENT_TASK_GATE) {
		interrupt_task_gate(error_code);
	} else {
		/* SEGMENT_16BIT_INTERRUPT_GATE */
		/* SEGMENT_32BIT_INTERRUPT_GATE */
		/* SEGMENT_16BIT_TRAP_GATE */
		/* SEGMENT_32BIT_TRAP_GATE */
		interrupt_gate(error_code);
	}
}

/******************************************************************************/

static void
reset(struct cpssp *_cpssp)
{
	cpssp = _cpssp;

	cpssp->NAME.exception_pending = 0;
	cpssp->NAME.exception_vector = 0;
	cpssp->NAME.exception_error_code = -1;
	cpssp->NAME.exception_is_interrupt = 0;
	cpssp->NAME.exception_double_page_fault = 0;

	cpssp->NAME.eax = 0x00000000;
	cpssp->NAME.ebx = 0x00000000;
	cpssp->NAME.ecx = 0x00000000;
	cpssp->NAME.edx = 0; /* FIXME: cpu version information */
	cpssp->NAME.ebp = 0x00000000;
	cpssp->NAME.esi = 0x00000000;
	cpssp->NAME.edi = 0x00000000;
	cpssp->NAME.esp = 0x00000000;

	cpssp->NAME.eflag_cf = 0;
	cpssp->NAME.eflag_pf = 0;
	cpssp->NAME.eflag_af = 0;
	cpssp->NAME.eflag_zf = 0;
	cpssp->NAME.eflag_sf = 0;
	cpssp->NAME.eflag_tf = 0;
	cpssp->NAME.eflag_if = 0;
	cpssp->NAME.eflag_df = 0;
	cpssp->NAME.eflag_of = 0;
	cpssp->NAME.eflag_iopl = 0;
	cpssp->NAME.eflag_nt = 0;
	cpssp->NAME.eflag_rf = 0;
	cpssp->NAME.eflag_vm = 0;

	cpssp->NAME.eip = 0x0000fff0;
	cpssp->NAME.eip_new = 0x0000fff0;

	/* NW(29) and CD(30) enabled; ET(4) disabled (no math coprocessor) */
	NAME_(set_CR0)(cpssp, 0x60000000);

	NAME_(set_CR2)(cpssp, 0);
	NAME_(set_CR3)(cpssp, 0);

	cpssp->NAME.cs_selector		= 0xf000;
	cpssp->NAME.cs_type		= SEGMENT_CODE_EXEC_READ;
	cpssp->NAME.cs_sflag		= 1;
	cpssp->cs_dpl			= 0;
	cpssp->NAME.cs_pflag		= 1;
	cpssp->cs_segment_limit		= 0xffff;
	cpssp->cs_segment_base		= 0xffff0000;
	cpssp->NAME.cs_segment_dflag	= 0;

	cpssp->NAME.ss_selector		= 0x0000;
	cpssp->NAME.ss_type		= SEGMENT_DATA_READ_WRITE;
	cpssp->NAME.ss_sflag		= 1;
	cpssp->NAME.ss_dpl		= 0;
	cpssp->NAME.ss_pflag		= 1;
	cpssp->ss_segment_limit		= 0xffff;
	cpssp->ss_segment_base		= 0x00000000;
	cpssp->NAME.ss_segment_dflag	= 0;

	cpssp->NAME.ds_selector		= 0x0000;
	cpssp->NAME.ds_type		= SEGMENT_DATA_READ_WRITE;
	cpssp->NAME.ds_sflag		= 1;
	cpssp->NAME.ds_dpl		= 0;
	cpssp->NAME.ds_pflag		= 1;
	cpssp->ds_segment_limit		= 0xffff;
	cpssp->ds_segment_base		= 0x00000000;
	cpssp->NAME.ds_segment_dflag	= 0;

	cpssp->NAME.es_selector		= 0x0000;
	cpssp->NAME.es_type		= SEGMENT_DATA_READ_WRITE;
	cpssp->NAME.es_sflag		= 1;
	cpssp->NAME.es_dpl		= 0;
	cpssp->NAME.es_pflag		= 1;
	cpssp->es_segment_limit		= 0xffff;
	cpssp->es_segment_base		= 0x00000000;
	cpssp->NAME.es_segment_dflag	= 0;

	cpssp->NAME.fs_selector		= 0x0000;
	cpssp->NAME.fs_type		= SEGMENT_DATA_READ_WRITE;
	cpssp->NAME.fs_sflag		= 1;
	cpssp->NAME.fs_dpl		= 0;
	cpssp->NAME.fs_pflag		= 1;
	cpssp->fs_segment_limit		= 0xffff;
	cpssp->fs_segment_base		= 0x00000000;
	cpssp->NAME.fs_segment_dflag	= 0;

	cpssp->NAME.gs_selector		= 0x0000;
	cpssp->NAME.gs_type		= SEGMENT_DATA_READ_WRITE;
	cpssp->NAME.gs_sflag		= 1;
	cpssp->NAME.gs_dpl		= 0;
	cpssp->NAME.gs_pflag		= 1;
	cpssp->gs_segment_limit		= 0xffff;
	cpssp->gs_segment_base		= 0x00000000;
	cpssp->NAME.gs_segment_dflag	= 0;

	cpssp->gdtr_base = 0x00000000;
	cpssp->gdtr_limit = 0xffff;

	cpssp->idtr_base = 0x00000000;
	cpssp->idtr_limit = 0xffff;

	cpssp->NAME.ldtr_selector = 0x0000;
	cpssp->ldtr_system_limit = 0xffff;
	cpssp->ldtr_system_base = 0x00000000;

	cpssp->NAME.tr_selector = 0x0000;
	cpssp->NAME.tr_type = SEGMENT_32BIT_BUSY_TSS;
	cpssp->tr_system_limit = 0xffff;
	cpssp->tr_system_base = 0x00000000;

	cpssp->NAME.interrupt_delay = 0;
	cpssp->NAME.halt_state = 0;
}

/******************************************************************************
 * Handler for one instruction including prefixes
 ******************************************************************************/

static void
NAME_(step)(struct cpssp *_cpssp)
{
	uint8_t vector;

	cpssp = _cpssp;

	if (setjmp(cpssp->NAME.jmp_env)) {
		goto end;
	}

	if (! cpssp->state_n_reset) {
		reset(cpssp);
		return;
	}

	if (cpssp->NAME.exception_pending) {
		interrupt(cpssp->NAME.exception_vector, cpssp->NAME.exception_is_interrupt,
				cpssp->NAME.exception_error_code);
		cpssp->NAME.exception_double_page_fault = 0;
		cpssp->NAME.exception_pending = 0;
		NAME_(commit_EIP)(cpssp);
		return;
	}

	if (cpssp->state_interrupt_pending
	 && NAME_(get_IF)(cpssp)
	 && ! cpssp->NAME.interrupt_delay
	 && cpssp->NAME.prefix_lock_repeat == LR_NONE
	 && cpssp->NAME.prefix_segment_override == SEGMENT_NONE
	 && ! cpssp->NAME.prefix_operand_size_override
	 && ! cpssp->NAME.prefix_address_size_override
	 && ! cpssp->NAME.instruction_opcode2) {
		vector = CHIP_(ack)(cpssp);
		interrupt(vector, 0, -1);
		NAME_(commit_EIP)(cpssp);
		cpssp->NAME.halt_state = 0;
		return;
	}

	if (cpssp->NAME.halt_state) {
		return;
	}

	cpssp->NAME.instruction_opcode = NAME_(smrb)(cpssp, cpssp->NAME.eip_new, SEGMENT_CS);
	cpssp->NAME.eip_new += 1;

	switch ((cpssp->NAME.instruction_opcode2 << 8) | cpssp->NAME.instruction_opcode) {
	default:
		ERROR_NYI();

	case 0x0f: /* 2-byte opcode */
		cpssp->NAME.instruction_opcode2 = 1;
		break;
	case 0x26:
	case 0x2e:
	case 0x36:
	case 0x3e:
	case 0x64:
	case 0x65:
		switch (cpssp->NAME.instruction_opcode) {
		case 0x26: cpssp->NAME.prefix_segment_override = SEGMENT_ES; break;
		case 0x2e: cpssp->NAME.prefix_segment_override = SEGMENT_CS; break;
		case 0x36: cpssp->NAME.prefix_segment_override = SEGMENT_SS; break;
		case 0x3e: cpssp->NAME.prefix_segment_override = SEGMENT_DS; break;
		case 0x64: cpssp->NAME.prefix_segment_override = SEGMENT_FS; break;
		case 0x65: cpssp->NAME.prefix_segment_override = SEGMENT_GS; break;
		default: assert(0); /* Cannot happen. */
		}
		break;
	case 0x66:
		cpssp->NAME.prefix_operand_size_override = 1;
		break;
	case 0x67:
		cpssp->NAME.prefix_address_size_override = 1;
		break;
	case 0xf0:
	case 0xf2:
	case 0xf3:
		switch (cpssp->NAME.instruction_opcode) {
		case 0xf0: cpssp->NAME.prefix_lock_repeat = LR_LOCK; break;
		case 0xf2: cpssp->NAME.prefix_lock_repeat = LR_REPNZ; break;
		case 0xf3: cpssp->NAME.prefix_lock_repeat = LR_REPZ; break;
		default: assert(0); /* Cannot happen. */
		}
		break;

	case 0x00:
	case 0x08:
	case 0x10:
	case 0x18:
	case 0x20:
	case 0x28:
	case 0x30:
	case 0x38:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb_Gb)(cpssp);
		NAME_(alub)(cpssp, (cpssp->NAME.instruction_opcode >> 3) & 7);
		if (cpssp->NAME.instruction_opcode != 0x38)
			NAME_(store_Eb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x01:
	case 0x09:
	case 0x11:
	case 0x19:
	case 0x21:
	case 0x29:
	case 0x31:
	case 0x39:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_Gv)(cpssp);
		NAME_(aluv)(cpssp, (cpssp->NAME.instruction_opcode >> 3) & 7);
		if (cpssp->NAME.instruction_opcode != 0x39)
			NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x02:
	case 0x0a:
	case 0x12:
	case 0x1a:
	case 0x22:
	case 0x2a:
	case 0x32:
	case 0x3a:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gb_Eb)(cpssp);
		NAME_(alub)(cpssp, (cpssp->NAME.instruction_opcode >> 3) & 7);
		if (cpssp->NAME.instruction_opcode != 0x3a)
			NAME_(store_Gb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x03:
	case 0x0b:
	case 0x13:
	case 0x1b:
	case 0x23:
	case 0x2b:
	case 0x33:
	case 0x3b:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv_Ev)(cpssp);
		NAME_(aluv)(cpssp, (cpssp->NAME.instruction_opcode >> 3) & 7);
		if (cpssp->NAME.instruction_opcode != 0x3b)
			NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x04:
	case 0x0c:
	case 0x14:
	case 0x1c:
	case 0x24:
	case 0x2c:
	case 0x34:
	case 0x3c:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_AL_Ib)(cpssp);
		NAME_(alub)(cpssp, (cpssp->NAME.instruction_opcode >> 3) & 7);
		if (cpssp->NAME.instruction_opcode != 0x3c)
			NAME_(store_AL)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x05:
	case 0x0d:
	case 0x15:
	case 0x1d:
	case 0x25:
	case 0x2d:
	case 0x35:
	case 0x3d:
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_eAX_Iv)(cpssp);
		NAME_(aluv)(cpssp, (cpssp->NAME.instruction_opcode >> 3) & 7);
		if (cpssp->NAME.instruction_opcode != 0x3d)
			NAME_(store_eAX)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x06:
	case 0x0e:
	case 0x16:
	case 0x1e:
		switch (cpssp->NAME.instruction_opcode) {
		case 0x06:
			NAME_(load_ES)(cpssp);
			break;
		case 0x0e:
			NAME_(load_CS)(cpssp);
			break;
		case 0x16:
			NAME_(load_SS)(cpssp);
			break;
		case 0x1e:
			NAME_(load_DS)(cpssp);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		NAME_(pushv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x07:
	/* case 0x0f: */ /* Special Case */
	case 0x17:
	case 0x1f:
		NAME_(popv)(cpssp);
		switch (cpssp->NAME.instruction_opcode) {
		case 0x07:
			NAME_(store_ES)(cpssp);
			break;
		case 0x17:
			NAME_(store_SS)(cpssp);
			break;
		case 0x1f:
			NAME_(store_DS)(cpssp);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		if (cpssp->NAME.instruction_opcode == 0x17) {
			NAME_(commit_EIP_and_delay_interrupts)(cpssp);
		} else {
			NAME_(commit_EIP)(cpssp);
		}
		break;
	case 0x27:
		NAME_(daa)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x2f:
		NAME_(das)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x37:
		NAME_(aaa)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x3f:
		NAME_(aas)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x40:
	case 0x41:
	case 0x42:
	case 0x43:
	case 0x44:
	case 0x45:
	case 0x46:
	case 0x47:
	case 0x48:
	case 0x49:
	case 0x4a:
	case 0x4b:
	case 0x4c:
	case 0x4d:
	case 0x4e:
	case 0x4f:
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00:
			NAME_(load_eAX)(cpssp);
			break;
		case 0x01:
			NAME_(load_eCX)(cpssp);
			break;
		case 0x02:
			NAME_(load_eDX)(cpssp);
			break;
		case 0x03:
			NAME_(load_eBX)(cpssp);
			break;
		case 0x04:
			NAME_(load_eSP)(cpssp);
			break;
		case 0x05:
			NAME_(load_eBP)(cpssp);
			break;
		case 0x06:
			NAME_(load_eSI)(cpssp);
			break;
		case 0x07:
			NAME_(load_eDI)(cpssp);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		switch (cpssp->NAME.instruction_opcode & 0x08) {
		case 0x00:
			NAME_(incv)(cpssp);
			break;
		case 0x08:
			NAME_(decv)(cpssp);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00:
			NAME_(store_eAX)(cpssp);
			break;
		case 0x01:
			NAME_(store_eCX)(cpssp);
			break;
		case 0x02:
			NAME_(store_eDX)(cpssp);
			break;
		case 0x03:
			NAME_(store_eBX)(cpssp);
			break;
		case 0x04:
			NAME_(store_eSP)(cpssp);
			break;
		case 0x05:
			NAME_(store_eBP)(cpssp);
			break;
		case 0x06:
			NAME_(store_eSI)(cpssp);
			break;
		case 0x07:
			NAME_(store_eDI)(cpssp);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAP)(cpssp);
		break;
	case 0x50:
	case 0x51:
	case 0x52:
	case 0x53:
	case 0x54:
	case 0x55:
	case 0x56:
	case 0x57:
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00: NAME_(load_eAX)(cpssp); break;
		case 0x01: NAME_(load_eCX)(cpssp); break;
		case 0x02: NAME_(load_eDX)(cpssp); break;
		case 0x03: NAME_(load_eBX)(cpssp); break;
		case 0x04: NAME_(load_eSP)(cpssp); break;
		case 0x05: NAME_(load_eBP)(cpssp); break;
		case 0x06: NAME_(load_eSI)(cpssp); break;
		case 0x07: NAME_(load_eDI)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(pushv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x58:
	case 0x59:
	case 0x5a:
	case 0x5b:
	case 0x5c:
	case 0x5d:
	case 0x5e:
	case 0x5f:
		NAME_(popv)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00: NAME_(store_eAX)(cpssp); break;
		case 0x01: NAME_(store_eCX)(cpssp); break;
		case 0x02: NAME_(store_eDX)(cpssp); break;
		case 0x03: NAME_(store_eBX)(cpssp); break;
		case 0x04: NAME_(store_eSP)(cpssp); break;
		case 0x05: NAME_(store_eBP)(cpssp); break;
		case 0x06: NAME_(store_eSI)(cpssp); break;
		case 0x07: NAME_(store_eDI)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x60: /* pusha */
		NAME_(pushav)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x61: /* popa */
		NAME_(popav)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x62: /* bound */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv_Ma)(cpssp);
		NAME_(boundv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x68:
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Iv)(cpssp);
		NAME_(pushv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x69:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Ev_Iv)(cpssp);
		NAME_(imul3vv)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x6a:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_IbSE)(cpssp);
		NAME_(pushv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x6b:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ev_IbSE)(cpssp);
		NAME_(imul3vb)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x6c:
	case 0x6d:
	case 0x6e:
	case 0x6f:
	case 0xa4:
	case 0xa5:
	case 0xa6:
	case 0xa7:
	case 0xaa:
	case 0xab:
	case 0xac:
	case 0xad:
	case 0xae:
	case 0xaf:
		switch (cpssp->NAME.instruction_opcode) {
		case 0x6c: NAME_(insb)(cpssp); break;
		case 0x6d: NAME_(insv)(cpssp); break;
		case 0x6e: NAME_(outsb)(cpssp); break;
		case 0x6f: NAME_(outsv)(cpssp); break;
		case 0xa4: NAME_(movsb)(cpssp); break;
		case 0xa5: NAME_(movsv)(cpssp); break;
		case 0xa6: NAME_(cmpsb)(cpssp); break;
		case 0xa7: NAME_(cmpsv)(cpssp); break;
		case 0xaa: NAME_(stosb)(cpssp); break;
		case 0xab: NAME_(stosv)(cpssp); break;
		case 0xac: NAME_(lodsb)(cpssp); break;
		case 0xad: NAME_(lodsv)(cpssp); break;
		case 0xae: NAME_(scasb)(cpssp); break;
		case 0xaf: NAME_(scasv)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x70:
	case 0x71:
	case 0x72:
	case 0x73:
	case 0x74:
	case 0x75:
	case 0x76:
	case 0x77:
	case 0x78:
	case 0x79:
	case 0x7a:
	case 0x7b:
	case 0x7c:
	case 0x7d:
	case 0x7e:
	case 0x7f:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Jb)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x0f) {
		case 0x00: NAME_(jo)(cpssp); break;
		case 0x01: NAME_(jno)(cpssp); break;
		case 0x02: NAME_(jb)(cpssp); break;
		case 0x03: NAME_(jnb)(cpssp); break;
		case 0x04: NAME_(jz)(cpssp); break;
		case 0x05: NAME_(jnz)(cpssp); break;
		case 0x06: NAME_(jbe)(cpssp); break;
		case 0x07: NAME_(jnbe)(cpssp); break;
		case 0x08: NAME_(js)(cpssp); break;
		case 0x09: NAME_(jns)(cpssp); break;
		case 0x0a: NAME_(jp)(cpssp); break;
		case 0x0b: NAME_(jnp)(cpssp); break;
		case 0x0c: NAME_(jl)(cpssp); break;
		case 0x0d: NAME_(jnl)(cpssp); break;
		case 0x0e: NAME_(jle)(cpssp); break;
		case 0x0f: NAME_(jnle)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x80:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Eb_Ib)(cpssp);
		NAME_(alub)(cpssp, cpssp->NAME.instruction_reg);
		if (cpssp->NAME.instruction_reg != 7) NAME_(store_Eb)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x81:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Ev_Iv)(cpssp);
		NAME_(aluv)(cpssp, cpssp->NAME.instruction_reg);
		if (cpssp->NAME.instruction_reg != 7) NAME_(store_Ev)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x82: /* FIXME: undocumented */
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Eb_Ib)(cpssp); /* load_Eb_IbSE? FIXME */
		NAME_(alub)(cpssp, cpssp->NAME.instruction_reg);
		if (cpssp->NAME.instruction_reg != 7) NAME_(store_Eb)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x83:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ev_IbSE)(cpssp);
		NAME_(aluv)(cpssp, cpssp->NAME.instruction_reg);
		if (cpssp->NAME.instruction_reg != 7) NAME_(store_Ev)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x84:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb_Gb)(cpssp);
		NAME_(testb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x85:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_Gv)(cpssp);
		NAME_(testv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x86:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb_Gb)(cpssp);
		NAME_(xchg)(cpssp);
		NAME_(store_Eb_Gb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x87:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_Gv)(cpssp);
		NAME_(xchg)(cpssp);
		NAME_(store_Ev_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x88:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gb)(cpssp);
		NAME_(movb)(cpssp);
		NAME_(store_Eb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x89:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv)(cpssp);
		NAME_(movv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x8a:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb)(cpssp);
		NAME_(movb)(cpssp);
		NAME_(store_Gb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x8b:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev)(cpssp);
		NAME_(movv)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x8c:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Sw)(cpssp);
		NAME_(movw)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x8d:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_effective_address)(cpssp);
		NAME_(lea)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x8e:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ew)(cpssp);
		NAME_(movw)(cpssp);
		NAME_(store_Sw)(cpssp);
		if (cpssp->NAME.instruction_reg == SEGMENT_SS) {
			NAME_(commit_EIP_and_delay_interrupts)(cpssp);
		} else {
			NAME_(commit_EIP)(cpssp);
		}
		break;
	case 0x8f:
		NAME_(fetch_modrm)(cpssp);
		NAME_(popv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x90:
	case 0x91:
	case 0x92:
	case 0x93:
	case 0x94:
	case 0x95:
	case 0x96:
	case 0x97:
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00: NAME_(load_eAX_eAX)(cpssp); break;
		case 0x01: NAME_(load_eAX_eCX)(cpssp); break;
		case 0x02: NAME_(load_eAX_eDX)(cpssp); break;
		case 0x03: NAME_(load_eAX_eBX)(cpssp); break;
		case 0x04: NAME_(load_eAX_eSP)(cpssp); break;
		case 0x05: NAME_(load_eAX_eBP)(cpssp); break;
		case 0x06: NAME_(load_eAX_eSI)(cpssp); break;
		case 0x07: NAME_(load_eAX_eDI)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(xchg)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00: NAME_(store_eAX_eAX)(cpssp); break;
		case 0x01: NAME_(store_eAX_eCX)(cpssp); break;
		case 0x02: NAME_(store_eAX_eDX)(cpssp); break;
		case 0x03: NAME_(store_eAX_eBX)(cpssp); break;
		case 0x04: NAME_(store_eAX_eSP)(cpssp); break;
		case 0x05: NAME_(store_eAX_eBP)(cpssp); break;
		case 0x06: NAME_(store_eAX_eSI)(cpssp); break;
		case 0x07: NAME_(store_eAX_eDI)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x98:
		if (NAME_(operand_size_32)(cpssp)) {
			NAME_(cwde)(cpssp);
		} else {
			NAME_(cbw)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x99:
		if (NAME_(operand_size_32)(cpssp)) {
			NAME_(cdq)(cpssp);
		} else {
			NAME_(cwd)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x9a: /* lcall im */
		NAME_(fetch_ptr)(cpssp);
		NAME_(load_Ap)(cpssp);
		NAME_(call_far)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x9b: /* fwait */
		NAME_(fwait)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x9c: /* pushf */
		NAME_(pushf)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x9d: /* popf */
		NAME_(popf)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x9e: /* sahf */
		NAME_(sahf)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x9f: /* lahf */
		NAME_(lahf)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xa0: /* mov mem, AL */
		NAME_(fetch_immv)(cpssp, NAME_(address_size_32)(cpssp));
		NAME_(load_Ob)(cpssp);
		NAME_(movb)(cpssp);
		NAME_(store_AL)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xa1: /* mov mem, eAX */
		NAME_(fetch_immv)(cpssp, NAME_(address_size_32)(cpssp));
		NAME_(load_Ov)(cpssp);
		NAME_(movv)(cpssp);
		NAME_(store_eAX)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xa2: /* mov AL, mem */
		NAME_(fetch_immv)(cpssp, NAME_(address_size_32)(cpssp));
		NAME_(load_AL)(cpssp);
		NAME_(movb)(cpssp);
		NAME_(store_Ob)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xa3: /* mov eAX, mem */
		NAME_(fetch_immv)(cpssp, NAME_(address_size_32)(cpssp));
		NAME_(load_eAX)(cpssp);
		NAME_(movv)(cpssp);
		NAME_(store_Ov)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xa8: /* test Ib, AL */
		NAME_(fetch_immb)(cpssp);
		NAME_(load_AL_Ib)(cpssp);
		NAME_(testb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0xa9: /* test Iv, eAX */
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_eAX_Iv)(cpssp);
		NAME_(testv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0xb0:
	case 0xb1:
	case 0xb2:
	case 0xb3:
	case 0xb4:
	case 0xb5:
	case 0xb6:
	case 0xb7: /* mov Ib, REG */
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(movb)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00: NAME_(store_AL)(cpssp); break;
		case 0x01: NAME_(store_CL)(cpssp); break;
		case 0x02: NAME_(store_DL)(cpssp); break;
		case 0x03: NAME_(store_BL)(cpssp); break;
		case 0x04: NAME_(store_AH)(cpssp); break;
		case 0x05: NAME_(store_CH)(cpssp); break;
		case 0x06: NAME_(store_DH)(cpssp); break;
		case 0x07: NAME_(store_BH)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xb8:
	case 0xb9:
	case 0xba:
	case 0xbb:
	case 0xbc:
	case 0xbd:
	case 0xbe:
	case 0xbf: /* mov Iv, REG */
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Iv)(cpssp);
		NAME_(movv)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x07) {
		case 0x00: NAME_(store_eAX)(cpssp); break;
		case 0x01: NAME_(store_eCX)(cpssp); break;
		case 0x02: NAME_(store_eDX)(cpssp); break;
		case 0x03: NAME_(store_eBX)(cpssp); break;
		case 0x04: NAME_(store_eSP)(cpssp); break;
		case 0x05: NAME_(store_eBP)(cpssp); break;
		case 0x06: NAME_(store_eSI)(cpssp); break;
		case 0x07: NAME_(store_eDI)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc0:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Eb_Ib)(cpssp);
		NAME_(shiftb)(cpssp, cpssp->NAME.instruction_reg);
		NAME_(store_Eb)(cpssp);
		if (cpssp->NAME.instruction_reg & 4) {
			NAME_(commit_OSZAPC)(cpssp);
		} else {
			NAME_(commit_OF)(cpssp); NAME_(commit_CF)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc1:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ev_Ib)(cpssp);
		NAME_(shiftv)(cpssp, cpssp->NAME.instruction_reg);
		NAME_(store_Ev)(cpssp);
		if (cpssp->NAME.instruction_reg & 4) {
			NAME_(commit_OSZAPC)(cpssp);
		} else {
			NAME_(commit_OF)(cpssp); NAME_(commit_CF)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc2:
		NAME_(fetch_immv)(cpssp, 0);
		NAME_(load_Iw)(cpssp);
		NAME_(ret_near)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc3:
		NAME_(load_0)(cpssp);
		NAME_(ret_near)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc4:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Mp)(cpssp);
		NAME_(les)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc5:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Mp)(cpssp);
		NAME_(lds)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc6:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(movb)(cpssp);
		NAME_(store_Eb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc7:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Iv)(cpssp);
		NAME_(movv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc8:
		NAME_(load_Iw_Ib)(cpssp);
		NAME_(enter)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xc9:
		NAME_(leave)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xca:
		NAME_(fetch_immv)(cpssp, 0);
		NAME_(load_Iw)(cpssp);
		NAME_(ret_far)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xcb:
		NAME_(load_0)(cpssp);
		NAME_(ret_far)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xcc:
		NAME_(int3)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xcd:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(intn)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xce:
		NAME_(into)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xcf:
		NAME_(iret)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd0: /* shift Eb, 1 */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb_1)(cpssp);
		NAME_(shiftb)(cpssp, cpssp->NAME.instruction_reg);
		NAME_(store_Eb)(cpssp);
		if (cpssp->NAME.instruction_reg & 4) {
			NAME_(commit_OSZAPC)(cpssp);
		} else {
			NAME_(commit_OF)(cpssp); NAME_(commit_CF)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd1: /* shift Ev, 1 */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_1)(cpssp);
		NAME_(shiftv)(cpssp, cpssp->NAME.instruction_reg);
		NAME_(store_Ev)(cpssp);
		if (cpssp->NAME.instruction_reg & 4) {
			NAME_(commit_OSZAPC)(cpssp);
		} else {
			NAME_(commit_OF)(cpssp); NAME_(commit_CF)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd2: /* shift Eb, CL */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb_CL)(cpssp);
		NAME_(shiftb)(cpssp, cpssp->NAME.instruction_reg);
		NAME_(store_Eb)(cpssp);
		if (cpssp->NAME.instruction_reg & 4) {
			NAME_(commit_OSZAPC)(cpssp);
		} else {
			NAME_(commit_OF)(cpssp); NAME_(commit_CF)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd3: /* shift Ev, CL */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_CL)(cpssp);
		NAME_(shiftv)(cpssp, cpssp->NAME.instruction_reg);
		NAME_(store_Ev)(cpssp);
		if (cpssp->NAME.instruction_reg & 4) {
			NAME_(commit_OSZAPC)(cpssp);
		} else {
			NAME_(commit_OF)(cpssp); NAME_(commit_CF)(cpssp);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd4: /* aam AX, Ib */
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(aam)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0xd5: /* aad AX, Ib */
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(aad)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0xd6: /* salc */ /* Undocumented: Set AL on Carry */
		NAME_(salc)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd7: /* xlat */
		NAME_(xlat)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xd8:
	case 0xd9:
	case 0xda:
	case 0xdb:
	case 0xdc:
	case 0xdd:
	case 0xde:
	case 0xdf:
		NAME_(fetch_modrm)(cpssp);
		if (NAME_(get_EM)(cpssp)) {
			NAME_(exception)(cpssp, EXCEPTION_NM, -1);
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe0:
	case 0xe1:
	case 0xe2:
	case 0xe3:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Jb)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x03) {
		case 0x00: NAME_(loopne)(cpssp); break;
		case 0x01: NAME_(loope)(cpssp); break;
		case 0x02: NAME_(loop)(cpssp); break;
		case 0x03: NAME_(jcxz)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe4:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(inb)(cpssp);
		NAME_(store_AL)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe5:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib)(cpssp);
		NAME_(inv)(cpssp);
		NAME_(store_eAX)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe6:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib_AL)(cpssp);
		NAME_(outb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe7:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ib_eAX)(cpssp);
		NAME_(outv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe8:
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Jv)(cpssp);
		NAME_(call_near)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xe9:
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Jv)(cpssp);
		NAME_(jmp_near)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xea:
		NAME_(fetch_ptr)(cpssp);
		NAME_(load_Ap)(cpssp);
		NAME_(jmp_far)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xeb:
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Jb)(cpssp);
		NAME_(jmp_near)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xec:
		NAME_(load_DX)(cpssp);
		NAME_(inb)(cpssp);
		NAME_(store_AL)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xed:
		NAME_(load_DX)(cpssp);
		NAME_(inv)(cpssp);
		NAME_(store_eAX)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xee:
		NAME_(load_DX_AL)(cpssp);
		NAME_(outb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xef:
		NAME_(load_DX_eAX)(cpssp);
		NAME_(outv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xf1:
		NAME_(icebp)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xf4:
		NAME_(hlt)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xf5:
		NAME_(cmc)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xf6:
		NAME_(fetch_modrm)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: DEBUG_ERROR_SWITCH();
		case 0:
			NAME_(fetch_immb)(cpssp);
			NAME_(load_Eb_Ib)(cpssp);
			NAME_(testb)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 1:
			/* FIXME: RESERVED */
			ERROR_NYI();
			break;
		case 2:
			NAME_(load_Eb)(cpssp);
			NAME_(notb)(cpssp);
			NAME_(store_Eb)(cpssp);
			break;
		case 3:
			NAME_(load_Eb)(cpssp);
			NAME_(negb)(cpssp);
			NAME_(store_Eb)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 4:
			NAME_(load_Eb)(cpssp);
			NAME_(mulb)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 5:
			NAME_(load_Eb)(cpssp);
			NAME_(imul1b)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 6:
			NAME_(load_Eb)(cpssp);
			NAME_(divb)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 7:
			NAME_(load_Eb)(cpssp);
			NAME_(idivb)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xf7:
		NAME_(fetch_modrm)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: DEBUG_ERROR_SWITCH();
		case 0:
			NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
			NAME_(load_Ev_Iv)(cpssp);
			NAME_(testv)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 1:
			/* FIXME: RESERVED */
			ERROR_NYI();
			break;
		case 2:
			NAME_(load_Ev)(cpssp);
			NAME_(notv)(cpssp);
			NAME_(store_Ev)(cpssp);
			break;
		case 3:
			NAME_(load_Ev)(cpssp);
			NAME_(negv)(cpssp);
			NAME_(store_Ev)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 4:
			NAME_(load_Ev)(cpssp);
			NAME_(mulv)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 5:
			NAME_(load_Ev)(cpssp);
			NAME_(imul1v)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 6:
			NAME_(load_Ev)(cpssp);
			NAME_(divv)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 7:
			NAME_(load_Ev)(cpssp);
			NAME_(idivv)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xf8:
	case 0xf9:
	case 0xfa:
	case 0xfb:
	case 0xfc:
	case 0xfd:
		switch (cpssp->NAME.instruction_opcode) {
		case 0xf8: NAME_(clc)(cpssp); NAME_(commit_CF)(cpssp); break;
		case 0xf9: NAME_(stc)(cpssp); NAME_(commit_CF)(cpssp); break;
		case 0xfa: NAME_(cli)(cpssp); NAME_(commit_IF)(cpssp); break;
		case 0xfb: NAME_(sti)(cpssp); NAME_(commit_IF)(cpssp); break;
		case 0xfc: NAME_(cld)(cpssp); NAME_(commit_DF)(cpssp); break;
		case 0xfd: NAME_(std)(cpssp); NAME_(commit_DF)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		if (cpssp->NAME.instruction_opcode == 0xfb) {
			NAME_(commit_EIP_and_delay_interrupts)(cpssp);
		} else {
			NAME_(commit_EIP)(cpssp);
		}
		break;
	case 0xfe:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: DEBUG_ERROR_SWITCH();
		case 0: NAME_(incb)(cpssp); NAME_(commit_OSZAP)(cpssp); break;
		case 1: NAME_(decb)(cpssp); NAME_(commit_OSZAP)(cpssp); break;
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		case 7:
			/* TODO reserved */
			ERROR_NYI();
		}
		NAME_(store_Eb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0xff:
		NAME_(fetch_modrm)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: DEBUG_ERROR_SWITCH();
		case 0:
			NAME_(load_Ev)(cpssp);
			NAME_(incv)(cpssp);
			NAME_(store_Ev)(cpssp);
			NAME_(commit_OSZAP)(cpssp);
			break;
		case 1:
			NAME_(load_Ev)(cpssp);
			NAME_(decv)(cpssp);
			NAME_(store_Ev)(cpssp);
			NAME_(commit_OSZAP)(cpssp);
			break;
		case 2:
			NAME_(load_Ev)(cpssp);
			NAME_(call_near_absolute)(cpssp);
			break;
		case 3:
			NAME_(load_Ep)(cpssp);
			NAME_(call_far)(cpssp);
			break;
		case 4:
			NAME_(load_Ev)(cpssp);
			NAME_(jmp_near_absolute)(cpssp);
			break;
		case 5:
			NAME_(load_Ep)(cpssp);
			NAME_(jmp_far)(cpssp);
			break;
		case 6:
			NAME_(load_Ev)(cpssp);
			NAME_(pushv)(cpssp);
			break;
		case 7:
			/* TODO reserved */
			ERROR_NYI();
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x100:
		NAME_(fetch_modrm)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: ERROR_NYI();
		/* TODO */
		case 0:
			NAME_(sldt)(cpssp);
			NAME_(store_Ew)(cpssp);
			break;
		case 1:
			NAME_(str)(cpssp);
			NAME_(store_Ew)(cpssp);
			break;
		case 2:
			NAME_(load_Ew)(cpssp);
			NAME_(lldt)(cpssp);
			break;
		case 3:
			NAME_(load_Ew)(cpssp);
			NAME_(ltr)(cpssp);
			break;
		/* TODO */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x101:
		NAME_(fetch_modrm)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: ERROR_NYI();
		case 0:
			NAME_(sgdt)(cpssp);
			NAME_(store_Ms)(cpssp);
			break;
		case 1:
			NAME_(sidt)(cpssp);
			NAME_(store_Ms)(cpssp);
			break;
		case 2:
			NAME_(load_Ms)(cpssp);
			NAME_(lgdt)(cpssp);
			break;
		case 3:
			NAME_(load_Ms)(cpssp);
			NAME_(lidt)(cpssp);
			break;
		case 4:
			NAME_(smsw)(cpssp);
			NAME_(store_Ew)(cpssp);
			break;
		/* TODO */
		case 6:
			NAME_(load_Ew)(cpssp);
			NAME_(lmsw)(cpssp);
			break;
		case 7:
			NAME_(exception)(cpssp, EXCEPTION_UD, -1); /* 486+ */
			break;
		}
		NAME_(commit_EIP)(cpssp);
		break;
	/* TODO */
	case 0x106:
		NAME_(clts)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
#if 0
	case 0x107: /* Undocumented: LOADALL: Load all registers from ES:ESI */
		loadall();
		NAME_(commit_EIP)(cpssp);
		break;
#endif
#if 80686 <= CONFIG_CPU
	case 0x10b: /* Undocumented: UD2: Undefined Opcode 2. */
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
#endif
#if 0
	case 0x110:
	case 0x111:
	case 0x112:
	case 0x113: /* Undocumented: UMOV: Move User Data */
		/* FIXME */
		break;
#endif
	/* TODO */
	case 0x120:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Cd)(cpssp);
		NAME_(movd)(cpssp);
		NAME_(store_Rd)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x121:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Dd)(cpssp);
		NAME_(movd)(cpssp);
		NAME_(store_Rd)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x122:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Rd)(cpssp);
		NAME_(movd)(cpssp);
		NAME_(store_Cd)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x123:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Rd)(cpssp);
		NAME_(movd)(cpssp);
		NAME_(store_Dd)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x124: /* Move from Test Register */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Td)(cpssp);
		NAME_(movd)(cpssp);
		NAME_(store_Rd)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x126: /* Move To Test Register */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Rd)(cpssp);
		NAME_(movd)(cpssp);
		NAME_(store_Td)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x16f:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1); /* MMX */
		break;
#if 80686 <= CONFIG_CPU
	case 0x140:
	case 0x141:
	case 0x142:
	case 0x143:
	case 0x144:
	case 0x145:
	case 0x146:
	case 0x147:
	case 0x148:
	case 0x149:
	case 0x14a:
	case 0x14b:
	case 0x14c:
	case 0x14d:
	case 0x14e:
	case 0x14f: /* Undocumented: CMOV: Conditional Move. */
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv_Ev)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0xf) {
		case 0x00: cmovo(); break;
		case 0x01: cmovno(); break;
		case 0x02: cmovb(); break;
		case 0x03: cmovnb(); break;
		case 0x04: cmovz(); break;
		case 0x05: cmovnz(); break;
		case 0x06: cmovbe(); break;
		case 0x07: cmovnbe(); break;
		case 0x08: cmovs(); break;
		case 0x09: cmovns(); break;
		case 0x0a: cmovp(); break;
		case 0x0b: cmovnp(); break;
		case 0x0c: cmovl(); break;
		case 0x0d: cmovnl(); break;
		case 0x0e: cmovle(); break;
		case 0x0f: cmovnle(); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(store_Gv)(cpssp);
		break;
#endif
	case 0x180:
	case 0x181:
	case 0x182:
	case 0x183:
	case 0x184:
	case 0x185:
	case 0x186:
	case 0x187:
	case 0x188:
	case 0x189:
	case 0x18a:
	case 0x18b:
	case 0x18c:
	case 0x18d:
	case 0x18e:
	case 0x18f:
		NAME_(fetch_immv)(cpssp, NAME_(operand_size_32)(cpssp));
		NAME_(load_Jv)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x0f) {
		case 0x00: NAME_(jo)(cpssp); break;
		case 0x01: NAME_(jno)(cpssp); break;
		case 0x02: NAME_(jb)(cpssp); break;
		case 0x03: NAME_(jnb)(cpssp); break;
		case 0x04: NAME_(jz)(cpssp); break;
		case 0x05: NAME_(jnz)(cpssp); break;
		case 0x06: NAME_(jbe)(cpssp); break;
		case 0x07: NAME_(jnbe)(cpssp); break;
		case 0x08: NAME_(js)(cpssp); break;
		case 0x09: NAME_(jns)(cpssp); break;
		case 0x0a: NAME_(jp)(cpssp); break;
		case 0x0b: NAME_(jnp)(cpssp); break;
		case 0x0c: NAME_(jl)(cpssp); break;
		case 0x0d: NAME_(jnl)(cpssp); break;
		case 0x0e: NAME_(jle)(cpssp); break;
		case 0x0f: NAME_(jnle)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x190:
	case 0x191:
	case 0x192:
	case 0x193:
	case 0x194:
	case 0x195:
	case 0x196:
	case 0x197:
	case 0x198:
	case 0x199:
	case 0x19a:
	case 0x19b:
	case 0x19c:
	case 0x19d:
	case 0x19e:
	case 0x19f:
		NAME_(fetch_modrm)(cpssp);
		switch (cpssp->NAME.instruction_opcode & 0x0f) {
		case 0x00: NAME_(seto)(cpssp); break;
		case 0x01: NAME_(setno)(cpssp); break;
		case 0x02: NAME_(setb)(cpssp); break;
		case 0x03: NAME_(setnb)(cpssp); break;
		case 0x04: NAME_(setz)(cpssp); break;
		case 0x05: NAME_(setnz)(cpssp); break;
		case 0x06: NAME_(setbe)(cpssp); break;
		case 0x07: NAME_(setnbe)(cpssp); break;
		case 0x08: NAME_(sets)(cpssp); break;
		case 0x09: NAME_(setns)(cpssp); break;
		case 0x0a: NAME_(setp)(cpssp); break;
		case 0x0b: NAME_(setnp)(cpssp); break;
		case 0x0c: NAME_(setl)(cpssp); break;
		case 0x0d: NAME_(setnl)(cpssp); break;
		case 0x0e: NAME_(setle)(cpssp); break;
		case 0x0f: NAME_(setnle)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(store_Eb)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1a0:
	case 0x1a8:
		switch (cpssp->NAME.instruction_opcode) {
		case 0xa0: NAME_(load_FS)(cpssp); break;
		case 0xa8: NAME_(load_GS)(cpssp); break;
		default: assert(0); /* Cannot happen. */
		}
		NAME_(pushv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1a1:
	case 0x1a9:
		NAME_(popv)(cpssp);
		switch (cpssp->NAME.instruction_opcode) {
		case 0xa1:
			NAME_(store_FS)(cpssp);
			break;
		case 0xa9:
			NAME_(store_GS)(cpssp);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1a2:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
	case 0x1a3:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_EvBIT_Gv)(cpssp);
		NAME_(btvv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_CF)(cpssp);
		break;
	case 0x1a4:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ev_Gv_Ib)(cpssp);
		NAME_(shldv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x1a5:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_Gv_CL)(cpssp);
		NAME_(shldv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x1ab:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_EvBIT_Gv)(cpssp);
		NAME_(btsvv)(cpssp);
		NAME_(store_EvBIT)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_CF)(cpssp);
		break;
	case 0x1ac:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		NAME_(load_Ev_Gv_Ib)(cpssp);
		NAME_(shrdv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	case 0x1ad:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ev_Gv_CL)(cpssp);
		NAME_(shrdv)(cpssp);
		NAME_(store_Ev)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	/* TODO */
	case 0x1af:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv_Ev)(cpssp);
		NAME_(imul2v)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		break;
	/* TODO */
	case 0x1b2:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Mp)(cpssp);
		NAME_(lss)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1b3:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_EvBIT_Gv)(cpssp);
		NAME_(btrvv)(cpssp);
		NAME_(store_EvBIT)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1b4:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Mp)(cpssp);
		NAME_(lfs)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1b5:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Mp)(cpssp);
		NAME_(lgs)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1b6:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb)(cpssp);
		NAME_(movzxb)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1b7:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ew)(cpssp);
		NAME_(movzxw)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	/* TODO */
#if 80686 <= CONFIG_CPU
	case 0x1b9: /* Undocumented: UD2: Undefined Opcode 2. */
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
#endif
	case 0x1ba:
		NAME_(fetch_modrm)(cpssp);
		NAME_(fetch_immb)(cpssp);
		switch (cpssp->NAME.instruction_reg) {
		default: ERROR_NYI();
		/* TODO */
		case 5:
			NAME_(load_Ev_Ib)(cpssp);
			NAME_(btsvb)(cpssp);
			NAME_(store_Ev)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 6:
			NAME_(load_Ev_Ib)(cpssp);
			NAME_(btrvb)(cpssp);
			NAME_(store_Ev)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		case 7:
			NAME_(load_Ev_Ib)(cpssp);
			NAME_(btcvb)(cpssp);
			NAME_(store_Ev)(cpssp);
			NAME_(commit_OSZAPC)(cpssp);
			break;
		}
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1bb:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_EvBIT_Gv)(cpssp);
		NAME_(btcvv)(cpssp);
		NAME_(store_EvBIT)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1bc:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv_Ev)(cpssp);
		NAME_(bsfv)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1bd:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Gv_Ev)(cpssp);
		NAME_(bsrv)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_OSZAPC)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1be:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Eb)(cpssp);
		NAME_(movsxb)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	case 0x1bf:
		NAME_(fetch_modrm)(cpssp);
		NAME_(load_Ew)(cpssp);
		NAME_(movsxw)(cpssp);
		NAME_(store_Gv)(cpssp);
		NAME_(commit_EIP)(cpssp);
		break;
	/* TODO */
	case 0x1c1:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1); /* 486+ */
		break;
	/* TODO */
	case 0x1c8:
	case 0x1c9:
	case 0x1ca:
	case 0x1cb:
	case 0x1cc:
	case 0x1cd:
	case 0x1ce:
	case 0x1cf:
		NAME_(exception)(cpssp, EXCEPTION_UD, -1);
		break;
	/* TODO */
	}
end:	;
}

#endif /* BEHAVIOR */
