/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    itkTimeStampTest.cxx
  Language:  C++

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif
#include <iostream>
#include "itkTimeStamp.h"
#include "itkMultiThreader.h"

// A helper struct for the test, the idea is to have one timestamp per thread.
// To ease the writing of the test, we use  MultiThreader::SingleMethodExecute
// with an array of timestamps in the shared data
typedef struct {
  std::vector<itk::TimeStamp> timestamps;
  std::vector<long int> counters;
} TimeStampTestHelper;
 
ITK_THREAD_RETURN_TYPE modified_function( void *ptr )
{
  typedef itk::MultiThreader::ThreadInfoStruct  ThreadInfoType;

  ThreadInfoType * infoStruct = static_cast< ThreadInfoType * >( ptr );

  const unsigned int threadId = infoStruct->ThreadID;

  TimeStampTestHelper * helper = 
    static_cast< TimeStampTestHelper * >( infoStruct->UserData );

  helper->timestamps[threadId].Modified();
  helper->counters[threadId]++;

  return ITK_THREAD_RETURN_VALUE;
}

int itkTimeStampTest(int, char*[])
{
  bool success = true;

  try
    {
    TimeStampTestHelper helper;
       
    // Set up the multithreader
    itk::MultiThreader::Pointer multithreader = itk::MultiThreader::New();
    multithreader->SetNumberOfThreads( ITK_MAX_THREADS+10 );// this will be clamped
    multithreader->SetSingleMethod( modified_function, &helper);

    // Test that the number of threads has actually been clamped 
    const long int numberOfThreads = 
      static_cast<long int>( multithreader->GetNumberOfThreads() );

    if( numberOfThreads > ITK_MAX_THREADS )
      {
      std::cerr << "[TEST FAILED]" << std::endl;
      std::cerr << "numberOfThreads > ITK_MAX_THREADS" << std::endl;
      return EXIT_FAILURE;
      }

    // Set up the helper class
    helper.counters.resize( numberOfThreads );
    helper.timestamps.resize( numberOfThreads );    
    for(int k=0; k < numberOfThreads; k++)
    {
       helper.counters[k] = 0;
    }

    // Declare an array to test whether the all modified times have
    // been used
    std::vector<bool> istimestamped( numberOfThreads );

    

    // Call Modified once  on any object to make it up-to-date
    multithreader->Modified();

    const long int init_mtime = multithreader->GetMTime();
    std::cout << "init_mtime: " << init_mtime << std::endl;

    long int prev_mtime = init_mtime;

    const int num_exp = 500;

    for( int i = 0; i < num_exp; i++ )
      {
      multithreader->SingleMethodExecute();

      long int min_mtime = helper.timestamps[0].GetMTime();
      long int max_mtime = helper.timestamps[0].GetMTime();
      for(int k=0; k < numberOfThreads; k++)
        {
        const long int & mtime = helper.timestamps[k].GetMTime();
        if ( mtime > max_mtime )
          {
          max_mtime = mtime;
          }
        else if ( mtime < min_mtime )
          {
          min_mtime = mtime;
          }

        // initialiaze the array to false
        istimestamped[k]=false;
        }

      bool iter_success =
             ( ((max_mtime-prev_mtime )==numberOfThreads) &&
               (min_mtime==prev_mtime+1) );

      if ( iter_success )
        {
        for(int k=0; k < numberOfThreads; k++)
          {
          // Test whether the all modified times have
          // been used
          const long int index = helper.timestamps[k].GetMTime()-min_mtime;

          if ( istimestamped[index] == true )
            {
            iter_success = false;
            std::cerr<<helper.timestamps[k].GetMTime()<<" was used twice as a timestamp!"<<std::endl;
            }
          else
            {
            istimestamped[index] = true;
            }

          // Test the counters
          if( helper.counters[k] != i+1 )
            {
            iter_success = false;
            std::cerr << "counter[" << k << "] = " << helper.counters[k];
            std::cerr << " at iteration " << i << std::endl;
            }
          }
      }
      
      if( !iter_success )
        {
        std::cerr << "[Iteration " << i << " FAILED]" << std::endl;
        std::cerr << "max_mtime       : " << max_mtime << std::endl;
        std::cerr << "min_mtime       : " << min_mtime << std::endl;
        std::cerr << "prev_mtime      : " << prev_mtime << std::endl;
        std::cerr << "num_threads     : " << numberOfThreads << std::endl;
        std::cerr << "max - prev mtime: " << max_mtime - prev_mtime << std::endl;
        std::cerr << std::endl;
        success = false;

        // Note that in a more general setting,  (max_mtime-prev_mtime)>numberOfThreads
        // might be a normal case since the modified time of a time stamp
        // is global. If a new itk object is created this will also increment
        // the time. In our specific test, there's no reason for another ITK object to be
        // modified though
        }

      prev_mtime = max_mtime;
      }
    }
  catch (itk::ExceptionObject &e)
    {
    std::cerr << "[TEST FAILED]" << std::endl;
    std::cerr << "Exception caught: "<< e << std::endl;
    return EXIT_FAILURE;
    }

  if (!success)
    {
    std::cerr << "[TEST FAILED]" << std::endl;
    return EXIT_FAILURE;
    }

  std::cout << "[TEST PASSED]" << std::endl;
  return EXIT_SUCCESS;
}
