"""Tests for handling of OpenStack credentials in config and environment"""

from juju.lib import (
    testing,
    )
from juju.providers.openstack import (
    credentials,
    )


class OpenStackCredentialsTests(testing.TestCase):

    def test_required_url(self):
        e = self.assertRaises(ValueError, credentials.OpenStackCredentials, {})
        self.assertIn("Missing config 'auth-url'", str(e))

    def test_required_mode_if_unguessable(self):
        e = self.assertRaises(ValueError, credentials.OpenStackCredentials, {
            'auth-url': "http://example.com",
            })
        self.assertIn("Missing config 'auth-mode'", str(e))

    def test_legacy(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com",
            'auth-mode': "legacy",
            'username': "luser",
            'access-key': "laccess",
            })
        self.assertEqual(creds.url, "http://example.com")
        self.assertEqual(creds.mode, "legacy")
        self.assertEqual(creds.username, "luser")
        self.assertEqual(creds.access_key, "laccess")

    def test_legacy_required_username(self):
        e = self.assertRaises(ValueError, credentials.OpenStackCredentials, {
            'auth-url': "http://example.com",
            'auth-mode': "legacy",
            'access-key': "laccess",
            })
        self.assertIn("Missing config 'username'", str(e))

    def test_legacy_required_access_key(self):
        e = self.assertRaises(ValueError, credentials.OpenStackCredentials, {
            'auth-url': "http://example.com",
            'auth-mode': "legacy",
            'username': "luser",
            })
        self.assertIn("Missing config 'access-key'", str(e))

    # v1.0 auth is gone from the upstream codebase so maybe remove support
    def test_legacy_guess_v1_0(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com/v1.0/",
            'username': "luser",
            'access-key': "laccess",
            })
        self.assertEqual(creds.mode, "legacy")

    def test_legacy_guess_v1_1(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com/v1.1/",
            'username': "luser",
            'access-key': "laccess",
            })
        self.assertEqual(creds.mode, "legacy")

    def test_userpass(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com",
            'auth-mode': "userpass",
            'username': "uuser",
            'password': "upass",
            'project-name': "uproject",
            })
        self.assertEqual(creds.url, "http://example.com")
        self.assertEqual(creds.mode, "userpass")
        self.assertEqual(creds.username, "uuser")
        self.assertEqual(creds.password, "upass")
        self.assertEqual(creds.project_name, "uproject")

    def test_userpass_guess_v2_0_no_slash(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com/v2.0",
            'username': "uuser",
            'password': "upass",
            'project-name': "uproject",
            })
        self.assertEqual(creds.mode, "userpass")

    def test_userpass_guess_v2_0_slash(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com/v2.0/",
            'username': "uuser",
            'password': "upass",
            'project-name': "uproject",
            })
        self.assertEqual(creds.mode, "userpass")

    def test_keypair(self):
        creds = credentials.OpenStackCredentials({
            'auth-url': "http://example.com",
            'auth-mode': "keypair",
            'access-key': "kaccess",
            'secret-key': "ksecret",
            'project-name': "kproject",
            })
        self.assertEqual(creds.url, "http://example.com")
        self.assertEqual(creds.mode, "keypair")
        self.assertEqual(creds.access_key, "kaccess")
        self.assertEqual(creds.secret_key, "ksecret")
        self.assertEqual(creds.project_name, "kproject")


class FromEnvironmentTests(testing.TestCase):

    def test_required_url(self):
        self.change_environment()
        e = self.assertRaises(ValueError,
            credentials.OpenStackCredentials.from_environment, {})
        self.assertIn("Missing config 'auth-url'", str(e))

    def test_required_mode_if_unguessable(self):
        self.change_environment(**{"NOVA_URL": "http://example.com"})
        e = self.assertRaises(ValueError,
            credentials.OpenStackCredentials.from_environment, {})
        self.assertIn("Missing config 'auth-mode'", str(e))

    def test_legacy(self):
        self.change_environment(**{
            "NOVA_URL": "http://example.com/v1.1/",
            "NOVA_USERNAME": "euser",
            "NOVA_API_KEY": "ekey",
            })
        creds = credentials.OpenStackCredentials.from_environment({})
        self.assertEqual(creds.mode, "legacy")
        self.assertEqual(creds.username, "euser")
        self.assertEqual(creds.access_key, "ekey")

    def test_keypair(self):
        self.change_environment(**{
            "NOVA_URL": "http://example.com/v2.0/",
            "NOVA_API_KEY": "eaccess",
            "NOVA_PROJECT_NAME": "eproject",
            "NOVA_PROJECT_ID": "349212",
            "NOVA_REGION_NAME": "eregion",
            "EC2_SECRET_KEY": "esecret",
            })
        creds = credentials.OpenStackCredentials.from_environment({
            'auth-mode': "keypair",
            })
        self.assertEqual(creds.access_key, "eaccess")
        self.assertEqual(creds.secret_key, "esecret")
        self.assertEqual(creds.project_name, "eproject")
        self.assertEqual(creds.region, "eregion")

    def test_userpass(self):
        self.change_environment(**{
            "OS_AUTH_URL": "http://example.com/v2.0/",
            "OS_USERNAME": "euser",
            "OS_PASSWORD": "epass",
            "OS_TENANT_NAME": "eproject",
            "OS_REGION_NAME": "eregion",
            })
        creds = credentials.OpenStackCredentials.from_environment({})
        self.assertEqual(creds.mode, "userpass")
        self.assertEqual(creds.username, "euser")
        self.assertEqual(creds.password, "epass")
        self.assertEqual(creds.project_name, "eproject")
        self.assertEqual(creds.region, "eregion")

    def test_prefer_os_auth_url(self):
        self.change_environment(**{
            "NOVA_URL": "http://example.com/v1.1/",
            "NOVA_API_KEY": "eaccess",
            "OS_AUTH_URL": "http://example.com/v2.0/",
            "OS_USERNAME": "euser",
            "OS_PASSWORD": "epass",
            "OS_TENANT_NAME": "eproject",
            })
        creds = credentials.OpenStackCredentials.from_environment({})
        self.assertEqual(creds.url, "http://example.com/v2.0/")
        self.assertEqual(creds.mode, "userpass")


class SetConfigDefaultsTests(testing.TestCase):

    def test_set_all(self):
        config = {
            'auth-url': "http://example.com",
            'auth-mode': "legacy",
            'username': "luser",
            'access-key': "laccess",
            }
        creds = credentials.OpenStackCredentials(config)
        new_config = {}
        creds.set_config_defaults(new_config)
        self.assertEqual(config, new_config)

    def test_set_only_missing(self):
        config = {
            'auth-url': "http://example.com/v1.1/",
            'username': "luser",
            'access-key': "laccess",
            }
        creds = credentials.OpenStackCredentials(config)
        new_config = {
            'username': "nuser",
            }
        creds.set_config_defaults(new_config)
        self.assertEqual({
            'auth-url': "http://example.com/v1.1/",
            'auth-mode': "legacy",
            'username': "nuser",
            'access-key': "laccess",
            }, new_config)
