from typing import Any, ClassVar, Dict, Sequence, List, Iterator, Optional
from typing import overload
import klayout.tl as tl
class Box:
    r"""
    @brief A box class with integer coordinates

    This object represents a box (a rectangular shape).

    The definition of the attributes is: p1 is the lower left point, p2 the 
    upper right one. If a box is constructed from two points (or four coordinates), the 
    coordinates are sorted accordingly.

    A box can be empty. An empty box represents no area
    (not even a point). Empty boxes behave neutral with respect to most operations. 
    Empty boxes return true on \empty?.

    A box can be a point or a single
    line. In this case, the area is zero but the box still
    can overlap other boxes for example and it is not empty. 

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bottom: int
    r"""
    Getter:
    @brief Gets the bottom coordinate of the box

    Setter:
    @brief Sets the bottom coordinate of the box
    """
    left: int
    r"""
    Getter:
    @brief Gets the left coordinate of the box

    Setter:
    @brief Sets the left coordinate of the box
    """
    p1: Point
    r"""
    Getter:
    @brief Gets the lower left point of the box

    Setter:
    @brief Sets the lower left point of the box
    """
    p2: Point
    r"""
    Getter:
    @brief Gets the upper right point of the box

    Setter:
    @brief Sets the upper right point of the box
    """
    right: int
    r"""
    Getter:
    @brief Gets the right coordinate of the box

    Setter:
    @brief Sets the right coordinate of the box
    """
    top: int
    r"""
    Getter:
    @brief Gets the top coordinate of the box

    Setter:
    @brief Sets the top coordinate of the box
    """
    @classmethod
    def from_dbox(cls, dbox: DBox) -> Box:
        r"""
        @brief Creates an integer coordinate box from a floating-point coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dbox'.
        """
    @classmethod
    def from_s(cls, s: str) -> Box:
        r"""
        @brief Creates a box object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Box:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    @classmethod
    def new(cls, dbox: DBox) -> Box:
        r"""
        @brief Creates an integer coordinate box from a floating-point coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dbox'.
        """
    @overload
    @classmethod
    def new(cls, left: int, bottom: int, right: int, top: int) -> Box:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    @classmethod
    def new(cls, lower_left: Point, upper_right: Point) -> Box:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    @classmethod
    def new(cls, w: int) -> Box:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, w: int, h: int) -> Box:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @classmethod
    def world(cls) -> Box:
        r"""
        @brief Gets the 'world' box
        The world box is the biggest box that can be represented. So it is basically 'all'. The world box behaves neutral on intersections for example. In other operations such as displacement or transformations, the world box may render unexpected results because of coordinate overflow.

        The world box can be used
        @ul
          @li for comparison ('==', '!=', '<') @/li
          @li in union and intersection ('+' and '&') @/li
          @li in relations (\contains?, \overlaps?, \touches?) @/li
          @li as 'all' argument in region queries @/li
        @/ul

        This method has been introduced in version 0.28.
        """
    @overload
    def __add__(self, box: Box) -> Box:
        r"""
        @brief Joins two boxes


        The + operator joins the first box with the one given as 
        the second argument. Joining constructs a box that encloses
        both boxes given. Empty boxes are neutral: they do not
        change another box when joining. Overwrites this box
        with the result.

        @param box The box to join with this box.

        @return The joined box
        """
    @overload
    def __add__(self, point: Point) -> Box:
        r"""
        @brief Joins box with a point


        The + operator joins a point with the box. The resulting box will enclose both the original box and the point.

        @param point The point to join with this box.

        @return The box joined with the point
        """
    def __and__(self, box: Box) -> Box:
        r"""
        @brief Returns the intersection of this box with another box


        The intersection of two boxes is the largest
        box common to both boxes. The intersection may be 
        empty if both boxes to not touch. If the boxes do
        not overlap but touch the result may be a single
        line or point with an area of zero. Overwrites this box
        with the result.

        @param box The box to take the intersection with

        @return The intersection box
        """
    def __copy__(self) -> Box:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Box:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is equal to the other box
        Returns true, if this box and the given box are equal 
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    def __init__(self, dbox: DBox) -> None:
        r"""
        @brief Creates an integer coordinate box from a floating-point coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dbox'.
        """
    @overload
    def __init__(self, left: int, bottom: int, right: int, top: int) -> None:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    def __init__(self, lower_left: Point, upper_right: Point) -> None:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    def __init__(self, w: int) -> None:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    def __init__(self, w: int, h: int) -> None:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    def __lt__(self, box: Box) -> bool:
        r"""
        @brief Returns true if this box is 'less' than another box
        Returns true, if this box is 'less' with respect to first and second point (in this order)
        """
    @overload
    def __mul__(self, box: Box) -> Box:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __mul__(self, scale_factor: float) -> Box:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is not equal to the other box
        Returns true, if this box and the given box are not equal 
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __rmul__(self, box: Box) -> Box:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __rmul__(self, scale_factor: float) -> Box:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __sub__(self, box: Box) -> Box:
        r"""
        @brief Subtraction of boxes


        The - operator subtracts the argument box from self.
        This will return the bounding box of the are covered by self, but not by argument box. Subtracting a box from itself will render an empty box. Subtracting another box from self will modify the first box only if the argument box covers one side entirely.

        @param box The box to subtract from this box.

        @return The result box

        This feature has been introduced in version 0.29.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Computes the box area

        Returns the box area or 0 if the box is empty
        """
    def assign(self, other: Box) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box
        This method is provided for consistency of the shape API is returns the box itself.

        This method has been introduced in version 0.27.
        """
    def center(self) -> Point:
        r"""
        @brief Gets the center of the box
        """
    @overload
    def contains(self, point: Point) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point is inside the box.
        It also returns true if the point is exactly on the box contour.

        @param p The point to test against.

        @return true if the point is inside the box.
        """
    @overload
    def contains(self, x: int, y: int) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point (x, y) is inside the box.
        It also returns true if the point is exactly on the box contour.

        @return true if the point is inside the box.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Box:
        r"""
        @brief Creates a copy of self
        """
    def empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the box is empty

        An empty box may be created with the default constructor for example. Such a box is neutral when combining it with other boxes and renders empty boxes if used in box intersections and false in geometrical relationship tests. 
        """
    @overload
    def enlarge(self, d: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return A reference to this box.
        """
    @overload
    def enlarge(self, dx: int, dy: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def enlarge(self, enlargement: Vector) -> Box:
        r"""
        @brief Enlarges the box by a certain amount.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return A reference to this box.
        """
    @overload
    def enlarged(self, d: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return The enlarged box.
        """
    @overload
    def enlarged(self, dx: int, dy: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return The enlarged box.
        """
    @overload
    def enlarged(self, enlargement: Vector) -> Box:
        r"""
        @brief Returns the enlarged box.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not modify this box. Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return The enlarged box.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    def height(self) -> int:
        r"""
        @brief Gets the height of the box
        """
    def inside(self, box: Box) -> bool:
        r"""
        @brief Tests if this box is inside the argument box


        Returns true, if this box is inside the given box, i.e. the box intersection renders this box
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_point(self) -> bool:
        r"""
        @brief Returns true, if the box is a single point
        """
    @overload
    def move(self, distance: Vector) -> Box:
        r"""
        @brief Moves the box by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not check for coordinate overflows.

        @param distance The offset to move the box.

        @return A reference to this box.
        """
    @overload
    def move(self, dx: int, dy: int) -> Box:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def moved(self, distance: Vector) -> Box:
        r"""
        @brief Returns the box moved by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not modify this box. Does not check for coordinate
        overflows.

        @param distance The offset to move the box.

        @return The moved box.
        """
    @overload
    def moved(self, dx: int, dy: int) -> Box:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return The moved box.
        """
    def overlaps(self, box: Box) -> bool:
        r"""
        @brief Tests if this box overlaps the argument box


        Returns true, if the intersection box of this box with the argument box exists and has a non-vanishing area
        """
    def perimeter(self) -> int:
        r"""
        @brief Returns the perimeter of the box

        This method is equivalent to 2*(width+height). For empty boxes, this method returns 0.

        This method has been introduced in version 0.23.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DBox:
        r"""
        @brief Converts the box to a floating-point coordinate box

        The database unit can be specified to translate the integer-coordinate box into a floating-point coordinate box in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def touches(self, box: Box) -> bool:
        r"""
        @brief Tests if this box touches the argument box


        Two boxes touch if they overlap or their boundaries share at least one common point. Touching is equivalent to a non-empty intersection ('!(b1 & b2).empty?').
        """
    @overload
    def transformed(self, t: CplxTrans) -> DBox:
        r"""
        @brief Returns the box transformed with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (a DBox now)
        """
    @overload
    def transformed(self, t: ICplxTrans) -> Box:
        r"""
        @brief Transforms the box with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (in this case an integer coordinate box)

        This method has been introduced in version 0.18.
        """
    @overload
    def transformed(self, t: Trans) -> Box:
        r"""
        @brief Returns the box transformed with the given simple transformation


        @param t The transformation to apply
        @return The transformed box
        """
    def width(self) -> int:
        r"""
        @brief Gets the width of the box
        """

class Cell:
    r"""
    @brief A cell

    A cell object consists of a set of shape containers (called layers),
    a set of child cell instances and auxiliary information such as
    the parent instance list.
    A cell is identified through an index given to the cell upon instantiation.
    Cell instances refer to single instances or array instances. Both are encapsulated in the
    same object, the \CellInstArray object. In the simple case, this object refers to a single instance.
    In the general case, this object may refer to a regular array of cell instances as well.

    Starting from version 0.16, the child_inst and erase_inst methods are no longer available since
    they were using index addressing which is no longer supported. Instead, instances are now addressed
    with the \Instance reference objects.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects like the Cell class.
    """
    ghost_cell: bool
    r"""
    Getter:
    @brief Returns a value indicating whether the cell is a "ghost cell"

    The ghost cell flag is used by the GDS reader for example to indicate that
    the cell is not located inside the file. Upon writing the reader can determine
    whether to write the cell or not.
    To satisfy the references inside the layout, a dummy cell is created in this case
    which has the "ghost cell" flag set to true.

    This method has been introduced in version 0.20.

    Setter:
    @brief Sets the "ghost cell" flag

    See \is_ghost_cell? for a description of this property.

    This method has been introduced in version 0.20.
    """
    name: str
    r"""
    Getter:
    @brief Gets the cell's name

    This may be an internal name for proxy cells. See \basic_name for the formal name (PCell name or library cell name).

    This method has been introduced in version 0.22.

    Setter:
    @brief Renames the cell
    Renaming a cell may cause name clashes, i.e. the name may be identical to the name
    of another cell. This does not have any immediate effect, but the cell needs to be renamed, for example when writing the layout to a GDS file.

    This method has been introduced in version 0.22.
    """
    prop_id: int
    r"""
    Getter:
    @brief Gets the properties ID associated with the cell

    This method has been introduced in version 0.23.
    Setter:
    @brief Sets the properties ID associated with the cell
    This method is provided, if a properties ID has been derived already. Usually it's more convenient to use \delete_property, \set_property or \property.

    This method has been introduced in version 0.23.
    """
    @classmethod
    def new(cls) -> Cell:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> Cell:
        r"""
        @brief Creates a copy of the cell

        This method will create a copy of the cell. The new cell will be member of the same layout the original cell was member of. The copy will inherit all shapes and instances, but get a different cell_index and a modified name as duplicate cell names are not allowed in the same layout.

        This method has been introduced in version 0.27.
        """
    def __deepcopy__(self) -> Cell:
        r"""
        @brief Creates a copy of the cell

        This method will create a copy of the cell. The new cell will be member of the same layout the original cell was member of. The copy will inherit all shapes and instances, but get a different cell_index and a modified name as duplicate cell names are not allowed in the same layout.

        This method has been introduced in version 0.27.
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_meta_info(self, info: LayoutMetaInfo) -> None:
        r"""
        @brief Adds meta information to the cell
        See \LayoutMetaInfo for details about cells and meta information.

        This method has been introduced in version 0.28.8.
        """
    def basic_name(self) -> str:
        r"""
        @brief Returns the name of the library or PCell or the real name of the cell
        For non-proxy cells (see \is_proxy?), this method simply returns the cell name.
        For proxy cells, this method returns the PCells definition name or the library
        cell name. This name may differ from the actual cell's name because to ensure
        that cell names are unique, KLayout may assign different names to the actual 
        cell compared to the source cell.

        This method has been introduced in version 0.22.
        """
    @overload
    def bbox(self) -> Box:
        r"""
        @brief Gets the bounding box of the cell

        @return The bounding box of the cell

        The bounding box is computed over all layers. To compute the bounding box over single layers, use \bbox with a layer index argument.
        """
    @overload
    def bbox(self, layer_index: int) -> Box:
        r"""
        @brief Gets the per-layer bounding box of the cell

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        'bbox' is the preferred synonym since version 0.28.
        """
    def bbox_per_layer(self, layer_index: int) -> Box:
        r"""
        @brief Gets the per-layer bounding box of the cell

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        'bbox' is the preferred synonym since version 0.28.
        """
    def begin_instances_rec(self) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.

        This method has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_overlapping(self, region: Box) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell using a region search
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box overlaps the given region.

        This method has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_overlapping(self, region: DBox) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell using a region search, with the region given in micrometer units
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box overlaps the given region.

        This variant has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_touching(self, region: Box) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box touches the given region.

        This method has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_touching(self, region: DBox) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell using a region search, with the region given in micrometer units
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box touches the given region.

        This variant has been added in version 0.27.
        """
    def begin_shapes_rec(self, layer: int) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer
        @param layer The layer from which to get the shapes
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.

        This method has been added in version 0.23.
        """
    @overload
    def begin_shapes_rec_overlapping(self, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.

        This method has been added in version 0.23.
        """
    @overload
    def begin_shapes_rec_overlapping(self, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search, with the region given in micrometer units
        @param layer The layer from which to get the shapes
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.

        This variant has been added in version 0.25.
        """
    @overload
    def begin_shapes_rec_touching(self, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.

        This method has been added in version 0.23.
        """
    @overload
    def begin_shapes_rec_touching(self, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search, with the region given in micrometer units
        @param layer The layer from which to get the shapes
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.

        This variant has been added in version 0.25.
        """
    def called_cells(self) -> List[int]:
        r"""
        @brief Gets a list of all called cells

        This method determines all cells which are called either directly or indirectly by the cell.
        It returns an array of cell indexes. Use the 'cell' method of \Layout to retrieve the corresponding Cell object.

        This method has been introduced in version 0.19.

        @return A list of cell indices.
        """
    def caller_cells(self) -> List[int]:
        r"""
        @brief Gets a list of all caller cells

        This method determines all cells which call this cell either directly or indirectly.
        It returns an array of cell indexes. Use the 'cell' method of \Layout to retrieve the corresponding Cell object.

        This method has been introduced in version 0.19.

        @return A list of cell indices.
        """
    def cell_index(self) -> int:
        r"""
        @brief Gets the cell index

        @return The cell index of the cell
        """
    def change_pcell_parameter(self, instance: Instance, name: str, value: Any) -> Instance:
        r"""
        @brief Changes a single parameter for an individual PCell instance given by name
        @return The new instance (the old may be invalid)
        This will set the PCell parameter named 'name' to the given value for the instance addressed by 'instance'. If no parameter with that name exists, the method will do nothing.

        This method has been introduced in version 0.23.
        """
    @overload
    def change_pcell_parameters(self, instance: Instance, dict: Dict[str, Any]) -> Instance:
        r"""
        @brief Changes the given parameter for an individual PCell instance
        @return The new instance (the old may be invalid)
        This version receives a dictionary of names and values. It will change the parameters given by the names to the values given by the values of the dictionary. The functionality is similar to the same function with an array, but more convenient to use.
        Values with unknown names are ignored.

        This method has been introduced in version 0.24.
        """
    @overload
    def change_pcell_parameters(self, instance: Instance, parameters: Sequence[Any]) -> Instance:
        r"""
        @brief Changes the parameters for an individual PCell instance
        @return The new instance (the old may be invalid)
        If necessary, this method creates a new variant and replaces the given instance
        by an instance of this variant.

        The parameters are given in the order the parameters are declared. Use \pcell_declaration on the instance to get the PCell declaration object of the cell. That PCellDeclaration object delivers the parameter declaration with its 'get_parameters' method.
        Each parameter in the variant list passed to the second list of values corresponds to one parameter declaration.

        There is a more convenient method (\change_pcell_parameter) that changes a single parameter by name.

        This method has been introduced in version 0.22.
        """
    def child_cells(self) -> int:
        r"""
        @brief Gets the number of child cells

        The number of child cells (not child instances!) is returned.
        CAUTION: this method is SLOW, in particular if many instances are present.
        """
    def child_instances(self) -> int:
        r"""
        @brief Gets the number of child instances

        @return Returns the number of cell instances
        """
    @overload
    def clear(self) -> None:
        r"""
        @brief Clears the cell (deletes shapes and instances)
        This method has been introduced in version 0.23.
        """
    @overload
    def clear(self, layer_index: int) -> None:
        r"""
        @brief Clears the shapes on the given layer
        """
    def clear_insts(self) -> None:
        r"""
        @brief Clears the instance list
        """
    def clear_meta_info(self) -> None:
        r"""
        @brief Clears the meta information of the cell
        See \LayoutMetaInfo for details about cells and meta information.

        This method has been introduced in version 0.28.8.
        """
    def clear_shapes(self) -> None:
        r"""
        @brief Clears all shapes in the cell
        """
    @overload
    def copy(self, src: int, dest: int) -> None:
        r"""
        @brief Copies the shapes from the source to the target layer

        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        If source are target layer are identical, this method does nothing.
        This method will copy shapes within the cell. To copy shapes from another cell to this cell, use the copy method with the cell parameter.

        This method has been introduced in version 0.19.

        @param src The layer index of the source layer
        @param dest The layer index of the destination layer
        """
    @overload
    def copy(self, src_cell: Cell, src_layer: int, dest: int) -> None:
        r"""
        @brief Copies shapes from another cell to the target layer in this cell

        This method will copy all shapes on layer 'src_layer' of cell 'src_cell' to the layer 'dest' of this cell.
        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        If the source cell lives in a layout with a different database unit than that current cell is in, the shapes will be transformed accordingly. The same way, shape properties are transformed as well. Note that the shape transformation may require rounding to smaller coordinates. This may result in a slight distortion of the original shapes, in particular when transforming into a layout with a bigger database unit.
        @param src_cell The cell where to take the shapes from
        @param src_layer The layer index of the layer from which to take the shapes
        @param dest The layer index of the destination layer
        """
    def copy_instances(self, source_cell: Cell) -> None:
        r"""
        @brief Copies the instances of child cells in the source cell to this cell
        @param source_cell The cell where the instances are copied from
        The source cell must reside in the same layout than this cell. The instances of child cells inside the source cell are copied to this cell. No new cells are created, just new instances are created to already existing cells in the target cell.

        The instances will be added to any existing instances in the cell.

        More elaborate methods of copying hierarchy trees between layouts or duplicating trees are provided through the \copy_tree_shapes (in cooperation with the \CellMapping class) or \copy_tree methods.

        This method has been added in version 0.23.
        """
    def copy_meta_info(self, other: Cell) -> None:
        r"""
        @brief Copies the meta information from the other cell into this cell
        See \LayoutMetaInfo for details about cells and meta information.
        The meta information from this cell will be replaced by the meta information from the other cell.

        This method has been introduced in version 0.28.16.
        """
    @overload
    def copy_shapes(self, source_cell: Cell) -> None:
        r"""
        @brief Copies the shapes from the given cell into this cell
        @param source_cell The cell from where to copy shapes
        All shapes are copied from the source cell to this cell. Instances are not copied.

        The source cell can reside in a different layout. In this case, the shapes are copied over from the other layout into this layout. Database unit conversion is done automatically if the database units differ between the layouts. Note that this may lead to grid snapping effects if the database unit of the target layout is not an integer fraction of the source layout.

        If source and target layout are different, the layers of the source and target layout are identified by their layer/datatype number or name (if no layer/datatype is present).
        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def copy_shapes(self, source_cell: Cell, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Copies the shapes from the given cell into this cell
        @param source_cell The cell from where to copy shapes
        @param layer_mapping A \LayerMapping object that specifies which layers are copied and where
        All shapes on layers specified in the layer mapping object are copied from the source cell to this cell. Instances are not copied.
        The target layer is taken from the mapping table.

        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    def copy_tree(self, source_cell: Cell) -> List[int]:
        r"""
        @brief Copies the cell tree of the given cell into this cell
        @param source_cell The cell from where to copy the cell tree
        @return A list of indexes of newly created cells
        The complete cell tree of the source cell is copied to the target cell plus all shapes in that tree are copied as well. This method will basically duplicate the cell tree of the source cell.

        The source cell may reside in a separate layout. This method therefore provides a way to copy over complete cell trees from one layout to another.

        The shapes and instances will be added to any shapes or instances already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def copy_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping) -> None:
        r"""
        @brief Copies the shapes from the given cell and the cell tree below into this cell or subcells of this cell
        @param source_cell The starting cell from where to copy shapes
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will copy the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        Layers are identified between the layouts by the layer/datatype number of name if no layer/datatype number is present.

        The shapes copied will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    @overload
    def copy_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Copies the shapes from the given cell and the cell tree below into this cell or subcells of this cell with layer mapping
        @param source_cell The cell from where to copy shapes and instances
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will copy the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        In addition, the layer mapping object can be specified which maps source to target layers. This feature can be used to restrict the copy operation to a subset of layers or to convert shapes to different layers in that step.

        The shapes copied will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    @overload
    def dbbox(self) -> DBox:
        r"""
        @brief Gets the bounding box of the cell in micrometer units

        @return The bounding box of the cell

        The bounding box is computed over all layers. To compute the bounding box over single layers, use \dbbox with a layer index argument.

        This method has been introduced in version 0.25.
        """
    @overload
    def dbbox(self, layer_index: int) -> DBox:
        r"""
        @brief Gets the per-layer bounding box of the cell in micrometer units

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        This method has been introduced in version 0.25. 'dbbox' is the preferred synonym since version 0.28.
        """
    def dbbox_per_layer(self, layer_index: int) -> DBox:
        r"""
        @brief Gets the per-layer bounding box of the cell in micrometer units

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        This method has been introduced in version 0.25. 'dbbox' is the preferred synonym since version 0.28.
        """
    def delete(self) -> None:
        r"""
        @brief Deletes this cell 

        This deletes the cell but not the sub cells of the cell.
        These subcells will likely become new top cells unless they are used
        otherwise.
        All instances of this cell are deleted as well.
        Hint: to delete multiple cells, use "delete_cells" which is 
        far more efficient in this case.

        After the cell has been deleted, the Cell object becomes invalid. Do not access methods or attributes of this object after deleting the cell.

        This method has been introduced in version 0.23.
        """
    def delete_property(self, key: Any) -> None:
        r"""
        @brief Deletes the user property with the given key
        This method is a convenience method that deletes the property with the given key. It does nothing if no property with that key exists. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID.

        This method has been introduced in version 0.23.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def display_title(self) -> str:
        r"""
        @brief Returns a nice looking name for display purposes

        For example, this name include PCell parameters for PCell proxy cells.

        This method has been introduced in version 0.22.
        """
    def dump_mem_statistics(self, detailed: Optional[bool] = ...) -> None:
        r"""
        @hide
        """
    def dup(self) -> Cell:
        r"""
        @brief Creates a copy of the cell

        This method will create a copy of the cell. The new cell will be member of the same layout the original cell was member of. The copy will inherit all shapes and instances, but get a different cell_index and a modified name as duplicate cell names are not allowed in the same layout.

        This method has been introduced in version 0.27.
        """
    def each_child_cell(self) -> Iterator[int]:
        r"""
        @brief Iterates over all child cells

        This iterator will report the child cell indices, not every instance.
        """
    def each_inst(self) -> Iterator[Instance]:
        r"""
        @brief Iterates over all child instances (which may actually be instance arrays)

        Starting with version 0.15, this iterator delivers \Instance objects rather than \CellInstArray objects.
        """
    def each_meta_info(self) -> Iterator[LayoutMetaInfo]:
        r"""
        @brief Iterates over the meta information of the cell
        See \LayoutMetaInfo for details about cells and meta information.

        This method has been introduced in version 0.28.8.
        """
    @overload
    def each_overlapping_inst(self, b: Box) -> Iterator[Instance]:
        r"""
        @brief Gets the instances overlapping the given rectangle

        This will iterate over all child cell
        instances overlapping with the given rectangle b. 

        @param b The region to iterate over

        Starting with version 0.15, this iterator delivers \Instance objects rather than \CellInstArray objects.
        """
    @overload
    def each_overlapping_inst(self, b: DBox) -> Iterator[Instance]:
        r"""
        @brief Gets the instances overlapping the given rectangle, with the rectangle in micrometer units

        This will iterate over all child cell
        instances overlapping with the given rectangle b. This method is identical to the \each_overlapping_inst version that takes a \Box object, but instead of taking database unit coordinates in will take a micrometer unit \DBox object.

        @param b The region to iterate over

        This variant has been introduced in version 0.25.
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: Box) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box

        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query

        This call is equivalent to each_overlapping_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: Box, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: DBox) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box, with the box given in micrometer units

        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query

        This call is equivalent to each_overlapping_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: DBox, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box, with the box given in micrometer units

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query
        """
    def each_parent_cell(self) -> Iterator[int]:
        r"""
        @brief Iterates over all parent cells

        This iterator will iterate over the parent cells, just returning their
        cell index.
        """
    def each_parent_inst(self) -> Iterator[ParentInstArray]:
        r"""
        @brief Iterates over the parent instance list (which may actually be instance arrays)

        The parent instances are basically inversions of the instances. Using parent instances it is possible to determine how a specific cell is called from where.
        """
    @overload
    def each_shape(self, layer_index: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer

        @param layer_index The layer on which to run the query

        This call is equivalent to each_shape(layer_index,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_shape(self, layer_index: int, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param layer_index The layer on which to run the query

        This iterator is equivalent to 'shapes(layer).each'.
        """
    @overload
    def each_touching_inst(self, b: Box) -> Iterator[Instance]:
        r"""
        @brief Gets the instances touching the given rectangle

        This will iterate over all child cell
        instances overlapping with the given rectangle b. 

        @param b The region to iterate over

        Starting with version 0.15, this iterator delivers \Instance objects rather than \CellInstArray objects.
        """
    @overload
    def each_touching_inst(self, b: DBox) -> Iterator[Instance]:
        r"""
        @brief Gets the instances touching the given rectangle, with the rectangle in micrometer units

        This will iterate over all child cell
        instances touching the given rectangle b. This method is identical to the \each_touching_inst version that takes a \Box object, but instead of taking database unit coordinates in will take a micrometer unit \DBox object.

        @param b The region to iterate over

        This variant has been introduced in version 0.25.
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: Box) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box

        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query

        This call is equivalent to each_touching_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: Box, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: DBox) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box, with the box given in micrometer units

        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query

        This call is equivalent to each_touching_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: DBox, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box, with the box given in micrometer units

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query
        """
    def erase(self, inst: Instance) -> None:
        r"""
        @brief Erases the instance given by the Instance object

        This method has been introduced in version 0.16. It can only be used in editable mode.
        """
    @overload
    def fill_region(self, region: Region, fill_cell_index: int, fc_bbox: Box, row_step: Vector, column_step: Vector, origin: Optional[Point] = ..., remaining_parts: Optional[Region] = ..., fill_margin: Optional[Vector] = ..., remaining_polygons: Optional[Region] = ..., glue_box: Optional[Box] = ...) -> None:
        r"""
        @brief Fills the given region with cells of the given type (skew step version)
        @param region The region to fill
        @param fill_cell_index The fill cell to place
        @param fc_bbox The fill cell's box to place
        @param row_step The 'rows' step vector
        @param column_step The 'columns' step vector
        @param origin The global origin of the fill pattern or nil to allow local (per-polygon) optimization
        @param remaining_parts See explanation in other version
        @param fill_margin See explanation in other version
        @param remaining_polygons See explanation in other version

        This version is similar to the version providing an orthogonal fill, but it offers more generic stepping of the fill cell.
        The step pattern is defined by an origin and two vectors (row_step and column_step) which span the axes of the fill cell pattern.

        The fill box and the step vectors are decoupled which means the fill box can be larger or smaller than the step pitch - it can be overlapping and there can be space between the fill box instances. Fill boxes are placed where they fit entirely into a polygon of the region. The fill boxes lower left corner is the reference for the fill pattern and aligns with the origin if given.

        This variant has been introduced in version 0.27.
        """
    @overload
    def fill_region(self, region: Region, fill_cell_index: int, fc_box: Box, origin: Optional[Point] = ..., remaining_parts: Optional[Region] = ..., fill_margin: Optional[Vector] = ..., remaining_polygons: Optional[Region] = ..., glue_box: Optional[Box] = ...) -> None:
        r"""
        @brief Fills the given region with cells of the given type (extended version)
        @param region The region to fill
        @param fill_cell_index The fill cell to place
        @param fc_box The fill cell's footprint
        @param origin The global origin of the fill pattern or nil to allow local (per-polygon) optimization
        @param remaining_parts See explanation below
        @param fill_margin See explanation below
        @param remaining_polygons See explanation below
        @param glue_box Guarantees fill cell compatibility to neighbor regions in enhanced mode

        This method creates a regular pattern of fill cells to cover the interior of the given region as far as possible. This process is also known as tiling. This implementation supports rectangular (not necessarily square) tile cells. The tile cell's footprint is given by the fc_box parameter and the cells will be arranged with their footprints forming a seamless array.

        The algorithm supports a global fill raster as well as local (per-polygon) origin optimization. In the latter case the origin of the regular raster is optimized per individual polygon of the fill region. To enable optimization, pass 'nil' to the 'origin' argument.

        The implementation will basically try to find a repetition pattern of the tile cell's footprint and produce instances which fit entirely into the fill region.

        There is also a version available which offers skew step vectors as a generalization of the orthogonal ones.

        If the 'remaining_parts' argument is non-nil, the corresponding region will receive the parts of the polygons which are not covered by tiles. Basically the tiles are subtracted from the original polygons. A margin can be specified which is applied separately in x and y direction before the subtraction is done ('fill_margin' parameter).

        If the 'remaining_polygons' argument is non-nil, the corresponding region will receive all polygons from the input region which could not be filled and where there is no chance of filling because not a single tile will fit into them.

        'remaining_parts' and 'remaining_polygons' can be identical with the input. In that case the input will be overwritten with the respective output. Otherwise, the respective polygons are added to these regions.

        This allows setting up a more elaborate fill scheme using multiple iterations and local origin-optimization ('origin' is nil):

        @code
        r = ...        # region to fill
        c = ...        # cell in which to produce the fill cells
        fc_index = ... # fill cell index
        fc_box = ...   # fill cell footprint

        fill_margin = RBA::Point::new(0, 0)   # x/y distance between tile cells with different origin

        # Iteration: fill a region and fill the remaining parts as long as there is anything left.
        # Polygons not worth being considered further are dropped (last argument is nil).
        while !r.is_empty?
          c.fill_region(r, fc_index, fc_box, nil, r, fill_margin, nil)
        end
        @/code

        The glue box parameter supports fill cell array compatibility with neighboring regions. This is specifically useful when putting the fill_cell method into a tiling processor. Fill cell array compatibility means that the fill cell array continues over tile boundaries. This is easy with an origin: you can chose the origin identically over all tiles which is sufficient to guarantee fill cell array compatibility across the tiles. However there is no freedom of choice of the origin then and fill cell placement may not be optimal. To enable the origin for the tile boundary only, a glue box can given. The origin will then be used only when the polygons to fill not entirely inside and not at the border of the glue box. Hence, while a certain degree of freedom is present for the placement of fill cells inside the glue box, the fill cells are guaranteed to be placed at the raster implied by origin at the glue box border and beyond. To ensure fill cell compatibility inside the tiling processor, it is sufficient to use the tile box as the glue box.

        This method has been introduced in version 0.23 and enhanced in version 0.27.
        """
    def fill_region_multi(self, region: Region, fill_cell_index: int, fc_bbox: Box, row_step: Vector, column_step: Vector, fill_margin: Optional[Vector] = ..., remaining_polygons: Optional[Region] = ..., glue_box: Optional[Box] = ...) -> None:
        r"""
        @brief Fills the given region with cells of the given type in enhanced mode with iterations
        This version operates like \fill_region, but repeats the fill generation until no further fill cells can be placed. As the fill pattern origin changes between the iterations, narrow regions can be filled which cannot with a fixed fill pattern origin. The \fill_margin parameter is important as it controls the distance between fill cells with a different origin and therefore introduces a safety distance between pitch-incompatible arrays.

        The origin is ignored unless a glue box is given. See \fill_region for a description of this concept.

        This method has been introduced in version 0.27.
        """
    @overload
    def flatten(self, levels: int, prune: bool) -> None:
        r"""
        @brief Flattens the given cell

        This method propagates all shapes from the specified number of hierarchy levels below into the given cell.
        It also removes the instances of the cells from which the shapes came from, but does not remove the cells themselves if prune is set to false.
        If prune is set to true, these cells are removed if not used otherwise.

        @param levels The number of hierarchy levels to flatten (-1: all, 0: none, 1: one level etc.)
        @param prune Set to true to remove orphan cells.

        This method has been introduced in version 0.23.
        """
    @overload
    def flatten(self, prune: bool) -> None:
        r"""
        @brief Flattens the given cell

        This method propagates all shapes from the hierarchy below into the given cell.
        It also removes the instances of the cells from which the shapes came from, but does not remove the cells themselves if prune is set to false.
        If prune is set to true, these cells are removed if not used otherwise.

        A version of this method exists which allows one to specify the number of hierarchy levels to which subcells are considered.

        @param prune Set to true to remove orphan cells.

        This method has been introduced in version 0.23.
        """
    def has_prop_id(self) -> bool:
        r"""
        @brief Returns true, if the cell has user properties

        This method has been introduced in version 0.23.
        """
    def hierarchy_levels(self) -> int:
        r"""
        @brief Returns the number of hierarchy levels below

        This method returns the number of call levels below the current cell. If there are no child cells, this method will return 0, if there are only direct children, it will return 1.

        CAUTION: this method may be expensive!
        """
    @overload
    def insert(self, cell_inst_array: CellInstArray) -> Instance:
        r"""
        @brief Inserts a cell instance (array)
        @return An Instance object representing the new instance
        With version 0.16, this method returns an Instance object that represents the new instance.
        It's use is discouraged in readonly mode, since it invalidates other Instance references.
        """
    @overload
    def insert(self, cell_inst_array: CellInstArray, property_id: int) -> Instance:
        r"""
        @brief Inserts a cell instance (array) with properties
        @return An \Instance object representing the new instance
        The property Id must be obtained from the \Layout object's property_id method which associates a property set with a property Id.
        With version 0.16, this method returns an Instance object that represents the new instance.
        It's use is discouraged in readonly mode, since it invalidates other Instance references.
        """
    @overload
    def insert(self, cell_inst_array: DCellInstArray) -> Instance:
        r"""
        @brief Inserts a cell instance (array) given in micron units
        @return An Instance object representing the new instance
        This method inserts an instance array, similar to \insert with a \CellInstArray parameter. But in this version, the argument is a cell instance array given in micrometer units. It is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def insert(self, cell_inst_array: DCellInstArray, property_id: int) -> Instance:
        r"""
        @brief Inserts a cell instance (array) given in micron units with properties
        @return An Instance object representing the new instance
        This method inserts an instance array, similar to \insert with a \CellInstArray parameter and a property set ID. But in this version, the argument is a cell instance array given in micrometer units. It is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def insert(self, inst: Instance) -> Instance:
        r"""
        @brief Inserts a cell instance given by another reference
        @return An Instance object representing the new instance
        This method allows one to copy instances taken from a reference (an \Instance object).
        This method is not suited to inserting instances from other Layouts into this cell. For this purpose, the hierarchical copy methods of \Layout have to be used.

        It has been added in version 0.16.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the cell is empty

        An empty cell is a cell not containing instances nor any shapes.

        This method has been introduced in version 0.20.
        """
    def is_ghost_cell(self) -> bool:
        r"""
        @brief Returns a value indicating whether the cell is a "ghost cell"

        The ghost cell flag is used by the GDS reader for example to indicate that
        the cell is not located inside the file. Upon writing the reader can determine
        whether to write the cell or not.
        To satisfy the references inside the layout, a dummy cell is created in this case
        which has the "ghost cell" flag set to true.

        This method has been introduced in version 0.20.
        """
    def is_leaf(self) -> bool:
        r"""
        @brief Gets a value indicating whether the cell is a leaf cell

        A cell is a leaf cell if there are no child instantiations.
        """
    def is_library_cell(self) -> bool:
        r"""
        @brief Returns true, if the cell is a proxy cell pointing to a library cell
        If the cell is imported from some library, this attribute returns true.
        Please note, that this attribute can combine with \is_pcell? for PCells imported from
        a library.

        This method has been introduced in version 0.22.
        """
    @overload
    def is_pcell_variant(self) -> bool:
        r"""
        @brief Returns true, if this cell is a pcell variant
        this method returns true, if this cell represents a pcell with a distinct
        set of parameters (a PCell proxy). This also is true, if the PCell is imported from a library.

        Technically, PCells imported from a library are library proxies which are 
        pointing to PCell variant proxies. This scheme can even proceed over multiple
        indirections, i.e. a library using PCells from another library.

        This method has been introduced in version 0.22.
        """
    @overload
    def is_pcell_variant(self, instance: Instance) -> bool:
        r"""
        @brief Returns true, if this instance is a PCell variant
        This method returns true, if this instance represents a PCell with a distinct
        set of parameters. This method also returns true, if it is a PCell imported from a library.

        This method has been introduced in version 0.22.
        """
    def is_proxy(self) -> bool:
        r"""
        @brief Returns true, if the cell presents some external entity   
        A cell may represent some data which is imported from some other source, i.e.
        a library. Such cells are called "proxy cells". For a library reference, the
        proxy cell is some kind of pointer to the library and the cell within the library.

        For PCells, this data can even be computed through some script.
        A PCell proxy represents all instances with a given set of parameters.

        Proxy cells cannot be modified, except that pcell parameters can be modified
        and PCell instances can be recomputed.

        This method has been introduced in version 0.22.
        """
    def is_top(self) -> bool:
        r"""
        @brief Gets a value indicating whether the cell is a top-level cell

        A cell is a top-level cell if there are no parent instantiations.
        """
    def is_valid(self, instance: Instance) -> bool:
        r"""
        @brief Tests if the given \Instance object is still pointing to a valid object
        This method has been introduced in version 0.16.
        If the instance represented by the given reference has been deleted, this method returns false. If however, another instance has been inserted already that occupies the original instances position, this method will return true again.
        """
    @overload
    def layout(self) -> Layout:
        r"""
        @brief Returns a reference to the layout where the cell resides

        this method has been introduced in version 0.22.
        """
    @overload
    def layout(self) -> Layout:
        r"""
        @brief Returns a reference to the layout where the cell resides (const references)

        this method has been introduced in version 0.22.
        """
    def library(self) -> Library:
        r"""
        @brief Returns a reference to the library from which the cell is imported
        if the cell is not imported from a library, this reference is nil.

        this method has been introduced in version 0.22.
        """
    def library_cell_index(self) -> int:
        r"""
        @brief Returns the index of the cell in the layout of the library (if it's a library proxy)
        Together with the \library method, it is possible to locate the source cell of
        a library proxy. The source cell can be retrieved from a cell "c" with 

        @code
        c.library.layout.cell(c.library_cell_index)
        @/code

        This cell may be itself a proxy,
        i.e. for pcell libraries, where the library cells are pcell variants which itself
        are proxies to a pcell.

        This method has been introduced in version 0.22.
        """
    def merge_meta_info(self, other: Cell) -> None:
        r"""
        @brief Merges the meta information from the other cell into this cell
        See \LayoutMetaInfo for details about cells and meta information.
        Existing keys in this cell will be overwritten by the respective values from the other cell.
        New keys will be added.

        This method has been introduced in version 0.28.16.
        """
    def meta_info(self, name: str) -> LayoutMetaInfo:
        r"""
        @brief Gets the meta information for a given name
        See \LayoutMetaInfo for details about cells and meta information.

        If no meta information with the given name exists, a default object with empty fields will be returned.

        This method has been introduced in version 0.28.8.
        """
    def meta_info_value(self, name: str) -> Any:
        r"""
        @brief Gets the meta information value for a given name
        See \LayoutMetaInfo for details about cells and meta information.

        If no meta information with the given name exists, a nil value will be returned.
        A more generic version that delivers all fields of the meta information is \meta_info.

        This method has been introduced in version 0.28.8.
        """
    @overload
    def move(self, src: int, dest: int) -> None:
        r"""
        @brief Moves the shapes from the source to the target layer

        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        This method will move shapes within the cell. To move shapes from another cell to this cell, use the copy method with the cell parameter.

        This method has been introduced in version 0.19.

        @param src The layer index of the source layer
        @param dest The layer index of the destination layer
        """
    @overload
    def move(self, src_cell: Cell, src_layer: int, dest: int) -> None:
        r"""
        @brief Moves shapes from another cell to the target layer in this cell

        This method will move all shapes on layer 'src_layer' of cell 'src_cell' to the layer 'dest' of this cell.
        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        If the source cell lives in a layout with a different database unit than that current cell is in, the shapes will be transformed accordingly. The same way, shape properties are transformed as well. Note that the shape transformation may require rounding to smaller coordinates. This may result in a slight distortion of the original shapes, in particular when transforming into a layout with a bigger database unit.
        @param src_cell The cell where to take the shapes from
        @param src_layer The layer index of the layer from which to take the shapes
        @param dest The layer index of the destination layer
        """
    def move_instances(self, source_cell: Cell) -> None:
        r"""
        @brief Moves the instances of child cells in the source cell to this cell
        @param source_cell The cell where the instances are moved from
        The source cell must reside in the same layout than this cell. The instances of child cells inside the source cell are moved to this cell. No new cells are created, just new instances are created to already existing cells in the target cell.

        The instances will be added to any existing instances in the cell.

        More elaborate methods of moving hierarchy trees between layouts are provided through the \move_tree_shapes (in cooperation with the \CellMapping class) or \move_tree methods.

        This method has been added in version 0.23.
        """
    @overload
    def move_shapes(self, source_cell: Cell) -> None:
        r"""
        @brief Moves the shapes from the given cell into this cell
        @param source_cell The cell from where to move shapes
        All shapes are moved from the source cell to this cell. Instances are not moved.

        The source cell can reside in a different layout. In this case, the shapes are moved over from the other layout into this layout. Database unit conversion is done automatically if the database units differ between the layouts. Note that this may lead to grid snapping effects if the database unit of the target layout is not an integer fraction of the source layout.

        If source and target layout are different, the layers of the source and target layout are identified by their layer/datatype number or name (if no layer/datatype is present).
        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def move_shapes(self, source_cell: Cell, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Moves the shapes from the given cell into this cell
        @param source_cell The cell from where to move shapes
        @param layer_mapping A \LayerMapping object that specifies which layers are moved and where
        All shapes on layers specified in the layer mapping object are moved from the source cell to this cell. Instances are not moved.
        The target layer is taken from the mapping table.

        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    def move_tree(self, source_cell: Cell) -> List[int]:
        r"""
        @brief Moves the cell tree of the given cell into this cell
        @param source_cell The cell from where to move the cell tree
        @return A list of indexes of newly created cells
        The complete cell tree of the source cell is moved to the target cell plus all shapes in that tree are moved as well. This method will basically rebuild the cell tree of the source cell and empty the source cell.

        The source cell may reside in a separate layout. This method therefore provides a way to move over complete cell trees from one layout to another.

        The shapes and instances will be added to any shapes or instances already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def move_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping) -> None:
        r"""
        @brief Moves the shapes from the given cell and the cell tree below into this cell or subcells of this cell
        @param source_cell The starting cell from where to move shapes
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will move the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        Layers are identified between the layouts by the layer/datatype number of name if no layer/datatype number is present.

        The shapes moved will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    @overload
    def move_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Moves the shapes from the given cell and the cell tree below into this cell or subcells of this cell with layer mapping
        @param source_cell The cell from where to move shapes and instances
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will move the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        In addition, the layer mapping object can be specified which maps source to target layers. This feature can be used to restrict the move operation to a subset of layers or to convert shapes to different layers in that step.

        The shapes moved will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    def parent_cells(self) -> int:
        r"""
        @brief Gets the number of parent cells 

        The number of parent cells (cells which reference our cell) is reported.
        """
    @overload
    def pcell_declaration(self) -> PCellDeclaration_Native:
        r"""
        @brief Returns a reference to the PCell declaration
        If this cell is not a PCell variant, this method returns nil.
        PCell variants are proxy cells which are PCell incarnations for a specific parameter set.
        The \PCellDeclaration object allows one to retrieve PCell parameter definitions for example.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_declaration(self, instance: Instance) -> PCellDeclaration_Native:
        r"""
        @brief Returns the PCell declaration of a pcell instance
        If the instance is not a PCell instance, this method returns nil.
        The \PCellDeclaration object allows one to retrieve PCell parameter definitions for example.

        This method has been introduced in version 0.22.
        """
    def pcell_id(self) -> int:
        r"""
        @brief Returns the PCell ID if the cell is a pcell variant
        This method returns the ID which uniquely identifies the PCell within the 
        layout where it's declared. It can be used to retrieve the PCell declaration 
        or to create new PCell variants.

        The method will be rarely used. It's more convenient to use \pcell_declaration to directly retrieve the PCellDeclaration object for example.

        This method has been introduced in version 0.22.
        """
    def pcell_library(self) -> Library:
        r"""
        @brief Returns the library where the PCell is declared if this cell is a PCell and it is not defined locally.
        A PCell often is not declared within the current layout but in some library. 
        This method returns a reference to that library, which technically is the last of the 
        chained library proxies. If this cell is not a PCell or it is not located in a 
        library, this method returns nil.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_parameter(self, instance: Instance, name: str) -> Any:
        r"""
        @brief Returns a PCell parameter by name for a pcell instance

        If the given instance is a PCell instance, this method returns the value of the PCell parameter with the given name.
        If the instance is not a PCell instance or the name is not a valid PCell parameter name, this
        method returns nil.

        This method has been introduced in version 0.25.
        """
    @overload
    def pcell_parameter(self, name: str) -> Any:
        r"""
        @brief Gets a PCell parameter by name if the cell is a PCell variant
        If the cell is a PCell variant, this method returns the parameter with the given name.
        If the cell is not a PCell variant or the name is not a valid PCell parameter name, the return value is nil.

        This method has been introduced in version 0.25.
        """
    @overload
    def pcell_parameters(self) -> List[Any]:
        r"""
        @brief Returns the PCell parameters for a pcell variant
        If the cell is a PCell variant, this method returns a list of
        values for the PCell parameters. If the cell is not a PCell variant, this
        method returns an empty list. This method also returns the PCell parameters if
        the cell is a PCell imported from a library.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_parameters(self, instance: Instance) -> List[Any]:
        r"""
        @brief Returns the PCell parameters for a pcell instance
        If the given instance is a PCell instance, this method returns a list of
        values for the PCell parameters. If the instance is not a PCell instance, this
        method returns an empty list.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_parameters_by_name(self) -> Dict[str, Any]:
        r"""
        @brief Returns the PCell parameters for a pcell variant as a name to value dictionary
        If the cell is a PCell variant, this method returns a dictionary of
        values for the PCell parameters with the parameter names as the keys. If the cell is not a PCell variant, this
        method returns an empty dictionary. This method also returns the PCell parameters if
        the cell is a PCell imported from a library.

        This method has been introduced in version 0.24.
        """
    @overload
    def pcell_parameters_by_name(self, instance: Instance) -> Dict[str, Any]:
        r"""
        @brief Returns the PCell parameters for a pcell instance as a name to value dictionary
        If the given instance is a PCell instance, this method returns a dictionary of
        values for the PCell parameters with the parameter names as the keys. If the instance is not a PCell instance, this
        method returns an empty dictionary.

        This method has been introduced in version 0.24.
        """
    def property(self, key: Any) -> Any:
        r"""
        @brief Gets the user property with the given key
        This method is a convenience method that gets the property with the given key. If no property with that key exists, it will return nil. Using that method is more convenient than using the layout object and the properties ID to retrieve the property value. 
        This method has been introduced in version 0.23.
        """
    @overload
    def prune_cell(self) -> None:
        r"""
        @brief Deletes the cell plus subcells not used otherwise

        This deletes the cell and also all sub cells of the cell which are not used otherwise.
        All instances of this cell are deleted as well.
        A version of this method exists which allows one to specify the number of hierarchy levels to which subcells are considered.

        After the cell has been deleted, the Cell object becomes invalid. Do not access methods or attributes of this object after deleting the cell.

        This method has been introduced in version 0.23.
        """
    @overload
    def prune_cell(self, levels: int) -> None:
        r"""
        @brief Deletes the cell plus subcells not used otherwise

        This deletes the cell and also all sub cells of the cell which are not used otherwise.
        The number of hierarchy levels to consider can be specified as well. One level of hierarchy means that only the direct children of the cell are deleted with the cell itself.
        All instances of this cell are deleted as well.

        After the cell has been deleted, the Cell object becomes invalid. Do not access methods or attributes of this object after deleting the cell.

        @param levels The number of hierarchy levels to consider (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.23.
        """
    @overload
    def prune_subcells(self) -> None:
        r"""
        @brief Deletes all sub cells of the cell which are not used otherwise

        This deletes all sub cells of the cell which are not used otherwise.
        All instances of the deleted cells are deleted as well.
        A version of this method exists which allows one to specify the number of hierarchy levels to which subcells are considered.

        This method has been introduced in version 0.23.
        """
    @overload
    def prune_subcells(self, levels: int) -> None:
        r"""
        @brief Deletes all sub cells of the cell which are not used otherwise down to the specified level of hierarchy

        This deletes all sub cells of the cell which are not used otherwise.
        All instances of the deleted cells are deleted as well.
        It is possible to specify how many levels of hierarchy below the given root cell are considered.

        @param levels The number of hierarchy levels to consider (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.23.
        """
    def qname(self) -> str:
        r"""
        @brief Returns the library-qualified name

        Library cells will be indicated by returning a qualified name composed of the library name, a dot and the basic cell name. For example: "Basic.TEXT" will be the qname of the TEXT cell of the Basic library. For non-library cells, the qname is identical to the basic name (see \name).

        This method has been introduced in version 0.25.
        """
    @overload
    def read(self, file_name: str) -> List[int]:
        r"""
        @brief Reads a layout file into this cell
        This version uses the default options for reading the file.

        This method has been introduced in version 0.28.
        """
    @overload
    def read(self, file_name: str, options: LoadLayoutOptions) -> List[int]:
        r"""
        @brief Reads a layout file into this cell

        @param file_name The path of the file to read
        @param options The reader options to use
        @return The indexes of the cells created during the reading (new child cells)

        The format of the file will be determined from the file name. The layout will be read into the cell, potentially creating new layers and a subhierarchy of cells below this cell.

        This feature is equivalent to the following code:

        @code
        def Cell.read(file_name, options)
          layout = RBA::Layout::new
          layout.read(file_name, options)
          cm = RBA::CellMapping::new
          cm.for_single_cell_full(self, layout.top_cell)
          self.move_tree_shapes(layout.top_cell)
        end
        @/code

        See \move_tree_shapes and \CellMapping for more details and how to implement more elaborate schemes.

        This method has been introduced in version 0.28.
        """
    def refresh(self) -> None:
        r"""
        @brief Refreshes a proxy cell

        If the cell is a PCell variant, this method recomputes the PCell.
        If the cell is a library proxy, this method reloads the information from the library, but not the library itself.
        Note that if the cell is an PCell variant for a PCell coming from a library, this method will not recompute the PCell. Instead, you can use \Library#refresh to recompute all PCells from that library.

        You can use \Layout#refresh to refresh all cells from a layout.

        This method has been introduced in version 0.22.
        """
    def remove_meta_info(self, name: str) -> None:
        r"""
        @brief Removes meta information from the cell
        See \LayoutMetaInfo for details about cells and meta information.

        This method has been introduced in version 0.28.8.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: CellInstArray) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one
        @return An \Instance object representing the new instance
        This method has been introduced in version 0.16. It can only be used in editable mode.
        The instance given by the instance object (first argument) is replaced by the given instance (second argument). The new object will not have any properties.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: CellInstArray, property_id: int) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one with properties
        @return An \Instance object representing the new instance
        This method has been introduced in version 0.16. It can only be used in editable mode.
        The instance given by the instance object (first argument) is replaced by the given instance (second argument) with the given properties Id.
        The property Id must be obtained from the \Layout object's property_id method which associates a property set with a property Id.
        The new object will not have any properties.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: DCellInstArray) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one, given in micrometer units
        @return An \Instance object representing the new instance
        This method is identical to the corresponding \replace variant with a \CellInstArray argument. It however accepts a micrometer-unit \DCellInstArray object which is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: DCellInstArray, property_id: int) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one and new properties, where the cell instance is given in micrometer units
        @return An \Instance object representing the new instance
        This method is identical to the corresponding \replace variant with a \CellInstArray argument and a property ID. It however accepts a micrometer-unit \DCellInstArray object which is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    def replace_prop_id(self, instance: Instance, property_id: int) -> Instance:
        r"""
        @brief Replaces (or install) the properties of a cell
        @return An Instance object representing the new instance
        This method has been introduced in version 0.16. It can only be used in editable mode.
        Changes the properties Id of the given instance or install a properties Id on that instance if it does not have one yet.
        The property Id must be obtained from the \Layout object's property_id method which associates a property set with a property Id.
        """
    def set_property(self, key: Any, value: Any) -> None:
        r"""
        @brief Sets the user property with the given key to the given value
        This method is a convenience method that sets the property with the given key to the given value. If no property with that key exists, it will create one. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID. Note: GDS only supports integer keys. OASIS supports numeric and string keys. 
        This method has been introduced in version 0.23.
        """
    @overload
    def shapes(self, layer_index: int) -> Shapes:
        r"""
        @brief Returns the shapes list of the given layer

        This method gives access to the shapes list on a certain layer.
        If the layer does not exist yet, it is created.

        @param index The layer index of the shapes list to retrieve

        @return A reference to the shapes list
        """
    @overload
    def shapes(self, layer_index: int) -> Shapes:
        r"""
        @brief Returns the shapes list of the given layer (const version)

        This method gives access to the shapes list on a certain layer. This is the const version - only const (reading) methods can be called on the returned object.

        @param index The layer index of the shapes list to retrieve

        @return A reference to the shapes list

        This variant has been introduced in version 0.26.4.
        """
    def swap(self, layer_index1: int, layer_index2: int) -> None:
        r"""
        @brief Swaps the layers given

        This method swaps two layers inside this cell.
        """
    @overload
    def transform(self, instance: Instance, trans: DCplxTrans) -> Instance:
        r"""
        @brief Transforms the instance with the given complex floating-point transformation given in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform method with a \ICplxTrans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform(self, instance: Instance, trans: DTrans) -> Instance:
        r"""
        @brief Transforms the instance with the transformation given in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform method with a \Trans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform(self, instance: Instance, trans: ICplxTrans) -> Instance:
        r"""
        @brief Transforms the instance with the given complex integer transformation
        @return A reference (an \Instance object) to the new instance
        This method has been introduced in version 0.23.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform(self, instance: Instance, trans: Trans) -> Instance:
        r"""
        @brief Transforms the instance with the given transformation
        @return A reference (an \Instance object) to the new instance
        This method has been introduced in version 0.16.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the cell by the given, micrometer-unit transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells. The difference is important in the presence of magnifications: "transform" will leave magnified instances while "transform_into" will not do so but expect the magnification to be applied inside the called cells too.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the cell by the given, micrometer-unit transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the cell by the given complex integer transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells. The difference is important in the presence of magnifications: "transform" will leave magnified instances while "transform_into" will not do so but expect the magnification to be applied inside the called cells too.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, trans: Trans) -> None:
        r"""
        @brief Transforms the cell by the given integer transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform_into(self, instance: Instance, trans: DCplxTrans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given complex transformation where the transformation is in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform_into method with a \ICplxTrans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, instance: Instance, trans: DTrans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given transformation where the transformation is in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform_into method with a \Trans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, instance: Instance, trans: ICplxTrans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given complex integer transformation
        @return A reference (an \Instance object) to the new instance

        See the comments for the simple-transformation version for a description of this method.
        This method has been introduced in version 0.23.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform_into(self, instance: Instance, trans: Trans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given transformation
        @return A reference (an \Instance object) to the new instance

        In contrast to the \transform method, this method allows propagation of the transformation into child cells. More precisely: it applies just a part of the given transformation to the instance, such that when transforming the cell instantiated and its shapes with the same transformation, the result will reflect the desired transformation. Mathematically spoken, the transformation of the instance (A) is transformed with the given transformation T using "A' = T * A * Tinv" where Tinv is the inverse of T. In effect, the transformation T commutes with the new instance transformation A' and can be applied to child cells as well. This method is therefore useful to transform a hierarchy of cells.

        This method has been introduced in version 0.23.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform_into(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given complex integer transformation where the transformation is in micrometer units
        This method is identical to the corresponding \transform_into method with a \ICplxTrans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given transformation where the transformation is in micrometer units
        This method is identical to the corresponding \transform_into method with a \Trans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given complex integer transformation

        See the comments for the simple-transformation version for a description of this method.
        This method has been introduced in version 0.23.
        """
    @overload
    def transform_into(self, trans: Trans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given transformation

        This method transforms all instances and all shapes. The instances are transformed in a way that allows propagation of the transformation into child cells. For this, it applies just a part of the given transformation to the instance such that when transforming the shapes of the cell instantiated, the result will reflect the desired transformation. Mathematically spoken, the transformation of the instance (A) is transformed with the given transformation T using "A' = T * A * Tinv" where Tinv is the inverse of T. In effect, the transformation T commutes with the new instance transformation A' and can be applied to child cells as well. This method is therefore useful to transform a hierarchy of cells.

        It has been introduced in version 0.23.
        """
    @overload
    def write(self, file_name: str) -> None:
        r"""
        @brief Writes the cell to a layout file
        The format of the file will be determined from the file name. Only the cell and its subtree below will be saved.

        This method has been introduced in version 0.23.
        """
    @overload
    def write(self, file_name: str, options: SaveLayoutOptions) -> None:
        r"""
        @brief Writes the cell to a layout file
        The format of the file will be determined from the file name. Only the cell and its subtree below will be saved.
        In contrast to the other 'write' method, this version allows one to specify save options, i.e. scaling etc.

        This method has been introduced in version 0.23.
        """

class CellInstArray:
    r"""
    @brief A single or array cell instance
    This object represents either single or array cell instances. A cell instance array is a regular array, described by two displacement vectors (a, b) and the instance count along that axes (na, nb). 

    In addition, this object represents either instances with simple transformations or instances with complex transformations. The latter includes magnified instances and instances rotated by an arbitrary angle.

    The cell which is instantiated is given by a cell index. The cell index can be converted to a cell pointer by using \Layout#cell. The cell index of a cell can be obtained using \Cell#cell_index.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    a: Vector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'a' axis

    Starting with version 0.25 the displacement is of vector type.

    Setter:
    @brief Sets the displacement vector for the 'a' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.

    This method was introduced in version 0.22. Starting with version 0.25 the displacement is of vector type.
    """
    b: Vector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'b' axis

    Starting with version 0.25 the displacement is of vector type.

    Setter:
    @brief Sets the displacement vector for the 'b' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.

    This method was introduced in version 0.22. Starting with version 0.25 the displacement is of vector type.
    """
    cell_index: int
    r"""
    Getter:
    @brief Gets the cell index of the cell instantiated 
    Use \Layout#cell to get the \Cell object from the cell index.
    Setter:
    @brief Sets the index of the cell this instance refers to
    """
    cplx_trans: ICplxTrans
    r"""
    Getter:
    @brief Gets the complex transformation of the first instance in the array
    This method is always applicable, compared to \trans, since simple transformations can be expressed as complex transformations as well.
    Setter:
    @brief Sets the complex transformation of the instance or the first instance in the array

    This method was introduced in version 0.22.
    """
    @property
    def cell(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the cell this instance refers to
        This is a convenience method and equivalent to 'cell_index = cell.cell_index()'. There is no getter for the cell pointer because the \CellInstArray object only knows about cell indexes.

        This convenience method has been introduced in version 0.28.
        """
    na: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'a' axis

    Setter:
    @brief Sets the number of instances in the 'a' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.

    This method was introduced in version 0.22.
    """
    nb: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'b' axis

    Setter:
    @brief Sets the number of instances in the 'b' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.

    This method was introduced in version 0.22.
    """
    trans: Trans
    r"""
    Getter:
    @brief Gets the transformation of the first instance in the array
    The transformation returned is only valid if the array does not represent a complex transformation array
    Setter:
    @brief Sets the transformation of the instance or the first instance in the array

    This method was introduced in version 0.22.
    """
    @overload
    @classmethod
    def new(cls) -> CellInstArray:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: Vector) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: ICplxTrans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: Trans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: Vector) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: ICplxTrans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: Trans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    def __copy__(self) -> CellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> CellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    def __init__(self, cell: Cell, disp: Vector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: ICplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: Trans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: Vector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, trans: ICplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell_index: int, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    @overload
    def __init__(self, cell_index: int, trans: Trans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell_index: int, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    def __len__(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def __lt__(self, other: CellInstArray) -> bool:
        r"""
        @brief Compares two arrays for 'less'
        The comparison provides an arbitrary sorting criterion and not specific sorting order. It is guaranteed that if an array a is less than b, b is not less than a. In addition, it a is not less than b and b is not less than a, then a is equal to b.
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for inequality
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts the array to a string

        This method was introduced in version 0.22.
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the array to a string

        This method was introduced in version 0.22.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: CellInstArray) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def bbox(self, layout: Layout) -> Box:
        r"""
        @brief Gets the bounding box of the array
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.
        """
    @overload
    def bbox(self, layout: Layout, layer_index: int) -> Box:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def bbox_per_layer(self, layout: Layout, layer_index: int) -> Box:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> CellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def each_cplx_trans(self) -> Iterator[ICplxTrans]:
        r"""
        @brief Gets the complex transformations represented by this instance
        For a single instance, this iterator will deliver the single, complex transformation. For array instances, the iterator will deliver each complex transformation of the expanded array.
        This iterator is a generalization of \each_trans for general complex transformations.

        This method has been introduced in version 0.25.
        """
    def each_trans(self) -> Iterator[Trans]:
        r"""
        @brief Gets the simple transformations represented by this instance
        For a single instance, this iterator will deliver the single, simple transformation. For array instances, the iterator will deliver each simple transformation of the expanded array.

        This iterator will only deliver valid transformations if the instance array is not of complex type (see \is_complex?). A more general iterator that delivers the complex transformations is \each_cplx_trans.

        This method has been introduced in version 0.25.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> None:
        r"""
        @brief Inverts the array reference

        The inverted array reference describes in which transformations the parent cell is
        seen from the current cell.
        """
    def is_complex(self) -> bool:
        r"""
        @brief Gets a value indicating whether the array is a complex array

        Returns true if the array represents complex instances (that is, with magnification and 
        arbitrary rotation angles).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_regular_array(self) -> bool:
        r"""
        @brief Gets a value indicating whether this instance is a regular array
        """
    def size(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def to_s(self) -> str:
        r"""
        @brief Converts the array to a string

        This method was introduced in version 0.22.
        """
    @overload
    def transform(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the cell instance with the given complex transformation

        This method has been introduced in version 0.20.
        """
    @overload
    def transform(self, trans: Trans) -> None:
        r"""
        @brief Transforms the cell instance with the given transformation

        This method has been introduced in version 0.20.
        """
    @overload
    def transformed(self, trans: ICplxTrans) -> CellInstArray:
        r"""
        @brief Gets the transformed cell instance (complex transformation)

        This method has been introduced in version 0.20.
        """
    @overload
    def transformed(self, trans: Trans) -> CellInstArray:
        r"""
        @brief Gets the transformed cell instance

        This method has been introduced in version 0.20.
        """

class CellMapping:
    r"""
    @brief A cell mapping (source to target layout)

    A cell mapping is an association of cells in two layouts forming pairs of cells, i.e. one cell corresponds to another cell in the other layout. The CellMapping object describes the mapping of cells of a source layout B to a target layout A. The cell mapping object is basically a table associating a cell in layout B with a cell in layout A.

    The cell mapping is of particular interest for providing the cell mapping recipe in \Cell#copy_tree_shapes or \Cell#move_tree_shapes.

    The mapping object is used to create and hold that table. There are three basic modes in which a table can be generated:

    @ul
      @li Top-level identity (\for_single_cell and \for_single_cell_full) @/li
      @li Top-level identify for multiple cells (\for_multi_cells_full and \for_multi_cells_full) @/li
      @li Geometrical identity (\from_geometry and \from_geometry_full)@/li
      @li Name identity (\from_names and \from_names_full) @/li
    @/ul

    'full' refers to the way cells are treated which are not mentioned. In the 'full' versions, cells for which no mapping is established explicitly - specifically all child cells in top-level identity modes - are created in the target layout and instantiated according to their source layout hierarchy. Then, these new cells become targets of the respective source cells. In the plain version (without 'full' cells), no additional cells are created. For the case of \Layout#copy_tree_shapes cells not explicitly mapped are flattened. Hence for example, \for_single_cell will flatten all children of the source cell during \Layout#copy_tree_shapes or \Layout#move_tree_shapes.

    Top-level identity means that only one cell (the top cell) is regarded identical. All child cells are not considered identical. In full mode (see below), this will create a new, identical cell tree below the top cell in layout A.

    Geometrical identity is defined by the exact identity of the set of expanded instances in each starting cell. Therefore, when a cell is mapped to another cell, shapes can be transferred from one cell to another while effectively rendering the same flat geometry (in the context of the given starting cells). Location identity is basically the safest way to map cells from one hierarchy into another, because it preserves the flat shape geometry. However in some cases the algorithm may find multiple mapping candidates. In that case it will make a guess about what mapping to choose.

    Name identity means that cells are identified by their names - for a source cell in layer B, a target cell with the same name is looked up in the target layout A and a mapping is created if a cell with the same name is found. However, name identity does not mean that the cells are actually equivalent because they may be placed differently. Hence, cell mapping by name is not a good choice when it is important to preserve the shape geometry of a layer.

    A cell might not be mapped to another cell which basically means that there is no corresponding cell. In this case, flattening to the next mapped cell is an option to transfer geometries despite the missing mapping. You can enforce a mapping by using the mapping generator methods in 'full' mode, i.e. \from_names_full or \from_geometry_full. These versions will create new cells and their corresponding instances in the target layout if no suitable target cell is found.

    This is a simple example for a cell mapping preserving the hierarchy of the source cell and creating a hierarchy copy in the top cell of the target layout ('hierarchical merge'):

    @code
    cell_names = [ "A", "B", "C" ]

    source = RBA::Layout::new
    source.read("input.gds")

    target = RBA::Layout::new
    target_top = target.create_cell("IMPORTED")

    cm = RBA::CellMapping::new
    # Copies the source layout hierarchy into the target top cell:
    cm.for_single_cell_full(target_top, source.top_cell)
    target.copy_tree_shapes(source, cm)
    @/code

    Without 'full', the effect is move-with-flattening (note we're using 'move' in this example):

    @code
    cell_names = [ "A", "B", "C" ]

    source = RBA::Layout::new
    source.read("input.gds")

    target = RBA::Layout::new
    target_top = target.create_cell("IMPORTED")

    cm = RBA::CellMapping::new
    # Flattens the source layout hierarchy into the target top cell:
    cm.for_single_cell(target_top, source.top_cell)
    target.move_tree_shapes(source, cm)
    @/code

    This is another example for using \CellMapping in multiple top cell identity mode. It extracts cells 'A', 'B' and 'C' from one layout and copies them to another. It will also copy all shapes and all child cells. Child cells which are shared between the three initial cells will be shared in the target layout too.

    @code
    cell_names = [ "A", "B", "C" ]

    source = RBA::Layout::new
    source.read("input.gds")

    target = RBA::Layout::new

    source_cells = cell_names.collect { |n| source.cell_by_name(n) }
    target_cells = cell_names.collect { |n| target.create_cell(n) }

    cm = RBA::CellMapping::new
    cm.for_multi_cells_full(target_cells, source_cells)
    target.copy_tree_shapes(source, cm)
    @/code
    """
    DropCell: ClassVar[int]
    r"""
    @brief A constant indicating the request to drop a cell

    If used as a pseudo-target for the cell mapping, this index indicates that the cell shall be dropped rather than created on the target side or skipped by flattening. Instead, all shapes of this cell are discarded and its children are not translated unless explicitly requested or if required are children for other cells.

    This constant has been introduced in version 0.25.
    """
    @classmethod
    def new(cls) -> CellMapping:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> CellMapping:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> CellMapping:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: CellMapping) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cell_mapping(self, cell_index_b: int) -> int:
        r"""
        @brief Determines cell mapping of a layout_b cell to the corresponding layout_a cell.


        @param cell_index_b The index of the cell in layout_b whose mapping is requested.
        @return The cell index in layout_a.

        Note that the returned index can be \DropCell to indicate the cell shall be dropped.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the mapping.

        This method has been introduced in version 0.23.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> CellMapping:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def for_multi_cells(self, cell_a: Sequence[Cell], cell_b: Sequence[Cell]) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param cell_a A list of target cells.
        @param cell_b A list of source cells.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_multi_cells(self, layout_a: Layout, cell_indexes_a: Sequence[int], layout_b: Layout, cell_indexes_b: Sequence[int]) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param layout_a The target layout.
        @param cell_indexes_a A list of cell indexes for the target cells.
        @param layout_b The source layout.
        @param cell_indexes_b A list of cell indexes for the source cells (same number of indexes than \cell_indexes_a).

        The cell mapping is created for cells from cell_indexes_b to cell from cell_indexes_a in the respective layouts. This method clears the mapping and creates one for each cell pair from cell_indexes_b vs. cell_indexes_a. If used for \Layout#copy_tree_shapes or \Layout#move_tree_shapes, this cell mapping will essentially flatten the source cells in the target layout.

        This method is equivalent to \clear, followed by \map(cell_index_a, cell_index_b) for each cell pair.

        This method has been introduced in version 0.27.
        """
    @overload
    def for_multi_cells_full(self, cell_a: Sequence[Cell], cell_b: Sequence[Cell]) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param cell_a A list of target cells.
        @param cell_b A list of source cells.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_multi_cells_full(self, layout_a: Layout, cell_indexes_a: Sequence[int], layout_b: Layout, cell_indexes_b: Sequence[int]) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param layout_a The target layout.
        @param cell_indexes_a A list of cell indexes for the target cells.
        @param layout_b The source layout.
        @param cell_indexes_b A list of cell indexes for the source cells (same number of indexes than \cell_indexes_a).

        The cell mapping is created for cells from cell_indexes_b to cell from cell_indexes_a in the respective layouts. This method clears the mapping and creates one for each cell pair from cell_indexes_b vs. cell_indexes_a. In addition and in contrast to \for_multi_cells, this method completes the mapping by adding all the child cells of all cells in cell_indexes_b to layout_a and creating the proper instances.

        This method has been introduced in version 0.27.
        """
    @overload
    def for_single_cell(self, cell_a: Cell, cell_b: Cell) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_single_cell(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param layout_a The target layout.
        @param cell_index_a The index of the target cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source cell.

        The cell mapping is created for cell_b to cell_a in the respective layouts. This method clears the mapping and creates one for the single cell pair. If used for \Cell#copy_tree or \Cell#move_tree, this cell mapping will essentially flatten the cell.

        This method is equivalent to \clear, followed by \map(cell_index_a, cell_index_b).

        This method has been introduced in version 0.23.
        """
    @overload
    def for_single_cell_full(self, cell_a: Cell, cell_b: Cell) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_single_cell_full(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param layout_a The target layout.
        @param cell_index_a The index of the target cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source cell.

        The cell mapping is created for cell_b to cell_a in the respective layouts. This method clears the mapping and creates one for the single cell pair. In addition and in contrast to \for_single_cell, this method completes the mapping by adding all the child cells of cell_b to layout_a and creating the proper instances.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_geometry(self, cell_a: Cell, cell_b: Cell) -> None:
        r"""
        @brief Initializes the cell mapping using the geometrical identity

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_geometry(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> None:
        r"""
        @brief Initializes the cell mapping using the geometrical identity

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts. This method employs geometrical identity to derive mappings for the child cells of the starting cell in layout A and B.
        If the geometrical identity is ambiguous, the algorithm will make an arbitrary choice.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_geometry_full(self, cell_a: Cell, cell_b: Cell) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the geometrical identity in full mapping mode

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_geometry_full(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the geometrical identity in full mapping mode

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.
        @return A list of indexes of cells created.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts. This method employs geometrical identity to derive mappings for the child cells of the starting cell in layout A and B.
        If the geometrical identity is ambiguous, the algorithm will make an arbitrary choice.

        Full mapping means that cells which are not found in the target layout A are created there plus their corresponding instances are created as well. The returned list will contain the indexes of all cells created for that reason.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_names(self, cell_a: Cell, cell_b: Cell) -> None:
        r"""
        @brief Initializes the cell mapping using the name identity

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_names(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> None:
        r"""
        @brief Initializes the cell mapping using the name identity

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts.
        This method employs name identity to derive mappings for the child cells of the starting cell in layout A and B.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_names_full(self, cell_a: Cell, cell_b: Cell) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the name identity in full mapping mode

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_names_full(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the name identity in full mapping mode

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.
        @return A list of indexes of cells created.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts.
        This method employs name identity to derive mappings for the child cells of the starting cell in layout A and B.

        Full mapping means that cells which are not found in the target layout A are created there plus their corresponding instances are created as well. The returned list will contain the indexes of all cells created for that reason.

        This method has been introduced in version 0.23.
        """
    def has_mapping(self, cell_index_b: int) -> bool:
        r"""
        @brief Returns as value indicating whether a cell of layout_b has a mapping to a layout_a cell.

        @param cell_index_b The index of the cell in layout_b whose mapping is requested.
        @return true, if the cell has a mapping

        Note that if the cell is supposed to be dropped (see \DropCell), the respective source cell will also be regarded "mapped", so has_mapping? will return true in this case.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def map(self, cell_index_b: int, cell_index_a: int) -> None:
        r"""
        @brief Explicitly specifies a mapping.


        @param cell_index_b The index of the cell in layout B (the "source")
        @param cell_index_a The index of the cell in layout A (the "target") - this index can be \DropCell

        Beside using the mapping generator algorithms provided through \from_names and \from_geometry, it is possible to explicitly specify cell mappings using this method.

        This method has been introduced in version 0.23.
        """
    def table(self) -> Dict[int, int]:
        r"""
        @brief Returns the mapping table.

        The mapping table is a dictionary where the keys are source layout cell indexes and the values are the target layout cell indexes.
        Note that the target cell index can be \DropCell to indicate that a cell is supposed to be dropped.

        This method has been introduced in version 0.25.
        """

class Circuit(NetlistObject):
    r"""
    @brief Circuits are the basic building blocks of the netlist
    A circuit has pins by which it can connect to the outside. Pins are created using \create_pin and are represented by the \Pin class.

    Furthermore, a circuit manages the components of the netlist. Components are devices (class \Device) and subcircuits (class \SubCircuit). Devices are basic devices such as resistors or transistors. Subcircuits are other circuits to which nets from this circuit connect. Devices are created using the \create_device method. Subcircuits are created using the \create_subcircuit method.

    Devices are connected through 'terminals', subcircuits are connected through their pins. Terminals and pins are described by integer ID's in the context of most methods.

    Finally, the circuit consists of the nets. Nets connect terminals of devices and pins of subcircuits or the circuit itself. Nets are created using \create_net and are represented by objects of the \Net class.
    See there for more about nets.

    The Circuit object is only valid if the netlist object is alive. Circuits must be added to a netlist using \Netlist#add to become part of the netlist.

    The Circuit class has been introduced in version 0.26.
    """
    boundary: DPolygon
    r"""
    Getter:
    @brief Gets the boundary of the circuit
    Setter:
    @brief Sets the boundary of the circuit
    """
    cell_index: int
    r"""
    Getter:
    @brief Gets the cell index of the circuit
    See \cell_index= for details.

    Setter:
    @brief Sets the cell index
    The cell index relates a circuit with a cell from a layout. It's intended to hold a cell index number if the netlist was extracted from a layout.
    """
    dont_purge: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the circuit can be purged on \Netlist#purge.

    Setter:
    @brief Sets a value indicating whether the circuit can be purged on \Netlist#purge.
    If this attribute is set to true, \Netlist#purge will never delete this circuit.
    This flag therefore marks this circuit as 'precious'.
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the circuit
    Setter:
    @brief Sets the name of the circuit
    """
    def _assign(self, other: NetlistObject) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> Circuit:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def blank(self) -> None:
        r"""
        @brief Blanks out the circuit
        This method will remove all the innards of the circuit and just leave the pins. The pins won't be connected to inside nets anymore, but the circuit can still be called by subcircuit references. This method will eventually create a 'circuit abstract' (or black box). It will set the \dont_purge flag to mark this circuit as 'intentionally empty'.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the circuit
        This method removes all objects and clears the other attributes.
        """
    def combine_devices(self) -> None:
        r"""
        @brief Combines devices where possible
        This method will combine devices that can be combined according to their device classes 'combine_devices' method.
        For example, serial or parallel resistors can be combined into a single resistor.
        """
    @overload
    def connect_pin(self, pin: Pin, net: Net) -> None:
        r"""
        @brief Connects the given pin with the given net.
        The net and the pin must be objects from inside the circuit. Any previous connected is resolved before this connection is made. A pin can only be connected to one net at a time.
        """
    @overload
    def connect_pin(self, pin_id: int, net: Net) -> None:
        r"""
        @brief Connects the given pin with the given net.
        The net must be one inside the circuit. Any previous connected is resolved before this connection is made. A pin can only be connected to one net at a time.
        """
    def create_device(self, device_class: DeviceClass, name: Optional[str] = ...) -> Device:
        r"""
        @brief Creates a new bound \Device object inside the circuit
        This object describes a device of the circuit. The device is already attached to the device class. The name is optional and is used to identify the device in a netlist file.

        For more details see the \Device class.
        """
    def create_net(self, name: Optional[str] = ...) -> Net:
        r"""
        @brief Creates a new \Net object inside the circuit
        This object will describe a net of the circuit. The nets are basically connections between the different components of the circuit (subcircuits, devices and pins).

        A net needs to be filled with references to connect to specific objects. See the \Net class for more details.
        """
    def create_pin(self, name: str) -> Pin:
        r"""
        @brief Creates a new \Pin object inside the circuit
        This object will describe a pin of the circuit. A circuit connects to the outside through such a pin. The pin is added after all existing pins. For more details see the \Pin class.

        Starting with version 0.26.8, this method returns a reference to a \Pin object rather than a copy.
        """
    def create_subcircuit(self, circuit: Circuit, name: Optional[str] = ...) -> SubCircuit:
        r"""
        @brief Creates a new bound \SubCircuit object inside the circuit
        This object describes an instance of another circuit inside the circuit. The subcircuit is already attached to the other circuit. The name is optional and is used to identify the subcircuit in a netlist file.

        For more details see the \SubCircuit class.
        """
    @overload
    def device_by_id(self, id: int) -> Device:
        r"""
        @brief Gets the device object for a given ID.
        If the ID is not a valid device ID, nil is returned.
        """
    @overload
    def device_by_id(self, id: int) -> Device:
        r"""
        @brief Gets the device object for a given ID (const version).
        If the ID is not a valid device ID, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def device_by_name(self, name: str) -> Device:
        r"""
        @brief Gets the device object for a given name.
        If the ID is not a valid device name, nil is returned.
        """
    @overload
    def device_by_name(self, name: str) -> Device:
        r"""
        @brief Gets the device object for a given name (const version).
        If the ID is not a valid device name, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def disconnect_pin(self, pin: Pin) -> None:
        r"""
        @brief Disconnects the given pin from any net.
        """
    @overload
    def disconnect_pin(self, pin_id: int) -> None:
        r"""
        @brief Disconnects the given pin from any net.
        """
    @overload
    def each_child(self) -> Iterator[Circuit]:
        r"""
        @brief Iterates over the child circuits of this circuit
        Child circuits are the ones that are referenced from this circuit via subcircuits.
        """
    @overload
    def each_child(self) -> Iterator[Circuit]:
        r"""
        @brief Iterates over the child circuits of this circuit (const version)
        Child circuits are the ones that are referenced from this circuit via subcircuits.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def each_device(self) -> Iterator[Device]:
        r"""
        @brief Iterates over the devices of the circuit
        """
    @overload
    def each_device(self) -> Iterator[Device]:
        r"""
        @brief Iterates over the devices of the circuit (const version)

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def each_net(self) -> Iterator[Net]:
        r"""
        @brief Iterates over the nets of the circuit
        """
    @overload
    def each_net(self) -> Iterator[Net]:
        r"""
        @brief Iterates over the nets of the circuit (const version)

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def each_parent(self) -> Iterator[Circuit]:
        r"""
        @brief Iterates over the parent circuits of this circuit
        Child circuits are the ones that are referencing this circuit via subcircuits.
        """
    @overload
    def each_parent(self) -> Iterator[Circuit]:
        r"""
        @brief Iterates over the parent circuits of this circuit (const version)
        Child circuits are the ones that are referencing this circuit via subcircuits.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def each_pin(self) -> Iterator[Pin]:
        r"""
        @brief Iterates over the pins of the circuit
        """
    @overload
    def each_pin(self) -> Iterator[Pin]:
        r"""
        @brief Iterates over the pins of the circuit (const version)

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def each_ref(self) -> Iterator[SubCircuit]:
        r"""
        @brief Iterates over the subcircuit objects referencing this circuit
        """
    @overload
    def each_ref(self) -> Iterator[SubCircuit]:
        r"""
        @brief Iterates over the subcircuit objects referencing this circuit (const version)


        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def each_subcircuit(self) -> Iterator[SubCircuit]:
        r"""
        @brief Iterates over the subcircuits of the circuit
        """
    @overload
    def each_subcircuit(self) -> Iterator[SubCircuit]:
        r"""
        @brief Iterates over the subcircuits of the circuit (const version)

        This constness variant has been introduced in version 0.26.8
        """
    def flatten_subcircuit(self, subcircuit: SubCircuit) -> None:
        r"""
        @brief Flattens a subcircuit
        This method will substitute the given subcircuit by its contents. The subcircuit is removed after this.
        """
    def has_refs(self) -> bool:
        r"""
        @brief Returns a value indicating whether the circuit has references
        A circuit has references if there is at least one subcircuit referring to it.
        """
    def join_nets(self, net: Net, with_: Net) -> None:
        r"""
        @brief Joins (connects) two nets into one
        This method will connect the 'with' net with 'net' and remove 'with'.

        This method has been introduced in version 0.26.4. Starting with version 0.28.13, net names will be formed from both input names, combining them with as a comma-separated list.
        """
    def net_by_cluster_id(self, cluster_id: int) -> Net:
        r"""
        @brief Gets the net object corresponding to a specific cluster ID
        If the ID is not a valid pin cluster ID, nil is returned.
        """
    @overload
    def net_by_name(self, name: str) -> Net:
        r"""
        @brief Gets the net object for a given name.
        If the ID is not a valid net name, nil is returned.
        """
    @overload
    def net_by_name(self, name: str) -> Net:
        r"""
        @brief Gets the net object for a given name (const version).
        If the ID is not a valid net name, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def net_for_pin(self, pin: Pin) -> Net:
        r"""
        @brief Gets the net object attached to a specific pin.
        This is the net object inside the circuit which attaches to the given outward-bound pin.
        This method returns nil if the pin is not connected or the pin object is nil.
        """
    @overload
    def net_for_pin(self, pin: Pin) -> Net:
        r"""
        @brief Gets the net object attached to a specific pin (const version).
        This is the net object inside the circuit which attaches to the given outward-bound pin.
        This method returns nil if the pin is not connected or the pin object is nil.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def net_for_pin(self, pin_id: int) -> Net:
        r"""
        @brief Gets the net object attached to a specific pin.
        This is the net object inside the circuit which attaches to the given outward-bound pin.
        This method returns nil if the pin is not connected or the pin ID is invalid.
        """
    @overload
    def net_for_pin(self, pin_id: int) -> Net:
        r"""
        @brief Gets the net object attached to a specific pin (const version).
        This is the net object inside the circuit which attaches to the given outward-bound pin.
        This method returns nil if the pin is not connected or the pin ID is invalid.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def netlist(self) -> Netlist:
        r"""
        @brief Gets the netlist object the circuit lives in
        """
    @overload
    def netlist(self) -> Netlist:
        r"""
        @brief Gets the netlist object the circuit lives in (const version)

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def nets_by_name(self, name_pattern: str) -> List[Net]:
        r"""
        @brief Gets the net objects for a given name filter.
        The name filter is a glob pattern. This method will return all \Net objects matching the glob pattern.

        This method has been introduced in version 0.27.3.
        """
    @overload
    def nets_by_name(self, name_pattern: str) -> List[Net]:
        r"""
        @brief Gets the net objects for a given name filter (const version).
        The name filter is a glob pattern. This method will return all \Net objects matching the glob pattern.


        This constness variant has been introduced in version 0.27.3
        """
    @overload
    def pin_by_id(self, id: int) -> Pin:
        r"""
        @brief Gets the \Pin object corresponding to a specific ID
        If the ID is not a valid pin ID, nil is returned.
        """
    @overload
    def pin_by_id(self, id: int) -> Pin:
        r"""
        @brief Gets the \Pin object corresponding to a specific ID (const version)
        If the ID is not a valid pin ID, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def pin_by_name(self, name: str) -> Pin:
        r"""
        @brief Gets the \Pin object corresponding to a specific name
        If the ID is not a valid pin name, nil is returned.
        """
    @overload
    def pin_by_name(self, name: str) -> Pin:
        r"""
        @brief Gets the \Pin object corresponding to a specific name (const version)
        If the ID is not a valid pin name, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """
    def pin_count(self) -> int:
        r"""
        @brief Gets the number of pins in the circuit
        """
    def purge_nets(self) -> None:
        r"""
        @brief Purges floating nets.
        Floating nets are nets with no device or subcircuit attached to. Such floating nets are removed in this step. If these nets are connected outward to a circuit pin, this circuit pin is also removed.
        """
    def purge_nets_keep_pins(self) -> None:
        r"""
        @brief Purges floating nets but keep pins.
        This method will remove floating nets like \purge_nets, but if these nets are attached to a pin, the pin will be left disconnected from any net.

        This method has been introduced in version 0.26.2.
        """
    def remove_device(self, device: Device) -> None:
        r"""
        @brief Removes the given device from the circuit
        """
    def remove_net(self, net: Net) -> None:
        r"""
        @brief Removes the given net from the circuit
        """
    def remove_pin(self, id: int) -> None:
        r"""
        @brief Removes the pin with the given ID from the circuit

        This method has been introduced in version 0.26.2.
        """
    def remove_subcircuit(self, subcircuit: SubCircuit) -> None:
        r"""
        @brief Removes the given subcircuit from the circuit
        """
    def rename_pin(self, id: int, new_name: str) -> None:
        r"""
        @brief Renames the pin with the given ID to 'new_name'

        This method has been introduced in version 0.26.8.
        """
    @overload
    def subcircuit_by_id(self, id: int) -> SubCircuit:
        r"""
        @brief Gets the subcircuit object for a given ID.
        If the ID is not a valid subcircuit ID, nil is returned.
        """
    @overload
    def subcircuit_by_id(self, id: int) -> SubCircuit:
        r"""
        @brief Gets the subcircuit object for a given ID (const version).
        If the ID is not a valid subcircuit ID, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def subcircuit_by_name(self, name: str) -> SubCircuit:
        r"""
        @brief Gets the subcircuit object for a given name.
        If the ID is not a valid subcircuit name, nil is returned.
        """
    @overload
    def subcircuit_by_name(self, name: str) -> SubCircuit:
        r"""
        @brief Gets the subcircuit object for a given name (const version).
        If the ID is not a valid subcircuit name, nil is returned.

        This constness variant has been introduced in version 0.26.8
        """

class CompoundRegionOperationNode:
    r"""
    @brief A base class for compound DRC operations

    This class is not intended to be used directly but rather provide a factory for various incarnations of compound operation nodes. Compound operations are a way to specify complex DRC operations put together by building a tree of operations. This operation tree then is executed with \Region#complex_op and will act on individual clusters of shapes and their interacting neighbors.

    A basic concept to the compound operations is the 'subject' (primary) and 'intruder' (secondary) input. The 'subject' is the Region, 'complex_op' with the operation tree is executed on. 'intruders' are regions inserted into the equation through secondary input nodes created with \new_secondary_node. The algorithm will execute the operation tree for every subject shape considering intruder shapes from the secondary inputs. The algorithm will only act on subject shapes primarily. As a consequence, 'lonely' intruder shapes without a subject shape are not considered at all. Only subject shapes trigger evaluation of the operation tree.

    The search distance for intruder shapes is determined by the operation and computed from the operation's requirements.

    This class has been introduced in version 0.27. The API is considered internal and will change without notice.
    """
    class GeometricalOp:
        r"""
        @brief This class represents the CompoundRegionOperationNode::GeometricalOp enum

        This enum has been introduced in version 0.27.
        """
        And: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '&' (and).
        """
        Not: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '-' (not).
        """
        Or: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '|' (or).
        """
        Xor: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '^' (xor).
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.GeometricalOp:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.GeometricalOp:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        def __hash__(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        def __int__(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.GeometricalOp) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def hash(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class LogicalOp:
        r"""
        @brief This class represents the CompoundRegionOperationNode::LogicalOp enum

        This enum has been introduced in version 0.27.
        """
        LogAnd: ClassVar[CompoundRegionOperationNode.LogicalOp]
        r"""
        @brief Indicates a logical '&&' (and).
        """
        LogOr: ClassVar[CompoundRegionOperationNode.LogicalOp]
        r"""
        @brief Indicates a logical '||' (or).
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.LogicalOp:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.LogicalOp:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        def __hash__(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        def __int__(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.LogicalOp) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def hash(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class ParameterType:
        r"""
        @brief This class represents the parameter type enum used in \CompoundRegionOperationNode#new_bbox_filter

        This enum has been introduced in version 0.27.
        """
        BoxAverageDim: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the average of width and height of the bounding box
        """
        BoxHeight: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the height of the bounding box
        """
        BoxMaxDim: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the maximum dimension of the bounding box
        """
        BoxMinDim: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the minimum dimension of the bounding box
        """
        BoxWidth: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the width of the bounding box
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.ParameterType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.ParameterType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        def __hash__(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        def __int__(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.ParameterType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def hash(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class RatioParameterType:
        r"""
        @brief This class represents the parameter type enum used in \CompoundRegionOperationNode#new_ratio_filter

        This enum has been introduced in version 0.27.
        """
        AreaRatio: ClassVar[CompoundRegionOperationNode.RatioParameterType]
        r"""
        @brief Measures the area ratio (bounding box area / polygon area)
        """
        AspectRatio: ClassVar[CompoundRegionOperationNode.RatioParameterType]
        r"""
        @brief Measures the aspect ratio of the bounding box (larger / smaller dimension)
        """
        RelativeHeight: ClassVar[CompoundRegionOperationNode.RatioParameterType]
        r"""
        @brief Measures the relative height (height / width)
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.RatioParameterType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.RatioParameterType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        def __hash__(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        def __int__(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.RatioParameterType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def hash(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class ResultType:
        r"""
        @brief This class represents the CompoundRegionOperationNode::ResultType enum

        This enum has been introduced in version 0.27.
        """
        EdgePairs: ClassVar[CompoundRegionOperationNode.ResultType]
        r"""
        @brief Indicates edge pair result type.
        """
        Edges: ClassVar[CompoundRegionOperationNode.ResultType]
        r"""
        @brief Indicates edge result type.
        """
        Region: ClassVar[CompoundRegionOperationNode.ResultType]
        r"""
        @brief Indicates polygon result type.
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.ResultType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.ResultType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        def __hash__(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        def __int__(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.ResultType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def hash(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    description: str
    r"""
    Getter:
    @brief Gets the description for this node
    Setter:
    @brief Sets the description for this node
    """
    distance: int
    r"""
    Getter:
    @brief Gets the distance value for this node
    Setter:
    @brief Sets the distance value for this nodeUsually it's not required to provide a distance because the nodes compute a distance based on their operation. If necessary you can supply a distance. The processor will use this distance or the computed one, whichever is larger.
    """
    @classmethod
    def new(cls) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a new object of this class
        """
    @classmethod
    def new_area_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., amin: Optional[int] = ..., amax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by area.
        This node renders the input if the area is between amin and amax (exclusively). If 'inverse' is set to true, the input shape is returned if the area is less than amin (exclusively) or larger than amax (inclusively).
        """
    @classmethod
    def new_area_sum_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., amin: Optional[int] = ..., amax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by area sum.
        Like \new_area_filter, but applies to the sum of all shapes in the current set.
        """
    @classmethod
    def new_bbox_filter(cls, input: CompoundRegionOperationNode, parameter: CompoundRegionOperationNode.ParameterType, inverse: Optional[bool] = ..., pmin: Optional[int] = ..., pmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by bounding box parameters.
        This node renders the input if the specified bounding box parameter of the input shape is between pmin and pmax (exclusively). If 'inverse' is set to true, the input shape is returned if the parameter is less than pmin (exclusively) or larger than pmax (inclusively).
        """
    @classmethod
    def new_case(cls, inputs: Sequence[CompoundRegionOperationNode]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a 'switch ladder' (case statement) compound operation node.

        The inputs are treated as a sequence of condition/result pairs: c1,r1,c2,r2 etc. If there is an odd number of inputs, the last element is taken as the default result. The implementation will evaluate c1 and if not empty, will render r1. Otherwise, c2 will be evaluated and r2 rendered if c2 isn't empty etc. If none of the conditions renders a non-empty set and a default result is present, the default will be returned. Otherwise, the result is empty.
        """
    @classmethod
    def new_centers(cls, input: CompoundRegionOperationNode, length: int, fraction: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a part at the center of each input edge.
        """
    @classmethod
    def new_convex_decomposition(cls, input: CompoundRegionOperationNode, mode: PreferredOrientation) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a composition into convex pieces.
        """
    @classmethod
    def new_corners_as_dots(cls, input: CompoundRegionOperationNode, angle_min: float, include_angle_min: bool, angle_max: float, include_angle_max: bool, inverse: Optional[bool] = ..., absolute: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node turning corners into dots (single-point edges).

        'absolute' and 'inverse' arguments have been added in version 0.29.1.
        """
    @classmethod
    def new_corners_as_edge_pairs(cls, input: CompoundRegionOperationNode, angle_min: float, include_angle_min: bool, angle_max: float, include_angle_max: bool, inverse: Optional[bool] = ..., absolute: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node turning corners into edge pairs containing the two edges adjacent to the corner.
        The first edge will be the incoming edge and the second one the outgoing edge.

        This feature has been introduced in version 0.27.1. 'absolute' and 'inverse' arguments have been added in version 0.29.1.
        """
    @classmethod
    def new_corners_as_rectangles(cls, input: CompoundRegionOperationNode, angle_min: float, include_angle_min: bool, angle_max: float, include_angle_max: bool, dim: int, inverse: Optional[bool] = ..., absolute: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node turning corners into rectangles.

        'absolute' and 'inverse' arguments have been added in version 0.29.1.
        """
    @classmethod
    def new_count_filter(cls, inputs: CompoundRegionOperationNode, invert: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node selecting results but their shape count.
        """
    @classmethod
    def new_edge_length_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., lmin: Optional[int] = ..., lmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering edges by their length.
        """
    @classmethod
    def new_edge_length_sum_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., lmin: Optional[int] = ..., lmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering edges by their length sum (over the local set).
        """
    @classmethod
    def new_edge_orientation_filter(cls, input: CompoundRegionOperationNode, inverse: bool, amin: float, include_amin: bool, amax: float, include_amax: bool, absolute_angle: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering edges by their orientation.

        'absolute_angle' has been introduced in version 0.29.1.
        """
    @classmethod
    def new_edge_pair_to_first_edges(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering the first edge of each edges pair.
        """
    @classmethod
    def new_edge_pair_to_second_edges(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering the second edge of each edges pair.
        """
    @classmethod
    def new_edges(cls, input: CompoundRegionOperationNode, mode: Optional[EdgeMode] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node converting polygons into its edges.
        The 'mode' argument allows selecting specific edges when generating edges from a polygon. See \EdgeMode for the various options. By default, all edges are generated from polygons.

        The 'mode' argument has been added in version 0.29.
        """
    @classmethod
    def new_empty(cls, type: CompoundRegionOperationNode.ResultType) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering an empty result of the given type
        """
    @classmethod
    def new_enclosed_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing an enclosed (secondary enclosing primary) check.

        This method has been added in version 0.27.5. The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_enclosing(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an inside selection operation between the inputs.
        """
    @classmethod
    def new_enclosing_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing an inside (enclosure) check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_end_segments(cls, input: CompoundRegionOperationNode, length: int, fraction: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a part at the end of each input edge.
        """
    @classmethod
    def new_extended(cls, input: CompoundRegionOperationNode, ext_b: int, ext_e: int, ext_o: int, ext_i: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a polygonized version of the edges with the four extension parameters.
        """
    @classmethod
    def new_extended_in(cls, input: CompoundRegionOperationNode, e: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a polygonized, inside-extended version of the edges.
        """
    @classmethod
    def new_extended_out(cls, input: CompoundRegionOperationNode, e: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a polygonized, inside-extended version of the edges.
        """
    @classmethod
    def new_extents(cls, input: CompoundRegionOperationNode, e: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node returning the extents of the objects.
        The 'e' parameter provides a generic enlargement which is applied to the boxes. This is helpful to cover dot-like edges or edge pairs in the input.
        """
    @classmethod
    def new_foreign(cls) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node object representing the primary input without the current polygon
        """
    @classmethod
    def new_geometrical_boolean(cls, op: CompoundRegionOperationNode.GeometricalOp, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing a geometrical boolean operation between the inputs.
        """
    @classmethod
    def new_hole_count_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., hmin: Optional[int] = ..., hmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by number of holes per polygon.
        This node renders the input if the hole count is between hmin and hmax (exclusively). If 'inverse' is set to true, the input shape is returned if the hole count is less than hmin (exclusively) or larger than hmax (inclusively).
        """
    @classmethod
    def new_holes(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node extracting the holes from polygons.
        """
    @classmethod
    def new_hulls(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node extracting the hulls from polygons.
        """
    @classmethod
    def new_inside(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an inside selection operation between the inputs.
        """
    @classmethod
    def new_interacting(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an interacting selection operation between the inputs.
        """
    @classmethod
    def new_isolated_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a isolated polygons (space between different polygons) check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_join(cls, inputs: Sequence[CompoundRegionOperationNode]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node that joins the inputs.
        """
    @classmethod
    def new_logical_boolean(cls, op: CompoundRegionOperationNode.LogicalOp, invert: bool, inputs: Sequence[CompoundRegionOperationNode]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing a logical boolean operation between the inputs.

        A logical AND operation will evaluate the arguments and render the subject shape when all arguments are non-empty. The logical OR operation will evaluate the arguments and render the subject shape when one argument is non-empty. Setting 'inverse' to true will reverse the result and return the subject shape when one argument is empty in the AND case and when all arguments are empty in the OR case.
        """
    @classmethod
    def new_merged(cls, input: CompoundRegionOperationNode, min_coherence: Optional[bool] = ..., min_wc: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing merged input polygons.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, e: Edge) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with an edge.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, p: Box) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a box.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, p: Polygon) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a polygon.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, p: Sequence[Point]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a point sequence forming a contour.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, e: Edge) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with an edge.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, p: Box) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a box.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, p: Polygon) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a polygon.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, p: Sequence[Point]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a point sequence forming a contour.
        """
    @classmethod
    def new_notch_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a intra-polygon space check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_outside(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an outside selection operation between the inputs.
        """
    @classmethod
    def new_overlap_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing an overlap check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_overlapping(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an overlapping selection operation between the inputs.
        """
    @classmethod
    def new_perimeter_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., pmin: Optional[int] = ..., pmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by perimeter.
        This node renders the input if the perimeter is between pmin and pmax (exclusively). If 'inverse' is set to true, the input shape is returned if the perimeter is less than pmin (exclusively) or larger than pmax (inclusively).
        """
    @classmethod
    def new_perimeter_sum_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., amin: Optional[int] = ..., amax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by area sum.
        Like \new_perimeter_filter, but applies to the sum of all shapes in the current set.
        """
    @classmethod
    def new_polygon_breaker(cls, input: CompoundRegionOperationNode, max_vertex_count: int, max_area_ratio: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a composition into parts with less than the given number of points and a smaller area ratio.
        """
    @classmethod
    def new_polygons(cls, input: CompoundRegionOperationNode, e: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node converting the input to polygons.
        @param e The enlargement parameter when converting edges or edge pairs to polygons.
        """
    @classmethod
    def new_primary(cls) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node object representing the primary input
        """
    @classmethod
    def new_ratio_filter(cls, input: CompoundRegionOperationNode, parameter: CompoundRegionOperationNode.RatioParameterType, inverse: Optional[bool] = ..., pmin: Optional[float] = ..., pmin_included: Optional[bool] = ..., pmax: Optional[float] = ..., pmax_included: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by ratio parameters.
        This node renders the input if the specified ratio parameter of the input shape is between pmin and pmax. If 'pmin_included' is true, the range will include pmin. Same for 'pmax_included' and pmax. If 'inverse' is set to true, the input shape is returned if the parameter is not within the specified range.
        """
    @classmethod
    def new_rectangle_filter(cls, input: CompoundRegionOperationNode, is_square: Optional[bool] = ..., inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input for rectangular or square shapes.
        If 'is_square' is true, only squares will be selected. If 'inverse' is true, the non-rectangle/non-square shapes are returned.
        """
    @classmethod
    def new_rectilinear_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input for rectilinear shapes (or non-rectilinear ones with 'inverse' set to 'true').
        """
    @classmethod
    def new_relative_extents(cls, input: CompoundRegionOperationNode, fx1: float, fy1: float, fx2: float, fy2: float, dx: int, dy: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node returning markers at specified locations of the extent (e.g. at the center).
        """
    @classmethod
    def new_relative_extents_as_edges(cls, input: CompoundRegionOperationNode, fx1: float, fy1: float, fx2: float, fy2: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node returning edges at specified locations of the extent (e.g. at the center).
        """
    @classmethod
    def new_rounded_corners(cls, input: CompoundRegionOperationNode, rinner: float, router: float, n: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node generating rounded corners.
        @param rinner The inner corner radius.@param router The outer corner radius.@param n The number if points per full circle.
        """
    @classmethod
    def new_secondary(cls, region: Region) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node object representing the secondary input from the given region
        """
    @classmethod
    def new_separation_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a separation check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_sized(cls, input: CompoundRegionOperationNode, dx: int, dy: int, mode: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing sizing.
        """
    @classmethod
    def new_smoothed(cls, input: CompoundRegionOperationNode, d: int, keep_hv: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node smoothing the polygons.
        @param d The tolerance to be applied for the smoothing.
        @param keep_hv If true, horizontal and vertical edges are maintained.
        """
    @classmethod
    def new_space_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a space check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    @classmethod
    def new_start_segments(cls, input: CompoundRegionOperationNode, length: int, fraction: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a part at the beginning of each input edge.
        """
    @classmethod
    def new_strange_polygons_filter(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node extracting strange polygons.
        'strange polygons' are ones which cannot be oriented - e.g. '8' shape polygons.
        """
    @classmethod
    def new_trapezoid_decomposition(cls, input: CompoundRegionOperationNode, mode: TrapezoidDecompositionMode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a composition into trapezoids.
        """
    @classmethod
    def new_width_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., zero_distance_mode: Optional[ZeroDistanceMode] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a width check.

        The zero_distance_mode argument has been inserted in version 0.29.
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def result_type(self) -> CompoundRegionOperationNode.ResultType:
        r"""
        @brief Gets the result type of this node
        """

class Connectivity:
    r"""
    @brief This class specifies connections between different layers.
    Connections are build using \connect. There are basically two flavours of connections: intra-layer and inter-layer.

    Intra-layer connections make nets begin propagated along different shapes on the same net. Without the intra-layer connections, nets are not propagated over shape boundaries. As this is usually intended, intra-layer connections should always be specified for each layer.

    Inter-layer connections connect shapes on different layers. Shapes which touch across layers will be connected if their layers are specified as being connected through inter-layer \connect.

    All layers are specified in terms of layer indexes. Layer indexes are layout layer indexes (see \Layout class).

    The connectivity object also manages the global nets. Global nets are substrate for example and they are propagated automatically from subcircuits to circuits. Global nets are defined by name and are managed through IDs. To get the name for a given ID, use \global_net_name.
    Starting with version 0.29, soft connections are supported. Soft connections attach to high-ohmic substrate or diffusion layers (the 'lower' layer) are upon netlist extraction it will be checked that no wiring is routed over such connections. See \soft_connect and \soft_global_connect for details.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls) -> Connectivity:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> Connectivity:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Connectivity:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def __repr__(self) -> str:
        r"""
        @hide
        """
    def __str__(self) -> str:
        r"""
        @hide
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: Connectivity) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def connect(self, layer: int) -> None:
        r"""
        @brief Specifies intra-layer connectivity.
        This method specifies a hard connection between shapes on the given layer. Without specifying such a connection, shapes on that layer do not form connection regions.
        """
    @overload
    def connect(self, layer_a: int, layer_b: int) -> None:
        r"""
        @brief Specifies inter-layer connectivity.
        This method specifies a hard connection between shapes on layer_a and layer_b.
        """
    def connect_global(self, layer: int, global_net_name: str) -> int:
        r"""
        @brief Connects the given layer to the global net given by name.
        Returns the ID of the global net.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Connectivity:
        r"""
        @brief Creates a copy of self
        """
    def global_net_id(self, global_net_name: str) -> int:
        r"""
        @brief Gets the ID for a given global net name.
        """
    def global_net_name(self, global_net_id: int) -> str:
        r"""
        @brief Gets the name for a given global net ID.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def soft_connect(self, layer_a: int, layer_b: int) -> None:
        r"""
        @brief Specifies a soft connection between layer_a and layer_b.
        @param layer_a The 'upper' layer
        @param layer_b The 'lower' layer
        Soft connections are made between a lower and an upper layer. The lower layer conceptually is a high-ohmic (i.e. substrate, diffusion) region that is not intended for signal wiring. The netlist extraction will check that no routing happens over such regions.

        Soft connections have in introduced in version 0.29.
        """
    def soft_connect_global(self, layer: int, global_net_name: str) -> int:
        r"""
        @brief Soft-connects the given layer to the global net given by name.
        Returns the ID of the global net.
        See \soft_connect for a description of the soft connection feature. The global net is always the 'lower' (i.e. high-ohmic, substrate) part of the soft connection.

        Soft connections have in introduced in version 0.29.
        """
    def to_s(self) -> str:
        r"""
        @hide
        """

class CplxTrans:
    r"""
    @brief A complex transformation

    A complex transformation provides magnification, mirroring at the x-axis, rotation by an arbitrary
    angle and a displacement. This is also the order, the operations are applied.
    This version can transform integer-coordinate objects into floating-point coordinate objects. This is the generic and exact case, for example for non-integer magnifications.

    Complex transformations are extensions of the simple transformation classes (\Trans or \DTrans in that case) and behave similar.

    Transformations can be used to transform points or other objects. Transformations can be combined with the '*' operator to form the transformation which is equivalent to applying the second and then the first. Here is some code:

    @code
    # Create a transformation that applies a magnification of 1.5, a rotation by 90 degree
    # and displacement of 10 in x and 20 units in y direction:
    t = RBA::DCplxTrans::new(1.5, 90, false, 10.0, 20.0)
    t.to_s            # r90 *1.5 10,20
    # compute the inverse:
    t.inverted.to_s   # r270 *0.666666667 -13,7
    # Combine with another displacement (applied after that):
    (RBA::DCplxTrans::new(5, 5) * t).to_s    # r90 *1.5 15,25
    # Transform a point:
    t.trans(RBA::DPoint::new(100, 200)).to_s # -290,170
    @/code

    The inverse type of the CplxTrans type is VCplxTrans which will transform floating-point to integer coordinate objects. Transformations of CplxTrans type can be concatenated (operator *) with either itself or with transformations of compatible input or output type. This means, the operator CplxTrans * ICplxTrans is allowed (output types of ICplxTrans and input of CplxTrans are identical) while CplxTrans * DCplxTrans is not.
    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    M0: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "mirrored at the x-axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M135: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "mirrored at the 135 degree axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M45: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "mirrored at the 45 degree axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M90: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "mirrored at the y (90 degree) axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R0: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "unrotated" (unit) transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R180: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "rotated by 180 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R270: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "rotated by 270 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R90: ClassVar[CplxTrans]
    r"""
    @brief A constant giving "rotated by 90 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    angle: float
    r"""
    Getter:
    @brief Gets the angle

    Note that the simple transformation returns the angle in units of 90 degree. Hence for a simple trans (i.e. \Trans), a rotation angle of 180 degree delivers a value of 2 for the angle attribute. The complex transformation, supporting any rotation angle returns the angle in degree.

    @return The rotation angle this transformation provides in degree units (0..360 deg).

    Setter:
    @brief Sets the angle
    @param a The new angle
    See \angle for a description of that attribute.
    """
    disp: DVector
    r"""
    Getter:
    @brief Gets the displacement

    Setter:
    @brief Sets the displacement
    @param u The new displacement
    """
    mag: float
    r"""
    Getter:
    @brief Gets the magnification

    Setter:
    @brief Sets the magnification
    @param m The new magnification
    """
    mirror: bool
    r"""
    Getter:
    @brief Gets the mirror flag

    If this property is true, the transformation is composed of a mirroring at the x-axis followed by a rotation by the angle given by the \angle property. 
    Setter:
    @brief Sets the mirror flag
    "mirroring" describes a reflection at the x-axis which is included in the transformation prior to rotation.@param m The new mirror flag
    """
    @classmethod
    def from_dtrans(cls, trans: DCplxTrans, dbu: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input space from floating-point units to integer units. Formally, the CplxTrans transformation is initialized with 'trans * from_dbu' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @classmethod
    def from_s(cls, s: str) -> CplxTrans:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> CplxTrans:
        r"""
        @brief Creates a unit transformation
        """
    @overload
    @classmethod
    def new(cls, c: CplxTrans, mag: Optional[float] = ..., u: Optional[DVector] = ...) -> CplxTrans:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param u The Additional displacement
        """
    @overload
    @classmethod
    def new(cls, c: CplxTrans, mag: Optional[float] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param x The Additional displacement (x)
        @param y The Additional displacement (y)
        """
    @overload
    @classmethod
    def new(cls, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., u: Optional[DVector] = ...) -> CplxTrans:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param u The displacement
        """
    @overload
    @classmethod
    def new(cls, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param x The x displacement
        @param y The y displacement
        """
    @overload
    @classmethod
    def new(cls, t: Trans, mag: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates a transformation from a simple transformation and a magnification

        Creates a magnifying transformation from a simple transformation and a magnification.
        """
    @overload
    @classmethod
    def new(cls, trans: DCplxTrans, dbu: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input space from floating-point units to integer units. Formally, the CplxTrans transformation is initialized with 'trans * from_dbu' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    @classmethod
    def new(cls, trans: ICplxTrans, dbu: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the output space from integer units to floating-point units. Formally, the CplxTrans transformation is initialized with 'from_dbu * trans' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    @classmethod
    def new(cls, trans: VCplxTrans, dbu: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input and output space from integer units to floating-point units and vice versa. Formally, the DCplxTrans transformation is initialized with 'from_dbu * trans * from_dbu' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    @classmethod
    def new(cls, u: DVector) -> CplxTrans:
        r"""
        @brief Creates a transformation from a displacement

        Creates a transformation with a displacement only.

        This method has been added in version 0.25.
        """
    @overload
    @classmethod
    def new(cls, x: float, y: float) -> CplxTrans:
        r"""
        @brief Creates a transformation from a x and y displacement

        This constructor will create a transformation with the specified displacement
        but no rotation.

        @param x The x displacement
        @param y The y displacement
        """
    def __copy__(self) -> CplxTrans:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> CplxTrans:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Tests for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given transformation. This method enables transformations as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates a unit transformation
        """
    @overload
    def __init__(self, c: CplxTrans, mag: Optional[float] = ..., u: Optional[DVector] = ...) -> None:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param u The Additional displacement
        """
    @overload
    def __init__(self, c: CplxTrans, mag: Optional[float] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param x The Additional displacement (x)
        @param y The Additional displacement (y)
        """
    @overload
    def __init__(self, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., u: Optional[DVector] = ...) -> None:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param u The displacement
        """
    @overload
    def __init__(self, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param x The x displacement
        @param y The y displacement
        """
    @overload
    def __init__(self, t: Trans, mag: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation from a simple transformation and a magnification

        Creates a magnifying transformation from a simple transformation and a magnification.
        """
    @overload
    def __init__(self, trans: DCplxTrans, dbu: Optional[float] = ...) -> None:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input space from floating-point units to integer units. Formally, the CplxTrans transformation is initialized with 'trans * from_dbu' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    def __init__(self, trans: ICplxTrans, dbu: Optional[float] = ...) -> None:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the output space from integer units to floating-point units. Formally, the CplxTrans transformation is initialized with 'from_dbu * trans' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    def __init__(self, trans: VCplxTrans, dbu: Optional[float] = ...) -> None:
        r"""
        @brief Creates an integer-to-floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input and output space from integer units to floating-point units and vice versa. Formally, the DCplxTrans transformation is initialized with 'from_dbu * trans * from_dbu' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    def __init__(self, u: DVector) -> None:
        r"""
        @brief Creates a transformation from a displacement

        Creates a transformation with a displacement only.

        This method has been added in version 0.25.
        """
    @overload
    def __init__(self, x: float, y: float) -> None:
        r"""
        @brief Creates a transformation from a x and y displacement

        This constructor will create a transformation with the specified displacement
        but no rotation.

        @param x The x displacement
        @param y The y displacement
        """
    def __lt__(self, other: CplxTrans) -> bool:
        r"""
        @brief Provides a 'less' criterion for sorting
        This method is provided to implement a sorting order. The definition of 'less' is opaque and might change in future versions.
        """
    @overload
    def __mul__(self, box: Box) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, d: int) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    @overload
    def __mul__(self, edge: Edge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, p: Point) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def __mul__(self, p: Vector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    @overload
    def __mul__(self, path: Path) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, polygon: Polygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, t: CplxTrans) -> CplxTrans:
        r"""
        @brief Returns the concatenated transformation

        The * operator returns self*t ("t is applied before this transformation").

        @param t The transformation to apply before
        @return The modified transformation
        """
    @overload
    def __mul__(self, t: ICplxTrans) -> CplxTrans:
        r"""
        @brief Multiplication (concatenation) of transformations

        The * operator returns self*t ("t is applied before this transformation").

        @param t The transformation to apply before
        @return The modified transformation
        """
    @overload
    def __mul__(self, t: VCplxTrans) -> DCplxTrans:
        r"""
        @brief Multiplication (concatenation) of transformations

        The * operator returns self*t ("t is applied before this transformation").

        @param t The transformation to apply before
        @return The modified transformation
        """
    @overload
    def __mul__(self, text: Text) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Tests for inequality
        """
    def __repr__(self, lazy: Optional[bool] = ..., dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If 'lazy' is true, some parts are omitted when not required.
        If a DBU is given, the output units will be micrometers.

        The lazy and DBU arguments have been added in version 0.27.6.
        """
    @overload
    def __rmul__(self, box: Box) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, d: int) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    @overload
    def __rmul__(self, edge: Edge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, p: Point) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def __rmul__(self, p: Vector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    @overload
    def __rmul__(self, path: Path) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, polygon: Polygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, text: Text) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    def __str__(self, lazy: Optional[bool] = ..., dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If 'lazy' is true, some parts are omitted when not required.
        If a DBU is given, the output units will be micrometers.

        The lazy and DBU arguments have been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: CplxTrans) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def ctrans(self, d: int) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> CplxTrans:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given transformation. This method enables transformations as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> CplxTrans:
        r"""
        @brief Inverts the transformation (in place)

        Inverts the transformation and replaces this transformation by its
        inverted one.

        @return The inverted transformation
        """
    def inverted(self) -> VCplxTrans:
        r"""
        @brief Returns the inverted transformation

        Returns the inverted transformation. This method does not modify the transformation.

        @return The inverted transformation
        """
    def is_complex(self) -> bool:
        r"""
        @brief Returns true if the transformation is a complex one

        If this predicate is false, the transformation can safely be converted to a simple transformation.
        Otherwise, this conversion will be lossy.
        The predicate value is equivalent to 'is_mag || !is_ortho'.

        This method has been introduced in version 0.27.5.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mag(self) -> bool:
        r"""
        @brief Tests, if the transformation is a magnifying one

        This is the recommended test for checking if the transformation represents
        a magnification.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Gets the mirror flag

        If this property is true, the transformation is composed of a mirroring at the x-axis followed by a rotation by the angle given by the \angle property. 
        """
    def is_ortho(self) -> bool:
        r"""
        @brief Tests, if the transformation is an orthogonal transformation

        If the rotation is by a multiple of 90 degree, this method will return true.
        """
    def is_unity(self) -> bool:
        r"""
        @brief Tests, whether this is a unit transformation
        """
    def rot(self) -> int:
        r"""
        @brief Returns the respective simple transformation equivalent rotation code if possible

        If this transformation is orthogonal (is_ortho () == true), then this method
        will return the corresponding fixpoint transformation, not taking into account
        magnification and displacement. If the transformation is not orthogonal, the result
        reflects the quadrant the rotation goes into.
        """
    def s_trans(self) -> Trans:
        r"""
        @brief Extracts the simple transformation part

        The simple transformation part does not reflect magnification or arbitrary angles.
        Rotation angles are rounded down to multiples of 90 degree. Magnification is fixed to 1.0.
        """
    def to_itrans(self, dbu: Optional[float] = ...) -> ICplxTrans:
        r"""
        @brief Converts the transformation to another transformation with integer input and output coordinates

        This method is redundant with the conversion constructors. Instead of 'to_itrans' use the conversion constructor:

        @code
        itrans = RBA::ICplxTrans::new(trans, dbu)
        @/code

        This method has been introduced in version 0.25 and was deprecated in version 0.29.
        """
    def to_s(self, lazy: Optional[bool] = ..., dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If 'lazy' is true, some parts are omitted when not required.
        If a DBU is given, the output units will be micrometers.

        The lazy and DBU arguments have been added in version 0.27.6.
        """
    def to_trans(self, dbu: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Converts the transformation to another transformation with floating-point input coordinates

        This method is redundant with the conversion constructors. Instead of 'to_trans' use the conversion constructor:

        @code
        dtrans = RBA::DCplxTrans::new(trans, dbu)
        @/code

        This method has been introduced in version 0.25 and was deprecated in version 0.29.
        """
    def to_vtrans(self, dbu: Optional[float] = ...) -> VCplxTrans:
        r"""
        @brief Converts the transformation to another transformation with integer output and floating-point input coordinates

        This method is redundant with the conversion constructors. Instead of 'to_vtrans' use the conversion constructor:

        @code
        vtrans = RBA::VCplxTrans::new(trans, dbu)
        @/code

        This method has been introduced in version 0.25 and was deprecated in version 0.29.
        """
    @overload
    def trans(self, box: Box) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, edge: Edge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, p: Point) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def trans(self, p: Vector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    @overload
    def trans(self, path: Path) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, polygon: Polygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, text: Text) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """

class DBox:
    r"""
    @brief A box class with floating-point coordinates

    This object represents a box (a rectangular shape).

    The definition of the attributes is: p1 is the lower left point, p2 the 
    upper right one. If a box is constructed from two points (or four coordinates), the 
    coordinates are sorted accordingly.

    A box can be empty. An empty box represents no area
    (not even a point). Empty boxes behave neutral with respect to most operations. 
    Empty boxes return true on \empty?.

    A box can be a point or a single
    line. In this case, the area is zero but the box still
    can overlap other boxes for example and it is not empty.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bottom: float
    r"""
    Getter:
    @brief Gets the bottom coordinate of the box

    Setter:
    @brief Sets the bottom coordinate of the box
    """
    left: float
    r"""
    Getter:
    @brief Gets the left coordinate of the box

    Setter:
    @brief Sets the left coordinate of the box
    """
    p1: DPoint
    r"""
    Getter:
    @brief Gets the lower left point of the box

    Setter:
    @brief Sets the lower left point of the box
    """
    p2: DPoint
    r"""
    Getter:
    @brief Gets the upper right point of the box

    Setter:
    @brief Sets the upper right point of the box
    """
    right: float
    r"""
    Getter:
    @brief Gets the right coordinate of the box

    Setter:
    @brief Sets the right coordinate of the box
    """
    top: float
    r"""
    Getter:
    @brief Gets the top coordinate of the box

    Setter:
    @brief Sets the top coordinate of the box
    """
    @classmethod
    def from_ibox(cls, box: Box) -> DBox:
        r"""
        @brief Creates a floating-point coordinate box from an integer coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ibox'.
        """
    @classmethod
    def from_s(cls, s: str) -> DBox:
        r"""
        @brief Creates a box object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DBox:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    @classmethod
    def new(cls, box: Box) -> DBox:
        r"""
        @brief Creates a floating-point coordinate box from an integer coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ibox'.
        """
    @overload
    @classmethod
    def new(cls, left: float, bottom: float, right: float, top: float) -> DBox:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    @classmethod
    def new(cls, lower_left: DPoint, upper_right: DPoint) -> DBox:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    @classmethod
    def new(cls, w: float) -> DBox:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, w: float, h: float) -> DBox:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @classmethod
    def world(cls) -> DBox:
        r"""
        @brief Gets the 'world' box
        The world box is the biggest box that can be represented. So it is basically 'all'. The world box behaves neutral on intersections for example. In other operations such as displacement or transformations, the world box may render unexpected results because of coordinate overflow.

        The world box can be used
        @ul
          @li for comparison ('==', '!=', '<') @/li
          @li in union and intersection ('+' and '&') @/li
          @li in relations (\contains?, \overlaps?, \touches?) @/li
          @li as 'all' argument in region queries @/li
        @/ul

        This method has been introduced in version 0.28.
        """
    @overload
    def __add__(self, box: DBox) -> DBox:
        r"""
        @brief Joins two boxes


        The + operator joins the first box with the one given as 
        the second argument. Joining constructs a box that encloses
        both boxes given. Empty boxes are neutral: they do not
        change another box when joining. Overwrites this box
        with the result.

        @param box The box to join with this box.

        @return The joined box
        """
    @overload
    def __add__(self, point: DPoint) -> DBox:
        r"""
        @brief Joins box with a point


        The + operator joins a point with the box. The resulting box will enclose both the original box and the point.

        @param point The point to join with this box.

        @return The box joined with the point
        """
    def __and__(self, box: DBox) -> DBox:
        r"""
        @brief Returns the intersection of this box with another box


        The intersection of two boxes is the largest
        box common to both boxes. The intersection may be 
        empty if both boxes to not touch. If the boxes do
        not overlap but touch the result may be a single
        line or point with an area of zero. Overwrites this box
        with the result.

        @param box The box to take the intersection with

        @return The intersection box
        """
    def __copy__(self) -> DBox:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DBox:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is equal to the other box
        Returns true, if this box and the given box are equal 
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    def __init__(self, box: Box) -> None:
        r"""
        @brief Creates a floating-point coordinate box from an integer coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ibox'.
        """
    @overload
    def __init__(self, left: float, bottom: float, right: float, top: float) -> None:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    def __init__(self, lower_left: DPoint, upper_right: DPoint) -> None:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    def __init__(self, w: float) -> None:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    def __init__(self, w: float, h: float) -> None:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    def __lt__(self, box: DBox) -> bool:
        r"""
        @brief Returns true if this box is 'less' than another box
        Returns true, if this box is 'less' with respect to first and second point (in this order)
        """
    @overload
    def __mul__(self, box: DBox) -> DBox:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __mul__(self, scale_factor: float) -> DBox:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is not equal to the other box
        Returns true, if this box and the given box are not equal 
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __rmul__(self, box: DBox) -> DBox:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __rmul__(self, scale_factor: float) -> DBox:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __sub__(self, box: DBox) -> DBox:
        r"""
        @brief Subtraction of boxes


        The - operator subtracts the argument box from self.
        This will return the bounding box of the are covered by self, but not by argument box. Subtracting a box from itself will render an empty box. Subtracting another box from self will modify the first box only if the argument box covers one side entirely.

        @param box The box to subtract from this box.

        @return The result box

        This feature has been introduced in version 0.29.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Computes the box area

        Returns the box area or 0 if the box is empty
        """
    def assign(self, other: DBox) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box
        This method is provided for consistency of the shape API is returns the box itself.

        This method has been introduced in version 0.27.
        """
    def center(self) -> DPoint:
        r"""
        @brief Gets the center of the box
        """
    @overload
    def contains(self, point: DPoint) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point is inside the box.
        It also returns true if the point is exactly on the box contour.

        @param p The point to test against.

        @return true if the point is inside the box.
        """
    @overload
    def contains(self, x: float, y: float) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point (x, y) is inside the box.
        It also returns true if the point is exactly on the box contour.

        @return true if the point is inside the box.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DBox:
        r"""
        @brief Creates a copy of self
        """
    def empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the box is empty

        An empty box may be created with the default constructor for example. Such a box is neutral when combining it with other boxes and renders empty boxes if used in box intersections and false in geometrical relationship tests. 
        """
    @overload
    def enlarge(self, d: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return A reference to this box.
        """
    @overload
    def enlarge(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def enlarge(self, enlargement: DVector) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return A reference to this box.
        """
    @overload
    def enlarged(self, d: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return The enlarged box.
        """
    @overload
    def enlarged(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return The enlarged box.
        """
    @overload
    def enlarged(self, enlargement: DVector) -> DBox:
        r"""
        @brief Returns the enlarged box.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not modify this box. Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return The enlarged box.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    def height(self) -> float:
        r"""
        @brief Gets the height of the box
        """
    def inside(self, box: DBox) -> bool:
        r"""
        @brief Tests if this box is inside the argument box


        Returns true, if this box is inside the given box, i.e. the box intersection renders this box
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_point(self) -> bool:
        r"""
        @brief Returns true, if the box is a single point
        """
    @overload
    def move(self, distance: DVector) -> DBox:
        r"""
        @brief Moves the box by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not check for coordinate overflows.

        @param distance The offset to move the box.

        @return A reference to this box.
        """
    @overload
    def move(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def moved(self, distance: DVector) -> DBox:
        r"""
        @brief Returns the box moved by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not modify this box. Does not check for coordinate
        overflows.

        @param distance The offset to move the box.

        @return The moved box.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return The moved box.
        """
    def overlaps(self, box: DBox) -> bool:
        r"""
        @brief Tests if this box overlaps the argument box


        Returns true, if the intersection box of this box with the argument box exists and has a non-vanishing area
        """
    def perimeter(self) -> float:
        r"""
        @brief Returns the perimeter of the box

        This method is equivalent to 2*(width+height). For empty boxes, this method returns 0.

        This method has been introduced in version 0.23.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Box:
        r"""
        @brief Converts the box to an integer coordinate box

        The database unit can be specified to translate the floating-point coordinate box in micron units to an integer-coordinate box in database units. The boxes coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def touches(self, box: DBox) -> bool:
        r"""
        @brief Tests if this box touches the argument box


        Two boxes touch if they overlap or their boundaries share at least one common point. Touching is equivalent to a non-empty intersection ('!(b1 & b2).empty?').
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DBox:
        r"""
        @brief Returns the box transformed with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (a DBox now)
        """
    @overload
    def transformed(self, t: DTrans) -> DBox:
        r"""
        @brief Returns the box transformed with the given simple transformation


        @param t The transformation to apply
        @return The transformed box
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Box:
        r"""
        @brief Transforms the box with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (in this case an integer coordinate box)

        This method has been introduced in version 0.25.
        """
    def width(self) -> float:
        r"""
        @brief Gets the width of the box
        """

class DCellInstArray:
    r"""
    @brief A single or array cell instance in micrometer units
    This object is identical to \CellInstArray, except that it holds coordinates in micron units instead of database units.

    This class has been introduced in version 0.25.
    """
    a: DVector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'a' axis

    Setter:
    @brief Sets the displacement vector for the 'a' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.
    """
    b: DVector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'b' axis

    Setter:
    @brief Sets the displacement vector for the 'b' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.
    """
    cell_index: int
    r"""
    Getter:
    @brief Gets the cell index of the cell instantiated 
    Use \Layout#cell to get the \Cell object from the cell index.
    Setter:
    @brief Sets the index of the cell this instance refers to
    """
    cplx_trans: DCplxTrans
    r"""
    Getter:
    @brief Gets the complex transformation of the first instance in the array
    This method is always applicable, compared to \trans, since simple transformations can be expressed as complex transformations as well.
    Setter:
    @brief Sets the complex transformation of the instance or the first instance in the array
    """
    @property
    def cell(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the cell this instance refers to
        This is a convenience method and equivalent to 'cell_index = cell.cell_index()'. There is no getter for the cell pointer because the \CellInstArray object only knows about cell indexes.

        This convenience method has been introduced in version 0.28.
        """
    na: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'a' axis

    Setter:
    @brief Sets the number of instances in the 'a' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.
    """
    nb: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'b' axis

    Setter:
    @brief Sets the number of instances in the 'b' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.
    """
    trans: DTrans
    r"""
    Getter:
    @brief Gets the transformation of the first instance in the array
    The transformation returned is only valid if the array does not represent a complex transformation array
    Setter:
    @brief Sets the transformation of the instance or the first instance in the array
    """
    @overload
    @classmethod
    def new(cls) -> DCellInstArray:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: DVector) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DCplxTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: DVector) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DCplxTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    def __copy__(self) -> DCellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DCellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    def __init__(self, cell: Cell, disp: DVector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DCplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DTrans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: DVector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, trans: DCplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell_index: int, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    @overload
    def __init__(self, cell_index: int, trans: DTrans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell_index: int, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    def __len__(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def __lt__(self, other: DCellInstArray) -> bool:
        r"""
        @brief Compares two arrays for 'less'
        The comparison provides an arbitrary sorting criterion and not specific sorting order. It is guaranteed that if an array a is less than b, b is not less than a. In addition, it a is not less than b and b is not less than a, then a is equal to b.
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for inequality
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts the array to a string
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the array to a string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DCellInstArray) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def bbox(self, layout: Layout) -> DBox:
        r"""
        @brief Gets the bounding box of the array
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.
        """
    @overload
    def bbox(self, layout: Layout, layer_index: int) -> DBox:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def bbox_per_layer(self, layout: Layout, layer_index: int) -> DBox:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DCellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def each_cplx_trans(self) -> Iterator[DCplxTrans]:
        r"""
        @brief Gets the complex transformations represented by this instance
        For a single instance, this iterator will deliver the single, complex transformation. For array instances, the iterator will deliver each complex transformation of the expanded array.
        This iterator is a generalization of \each_trans for general complex transformations.
        """
    def each_trans(self) -> Iterator[DTrans]:
        r"""
        @brief Gets the simple transformations represented by this instance
        For a single instance, this iterator will deliver the single, simple transformation. For array instances, the iterator will deliver each simple transformation of the expanded array.

        This iterator will only deliver valid transformations if the instance array is not of complex type (see \is_complex?). A more general iterator that delivers the complex transformations is \each_cplx_trans.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> None:
        r"""
        @brief Inverts the array reference

        The inverted array reference describes in which transformations the parent cell is
        seen from the current cell.
        """
    def is_complex(self) -> bool:
        r"""
        @brief Gets a value indicating whether the array is a complex array

        Returns true if the array represents complex instances (that is, with magnification and 
        arbitrary rotation angles).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_regular_array(self) -> bool:
        r"""
        @brief Gets a value indicating whether this instance is a regular array
        """
    def size(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def to_s(self) -> str:
        r"""
        @brief Converts the array to a string
        """
    @overload
    def transform(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the cell instance with the given complex transformation
        """
    @overload
    def transform(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the cell instance with the given transformation
        """
    @overload
    def transformed(self, trans: DCplxTrans) -> DCellInstArray:
        r"""
        @brief Gets the transformed cell instance (complex transformation)
        """
    @overload
    def transformed(self, trans: DTrans) -> DCellInstArray:
        r"""
        @brief Gets the transformed cell instance
        """

class DCplxTrans:
    r"""
    @brief A complex transformation

    A complex transformation provides magnification, mirroring at the x-axis, rotation by an arbitrary
    angle and a displacement. This is also the order, the operations are applied.

    A complex transformation provides a superset of the simple transformation.
    In many applications, a complex transformation computes floating-point coordinates to minimize rounding effects.
    This version can transform floating-point coordinate objects.

    Complex transformations are extensions of the simple transformation classes (\DTrans in that case) and behave similar.

    Transformations can be used to transform points or other objects. Transformations can be combined with the '*' operator to form the transformation which is equivalent to applying the second and then the first. Here is some code:

    @code
    # Create a transformation that applies a magnification of 1.5, a rotation by 90 degree
    # and displacement of 10 in x and 20 units in y direction:
    t = RBA::CplxTrans::new(1.5, 90, false, 10.0, 20.0)
    t.to_s            # r90 *1.5 10,20
    # compute the inverse:
    t.inverted.to_s   # r270 *0.666666667 -13,7
    # Combine with another displacement (applied after that):
    (RBA::CplxTrans::new(5, 5) * t).to_s    # r90 *1.5 15,25
    # Transform a point:
    t.trans(RBA::Point::new(100, 200)).to_s # -290,170
    @/code

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    M0: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "mirrored at the x-axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M135: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "mirrored at the 135 degree axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M45: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "mirrored at the 45 degree axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M90: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "mirrored at the y (90 degree) axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R0: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "unrotated" (unit) transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R180: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "rotated by 180 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R270: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "rotated by 270 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R90: ClassVar[DCplxTrans]
    r"""
    @brief A constant giving "rotated by 90 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    angle: float
    r"""
    Getter:
    @brief Gets the angle

    Note that the simple transformation returns the angle in units of 90 degree. Hence for a simple trans (i.e. \Trans), a rotation angle of 180 degree delivers a value of 2 for the angle attribute. The complex transformation, supporting any rotation angle returns the angle in degree.

    @return The rotation angle this transformation provides in degree units (0..360 deg).

    Setter:
    @brief Sets the angle
    @param a The new angle
    See \angle for a description of that attribute.
    """
    disp: DVector
    r"""
    Getter:
    @brief Gets the displacement

    Setter:
    @brief Sets the displacement
    @param u The new displacement
    """
    mag: float
    r"""
    Getter:
    @brief Gets the magnification

    Setter:
    @brief Sets the magnification
    @param m The new magnification
    """
    mirror: bool
    r"""
    Getter:
    @brief Gets the mirror flag

    If this property is true, the transformation is composed of a mirroring at the x-axis followed by a rotation by the angle given by the \angle property. 
    Setter:
    @brief Sets the mirror flag
    "mirroring" describes a reflection at the x-axis which is included in the transformation prior to rotation.@param m The new mirror flag
    """
    @classmethod
    def from_itrans(cls, trans: CplxTrans, dbu: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input space from integer units to floating-point units. Formally, the DCplxTrans transformation is initialized with 'trans * to_dbu' where 'to_dbu' is the transformation into DBU space, or more precisely 'VCplxTrans(mag=1/dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @classmethod
    def from_s(cls, s: str) -> DCplxTrans:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DCplxTrans:
        r"""
        @brief Creates a unit transformation
        """
    @overload
    @classmethod
    def new(cls, c: DCplxTrans, mag: Optional[float] = ..., u: Optional[DVector] = ...) -> DCplxTrans:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param u The Additional displacement
        """
    @overload
    @classmethod
    def new(cls, c: DCplxTrans, mag: Optional[float] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param x The Additional displacement (x)
        @param y The Additional displacement (y)
        """
    @overload
    @classmethod
    def new(cls, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., u: Optional[DVector] = ...) -> DCplxTrans:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param u The displacement
        """
    @overload
    @classmethod
    def new(cls, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param x The x displacement
        @param y The y displacement
        """
    @overload
    @classmethod
    def new(cls, t: DTrans, mag: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a transformation from a simple transformation and a magnification

        Creates a magnifying transformation from a simple transformation and a magnification.
        """
    @overload
    @classmethod
    def new(cls, trans: CplxTrans, dbu: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input space from integer units to floating-point units. Formally, the DCplxTrans transformation is initialized with 'trans * to_dbu' where 'to_dbu' is the transformation into DBU space, or more precisely 'VCplxTrans(mag=1/dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    @classmethod
    def new(cls, trans: ICplxTrans, dbu: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input and output space from integer units to floating-point units and vice versa. Formally, the DCplxTrans transformation is initialized with 'from_dbu * trans * to_dbu' where 'to_dbu' is the transformation into DBU space, or more precisely 'VCplxTrans(mag=1/dbu)'. 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    @classmethod
    def new(cls, trans: VCplxTrans, dbu: Optional[float] = ...) -> DCplxTrans:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the output space from integer units to floating-point units. Formally, the DCplxTrans transformation is initialized with 'from_dbu * trans' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    @classmethod
    def new(cls, u: DVector) -> DCplxTrans:
        r"""
        @brief Creates a transformation from a displacement

        Creates a transformation with a displacement only.

        This method has been added in version 0.25.
        """
    @overload
    @classmethod
    def new(cls, x: float, y: float) -> DCplxTrans:
        r"""
        @brief Creates a transformation from a x and y displacement

        This constructor will create a transformation with the specified displacement
        but no rotation.

        @param x The x displacement
        @param y The y displacement
        """
    def __copy__(self) -> DCplxTrans:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DCplxTrans:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Tests for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given transformation. This method enables transformations as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates a unit transformation
        """
    @overload
    def __init__(self, c: DCplxTrans, mag: Optional[float] = ..., u: Optional[DVector] = ...) -> None:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param u The Additional displacement
        """
    @overload
    def __init__(self, c: DCplxTrans, mag: Optional[float] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation from another transformation plus a magnification and displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param x The Additional displacement (x)
        @param y The Additional displacement (y)
        """
    @overload
    def __init__(self, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., u: Optional[DVector] = ...) -> None:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param u The displacement
        """
    @overload
    def __init__(self, mag: Optional[float] = ..., rot: Optional[float] = ..., mirrx: Optional[bool] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation using magnification, angle, mirror flag and displacement

        The sequence of operations is: magnification, mirroring at x axis,
        rotation, application of displacement.

        @param mag The magnification
        @param rot The rotation angle in units of degree
        @param mirrx True, if mirrored at x axis
        @param x The x displacement
        @param y The y displacement
        """
    @overload
    def __init__(self, t: DTrans, mag: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation from a simple transformation and a magnification

        Creates a magnifying transformation from a simple transformation and a magnification.
        """
    @overload
    def __init__(self, trans: CplxTrans, dbu: Optional[float] = ...) -> None:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input space from integer units to floating-point units. Formally, the DCplxTrans transformation is initialized with 'trans * to_dbu' where 'to_dbu' is the transformation into DBU space, or more precisely 'VCplxTrans(mag=1/dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    def __init__(self, trans: ICplxTrans, dbu: Optional[float] = ...) -> None:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the input and output space from integer units to floating-point units and vice versa. Formally, the DCplxTrans transformation is initialized with 'from_dbu * trans * to_dbu' where 'to_dbu' is the transformation into DBU space, or more precisely 'VCplxTrans(mag=1/dbu)'. 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    def __init__(self, trans: VCplxTrans, dbu: Optional[float] = ...) -> None:
        r"""
        @brief Creates a floating-point coordinate transformation from another coordinate flavour
        The 'dbu' argument is used to transform the output space from integer units to floating-point units. Formally, the DCplxTrans transformation is initialized with 'from_dbu * trans' where 'from_dbu' is the transformation into micrometer space, or more precisely 'CplxTrans(mag=dbu)'.

        This constructor has been introduced in version 0.25. The 'dbu' argument has been added in version 0.29.
        """
    @overload
    def __init__(self, u: DVector) -> None:
        r"""
        @brief Creates a transformation from a displacement

        Creates a transformation with a displacement only.

        This method has been added in version 0.25.
        """
    @overload
    def __init__(self, x: float, y: float) -> None:
        r"""
        @brief Creates a transformation from a x and y displacement

        This constructor will create a transformation with the specified displacement
        but no rotation.

        @param x The x displacement
        @param y The y displacement
        """
    def __lt__(self, other: DCplxTrans) -> bool:
        r"""
        @brief Provides a 'less' criterion for sorting
        This method is provided to implement a sorting order. The definition of 'less' is opaque and might change in future versions.
        """
    @overload
    def __mul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, d: float) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    @overload
    def __mul__(self, edge: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def __mul__(self, p: DVector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    @overload
    def __mul__(self, path: DPath) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, polygon: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, t: CplxTrans) -> CplxTrans:
        r"""
        @brief Multiplication (concatenation) of transformations

        The * operator returns self*t ("t is applied before this transformation").

        @param t The transformation to apply before
        @return The modified transformation
        """
    @overload
    def __mul__(self, t: DCplxTrans) -> DCplxTrans:
        r"""
        @brief Returns the concatenated transformation

        The * operator returns self*t ("t is applied before this transformation").

        @param t The transformation to apply before
        @return The modified transformation
        """
    @overload
    def __mul__(self, text: DText) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Tests for inequality
        """
    def __repr__(self, lazy: Optional[bool] = ..., dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If 'lazy' is true, some parts are omitted when not required.
        If a DBU is given, the output units will be micrometers.

        The lazy and DBU arguments have been added in version 0.27.6.
        """
    @overload
    def __rmul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, d: float) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    @overload
    def __rmul__(self, edge: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def __rmul__(self, p: DVector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    @overload
    def __rmul__(self, path: DPath) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, polygon: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, text: DText) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    def __str__(self, lazy: Optional[bool] = ..., dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If 'lazy' is true, some parts are omitted when not required.
        If a DBU is given, the output units will be micrometers.

        The lazy and DBU arguments have been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DCplxTrans) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def ctrans(self, d: float) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DCplxTrans:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given transformation. This method enables transformations as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> DCplxTrans:
        r"""
        @brief Inverts the transformation (in place)

        Inverts the transformation and replaces this transformation by its
        inverted one.

        @return The inverted transformation
        """
    def inverted(self) -> DCplxTrans:
        r"""
        @brief Returns the inverted transformation

        Returns the inverted transformation. This method does not modify the transformation.

        @return The inverted transformation
        """
    def is_complex(self) -> bool:
        r"""
        @brief Returns true if the transformation is a complex one

        If this predicate is false, the transformation can safely be converted to a simple transformation.
        Otherwise, this conversion will be lossy.
        The predicate value is equivalent to 'is_mag || !is_ortho'.

        This method has been introduced in version 0.27.5.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mag(self) -> bool:
        r"""
        @brief Tests, if the transformation is a magnifying one

        This is the recommended test for checking if the transformation represents
        a magnification.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Gets the mirror flag

        If this property is true, the transformation is composed of a mirroring at the x-axis followed by a rotation by the angle given by the \angle property. 
        """
    def is_ortho(self) -> bool:
        r"""
        @brief Tests, if the transformation is an orthogonal transformation

        If the rotation is by a multiple of 90 degree, this method will return true.
        """
    def is_unity(self) -> bool:
        r"""
        @brief Tests, whether this is a unit transformation
        """
    def rot(self) -> int:
        r"""
        @brief Returns the respective simple transformation equivalent rotation code if possible

        If this transformation is orthogonal (is_ortho () == true), then this method
        will return the corresponding fixpoint transformation, not taking into account
        magnification and displacement. If the transformation is not orthogonal, the result
        reflects the quadrant the rotation goes into.
        """
    def s_trans(self) -> DTrans:
        r"""
        @brief Extracts the simple transformation part

        The simple transformation part does not reflect magnification or arbitrary angles.
        Rotation angles are rounded down to multiples of 90 degree. Magnification is fixed to 1.0.
        """
    def to_itrans(self, dbu: Optional[float] = ...) -> ICplxTrans:
        r"""
        @brief Converts the transformation to another transformation with integer input and output coordinates

        The database unit can be specified to translate the floating-point coordinate displacement in micron units to an integer-coordinate displacement in database units. The displacement's' coordinates will be divided by the database unit.

        This method is redundant with the conversion constructors. Instead of 'to_itrans' use the conversion constructor:

        @code
        itrans = RBA::ICplxTrans::new(dtrans, dbu)
        @/code

        This method has been introduced in version 0.25 and was deprecated in version 0.29.
        """
    def to_s(self, lazy: Optional[bool] = ..., dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If 'lazy' is true, some parts are omitted when not required.
        If a DBU is given, the output units will be micrometers.

        The lazy and DBU arguments have been added in version 0.27.6.
        """
    def to_trans(self, dbu: Optional[float] = ...) -> CplxTrans:
        r"""
        @brief Converts the transformation to another transformation with integer input coordinates

        This method is redundant with the conversion constructors. Instead of 'to_trans' use the conversion constructor:

        @code
        trans = RBA::CplxTrans::new(dtrans, dbu)
        @/code

        This method has been introduced in version 0.25 and was deprecated in version 0.29.
        """
    def to_vtrans(self, dbu: Optional[float] = ...) -> VCplxTrans:
        r"""
        @brief Converts the transformation to another transformation with integer output coordinates

        The database unit can be specified to translate the floating-point coordinate displacement in micron units to an integer-coordinate displacement in database units. The displacement's' coordinates will be divided by the database unit.

        This method is redundant with the conversion constructors. Instead of 'to_vtrans' use the conversion constructor:

        @code
        vtrans = RBA::VCplxTrans::new(dtrans, dbu)
        @/code

        This method has been introduced in version 0.25 and was deprecated in version 0.29.
        """
    @overload
    def trans(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, edge: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def trans(self, p: DVector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    @overload
    def trans(self, path: DPath) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, polygon: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, text: DText) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """

class DEdge:
    r"""
    @brief An edge class

    An edge is a connection between points, usually participating in a larger context such as a polygon. An edge has a defined direction (from p1 to p2). Edges play a role in the database as parts of polygons and to describe a line through both points.
    The \Edge object is also used inside the boolean processor (\EdgeProcessor).
    Although supported, edges are rarely used as individual database objects.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects like the Edge class.
    """
    p1: DPoint
    r"""
    Getter:
    @brief The first point.

    Setter:
    @brief Sets the first point.
    This method has been added in version 0.23.
    """
    p2: DPoint
    r"""
    Getter:
    @brief The second point.

    Setter:
    @brief Sets the second point.
    This method has been added in version 0.23.
    """
    x1: float
    r"""
    Getter:
    @brief Shortcut for p1.x

    Setter:
    @brief Sets p1.x
    This method has been added in version 0.23.
    """
    x2: float
    r"""
    Getter:
    @brief Shortcut for p2.x

    Setter:
    @brief Sets p2.x
    This method has been added in version 0.23.
    """
    y1: float
    r"""
    Getter:
    @brief Shortcut for p1.y

    Setter:
    @brief Sets p1.y
    This method has been added in version 0.23.
    """
    y2: float
    r"""
    Getter:
    @brief Shortcut for p2.y

    Setter:
    @brief Sets p2.y
    This method has been added in version 0.23.
    """
    @classmethod
    def from_iedge(cls, edge: Edge) -> DEdge:
        r"""
        @brief Creates a floating-point coordinate edge from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge'.
        """
    @classmethod
    def from_s(cls, s: str) -> DEdge:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DEdge:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    @classmethod
    def new(cls, edge: Edge) -> DEdge:
        r"""
        @brief Creates a floating-point coordinate edge from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge'.
        """
    @overload
    @classmethod
    def new(cls, p1: DPoint, p2: DPoint) -> DEdge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    @classmethod
    def new(cls, x1: float, y1: float, x2: float, y2: float) -> DEdge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    @classmethod
    def new_pp(cls, p1: DPoint, p2: DPoint) -> DEdge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @classmethod
    def new_xyxy(cls, x1: float, y1: float, x2: float, y2: float) -> DEdge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __copy__(self) -> DEdge:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DEdge:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, e: object) -> bool:
        r"""
        @brief Equality test
        @param e The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    def __init__(self, edge: Edge) -> None:
        r"""
        @brief Creates a floating-point coordinate edge from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge'.
        """
    @overload
    def __init__(self, p1: DPoint, p2: DPoint) -> None:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    def __init__(self, x1: float, y1: float, x2: float, y2: float) -> None:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __lt__(self, e: DEdge) -> bool:
        r"""
        @brief Less operator
        @param e The object to compare against
        @return True, if the edge is 'less' as the other edge with respect to first and second point
        """
    def __mul__(self, scale_factor: float) -> DEdge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __ne__(self, e: object) -> bool:
        r"""
        @brief Inequality test
        @param e The object to compare against
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __rmul__(self, scale_factor: float) -> DEdge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DEdge) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Return the bounding box of the edge.
        """
    def clipped(self, box: DBox) -> Any:
        r"""
        @brief Returns the edge clipped at the given box

        @param box The clip box.
        @return The clipped edge or nil if the edge does not intersect with the box.

        This method has been introduced in version 0.26.2.
        """
    def clipped_line(self, box: DBox) -> Any:
        r"""
        @brief Returns the line through the edge clipped at the given box

        @param box The clip box.
        @return The part of the line through the box or nil if the line does not intersect with the box.

        In contrast to \clipped, this method will consider the edge extended infinitely (a "line"). The returned edge will be the part of this line going through the box.

        This method has been introduced in version 0.26.2.
        """
    def coincident(self, e: DEdge) -> bool:
        r"""
        @brief Coincidence check.

        Checks whether a edge is coincident with another edge. 
        Coincidence is defined by being parallel and that 
        at least one point of one edge is on the other edge.

        @param e the edge to test with

        @return True if the edges are coincident.
        """
    def contains(self, p: DPoint) -> bool:
        r"""
        @brief Tests whether a point is on an edge.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge.
        """
    def contains_excl(self, p: DPoint) -> bool:
        r"""
        @brief Tests whether a point is on an edge excluding the endpoints.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge but not equal p1 or p2.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def crossed_by(self, e: DEdge) -> bool:
        r"""
        @brief Checks, if the line given by self is crossed by the edge e

        self if considered an infinite line. This predicate renders true if the edge e is cut by this line. In other words: this method returns true if e.p1 is in one semispace of self 
        while e.p2 is in the other or one of them is exactly on self.

        @param e The edge representing the line that the edge must be crossing.
        """
    def crossing_point(self, e: DEdge) -> DPoint:
        r"""
        @brief Returns the crossing point on two edges. 

        This method delivers the point where the given line (self) crosses the edge given by the argument "e". self is considered infinitely long and is required to cut through the edge "e". If self does not cut this line, the result is undefined. See \crossed_by? for a description of the crossing predicate.

        @param e The edge representing the line that self must be crossing.
        @return The point where self crosses the line given by "e".

        This method has been introduced in version 0.19.
        """
    def cut_point(self, e: DEdge) -> Any:
        r"""
        @brief Returns the intersection point of the lines through the two edges.

        This method delivers the intersection point between the lines through the two edges. If the lines are parallel and do not intersect, the result will be nil.
        In contrast to \intersection_point, this method will regard the edges as infinitely extended and intersection is not confined to the edge span.

        @param e The edge to test.
        @return The point where the lines intersect.

        This method has been introduced in version 0.27.1.
        """
    def d(self) -> DVector:
        r"""
        @brief Gets the edge extension as a vector.
        This method is equivalent to p2 - p1.
        This method has been introduced in version 0.26.2.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, p: DPoint) -> float:
        r"""
        @brief Gets the distance of the point from the line through the edge.

        Returns the distance between the edge and the point. The 
        distance is signed which is negative if the point is to the
        "right" of the edge and positive if the point is to the "left".
        The distance is measured by projecting the point onto the
        line through the edge. If the edge is degenerated, the distance
        is not defined.

        This method considers the edge to define an infinite line running through it.
        \distance returns the distance of 'p' to this line.
        A similar method is \euclidian_distance, but the latter regards
        the edge a finite set of points between the endpoints.

        @param p The point to test.

        @return The distance
        """
    def distance_abs(self, p: DPoint) -> float:
        r"""
        @brief Absolute distance between the edge and a point.

        Returns the distance between the edge and the point. 

        @param p The point to test.

        @return The distance
        """
    def dup(self) -> DEdge:
        r"""
        @brief Creates a copy of self
        """
    def dx(self) -> float:
        r"""
        @brief The horizontal extend of the edge.
        """
    def dx_abs(self) -> float:
        r"""
        @brief The absolute value of the horizontal extend of the edge.
        """
    def dy(self) -> float:
        r"""
        @brief The vertical extend of the edge.
        """
    def dy_abs(self) -> float:
        r"""
        @brief The absolute value of the vertical extend of the edge.
        """
    def enlarge(self, p: DVector) -> DEdge:
        r"""
        @brief Enlarges the edge.

        Enlarges the edge by the given distance and returns the 
        enlarged edge. The edge is overwritten.
        Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def enlarged(self, p: DVector) -> DEdge:
        r"""
        @brief Returns the enlarged edge (does not modify self)

        Enlarges the edge by the given offset and returns the 
        enlarged edge. The edge is not modified. Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def euclidian_distance(self, p: DPoint) -> float:
        r"""
        @brief Gets the distance of the point from the the edge.

        Returns the minimum distance of the point to any point on the edge.
        Unlike \distance, the edge is considered a finite set of points between
        the endpoints. The result is also not signed like it is the case for \distance.

        This method has been introduced in version 0.28.14.

        @param p The point to test.

        @return The distance
        """
    def extend(self, d: float) -> DEdge:
        r"""
        @brief Extends the edge (modifies self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extended is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge (self).
        """
    def extended(self, d: float) -> DEdge:
        r"""
        @brief Returns the extended edge (does not modify self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extend is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    def intersect(self, e: DEdge) -> bool:
        r"""
        @brief Intersection test. 

        Returns true if the edges intersect. Two edges intersect if they share at least one point. 
        If the edges coincide, they also intersect.
        If one of the edges is degenerate (both points are identical), that point is required to sit exaclty on the other edge. If both edges are degenerate, their points are required to be identical.

        @param e The edge to test.

        The 'intersects' (with an 's') synonym has been introduced in version 0.28.12.
        """
    def intersection_point(self, e: DEdge) -> Any:
        r"""
        @brief Returns the intersection point of two edges. 

        This method delivers the intersection point. If the edges do not intersect, the result will be nil.

        @param e The edge to test.
        @return The point where the edges intersect.

        This method has been introduced in version 0.19.
        From version 0.26.2, this method will return nil in case of non-intersection.
        """
    def intersects(self, e: DEdge) -> bool:
        r"""
        @brief Intersection test. 

        Returns true if the edges intersect. Two edges intersect if they share at least one point. 
        If the edges coincide, they also intersect.
        If one of the edges is degenerate (both points are identical), that point is required to sit exaclty on the other edge. If both edges are degenerate, their points are required to be identical.

        @param e The edge to test.

        The 'intersects' (with an 's') synonym has been introduced in version 0.28.12.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_degenerate(self) -> bool:
        r"""
        @brief Test for degenerated edge

        An edge is degenerate, if both end and start point are identical.
        """
    def is_parallel(self, e: DEdge) -> bool:
        r"""
        @brief Test for being parallel

        @param e The edge to test against

        @return True if both edges are parallel
        """
    def length(self) -> float:
        r"""
        @brief The length of the edge
        """
    @overload
    def move(self, dx: float, dy: float) -> DEdge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    @overload
    def move(self, p: DVector) -> DEdge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DEdge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: DVector) -> DEdge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    def ortho_length(self) -> float:
        r"""
        @brief The orthogonal length of the edge ("manhattan-length")

        @return The orthogonal length (abs(dx)+abs(dy))
        """
    def shift(self, d: float) -> DEdge:
        r"""
        @brief Shifts the edge (modifies self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shifted is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge (self).
        """
    def shifted(self, d: float) -> DEdge:
        r"""
        @brief Returns the shifted edge (does not modify self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shift is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge.
        """
    def side_of(self, p: DPoint) -> int:
        r"""
        @brief Indicates at which side the point is located relative to the edge.

        Returns 1 if the point is "left" of the edge, 0 if on
        and -1 if the point is "right" of the edge.

        @param p The point to test.

        @return The side value
        """
    def sq_length(self) -> float:
        r"""
        @brief The square of the length of the edge
        """
    def swap_points(self) -> DEdge:
        r"""
        @brief Swap the points of the edge

        This version modifies self. A version that does not modify self is \swapped_points. Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def swapped_points(self) -> DEdge:
        r"""
        @brief Returns an edge in which both points are swapped

        Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Edge:
        r"""
        @brief Converts the edge to an integer coordinate edge

        The database unit can be specified to translate the floating-point coordinate edge in micron units to an integer-coordinate edge in database units. The edges coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    @overload
    def transformed(self, t: DTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Edge:
        r"""
        @brief Transforms the edge with the given complex transformation

        @param t The magnifying transformation to apply
        @return The transformed edge (in this case an integer coordinate edge)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """

class DEdgePair:
    r"""
    @brief An edge pair (a pair of two edges)
    Edge pairs are objects representing two edges or parts of edges. They play a role mainly in the context of DRC functions, where they specify a DRC violation by connecting two edges which violate the condition checked. Within the framework of polygon and edge collections which provide DRC functionality, edges pairs with integer coordinates (\EdgePair type) are used in the form of edge pair collections (\EdgePairs).

    Edge pairs basically consist of two edges, called first and second. If created by a two-layer DRC function, the first edge will correspond to edges from the first layer and the second to edges from the second layer.

    This class has been introduced in version 0.23.
    """
    first: DEdge
    r"""
    Getter:
    @brief Gets the first edge

    Setter:
    @brief Sets the first edge
    """
    second: DEdge
    r"""
    Getter:
    @brief Gets the second edge

    Setter:
    @brief Sets the second edge
    """
    symmetric: bool
    r"""
    Getter:
    @brief Returns a value indicating whether the edge pair is symmetric
    For symmetric edge pairs, the edges are commutable. Specifically, a symmetric edge pair with (e1,e2) is identical to (e2,e1). Symmetric edge pairs are generated by some checks for which there is no directed error marker (width, space, notch, isolated).

    Symmetric edge pairs have been introduced in version 0.27.

    Setter:
    @brief Sets a value indicating whether the edge pair is symmetric
    See \symmetric? for a description of this attribute.

    Symmetric edge pairs have been introduced in version 0.27.
    """
    @classmethod
    def from_s(cls, s: str) -> DEdgePair:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DEdgePair:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    @classmethod
    def new(cls, edge_pair: EdgePair) -> DEdgePair:
        r"""
        @brief Creates a floating-point coordinate edge pair from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge_pair'.
        """
    @overload
    @classmethod
    def new(cls, first: DEdge, second: DEdge, symmetric: Optional[bool] = ...) -> DEdgePair:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __copy__(self) -> DEdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DEdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Equality
        Returns true, if this edge pair and the given one are equal

        This method has been introduced in version 0.25.
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    def __init__(self, edge_pair: EdgePair) -> None:
        r"""
        @brief Creates a floating-point coordinate edge pair from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge_pair'.
        """
    @overload
    def __init__(self, first: DEdge, second: DEdge, symmetric: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __lt__(self, box: DEdgePair) -> bool:
        r"""
        @brief Less operator
        Returns true, if this edge pair is 'less' with respect to first and second edge

        This method has been introduced in version 0.25.
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Inequality
        Returns true, if this edge pair and the given one are not equal

        This method has been introduced in version 0.25.
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Gets the area between the edges of the edge pair

        This attribute has been introduced in version 0.28.
        """
    def assign(self, other: DEdgePair) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Gets the bounding box of the edge pair
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self) -> float:
        r"""
        @brief Gets the distance of the edges in the edge pair

        The distance between the two edges is defined as the minimum distance between any two points on the two edges.

        This attribute has been introduced in version 0.28.14.
        """
    def dup(self) -> DEdgePair:
        r"""
        @brief Creates a copy of self
        """
    def greater(self) -> DEdge:
        r"""
        @brief Gets the 'greater' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'second' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def lesser(self) -> DEdge:
        r"""
        @brief Gets the 'lesser' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'first' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def normalized(self) -> DEdgePair:
        r"""
        @brief Normalizes the edge pair
        This method normalized the edge pair such that when connecting the edges at their 
        start and end points a closed loop is formed which is oriented clockwise. To achieve this, the points of the first and/or first and second edge are swapped. Normalization is a first step recommended before converting an edge pair to a polygon, because that way the polygons won't be self-overlapping and the enlargement parameter is applied properly.
        """
    def perimeter(self) -> float:
        r"""
        @brief Gets the perimeter of the edge pair

        The perimeter is defined as the sum of the lengths of both edges ('active perimeter').

        This attribute has been introduced in version 0.28.
        """
    def polygon(self, e: float) -> DPolygon:
        r"""
        @brief Convert an edge pair to a polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to simple polygons is \simple_polygon which renders a \SimplePolygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def simple_polygon(self, e: float) -> DSimplePolygon:
        r"""
        @brief Convert an edge pair to a simple polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to polygons is \polygon which renders a \Polygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def to_itype(self, dbu: Optional[float] = ...) -> EdgePair:
        r"""
        @brief Converts the edge pair to an integer coordinate edge pair

        The database unit can be specified to translate the floating-point coordinate edge pair in micron units to an integer-coordinate edge pair in database units. The edge pair's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DEdgePair:
        r"""
        @brief Returns the transformed edge pair

        Transforms the edge pair with the given complex transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """
    @overload
    def transformed(self, t: DTrans) -> DEdgePair:
        r"""
        @brief Returns the transformed pair

        Transforms the edge pair with the given transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """
    @overload
    def transformed(self, t: VCplxTrans) -> EdgePair:
        r"""
        @brief Transforms the edge pair with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed edge pair (in this case an integer coordinate edge pair)

        This method has been introduced in version 0.25.
        """

class DPath:
    r"""
    @brief A path class

    A path consists of an sequence of line segments forming the 'spine' of the path and a width. In addition, the starting point can be drawn back by a certain extent (the 'begin extension') and the end point can be pulled forward somewhat (by the 'end extension').

    A path may have round ends for special purposes. In particular, a round-ended path with a single point can represent a circle. Round-ended paths should have being and end extensions equal to half the width. Non-round-ended paths with a single point are allowed but the definition of the resulting shape in not well defined and may differ in other tools.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bgn_ext: float
    r"""
    Getter:
    @brief Get the begin extension

    Setter:
    @brief Set the begin extension
    """
    end_ext: float
    r"""
    Getter:
    @brief Get the end extension

    Setter:
    @brief Set the end extension
    """
    points: int
    r"""
    Getter:
    @brief Get the number of points
    Setter:
    @brief Set the points of the path
    @param p An array of points to assign to the path's spine
    """
    round: bool
    r"""
    Getter:
    @brief Returns true, if the path has round ends

    Setter:
    @brief Set the 'round ends' flag
    A path with round ends show half circles at the ends, instead of square or rectangular ends. Paths with this flag set should use a begin and end extension of half the width (see \bgn_ext and \end_ext). The interpretation of such paths in other tools may differ otherwise.
    """
    width: float
    r"""
    Getter:
    @brief Get the width

    Setter:
    @brief Set the width
    """
    @classmethod
    def from_ipath(cls, path: Path) -> DPath:
        r"""
        @brief Creates a floating-point coordinate path from an integer coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipath'.
        """
    @classmethod
    def from_s(cls, s: str) -> DPath:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DPath:
        r"""
        @brief Default constructor: creates an empty (invalid) path with width 0
        """
    @overload
    @classmethod
    def new(cls, path: Path) -> DPath:
        r"""
        @brief Creates a floating-point coordinate path from an integer coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipath'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], width: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float, round: bool) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    @classmethod
    def new_pw(cls, pts: Sequence[DPoint], width: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @classmethod
    def new_pwx(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @classmethod
    def new_pwxr(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float, round: bool) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    def __copy__(self) -> DPath:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DPath:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Equality test
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates an empty (invalid) path with width 0
        """
    @overload
    def __init__(self, path: Path) -> None:
        r"""
        @brief Creates a floating-point coordinate path from an integer coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipath'.
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], width: float) -> None:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float) -> None:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float, round: bool) -> None:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    def __lt__(self, p: DPath) -> bool:
        r"""
        @brief Less operator
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order
        """
    def __mul__(self, f: float) -> DPath:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Inequality test
        @param p The object to compare against
        """
    def __repr__(self) -> str:
        r"""
        @brief Convert to a string
        """
    def __rmul__(self, f: float) -> DPath:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Convert to a string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Returns the approximate area of the path
        This method returns the approximate value of the area. It is computed from the length times the width. end extensions are taken into account correctly, but not effects of the corner interpolation.
        This method was added in version 0.22.
        """
    def assign(self, other: DPath) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box of the path
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DPath:
        r"""
        @brief Creates a copy of self
        """
    def each_point(self) -> Iterator[DPoint]:
        r"""
        @brief Get the points that make up the path's spine
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_round(self) -> bool:
        r"""
        @brief Returns true, if the path has round ends
        """
    def length(self) -> float:
        r"""
        @brief Returns the length of the path
        the length of the path is determined by summing the lengths of the segments and adding begin and end extensions. For round-ended paths the length of the paths between the tips of the ends.

        This method was added in version 0.23.
        """
    @overload
    def move(self, dx: float, dy: float) -> DPath:
        r"""
        @brief Moves the path.

        Moves the path by the given offset and returns the 
        moved path. The path is overwritten.

        @param dx The x distance to move the path.
        @param dy The y distance to move the path.

        @return The moved path.

        This version has been added in version 0.23.
        """
    @overload
    def move(self, p: DVector) -> DPath:
        r"""
        @brief Moves the path.

        Moves the path by the given offset and returns the 
        moved path. The path is overwritten.

        @param p The distance to move the path.

        @return The moved path.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DPath:
        r"""
        @brief Returns the moved path (does not change self)

        Moves the path by the given offset and returns the 
        moved path. The path is not modified.

        @param dx The x distance to move the path.
        @param dy The y distance to move the path.

        @return The moved path.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: DVector) -> DPath:
        r"""
        @brief Returns the moved path (does not change self)

        Moves the path by the given offset and returns the 
        moved path. The path is not modified.

        @param p The distance to move the path.

        @return The moved path.
        """
    def num_points(self) -> int:
        r"""
        @brief Get the number of points
        """
    def perimeter(self) -> float:
        r"""
        @brief Returns the approximate perimeter of the path
        This method returns the approximate value of the perimeter. It is computed from the length and the width. end extensions are taken into account correctly, but not effects of the corner interpolation.
        This method was added in version 0.24.4.
        """
    def polygon(self) -> DPolygon:
        r"""
        @brief Convert the path to a polygon
        The returned polygon is not guaranteed to be non-self overlapping. This may happen if the path overlaps itself or contains very short segments.
        """
    def round_corners(self, radius: float, npoints: int, accuracy: float) -> DPath:
        r"""
        @brief Creates a new path whose corners are interpolated with circular bends

        @param radius The radius of the bends
        @param npoints The number of points (per full circle) used for interpolating the bends
        @param accuracy The numerical accuracy of the computation

        The accuracy parameter controls the numerical resolution of the approximation process and should be in the order of half the database unit. This accuracy is used for suppressing redundant points and simplification of the resulting path.

        This method has been introduced in version 0.25.
        """
    def simple_polygon(self) -> DSimplePolygon:
        r"""
        @brief Convert the path to a simple polygon
        The returned polygon is not guaranteed to be non-selfoverlapping. This may happen if the path overlaps itself or contains very short segments.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Path:
        r"""
        @brief Converts the path to an integer coordinate path

        The database unit can be specified to translate the floating-point coordinate path in micron units to an integer-coordinate path in database units. The path's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert to a string
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """
    @overload
    def transformed(self, t: DTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Path:
        r"""
        @brief Transforms the polygon with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed path (in this case an integer coordinate path)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """

class DPoint:
    r"""
    @brief A point class with double (floating-point) coordinates
    Points represent a coordinate in the two-dimensional coordinate space of layout. They are not geometrical objects by itself. But they are frequently used in the database API for various purposes. Other than the integer variant (\Point), points with floating-point coordinates can represent fractions of a database unit.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    x: float
    r"""
    Getter:
    @brief Accessor to the x coordinate

    Setter:
    @brief Write accessor to the x coordinate
    """
    y: float
    r"""
    Getter:
    @brief Accessor to the y coordinate

    Setter:
    @brief Write accessor to the y coordinate
    """
    @classmethod
    def from_ipoint(cls, point: Point) -> DPoint:
        r"""
        @brief Creates a floating-point coordinate point from an integer coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoint'.
        """
    @classmethod
    def from_s(cls, s: str) -> DPoint:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DPoint:
        r"""
        @brief Default constructor: creates a point at 0,0
        """
    @overload
    @classmethod
    def new(cls, point: Point) -> DPoint:
        r"""
        @brief Creates a floating-point coordinate point from an integer coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoint'.
        """
    @overload
    @classmethod
    def new(cls, v: DVector) -> DPoint:
        r"""
        @brief Default constructor: creates a point at from an vector
        This constructor is equivalent to computing point(0,0)+v.
        This method has been introduced in version 0.25.
        """
    @overload
    @classmethod
    def new(cls, x: float, y: float) -> DPoint:
        r"""
        @brief Constructor for a point from two coordinate values

        """
    def __add__(self, v: DVector) -> DPoint:
        r"""
        @brief Adds a vector to a point


        Adds vector v to self by adding the coordinates.

        Starting with version 0.25, this method expects a vector argument.
        """
    def __copy__(self) -> DPoint:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DPoint:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Equality test operator

        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given point. This method enables points as hash keys.

        This method has been introduced in version 0.25.
        """
    def __imul__(self, f: float) -> DPoint:
        r"""
        @brief Scaling by some factor


        Scales object in place. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a point at 0,0
        """
    @overload
    def __init__(self, point: Point) -> None:
        r"""
        @brief Creates a floating-point coordinate point from an integer coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoint'.
        """
    @overload
    def __init__(self, v: DVector) -> None:
        r"""
        @brief Default constructor: creates a point at from an vector
        This constructor is equivalent to computing point(0,0)+v.
        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self, x: float, y: float) -> None:
        r"""
        @brief Constructor for a point from two coordinate values

        """
    def __itruediv__(self, d: float) -> DPoint:
        r"""
        @brief Division by some divisor


        Divides the object in place. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def __lt__(self, p: DPoint) -> bool:
        r"""
        @brief "less" comparison operator


        This operator is provided to establish a sorting
        order
        """
    def __mul__(self, f: float) -> DPoint:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Inequality test operator

        """
    def __neg__(self) -> DPoint:
        r"""
        @brief Compute the negative of a point


        Returns a new point with -x, -y.

        This method has been added in version 0.23.
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __rmul__(self, f: float) -> DPoint:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __sub__(self, p: DPoint) -> DVector:
        r"""
        @brief Subtract one point from another


        Subtract point p from self by subtracting the coordinates. This renders a vector.

        Starting with version 0.25, this method renders a vector.
        """
    @overload
    def __sub__(self, v: DVector) -> DPoint:
        r"""
        @brief Subtract one vector from a point


        Subtract vector v from from self by subtracting the coordinates. This renders a point.

        This method has been added in version 0.27.
        """
    def __truediv__(self, d: float) -> DPoint:
        r"""
        @brief Division by some divisor


        Returns the scaled object. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def abs(self) -> float:
        r"""
        @brief The absolute value of the point (Euclidian distance to 0,0)

        The returned value is 'sqrt(x*x+y*y)'.

        This method has been introduced in version 0.23.
        """
    def assign(self, other: DPoint) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, d: DPoint) -> float:
        r"""
        @brief The Euclidian distance to another point


        @param d The other point to compute the distance to.
        """
    def dup(self) -> DPoint:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given point. This method enables points as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def sq_abs(self) -> float:
        r"""
        @brief The square of the absolute value of the point (Euclidian distance to 0,0)

        The returned value is 'x*x+y*y'.

        This method has been introduced in version 0.23.
        """
    def sq_distance(self, d: DPoint) -> float:
        r"""
        @brief The square Euclidian distance to another point


        @param d The other point to compute the distance to.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Point:
        r"""
        @brief Converts the point to an integer coordinate point

        The database unit can be specified to translate the floating-point coordinate point in micron units to an integer-coordinate point in database units. The point's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def to_v(self) -> DVector:
        r"""
        @brief Turns the point into a vector
        This method returns a vector representing the distance from (0,0) to the point.This method has been introduced in version 0.25.
        """

class DPolygon:
    r"""
    @brief A polygon class

    A polygon consists of an outer hull and zero to many
    holes. Each contour consists of several points. The point
    list is normalized such that the leftmost, lowest point is 
    the first one. The orientation is normalized such that
    the orientation of the hull contour is clockwise, while
    the orientation of the holes is counterclockwise.

    It is in no way checked that the contours are not overlapping.
    This must be ensured by the user of the object
    when filling the contours.

    A polygon can be asked for the number of holes using the \holes method. \each_point_hull delivers the points of the hull contour. \each_point_hole delivers the points of a specific hole. \each_edge delivers the edges (point-to-point connections) of both hull and holes. \bbox delivers the bounding box, \area the area and \perimeter the perimeter of the polygon.

    Here's an example of how to create a polygon:

    @code
    hull =  [ RBA::DPoint::new(0, 0),       RBA::DPoint::new(6000, 0), 
              RBA::DPoint::new(6000, 3000), RBA::DPoint::new(0, 3000) ]
    hole1 = [ RBA::DPoint::new(1000, 1000), RBA::DPoint::new(2000, 1000), 
              RBA::DPoint::new(2000, 2000), RBA::DPoint::new(1000, 2000) ]
    hole2 = [ RBA::DPoint::new(3000, 1000), RBA::DPoint::new(4000, 1000), 
              RBA::DPoint::new(4000, 2000), RBA::DPoint::new(3000, 2000) ]
    poly = RBA::DPolygon::new(hull)
    poly.insert_hole(hole1)
    poly.insert_hole(hole2)

    # ask the polygon for some properties
    poly.holes      # -> 2
    poly.area       # -> 16000000.0
    poly.perimeter  # -> 26000.0
    poly.bbox       # -> (0,0;6000,3000)
    @/code

    The \DPolygon class stores coordinates in floating-point format which gives a higher precision for some operations. A class that stores integer coordinates is \Polygon.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    @property
    def hull(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the points of the hull of polygon
        @param p An array of points to assign to the polygon's hull
        The 'assign_hull' variant is provided in analogy to 'assign_hole'.
        """
    @classmethod
    def ellipse(cls, box: DBox, n: int) -> DPolygon:
        r"""
        @brief Creates a simple polygon approximating an ellipse

        @param box The bounding box of the ellipse
        @param n The number of points that will be used to approximate the ellipse

        This method has been introduced in version 0.23.
        """
    @classmethod
    def from_ipoly(cls, polygon: Polygon) -> DPolygon:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipolygon'.
        """
    @classmethod
    def from_s(cls, s: str) -> DPolygon:
        r"""
        @brief Creates a polygon from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DPolygon:
        r"""
        @brief Creates an empty (invalid) polygon
        """
    @overload
    @classmethod
    def new(cls, box: DBox) -> DPolygon:
        r"""
        @brief Creates a polygon from a box

        @param box The box to convert to a polygon
        """
    @overload
    @classmethod
    def new(cls, polygon: Polygon) -> DPolygon:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipolygon'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> DPolygon:
        r"""
        @brief Creates a polygon from a point array for the hull

        @param pts The points forming the polygon hull
        @param raw If true, the point list won't be modified (see \assign_hull)

        The 'raw' argument was added in version 0.24.
        """
    @overload
    @classmethod
    def new(cls, sp: DSimplePolygon) -> DPolygon:
        r"""
        @brief Creates a polygon from a simple polygon
        @param sp The simple polygon that is converted into the polygon
        This method was introduced in version 0.22.
        """
    def __copy__(self) -> DPolygon:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DPolygon:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether the polygons are equal
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates an empty (invalid) polygon
        """
    @overload
    def __init__(self, box: DBox) -> None:
        r"""
        @brief Creates a polygon from a box

        @param box The box to convert to a polygon
        """
    @overload
    def __init__(self, polygon: Polygon) -> None:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipolygon'.
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Creates a polygon from a point array for the hull

        @param pts The points forming the polygon hull
        @param raw If true, the point list won't be modified (see \assign_hull)

        The 'raw' argument was added in version 0.24.
        """
    @overload
    def __init__(self, sp: DSimplePolygon) -> None:
        r"""
        @brief Creates a polygon from a simple polygon
        @param sp The simple polygon that is converted into the polygon
        This method was introduced in version 0.22.
        """
    def __lt__(self, p: DPolygon) -> bool:
        r"""
        @brief Returns a value indicating whether self is less than p
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order
        """
    def __mul__(self, f: float) -> DPolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether the polygons are not equal
        @param p The object to compare against
        """
    def __repr__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def __rmul__(self, f: float) -> DPolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Gets the area of the polygon
        The area is correct only if the polygon is not self-overlapping and the polygon is oriented clockwise.Orientation is ensured automatically in most cases.
        """
    def area2(self) -> float:
        r"""
        @brief Gets the double area of the polygon
        This method is provided because the area for an integer-type polygon is a multiple of 1/2. Hence the double area can be expresses precisely as an integer for these types.

        This method has been introduced in version 0.26.1
        """
    def assign(self, other: DPolygon) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def assign_hole(self, n: int, b: DBox) -> None:
        r"""
        @brief Sets the box as the given hole of the polygon
        @param n The index of the hole to which the points should be assigned
        @param b The box to assign to the polygon's hole
        If the hole index is not valid, this method does nothing.
        This method was introduced in version 0.23.
        """
    @overload
    def assign_hole(self, n: int, p: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the given hole of the polygon
        @param n The index of the hole to which the points should be assigned
        @param p An array of points to assign to the polygon's hole
        @param raw If true, the points won't be compressed (see \assign_hull)
        If the hole index is not valid, this method does nothing.

        This method was introduced in version 0.18.
        The 'raw' argument was added in version 0.24.
        """
    def assign_hull(self, p: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the hull of polygon
        @param p An array of points to assign to the polygon's hull
        @param raw If true, the points won't be compressed

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'assign_hull' variant is provided in analogy to 'assign_hole'.

        The 'raw' argument was added in version 0.24.
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box of the polygon
        The bounding box is the box enclosing all points of the polygon.
        """
    def break_(self, max_vertex_count: int, max_area_ratio: float) -> List[DPolygon]:
        r"""
        @brief Splits the polygon into parts with a maximum vertex count and area ratio
        The area ratio is the ratio between the bounding box area and the polygon area. Higher values mean more 'skinny' polygons.

        This method will split the input polygon into pieces having a maximum of 'max_vertex_count' vertices and an area ratio less than 'max_area_ratio'. 'max_vertex_count' can be zero. In this case the limit is ignored. Also 'max_area_ratio' can be zero, in which case it is ignored as well.

        The method of splitting is unspecified. The algorithm will apply 'split' recursively until the parts satisfy the limits.

        This method has been introduced in version 0.29.
        """
    def compress(self, remove_reflected: bool) -> None:
        r"""
        @brief Compresses the polygon.

        This method removes redundant points from the polygon, such as points being on a line formed by two other points.
        If remove_reflected is true, points are also removed if the two adjacent edges form a spike.

        @param remove_reflected See description of the functionality.

        This method was introduced in version 0.18.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DPolygon:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def each_edge(self) -> Iterator[DEdge]:
        r"""
        @brief Iterates over the edges that make up the polygon

        This iterator will deliver all edges, including those of the holes. Hole edges are oriented counterclockwise while hull edges are oriented clockwise.
        """
    @overload
    def each_edge(self, contour: int) -> Iterator[DEdge]:
        r"""
        @brief Iterates over the edges of one contour of the polygon

        @param contour The contour number (0 for hull, 1 for first hole ...)

        This iterator will deliver all edges of the contour specified by the contour parameter. The hull has contour number 0, the first hole has contour 1 etc.
        Hole edges are oriented counterclockwise while hull edges are oriented clockwise.

        This method was introduced in version 0.24.
        """
    def each_point_hole(self, n: int) -> Iterator[DPoint]:
        r"""
        @brief Iterates over the points that make up the nth hole
        The hole number must be less than the number of holes (see \holes)
        """
    def each_point_hull(self) -> Iterator[DPoint]:
        r"""
        @brief Iterates over the points that make up the hull
        """
    def extract_rad(self) -> List[Any]:
        r"""
        @brief Extracts the corner radii from a rounded polygon

        Attempts to extract the radii of rounded corner polygon. This is essentially the inverse of the \round_corners method. If this method succeeds, if will return an array of four elements: @ul
        @li The polygon with the rounded corners replaced by edgy ones @/li
        @li The radius of the inner corners @/li
        @li The radius of the outer corners @/li
        @li The number of points per full circle @/li
        @/ul

        This method is based on some assumptions and may fail. In this case, an empty array is returned.

        If successful, the following code will more or less render the original polygon and parameters

        @code
        p = ...   # some polygon
        p.round_corners(ri, ro, n)
        (p2, ri2, ro2, n2) = p.extract_rad
        # -> p2 == p, ro2 == ro, ri2 == ri, n2 == n (within some limits)
        @/code

        This method was introduced in version 0.25.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def holes(self) -> int:
        r"""
        @brief Returns the number of holes
        """
    @overload
    def insert_hole(self, b: DBox) -> None:
        r"""
        @brief Inserts a hole from the given box
        @param b The box to insert as a new hole
        This method was introduced in version 0.23.
        """
    @overload
    def insert_hole(self, p: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Inserts a hole with the given points
        @param p An array of points to insert as a new hole
        @param raw If true, the points won't be compressed (see \assign_hull)

        The 'raw' argument was added in version 0.24.
        """
    def inside(self, p: DPoint) -> bool:
        r"""
        @brief Tests, if the given point is inside the polygon
        If the given point is inside or on the edge of the polygon, true is returned. This tests works well only if the polygon is not self-overlapping and oriented clockwise. 
        """
    def is_box(self) -> bool:
        r"""
        @brief Returns true, if the polygon is a simple box.

        A polygon is a box if it is identical to its bounding box.

        @return True if the polygon is a box.

        This method was introduced in version 0.23.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is empty
        """
    def is_halfmanhattan(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is half-manhattan
        Half-manhattan polygons have edges which are multiples of 45 degree. These polygons can be clipped at a rectangle without potential grid snapping.

        This predicate was introduced in version 0.27.
        """
    def is_rectilinear(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is rectilinear
        """
    @overload
    def move(self, p: DVector) -> DPolygon:
        r"""
        @brief Moves the polygon.

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is overwritten.

        @param p The distance to move the polygon.

        @return The moved polygon (self).

        This method has been introduced in version 0.23.
        """
    @overload
    def move(self, x: float, y: float) -> DPolygon:
        r"""
        @brief Moves the polygon.

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is overwritten.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon (self).
        """
    @overload
    def moved(self, p: DVector) -> DPolygon:
        r"""
        @brief Returns the moved polygon (does not modify self)

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is not modified.

        @param p The distance to move the polygon.

        @return The moved polygon.

        This method has been introduced in version 0.23.
        """
    @overload
    def moved(self, x: float, y: float) -> DPolygon:
        r"""
        @brief Returns the moved polygon (does not modify self)

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is not modified.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon.

        This method has been introduced in version 0.23.
        """
    def num_points(self) -> int:
        r"""
        @brief Gets the total number of points (hull plus holes)
        This method was introduced in version 0.18.
        """
    def num_points_hole(self, n: int) -> int:
        r"""
        @brief Gets the number of points of the given hole
        The argument gives the index of the hole of which the number of points are requested. The index must be less than the number of holes (see \holes). 
        """
    def num_points_hull(self) -> int:
        r"""
        @brief Gets the number of points of the hull
        """
    def perimeter(self) -> float:
        r"""
        @brief Gets the perimeter of the polygon
        The perimeter is sum of the lengths of all edges making up the polygon.

        This method has been introduce in version 0.23.
        """
    def point_hole(self, n: int, p: int) -> DPoint:
        r"""
        @brief Gets a specific point of a hole
        @param n The index of the hole to which the points should be assigned
        @param p The index of the point to get
        If the index of the point or of the hole is not valid, a default value is returned.
        This method was introduced in version 0.18.
        """
    def point_hull(self, p: int) -> DPoint:
        r"""
        @brief Gets a specific point of the hull
        @param p The index of the point to get
        If the index of the point is not a valid index, a default value is returned.
        This method was introduced in version 0.18.
        """
    def round_corners(self, rinner: float, router: float, n: int) -> DPolygon:
        r"""
        @brief Rounds the corners of the polygon

        Replaces the corners of the polygon with circle segments.

        @param rinner The circle radius of inner corners (in database units).
        @param router The circle radius of outer corners (in database units).
        @param n The number of points per full circle.

        @return The new polygon.

        This method was introduced in version 0.20 for integer coordinates and in 0.25 for all coordinate types.
        """
    @overload
    def size(self, d: float, mode: Optional[int] = ...) -> None:
        r"""
        @brief Sizes the polygon (biasing)

        Shifts the contour outwards (d>0) or inwards (d<0).
        This method is equivalent to
        @code
        size(d, d, mode)
        @/code

        See \size for a detailed description.

        This method has been introduced in version 0.23.
        """
    @overload
    def size(self, dv: Vector, mode: Optional[int] = ...) -> None:
        r"""
        @brief Sizes the polygon (biasing)

        This method is equivalent to
        @code
        size(dv.x, dv.y, mode)
        @/code

        See \size for a detailed description.

        This version has been introduced in version 0.28.
        """
    @overload
    def size(self, dx: float, dy: float, mode: int) -> None:
        r"""
        @brief Sizes the polygon (biasing)

        Shifts the contour outwards (dx,dy>0) or inwards (dx,dy<0).
        dx is the sizing in x-direction and dy is the sizing in y-direction. The sign of dx and dy should be identical.
        The sizing operation create invalid (self-overlapping, reverse oriented) contours. 

        The mode defines at which bending angle cutoff occurs 
        (0:>0, 1:>45, 2:>90, 3:>135, 4:>approx. 168, other:>approx. 179)

        In order to obtain a proper polygon in the general case, the
        sized polygon must be merged in 'greater than zero' wrap count mode. This is necessary since in the general case,
        sizing can be complicated operation which lets a single polygon fall apart into disjoint pieces for example.
        This can be achieved using the \EdgeProcessor class for example:

        @code
        poly = ... # a RBA::Polygon
        poly.size(-50, 2)
        ep = RBA::EdgeProcessor::new
        # result is an array of RBA::Polygon objects
        result = ep.simple_merge_p2p([ poly ], false, false, 1)
        @/code
        """
    @overload
    def sized(self, d: float, mode: Optional[int] = ...) -> DPolygon:
        r"""
        @brief Sizes the polygon (biasing) without modifying self

        Shifts the contour outwards (d>0) or inwards (d<0).
        This method is equivalent to
        @code
        sized(d, d, mode)
        @/code

        See \size and \sized for a detailed description.
        """
    @overload
    def sized(self, dv: Vector, mode: Optional[int] = ...) -> DPolygon:
        r"""
        @brief Sizes the polygon (biasing) without modifying self

        This method is equivalent to
        @code
        sized(dv.x, dv.y, mode)
        @/code

        See \size and \sized for a detailed description.

        This version has been introduced in version 0.28.
        """
    @overload
    def sized(self, dx: float, dy: float, mode: int) -> DPolygon:
        r"""
        @brief Sizes the polygon (biasing) without modifying self

        This method applies sizing to the polygon but does not modify self. Instead a sized copy is returned.
        See \size for a description of the operation.

        This method has been introduced in version 0.23.
        """
    def sort_holes(self) -> None:
        r"""
        @brief Brings the holes in a specific order
        This function is normalize the hole order so the comparison of two polygons does not depend on the order the holes were inserted. Polygons generated by KLayout's alorithms have their holes sorted.

        This method has been introduced in version 0.28.8.
        """
    def split(self) -> List[DPolygon]:
        r"""
        @brief Splits the polygon into two or more parts
        This method will break the polygon into parts. The exact breaking algorithm is unspecified, the result are smaller polygons of roughly equal number of points and 'less concave' nature. Usually the returned polygon set consists of two polygons, but there can be more. The merged region of the resulting polygons equals the original polygon with the exception of small snapping effects at new vertexes.

        The intended use for this method is a iteratively split polygons until the satisfy some maximum number of points limit.

        This method has been introduced in version 0.25.3.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Polygon:
        r"""
        @brief Converts the polygon to an integer coordinate polygon

        The database unit can be specified to translate the floating-point coordinate polygon in micron units to an integer-coordinate polygon in database units. The polygons coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    @overload
    def touches(self, box: DBox) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given box.
        The box and the polygon touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, edge: DEdge) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given edge.
        The edge and the polygon touch if they overlap or the edge shares at least one point with the polygon's contour.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, polygon: DPolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, simple_polygon: DSimplePolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def transform(self, t: DCplxTrans) -> DPolygon:
        r"""
        @brief Transforms the polygon with a complex transformation (in-place)

        Transforms the polygon with the given complex transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transform(self, t: DTrans) -> DPolygon:
        r"""
        @brief Transforms the polygon (in-place)

        Transforms the polygon with the given transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DPolygon:
        r"""
        @brief Transforms the polygon with a complex transformation

        Transforms the polygon with the given complex transformation.
        Does not modify the polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """
    @overload
    def transformed(self, t: DTrans) -> DPolygon:
        r"""
        @brief Transforms the polygon

        Transforms the polygon with the given transformation.
        Does not modify the polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed polygon.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Polygon:
        r"""
        @brief Transforms the polygon with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed polygon (in this case an integer coordinate polygon)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DPolygon:
        r"""
        @brief Transforms the polygon with a complex transformation

        Transforms the polygon with the given complex transformation.
        Does not modify the polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """

class DSimplePolygon:
    r"""
    @brief A simple polygon class

    A simple polygon consists of an outer hull only. To support polygons with holes, use \DPolygon.
    The contour consists of several points. The point
    list is normalized such that the leftmost, lowest point is 
    the first one. The orientation is normalized such that
    the orientation of the hull contour is clockwise.

    It is in no way checked that the contours are not over-
    lapping. This must be ensured by the user of the object
    when filling the contours.

    The \DSimplePolygon class stores coordinates in floating-point format which gives a higher precision for some operations. A class that stores integer coordinates is \SimplePolygon.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    @property
    def points(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the points of the simple polygon

        @param pts An array of points to assign to the simple polygon

        See the constructor description for details about raw mode.
        """
    @classmethod
    def ellipse(cls, box: DBox, n: int) -> DSimplePolygon:
        r"""
        @brief Creates a simple polygon approximating an ellipse

        @param box The bounding box of the ellipse
        @param n The number of points that will be used to approximate the ellipse

        This method has been introduced in version 0.23.
        """
    @classmethod
    def from_ipoly(cls, polygon: SimplePolygon) -> DSimplePolygon:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon
        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoly'.
        """
    @classmethod
    def from_s(cls, s: str) -> DSimplePolygon:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DSimplePolygon:
        r"""
        @brief Default constructor: creates an empty (invalid) polygon
        """
    @overload
    @classmethod
    def new(cls, box: DBox) -> DSimplePolygon:
        r"""
        @brief Constructor converting a box to a polygon

        @param box The box to convert to a polygon
        """
    @overload
    @classmethod
    def new(cls, polygon: SimplePolygon) -> DSimplePolygon:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon
        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoly'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> DSimplePolygon:
        r"""
        @brief Constructor given the points of the simple polygon

        @param pts The points forming the simple polygon
        @param raw If true, the points are taken as they are (see below)

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'raw' argument has been added in version 0.24.
        """
    def __copy__(self) -> DSimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DSimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether self is equal to p
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates an empty (invalid) polygon
        """
    @overload
    def __init__(self, box: DBox) -> None:
        r"""
        @brief Constructor converting a box to a polygon

        @param box The box to convert to a polygon
        """
    @overload
    def __init__(self, polygon: SimplePolygon) -> None:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon
        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoly'.
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor given the points of the simple polygon

        @param pts The points forming the simple polygon
        @param raw If true, the points are taken as they are (see below)

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'raw' argument has been added in version 0.24.
        """
    def __lt__(self, p: DSimplePolygon) -> bool:
        r"""
        @brief Returns a value indicating whether self is less than p
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order

        This method has been introduced in version 0.25.
        """
    def __mul__(self, f: float) -> DSimplePolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether self is not equal to p
        @param p The object to compare against
        """
    def __repr__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def __rmul__(self, f: float) -> DSimplePolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Gets the area of the polygon
        The area is correct only if the polygon is not self-overlapping and the polygon is oriented clockwise.
        """
    def area2(self) -> float:
        r"""
        @brief Gets the double area of the polygon
        This method is provided because the area for an integer-type polygon is a multiple of 1/2. Hence the double area can be expresses precisely as an integer for these types.

        This method has been introduced in version 0.26.1
        """
    def assign(self, other: DSimplePolygon) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box of the simple polygon
        """
    def break_(self, max_vertex_count: int, max_area_ratio: float) -> List[DSimplePolygon]:
        r"""
        @brief Splits the polygon into parts with a maximum vertex count and area ratio
        The area ratio is the ratio between the bounding box area and the polygon area. Higher values mean more 'skinny' polygons.

        This method will split the input polygon into pieces having a maximum of 'max_vertex_count' vertices and an area ratio less than 'max_area_ratio'. 'max_vertex_count' can be zero. In this case the limit is ignored. Also 'max_area_ratio' can be zero, in which case it is ignored as well.

        The method of splitting is unspecified. The algorithm will apply 'split' recursively until the parts satisfy the limits.

        This method has been introduced in version 0.29.
        """
    def compress(self, remove_reflected: bool) -> None:
        r"""
        @brief Compressed the simple polygon.

        This method removes redundant points from the polygon, such as points being on a line formed by two other points.
        If remove_reflected is true, points are also removed if the two adjacent edges form a spike.

        @param remove_reflected See description of the functionality.

        This method was introduced in version 0.18.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DSimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def each_edge(self) -> Iterator[DEdge]:
        r"""
        @brief Iterates over the edges that make up the simple polygon
        """
    def each_point(self) -> Iterator[DPoint]:
        r"""
        @brief Iterates over the points that make up the simple polygon
        """
    def extract_rad(self) -> List[Any]:
        r"""
        @brief Extracts the corner radii from a rounded polygon

        Attempts to extract the radii of rounded corner polygon. This is essentially the inverse of the \round_corners method. If this method succeeds, if will return an array of four elements: @ul
        @li The polygon with the rounded corners replaced by edgy ones @/li
        @li The radius of the inner corners @/li
        @li The radius of the outer corners @/li
        @li The number of points per full circle @/li
        @/ul

        This method is based on some assumptions and may fail. In this case, an empty array is returned.

        If successful, the following code will more or less render the original polygon and parameters

        @code
        p = ...   # some polygon
        p.round_corners(ri, ro, n)
        (p2, ri2, ro2, n2) = p.extract_rad
        # -> p2 == p, ro2 == ro, ri2 == ri, n2 == n (within some limits)
        @/code

        This method was introduced in version 0.25.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def inside(self, p: DPoint) -> bool:
        r"""
        @brief Gets a value indicating whether the given point is inside the polygon
        If the given point is inside or on the edge the polygon, true is returned. This tests works well only if the polygon is not self-overlapping and oriented clockwise. 
        """
    def is_box(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is a simple box.

        A polygon is a box if it is identical to its bounding box.

        @return True if the polygon is a box.

        This method was introduced in version 0.23.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is empty
        """
    def is_halfmanhattan(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is half-manhattan
        Half-manhattan polygons have edges which are multiples of 45 degree. These polygons can be clipped at a rectangle without potential grid snapping.

        This predicate was introduced in version 0.27.
        """
    def is_rectilinear(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is rectilinear
        """
    @overload
    def move(self, p: DVector) -> DSimplePolygon:
        r"""
        @brief Moves the simple polygon.

        Moves the simple polygon by the given offset and returns the 
        moved simple polygon. The polygon is overwritten.

        @param p The distance to move the simple polygon.

        @return The moved simple polygon.
        """
    @overload
    def move(self, x: float, y: float) -> DSimplePolygon:
        r"""
        @brief Moves the polygon.

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is overwritten.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon (self).
        """
    @overload
    def moved(self, p: DVector) -> DSimplePolygon:
        r"""
        @brief Returns the moved simple polygon

        Moves the simple polygon by the given offset and returns the 
        moved simple polygon. The polygon is not modified.

        @param p The distance to move the simple polygon.

        @return The moved simple polygon.
        """
    @overload
    def moved(self, x: float, y: float) -> DSimplePolygon:
        r"""
        @brief Returns the moved polygon (does not modify self)

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is not modified.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon.

        This method has been introduced in version 0.23.
        """
    def num_points(self) -> int:
        r"""
        @brief Gets the number of points
        """
    def perimeter(self) -> float:
        r"""
        @brief Gets the perimeter of the polygon
        The perimeter is sum of the lengths of all edges making up the polygon.
        """
    def point(self, p: int) -> DPoint:
        r"""
        @brief Gets a specific point of the contour@param p The index of the point to get
        If the index of the point is not a valid index, a default value is returned.
        This method was introduced in version 0.18.
        """
    def round_corners(self, rinner: float, router: float, n: int) -> DSimplePolygon:
        r"""
        @brief Rounds the corners of the polygon

        Replaces the corners of the polygon with circle segments.

        @param rinner The circle radius of inner corners (in database units).
        @param router The circle radius of outer corners (in database units).
        @param n The number of points per full circle.

        @return The new polygon.

        This method was introduced in version 0.22 for integer coordinates and in 0.25 for all coordinate types.
        """
    def set_points(self, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the simple polygon

        @param pts An array of points to assign to the simple polygon
        @param raw If true, the points are taken as they are

        See the constructor description for details about raw mode.

        This method has been added in version 0.24.
        """
    def split(self) -> List[DSimplePolygon]:
        r"""
        @brief Splits the polygon into two or more parts
        This method will break the polygon into parts. The exact breaking algorithm is unspecified, the result are smaller polygons of roughly equal number of points and 'less concave' nature. Usually the returned polygon set consists of two polygons, but there can be more. The merged region of the resulting polygons equals the original polygon with the exception of small snapping effects at new vertexes.

        The intended use for this method is a iteratively split polygons until the satisfy some maximum number of points limit.

        This method has been introduced in version 0.25.3.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> SimplePolygon:
        r"""
        @brief Converts the polygon to an integer coordinate polygon
        The database unit can be specified to translate the floating-point coordinate polygon in micron units to an integer-coordinate polygon in database units. The polygon's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    @overload
    def touches(self, box: DBox) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given box.
        The box and the polygon touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, edge: DEdge) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given edge.
        The edge and the polygon touch if they overlap or the edge shares at least one point with the polygon's contour.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, polygon: DPolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, simple_polygon: DSimplePolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def transform(self, t: DCplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon with a complex transformation (in-place)

        Transforms the simple polygon with the given complex transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transform(self, t: DTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon (in-place)

        Transforms the simple polygon with the given transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """
    @overload
    def transformed(self, t: DTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> SimplePolygon:
        r"""
        @brief Transforms the polygon with the given complex transformation

        @param t The magnifying transformation to apply
        @return The transformed polygon (in this case an integer coordinate polygon)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """

class DText:
    r"""
    @brief A text object

    A text object has a point (location), a text, a text transformation,
    a text size and a font id. Text size and font id are provided to be
    be able to render the text correctly.
    Text objects are used as labels (i.e. for pins) or to indicate a particular position.

    The \DText class uses floating-point coordinates. A class that operates with integer coordinates is \Text.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    HAlignCenter: ClassVar[HAlign]
    r"""
    @brief Centered horizontal alignment
    """
    HAlignLeft: ClassVar[HAlign]
    r"""
    @brief Left horizontal alignment
    """
    HAlignRight: ClassVar[HAlign]
    r"""
    @brief Right horizontal alignment
    """
    NoHAlign: ClassVar[HAlign]
    r"""
    @brief Undefined horizontal alignment
    """
    NoVAlign: ClassVar[VAlign]
    r"""
    @brief Undefined vertical alignment
    """
    VAlignBottom: ClassVar[VAlign]
    r"""
    @brief Bottom vertical alignment
    """
    VAlignCenter: ClassVar[VAlign]
    r"""
    @brief Centered vertical alignment
    """
    VAlignTop: ClassVar[VAlign]
    r"""
    @brief Top vertical alignment
    """
    font: int
    r"""
    Getter:
    @brief Gets the font number
    See \font= for a description of this property.
    Setter:
    @brief Sets the font number
    The font number does not play a role for KLayout. This property is provided for compatibility with other systems which allow using different fonts for the text objects.
    """
    halign: HAlign
    r"""
    Getter:
    @brief Gets the horizontal alignment

    See \halign= for a description of this property.

    Setter:
    @brief Sets the horizontal alignment

    This is the version accepting integer values. It's provided for backward compatibility.
    """
    size: float
    r"""
    Getter:
    @brief Gets the text height

    Setter:
    @brief Sets the text height of this object
    """
    string: str
    r"""
    Getter:
    @brief Get the text string

    Setter:
    @brief Assign a text string to this object
    """
    trans: DTrans
    r"""
    Getter:
    @brief Gets the transformation

    Setter:
    @brief Assign a transformation (text position and orientation) to this object
    """
    valign: VAlign
    r"""
    Getter:
    @brief Gets the vertical alignment

    See \valign= for a description of this property.

    Setter:
    @brief Sets the vertical alignment

    This is the version accepting integer values. It's provided for backward compatibility.
    """
    x: float
    r"""
    Getter:
    @brief Gets the x location of the text

    This method has been introduced in version 0.23.

    Setter:
    @brief Sets the x location of the text

    This method has been introduced in version 0.23.
    """
    y: float
    r"""
    Getter:
    @brief Gets the y location of the text

    This method has been introduced in version 0.23.

    Setter:
    @brief Sets the y location of the text

    This method has been introduced in version 0.23.
    """
    @classmethod
    def from_s(cls, s: str) -> DText:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DText:
        r"""
        @brief Default constructor

        Creates a text with unit transformation and empty text.
        """
    @overload
    @classmethod
    def new(cls, Text: Text) -> DText:
        r"""
        @brief Creates a floating-point coordinate text from an integer coordinate text

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_itext'.
        """
    @overload
    @classmethod
    def new(cls, string: str, trans: DTrans) -> DText:
        r"""
        @brief Constructor with string and transformation


        A string and a transformation is provided to this constructor. The transformation specifies the location and orientation of the text object.
        """
    @overload
    @classmethod
    def new(cls, string: str, trans: DTrans, height: float, font: int) -> DText:
        r"""
        @brief Constructor with string, transformation, text height and font


        A string and a transformation is provided to this constructor. The transformation specifies the location and orientation of the text object. In addition, the text height and font can be specified.
        """
    @overload
    @classmethod
    def new(cls, string: str, x: float, y: float) -> DText:
        r"""
        @brief Constructor with string and location


        A string and a location is provided to this constructor. The location is specifies as a pair of x and y coordinates.

        This method has been introduced in version 0.23.
        """
    def __copy__(self) -> DText:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DText:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, text: object) -> bool:
        r"""
        @brief Equality


        Return true, if this text object and the given text are equal 
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given text object. This method enables texts as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        Creates a text with unit transformation and empty text.
        """
    @overload
    def __init__(self, Text: Text) -> None:
        r"""
        @brief Creates a floating-point coordinate text from an integer coordinate text

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_itext'.
        """
    @overload
    def __init__(self, string: str, trans: DTrans) -> None:
        r"""
        @brief Constructor with string and transformation


        A string and a transformation is provided to this constructor. The transformation specifies the location and orientation of the text object.
        """
    @overload
    def __init__(self, string: str, trans: DTrans, height: float, font: int) -> None:
        r"""
        @brief Constructor with string, transformation, text height and font


        A string and a transformation is provided to this constructor. The transformation specifies the location and orientation of the text object. In addition, the text height and font can be specified.
        """
    @overload
    def __init__(self, string: str, x: float, y: float) -> None:
        r"""
        @brief Constructor with string and location


        A string and a location is provided to this constructor. The location is specifies as a pair of x and y coordinates.

        This method has been introduced in version 0.23.
        """
    def __lt__(self, t: DText) -> bool:
        r"""
        @brief Less operator
        @param t The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order
        """
    def __ne__(self, text: object) -> bool:
        r"""
        @brief Inequality


        Return true, if this text object and the given text are not equal 
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Converts the object to a string.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Converts the object to a string.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DText) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Gets the bounding box of the text
        The bounding box of the text is a single point - the location of the text. Both points of the box are identical.

        This method has been added in version 0.28.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DText:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given text object. This method enables texts as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    @overload
    def move(self, distance: DVector) -> DText:
        r"""
        @brief Moves the text by a certain distance (modifies self)


        Moves the text by a given offset and returns the moved
        text. Does not check for coordinate overflows.

        @param p The offset to move the text.

        @return A reference to this text object
        """
    @overload
    def move(self, dx: float, dy: float) -> DText:
        r"""
        @brief Moves the text by a certain distance (modifies self)


        Moves the text by a given distance in x and y direction and returns the moved
        text. Does not check for coordinate overflows.

        @param dx The x distance to move the text.
        @param dy The y distance to move the text.

        @return A reference to this text object

        This method was introduced in version 0.23.
        """
    @overload
    def moved(self, distance: DVector) -> DText:
        r"""
        @brief Returns the text moved by a certain distance (does not modify self)


        Moves the text by a given offset and returns the moved
        text. Does not modify *this. Does not check for coordinate
        overflows.

        @param p The offset to move the text.

        @return The moved text.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DText:
        r"""
        @brief Returns the text moved by a certain distance (does not modify self)


        Moves the text by a given offset and returns the moved
        text. Does not modify *this. Does not check for coordinate
        overflows.

        @param dx The x distance to move the text.
        @param dy The y distance to move the text.

        @return The moved text.

        This method was introduced in version 0.23.
        """
    def position(self) -> DPoint:
        r"""
        @brief Gets the position of the text

        This convenience method has been added in version 0.28.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Text:
        r"""
        @brief Converts the text to an integer coordinate text

        The database unit can be specified to translate the floating-point coordinate Text in micron units to an integer-coordinate text in database units. The text's coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Converts the object to a string.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DText:
        r"""
        @brief Transforms the text with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed text (a DText now)
        """
    @overload
    def transformed(self, t: DTrans) -> DText:
        r"""
        @brief Transforms the text with the given simple transformation


        @param t The transformation to apply
        @return The transformed text
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Text:
        r"""
        @brief Transforms the text with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed text (in this case an integer coordinate text)

        This method has been introduced in version 0.25.
        """

class DTrans:
    r"""
    @brief A simple transformation

    Simple transformations only provide rotations about angles which a multiples of 90 degree.
    Together with the mirror options, this results in 8 distinct orientations (fixpoint transformations).
    These can be combined with a displacement which is applied after the rotation/mirror.
    This version acts on floating-point coordinates. A version for integer coordinates is \Trans.

    Here are some examples for using the DTrans class:

    @code
    t = RBA::DTrans::new(0, 100)  # displacement by 100 DBU in y direction
    # the inverse: -> "r0 0,-100"
    t.inverted.to_s
    # concatenation: -> "r90 -100,0"
    (RBA::DTrans::new(RBA::DTrans::R90) * t).to_s
    # apply to a point: -> "0,100"
    RBA::DTrans::new(RBA::DTrans::R90).trans(RBA::DPoint::new(100, 0))
    @/code

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    M0: ClassVar[DTrans]
    r"""
    @brief A constant giving "mirrored at the x-axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M135: ClassVar[DTrans]
    r"""
    @brief A constant giving "mirrored at the 135 degree axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M45: ClassVar[DTrans]
    r"""
    @brief A constant giving "mirrored at the 45 degree axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    M90: ClassVar[DTrans]
    r"""
    @brief A constant giving "mirrored at the y (90 degree) axis" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R0: ClassVar[DTrans]
    r"""
    @brief A constant giving "unrotated" (unit) transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R180: ClassVar[DTrans]
    r"""
    @brief A constant giving "rotated by 180 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R270: ClassVar[DTrans]
    r"""
    @brief A constant giving "rotated by 270 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    R90: ClassVar[DTrans]
    r"""
    @brief A constant giving "rotated by 90 degree counterclockwise" transformation
    The previous integer constant has been turned into a transformation in version 0.25.
    """
    angle: int
    r"""
    Getter:
    @brief Gets the angle in units of 90 degree

    This value delivers the rotation component. In addition, a mirroring at the x axis may be applied before if the \is_mirror? property is true. 
    Setter:
    @brief Sets the angle in units of 90 degree
    @param a The new angle

    This method was introduced in version 0.20.
    """
    disp: DVector
    r"""
    Getter:
    @brief Gets to the displacement vector

    Staring with version 0.25 the displacement type is a vector.
    Setter:
    @brief Sets the displacement
    @param u The new displacement

    This method was introduced in version 0.20.
    Staring with version 0.25 the displacement type is a vector.
    """
    mirror: bool
    r"""
    Getter:
    @brief Gets the mirror flag

    If this property is true, the transformation is composed of a mirroring at the x-axis followed by a rotation by the angle given by the \angle property. 
    Setter:
    @brief Sets the mirror flag
    "mirroring" describes a reflection at the x-axis which is included in the transformation prior to rotation.@param m The new mirror flag

    This method was introduced in version 0.20.
    """
    rot: int
    r"""
    Getter:
    @brief Gets the angle/mirror code

    The angle/mirror code is one of the constants R0, R90, R180, R270, M0, M45, M90 and M135. rx is the rotation by an angle of x counter clockwise. mx is the mirroring at the axis given by the angle x (to the x-axis). 
    Setter:
    @brief Sets the angle/mirror code
    @param r The new angle/rotation code (see \rot property)

    This method was introduced in version 0.20.
    """
    @classmethod
    def from_itrans(cls, trans: Trans) -> DTrans:
        r"""
        @brief Creates a floating-point coordinate transformation from an integer coordinate transformation

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_itrans'.
        """
    @classmethod
    def from_s(cls, s: str) -> DTrans:
        r"""
        @brief Creates a transformation from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DTrans:
        r"""
        @brief Creates a unit transformation
        """
    @overload
    @classmethod
    def new(cls, c: DTrans, u: Optional[DVector] = ...) -> DTrans:
        r"""
        @brief Creates a transformation from another transformation plus a displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param u The Additional displacement
        """
    @overload
    @classmethod
    def new(cls, c: DTrans, x: Optional[float] = ..., y: Optional[float] = ...) -> DTrans:
        r"""
        @brief Creates a transformation from another transformation plus a displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param x The Additional displacement (x)
        @param y The Additional displacement (y)
        """
    @overload
    @classmethod
    def new(cls, rot: Optional[int] = ..., mirrx: Optional[bool] = ..., u: Optional[DVector] = ...) -> DTrans:
        r"""
        @brief Creates a transformation using angle and mirror flag

        The sequence of operations is: mirroring at x axis,
        rotation, application of displacement.

        @param rot The rotation in units of 90 degree
        @param mirrx True, if mirrored at x axis
        @param u The displacement
        """
    @overload
    @classmethod
    def new(cls, rot: Optional[int] = ..., mirrx: Optional[bool] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> DTrans:
        r"""
        @brief Creates a transformation using angle and mirror flag and two coordinate values for displacement

        The sequence of operations is: mirroring at x axis,
        rotation, application of displacement.

        @param rot The rotation in units of 90 degree
        @param mirrx True, if mirrored at x axis
        @param x The horizontal displacement
        @param y The vertical displacement
        """
    @overload
    @classmethod
    def new(cls, trans: Trans) -> DTrans:
        r"""
        @brief Creates a floating-point coordinate transformation from an integer coordinate transformation

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_itrans'.
        """
    @overload
    @classmethod
    def new(cls, u: DVector) -> DTrans:
        r"""
        @brief Creates a transformation using a displacement only

        @param u The displacement
        """
    @overload
    @classmethod
    def new(cls, x: float, y: float) -> DTrans:
        r"""
        @brief Creates a transformation using a displacement given as two coordinates

        @param x The horizontal displacement
        @param y The vertical displacement
        """
    def __copy__(self) -> DTrans:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DTrans:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Tests for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given transformation. This method enables transformations as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates a unit transformation
        """
    @overload
    def __init__(self, c: DTrans, u: Optional[DVector] = ...) -> None:
        r"""
        @brief Creates a transformation from another transformation plus a displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param u The Additional displacement
        """
    @overload
    def __init__(self, c: DTrans, x: Optional[float] = ..., y: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation from another transformation plus a displacement

        Creates a new transformation from a existing transformation. This constructor is provided for creating duplicates and backward compatibility since the constants are transformations now. It will copy the original transformation and add the given displacement.

        This variant has been introduced in version 0.25.

        @param c The original transformation
        @param x The Additional displacement (x)
        @param y The Additional displacement (y)
        """
    @overload
    def __init__(self, rot: Optional[int] = ..., mirrx: Optional[bool] = ..., u: Optional[DVector] = ...) -> None:
        r"""
        @brief Creates a transformation using angle and mirror flag

        The sequence of operations is: mirroring at x axis,
        rotation, application of displacement.

        @param rot The rotation in units of 90 degree
        @param mirrx True, if mirrored at x axis
        @param u The displacement
        """
    @overload
    def __init__(self, rot: Optional[int] = ..., mirrx: Optional[bool] = ..., x: Optional[float] = ..., y: Optional[float] = ...) -> None:
        r"""
        @brief Creates a transformation using angle and mirror flag and two coordinate values for displacement

        The sequence of operations is: mirroring at x axis,
        rotation, application of displacement.

        @param rot The rotation in units of 90 degree
        @param mirrx True, if mirrored at x axis
        @param x The horizontal displacement
        @param y The vertical displacement
        """
    @overload
    def __init__(self, trans: Trans) -> None:
        r"""
        @brief Creates a floating-point coordinate transformation from an integer coordinate transformation

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_itrans'.
        """
    @overload
    def __init__(self, u: DVector) -> None:
        r"""
        @brief Creates a transformation using a displacement only

        @param u The displacement
        """
    @overload
    def __init__(self, x: float, y: float) -> None:
        r"""
        @brief Creates a transformation using a displacement given as two coordinates

        @param x The horizontal displacement
        @param y The vertical displacement
        """
    def __lt__(self, other: DTrans) -> bool:
        r"""
        @brief Provides a 'less' criterion for sorting
        This method is provided to implement a sorting order. The definition of 'less' is opaque and might change in future versions.
        """
    @overload
    def __mul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, d: float) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    @overload
    def __mul__(self, edge: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def __mul__(self, path: DPath) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, polygon: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, t: DTrans) -> DTrans:
        r"""
        @brief Returns the concatenated transformation

        The * operator returns self*t ("t is applied before this transformation").

        @param t The transformation to apply before
        @return The modified transformation
        """
    @overload
    def __mul__(self, text: DText) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __mul__(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Tests for inequality
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __rmul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, d: float) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    @overload
    def __rmul__(self, edge: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def __rmul__(self, path: DPath) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, polygon: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, text: DText) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def __rmul__(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DTrans) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def ctrans(self, d: float) -> float:
        r"""
        @brief Transforms a distance

        The "ctrans" method transforms the given distance.
        e = t(d). For the simple transformations, there
        is no magnification and no modification of the distance
        therefore.

        @param d The distance to transform
        @return The transformed distance

        The product '*' has been added as a synonym in version 0.28.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DTrans:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given transformation. This method enables transformations as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> DTrans:
        r"""
        @brief Inverts the transformation (in place)

        Inverts the transformation and replaces this object by the
        inverted one.

        @return The inverted transformation
        """
    def inverted(self) -> DTrans:
        r"""
        @brief Returns the inverted transformation
        Returns the inverted transformation

        @return The inverted transformation
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Gets the mirror flag

        If this property is true, the transformation is composed of a mirroring at the x-axis followed by a rotation by the angle given by the \angle property. 
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Trans:
        r"""
        @brief Converts the transformation to an integer coordinate transformation

        The database unit can be specified to translate the floating-point coordinate transformation in micron units to an integer-coordinate transformation in database units. The transformation's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def trans(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box

        't*box' or 't.trans(box)' is equivalent to box.transformed(t).

        @param box The box to transform
        @return The transformed box

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, edge: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge

        't*edge' or 't.trans(edge)' is equivalent to edge.transformed(t).

        @param edge The edge to transform
        @return The transformed edge

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point

        The "trans" method or the * operator transforms the given point.
        q = t(p)

        The * operator has been introduced in version 0.25.

        @param p The point to transform
        @return The transformed point
        """
    @overload
    def trans(self, path: DPath) -> DPath:
        r"""
        @brief Transforms a path

        't*path' or 't.trans(path)' is equivalent to path.transformed(t).

        @param path The path to transform
        @return The transformed path

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, polygon: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon

        't*polygon' or 't.trans(polygon)' is equivalent to polygon.transformed(t).

        @param polygon The polygon to transform
        @return The transformed polygon

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, text: DText) -> DText:
        r"""
        @brief Transforms a text

        't*text' or 't.trans(text)' is equivalent to text.transformed(t).

        @param text The text to transform
        @return The transformed text

        This convenience method has been introduced in version 0.25.
        """
    @overload
    def trans(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector

        The "trans" method or the * operator transforms the given vector.
        w = t(v)

        Vector transformation has been introduced in version 0.25.

        @param v The vector to transform
        @return The transformed vector
        """

class DVector:
    r"""
    @brief A vector class with double (floating-point) coordinates
    A vector is a distance in cartesian, 2 dimensional space. A vector is given by two coordinates (x and y) and represents the distance between two points. Being the distance, transformations act differently on vectors: the displacement is not applied. 
    Vectors are not geometrical objects by itself. But they are frequently used in the database API for various purposes. Other than the integer variant (\Vector), points with floating-point coordinates can represent fractions of a database unit or vectors in physical (micron) units.

    This class has been introduced in version 0.25.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    x: float
    r"""
    Getter:
    @brief Accessor to the x coordinate

    Setter:
    @brief Write accessor to the x coordinate
    """
    y: float
    r"""
    Getter:
    @brief Accessor to the y coordinate

    Setter:
    @brief Write accessor to the y coordinate
    """
    @classmethod
    def from_s(cls, s: str) -> DVector:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)
        """
    @overload
    @classmethod
    def new(cls) -> DVector:
        r"""
        @brief Default constructor: creates a null vector with coordinates (0,0)
        """
    @overload
    @classmethod
    def new(cls, p: DPoint) -> DVector:
        r"""
        @brief Default constructor: creates a vector from a point
        This constructor is equivalent to computing p-point(0,0).
        This method has been introduced in version 0.25.
        """
    @overload
    @classmethod
    def new(cls, vector: Vector) -> DVector:
        r"""
        @brief Creates a floating-point coordinate vector from an integer coordinate vector
        """
    @overload
    @classmethod
    def new(cls, x: float, y: float) -> DVector:
        r"""
        @brief Constructor for a vector from two coordinate values

        """
    @overload
    def __add__(self, p: DPoint) -> DPoint:
        r"""
        @brief Adds a vector and a point


        Returns the point p shifted by the vector.
        """
    @overload
    def __add__(self, v: DVector) -> DVector:
        r"""
        @brief Adds two vectors


        Adds vector v to self by adding the coordinates.
        """
    def __copy__(self) -> DVector:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DVector:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, v: object) -> bool:
        r"""
        @brief Equality test operator

        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given vector. This method enables vectors as hash keys.

        This method has been introduced in version 0.25.
        """
    def __imul__(self, f: float) -> DVector:
        r"""
        @brief Scaling by some factor


        Scales object in place. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a null vector with coordinates (0,0)
        """
    @overload
    def __init__(self, p: DPoint) -> None:
        r"""
        @brief Default constructor: creates a vector from a point
        This constructor is equivalent to computing p-point(0,0).
        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self, vector: Vector) -> None:
        r"""
        @brief Creates a floating-point coordinate vector from an integer coordinate vector
        """
    @overload
    def __init__(self, x: float, y: float) -> None:
        r"""
        @brief Constructor for a vector from two coordinate values

        """
    def __itruediv__(self, d: float) -> DVector:
        r"""
        @brief Division by some divisor


        Divides the object in place. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def __lt__(self, v: DVector) -> bool:
        r"""
        @brief "less" comparison operator


        This operator is provided to establish a sorting
        order
        """
    @overload
    def __mul__(self, f: float) -> DVector:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    @overload
    def __mul__(self, v: DVector) -> float:
        r"""
        @brief Computes the scalar product between self and the given vector


        The scalar product of a and b is defined as: vp = ax*bx+ay*by.
        """
    def __ne__(self, v: object) -> bool:
        r"""
        @brief Inequality test operator

        """
    def __neg__(self) -> DVector:
        r"""
        @brief Compute the negative of a vector


        Returns a new vector with -x,-y.
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __rmul__(self, f: float) -> DVector:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    @overload
    def __rmul__(self, v: DVector) -> float:
        r"""
        @brief Computes the scalar product between self and the given vector


        The scalar product of a and b is defined as: vp = ax*bx+ay*by.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __sub__(self, v: DVector) -> DVector:
        r"""
        @brief Subtract two vectors


        Subtract vector v from self by subtracting the coordinates.
        """
    def __truediv__(self, d: float) -> DVector:
        r"""
        @brief Division by some divisor


        Returns the scaled object. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def abs(self) -> float:
        r"""
        @brief Returns the length of the vector
        'abs' is an alias provided for compatibility with the former point type.
        """
    def assign(self, other: DVector) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DVector:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given vector. This method enables vectors as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def length(self) -> float:
        r"""
        @brief Returns the length of the vector
        'abs' is an alias provided for compatibility with the former point type.
        """
    def sprod(self, v: DVector) -> float:
        r"""
        @brief Computes the scalar product between self and the given vector


        The scalar product of a and b is defined as: vp = ax*bx+ay*by.
        """
    def sprod_sign(self, v: DVector) -> int:
        r"""
        @brief Computes the scalar product between self and the given vector and returns a value indicating the sign of the product


        @return 1 if the scalar product is positive, 0 if it is zero and -1 if it is negative.
        """
    def sq_abs(self) -> float:
        r"""
        @brief The square length of the vector
        'sq_abs' is an alias provided for compatibility with the former point type.
        """
    def sq_length(self) -> float:
        r"""
        @brief The square length of the vector
        'sq_abs' is an alias provided for compatibility with the former point type.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Vector:
        r"""
        @brief Converts the point to an integer coordinate point

        The database unit can be specified to translate the floating-point coordinate vector in micron units to an integer-coordinate vector in database units. The vector's' coordinates will be divided by the database unit.
        """
    def to_p(self) -> DPoint:
        r"""
        @brief Turns the vector into a point
        This method returns the point resulting from adding the vector to (0,0).
        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def vprod(self, v: DVector) -> float:
        r"""
        @brief Computes the vector product between self and the given vector


        The vector product of a and b is defined as: vp = ax*by-ay*bx.
        """
    def vprod_sign(self, v: DVector) -> int:
        r"""
        @brief Computes the vector product between self and the given vector and returns a value indicating the sign of the product


        @return 1 if the vector product is positive, 0 if it is zero and -1 if it is negative.
        """

class DeepShapeStore:
    r"""
    @brief An opaque layout heap for the deep region processor

    This class is used for keeping intermediate, hierarchical data for the deep region processor. It is used in conjunction with the region constructor to create a deep (hierarchical) region.
    @code
    layout = ... # a layout
    layer = ...  # a layer
    cell = ...   # a cell (initial cell for the deep region)
    dss = RBA::DeepShapeStore::new
    region = RBA::Region::new(cell.begin(layer), dss)
    @/code

    The DeepShapeStore object also supplies some configuration options for the operations acting on the deep regions. See for example \threads=.

    This class has been introduced in version 0.26.
    """
    @property
    def subcircuit_hierarchy_for_nets(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets a value indicating whether to build a subcircuit hierarchy per net


        This flag is used to determine the way, net subcircuit hierarchies are built:
        when true, subcells are created for subcircuits on a net. Otherwise the net
        shapes are produced flat inside the cell they appear on.

        This attribute has been introduced in version 0.28.4
        """
    max_area_ratio: float
    r"""
    Getter:
    @brief Gets the max. area ratio.

    Setter:
    @brief Sets the max. area ratio for bounding box vs. polygon area

    This parameter is used to simplify complex polygons. It is used by
    create_polygon_layer with the default parameters. It's also used by
    boolean operations when they deliver their output.
    """
    max_vertex_count: int
    r"""
    Getter:
    @brief Gets the maximum vertex count.

    Setter:
    @brief Sets the maximum vertex count default value

    This parameter is used to simplify complex polygons. It is used by
    create_polygon_layer with the default parameters. It's also used by
    boolean operations when they deliver their output.
    """
    reject_odd_polygons: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether to reject odd polygons.
    This attribute has been introduced in version 0.27.
    Setter:
    @brief Sets a flag indicating whether to reject odd polygons

    Some kind of 'odd' (e.g. non-orientable) polygons may spoil the functionality because they cannot be handled properly. By using this flag, the shape store we reject these kind of polygons. The default is 'accept' (without warning).

    This attribute has been introduced in version 0.27.
    """
    text_enlargement: int
    r"""
    Getter:
    @brief Gets the text enlargement value.

    Setter:
    @brief Sets the text enlargement value

    If set to a non-negative value, text objects are converted to boxes with the
    given enlargement (width = 2 * enlargement). The box centers are identical
    to the original location of the text.
    If this value is negative (the default), texts are ignored.
    """
    text_property_name: Any
    r"""
    Getter:
    @brief Gets the text property name.

    Setter:
    @brief Sets the text property name.

    If set to a non-null variant, text strings are attached to the generated boxes
    as properties with this particular name. This option has an effect only if the
    text_enlargement property is not negative.
    By default, the name is empty.
    """
    threads: int
    r"""
    Getter:
    @brief Gets the number of threads.

    Setter:
    @brief Sets the number of threads to allocate for the hierarchical processor
    """
    wants_all_cells: bool
    r"""
    Getter:
    @brief Gets a flag wether to copy the full hierarchy for the working layouts
    This attribute has been introduced in version 0.28.10.
    Setter:
    @brief Sets a flag wether to copy the full hierarchy for the working layouts

    The DeepShapeStore object keeps a copy of the original hierarchy internally for the working layouts.
    By default, this hierarchy is mapping only non-empty cells. While the operations proceed, more cells may need to be added. This conservative approach saves some memory, but the update operations may reduce overall performance. Setting this flag to 'true' switches to a mode where the full hierarchy is copied always. This will take more memory but may save CPU time.

    This attribute has been introduced in version 0.28.10.
    """
    @classmethod
    def instance_count(cls) -> int:
        r"""
        @hide
        """
    @classmethod
    def new(cls) -> DeepShapeStore:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_breakout_cell(self, layout_index: int, cell_index: Sequence[int]) -> None:
        r"""
        @brief Adds a cell indexe to the breakout cell list for the given layout inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """
    @overload
    def add_breakout_cells(self, layout_index: int, cells: Sequence[int]) -> None:
        r"""
        @brief Adds cell indexes to the breakout cell list for the given layout inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """
    @overload
    def add_breakout_cells(self, layout_index: int, pattern: str) -> None:
        r"""
        @brief Adds cells (given by a cell name pattern) to the breakout cell list for the given layout inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """
    @overload
    def add_breakout_cells(self, pattern: str) -> None:
        r"""
        @brief Adds cells (given by a cell name pattern) to the breakout cell list to all layouts inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """
    @overload
    def clear_breakout_cells(self) -> None:
        r"""
        @brief Clears the breakout cells
        See the other variant of \clear_breakout_cells for details.

        This method has been added in version 0.26.1
        """
    @overload
    def clear_breakout_cells(self, layout_index: int) -> None:
        r"""
        @brief Clears the breakout cells
        Breakout cells are a feature by which hierarchy handling can be disabled for specific cells. If cells are specified as breakout cells, they don't interact with neighbor or parent cells, hence are virtually isolated. Breakout cells are useful to shortcut hierarchy evaluation for cells which are otherwise difficult to handle. An example are memory array cells with overlaps to their neighbors: a precise handling of such cells would generate variants and the boundary of the array. Although precise, this behavior leads to partial flattening and propagation of shapes. In consequence, this will also result in wrong device detection in LVS applications. In such cases, these array cells can be declared 'breakout cells' which makes them isolated entities and variant generation does not happen.

        See also \set_breakout_cells and \add_breakout_cells.

        This method has been added in version 0.26.1
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_singular(self) -> bool:
        r"""
        @brief Gets a value indicating whether there is a single layout variant

        Specifically for network extraction, singular DSS objects are required. Multiple layouts may be present if different sources of layouts have been used. Such DSS objects are not usable for network extraction.
        """
    def pop_state(self) -> None:
        r"""
        @brief Restores the store's state on the state state
        This will restore the state pushed by \push_state.

        This method has been added in version 0.26.1
        """
    def push_state(self) -> None:
        r"""
        @brief Pushes the store's state on the state state
        This will save the stores state (\threads, \max_vertex_count, \max_area_ratio, breakout cells ...) on the state stack. \pop_state can be used to restore the state.

        This method has been added in version 0.26.1
        """
    @overload
    def set_breakout_cells(self, layout_index: int, cells: Sequence[int]) -> None:
        r"""
        @brief Sets the breakout cell list (as cell indexes) for the given layout inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """
    @overload
    def set_breakout_cells(self, layout_index: int, pattern: str) -> None:
        r"""
        @brief Sets the breakout cell list (as cell name pattern) for the given layout inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """
    @overload
    def set_breakout_cells(self, pattern: str) -> None:
        r"""
        @brief Sets the breakout cell list (as cell name pattern) for the all layouts inside the store
        See \clear_breakout_cells for an explanation of breakout cells.

        This method has been added in version 0.26.1
        """

class Device(NetlistObject):
    r"""
    @brief A device inside a circuit.
    Device object represent atomic devices such as resistors, diodes or transistors. The \Device class represents a particular device with specific parameters. The type of device is represented by a \DeviceClass object. Device objects live in \Circuit objects, the device class objects live in the \Netlist object.

    Devices connect to nets through terminals. Terminals are described by a terminal ID which is essentially the zero-based index of the terminal. Terminal definitions can be obtained from the device class using the \DeviceClass#terminal_definitions method.

    Devices connect to nets through the \Device#connect_terminal method. Device terminals can be disconnected using \Device#disconnect_terminal.

    Device objects are created inside a circuit with \Circuit#create_device.

    This class has been added in version 0.26.
    """
    device_abstract: DeviceAbstract
    r"""
    Getter:
    @brief Gets the device abstract for this device instance.
    See \DeviceAbstract for more details.

    Setter:
    @hide
    Provided for test purposes mainly. Be careful with pointers!
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the device.

    Setter:
    @brief Sets the name of the device.
    Device names are used to name a device inside a netlist file. Device names should be unique within a circuit.
    """
    trans: DCplxTrans
    r"""
    Getter:
    @brief Gets the location of the device.
    See \trans= for details about this method.
    Setter:
    @brief Sets the location of the device.
    The device location is essentially describing the position of the device. The position is typically the center of some recognition shape. In this case the transformation is a plain displacement to the center of this shape.
    """
    def _assign(self, other: NetlistObject) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> Device:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_combined_abstract(self, ref: DeviceAbstractRef) -> None:
        r"""
        @hide
        Provided for test purposes mainly.
        """
    def add_reconnected_terminal_for(self, outer_terminal: int, descriptor: DeviceReconnectedTerminal) -> None:
        r"""
        @hide
        Provided for test purposes mainly.
        """
    @overload
    def circuit(self) -> Circuit:
        r"""
        @brief Gets the circuit the device lives in.
        """
    @overload
    def circuit(self) -> Circuit:
        r"""
        @brief Gets the circuit the device lives in (non-const version).

        This constness variant has been introduced in version 0.26.8
        """
    def clear_combined_abstracts(self) -> None:
        r"""
        @hide
        Provided for test purposes mainly.
        """
    def clear_reconnected_terminals(self) -> None:
        r"""
        @hide
        Provided for test purposes mainly.
        """
    @overload
    def connect_terminal(self, terminal_id: int, net: Net) -> None:
        r"""
        @brief Connects the given terminal to the specified net.
        """
    @overload
    def connect_terminal(self, terminal_name: str, net: Net) -> None:
        r"""
        @brief Connects the given terminal to the specified net.
        This version accepts a terminal name. If the name is not a valid terminal name, an exception is raised.
        If the terminal has been connected to a global net, it will be disconnected from there.
        """
    def device_class(self) -> DeviceClass:
        r"""
        @brief Gets the device class the device belongs to.
        """
    @overload
    def disconnect_terminal(self, terminal_id: int) -> None:
        r"""
        @brief Disconnects the given terminal from any net.
        If the terminal has been connected to a global, this connection will be disconnected too.
        """
    @overload
    def disconnect_terminal(self, terminal_name: str) -> None:
        r"""
        @brief Disconnects the given terminal from any net.
        This version accepts a terminal name. If the name is not a valid terminal name, an exception is raised.
        """
    def each_combined_abstract(self) -> Iterator[DeviceAbstractRef]:
        r"""
        @brief Iterates over the combined device specifications.
        This feature applies to combined devices. This iterator will deliver all device abstracts present in addition to the default device abstract.
        """
    def each_reconnected_terminal_for(self, terminal_id: int) -> Iterator[DeviceReconnectedTerminal]:
        r"""
        @brief Iterates over the reconnected terminal specifications for a given outer terminal.
        This feature applies to combined devices. This iterator will deliver all device-to-abstract terminal reroutings.
        """
    def expanded_name(self) -> str:
        r"""
        @brief Gets the expanded name of the device.
        The expanded name takes the name of the device. If the name is empty, the numeric ID will be used to build a name. 
        """
    def id(self) -> int:
        r"""
        @brief Gets the device ID.
        The ID is a unique integer which identifies the device.
        It can be used to retrieve the device from the circuit using \Circuit#device_by_id.
        When assigned, the device ID is not 0.
        """
    def is_combined_device(self) -> bool:
        r"""
        @brief Returns true, if the device is a combined device.
        Combined devices feature multiple device abstracts and device-to-abstract terminal connections.
        See \each_reconnected_terminal and \each_combined_abstract for more details.
        """
    @overload
    def net_for_terminal(self, terminal_id: int) -> Net:
        r"""
        @brief Gets the net connected to the specified terminal.
        If the terminal is not connected, nil is returned for the net.
        """
    @overload
    def net_for_terminal(self, terminal_id: int) -> Net:
        r"""
        @brief Gets the net connected to the specified terminal (non-const version).
        If the terminal is not connected, nil is returned for the net.

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def net_for_terminal(self, terminal_name: str) -> Net:
        r"""
        @brief Gets the net connected to the specified terminal (non-const version).
        If the terminal is not connected, nil is returned for the net.

        This convenience method has been introduced in version 0.27.3.
        """
    @overload
    def net_for_terminal(self, terminal_name: str) -> Net:
        r"""
        @brief Gets the net connected to the specified terminal.
        If the terminal is not connected, nil is returned for the net.

        This convenience method has been introduced in version 0.27.3.
        """
    @overload
    def parameter(self, param_id: int) -> float:
        r"""
        @brief Gets the parameter value for the given parameter ID.
        """
    @overload
    def parameter(self, param_name: str) -> float:
        r"""
        @brief Gets the parameter value for the given parameter name.
        If the parameter name is not valid, an exception is thrown.
        """
    @overload
    def set_parameter(self, param_id: int, value: float) -> None:
        r"""
        @brief Sets the parameter value for the given parameter ID.
        """
    @overload
    def set_parameter(self, param_name: str, value: float) -> None:
        r"""
        @brief Sets the parameter value for the given parameter name.
        If the parameter name is not valid, an exception is thrown.
        """

class DeviceAbstract:
    r"""
    @brief A geometrical device abstract
    This class represents the geometrical model for the device. It links into the extracted layout to a cell which holds the terminal shapes for the device.

    This class has been added in version 0.26.
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the device abstract.

    Setter:
    @brief Sets the name of the device abstract.
    Device names are used to name a device abstract inside a netlist file. Device names should be unique within a netlist.
    """
    @classmethod
    def new(cls) -> DeviceAbstract:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> DeviceAbstract:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceAbstract:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DeviceAbstract) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cell_index(self) -> int:
        r"""
        @brief Gets the cell index of the device abstract.
        This is the cell that represents the device.
        """
    def cluster_id_for_terminal(self, terminal_id: int) -> int:
        r"""
        @brief Gets the cluster ID for the given terminal.
        The cluster ID links the terminal to geometrical shapes within the clusters of the cell (see \cell_index)
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def device_class(self) -> DeviceClass:
        r"""
        @brief Gets the device class of the device.
        """
    def dup(self) -> DeviceAbstract:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    @overload
    def netlist(self) -> Netlist:
        r"""
        @brief Gets the netlist the device abstract lives in (non-const version).

        This constness variant has been introduced in version 0.26.8
        """
    @overload
    def netlist(self) -> Netlist:
        r"""
        @brief Gets the netlist the device abstract lives in.
        """

class DeviceAbstractRef:
    r"""
    @brief Describes an additional device abstract reference for combined devices.
    Combined devices are implemented as a generalization of the device abstract concept in \Device. For combined devices, multiple \DeviceAbstract references are present. This class describes such an additional reference. A reference is a pointer to an abstract plus a transformation by which the abstract is transformed geometrically as compared to the first (initial) abstract.

    This class has been introduced in version 0.26.
    """
    device_abstract: DeviceAbstract
    r"""
    Getter:
    @brief The getter for the device abstract reference.
    See the class description for details.
    Setter:
    @brief The setter for the device abstract reference.
    See the class description for details.
    """
    trans: DCplxTrans
    r"""
    Getter:
    @brief The getter for the relative transformation of the instance.
    See the class description for details.
    Setter:
    @brief The setter for the relative transformation of the instance.
    See the class description for details.
    """
    @classmethod
    def new(cls) -> DeviceAbstractRef:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> DeviceAbstractRef:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceAbstractRef:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DeviceAbstractRef) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DeviceAbstractRef:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class DeviceClass:
    r"""
    @brief A class describing a specific type of device.
    Device class objects live in the context of a \Netlist object. After a device class is created, it must be added to the netlist using \Netlist#add. The netlist will own the device class object. When the netlist is destroyed, the device class object will become invalid.

    The \DeviceClass class is the base class for other device classes.

    This class has been added in version 0.26. In version 0.27.3, the 'GenericDeviceClass' has been integrated with \DeviceClass and the device class was made writeable in most respects. This enables manipulating built-in device classes.
    """
    combiner: GenericDeviceCombiner
    r"""
    Getter:
    @brief Gets a device combiner or nil if none is registered.

    This method has been added in version 0.27.3.

    Setter:
    @brief Specifies a device combiner (parallel or serial device combination).

    You can assign nil for the combiner to remove it.

    In special cases, you can even implement a custom combiner by deriving your own comparer from the \GenericDeviceCombiner class.

    This method has been added in version 0.27.3.
    """
    @property
    def supports_parallel_combination(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Specifies whether the device supports parallel device combination.
        Parallel device combination means that all terminals of two combination candidates are connected to the same nets. If the device does not support this combination mode, this predicate can be set to false. This will make the device extractor skip the combination test in parallel mode and improve performance somewhat.

        This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
        """
    @property
    def supports_serial_combination(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Specifies whether the device supports serial device combination.
        Serial device combination means that the devices are connected by internal nodes. If the device does not support this combination mode, this predicate can be set to false. This will make the device extractor skip the combination test in serial mode and improve performance somewhat.

        This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
        """
    description: str
    r"""
    Getter:
    @brief Gets the description text of the device class.
    Setter:
    @brief Sets the description of the device class.
    """
    equal_parameters: EqualDeviceParameters
    r"""
    Getter:
    @brief Gets the device parameter comparer for netlist verification or nil if no comparer is registered.
    See \equal_parameters= for the setter.

    This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.

    Setter:
    @brief Specifies a device parameter comparer for netlist verification.
    By default, all devices are compared with all parameters. If you want to select only certain parameters for comparison or use a fuzzy compare criterion, use an \EqualDeviceParameters object and assign it to the device class of one netlist. You can also chain multiple \EqualDeviceParameters objects with the '+' operator for specifying multiple parameters in the equality check.

    You can assign nil for the parameter comparer to remove it.

    In special cases, you can even implement a custom compare scheme by deriving your own comparer from the \GenericDeviceParameterCompare class.

    This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the device class.
    Setter:
    @brief Sets the name of the device class.
    """
    strict: bool
    r"""
    Getter:
    @brief Gets a value indicating whether this class performs strict terminal mapping
    See \strict= for details about this attribute.
    Setter:
    @brief Sets a value indicating whether this class performs strict terminal mapping

    Classes with this flag set never allow terminal swapping, even if the device symmetry supports that. If two classes are involved in a netlist compare,
    terminal swapping will be disabled if one of the classes is in strict mode.

    By default, device classes are not strict and terminal swapping is allowed as far as the device symmetry supports that.
    """
    @classmethod
    def new(cls) -> DeviceClass:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> DeviceClass:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceClass:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_parameter(self, parameter_def: DeviceParameterDefinition) -> None:
        r"""
        @brief Adds the given parameter definition to the device class
        This method will define a new parameter. The new parameter is added at the end of existing parameters. The parameter definition object passed as the argument is modified to contain the new ID of the parameter.
        The parameter is copied into the device class. Modifying the parameter object later does not have the effect of changing the parameter definition.

        This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
        """
    def add_terminal(self, terminal_def: DeviceTerminalDefinition) -> None:
        r"""
        @brief Adds the given terminal definition to the device class
        This method will define a new terminal. The new terminal is added at the end of existing terminals. The terminal definition object passed as the argument is modified to contain the new ID of the terminal.

        The terminal is copied into the device class. Modifying the terminal object later does not have the effect of changing the terminal definition.

        This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
        """
    def assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def clear_equivalent_terminal_ids(self) -> None:
        r"""
        @brief Clears all equivalent terminal ids

        This method has been added in version 0.27.3.
        """
    def clear_parameters(self) -> None:
        r"""
        @brief Clears the list of parameters

        This method has been added in version 0.27.3.
        """
    def clear_terminals(self) -> None:
        r"""
        @brief Clears the list of terminals

        This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DeviceClass:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def enable_parameter(self, parameter_id: int, enable: bool) -> None:
        r"""
        @brief Enables or disables a parameter.
        Some parameters are 'secondary' parameters which are extracted but not handled in device compare and are not shown in the netlist browser. For example, the 'W' parameter of the resistor is such a secondary parameter. This method allows turning a parameter in a primary one ('enable') or into a secondary one ('disable').

        This method has been introduced in version 0.27.3.
        """
    @overload
    def enable_parameter(self, parameter_name: str, enable: bool) -> None:
        r"""
        @brief Enables or disables a parameter.
        Some parameters are 'secondary' parameters which are extracted but not handled in device compare and are not shown in the netlist browser. For example, the 'W' parameter of the resistor is such a secondary parameter. This method allows turning a parameter in a primary one ('enable') or into a secondary one ('disable').

        This version accepts a parameter name.

        This method has been introduced in version 0.27.3.
        """
    def equivalent_terminal_id(self, original_id: int, equivalent_id: int) -> None:
        r"""
        @brief Specifies a terminal to be equivalent to another.
        Use this method to specify two terminals to be exchangeable. For example to make S and D of a MOS transistor equivalent, call this method with S and D terminal IDs. In netlist matching, S will be translated to D and thus made equivalent to D.

        Note that terminal equivalence is not effective if the device class operates in strict mode (see \DeviceClass#strict=).

        This method has been moved from 'GenericDeviceClass' to 'DeviceClass' in version 0.27.3.
        """
    def has_parameter(self, name: str) -> bool:
        r"""
        @brief Returns true, if the device class has a parameter with the given name.
        """
    def has_terminal(self, name: str) -> bool:
        r"""
        @brief Returns true, if the device class has a terminal with the given name.
        """
    def id(self) -> int:
        r"""
        @brief Gets the unique ID of the device class
        The ID is a unique integer that identifies the device class. Use the ID to check for object identity - i.e. to determine whether two devices share the same device class.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def netlist(self) -> Netlist:
        r"""
        @brief Gets the netlist the device class lives in.
        """
    @overload
    def parameter_definition(self, parameter_id: int) -> DeviceParameterDefinition:
        r"""
        @brief Gets the parameter definition object for a given ID.
        Parameter definition IDs are used in some places to reference a specific parameter of a device. This method obtains the corresponding definition object.
        """
    @overload
    def parameter_definition(self, parameter_name: str) -> DeviceParameterDefinition:
        r"""
        @brief Gets the parameter definition object for a given ID.
        Parameter definition IDs are used in some places to reference a specific parameter of a device. This method obtains the corresponding definition object.
        This version accepts a parameter name.

        This method has been introduced in version 0.27.3.
        """
    def parameter_definitions(self) -> List[DeviceParameterDefinition]:
        r"""
        @brief Gets the list of parameter definitions of the device.
        See the \DeviceParameterDefinition class description for details.
        """
    def parameter_id(self, name: str) -> int:
        r"""
        @brief Returns the parameter ID of the parameter with the given name.
        An exception is thrown if there is no parameter with the given name. Use \has_parameter to check whether the name is a valid parameter name.
        """
    def terminal_definition(self, terminal_id: int) -> DeviceTerminalDefinition:
        r"""
        @brief Gets the terminal definition object for a given ID.
        Terminal definition IDs are used in some places to reference a specific terminal of a device. This method obtains the corresponding definition object.
        """
    def terminal_definitions(self) -> List[DeviceTerminalDefinition]:
        r"""
        @brief Gets the list of terminal definitions of the device.
        See the \DeviceTerminalDefinition class description for details.
        """
    def terminal_id(self, name: str) -> int:
        r"""
        @brief Returns the terminal ID of the terminal with the given name.
        An exception is thrown if there is no terminal with the given name. Use \has_terminal to check whether the name is a valid terminal name.
        """

class DeviceClassBJT3Transistor(DeviceClass):
    r"""
    @brief A device class for a bipolar transistor.
    This class describes a bipolar transistor. Bipolar transistors have tree terminals: the collector (C), the base (B) and the emitter (E).
    Multi-emitter transistors are resolved in individual devices.
    The parameters are AE, AB and AC for the emitter, base and collector areas in square micrometers and PE, PB and PC for the emitter, base and collector perimeters in micrometers.
    In addition, the emitter count (NE) is given. The emitter count is 1 always for a transistor extracted initially. Upon combination of devices, the emitter counts are added, thus forming multi-emitter devices.

    This class has been introduced in version 0.26.
    """
    PARAM_AB: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter AB (base area)
    """
    PARAM_AC: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter AC (collector area)
    """
    PARAM_AE: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter AE (emitter area)
    """
    PARAM_NE: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter NE (emitter count)
    """
    PARAM_PB: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter PB (base perimeter)
    """
    PARAM_PC: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter PC (collector perimeter)
    """
    PARAM_PE: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter PE (emitter perimeter)
    """
    TERMINAL_B: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal B (base)
    """
    TERMINAL_C: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal C (collector)
    """
    TERMINAL_E: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal E (emitter)
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassBJT3Transistor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassBJT4Transistor(DeviceClassBJT3Transistor):
    r"""
    @brief A device class for a 4-terminal bipolar transistor.
    This class describes a bipolar transistor with a substrate terminal. A device class for a bipolar transistor without a substrate terminal is \DeviceClassBJT3Transistor. 
    The additional terminal is 'S' for the substrate terminal.
    BJT4 transistors combine in parallel if both substrate terminals are connected to the same net.

    This class has been introduced in version 0.26.
    """
    TERMINAL_S: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal S
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassBJT4Transistor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassCapacitor(DeviceClass):
    r"""
    @brief A device class for a capacitor.
    This describes a capacitor. Capacitors are defined by their combination behavior and the basic parameter 'C' which is the capacitance in Farad.

    A capacitor has two terminals, A and B.
    The parameters of a capacitor are C (the value in Farad) and A and P (area and perimeter in square micrometers and micrometers respectively).

    This class has been introduced in version 0.26.
    """
    PARAM_A: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter A
    """
    PARAM_C: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter C
    """
    PARAM_P: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter P
    """
    TERMINAL_A: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal A
    """
    TERMINAL_B: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal B
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassCapacitor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassCapacitorWithBulk(DeviceClassCapacitor):
    r"""
    @brief A device class for a capacitor with a bulk terminal (substrate, well).
    This class is similar to \DeviceClassCapacitor, but provides an additional terminal (BULK) for the well or substrate the capacitor is embedded in.

    The additional terminal is 'W' for the well/substrate terminal.

    This class has been introduced in version 0.26.
    """
    TERMINAL_W: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal W (well, bulk)
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassCapacitorWithBulk:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassDiode(DeviceClass):
    r"""
    @brief A device class for a diode.
    This class describes a diode.
    A diode has two terminals, A (anode) and C (cathode).
    It has two parameters: The diode area in square micrometers (A) and the diode area perimeter in micrometers (P).

    Diodes only combine when parallel and in the same direction. In this case, their areas and perimeters are added.
    This class has been introduced in version 0.26.
    """
    PARAM_A: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter A
    """
    PARAM_P: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter P
    """
    TERMINAL_A: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal A
    """
    TERMINAL_C: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal C
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassDiode:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassFactory:
    r"""
    @brief A factory for creating specific device classes for the standard device extractors
    Use a reimplementation of this class to provide a device class generator for built-in device extractors such as \DeviceExtractorMOS3Transistor. The constructor of this extractor has a 'factory' parameter which takes an object of \DeviceClassFactory type.

    If such an object is provided, this factory is used to create the actual device class. The following code shows an example:

    @code
    class MyClass < RBA::DeviceClassMOS3Transistor
      ... overrides some methods ...
    end

    class MyFactory < RBA::DeviceClassFactory
      def create_class
        MyClass.new
      end
    end

    extractor = RBA::DeviceExtractorMOS3Transistor::new("NMOS", false, MyFactory.new)
    @/code

    When using a factory with a device extractor, make sure it creates a corresponding device class, e.g. for the \DeviceExtractorMOS3Transistor extractor create a device class derived from \DeviceClassMOS3Transistor.

    This class has been introduced in version 0.27.3.
    """
    @classmethod
    def new(cls) -> DeviceClassFactory:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> DeviceClassFactory:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceClassFactory:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DeviceClassFactory) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DeviceClassFactory:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class DeviceClassInductor(DeviceClass):
    r"""
    @brief A device class for an inductor.
    This class describes an inductor. Inductors are defined by their combination behavior and the basic parameter 'L' which is the inductance in Henry.

    An inductor has two terminals, A and B.

    This class has been introduced in version 0.26.
    """
    PARAM_L: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter L
    """
    TERMINAL_A: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal A
    """
    TERMINAL_B: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal B
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassInductor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassMOS3Transistor(DeviceClass):
    r"""
    @brief A device class for a 3-terminal MOS transistor.
    This class describes a MOS transistor without a bulk terminal. A device class for a MOS transistor with a bulk terminal is \DeviceClassMOS4Transistor. MOS transistors are defined by their combination behavior and the basic parameters.

    The parameters are L, W, AS, AD, PS and PD for the gate length and width in micrometers, source and drain area in square micrometers and the source and drain perimeter in micrometers.

    The terminals are S, G and D for source, gate and drain.

    MOS transistors combine in parallel mode, when both gate lengths are identical and their gates are connected (source and drain can be swapped). In this case, their widths and source and drain areas are added.

    This class has been introduced in version 0.26.
    """
    PARAM_AD: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter AD
    """
    PARAM_AS: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter AS
    """
    PARAM_L: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter L
    """
    PARAM_PD: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter PD
    """
    PARAM_PS: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter PS
    """
    PARAM_W: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter W
    """
    TERMINAL_D: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal D
    """
    TERMINAL_G: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal G
    """
    TERMINAL_S: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal S
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassMOS3Transistor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def join_split_gates(self, circuit: Circuit) -> None:
        r"""
        @brief Joins source/drain nets from 'split gate' transistor strings on the given circuit
        This method has been introduced in version 0.27.9
        """

class DeviceClassMOS4Transistor(DeviceClassMOS3Transistor):
    r"""
    @brief A device class for a 4-terminal MOS transistor.
    This class describes a MOS transistor with a bulk terminal. A device class for a MOS transistor without a bulk terminal is \DeviceClassMOS3Transistor. MOS transistors are defined by their combination behavior and the basic parameters.

    The additional terminal is 'B' for the bulk terminal.
    MOS4 transistors combine in parallel if both bulk terminals are connected to the same net.

    This class has been introduced in version 0.26.
    """
    TERMINAL_B: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal B
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassMOS4Transistor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassResistor(DeviceClass):
    r"""
    @brief A device class for a resistor.
    This class describes a resistor. Resistors are defined by their combination behavior and the basic parameter 'R' which is the resistance in Ohm.

    A resistor has two terminals, A and B.
    The parameters of a resistor are R (the value in Ohms), L and W (length and width in micrometers) and A and P (area and perimeter in square micrometers and micrometers respectively).

    This class has been introduced in version 0.26.
    """
    PARAM_A: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter A
    """
    PARAM_L: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter L
    """
    PARAM_P: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter P
    """
    PARAM_R: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter R
    """
    PARAM_W: ClassVar[int]
    r"""
    @brief A constant giving the parameter ID for parameter W
    """
    TERMINAL_A: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal A
    """
    TERMINAL_B: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal B
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassResistor:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceClassResistorWithBulk(DeviceClassResistor):
    r"""
    @brief A device class for a resistor with a bulk terminal (substrate, well).
    This class is similar to \DeviceClassResistor, but provides an additional terminal (BULK) for the well or substrate the resistor is embedded in.

    The additional terminal is 'W' for the well/substrate terminal.

    This class has been introduced in version 0.26.
    """
    TERMINAL_W: ClassVar[int]
    r"""
    @brief A constant giving the terminal ID for terminal W (well, bulk)
    """
    def _assign(self, other: DeviceClass) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> DeviceClassResistorWithBulk:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorBJT3Transistor(DeviceExtractorBase):
    r"""
    @brief A device extractor for a bipolar transistor (BJT)

    This class supplies the generic extractor for a bipolar transistor device.

    Extraction of vertical and lateral transistors is supported through a generic geometry model: The basic area is the base area. A marker shape must be provided for this area. The emitter of the transistor is defined by emitter layer shapes inside the base area. Multiple emitter shapes can be present. In this case, multiple transistor devices sharing the same base and collector are generated.
    Finally, a collector layer can be given. If non-empty, the parts inside the base region will define the collector terminals. If empty, the collector is formed by the substrate. In this case, the base region will be output to the 'tC' terminal output layer. This layer then needs to be connected to a global net to form the net connection.

    The device class produced by this extractor is \DeviceClassBJT3Transistor.
    The extractor delivers these parameters:

    @ul
    @li 'AE', 'AB' and 'AC' - the emitter, base and collector areas in square micrometer units @/li
    @li 'PE', 'PB' and 'PC' - the emitter, base and collector perimeters in micrometer units @/li
    @li 'NE' - emitter count (initially 1 but increases when devices are combined) @/li
    @/ul

    The device layer names are:

    @ul
    @li 'E' - emitter. @/li
    @li 'B' - base. @/li
    @li 'C' - collector. @/li
    @/ul

    The terminals are output on these layers:
    @ul
    @li 'tE' - emitter. Default output is 'E'. @/li
    @li 'tB' - base. Default output is 'B'. @/li
    @li 'tC' - collector. Default output is 'C'. @/li
    @/ul

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorBJT3Transistor:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorBJT4Transistor(DeviceExtractorBJT3Transistor):
    r"""
    @brief A device extractor for a four-terminal bipolar transistor (BJT)

    This class supplies the generic extractor for a bipolar transistor device.
    It is based on the \DeviceExtractorBJT3Transistor class with the extension of a substrate terminal and corresponding substrate terminal output (annotation) layer.

    Two new layers are introduced:

    @ul
    @li 'S' - the bulk (substrate) layer. Currently this layer is ignored and can be empty. @/li@li 'tS' - the bulk terminal output layer (defaults to 'S'). @/li@/ul

    The bulk terminal layer ('tS') can be an empty layer representing the wafer substrate.
    In this use mode the substrate terminal shapes will be produced on the 'tS' layer. This
    layer then needs to be connected to a global net to establish the net connection.

    The device class produced by this extractor is \DeviceClassBJT4Transistor.
    The This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorBJT4Transistor:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorBase:
    r"""
    @brief The base class for all device extractors.
    This is an abstract base class for device extractors. See \GenericDeviceExtractor for a generic class which you can reimplement to supply your own customized device extractor. In many cases using one of the preconfigured specific device extractors may be useful already and it's not required to implement a custom one. For an example about a preconfigured device extractor see \DeviceExtractorMOS3Transistor.

    This class cannot and should not be instantiated explicitly. Use one of the subclasses instead.

    This class has been introduced in version 0.26.
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the device extractor and the device class.
    Setter:
    @brief Sets the name of the device extractor and the device class.
    """
    @classmethod
    def new(cls) -> DeviceExtractorBase:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def device_class(self) -> DeviceClass:
        r"""
        @brief Gets the device class used during extraction
        The attribute will hold the actual device class used in the device extraction. It is valid only after 'extract_devices'.

        This method has been added in version 0.27.3.
        """
    def each_error(self) -> Iterator[LogEntryData]:
        r"""
        @brief Iterates over all log entries collected in the device extractor.Starting with version 0.28.13, the preferred name of the method is 'each_log_entry' as log entries have been generalized to become warnings too.
        """
    def each_layer_definition(self) -> Iterator[NetlistDeviceExtractorLayerDefinition]:
        r"""
        @brief Iterates over all layer definitions.
        """
    def each_log_entry(self) -> Iterator[LogEntryData]:
        r"""
        @brief Iterates over all log entries collected in the device extractor.Starting with version 0.28.13, the preferred name of the method is 'each_log_entry' as log entries have been generalized to become warnings too.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def test_initialize(self, netlist: Netlist) -> None:
        r"""
        @hide
        """

class DeviceExtractorCapacitor(DeviceExtractorBase):
    r"""
    @brief A device extractor for a two-terminal capacitor

    This class supplies the generic extractor for a capacitor device.
    The device is defined by two geometry layers forming the 'plates' of the capacitor.
    The capacitance is computed from the overlapping area of the plates using 'C = A * area_cap' (area_cap is the capacitance per square micrometer area).

    Although 'area_cap' can be given in any unit, Farad should be preferred as this is the convention used for output into a netlist.

    The device class produced by this extractor is \DeviceClassCapacitor.
    The extractor produces three parameters:

    @ul
    @li 'C' - the capacitance @/li
    @li 'A' - the capacitor's area in square micrometer units @/li
    @li 'P' - the capacitor's perimeter in micrometer units @/li
    @/ul

    The device layer names are:

    @ul
    @li 'P1', 'P2' - the two plates. @/li
    @/ul

    The terminals are output on these layers:
    @ul
    @li 'tA', 'tB' - the two terminals. Defaults to 'P1' and 'P2'. @/li
    @/ul

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, area_cap: float, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorCapacitor:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, area_cap: float, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorCapacitorWithBulk(DeviceExtractorBase):
    r"""
    @brief A device extractor for a capacitor with a bulk terminal

    This class supplies the generic extractor for a capacitor device including a bulk terminal.
    The device is defined the same way than devices are defined for \DeviceExtractorCapacitor.

    The device class produced by this extractor is \DeviceClassCapacitorWithBulk.
    The extractor produces three parameters:

    @ul
    @li 'C' - the capacitance @/li
    @li 'A' - the capacitor's area in square micrometer units @/li
    @li 'P' - the capacitor's perimeter in micrometer units @/li
    @/ul

    The device layer names are:

    @ul
    @li 'P1', 'P2' - the two plates. @/li
    @li 'W' - well, bulk. Currently this layer is ignored for the extraction and can be empty. @/li
    @/ul

    The terminals are output on these layers:
    @ul
    @li 'tA', 'tB' - the two terminals. Defaults to 'P1' and 'P2'. @/li
    @li 'tW' - the bulk terminal (copy of the resistor area). @/li
    @/ul

    The bulk terminal layer can be an empty layer representing the substrate. In this case, it needs to be connected globally.

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, sheet_rho: float, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorCapacitorWithBulk:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, sheet_rho: float, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorDiode(DeviceExtractorBase):
    r"""
    @brief A device extractor for a planar diode

    This class supplies the generic extractor for a planar diode.
    The diode is defined by two layers whose overlap area forms
    the diode. The p-type layer forms the anode, the n-type layer
    the cathode.

    The device class produced by this extractor is \DeviceClassDiode.
    The extractor extracts the two parameters of this class:

    @ul
    @li 'A' - the diode area in square micrometer units. @/li
    @li 'P' - the diode perimeter in micrometer units. @/li
    @/ul

    The device layers are:

    @ul
    @li 'P' - the p doped area. @/li
    @li 'N' - the n doped area. @/li
    @/ul

    The diode region is defined by the overlap of p and n regions.

    The terminal output layers are:

    @ul
    @li 'tA' - anode. Defaults to 'P'. @/li
    @li 'tC' - cathode. Defaults to 'N'. @/li
    @/ul

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorDiode:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorMOS3Transistor(DeviceExtractorBase):
    r"""
    @brief A device extractor for a three-terminal MOS transistor

    This class supplies the generic extractor for a MOS device.
    The device is defined by two basic input layers: the diffusion area
    (source and drain) and the gate area. It requires a third layer
    (poly) to put the gate terminals on. The separation between poly
    and allows separating the device recognition layer (gate) from the
    conductive layer.

    The device class produced by this extractor is \DeviceClassMOS3Transistor.

    The extractor delivers six parameters:

    @ul
    @li 'L' - the gate length in micrometer units @/li
    @li 'W' - the gate width in micrometer units @/li
    @li 'AS' and 'AD' - the source and drain region areas in square micrometers @/li
    @li 'PS' and 'PD' - the source and drain region perimeters in micrometer units @/li
    @/ul

    The device layer names are:

    @ul
    @li In strict mode: 'S' (source), 'D' (drain) and 'G' (gate). @/li
    @li In non-strict mode: 'SD' (source and drain) and 'G' (gate). @/li
    @/ul

    The terminals are output on these layers:
    @ul
    @li 'tS' - source. Default output is 'S' (strict mode) or 'SD' (otherwise). @/li
    @li 'tD' - drain. Default output is 'D' (strict mode) or 'SD' (otherwise). @/li
    @li 'tG' - gate. Default output is 'G'. @/li
    @/ul

    The source/drain (diffusion) area is distributed on the number of gates connecting to
    the particular source or drain area.

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, strict: Optional[bool] = ..., factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorMOS3Transistor:
        r"""
        @brief Creates a new device extractor with the given name.
        If \strict is true, the MOS device extraction will happen in strict mode. That is, source and drain are not interchangeable.

        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, strict: Optional[bool] = ..., factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name.
        If \strict is true, the MOS device extraction will happen in strict mode. That is, source and drain are not interchangeable.

        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def strict(self) -> bool:
        r"""
        @brief Returns a value indicating whether extraction happens in strict mode.
        """

class DeviceExtractorMOS4Transistor(DeviceExtractorBase):
    r"""
    @brief A device extractor for a four-terminal MOS transistor

    This class supplies the generic extractor for a MOS device.
    It is based on the \DeviceExtractorMOS3Transistor class with the extension of a bulk terminal and corresponding bulk terminal output (annotation) layer.

    The parameters of a MOS4 device are the same than for MOS3 devices. For the device layers the bulk layer is added.

    @ul
    @li 'B' (bulk) - currently this layer is not used and can be empty. @/li
    @/ul

    The bulk terminals are output on this layer:
    @ul
    @li 'tB' - bulk terminal (a copy of the gate shape). Default output is 'B'. @/li
    @/ul

    The bulk terminal layer can be empty. In this case, it needs 
    to be connected to a global net to establish the net connection.

    The device class produced by this extractor is \DeviceClassMOS4Transistor.

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, strict: Optional[bool] = ..., factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorMOS4Transistor:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, strict: Optional[bool] = ..., factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorResistor(DeviceExtractorBase):
    r"""
    @brief A device extractor for a two-terminal resistor

    This class supplies the generic extractor for a resistor device.
    The device is defined by two geometry layers: the resistor 'wire' and two contacts per wire. The contacts should be attached to the ends of the wire. The wire length and width is computed from the edge lengths between the contacts and along the contacts respectively.

    This simple computation is precise only when the resistor shape is a rectangle.

    Using the given sheet resistance, the resistance value is computed by 'R = L / W * sheet_rho'.

    The device class produced by this extractor is \DeviceClassResistor.
    The extractor produces three parameters:

    @ul
    @li 'R' - the resistance in Ohm @/li
    @li 'A' - the resistor's area in square micrometer units @/li
    @li 'P' - the resistor's perimeter in micrometer units @/li
    @/ul

    The device layer names are:

    @ul
    @li 'R' - resistor path. This is the geometry that defines the resistor's current path. @/li
    @li 'C' - contacts. These areas form the contact regions at the ends of the resistor path. @/li
    @/ul

    The terminals are output on these layers:
    @ul
    @li 'tA', 'tB' - the two terminals of the resistor. @/li
    @/ul

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, sheet_rho: float, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorResistor:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, sheet_rho: float, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceExtractorResistorWithBulk(DeviceExtractorBase):
    r"""
    @brief A device extractor for a resistor with a bulk terminal

    This class supplies the generic extractor for a resistor device including a bulk terminal.
    The device is defined the same way than devices are defined for \DeviceExtractorResistor.

    The device class produced by this extractor is \DeviceClassResistorWithBulk.
    The extractor produces three parameters:

    @ul
    @li 'R' - the resistance in Ohm @/li
    @li 'A' - the resistor's area in square micrometer units @/li
    @li 'P' - the resistor's perimeter in micrometer units @/li
    @/ul

    The device layer names are:

    @ul
    @li 'R' - resistor path. This is the geometry that defines the resistor's current path. @/li
    @li 'C' - contacts. These areas form the contact regions at the ends of the resistor path. @/li
    @li 'W' - well, bulk. Currently this layer is ignored for the extraction and can be empty. @/li
    @/ul

    The terminals are output on these layers:
    @ul
    @li 'tA', 'tB' - the two terminals of the resistor. @/li
    @li 'tW' - the bulk terminal (copy of the resistor area). @/li
    @/ul

    The bulk terminal layer can be an empty layer representing the substrate. In this case, it needs to be connected globally.

    This class is a closed one and methods cannot be reimplemented. To reimplement specific methods, see \DeviceExtractor.

    This class has been introduced in version 0.26.
    """
    @classmethod
    def new(cls, name: str, sheet_rho: float, factory: Optional[DeviceClassFactory] = ...) -> DeviceExtractorResistorWithBulk:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def __init__(self, name: str, sheet_rho: float, factory: Optional[DeviceClassFactory] = ...) -> None:
        r"""
        @brief Creates a new device extractor with the given name
        For the 'factory' parameter see \DeviceClassFactory. It has been added in version 0.27.3.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """

class DeviceParameterDefinition:
    r"""
    @brief A parameter descriptor
    This class is used inside the \DeviceClass class to describe a parameter of the device.

    This class has been added in version 0.26.
    """
    default_value: float
    r"""
    Getter:
    @brief Gets the default value of the parameter.
    Setter:
    @brief Sets the default value of the parameter.
    The default value is used to initialize parameters of \Device objects.
    """
    description: str
    r"""
    Getter:
    @brief Gets the description of the parameter.
    Setter:
    @brief Sets the description of the parameter.
    """
    geo_scaling_exponent: float
    r"""
    Getter:
    @brief Gets the geometry scaling exponent.
    This value is used when applying '.options scale' in the SPICE reader for example. It is zero for 'no scaling', 1.0 for linear scaling and 2.0 for quadratic scaling.

    This attribute has been added in version 0.28.6.
    Setter:
    @brief Sets the geometry scaling exponent.
    See \geo_scaling_exponent for details.

    This attribute has been added in version 0.28.6.
    """
    is_primary: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the parameter is a primary parameter
    See \is_primary= for details about this predicate.
    Setter:
    @brief Sets a value indicating whether the parameter is a primary parameter
    If this flag is set to true (the default), the parameter is considered a primary parameter.
    Only primary parameters are compared by default.
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the parameter.
    Setter:
    @brief Sets the name of the parameter.
    """
    si_scaling: float
    r"""
    Getter:
    @brief Gets the scaling factor to SI units.
    For parameters in micrometers - for example W and L of MOS devices - this factor can be set to 1e-6 to reflect the unit.
    Setter:
    @brief Sets the scaling factor to SI units.

    This setter has been added in version 0.28.6.
    """
    @classmethod
    def new(cls, name: str, description: Optional[str] = ..., default_value: Optional[float] = ..., is_primary: Optional[bool] = ..., si_scaling: Optional[float] = ..., geo_scaling_exponent: Optional[float] = ...) -> DeviceParameterDefinition:
        r"""
        @brief Creates a new parameter definition.
        @param name The name of the parameter
        @param description The human-readable description
        @param default_value The initial value
        @param is_primary True, if the parameter is a primary parameter (see \is_primary=)
        @param si_scaling The scaling factor to SI units
        @param geo_scaling_exponent Indicates how the parameter scales with geometrical scaling (0: no scaling, 1.0: linear, 2.0: quadratic)
        """
    def __copy__(self) -> DeviceParameterDefinition:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceParameterDefinition:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self, name: str, description: Optional[str] = ..., default_value: Optional[float] = ..., is_primary: Optional[bool] = ..., si_scaling: Optional[float] = ..., geo_scaling_exponent: Optional[float] = ...) -> None:
        r"""
        @brief Creates a new parameter definition.
        @param name The name of the parameter
        @param description The human-readable description
        @param default_value The initial value
        @param is_primary True, if the parameter is a primary parameter (see \is_primary=)
        @param si_scaling The scaling factor to SI units
        @param geo_scaling_exponent Indicates how the parameter scales with geometrical scaling (0: no scaling, 1.0: linear, 2.0: quadratic)
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DeviceParameterDefinition) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DeviceParameterDefinition:
        r"""
        @brief Creates a copy of self
        """
    def id(self) -> int:
        r"""
        @brief Gets the ID of the parameter.
        The ID of the parameter is used in some places to refer to a specific parameter (e.g. in the \NetParameterRef object).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class DeviceReconnectedTerminal:
    r"""
    @brief Describes a terminal rerouting in combined devices.
    Combined devices are implemented as a generalization of the device abstract concept in \Device. For combined devices, multiple \DeviceAbstract references are present. To support different combination schemes, device-to-abstract routing is supported. Parallel combinations will route all outer terminals to corresponding terminals of all device abstracts (because of terminal swapping these may be different ones).

    This object describes one route to an abstract's terminal. The device index is 0 for the main device abstract and 1 for the first combined device abstract.

    This class has been introduced in version 0.26.
    """
    device_index: int
    r"""
    Getter:
    @brief The device abstract index getter.
    See the class description for details.
    Setter:
    @brief The device abstract index setter.
    See the class description for details.
    """
    other_terminal_id: int
    r"""
    Getter:
    @brief The getter for the abstract's connected terminal.
    See the class description for details.
    Setter:
    @brief The setter for the abstract's connected terminal.
    See the class description for details.
    """
    @classmethod
    def new(cls) -> DeviceReconnectedTerminal:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> DeviceReconnectedTerminal:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceReconnectedTerminal:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DeviceReconnectedTerminal) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DeviceReconnectedTerminal:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class DeviceTerminalDefinition:
    r"""
    @brief A terminal descriptor
    This class is used inside the \DeviceClass class to describe a terminal of the device.

    This class has been added in version 0.26.
    """
    description: str
    r"""
    Getter:
    @brief Gets the description of the terminal.
    Setter:
    @brief Sets the description of the terminal.
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the terminal.
    Setter:
    @brief Sets the name of the terminal.
    """
    @classmethod
    def new(cls, name: str, description: Optional[str] = ...) -> DeviceTerminalDefinition:
        r"""
        @brief Creates a new terminal definition.
        """
    def __copy__(self) -> DeviceTerminalDefinition:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DeviceTerminalDefinition:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self, name: str, description: Optional[str] = ...) -> None:
        r"""
        @brief Creates a new terminal definition.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DeviceTerminalDefinition) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DeviceTerminalDefinition:
        r"""
        @brief Creates a copy of self
        """
    def id(self) -> int:
        r"""
        @brief Gets the ID of the terminal.
        The ID of the terminal is used in some places to refer to a specific terminal (e.g. in the \NetTerminalRef object).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class Edge:
    r"""
    @brief An edge class

    An edge is a connection between points, usually participating in a larger context such as a polygon. An edge has a defined direction (from p1 to p2). Edges play a role in the database as parts of polygons and to describe a line through both points.
    Although supported, edges are rarely used as individual database objects.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects like the Edge class.
    """
    p1: Point
    r"""
    Getter:
    @brief The first point.

    Setter:
    @brief Sets the first point.
    This method has been added in version 0.23.
    """
    p2: Point
    r"""
    Getter:
    @brief The second point.

    Setter:
    @brief Sets the second point.
    This method has been added in version 0.23.
    """
    x1: int
    r"""
    Getter:
    @brief Shortcut for p1.x

    Setter:
    @brief Sets p1.x
    This method has been added in version 0.23.
    """
    x2: int
    r"""
    Getter:
    @brief Shortcut for p2.x

    Setter:
    @brief Sets p2.x
    This method has been added in version 0.23.
    """
    y1: int
    r"""
    Getter:
    @brief Shortcut for p1.y

    Setter:
    @brief Sets p1.y
    This method has been added in version 0.23.
    """
    y2: int
    r"""
    Getter:
    @brief Shortcut for p2.y

    Setter:
    @brief Sets p2.y
    This method has been added in version 0.23.
    """
    @classmethod
    def from_dedge(cls, dedge: DEdge) -> Edge:
        r"""
        @brief Creates an integer coordinate edge from a floating-point coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge'.
        """
    @classmethod
    def from_s(cls, s: str) -> Edge:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Edge:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    @classmethod
    def new(cls, dedge: DEdge) -> Edge:
        r"""
        @brief Creates an integer coordinate edge from a floating-point coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge'.
        """
    @overload
    @classmethod
    def new(cls, p1: Point, p2: Point) -> Edge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    @classmethod
    def new(cls, x1: int, y1: int, x2: int, y2: int) -> Edge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    @classmethod
    def new_pp(cls, p1: Point, p2: Point) -> Edge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @classmethod
    def new_xyxy(cls, x1: int, y1: int, x2: int, y2: int) -> Edge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __copy__(self) -> Edge:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Edge:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, e: object) -> bool:
        r"""
        @brief Equality test
        @param e The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    def __init__(self, dedge: DEdge) -> None:
        r"""
        @brief Creates an integer coordinate edge from a floating-point coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge'.
        """
    @overload
    def __init__(self, p1: Point, p2: Point) -> None:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    def __init__(self, x1: int, y1: int, x2: int, y2: int) -> None:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __lt__(self, e: Edge) -> bool:
        r"""
        @brief Less operator
        @param e The object to compare against
        @return True, if the edge is 'less' as the other edge with respect to first and second point
        """
    def __mul__(self, scale_factor: float) -> Edge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __ne__(self, e: object) -> bool:
        r"""
        @brief Inequality test
        @param e The object to compare against
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __rmul__(self, scale_factor: float) -> Edge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: Edge) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Return the bounding box of the edge.
        """
    def clipped(self, box: Box) -> Any:
        r"""
        @brief Returns the edge clipped at the given box

        @param box The clip box.
        @return The clipped edge or nil if the edge does not intersect with the box.

        This method has been introduced in version 0.26.2.
        """
    def clipped_line(self, box: Box) -> Any:
        r"""
        @brief Returns the line through the edge clipped at the given box

        @param box The clip box.
        @return The part of the line through the box or nil if the line does not intersect with the box.

        In contrast to \clipped, this method will consider the edge extended infinitely (a "line"). The returned edge will be the part of this line going through the box.

        This method has been introduced in version 0.26.2.
        """
    def coincident(self, e: Edge) -> bool:
        r"""
        @brief Coincidence check.

        Checks whether a edge is coincident with another edge. 
        Coincidence is defined by being parallel and that 
        at least one point of one edge is on the other edge.

        @param e the edge to test with

        @return True if the edges are coincident.
        """
    def contains(self, p: Point) -> bool:
        r"""
        @brief Tests whether a point is on an edge.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge.
        """
    def contains_excl(self, p: Point) -> bool:
        r"""
        @brief Tests whether a point is on an edge excluding the endpoints.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge but not equal p1 or p2.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def crossed_by(self, e: Edge) -> bool:
        r"""
        @brief Checks, if the line given by self is crossed by the edge e

        self if considered an infinite line. This predicate renders true if the edge e is cut by this line. In other words: this method returns true if e.p1 is in one semispace of self 
        while e.p2 is in the other or one of them is exactly on self.

        @param e The edge representing the line that the edge must be crossing.
        """
    def crossing_point(self, e: Edge) -> Point:
        r"""
        @brief Returns the crossing point on two edges. 

        This method delivers the point where the given line (self) crosses the edge given by the argument "e". self is considered infinitely long and is required to cut through the edge "e". If self does not cut this line, the result is undefined. See \crossed_by? for a description of the crossing predicate.

        @param e The edge representing the line that self must be crossing.
        @return The point where self crosses the line given by "e".

        This method has been introduced in version 0.19.
        """
    def cut_point(self, e: Edge) -> Any:
        r"""
        @brief Returns the intersection point of the lines through the two edges.

        This method delivers the intersection point between the lines through the two edges. If the lines are parallel and do not intersect, the result will be nil.
        In contrast to \intersection_point, this method will regard the edges as infinitely extended and intersection is not confined to the edge span.

        @param e The edge to test.
        @return The point where the lines intersect.

        This method has been introduced in version 0.27.1.
        """
    def d(self) -> Vector:
        r"""
        @brief Gets the edge extension as a vector.
        This method is equivalent to p2 - p1.
        This method has been introduced in version 0.26.2.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, p: Point) -> int:
        r"""
        @brief Gets the distance of the point from the line through the edge.

        Returns the distance between the edge and the point. The 
        distance is signed which is negative if the point is to the
        "right" of the edge and positive if the point is to the "left".
        The distance is measured by projecting the point onto the
        line through the edge. If the edge is degenerated, the distance
        is not defined.

        This method considers the edge to define an infinite line running through it.
        \distance returns the distance of 'p' to this line.
        A similar method is \euclidian_distance, but the latter regards
        the edge a finite set of points between the endpoints.

        @param p The point to test.

        @return The distance
        """
    def distance_abs(self, p: Point) -> int:
        r"""
        @brief Absolute distance between the edge and a point.

        Returns the distance between the edge and the point. 

        @param p The point to test.

        @return The distance
        """
    def dup(self) -> Edge:
        r"""
        @brief Creates a copy of self
        """
    def dx(self) -> int:
        r"""
        @brief The horizontal extend of the edge.
        """
    def dx_abs(self) -> int:
        r"""
        @brief The absolute value of the horizontal extend of the edge.
        """
    def dy(self) -> int:
        r"""
        @brief The vertical extend of the edge.
        """
    def dy_abs(self) -> int:
        r"""
        @brief The absolute value of the vertical extend of the edge.
        """
    def enlarge(self, p: Vector) -> Edge:
        r"""
        @brief Enlarges the edge.

        Enlarges the edge by the given distance and returns the 
        enlarged edge. The edge is overwritten.
        Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def enlarged(self, p: Vector) -> Edge:
        r"""
        @brief Returns the enlarged edge (does not modify self)

        Enlarges the edge by the given offset and returns the 
        enlarged edge. The edge is not modified. Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def euclidian_distance(self, p: Point) -> int:
        r"""
        @brief Gets the distance of the point from the the edge.

        Returns the minimum distance of the point to any point on the edge.
        Unlike \distance, the edge is considered a finite set of points between
        the endpoints. The result is also not signed like it is the case for \distance.

        This method has been introduced in version 0.28.14.

        @param p The point to test.

        @return The distance
        """
    def extend(self, d: int) -> Edge:
        r"""
        @brief Extends the edge (modifies self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extended is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge (self).
        """
    def extended(self, d: int) -> Edge:
        r"""
        @brief Returns the extended edge (does not modify self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extend is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    def intersect(self, e: Edge) -> bool:
        r"""
        @brief Intersection test. 

        Returns true if the edges intersect. Two edges intersect if they share at least one point. 
        If the edges coincide, they also intersect.
        If one of the edges is degenerate (both points are identical), that point is required to sit exaclty on the other edge. If both edges are degenerate, their points are required to be identical.

        @param e The edge to test.

        The 'intersects' (with an 's') synonym has been introduced in version 0.28.12.
        """
    def intersection_point(self, e: Edge) -> Any:
        r"""
        @brief Returns the intersection point of two edges. 

        This method delivers the intersection point. If the edges do not intersect, the result will be nil.

        @param e The edge to test.
        @return The point where the edges intersect.

        This method has been introduced in version 0.19.
        From version 0.26.2, this method will return nil in case of non-intersection.
        """
    def intersects(self, e: Edge) -> bool:
        r"""
        @brief Intersection test. 

        Returns true if the edges intersect. Two edges intersect if they share at least one point. 
        If the edges coincide, they also intersect.
        If one of the edges is degenerate (both points are identical), that point is required to sit exaclty on the other edge. If both edges are degenerate, their points are required to be identical.

        @param e The edge to test.

        The 'intersects' (with an 's') synonym has been introduced in version 0.28.12.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_degenerate(self) -> bool:
        r"""
        @brief Test for degenerated edge

        An edge is degenerate, if both end and start point are identical.
        """
    def is_parallel(self, e: Edge) -> bool:
        r"""
        @brief Test for being parallel

        @param e The edge to test against

        @return True if both edges are parallel
        """
    def length(self) -> int:
        r"""
        @brief The length of the edge
        """
    @overload
    def move(self, dx: int, dy: int) -> Edge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    @overload
    def move(self, p: Vector) -> Edge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    @overload
    def moved(self, dx: int, dy: int) -> Edge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: Vector) -> Edge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    def ortho_length(self) -> int:
        r"""
        @brief The orthogonal length of the edge ("manhattan-length")

        @return The orthogonal length (abs(dx)+abs(dy))
        """
    def shift(self, d: int) -> Edge:
        r"""
        @brief Shifts the edge (modifies self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shifted is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge (self).
        """
    def shifted(self, d: int) -> Edge:
        r"""
        @brief Returns the shifted edge (does not modify self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shift is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge.
        """
    def side_of(self, p: Point) -> int:
        r"""
        @brief Indicates at which side the point is located relative to the edge.

        Returns 1 if the point is "left" of the edge, 0 if on
        and -1 if the point is "right" of the edge.

        @param p The point to test.

        @return The side value
        """
    def sq_length(self) -> int:
        r"""
        @brief The square of the length of the edge
        """
    def swap_points(self) -> Edge:
        r"""
        @brief Swap the points of the edge

        This version modifies self. A version that does not modify self is \swapped_points. Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def swapped_points(self) -> Edge:
        r"""
        @brief Returns an edge in which both points are swapped

        Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DEdge:
        r"""
        @brief Converts the edge to a floating-point coordinate edge

        The database unit can be specified to translate the integer-coordinate edge into a floating-point coordinate edge in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: CplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> Edge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge (in this case an integer coordinate edge).

        This method has been introduced in version 0.18.
        """
    @overload
    def transformed(self, t: Trans) -> Edge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    def transformed_cplx(self, t: CplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """

class EdgeFilter:
    r"""
    @brief A generic edge filter adaptor

    Edge filters are an efficient way to filter edge from a Edges collection. To apply a filter, derive your own filter class and pass an instance to the \Edges#filter or \Edges#filtered method.

    Conceptually, these methods take each edge from the collection and present it to the filter's 'selected' method.
    Based on the result of this evaluation, the edge is kept or discarded.

    The magic happens when deep mode edge collections are involved. In that case, the filter will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the filter behaves. You need to configure the filter by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using the filter.

    You can skip this step, but the filter algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    Here is some example that filters edges parallel to a given one:
    @code
    class ParallelFilter < RBA::EdgeFilter

      # Constructor
      def initialize(ref_edge)
        self.is_scale_invariant   # orientation matters, but scale does not
        @ref_edge = ref_edge
      end
  
      # Select only parallel ones
      def selected(edge)
        return edge.is_parallel?(@ref_edge)
      end

    end

    edges = ... # some Edges object
    ref_edge = ... # some Edge
    parallel_only = edges.filtered(ParallelFilter::new(ref_edge))
    @/code

    This class has been introduced in version 0.29.
    """
    requires_raw_input: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter needs raw (unmerged) input
    See \requires_raw_input= for details.

    Setter:
    @brief Sets a value indicating whether the filter needs raw (unmerged) input
    This flag must be set before using this filter. It tells the filter implementation whether the filter wants to have raw input (unmerged). The default value is 'false', meaning that
    the filter will receive merged polygons ('merged semantics').

    Setting this value to false potentially saves some CPU time needed for merging the polygons.
    Also, raw input means that strange shapes such as dot-like edges, self-overlapping polygons, empty or degenerated polygons are preserved.
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgeFilter:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) filters are area or perimeter filters. The area or perimeter of a polygon depends on the scale, but not on the orientation of the polygon.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) filter is the square selector. Whether a polygon is a square or not does not depend on the polygon's orientation nor scale.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) filter is the bounding box aspect ratio (height/width) filter. The definition of heigh and width depends on the orientation, but the ratio is independent on scale.
        """

class EdgeMode:
    r"""
    @brief This class represents the edge mode type for \Region#edges.

    This enum has been introduced in version 0.29.
    """
    All: ClassVar[EdgeMode]
    r"""
    @brief Selects all edges
    """
    Concave: ClassVar[EdgeMode]
    r"""
    @brief Selects only concave edges
    """
    Convex: ClassVar[EdgeMode]
    r"""
    @brief Selects only convex edges
    """
    NotConcave: ClassVar[EdgeMode]
    r"""
    @brief Selects only edges which are not concave
    """
    NotConvex: ClassVar[EdgeMode]
    r"""
    @brief Selects only edges which are not convex
    """
    NotStep: ClassVar[EdgeMode]
    r"""
    @brief Selects only edges which are not steps
    """
    NotStepIn: ClassVar[EdgeMode]
    r"""
    @brief Selects only edges which are not steps leading inside
    """
    NotStepOut: ClassVar[EdgeMode]
    r"""
    @brief Selects only edges which are not steps leading outside
    """
    Step: ClassVar[EdgeMode]
    r"""
    @brief Selects only step edges leading inside or outside
    """
    StepIn: ClassVar[EdgeMode]
    r"""
    @brief Selects only step edges leading inside
    """
    StepOut: ClassVar[EdgeMode]
    r"""
    @brief Selects only step edges leading outside
    """
    @overload
    @classmethod
    def new(cls, i: int) -> EdgeMode:
        r"""
        @brief Creates an enum from an integer value
        """
    @overload
    @classmethod
    def new(cls, s: str) -> EdgeMode:
        r"""
        @brief Creates an enum from a string value
        """
    def __copy__(self) -> EdgeMode:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgeMode:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two enums
        """
    @overload
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares an enum with an integer value
        """
    def __hash__(self) -> int:
        r"""
        @brief Gets the hash value from the enum
        """
    @overload
    def __init__(self, i: int) -> None:
        r"""
        @brief Creates an enum from an integer value
        """
    @overload
    def __init__(self, s: str) -> None:
        r"""
        @brief Creates an enum from a string value
        """
    def __int__(self) -> int:
        r"""
        @brief Gets the integer value from the enum
        """
    @overload
    def __lt__(self, other: EdgeMode) -> bool:
        r"""
        @brief Returns true if the first enum is less (in the enum symbol order) than the second
        """
    @overload
    def __lt__(self, other: int) -> bool:
        r"""
        @brief Returns true if the enum is less (in the enum symbol order) than the integer value
        """
    @overload
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares an enum with an integer for inequality
        """
    @overload
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two enums for inequality
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts an enum to a visual string
        """
    def __str__(self) -> str:
        r"""
        @brief Gets the symbolic string from an enum
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: EdgeMode) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> EdgeMode:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Gets the hash value from the enum
        """
    def inspect(self) -> str:
        r"""
        @brief Converts an enum to a visual string
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def to_i(self) -> int:
        r"""
        @brief Gets the integer value from the enum
        """
    def to_s(self) -> str:
        r"""
        @brief Gets the symbolic string from an enum
        """

class EdgeOperator:
    r"""
    @brief A generic edge-to-polygon operator

    Edge processors are an efficient way to process edges from an edge collection. To apply a processor, derive your own operator class and pass an instance to the \Edges#processed method.

    Conceptually, these methods take each edge from the edge collection and present it to the operator's 'process' method.
    The result of this call is a list of zero to many output edges derived from the input edge.
    The output edge collection is the sum over all these individual results.

    The magic happens when deep mode edge collections are involved. In that case, the processor will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the operator behaves. You need to configure the operator by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using it.

    You can skip this step, but the processor algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    Here is some example that shrinks every edge to half of the size, but does not change the position.
    In this example the 'position' is defined by the center of the edge:
    @code
    class ShrinkToHalf < RBA::EdgeOperator

      # Constructor
      def initialize
        self.is_isotropic_and_scale_invariant   # scale or orientation do not matter
      end
  
      # Shrink to half size
      def process(edge)
        shift = edge.bbox.center - RBA::Point::new   # shift vector
        return [ (edge.moved(-shift) * 0.5).moved(shift) ]
      end

    end

    edges = ... # some Edges collection
    shrinked_to_half = edges.processed(ShrinkToHalf::new)
    @/code

    This class has been introduced in version 0.29.
    """
    requires_raw_input: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor needs raw (unmerged) input
    See \requires_raw_input= for details.

    Setter:
    @brief Sets a value indicating whether the processor needs raw (unmerged) input
    This flag must be set before using this processor. It tells the processor implementation whether the processor wants to have raw input (unmerged). The default value is 'false', meaning that
    the processor will receive merged polygons ('merged semantics').

    Setting this value to false potentially saves some CPU time needed for merging the polygons.
    Also, raw input means that strange shapes such as dot-like edges, self-overlapping polygons, empty or degenerated polygons are preserved.
    """
    result_is_merged: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor delivers merged output
    See \result_is_merged= for details.

    Setter:
    @brief Sets a value indicating whether the processor delivers merged output
    This flag must be set before using this processor. If the processor maintains the merged condition
    by design (output is merged if input is), it is a good idea to set this predicate to 'true'.
    This will avoid additional merge steps when the resulting collection is used in further operations
    that need merged input
    .
    """
    result_must_not_be_merged: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor's output must not be merged
    See \result_must_not_be_merged= for details.

    Setter:
    @brief Sets a value indicating whether the processor's output must not be merged
    This flag must be set before using this processor. The processor can set this flag if it wants to
    deliver shapes that must not be merged - e.g. point-like edges or strange or degenerated polygons.
    .
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgeOperator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) processors are size or shrink operators. Size or shrink is not dependent on orientation unless size or shrink needs to be different in x and y direction.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) processor is the convex decomposition operator. The decomposition of a polygon into convex parts is an operation that is not depending on scale nor orientation.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) processor is the rotation operator. Rotation is not depending on scale, but on the original orientation as mirrored versions need to be rotated differently.
        """

class EdgePair:
    r"""
    @brief An edge pair (a pair of two edges)
    Edge pairs are objects representing two edges or parts of edges. They play a role mainly in the context of DRC functions, where they specify a DRC violation by connecting two edges which violate the condition checked. Within the framework of polygon and edge collections which provide DRC functionality, edges pairs are used in the form of edge pair collections (\EdgePairs).

    Edge pairs basically consist of two edges, called first and second. If created by a two-layer DRC function, the first edge will correspond to edges from the first layer and the second to edges from the second layer.

    This class has been introduced in version 0.23.
    """
    first: Edge
    r"""
    Getter:
    @brief Gets the first edge

    Setter:
    @brief Sets the first edge
    """
    second: Edge
    r"""
    Getter:
    @brief Gets the second edge

    Setter:
    @brief Sets the second edge
    """
    symmetric: bool
    r"""
    Getter:
    @brief Returns a value indicating whether the edge pair is symmetric
    For symmetric edge pairs, the edges are commutable. Specifically, a symmetric edge pair with (e1,e2) is identical to (e2,e1). Symmetric edge pairs are generated by some checks for which there is no directed error marker (width, space, notch, isolated).

    Symmetric edge pairs have been introduced in version 0.27.

    Setter:
    @brief Sets a value indicating whether the edge pair is symmetric
    See \symmetric? for a description of this attribute.

    Symmetric edge pairs have been introduced in version 0.27.
    """
    @classmethod
    def from_s(cls, s: str) -> EdgePair:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> EdgePair:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    @classmethod
    def new(cls, dedge_pair: DEdgePair) -> EdgePair:
        r"""
        @brief Creates an integer coordinate edge pair from a floating-point coordinate edge pair

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge_pair'.
        """
    @overload
    @classmethod
    def new(cls, first: Edge, second: Edge, symmetric: Optional[bool] = ...) -> EdgePair:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __copy__(self) -> EdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Equality
        Returns true, if this edge pair and the given one are equal

        This method has been introduced in version 0.25.
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    def __init__(self, dedge_pair: DEdgePair) -> None:
        r"""
        @brief Creates an integer coordinate edge pair from a floating-point coordinate edge pair

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge_pair'.
        """
    @overload
    def __init__(self, first: Edge, second: Edge, symmetric: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __lt__(self, box: EdgePair) -> bool:
        r"""
        @brief Less operator
        Returns true, if this edge pair is 'less' with respect to first and second edge

        This method has been introduced in version 0.25.
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Inequality
        Returns true, if this edge pair and the given one are not equal

        This method has been introduced in version 0.25.
        """
    def __repr__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> int:
        r"""
        @brief Gets the area between the edges of the edge pair

        This attribute has been introduced in version 0.28.
        """
    def assign(self, other: EdgePair) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Gets the bounding box of the edge pair
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self) -> int:
        r"""
        @brief Gets the distance of the edges in the edge pair

        The distance between the two edges is defined as the minimum distance between any two points on the two edges.

        This attribute has been introduced in version 0.28.14.
        """
    def dup(self) -> EdgePair:
        r"""
        @brief Creates a copy of self
        """
    def greater(self) -> Edge:
        r"""
        @brief Gets the 'greater' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'second' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def lesser(self) -> Edge:
        r"""
        @brief Gets the 'lesser' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'first' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def normalized(self) -> EdgePair:
        r"""
        @brief Normalizes the edge pair
        This method normalized the edge pair such that when connecting the edges at their 
        start and end points a closed loop is formed which is oriented clockwise. To achieve this, the points of the first and/or first and second edge are swapped. Normalization is a first step recommended before converting an edge pair to a polygon, because that way the polygons won't be self-overlapping and the enlargement parameter is applied properly.
        """
    def perimeter(self) -> int:
        r"""
        @brief Gets the perimeter of the edge pair

        The perimeter is defined as the sum of the lengths of both edges ('active perimeter').

        This attribute has been introduced in version 0.28.
        """
    def polygon(self, e: int) -> Polygon:
        r"""
        @brief Convert an edge pair to a polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to simple polygons is \simple_polygon which renders a \SimplePolygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def simple_polygon(self, e: int) -> SimplePolygon:
        r"""
        @brief Convert an edge pair to a simple polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to polygons is \polygon which renders a \Polygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DEdgePair:
        r"""
        @brief Converts the edge pair to a floating-point coordinate edge pair

        The database unit can be specified to translate the integer-coordinate edge pair into a floating-point coordinate edge pair in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: CplxTrans) -> DEdgePair:
        r"""
        @brief Returns the transformed edge pair

        Transforms the edge pair with the given complex transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """
    @overload
    def transformed(self, t: ICplxTrans) -> EdgePair:
        r"""
        @brief Returns the transformed edge pair

        Transforms the edge pair with the given complex transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair (in this case an integer coordinate edge pair).
        """
    @overload
    def transformed(self, t: Trans) -> EdgePair:
        r"""
        @brief Returns the transformed pair

        Transforms the edge pair with the given transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """

class EdgePairFilter:
    r"""
    @brief A generic edge pair filter adaptor

    EdgePair filters are an efficient way to filter edge pairs from a EdgePairs collection. To apply a filter, derive your own filter class and pass an instance to \EdgePairs#filter or \EdgePairs#filtered method.

    Conceptually, these methods take each edge pair from the collection and present it to the filter's 'selected' method.
    Based on the result of this evaluation, the edge pair is kept or discarded.

    The magic happens when deep mode edge pair collections are involved. In that case, the filter will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the filter behaves. You need to configure the filter by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using the filter.

    You can skip this step, but the filter algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    Here is some example that filters edge pairs where the edges are perpendicular:
    @code
    class PerpendicularEdgesFilter < RBA::EdgePairFilter

      # Constructor
      def initialize
        self.is_isotropic_and_scale_invariant   # orientation and scale do not matter
      end
  
      # Select edge pairs where the edges are perpendicular
      def selected(edge_pair)
        return edge_pair.first.d.sprod_sign(edge_pair.second.d) == 0
      end

    end

    edge_pairs = ... # some EdgePairs object
    perpendicular_only = edge_pairs.filtered(PerpendicularEdgesFilter::new)
    @/code

    This class has been introduced in version 0.29.
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgePairFilter:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) filters are area or perimeter filters. The area or perimeter of a polygon depends on the scale, but not on the orientation of the polygon.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) filter is the square selector. Whether a polygon is a square or not does not depend on the polygon's orientation nor scale.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) filter is the bounding box aspect ratio (height/width) filter. The definition of heigh and width depends on the orientation, but the ratio is independent on scale.
        """

class EdgePairOperator:
    r"""
    @brief A generic edge-pair operator

    Edge pair processors are an efficient way to process edge pairs from an edge pair collection. To apply a processor, derive your own operator class and pass an instance to the \EdgePairs#processed or \EdgePairs#process method.

    Conceptually, these methods take each edge pair from the edge pair collection and present it to the operator's 'process' method.
    The result of this call is a list of zero to many output edge pairs derived from the input edge pair.
    The output edge pair collection is the sum over all these individual results.

    The magic happens when deep mode edge pair collections are involved. In that case, the processor will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the operator behaves. You need to configure the operator by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using it.

    You can skip this step, but the processor algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    Here is some example that flips the edge pairs (swaps first and second edge):
    @code
    class FlipEdgePairs < RBA::EdgePairOperator

      # Constructor
      def initialize
        self.is_isotropic_and_scale_invariant   # orientation and scale do not matter
      end
  
      # Flips the edge pair
      def process(edge_pair)
        return [ RBA::EdgePair::new(edge_pair.second, edge_pair.first) ]
      end

    end

    edge_pairs = ... # some EdgePairs object
    flipped = edge_pairs.processed(FlipEdgePairs::new)
    @/code

    This class has been introduced in version 0.29.
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgePairOperator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) processors are size or shrink operators. Size or shrink is not dependent on orientation unless size or shrink needs to be different in x and y direction.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) processor is the convex decomposition operator. The decomposition of a polygon into convex parts is an operation that is not depending on scale nor orientation.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) processor is the rotation operator. Rotation is not depending on scale, but on the original orientation as mirrored versions need to be rotated differently.
        """

class EdgePairToEdgeOperator:
    r"""
    @brief A generic edge-pair-to-edge operator

    Edge processors are an efficient way to process edge pairs from an edge pair collection. To apply a processor, derive your own operator class and pass an instance to \EdgePairs#processed method.

    Conceptually, these methods take each edge from the edge collection and present it to the operator's 'process' method.
    The result of this call is a list of zero to many output edges derived from the input edge pair.
    The output edge pair collection is the sum over all these individual results.

    The magic happens when deep mode edge pair collections are involved. In that case, the processor will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the operator behaves. You need to configure the operator by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using it.

    You can skip this step, but the processor algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    For a basic example see the \EdgeToEdgePairOperator class, with the exception that this incarnation has to deliver edges and takes edge pairs.

    This class has been introduced in version 0.29.
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgePairToEdgeOperator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) processors are size or shrink operators. Size or shrink is not dependent on orientation unless size or shrink needs to be different in x and y direction.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) processor is the convex decomposition operator. The decomposition of a polygon into convex parts is an operation that is not depending on scale nor orientation.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) processor is the rotation operator. Rotation is not depending on scale, but on the original orientation as mirrored versions need to be rotated differently.
        """

class EdgePairToPolygonOperator:
    r"""
    @brief A generic edge-pair-to-polygon operator

    Edge pair processors are an efficient way to process edge pairs from an edge pair collection. To apply a processor, derive your own operator class and pass an instance to the \EdgePairs#processed method.

    Conceptually, these methods take each edge pair from the edge pair collection and present it to the operator's 'process' method.
    The result of this call is a list of zero to many output polygons derived from the input edge pair.
    The output region is the sum over all these individual results.

    The magic happens when deep mode edge pair collections are involved. In that case, the processor will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the operator behaves. You need to configure the operator by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using it.

    You can skip this step, but the processor algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    For a basic example see the \EdgeToPolygonOperator class, with the exception that this incarnation receives edge pairs.

    This class has been introduced in version 0.29.
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgePairToPolygonOperator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) processors are size or shrink operators. Size or shrink is not dependent on orientation unless size or shrink needs to be different in x and y direction.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) processor is the convex decomposition operator. The decomposition of a polygon into convex parts is an operation that is not depending on scale nor orientation.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) processor is the rotation operator. Rotation is not depending on scale, but on the original orientation as mirrored versions need to be rotated differently.
        """

class EdgePairs(ShapeCollection):
    r"""
    @brief EdgePairs (a collection of edge pairs)

    Edge pairs are used mainly in the context of the DRC functions (width_check, space_check etc.) of \Region and \Edges. A single edge pair represents two edges participating in a DRC violation. In the two-layer checks (inside, overlap) The first edge represents an edge from the first layer and the second edge an edge from the second layer. For single-layer checks (width, space) the order of the edges is arbitrary.

    This class has been introduced in version 0.23.
    """
    @overload
    @classmethod
    def new(cls) -> EdgePairs:
        r"""
        @brief Default constructor

        This constructor creates an empty edge pair collection.
        """
    @overload
    @classmethod
    def new(cls, array: Sequence[EdgePair]) -> EdgePairs:
        r"""
        @brief Constructor from an edge pair array

        This constructor creates an edge pair collection from an array of \EdgePair objects.

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, edge_pair: EdgePair) -> EdgePairs:
        r"""
        @brief Constructor from a single edge pair object

        This constructor creates an edge pair collection with a single edge pair.

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator) -> EdgePairs:
        r"""
        @brief Constructor from a hierarchical shape set

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore) -> EdgePairs:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans) -> EdgePairs:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans) -> EdgePairs:
        r"""
        @brief Constructor from a hierarchical shape set with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        The given transformation is applied to each edge pair taken.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shapes: Shapes) -> EdgePairs:
        r"""
        @brief Shapes constructor

        This constructor creates an edge pair collection from a \Shapes collection.

        This constructor has been introduced in version 0.26.
        """
    def __add__(self, other: EdgePairs) -> EdgePairs:
        r"""
        @brief Returns the combined edge pair collection of self and the other one

        @return The resulting edge pair collection

        This operator adds the edge pairs of the other collection to self and returns a new combined set.

        This method has been introduced in version 0.24.
        The 'join' alias has been introduced in version 0.28.12.
        """
    def __copy__(self) -> EdgePairs:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgePairs:
        r"""
        @brief Creates a copy of self
        """
    def __getitem__(self, n: int) -> EdgePair:
        r"""
        @brief Returns the nth edge pair

        This method returns nil if the index is out of range. It is available for flat edge pairs only - i.e. those for which \has_valid_edge_pairs? is true. Use \flatten to explicitly flatten an edge pair collection.

        The \each iterator is the more general approach to access the edge pairs.
        """
    def __iadd__(self, other: EdgePairs) -> EdgePairs:
        r"""
        @brief Adds the edge pairs of the other edge pair collection to self

        @return The edge pair collection after modification (self)

        This operator adds the edge pairs of the other collection to self.

        This method has been introduced in version 0.24.

        Note that in Ruby, the '+=' operator actually does not exist, but is emulated by '+' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'join_with' instead.

        The 'join_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an empty edge pair collection.
        """
    @overload
    def __init__(self, array: Sequence[EdgePair]) -> None:
        r"""
        @brief Constructor from an edge pair array

        This constructor creates an edge pair collection from an array of \EdgePair objects.

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, edge_pair: EdgePair) -> None:
        r"""
        @brief Constructor from a single edge pair object

        This constructor creates an edge pair collection with a single edge pair.

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator) -> None:
        r"""
        @brief Constructor from a hierarchical shape set

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore) -> None:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans) -> None:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans) -> None:
        r"""
        @brief Constructor from a hierarchical shape set with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        The given transformation is applied to each edge pair taken.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shapes: Shapes) -> None:
        r"""
        @brief Shapes constructor

        This constructor creates an edge pair collection from a \Shapes collection.

        This constructor has been introduced in version 0.26.
        """
    def __iter__(self) -> Iterator[EdgePair]:
        r"""
        @brief Returns each edge pair of the edge pair collection
        """
    def __len__(self) -> int:
        r"""
        @brief Returns the (flat) number of edge pairs in the edge pair collection

        The count is computed 'as if flat', i.e. edge pairs inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        The length of the output is limited to 20 edge pairs to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        The length of the output is limited to 20 edge pairs to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: ShapeCollection) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Return the bounding box of the edge pair collection
        The bounding box is the box enclosing all points of all edge pairs.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the edge pair collection
        """
    def count(self) -> int:
        r"""
        @brief Returns the (flat) number of edge pairs in the edge pair collection

        The count is computed 'as if flat', i.e. edge pairs inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def data_id(self) -> int:
        r"""
        @brief Returns the data ID (a unique identifier for the underlying data storage)

        This method has been added in version 0.26.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disable_progress(self) -> None:
        r"""
        @brief Disable progress reporting
        Calling this method will disable progress reporting. See \enable_progress.
        """
    def dup(self) -> EdgePairs:
        r"""
        @brief Creates a copy of self
        """
    def each(self) -> Iterator[EdgePair]:
        r"""
        @brief Returns each edge pair of the edge pair collection
        """
    def edges(self) -> Edges:
        r"""
        @brief Decomposes the edge pairs into single edges
        @return An edge collection containing the individual edges
        """
    def enable_progress(self, label: str) -> None:
        r"""
        @brief Enable progress reporting
        After calling this method, the edge pair collection will report the progress through a progress bar while expensive operations are running.
        The label is a text which is put in front of the progress bar.
        Using a progress bar will imply a performance penalty of a few percent typically.
        """
    def enable_properties(self) -> None:
        r"""
        @brief Enables properties for the given container.
        This method has an effect mainly on original layers and will import properties from such layers. By default, properties are not enabled on original layers. Alternatively you can apply \filter_properties or \map_properties to enable properties with a specific name key.

        This method has been introduced in version 0.28.4.
        """
    @overload
    def extents(self) -> Region:
        r"""
        @brief Returns a region with the bounding boxes of the edge pairs
        This method will return a region consisting of the bounding boxes of the edge pairs.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    @overload
    def extents(self, d: int) -> Region:
        r"""
        @brief Returns a region with the enlarged bounding boxes of the edge pairs
        This method will return a region consisting of the bounding boxes of the edge pairs enlarged by the given distance d.
        The enlargement is specified per edge, i.e the width and height will be increased by 2*d.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    @overload
    def extents(self, dx: int, dy: int) -> Region:
        r"""
        @brief Returns a region with the enlarged bounding boxes of the edge pairs
        This method will return a region consisting of the bounding boxes of the edge pairs enlarged by the given distance dx in x direction and dy in y direction.
        The enlargement is specified per edge, i.e the width will be increased by 2*dx.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    def filter(self, filter: EdgePairFilter) -> None:
        r"""
        @brief Applies a generic filter in place (replacing the edge pairs from the EdgePair collection)
        See \EdgePairFilter for a description of this feature.

        This method has been introduced in version 0.29.
        """
    def filter_properties(self, keys: Sequence[Any]) -> None:
        r"""
        @brief Filters properties by certain keys.
        Calling this method on a container will reduce the properties to values with name keys from the 'keys' list.
        As a side effect, this method enables properties on original layers.

        This method has been introduced in version 0.28.4.
        """
    def filtered(self, filtered: EdgePairFilter) -> EdgePairs:
        r"""
        @brief Applies a generic filter and returns a filtered copy
        See \EdgePairFilter for a description of this feature.

        This method has been introduced in version 0.29.
        """
    def first_edges(self) -> Edges:
        r"""
        @brief Returns the first one of all edges
        @return An edge collection containing the first edges
        """
    def flatten(self) -> None:
        r"""
        @brief Explicitly flattens an edge pair collection

        If the collection is already flat (i.e. \has_valid_edge_pairs? returns true), this method will not change the collection.

        This method has been introduced in version 0.26.
        """
    def has_valid_edge_pairs(self) -> bool:
        r"""
        @brief Returns true if the edge pair collection is flat and individual edge pairs can be accessed randomly

        This method has been introduced in version 0.26.
        """
    def hier_count(self) -> int:
        r"""
        @brief Returns the (hierarchical) number of edge pairs in the edge pair collection

        The count is computed 'hierarchical', i.e. edge pairs inside a cell are counted once even if the cell is instantiated multiple times.

        This method has been introduced in version 0.27.
        """
    @overload
    def insert(self, edge_pair: EdgePair) -> None:
        r"""
        @brief Inserts an edge pair into the collection
        """
    @overload
    def insert(self, edge_pairs: EdgePairs) -> None:
        r"""
        @brief Inserts all edge pairs from the other edge pair collection into this edge pair collection
        This method has been introduced in version 0.25.
        """
    @overload
    def insert(self, first: Edge, second: Edge) -> None:
        r"""
        @brief Inserts an edge pair into the collection
        """
    def insert_into(self, layout: Layout, cell_index: int, layer: int) -> None:
        r"""
        @brief Inserts this edge pairs into the given layout, below the given cell and into the given layer.
        If the edge pair collection is a hierarchical one, a suitable hierarchy will be built below the top cell or and existing hierarchy will be reused.

        This method has been introduced in version 0.26.
        """
    def insert_into_as_polygons(self, layout: Layout, cell_index: int, layer: int, e: int) -> None:
        r"""
        @brief Inserts this edge pairs into the given layout, below the given cell and into the given layer.
        If the edge pair collection is a hierarchical one, a suitable hierarchy will be built below the top cell or and existing hierarchy will be reused.

        The edge pairs will be converted to polygons with the enlargement value given be 'e'.

        This method has been introduced in version 0.26.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_deep(self) -> bool:
        r"""
        @brief Returns true if the edge pair collection is a deep (hierarchical) one

        This method has been added in version 0.26.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns true if the collection is empty
        """
    def join(self, other: EdgePairs) -> EdgePairs:
        r"""
        @brief Returns the combined edge pair collection of self and the other one

        @return The resulting edge pair collection

        This operator adds the edge pairs of the other collection to self and returns a new combined set.

        This method has been introduced in version 0.24.
        The 'join' alias has been introduced in version 0.28.12.
        """
    def join_with(self, other: EdgePairs) -> EdgePairs:
        r"""
        @brief Adds the edge pairs of the other edge pair collection to self

        @return The edge pair collection after modification (self)

        This operator adds the edge pairs of the other collection to self.

        This method has been introduced in version 0.24.

        Note that in Ruby, the '+=' operator actually does not exist, but is emulated by '+' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'join_with' instead.

        The 'join_with' alias has been introduced in version 0.28.12.
        """
    def map_properties(self, key_map: Dict[Any, Any]) -> None:
        r"""
        @brief Maps properties by name key.
        Calling this method on a container will reduce the properties to values with name keys from the 'keys' hash and renames the properties. Properties not listed in the key map will be removed.
        As a side effect, this method enables properties on original layers.

        This method has been introduced in version 0.28.4.
        """
    @overload
    def move(self, p: Vector) -> EdgePairs:
        r"""
        @brief Moves the edge pair collection

        Moves the edge pairs by the given offset and returns the 
        moved edge pair collection. The edge pair collection is overwritten.

        @param p The distance to move the edge pairs.

        @return The moved edge pairs (self).

        Starting with version 0.25 the displacement is of vector type.
        """
    @overload
    def move(self, x: int, y: int) -> EdgePairs:
        r"""
        @brief Moves the edge pair collection

        Moves the edge pairs by the given offset and returns the 
        moved edge pairs. The edge pair collection is overwritten.

        @param x The x distance to move the edge pairs.
        @param y The y distance to move the edge pairs.

        @return The moved edge pairs (self).
        """
    @overload
    def moved(self, p: Vector) -> EdgePairs:
        r"""
        @brief Returns the moved edge pair collection (does not modify self)

        Moves the edge pairs by the given offset and returns the 
        moved edge pairs. The edge pair collection is not modified.

        @param p The distance to move the edge pairs.

        @return The moved edge pairs.

        Starting with version 0.25 the displacement is of vector type.
        """
    @overload
    def moved(self, x: int, y: int) -> EdgePairs:
        r"""
        @brief Returns the moved edge pair collection (does not modify self)

        Moves the edge pairs by the given offset and returns the 
        moved edge pairs. The edge pair collection is not modified.

        @param x The x distance to move the edge pairs.
        @param y The y distance to move the edge pairs.

        @return The moved edge pairs.
        """
    @overload
    def polygons(self) -> Region:
        r"""
        @brief Converts the edge pairs to polygons
        This method creates polygons from the edge pairs. Each polygon will be a triangle or quadrangle which connects the start and end points of the edges forming the edge pair.
        """
    @overload
    def polygons(self, e: int) -> Region:
        r"""
        @brief Converts the edge pairs to polygons
        This method creates polygons from the edge pairs. Each polygon will be a triangle or quadrangle which connects the start and end points of the edges forming the edge pair. This version allows one to specify an enlargement which is applied to the edges. The length of the edges is modified by applying the enlargement and the edges are shifted by the enlargement. By specifying an enlargement it is possible to give edge pairs an area which otherwise would not have one (coincident edges, two point-like edges).
        """
    def process(self, process: EdgePairOperator) -> None:
        r"""
        @brief Applies a generic edge pair processor in place (replacing the edge pairs from the EdgePairs collection)
        See \EdgePairProcessor for a description of this feature.

        This method has been introduced in version 0.29.
        """
    @overload
    def processed(self, processed: EdgePairOperator) -> EdgePairs:
        r"""
        @brief Applies a generic edge pair processor and returns a processed copy
        See \EdgePairProcessor for a description of this feature.

        This method has been introduced in version 0.29.
        """
    @overload
    def processed(self, processed: EdgePairToEdgeOperator) -> Edges:
        r"""
        @brief Applies a generic edge-pair-to-edge processor and returns an edge collection with the results
        See \EdgePairToEdgeProcessor for a description of this feature.

        This method has been introduced in version 0.29.
        """
    @overload
    def processed(self, processed: EdgePairToPolygonOperator) -> Region:
        r"""
        @brief Applies a generic edge-pair-to-polygon processor and returns an Region with the results
        See \EdgePairToPolygonProcessor for a description of this feature.

        This method has been introduced in version 0.29.
        """
    def remove_properties(self) -> None:
        r"""
        @brief Removes properties for the given container.
        This will remove all properties on the given container.

        This method has been introduced in version 0.28.4.
        """
    def second_edges(self) -> Edges:
        r"""
        @brief Returns the second one of all edges
        @return An edge collection containing the second edges
        """
    def size(self) -> int:
        r"""
        @brief Returns the (flat) number of edge pairs in the edge pair collection

        The count is computed 'as if flat', i.e. edge pairs inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def swap(self, other: EdgePairs) -> None:
        r"""
        @brief Swap the contents of this collection with the contents of another collection
        This method is useful to avoid excessive memory allocation in some cases. For managed memory languages such as Ruby, those cases will be rare. 
        """
    @overload
    def to_s(self) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        The length of the output is limited to 20 edge pairs to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    @overload
    def to_s(self, max_count: int) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        This version allows specification of the maximum number of edge pairs contained in the string.
        """
    @overload
    def transform(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation (modifies self)

        Transforms the edge pair collection with the given transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.
        """
    @overload
    def transform(self, t: IMatrix2d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection (modifies self)

        Transforms the edge pair collection with the given 2d matrix transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transform(self, t: IMatrix3d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection (modifies self)

        Transforms the edge pair collection with the given 3d matrix transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transform(self, t: Trans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection (modifies self)

        Transforms the edge pair collection with the given transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.
        """
    def transform_icplx(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation (modifies self)

        Transforms the edge pair collection with the given transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation

        Transforms the edge pairs with the given complex transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.
        """
    @overload
    def transformed(self, t: IMatrix2d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection

        Transforms the edge pairs with the given 2d matrix transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transformed(self, t: IMatrix3d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection

        Transforms the edge pairs with the given 3d matrix transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transformed(self, t: Trans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection

        Transforms the edge pairs with the given transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.
        """
    def transformed_icplx(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation

        Transforms the edge pairs with the given complex transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.
        """
    @overload
    def with_abs_angle(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges

        This method behaves like \with_angle, but angles are always positive - i.e. there is no differentiation between edges sloping 'down' vs. edges sloping 'up.

        This method has been added in version 0.29.1.
        """
    @overload
    def with_abs_angle(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges

        This method behaves like \with_angle, but angles are always positive - i.e. there is no differentiation between edges sloping 'down' vs. edges sloping 'up.

        This method has been added in version 0.29.1.
        """
    @overload
    def with_abs_angle_both(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of both of their edges

        This method behaves like \with_angle_both, but angles are always positive - i.e. there is no differentiation between edges sloping 'down' vs. edges sloping 'up.

        This method has been added in version 0.29.1.
        """
    @overload
    def with_abs_angle_both(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""

        This method behaves like \with_angle_both, but angles are always positive - i.e. there is no differentiation between edges sloping 'down' vs. edges sloping 'up.

        This method has been added in version 0.29.1.
        """
    @overload
    def with_angle(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with at least one edge having the given angle to the x-axis are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This will filter edge pairs with at least one horizontal edge:

        @code
        horizontal = edge_pairs.with_angle(0, false)
        @/code

        Note that the inverse @b result @/b of \with_angle is delivered by \with_angle_both with the inverse flag set as edge pairs are unselected when both edges fail to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle(0, false)
        others = edge_pairs.with_angle_both(0, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with at least one edge having an angle between min_angle and max_angle are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        Note that the inverse @b result @/b of \with_angle is delivered by \with_angle_both with the inverse flag set as edge pairs are unselected when both edges fail to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle(0, 45, false)
        others = edge_pairs.with_angle_both(0, 45, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle(self, type: Edges.EdgeType, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with at least one edge having an angle of the given type are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This version allows specifying an edge type instead of an angle. Edge types include multiple distinct orientations and are specified using one of the \Edges#OrthoEdges, \Edges#DiagonalEdges or \Edges#OrthoDiagonalEdges types.

        Note that the inverse @b result @/b of \with_angle is delivered by \with_angle_both with the inverse flag set as edge pairs are unselected when both edges fail to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle(RBA::Edges::Ortho, false)
        others = edge_pairs.with_angle_both(RBA::Edges::Ortho, true)
        @/code

        This method has been added in version 0.28.
        """
    @overload
    def with_angle_both(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of both of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with both edges having the given angle to the x-axis are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This will filter edge pairs with at least one horizontal edge:

        @code
        horizontal = edge_pairs.with_angle_both(0, false)
        @/code

        Note that the inverse @b result @/b of \with_angle_both is delivered by \with_angle with the inverse flag set as edge pairs are unselected when one edge fails to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle_both(0, false)
        others = edge_pairs.with_angle(0, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle_both(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of both of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with both edges having an angle between min_angle and max_angle are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        Note that the inverse @b result @/b of \with_angle_both is delivered by \with_angle with the inverse flag set as edge pairs are unselected when one edge fails to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle_both(0, 45, false)
        others = edge_pairs.with_angle(0, 45, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle_both(self, type: Edges.EdgeType, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with both edges having an angle of the given type are returned. If "inverse" is true, edge pairs not fulfilling this criterion for both edges are returned.

        This version allows specifying an edge type instead of an angle. Edge types include multiple distinct orientations and are specified using one of the \Edges#OrthoEdges, \Edges#DiagonalEdges or \Edges#OrthoDiagonalEdges types.

        Note that the inverse @b result @/b of \with_angle_both is delivered by \with_angle with the inverse flag set as edge pairs are unselected when one edge fails to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle_both(RBA::Edges::Ortho, false)
        others = edge_pairs.with_angle(RBA::Edges::Ortho, true)
        @/code

        This method has been added in version 0.28.
        """
    @overload
    def with_area(self, area: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the enclosed area
        Filters the edge pairs in the edge pair collection by enclosed area. If "inverse" is false, only edge pairs with the given area are returned. If "inverse" is true, edge pairs not with the given area are returned.

        This method has been added in version 0.27.2.
        """
    @overload
    def with_area(self, min_area: int, max_area: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the enclosed area
        Filters the edge pairs in the edge pair collection by enclosed area. If "inverse" is false, only edge pairs with an area between min_area and max_area (max_area itself is excluded) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This method has been added in version 0.27.2.
        """
    @overload
    def with_distance(self, distance: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the distance of the edges
        Filters the edge pairs in the edge pair collection by distance of the edges. If "inverse" is false, only edge pairs where both edges have the given distance are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        Distance is measured as the shortest distance between any of the points on the edges.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_distance(self, min_distance: Any, max_distance: Any, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the distance of the edges
        Filters the edge pairs in the edge pair collection by distance of the edges. If "inverse" is false, only edge pairs where both edges have a distance between min_distance and max_distance (max_distance itself is excluded) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        Distance is measured as the shortest distance between any of the points on the edges.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_internal_angle(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the angle between their edges
        Filters the edge pairs in the edge pair collection by the angle between their edges. If "inverse" is false, only edge pairs with the given angle are returned. If "inverse" is true, edge pairs not with the given angle are returned.

        The angle is measured between the two edges. It is between 0 (parallel or anti-parallel edges) and 90 degree (perpendicular edges).

        This method has been added in version 0.27.2.
        """
    @overload
    def with_internal_angle(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the angle between their edges
        Filters the edge pairs in the edge pair collection by the angle between their edges. If "inverse" is false, only edge pairs with an angle between min_angle and max_angle (max_angle itself is excluded) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        The angle is measured between the two edges. It is between 0 (parallel or anti-parallel edges) and 90 degree (perpendicular edges).

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        This method has been added in version 0.27.2.
        """
    @overload
    def with_length(self, length: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of one of their edges
        Filters the edge pairs in the edge pair collection by length of at least one of their edges. If "inverse" is false, only edge pairs with at least one edge having the given length are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_length(self, min_length: Any, max_length: Any, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of one of their edges
        Filters the edge pairs in the edge pair collection by length of at least one of their edges. If "inverse" is false, only edge pairs with at least one edge having a length between min_length and max_length (excluding max_length itself) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        If you don't want to specify a lower or upper limit, pass nil to that parameter.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_length_both(self, length: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of both of their edges
        Filters the edge pairs in the edge pair collection by length of both of their edges. If "inverse" is false, only edge pairs where both edges have the given length are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_length_both(self, min_length: Any, max_length: Any, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of both of their edges
        Filters the edge pairs in the edge pair collection by length of both of their edges. If "inverse" is false, only edge pairs with both edges having a length between min_length and max_length (excluding max_length itself) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        If you don't want to specify a lower or upper limit, pass nil to that parameter.

        This method has been added in version 0.27.1.
        """
    def write(self, filename: str) -> None:
        r"""
        @brief Writes the region to a file
        This method is provided for debugging purposes. It writes the object to a flat layer 0/0 in a single top cell.

        This method has been introduced in version 0.29.
        """

class EdgeProcessor:
    r"""
    @brief The edge processor (boolean, sizing, merge)

    The edge processor implements the boolean and edge set operations (size, merge). Because the edge processor might allocate resources which can be reused in later operations, it is implemented as an object that can be used several times.

    Here is a simple example of how to use the edge processor:

    @code
    ep = RBA::EdgeProcessor::new
    # Prepare two boxes
    a = [ RBA::Polygon::new(RBA::Box::new(0, 0, 300, 300)) ]
    b = [ RBA::Polygon::new(RBA::Box::new(100, 100, 200, 200)) ]
    # Run an XOR -> creates a polygon with a hole, since the 'resolve_holes' parameter
    # is false:
    out = ep.boolean_p2p(a, b, RBA::EdgeProcessor::ModeXor, false, false)
    out.to_s    # -> [(0,0;0,300;300,300;300,0/100,100;200,100;200,200;100,200)]
    @/code
    """
    ModeANotB: ClassVar[int]
    r"""
    @brief boolean method's mode value for A NOT B operation
    """
    ModeAnd: ClassVar[int]
    r"""
    @brief boolean method's mode value for AND operation
    """
    ModeBNotA: ClassVar[int]
    r"""
    @brief boolean method's mode value for B NOT A operation
    """
    ModeOr: ClassVar[int]
    r"""
    @brief boolean method's mode value for OR operation
    """
    ModeXor: ClassVar[int]
    r"""
    @brief boolean method's mode value for XOR operation
    """
    @classmethod
    def mode_and(cls) -> int:
        r"""
        @brief boolean method's mode value for AND operation
        """
    @classmethod
    def mode_anotb(cls) -> int:
        r"""
        @brief boolean method's mode value for A NOT B operation
        """
    @classmethod
    def mode_bnota(cls) -> int:
        r"""
        @brief boolean method's mode value for B NOT A operation
        """
    @classmethod
    def mode_or(cls) -> int:
        r"""
        @brief boolean method's mode value for OR operation
        """
    @classmethod
    def mode_xor(cls) -> int:
        r"""
        @brief boolean method's mode value for XOR operation
        """
    @classmethod
    def new(cls) -> EdgeProcessor:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> EdgeProcessor:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgeProcessor:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: EdgeProcessor) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def boolean(self, a: Sequence[Edge], b: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given edges, creating edges

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input edges (first operand)
        @param b The input edges (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @return The output edges
        """
    @overload
    def boolean(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given polygons, creating edges

        This method computes the result for the given boolean operation on two sets of polygons.
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode
        @return The output edges
        """
    def boolean_e2e(self, a: Sequence[Edge], b: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given edges, creating edges

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input edges (first operand)
        @param b The input edges (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @return The output edges
        """
    def boolean_e2p(self, a: Sequence[Edge], b: Sequence[Edge], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given edges, creating polygons

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    def boolean_p2e(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given polygons, creating edges

        This method computes the result for the given boolean operation on two sets of polygons.
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode
        @return The output edges
        """
    def boolean_p2p(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given polygons, creating polygons

        This method computes the result for the given boolean operation on two sets of polygons.
        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def boolean_to_polygon(self, a: Sequence[Edge], b: Sequence[Edge], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given edges, creating polygons

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def boolean_to_polygon(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given polygons, creating polygons

        This method computes the result for the given boolean operation on two sets of polygons.
        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disable_progress(self) -> None:
        r"""
        @brief Disable progress reporting
        Calling this method will stop the edge processor from showing a progress bar. See \enable_progress.

        This method has been introduced in version 0.23.
        """
    def dup(self) -> EdgeProcessor:
        r"""
        @brief Creates a copy of self
        """
    def enable_progress(self, label: str) -> None:
        r"""
        @brief Enable progress reporting
        After calling this method, the edge processor will report the progress through a progress bar.
        The label is a text which is put in front of the progress bar.
        Using a progress bar will imply a performance penalty of a few percent typically.

        This method has been introduced in version 0.23.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def merge(self, in_: Sequence[Polygon], min_wc: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @return The output edges
        """
    def merge_p2e(self, in_: Sequence[Polygon], min_wc: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @return The output edges
        """
    def merge_p2p(self, in_: Sequence[Polygon], min_wc: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    def merge_to_polygon(self, in_: Sequence[Polygon], min_wc: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge(self, in_: Sequence[Edge]) -> List[Edge]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge(self, in_: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge(self, in_: Sequence[Polygon]) -> List[Edge]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge(self, in_: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge_e2e(self, in_: Sequence[Edge]) -> List[Edge]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge_e2e(self, in_: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge_e2p(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion into polygons

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_e2p(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_p2e(self, in_: Sequence[Polygon]) -> List[Edge]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge_p2e(self, in_: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge_p2p(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion into polygons

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_p2p(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion into polygons

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion into polygons

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size(self, in_: Sequence[Polygon], d: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @return The output edges
        """
    @overload
    def size(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons 

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping contours, but no self-overlaps. 

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @return The output edges
        """
    @overload
    def size_p2e(self, in_: Sequence[Polygon], d: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @return The output edges
        """
    @overload
    def size_p2e(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons 

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping contours, but no self-overlaps. 

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @return The output edges
        """
    @overload
    def size_p2p(self, in_: Sequence[Polygon], d: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size_p2p(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping polygons, but no self-overlapping ones. 
        Polygon overlap occurs if the polygons are close enough, so a positive sizing makes polygons overlap.

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size_to_polygon(self, in_: Sequence[Polygon], d: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size_to_polygon(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping polygons, but no self-overlapping ones. 
        Polygon overlap occurs if the polygons are close enough, so a positive sizing makes polygons overlap.

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """

class EdgeToEdgePairOperator:
    r"""
    @brief A generic edge-to-edge-pair operator

    Edge processors are an efficient way to process edges from an edge collection. To apply a processor, derive your own operator class and pass an instance to the \Edges#processed method.

    Conceptually, these methods take each edge from the edge collection and present it to the operator's 'process' method.
    The result of this call is a list of zero to many output edge pairs derived from the input edge.
    The output edge pair collection is the sum over all these individual results.

    The magic happens when deep mode edge collections are involved. In that case, the processor will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the operator behaves. You need to configure the operator by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using it.

    You can skip this step, but the processor algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    For a basic example see the \EdgeOperator class, with the exception that this incarnation has to deliver edge pairs.

    This class has been introduced in version 0.29.
    """
    requires_raw_input: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor needs raw (unmerged) input
    See \requires_raw_input= for details.

    Setter:
    @brief Sets a value indicating whether the processor needs raw (unmerged) input
    This flag must be set before using this processor. It tells the processor implementation whether the processor wants to have raw input (unmerged). The default value is 'false', meaning that
    the processor will receive merged polygons ('merged semantics').

    Setting this value to false potentially saves some CPU time needed for merging the polygons.
    Also, raw input means that strange shapes such as dot-like edges, self-overlapping polygons, empty or degenerated polygons are preserved.
    """
    result_is_merged: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor delivers merged output
    See \result_is_merged= for details.

    Setter:
    @brief Sets a value indicating whether the processor delivers merged output
    This flag must be set before using this processor. If the processor maintains the merged condition
    by design (output is merged if input is), it is a good idea to set this predicate to 'true'.
    This will avoid additional merge steps when the resulting collection is used in further operations
    that need merged input
    .
    """
    result_must_not_be_merged: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor's output must not be merged
    See \result_must_not_be_merged= for details.

    Setter:
    @brief Sets a value indicating whether the processor's output must not be merged
    This flag must be set before using this processor. The processor can set this flag if it wants to
    deliver shapes that must not be merged - e.g. point-like edges or strange or degenerated polygons.
    .
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgeToEdgePairOperator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) processors are size or shrink operators. Size or shrink is not dependent on orientation unless size or shrink needs to be different in x and y direction.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) processor is the convex decomposition operator. The decomposition of a polygon into convex parts is an operation that is not depending on scale nor orientation.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) processor is the rotation operator. Rotation is not depending on scale, but on the original orientation as mirrored versions need to be rotated differently.
        """

class EdgeToPolygonOperator:
    r"""
    @brief A generic edge-to-polygon operator

    Edge processors are an efficient way to process edges from an edge collection. To apply a processor, derive your own operator class and pass an instance to the \Edges#processed method.

    Conceptually, these methods take each edge from the edge collection and present it to the operator's 'process' method.
    The result of this call is a list of zero to many output polygons derived from the input edge.
    The output region is the sum over all these individual results.

    The magic happens when deep mode edge collections are involved. In that case, the processor will use as few calls as possible and exploit the hierarchical compression if possible. It needs to know however, how the operator behaves. You need to configure the operator by calling \is_isotropic, \is_scale_invariant or \is_isotropic_and_scale_invariant before using it.

    You can skip this step, but the processor algorithm will assume the worst case then. This usually leads to cell variant formation which is not always desired and blows up the hierarchy.

    For a basic example see the \EdgeOperator class, with the exception that this incarnation has to deliver edges.

    This class has been introduced in version 0.29.
    """
    requires_raw_input: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor needs raw (unmerged) input
    See \requires_raw_input= for details.

    Setter:
    @brief Sets a value indicating whether the processor needs raw (unmerged) input
    This flag must be set before using this processor. It tells the processor implementation whether the processor wants to have raw input (unmerged). The default value is 'false', meaning that
    the processor will receive merged polygons ('merged semantics').

    Setting this value to false potentially saves some CPU time needed for merging the polygons.
    Also, raw input means that strange shapes such as dot-like edges, self-overlapping polygons, empty or degenerated polygons are preserved.
    """
    result_is_merged: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor delivers merged output
    See \result_is_merged= for details.

    Setter:
    @brief Sets a value indicating whether the processor delivers merged output
    This flag must be set before using this processor. If the processor maintains the merged condition
    by design (output is merged if input is), it is a good idea to set this predicate to 'true'.
    This will avoid additional merge steps when the resulting collection is used in further operations
    that need merged input
    .
    """
    result_must_not_be_merged: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the processor's output must not be merged
    See \result_must_not_be_merged= for details.

    Setter:
    @brief Sets a value indicating whether the processor's output must not be merged
    This flag must be set before using this processor. The processor can set this flag if it wants to
    deliver shapes that must not be merged - e.g. point-like edges or strange or degenerated polygons.
    .
    """
    wants_variants: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the filter prefers cell variants
    See \wants_variants= for details.

    Setter:
    @brief Sets a value indicating whether the filter prefers cell variants
    This flag must be set before using this filter for hierarchical applications (deep mode). It tells the filter implementation whether cell variants should be created (true, the default) or shape propagation will be applied (false).

    This decision needs to be made, if the filter indicates that it will deliver different results
    for scaled or rotated versions of the shape (see \is_isotropic and the other hints). If a cell
    is present with different qualities - as seen from the top cell - the respective instances
    need to be differentiated. Cell variant formation is one way, shape propagation the other way.
    Typically, cell variant formation is less expensive, but the hierarchy will be modified.
    """
    @classmethod
    def new(cls) -> EdgeToPolygonOperator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_isotropic(self) -> None:
        r"""
        @brief Indicates that the filter has isotropic properties
        Call this method before using the filter to indicate that the selection is independent of the orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        Examples for isotropic (polygon) processors are size or shrink operators. Size or shrink is not dependent on orientation unless size or shrink needs to be different in x and y direction.
        """
    def is_isotropic_and_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is isotropic and scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale and orientation of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for such a (polygon) processor is the convex decomposition operator. The decomposition of a polygon into convex parts is an operation that is not depending on scale nor orientation.
        """
    def is_scale_invariant(self) -> None:
        r"""
        @brief Indicates that the filter is scale invariant
        Call this method before using the filter to indicate that the selection is independent of the scale of the shape. This helps the filter algorithm optimizing the filter run, specifically in hierarchical mode.

        An example for a scale invariant (polygon) processor is the rotation operator. Rotation is not depending on scale, but on the original orientation as mirrored versions need to be rotated differently.
        """

class Edges(ShapeCollection):
    r"""
    @brief A collection of edges (Not necessarily describing closed contours)


    This class was introduced to simplify operations on edges sets. See \Edge for a description of the individual edge object. The edge collection contains an arbitrary number of edges and supports operations to select edges by various criteria, produce polygons from the edges by applying an extension, filtering edges against other edges collections and checking geometrical relations to other edges (DRC functionality).

    The edge collection is supposed to work closely with the \Region polygon set. Both are related, although the edge collection has a lower rank since it potentially represents a disconnected collection of edges. Edge collections may form closed contours, for example immediately after they have been derived from a polygon set using \Region#edges. But this state is volatile and can easily be destroyed by filtering edges. Hence the connected state does not play an important role in the edge collection's API.

    Edge collections may also contain points (degenerated edges with identical start and end points). Such point-like objects participate in some although not all methods of the edge collection class. 
    Edge collections can be used in two different flavors: in raw mode or merged semantics. With merged semantics (the default), connected edges are considered to belong together and are effectively merged.
    Overlapping parts are counted once in that mode. Dot-like edges are not considered in merged semantics.
    In raw mode (without merged semantics), each edge is considered as it is. Overlaps between edges
    may exists and merging has to be done explicitly using the \merge method. The semantics can be
    selected using \merged_semantics=.


    This class has been introduced in version 0.23.
    """
    class EdgeType:
        r"""
        @brief This enum specifies the edge type for edge angle filters.

        This enum was introduced in version 0.28.
        """
        DiagonalEdges: ClassVar[Edges.EdgeType]
        r"""
        @brief Diagonal edges are selected (-45 and 45 degree)
        """
        OrthoDiagonalEdges: ClassVar[Edges.EdgeType]
        r"""
        @brief Diagonal or orthogonal edges are selected (0, 90, -45 and 45 degree)
        """
        OrthoEdges: ClassVar[Edges.EdgeType]
        r"""
        @brief Horizontal and vertical edges are selected
        """
        @overload
        @classmethod
        def new(cls, i: int) -> Edges.EdgeType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> Edges.EdgeType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        def __hash__(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        def __int__(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        @overload
        def __lt__(self, other: Edges.EdgeType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def hash(self) -> int:
            r"""
            @brief Gets the hash value from the enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    AlwaysIncludeZeroDistance: ClassVar[ZeroDistanceMode]
    r"""
    @hide
    @brief Specifies that check functions should always include edges with zero distance
    This mode has little practical value.
    """
    DiagonalEdges: ClassVar[Edges.EdgeType]
    r"""
    @brief Diagonal edges are selected (-45 and 45 degree)
    """
    DifferentPropertiesConstraint: ClassVar[PropertyConstraint]
    r"""
    @brief Specifies to consider shapes only if their user properties are different
    When using this constraint - for example on a boolean operation - shapes are considered only if their user properties are different. Properties are generated on the output shapes where applicable.
    """
    DifferentPropertiesConstraintDrop: ClassVar[PropertyConstraint]
    r"""
    @brief Specifies to consider shapes only if their user properties are different
    When using this constraint - for example on a boolean operation - shapes are considered only if their user properties are the same. No properties are generated on the output shapes.
    """
    Euclidian: ClassVar[Metrics]
    r"""
    @brief Specifies Euclidian metrics for the check functions
    This value can be used for the metrics parameter in the check functions, i.e. \width_check. This value specifies Euclidian metrics, i.e. the distance between two points is measured by:

    @code
    d = sqrt(dx^2 + dy^2)
    @/code

    All points within a circle with radius d around one point are considered to have a smaller distance than d.
    """
    IgnoreProperties: ClassVar[PropertyConstraint]
    r"""
    @brief Specifies to ignore properties
    When using this constraint - for example on a boolean operation - properties are ignored and are not generated in the output.
    """
    IncludeZeroDistanceWhenCollinearAndTouching: ClassVar[ZeroDistanceMode]
    r"""
    @brief Specifies that check functions should include edges when they are collinear and touch
    With this specification, the check functions will also check edges if they share at least one common point and are collinear. This is the mode that includes checking the 'kissing corner' cases when the kissing edges are collinear. This mode was default up to version 0.28. 
    """
    IncludeZeroDistanceWhenOverlapping: ClassVar[ZeroDistanceMode]
    r"""
    @brief Specifies that check functions should include edges when they overlap
    With this specification, the check functions will also check edges which are collinear and share more than a single point. This is the mode that excludes the 'kissing corner' cases.
    """
    IncludeZeroDistanceWhenTouching: ClassVar[ZeroDistanceMode]
    r"""
    @brief Specifies that check functions should include edges when they touch
    With this specification, the check functions will also check edges if they share at least one common point. This is the mode that includes checking the 'kissing corner' cases. This mode is default for version 0.28.16 and later. 
    """
    NeverIncludeZeroDistance: ClassVar[ZeroDistanceMode]
    r"""
    @brief Specifies that check functions should never include edges with zero distance.
    With this specification, the check functions will ignore edges which are collinear or touch.
    """
    NoPropertyConstraint: ClassVar[PropertyConstraint]
    r"""
    @brief Specifies not to apply any property constraint
    When using this constraint - for example on a boolean operation - shapes are considered regardless of their user properties. Properties are generated on the output shapes where applicable.
    """
    OrthoDiagonalEdges: ClassVar[Edges.EdgeType]
    r"""
    @brief Diagonal or orthogonal edges are selected (0, 90, -45 and 45 degree)
    """
    OrthoEdges: ClassVar[Edges.EdgeType]
    r"""
    @brief Horizontal and vertical edges are selected
    """
    Projection: ClassVar[Metrics]
    r"""
    @brief Specifies projected distance metrics for the check functions
    This value can be used for the metrics parameter in the check functions, i.e. \width_check. This value specifies projected metrics, i.e. the distance is defined as the minimum distance measured perpendicular to one edge. That implies that the distance is defined only where two edges have a non-vanishing projection onto each other.
    """
    SamePropertiesConstraint: ClassVar[PropertyConstraint]
    r"""
    @brief Specifies to consider shapes only if their user properties are the same
    When using this constraint - for example on a boolean operation - shapes are considered only if their user properties are the same. Properties are generated on the output shapes where applicable.
    """
    SamePropertiesConstraintDrop: ClassVar[PropertyConstraint]
    r"""
    @brief Specifies to consider shapes only if their user properties are the same
    When using this constraint - for example on a boolean operation - shapes are considered only if their user properties are the same. No properties are generated on the output shapes.
    """
    Square: ClassVar[Metrics]
    r"""
    @brief Specifies square metrics for the check functions
    This value can be used for the metrics parameter in the check functions, i.e. \width_check. This value specifies square metrics, i.e. the distance between two points is measured by:

    @code
    d = max(abs(dx), abs(dy))
    @/code

    All points within a square with length 2*d around one point are considered to have a smaller distance than d in this metrics.
    """
    merged_semantics: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether merged semantics is enabled
    See \merged_semantics= for a description of this attribute.

    Setter:
    @brief Enable or disable merged semantics
    If merged semantics is enabled (the default), colinear, connected or overlapping edges will be considered
    as single edges.
    """
    @overload
    @classmethod
    def new(cls) -> Edges:
        r"""
        @brief Default constructor

        This constructor creates an empty edge collection.
        """
    @overload
    @classmethod
    def new(cls, array: Sequence[Edge]) -> Edges:
        r"""
        @brief Constructor from an edge array

        This constructor creates an edge collection from an array of edges.
        """
    @overload
    @classmethod
    def new(cls, array: Sequence[Polygon]) -> Edges:
        r"""
        @brief Constructor from a polygon array

        This constructor creates an edge collection from an array of polygons.
        The edges form the contours of the polygons.
        """
    @overload
    @classmethod
    def new(cls, box: Box) -> Edges:
        r"""
        @brief Box constructor

        This constructor creates an edge collection from a box.
        The edges form the contour of the box.
        """
    @overload
    @classmethod
    def new(cls, edge: Edge) -> Edges:
        r"""
        @brief Constructor from a single edge

        This constructor creates an edge collection with a single edge.
        """
    @overload
    @classmethod
    def new(cls, path: Path) -> Edges:
        r"""
        @brief Path constructor

        This constructor creates an edge collection from a path.
        The edges form the contour of the path.
        """
    @overload
    @classmethod
    def new(cls, polygon: Polygon) -> Edges:
        r"""
        @brief Polygon constructor

        This constructor creates an edge collection from a polygon.
        The edges form the contour of the polygon.
        """
    @overload
    @classmethod
    def new(cls, polygon: SimplePolygon) -> Edges:
        r"""
        @brief Simple polygon constructor

        This constructor creates an edge collection from a simple polygon.
        The edges form the contour of the polygon.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), false)
        @/code
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a hierarchical edge collection

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, false)
        @/code
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, expr: str, as_pattern: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param dss The \DeepShapeStore object that acts as a heap for hierarchical operations.
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create a deep edge set from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        region = RBA::Region::new(iter, dss, "*")           # all texts
        region = RBA::Region::new(iter, dss, "A*")          # all texts starting with an 'A'
        region = RBA::Region::new(iter, dss, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a hierarchical edge collection with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, RBA::ICplxTrans::new(layout.dbu / dbu), false)
        @/code
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, expr: str, as_pattern: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create dot-like edges from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        dots = RBA::Edges::new(iter, "*")           # all texts
        dots = RBA::Edges::new(iter, "A*")          # all texts starting with an 'A'
        dots = RBA::Edges::new(iter, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code
        """
    @overload
    @classmethod
    def new(cls, shapes: Shapes, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a flat edge collection from a \Shapes container

        If 'as_edges' is true, the shapes from the container will be converted to edges (i.e. polygon contours to edges). Otherwise, only edges will be taken from the container.

        This method has been introduced in version 0.26.
        """
    def __add__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the combined edge set of self and the other one

        @return The resulting edge set

        This operator adds the edges of the other edge set to self and returns a new combined edge set. This usually creates unmerged edge sets and edges may overlap. Use \merge if you want to ensure the result edge set is merged.

        The 'join' alias has been introduced in version 0.28.12.
        """
    @overload
    def __and__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean AND between self and the other edge collection

        @return The result of the boolean AND operation

        The boolean AND operation will return all parts of the edges in this collection which are coincident with parts of the edges in the other collection.The result will be a merged edge collection.

        The 'and' alias has been introduced in version 0.28.12.
        """
    @overload
    def __and__(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges inside the given region

        @return The edges inside the given region

        This operation returns the parts of the edges which are inside the given region.
        Edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.The 'and' alias has been introduced in version 0.28.12.
        """
    def __copy__(self) -> Edges:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Edges:
        r"""
        @brief Creates a copy of self
        """
    def __getitem__(self, n: int) -> Edge:
        r"""
        @brief Returns the nth edge of the collection

        This method returns nil if the index is out of range. It is available for flat edge collections only - i.e. those for which \has_valid_edges? is true. Use \flatten to explicitly flatten an edge collection.
        This method returns the raw edge (not merged edges, even if merged semantics is enabled).

        The \each iterator is the more general approach to access the edges.
        """
    def __iadd__(self, other: Edges) -> Edges:
        r"""
        @brief Adds the edges of the other edge collection to self

        @return The edge set after modification (self)

        This operator adds the edges of the other edge set to self. This usually creates unmerged edge sets and edges may overlap. Use \merge if you want to ensure the result edge set is merged.

        Note that in Ruby, the '+=' operator actually does not exist, but is emulated by '+' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'join_with' instead.

        The 'join_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def __iand__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean AND between self and the other edge collection in-place (modifying self)

        @return The edge collection after modification (self)

        The boolean AND operation will return all parts of the edges in this collection which are coincident with parts of the edges in the other collection.The result will be a merged edge collection.

        Note that in Ruby, the '&=' operator actually does not exist, but is emulated by '&' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'and_with' instead.

        The 'and_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def __iand__(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges inside the given region in-place (modifying self)

        @return The edge collection after modification (self)

        This operation selects the parts of the edges which are inside the given region.
        Edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        Note that in Ruby, the '&=' operator actually does not exist, but is emulated by '&' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'and_with' instead.

        The 'and_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an empty edge collection.
        """
    @overload
    def __init__(self, array: Sequence[Edge]) -> None:
        r"""
        @brief Constructor from an edge array

        This constructor creates an edge collection from an array of edges.
        """
    @overload
    def __init__(self, array: Sequence[Polygon]) -> None:
        r"""
        @brief Constructor from a polygon array

        This constructor creates an edge collection from an array of polygons.
        The edges form the contours of the polygons.
        """
    @overload
    def __init__(self, box: Box) -> None:
        r"""
        @brief Box constructor

        This constructor creates an edge collection from a box.
        The edges form the contour of the box.
        """
    @overload
    def __init__(self, edge: Edge) -> None:
        r"""
        @brief Constructor from a single edge

        This constructor creates an edge collection with a single edge.
        """
    @overload
    def __init__(self, path: Path) -> None:
        r"""
        @brief Path constructor

        This constructor creates an edge collection from a path.
        The edges form the contour of the path.
        """
    @overload
    def __init__(self, polygon: Polygon) -> None:
        r"""
        @brief Polygon constructor

        This constructor creates an edge collection from a polygon.
        The edges form the contour of the polygon.
        """
    @overload
    def __init__(self, polygon: SimplePolygon) -> None:
        r"""
        @brief Simple polygon constructor

        This constructor creates an edge collection from a simple polygon.
        The edges form the contour of the polygon.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), false)
        @/code
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a hierarchical edge collection

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, false)
        @/code
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, expr: str, as_pattern: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param dss The \DeepShapeStore object that acts as a heap for hierarchical operations.
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create a deep edge set from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        region = RBA::Region::new(iter, dss, "*")           # all texts
        region = RBA::Region::new(iter, dss, "A*")          # all texts starting with an 'A'
        region = RBA::Region::new(iter, dss, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a hierarchical edge collection with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, RBA::ICplxTrans::new(layout.dbu / dbu), false)
        @/code
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, expr: str, as_pattern: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create dot-like edges from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        dots = RBA::Edges::new(iter, "*")           # all texts
        dots = RBA::Edges::new(iter, "A*")          # all texts starting with an 'A'
        dots = RBA::Edges::new(iter, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code
        """
    @overload
    def __init__(self, shapes: Shapes, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a flat edge collection from a \Shapes container

        If 'as_edges' is true, the shapes from the container will be converted to edges (i.e. polygon contours to edges). Otherwise, only edges will be taken from the container.

        This method has been introduced in version 0.26.
        """
    def __ior__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean OR between self and the other edge set in-place (modifying self)

        @return The edge collection after modification (self)

        The boolean OR is implemented by merging the edges of both edge sets. To simply join the edge collections without merging, the + operator is more efficient.
        Note that in Ruby, the '|=' operator actually does not exist, but is emulated by '|' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'or_with' instead.

        The 'or_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def __isub__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean NOT between self and the other edge collection in-place (modifying self)

        @return The edge collection after modification (self)

        The boolean NOT operation will return all parts of the edges in this collection which are not coincident with parts of the edges in the other collection.The result will be a merged edge collection.

        Note that in Ruby, the '-=' operator actually does not exist, but is emulated by '-' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'not_with' instead.

        The 'not_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def __isub__(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges outside the given region in-place (modifying self)

        @return The edge collection after modification (self)

        This operation selects the parts of the edges which are outside the given region.
        Edges on the borders of the polygons are not included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        Note that in Ruby, the '-=' operator actually does not exist, but is emulated by '-' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'not_with' instead.

        This method has been introduced in version 0.24.The 'not_with' alias has been introduced in version 0.28.12.
        """
    def __iter__(self) -> Iterator[Edge]:
        r"""
        @brief Returns each edge of the region
        """
    def __ixor__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean XOR between self and the other edge collection in-place (modifying self)

        @return The edge collection after modification (self)

        The boolean XOR operation will return all parts of the edges in this and the other collection except the parts where both are coincident.
        The result will be a merged edge collection.

        Note that in Ruby, the '^=' operator actually does not exist, but is emulated by '^' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'xor_with' instead.

        The 'xor_with' alias has been introduced in version 0.28.12.
        """
    def __len__(self) -> int:
        r"""
        @brief Returns the (flat) number of edges in the edge collection

        This returns the number of raw edges (not merged edges if merged semantics is enabled).
        The count is computed 'as if flat', i.e. edges inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def __or__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean OR between self and the other edge set

        @return The resulting edge collection

        The boolean OR is implemented by merging the edges of both edge sets. To simply join the edge collections without merging, the + operator is more efficient.
        The 'or' alias has been introduced in version 0.28.12.
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts the edge collection to a string
        The length of the output is limited to 20 edges to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the edge collection to a string
        The length of the output is limited to 20 edges to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    @overload
    def __sub__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean NOT between self and the other edge collection

        @return The result of the boolean NOT operation

        The boolean NOT operation will return all parts of the edges in this collection which are not coincident with parts of the edges in the other collection.The result will be a merged edge collection.

        The 'not' alias has been introduced in version 0.28.12.
        """
    @overload
    def __sub__(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges outside the given region

        @return The edges outside the given region

        This operation returns the parts of the edges which are outside the given region.
        Edges on the borders of the polygons are not included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.The 'not' alias has been introduced in version 0.28.12.
        """
    def __xor__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean XOR between self and the other edge collection

        @return The result of the boolean XOR operation

        The boolean XOR operation will return all parts of the edges in this and the other collection except the parts where both are coincident.
        The result will be a merged edge collection.

        The 'xor' alias has been introduced in version 0.28.12.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    @overload
    def and_(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean AND between self and the other edge collection

        @return The result of the boolean AND operation

        The boolean AND operation will return all parts of the edges in this collection which are coincident with parts of the edges in the other collection.The result will be a merged edge collection.

        The 'and' alias has been introduced in version 0.28.12.
        """
    @overload
    def and_(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges inside the given region

        @return The edges inside the given region

        This operation returns the parts of the edges which are inside the given region.
        Edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.The 'and' alias has been introduced in version 0.28.12.
        """
    @overload
    def and_with(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean AND between self and the other edge collection in-place (modifying self)

        @return The edge collection after modification (self)

        The boolean AND operation will return all parts of the edges in this collection which are coincident with parts of the edges in the other collection.The result will be a merged edge collection.

        Note that in Ruby, the '&=' operator actually does not exist, but is emulated by '&' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'and_with' instead.

        The 'and_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def and_with(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges inside the given region in-place (modifying self)

        @return The edge collection after modification (self)

        This operation selects the parts of the edges which are inside the given region.
        Edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        Note that in Ruby, the '&=' operator actually does not exist, but is emulated by '&' followed by an assignment. This is less efficient than the in-place operation, so it is recommended to use 'and_with' instead.

        The 'and_with' alias has been introduced in version 0.28.12.
        """
    @overload
    def andnot(self, other: Edges) -> List[Edges]:
        r"""
        @brief Returns the boolean AND and NOT between self and the other edge set

        @return A two-element array of edge collections with the first one being the AND result and the second one being the NOT result

        This method will compute the boolean AND and NOT between two edge sets simultaneously. Because this requires a single sweep only, using this method is faster than doing AND and NOT separately.

        This method has been added in version 0.28.
        """
    @overload
    def andnot(self, other: Region) -> List[Edges]:
        r"""
        @brief Returns the boolean AND and NOT between self and the region

        @return A two-element array of edge collections with the first one being the AND result and the second one being the NOT result

        This method will compute the boolean AND and NOT simultaneously. Because this requires a single sweep only, using this method is faster than doing AND and NOT separately.

        This method has been added in version 0.28.
        """
    def assign(self, other: ShapeCollection) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box of the edge collection
        The bounding box is the box enclosing all points of all edges.
        """
    def centers(self, length: int, fraction: float) -> Edges:
        r"""
        @brief Returns edges representing the center part of the edges
        @return A new collection of edges representing the part around the center
        This method allows one to specify the length of these segments in a twofold way: either as a fixed length or by specifying a fraction of the original length:

        @code
        edges = ...  # An edge collection
        edges.centers(100, 0.0)     # All segments have a length of 100 DBU
        edges.centers(0, 50.0)      # All segments have a length of half the original length
        edges.centers(100, 50.0)    # All segments have a length of half the original length
                                    # or 100 DBU, whichever is larger
        @/code

        It is possible to specify 0 for both values. In this case, degenerated edges (points) are delivered which specify the centers of the edges but can't participate in some functions.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the edge collection
        """
    def count(self) -> int:
        r"""
        @brief Returns the (flat) number of edges in the edge collection

        This returns the number of raw edges (not merged edges if merged semantics is enabled).
        The count is computed 'as if flat', i.e. edges inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def data_id(self) -> int:
        r"""
        @brief Returns the data ID (a unique identifier for the underlying data storage)

        This method has been added in version 0.26.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disable_progress(self) -> None:
        r"""
        @brief Disable progress reporting
        Calling this method will disable progress reporting. See \enable_progress.
        """
    def dup(self) -> Edges:
        r"""
        @brief Creates a copy of self
        """
    def each(self) -> Iterator[Edge]:
        r"""
        @brief Returns each edge of the region
        """
    def each_merged(self) -> Iterator[Edge]:
        r"""
        @brief Returns each edge of the region

        In contrast to \each, this method delivers merged edges if merge semantics applies while \each delivers the original edges only.

        This method has been introduced in version 0.25.
        """
    def enable_progress(self, label: str) -> None:
        r"""
        @brief Enable progress reporting
        After calling this method, the edge collection will report the progress through a progress bar while expensive operations are running.
        The label is a text which is put in front of the progress bar.
        Using a progress bar will imply a performance penalty of a few percent typically.
        """
    def enable_properties(self) -> None:
        r"""
        @brief Enables properties for the given container.
        This method has an effect mainly o