/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <cmath>
#include <cairomm/context.h>

#include "widget_chart.hpp"


using namespace LIFEO;


ChartPoints::ChartPoints( void )
:   value_min( -1 ), value_max( -1 ), flag_use_current( false )
{
}

void
ChartPoints::add( Date::date_t date, int value )
{
    insert( value_type( date, value ) );

    if( value > value_max )
        value_max = value;
    if( value < value_min || value_min < 0 )
        value_min = value;
}

void
ChartPoints::increase( Date::date_t date, unsigned int increment )
{
    iterator iter( find( date ) );
    if( iter != end() )
    {
        iter->second += increment;
        if( iter->second > value_max )
            value_max = iter->second;
    }
    else
        add( date, increment );
}

int
ChartPoints::get_value_at( Date::date_t date )
{
    iterator iter( find( date ) );
    return( iter == end() ? 0 : iter->second );
}

unsigned int
ChartPoints::get_span( void )
{
    if( value_max < 0 )
        return 0;

    return( Date::calculate_months_between( begin()->first, rbegin()->first ) + 1 );
}

// WIDGETCHART =====================================================================================
WidgetChart::WidgetChart( BaseObjectType *cobject,
                          const Glib::RefPtr< Gtk::Builder >& )
:   DrawingArea( cobject ), m_points( NULL ), m_width( -1 ), m_overview_height( 0.0 ),
    m_flag_button_pressed( false ), m_flag_pointer_hovered( false )
{
    set_events( Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK |
                Gdk::POINTER_MOTION_MASK | Gdk::LEAVE_NOTIFY_MASK | Gdk::SCROLL_MASK );
    m_font_main = Cairo::ToyFontFace::create( "FreeSans",
                  Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_BOLD );
}

void
WidgetChart::update_col_geom( bool flag_new )
{
    if( m_points )
    {
        m_col_count = m_points->get_span();
        if( m_col_count > m_col_count_max )
            m_col_count = m_col_count_max;

        m_step_x = m_col_count == 1 ? m_length : m_length / ( m_col_count - 1 );

        m_overview_height = m_col_count < m_points->get_span() ?
                log10( m_height ) * OVERVIEW_COEFFICIENT : 0.0;
        m_y_max = m_height - 2 * bar_height - m_overview_height;
        m_y_mid = ( m_y_max + s_y_min ) / 2;
        m_amplitude = m_y_max - s_y_min;
        m_coefficient = m_points->value_max == m_points->value_min ? 0.0 :
                m_amplitude / ( m_points->value_max - m_points->value_min );

        const int col_start_max = m_points->get_span() - m_col_count;
        if( flag_new )
        {
            /* not used
            if( m_points->flag_use_current )
            {
                int col_start = m_points->current - m_col_count / 2;
                if( col_start < 0 )
                    m_col_start = 0;
                else
                if( col_start > col_start_max )
                    m_col_start = col_start_max;
                else
                    m_col_start = col_start;
            }
            else*/
                m_col_start = col_start_max;
        }
        else if( int( m_col_start ) > col_start_max )
            m_col_start = col_start_max;
    }
}

void
WidgetChart::set_points( ChartPoints *points )
{
    if( m_points )
        delete m_points;
    m_points = points;

    if( m_width > 0 ) // if on_size_allocate is executed before
    {
        update_col_geom( true );
        if( Glib::RefPtr< Gdk::Window > window = get_window() )
            window->invalidate( false );
    }
}

bool
WidgetChart::on_scroll_event( GdkEventScroll *event )
{
    if( m_points )
    {
        if( event->direction == GDK_SCROLL_UP && m_col_start > 0 )
            m_col_start--;
        else
        if( event->direction == GDK_SCROLL_DOWN &&
            m_col_start < ( m_points->get_span() - m_col_count ) )
            m_col_start++;
        else
            return true;

        get_window()->invalidate( false );
    }
    return true;
}

bool
WidgetChart::on_button_press_event( GdkEventButton *event )
{
    if( m_points && event->button == 1 && event->y > m_height - m_overview_height )
    {
        int col_start = ( event->x / m_width ) * m_points->get_span() - m_col_count / 2;
        if( col_start > int( m_points->get_span() - m_col_count ) )
            col_start = m_points->get_span() - m_col_count;
        else
        if( col_start < 0 )
            col_start = 0;
        m_col_start = col_start;

        get_window()->invalidate( false );

        m_flag_button_pressed = true;
    }

    return true;
}

bool
WidgetChart::on_button_release_event( GdkEventButton *event )
{
    if( event->button == 1 )
    {
        m_flag_button_pressed = false;
    }

    return true;
}

bool
WidgetChart::on_motion_notify_event( GdkEventMotion *event )
{
    if( m_points )
    {
        if( m_flag_button_pressed )
        {
            int col_start = ( event->x / m_width ) * m_points->get_span() - m_col_count / 2;
            if( col_start > int( m_points->get_span() - m_col_count ) )
                col_start = m_points->get_span() - m_col_count;
            else
            if( col_start < 0 )
                col_start = 0;

            if( col_start != ( int ) m_col_start )
            {
                m_col_start = col_start;
                get_window()->invalidate( false );
            }
        }
        bool flag_pointer_hovered = ( event->y > m_height - m_overview_height );
        if( flag_pointer_hovered != m_flag_pointer_hovered )
        {
            m_flag_pointer_hovered = flag_pointer_hovered;
            get_window()->invalidate( false );  // TODO: limit to scrollbar only
        }
    }

    return Gtk::DrawingArea::on_motion_notify_event( event );
}

bool
WidgetChart::on_leave_notify_event( GdkEventCrossing *event )
{
    if( m_points )
    {
        m_flag_pointer_hovered = false;
        get_window()->invalidate( false );  // TODO: limit to scrollbar only
    }

    return true;
}

void
WidgetChart::on_size_allocate( Gtk::Allocation &allocation )
{
    bool flag_first( m_width < 0 );

    Gtk::Widget::on_size_allocate( allocation );
    m_width = allocation.get_width();
    m_height = allocation.get_height();

    m_x_max = m_width - border_curve;
    m_length = m_x_max - s_x_min;
    m_col_count_max = ( m_length / COLUMN_WIDTH_MIN ) + 1;

    update_col_geom( flag_first );
}

bool
WidgetChart::on_draw( const Cairo::RefPtr< Cairo::Context > &cr )
{
    /* TODO
    if( event )
    {
        // clip to the area indicated by the expose event so that we only
        // redraw the portion of the window that needs to be redrawn
        cr->rectangle( event->area.x, event->area.y,
                event->area.width, event->area.height );
        cr->clip();
    }*/

    // FONT FACE
    cr->set_font_face( m_font_main );

    // BACKGROUND
    cr->rectangle( 0.0, 0.0, m_width, m_y_max );
    cr->set_source_rgba( 1.0, 1.0, 1.0, 0.6 );
    cr->fill();

    // HANDLE THERE-IS-TOO-FEW-ENTRIES-CASE SPECIALLY
    if( m_points->get_span() < 2 )
    {
        cr->set_font_size( 1.5 * label_height );
        cr->set_source_rgb( 0.0, 0.0, 0.0 );
        Cairo::TextExtents te;
        cr->get_text_extents( _( "INSUFFICIENT DATA" ), te );
        cr->move_to( ( m_width - te.width ) / 2 , m_height / 2 );
        cr->show_text( _( "INSUFFICIENT DATA" ) );
        return true;
    }

    // END REGIONS
    if( m_col_start == 0 )
    {
        cr->set_source_rgb( 0.8, 0.8, 0.8 );
        cr->rectangle( 0.0, 0.0, s_x_min, m_y_max );
        cr->fill();
    }
    if( m_col_start == m_points->get_span() - m_col_count )
    {
        cr->set_source_rgb( 0.8, 0.8, 0.8 );
        cr->rectangle( m_width - border_curve, 0.0, border_curve, m_y_max );
        cr->fill();
    }

    // MONTHS BAR
    cr->rectangle( 0.0, m_y_max, m_width, bar_height );
    cr->set_source_rgb( 0.85, 0.85, 0.85 );
    cr->fill();
    // YEARS BAR
    cr->rectangle( 0.0, m_y_max + bar_height, m_width, bar_height );
    cr->set_source_rgb( 0.75, 0.75, 0.75 );
    cr->fill();

    // YEAR & MONTH LABELS
    cr->set_source_rgb( 0.0, 0.0, 0.0 );
    cr->set_font_size( label_height );
    unsigned int year_last = 0;
    Date date( m_points->begin()->first );
    date.forward_months( m_col_start );
    const Date::date_t date_start( date.m_date );

    bool flag_show_year_label( date.get_month() + m_col_count > 16 );

    for( unsigned int i = 0; i < m_col_count; ++i )
    {
        cr->move_to( s_x_min + m_step_x * i, m_y_max + label_y );
        cr->show_text( Glib::ustring::compose( "%1", date.get_month() ) );

        if( date.get_year() != year_last )
        {
            year_last = date.get_year();
            if( i != 0 || flag_show_year_label == false )
            {
                cr->move_to( s_x_min + m_step_x * i, m_y_max + bar_height + label_y );
                cr->show_text( Glib::ustring::compose( "%1", year_last ) );
            }
        }

        date.forward_month();
    }

    // y LABELS
    cr->move_to( border_label, s_y_min - offset_label );
    cr->show_text( Glib::ustring::compose( "%1", m_points->value_max ) );
    cr->move_to( border_label, m_y_max - offset_label );
    cr->show_text( Glib::ustring::compose( "%1", m_points->value_min ) );

    // HORIZONTAL LINES
    cr->set_source_rgb( 0.5, 0.5, 0.5 );
    cr->set_line_width( 1.0 );
    // + 0.5 offset needed to get crisp lines:
    cr->move_to( border_label, s_y_min + 0.5 );
    cr->line_to( m_width - border_label, s_y_min + 0.5 );
    cr->move_to( border_label, m_y_mid + 0.5 );
    cr->line_to( m_width - border_label, m_y_mid + 0.5 );
    cr->stroke();

    // CURRENT LINE (NOT USED)
    if( m_points->flag_use_current &&
        m_points->current >= m_col_start &&
        m_points->current - m_col_start < m_col_count )
    {
        cr->set_line_width( 2.0 );
        cr->set_source_rgb( 0.6, 0.0, 0.5 );
        cr->move_to( s_x_min + m_step_x * ( m_points->current - m_col_start ), 0.0 );
        cr->line_to( s_x_min + m_step_x * ( m_points->current - m_col_start ), m_y_max );
        cr->stroke();
    }

    // GRAPH LINE
    cr->set_source_rgb( 1.0, 0.0, 0.0 );
    cr->set_line_join( Cairo::LINE_JOIN_BEVEL );
    cr->set_line_width( 3.0 );

    date.m_date = date_start;
    cr->move_to( s_x_min, m_y_max - m_coefficient *
                         ( m_points->get_value_at( date.m_date ) - m_points->value_min ) );
    for( unsigned int i = 0; i < m_col_count; ++i )
    {
        cr->line_to( s_x_min + m_step_x * i, m_y_max - m_coefficient *
                             ( m_points->get_value_at( date.m_date ) - m_points->value_min ) );
        date.forward_month();
    }
    cr->stroke();

    // GRAPH DOTS
    date.m_date = date_start;
    for( unsigned int i = 0; i < m_col_count; ++i )
    {
        cr->arc( s_x_min + m_step_x * i, m_y_max - m_coefficient *
                         ( m_points->get_value_at( date.m_date ) - m_points->value_min ),
                 5.0, 0.0, 2.0*M_PI);
        cr->fill();

        date.forward_month();
    }

    if( m_col_count < m_points->get_span() )
    {
        // OVERVIEW PARAMETERS
        float ampli_ov = m_overview_height - 2 * offset_label;
        float coeff_ov = m_points->value_max == m_points->value_min ? 0.5 :
                ampli_ov / ( m_points->value_max - m_points->value_min );
        float step_x_ov = m_width - 2 * offset_label;
        if( m_points->get_span() > 1 )
            step_x_ov /= m_points->get_span() - 1;

        // OVERVIEW REGION
        cr->save();
        cr->set_source_rgb( 0.7, 0.7, 0.7 );
        cr->rectangle( 0.0, m_height - m_overview_height, m_width, m_overview_height );
        cr->fill();

        if( m_flag_pointer_hovered )
            cr->set_source_rgb( 1.0, 1.0, 1.0 );
        else
            cr->set_source_rgb( 0.95, 0.95, 0.95 );
        cr->rectangle( offset_label + m_col_start * step_x_ov, m_height - m_overview_height,
                       ( m_col_count - 1 ) * step_x_ov, m_overview_height );
        cr->fill();
        cr->restore();

        // CURRENT LINE (NOT USED)
        if( m_points->flag_use_current )
        {
            cr->save();
            cr->set_line_width( 1.0 );
            cr->set_source_rgb( 0.6, 0.0, 0.5 );
            cr->move_to( offset_label + step_x_ov * m_points->current,
                         m_height - m_overview_height );
            cr->line_to( offset_label + step_x_ov * m_points->current, m_height );
            cr->stroke();
            cr->restore();
        }

        // OVERVIEW LINE
        cr->set_line_width( 2.0 );

        date.m_date = m_points->begin()->first;
        cr->move_to( offset_label, m_height - offset_label - coeff_ov *
                     ( m_points->get_value_at( date.m_date ) - m_points->value_min ) );
        for( unsigned int i = 1; i < m_points->get_span(); ++i )
        {
            date.forward_month();
            cr->line_to( offset_label + step_x_ov * i, m_height - offset_label - coeff_ov *
                         ( m_points->get_value_at( date.m_date ) - m_points->value_min ) );
        }
        cr->stroke();

        // DIVIDER
        if( m_flag_pointer_hovered )
            cr->set_source_rgb( 0.2, 0.2, 0.2 );
        else
            cr->set_source_rgb( 0.45, 0.45, 0.45 );
        cr->rectangle( 1.0, m_height - m_overview_height,
                       m_width - 2.0, m_overview_height - 1.0 );
        cr->stroke();
    }

    return true;
}
