/*
 *
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.cache.marshall;


import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.Region;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.config.Configuration.CacheMode;
import org.jboss.cache.factories.UnitTestCacheConfigurationFactory;
import org.jboss.cache.marshall.data.Address;
import org.jboss.cache.marshall.data.Person;
import static org.testng.AssertJUnit.*;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.transaction.NotSupportedException;
import javax.transaction.SystemException;
import javax.transaction.TransactionManager;
import java.lang.reflect.Method;
import java.util.HashMap;

/**
 * Test case for marshalling using Sync mode.
 *
 * @author Ben Wang
 * @version $Revision: 5919 $
 */
@Test(groups = {"functional", "jgroups"})
public class SyncReplTest extends RegionBasedMarshallingTestBase
{
   CacheSPI<Object, Object> cache1, cache2;
   String props = null;
   Person ben_;
   Address addr_;
   Throwable ex_;
   private Fqn aop = Fqn.fromString("/aop");
   protected boolean useMarshalledValues = false;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      log("creating cache1");
      cache1 = createCache("TestCache");

      log("creating cache2");

      cache2 = createCache("TestCache");
      addr_ = new Address();
      addr_.setCity("San Jose");
      ben_ = new Person();
      ben_.setName("Ben");
      ben_.setAddress(addr_);

      // Pause to give caches time to see each other
      TestingUtil.blockUntilViewsReceived(new CacheSPI[]{cache1, cache2}, 60000);
   }

   private CacheSPI<Object, Object> createCache(String name)
   {
      CacheSPI<Object, Object> cache = (CacheSPI<Object, Object>) new DefaultCacheFactory().createCache(UnitTestCacheConfigurationFactory.createConfiguration(CacheMode.REPL_SYNC), false);
      cache.getConfiguration().setClusterName(name);
      // Use marshaller
      cache.getConfiguration().setUseLazyDeserialization(useMarshalledValues);
      cache.getConfiguration().setUseRegionBasedMarshalling(!useMarshalledValues);

      cache.create();
      cache.start();
      return cache;
   }

   @AfterMethod(alwaysRun = true)
   public void tearDown() throws Exception
   {
      cache1.removeNode(Fqn.ROOT);
      if (cache1 != null)
      {
         log("stopping cache1");
         cache1.stop();
      }

      if (cache2 != null)
      {
         log("stopping cache2");
         cache2.stop();
      }

      resetContextClassLoader();
   }

   public void testPlainPut() throws Exception
   {
      cache1.put(aop, "person", ben_);
      Person ben2 = (Person) cache2.get(aop, "person");
      assertNotNull("Person from 2nd cache should not be null ", ben2);
      assertEquals(ben_.toString(), ben2.toString());
   }

   public void testCCE() throws Exception
   {
      ClassLoader c1 = getClassLoader();
      ClassLoader c2 = getClassLoader();

      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(c1);


         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(c2);
      }

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(c1);
      cache1.put(aop, "person", ben_);
      if (useMarshalledValues) resetContextClassLoader();
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(getFailingClassLoader());
      try
      {
         if (useMarshalledValues) Thread.currentThread().setContextClassLoader(c2);
         @SuppressWarnings("unused")
         Person person = (Person) cache2.get(aop, "person");
      }
      catch (ClassCastException ex)
      {
         // That's ok.
         return;
      }
      fail("Should have thrown an exception");
   }

   public void testPut() throws Exception
   {
      ClassLoader cla = getClassLoader();
      ClassLoader clb = getClassLoader();
      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);
         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      cache1.put(aop, "person", ben_);

      Object ben2;
      // Can't cast it to Person. CCE will result.
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      ben2 = cache2.get(aop, "person");
      assertEquals(ben_.toString(), ben2.toString());
   }

   public void testCLSet() throws Exception
   {
      ClassLoader cla = getClassLoader();
      ClassLoader clb = getClassLoader();
      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);
         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      cache1.put(aop, "person", ben_);

      Object ben2;
      // Can't cast it to Person. CCE will result.
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      ben2 = cache2.get(aop, "person");
      assertEquals(ben_.toString(), ben2.toString());

      Class claz = clb.loadClass(ADDRESS_CLASSNAME);
      Object add = claz.newInstance();
      Method setValue = claz.getMethod("setCity", String.class);
      setValue.invoke(add, "Sunnyvale");
      Class clasz1 = clb.loadClass(PERSON_CLASSNAME);
      setValue = clasz1.getMethod("setAddress", claz);
      setValue.invoke(ben2, add);
   }

   /**
    * Test replication with classloaders.
    *
    * @throws Exception
    */
   public void testCLSet2() throws Exception
   {
      ClassLoader cla = getClassLoader();
      ClassLoader clb = getClassLoader();

      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);
         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      cache1.put(aop, "person", ben_);

      Object ben2;
      // Can't cast it to Person. CCE will result.
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      ben2 = cache2.get(aop, "person");
      assertEquals(ben_.toString(), ben2.toString());

      Class claz = clb.loadClass(ADDRESS_CLASSNAME);
      Object add = claz.newInstance();
      Method setValue = claz.getMethod("setCity", String.class);
      setValue.invoke(add, "Sunnyvale");
      Class clasz1 = clb.loadClass(PERSON_CLASSNAME);
      setValue = clasz1.getMethod("setAddress", claz);
      setValue.invoke(ben2, add);

      // Set it back to the cache
      // Can't cast it to Person. CCE will resutl.
      cache2.put(aop, "person", ben2);

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      Object ben3 = cache1.get(aop, "person");
      assertEquals(ben2.toString(), ben3.toString());
   }

   public void testPuts() throws Exception
   {
      ClassLoader cla = getClassLoader();
      ClassLoader clb = getClassLoader();

      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);
         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }

      // Create an empty Person loaded by this classloader
      Object scopedBen1 = getPersonFromClassloader(cla);
      // Create another empty Person loaded by this classloader
      Object scopedBen2 = getPersonFromClassloader(clb);

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      cache1.put(Fqn.fromString("/aop/1"), "person", ben_);
      cache1.put(Fqn.fromString("/aop/2"), "person", scopedBen1);
      if (useMarshalledValues) resetContextClassLoader();
      Object ben2;
      // Can't cast it to Person. CCE will resutl.
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      ben2 = cache2.get(Fqn.fromString("/aop/1"), "person");
      if (useMarshalledValues) resetContextClassLoader();
      assertEquals(ben_.toString(), ben2.toString());

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      ben2 = cache2.get(Fqn.fromString("/aop/2"), "person");
      if (useMarshalledValues) resetContextClassLoader();
      assertFalse("cache2 deserialized with scoped classloader", ben2 instanceof Person);
      assertFalse("cache2 deserialized with cache2 classloader", scopedBen1.equals(ben2));
      assertEquals("scopedBen deserialized properly", scopedBen2, ben2);
   }

   public void testMethodCall() throws Exception
   {
      ClassLoader cla = getClassLoader();
      ClassLoader clb = getClassLoader();

      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);

         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      cache1.put(Fqn.fromString("/aop/1"), "person", ben_);
      cache1.remove(Fqn.fromString("/aop/1"), "person");
      HashMap<Object, Object> map = new HashMap<Object, Object>();
      map.put("1", "1");
      map.put("2", "2");

      cache1.put(Fqn.fromString("/aop/2"), map);
      cache1.removeNode(Fqn.fromString("/aop/2"));
      if (useMarshalledValues) resetContextClassLoader();
//      TestingUtil.sleepThread(1000);
   }

   public void testTxMethodCall() throws Exception
   {
      ClassLoader cla = getClassLoader();
      ClassLoader clb = getClassLoader();

      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);

         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }

      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      TransactionManager tm = beginTransaction();
      cache1.put(Fqn.fromString("/aop/1"), "person", ben_);
      cache1.remove(Fqn.fromString("/aop/1"), "person");
      HashMap<Object, Object> map = new HashMap<Object, Object>();
      map.put("1", "1");
      map.put("2", "2");
      cache1.put(Fqn.fromString("/aop/2"), map);
      cache1.removeNode(Fqn.fromString("/aop/2"));
      tm.commit();

//      TestingUtil.sleepThread(1000);
      if (useMarshalledValues) resetContextClassLoader();
   }

   public void testTxCLSet2() throws Exception
   {
      ClassLoader cla = getClassLoader();
      if (!useMarshalledValues)
      {
         Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
         r1.registerContextClassLoader(cla);
      }
      ClassLoader clb = getClassLoader();
      if (!useMarshalledValues)
      {
         Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
         r2.registerContextClassLoader(clb);
      }

      TransactionManager tm = beginTransaction();
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      cache1.put(aop, "person", ben_);
      tm.commit();
      if (useMarshalledValues) resetContextClassLoader();

      Object ben2;
      // Can't cast it to Person. CCE will resutl.
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      ben2 = cache2.get(aop, "person");
      if (useMarshalledValues) resetContextClassLoader();
      assertEquals(ben_.toString(), ben2.toString());

      Class claz = clb.loadClass(ADDRESS_CLASSNAME);
      Object add = claz.newInstance();
      Method setValue = claz.getMethod("setCity", String.class);
      setValue.invoke(add, "Sunnyvale");
      Class clasz1 = clb.loadClass(PERSON_CLASSNAME);
      setValue = clasz1.getMethod("setAddress", claz);
      setValue.invoke(ben2, add);

      // Set it back to the cache
      // Can't cast it to Person. CCE will resutl.
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(clb);
      cache2.put(aop, "person", ben2);
      if (useMarshalledValues) resetContextClassLoader();
      if (useMarshalledValues) Thread.currentThread().setContextClassLoader(cla);
      Object ben3 = cache1.get(aop, "person");
      if (useMarshalledValues) resetContextClassLoader();
      assertEquals(ben2.toString(), ben3.toString());
   }

   public void testStateTransfer() throws Exception
   {
      // Need to test out if app is not registered with beforehand??
   }

   @SuppressWarnings("unchecked")
   public void testCustomFqn() throws Exception
   {
      FooClassLoader cl1 = new FooClassLoader(Thread.currentThread().getContextClassLoader());
      Region r1 = cache1.getRegion(aop, false) == null ? cache1.getRegion(aop, true) : cache1.getRegion(aop, false);
      r1.registerContextClassLoader(cl1);
      FooClassLoader cl2 = new FooClassLoader(Thread.currentThread().getContextClassLoader());
      Region r2 = cache2.getRegion(aop, false) == null ? cache2.getRegion(aop, true) : cache2.getRegion(aop, false);
      r2.registerContextClassLoader(cl2);

      Class clazz = cl1.loadFoo();
      Object custom1 = clazz.newInstance();

      clazz = cl2.loadFoo();
      Object custom2 = clazz.newInstance();

      cache1.put(Fqn.fromElements("aop", custom1), "key", "value");

      try
      {
         Object val = cache2.get(Fqn.fromElements("aop", custom2), "key");
         assertEquals("value", val);
      }
      catch (Exception ex)
      {
         fail("Test fails with exception " + ex);
      }
   }

   private TransactionManager beginTransaction() throws SystemException, NotSupportedException
   {
      TransactionManager mgr = cache1.getConfiguration().getRuntimeConfig().getTransactionManager();
      mgr.begin();
      return mgr;
   }

   protected Object getPersonFromClassloader(ClassLoader cl) throws Exception
   {
      Class clazz = cl.loadClass(PERSON_CLASSNAME);
      return clazz.newInstance();
   }

   private void log(String msg)
   {
      System.out.println("-- [" + Thread.currentThread() + "]: " + msg);
   }

}
