/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/f/cdf' );
import F = require( './../../../../../base/dists/f/ctor' );
import entropy = require( './../../../../../base/dists/f/entropy' );
import kurtosis = require( './../../../../../base/dists/f/kurtosis' );
import mean = require( './../../../../../base/dists/f/mean' );
import mode = require( './../../../../../base/dists/f/mode' );
import pdf = require( './../../../../../base/dists/f/pdf' );
import quantile = require( './../../../../../base/dists/f/quantile' );
import skewness = require( './../../../../../base/dists/f/skewness' );
import stdev = require( './../../../../../base/dists/f/stdev' );
import variance = require( './../../../../../base/dists/f/variance' );

/**
* Interface describing the `f` namespace.
*/
interface Namespace {
	/**
	* F distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 1.0, 1.0 );
	* // returns ~0.608
	*
	* y = ns.cdf( 2.0, 8.0, 4.0 );
	* // returns ~0.737
	*
	* y = ns.cdf( -1.0, 2.0, 2.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 10.0, 2.0 );
	*
	* y = mycdf( 10.0 );
	* // returns ~0.906
	*
	* y = mycdf( 8.0 );
	* // returns ~0.884
	*/
	cdf: typeof cdf;

	/**
	* F distribution.
	*/
	F: typeof F;

	/**
	* Returns the differential entropy of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 3.0, 7.0 );
	* // returns ~1.298
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~1.12
	*
	* @example
	* var v = ns.entropy( 8.0, 7.0 );
	* // returns ~1.193
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 3.0, 9.0 );
	* // returns ~124.667
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns ~26.143
	*
	* @example
	* var v = ns.kurtosis( 8.0, 9.0 );
	* // returns ~100.167
	*
	* @example
	* var v = ns.kurtosis( 1.0, 8.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Returns the expected value of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 3.0, 5.0 );
	* // returns ~1.667
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns ~1.2
	*
	* @example
	* var v = ns.mean( 8.0, 4.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the mode of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns mode
	*
	* @example
	* var v = ns.mode( 3.0, 5.0 );
	* // returns ~0.238
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns ~0.429
	*
	* @example
	* var v = ns.mode( 8.0, 4.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mode( 3.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 3.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* F distribution probability density function (PDF).
	*
	* @param x - input value
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.5, 1.0 );
	* // returns ~0.057
	*
	* y = ns.pdf( 0.1, 1.0, 1.0 );
	* // returns ~0.915
	*
	* var mypdf = ns.pdf.factory( 6.0, 7.0 );
	* y = mypdf( 7.0 );
	* // returns ~0.004
	*
	* y = mypdf( 2.0 );
	* // returns ~0.166
	*/
	pdf: typeof pdf;

	/**
	* F distribution quantile function.
	*
	* @param p - input value
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 1.0, 1.0 );
	* // returns ~9.472
	*
	* y = ns.quantile( 0.5, 4.0, 2.0 );
	* // returns ~1.207
	*
	* var myquantile = ns.quantile.factory( 10.0, 2.0 );
	*
	* y = myquantile( 0.2 );
	* // returns ~0.527
	*
	* y = myquantile( 0.8 );
	* // returns ~4.382
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 3.0, 7.0 );
	* // returns 11.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~3.207
	*
	* @example
	* var v = ns.skewness( 8.0, 7.0 );
	* // returns ~10.088
	*
	* @example
	* var v = ns.skewness( 1.0, 4.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 3.0, 5.0 );
	* // returns ~3.333
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~1.122
	*
	* @example
	* var v = ns.stdev( 8.0, 5.0 );
	* // returns ~2.764
	*
	* @example
	* var v = ns.stdev( 1.0, 4.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of an F distribution.
	*
	* ## Notes
	*
	* -   If `d1 <= 0` or `d2 <= 0`, the function returns `NaN`.
	*
	* @param d1 - numerator degrees of freedom
	* @param d2 - denominator degrees of freedom
	* @returns variance
	*
	* @example
	* var v = ns.variance( 3.0, 5.0 );
	* // returns ~11.111
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~1.26
	*
	* @example
	* var v = ns.variance( 8.0, 5.0 );
	* // returns ~7.639
	*
	* @example
	* var v = ns.variance( 1.0, 4.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Fisher's F distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
