/* NVTV Gui -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: gui.c,v 1.52 2004/01/30 08:47:02 dthierbach Exp $
 *
 * Contents:
 *
 * The GTK graphical user interface.
 */

#include "local.h" /* before everything else */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>

#include <X11/Xmu/WinUtil.h>

#include <gdk/gdkprivate.h>
#include <gdk/gdkx.h>

#include "tv_chip.h"
#include "nvtv.h"
#include "data.h"
#include "data_bt.h"
#include "data_ch.h"
#include "gui.h"
#include "gui_bt.h"
#include "gui_cx.h"
#include "gui_ch.h"
#include "gui_ph.h"
#include "gui_nx.h"
#include "gui_nv.h"
#include "gui_tdfx.h"
#include "gui_i810.h"
#include "actions.h"
#include "print.h"
#include "backend.h"

/* -------- Global variables -------- */

GtkWidget *gui_main_window;

/* ---- GUI state ---- */

int gui_main_head, gui_tv_head, gui_video_head;
int gui_max_head;
Bool gui_change_heads = FALSE;

TVRegs gui_regs;

#define gui_set opt_set  /* FIXME */

int gui_set_saved = (TV_CAP_MASK & ~TV_CAP_DUALVIEW) | TV_DESC_DUALVIEW;
/* Dualview is set by default */

TVMode gui_orig_mode = {
  {TV_SYSTEM_NONE, 0, 0, "Monitor", "---", 0.0, 0.0}, descFlags: 0
};
TVMode gui_fetch_mode = {
  {TV_SYSTEM_NONE, 0, 0, "Fetch", "---", 0.0, 0.0}, descFlags: 0
};
TVRegs *gui_orig_regs = NULL; /* either NULL or points into gui_orig_mode */

static int gui_service_delays [] = 
  { 50, 100, 200, 400, 0 }; /* in ms */

int gui_service_delay = 0;
int gui_service_flags = 0;

TVChip gui_tv_chip = TV_NO_CHIP;

TVSystem gui_system = TV_SYSTEM_NONE;
TVMode *gui_list_mode = NULL;
TVMode *gui_act_mode = NULL;

Window gui_window = None;
int gui_window_x, gui_window_y;
int gui_window_width, gui_window_height;

Bool gui_auto = FALSE;
Bool gui_bypass = FALSE;
Bool gui_tvstate = FALSE; /* FALSE=off, TRUE=on */

DataFunc *gui_func = NULL;

gint gui_service_timeout_id = -1; /* service timeout */

/* ---- other variables ---- */

static CardPtr gui_card_list;

static CardPtr gui_card = NULL;

/* ---- "Changed" signal chains ---- */

GuiNotify *update_crt;
GuiNotify *update_chip;
GuiNotify *update_port;
GuiNotify *update_mode;
GuiNotify *update_set;
GuiNotify *update_service;
GuiNotify *changed_all;
GuiNotify *show_bt, *hide_bt;
GuiNotify *show_cx, *hide_cx;
GuiNotify *show_ch, *hide_ch;
GuiNotify *show_ph1, *hide_ph1;
GuiNotify *show_ph2, *hide_ph2;
GuiNotify *show_nx, *hide_nx;
GuiNotify *show_card[CARD_LAST], *hide_card[CARD_LAST];
GuiNotify *show_head, *hide_head;
GuiNotify *update_bypass;

/* -------- Accelerators -------- */

GuiAccel gui_accel[ACCEL_LAST] = {
  {"Switch TV on:",    "F1"},
  {"Switch TV off:",   "F2"},
  {"Switch X mode:",   "F4"},
  {"Center X window:", "F5"},
  {"Resize X window:", "F6"},
  {"Reset values:",    "F9"},
  {"Print values:",    "F10"},
};

GtkAccelGroup *gui_main_accel_group;

/* -------- GUI Masks -------- */

struct mask_mode_page {
  GtkOptionMenu *system;
  GtkCList *modes;
  gint column; /* last column sorted */
  GtkLabel *res;
  GtkLabel *size;
  GtkLabel *aspect;
  GtkLabel *overscan;
  GtkLabel *tv_state, *mon_state;
  GtkLabel *xwin_name;
  GtkWidget *xwin_frame;
  GtkWidget *xcenter, *xadjust;
  GtkWidget *xwin_pos, *xwin_size;
};

struct mask_mode_page gui_mode_mask;

struct mask_config_page {
  GtkLabel *io_addr, *probe;
  GtkOptionMenu *chip;
};

struct mask_config_page gui_config_mask;

struct mask_offset {
  GtkAdjustment *hadj, *vadj;
  gfloat hcenter, vcenter;
};

struct mask_offset gui_tv_offset, gui_mon_offset;
struct mask_offset gui_xwin_pos, gui_xwin_size;

struct mask_offset gui_view_pos;

struct mask_settings {
  GtkAdjustment *brightness_sig, *contrast_sig, *saturation_sig;
  GtkToggleButton *dualview, *macrovis, *monochrome, *noninterlaced,
    *colorfix;
  GSList *output;
  int connect;
};

struct mask_settings gui_settings;

struct mask_heads {
  GtkAdjustment *main, *tv, *video;
  GtkLabel *timer, *twin;
  GtkToggleButton *shared;
  GtkFrame *disp_mon[2];
  GtkFrame *disp_tv[2];
  GtkFrame *disp_fp[2];
};

struct mask_heads gui_heads;

/* -------- */

typedef struct set_mask {
  char *label;
  int xl, xr;
  int min, max;
  GtkAdjustment **adj;
  int *addr;
} GuiSetMask;

#define SET_ADJ(m) adj:&gui_settings.m 
#define SET_VAL(m) addr:&gui_set.m

static GuiSetMask gui_set_mask [] = {
  {"Contrast",    xl:1, xr:3, min:-100, max:100, SET_VAL(contrast)},
  {"Saturation",  xl:1, xr:3, min:-100, max:100, SET_VAL(saturation)},
  {"Brightness#", xl:1, xr:5, min: -50, max: 50, SET_VAL(brightness_sig), 
                                                 SET_ADJ(brightness_sig)},
  {"Contrast#",   xl:1, xr:5, min: -50, max: 50, SET_VAL(contrast_sig),
                                                 SET_ADJ(contrast_sig)},
  {"Saturation#", xl:1, xr:5, min: -50, max: 50, SET_VAL(saturation_sig),
                                                 SET_ADJ(saturation_sig)},
  {"Phase",       xl:1, xr:5, min: -60, max: 60, SET_VAL(phase)},
  {"Hue",         xl:1, xr:5, min: -60, max: 60, SET_VAL(hue)},
  {"Flicker",     xl:1, xr:3, min:   0, max:100, SET_VAL(flicker)},
  {"+",           xl:3, xr:5, min:   0, max:100, SET_VAL(flicker_adapt)},
  {"Bandwidth",   xl:1, xr:3, min:   0, max:100, SET_VAL(luma_bandwidth)},
  {"+",           xl:3, xr:5, min:   0, max:100, SET_VAL(chroma_bandwidth)},
  {"Sharpness",   xl:1, xr:3, min:   0, max:100, SET_VAL(sharpness)},
  {"+",           xl:3, xr:5, min:   0, max:100, SET_VAL(cross_color)},
  {NULL}
};

#define FIELD_FLAG(m) addr:&m, size:sizeof(int)

static GuiFlagMask service_mask_flag [] = {
  {label:"", mask: BACK_SERVICE_CURSOR,      FIELD_FLAG(gui_service_flags)},
  {label:"", mask: BACK_SERVICE_VIEW_CURSOR, FIELD_FLAG(gui_service_flags)},
  {label:"", mask: BACK_SERVICE_VIEW_MAIN,   FIELD_FLAG(gui_service_flags)}
};

static GuiFlagMask dev_port_mask_flag [] = {
  {label:"Monitor",   mask: DEV_MONITOR,    FIELD_FLAG(gui_regs.devFlags)},
  {label:"TV",        mask: DEV_TELEVISION, FIELD_FLAG(gui_regs.devFlags)},
  {label:"LCD",       mask: DEV_FLATPANEL,  FIELD_FLAG(gui_regs.devFlags)},
  {label:"Overlay",   mask: DEV_OVERLAY,    FIELD_FLAG(gui_regs.devFlags)},
  {label:"Intern",    mask: DEV_INTERNAL,   FIELD_FLAG(gui_regs.devFlags)},
  {NULL}
};

static GuiFlagMask host_port_mask_flag [] = {
  {label:"", mask: PORT_PCLK_MODE,         FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_PCLK_POLARITY,     FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_SYNC_DIR,          FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_HSYNC_POLARITY,    FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_VSYNC_POLARITY,    FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_BLANK_DIR,         FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_BLANK_POLARITY,    FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_BLANK_REGION,      FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_FORMAT_MASK_COLOR, FIELD_FLAG(gui_regs.portHost)},
  {label:"", mask: PORT_FORMAT_MASK_ALT,   FIELD_FLAG(gui_regs.portHost)},
};

static GuiFlagMask enc_port_mask_flag [] = {
  {label:"", mask: PORT_PCLK_MODE,         FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_PCLK_POLARITY,     FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_SYNC_DIR,          FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_HSYNC_POLARITY,    FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_VSYNC_POLARITY,    FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_BLANK_DIR,         FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_BLANK_POLARITY,    FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_BLANK_REGION,      FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_FORMAT_MASK_COLOR, FIELD_FLAG(gui_regs.portEnc)},
  {label:"", mask: PORT_FORMAT_MASK_ALT,   FIELD_FLAG(gui_regs.portEnc)},
};

/* -------- GTK Helper Routines -------- */

/*
 *  Create an option menu with labels. 
 */

GtkWidget* create_option_menu (GtkSignalFunc func, ...)
{
  va_list ap;
  GtkWidget *opt, *menu, *item;
  char *label;
  int i;

  va_start (ap, func);
  opt = gtk_option_menu_new();
  menu = gtk_menu_new();
  for (i = 0; TRUE; i++) {
    label = va_arg (ap, char *);
    if (!label) break;
    item = gtk_menu_item_new_with_label (label);
    if (func) 
      gtk_signal_connect (GTK_OBJECT (item), "activate", func, (gpointer) i);
    gtk_menu_append (GTK_MENU (menu), item);
  }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);
  va_end (ap);
  return opt;
}

/*
 *  Attempt to control the size of an entry 
 */

#if GTK_MAJOR_VERSION == 1
void entry_size_request_cb (GtkWidget *widget, GtkRequisition *req)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_ENTRY (widget));
  g_return_if_fail (req != NULL);

  /* No info about max or avg xsize, so guess */
  if (((GtkEntry *) widget)->text_max_length > 0) {
    req->width = ((GtkEntry *) widget)->text_max_length 
      * (widget->style->font->ascent + widget->style->font->descent) 
      + (widget->style->klass->xthickness + 2) * 2;
  }
}
#endif

GtkWidget *create_entry_with_size (gint max_length)
{
  GtkWidget *entry;

  entry = gtk_entry_new_with_max_length (max_length);
#if GTK_MAJOR_VERSION == 1
  gtk_signal_connect_after (GTK_OBJECT(entry), "size_request",  
			    GTK_SIGNAL_FUNC(entry_size_request_cb), NULL);
#endif
  return entry;
}

/*
 *  Attempt to control the size of a spinbutton
 */

#if GTK_MAJOR_VERSION == 1
void spinbutton_size_request_cb (GtkWidget *widget, GtkRequisition *req)
{
  GtkAdjustment *adj;
  register int s;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SPIN_BUTTON (widget));
  g_return_if_fail (req != NULL);

  /* WARNING: This is a very crude estimation of average font character width
     and arrow size using ascent and descent, and probably will not work with 
     all fonts. */
  adj = ((GtkSpinButton *) widget)->adjustment;

  if (adj) {
    s = ((int) log10 (adj->upper - adj->lower)) + 1 + 
        ((GtkSpinButton *) widget)->digits;
    s = s * (widget->style->font->ascent - widget->style->font->descent) +
        6 * widget->style->klass->xthickness + 4 +
        (widget->style->font->ascent + widget->style->font->descent - 1) / 2;
    /* ARROW_SIZE == 11 */
    req->width = s;
  }
}
#endif

GtkWidget *create_spinbutton_with_size (GtkAdjustment *adjustment,
  gfloat climb_rate, guint digits)
{
  GtkWidget *spin;

  spin = gtk_spin_button_new (adjustment, climb_rate, digits);
#if GTK_MAJOR_VERSION == 1
  gtk_signal_connect_after (GTK_OBJECT(spin), "size_request",  
			    GTK_SIGNAL_FUNC(spinbutton_size_request_cb), NULL);
#endif
  return spin;
}

/*
 *  Create an arrow button and attach to table.
 */

GtkWidget *create_arrow_button (GtkTable *table, GtkArrowType arrow_type,
  int x0, int x1, int y0, int y1)
{
  GtkWidget *button;
  GtkWidget *arrow;

  button = gtk_button_new();
  if (arrow_type >= GTK_ARROW_UP && arrow_type <= GTK_ARROW_RIGHT) {
    arrow = gtk_arrow_new (arrow_type, GTK_SHADOW_OUT);
    gtk_widget_set_usize (arrow, 20, 20); 
    gtk_container_add (GTK_CONTAINER (button), arrow);
  }
  gtk_table_attach (table, button, x0, x1, y0, y1, 
    GTK_FILL, GTK_FILL, 0,0);
  return button;
}

/*
 *  Create a vertical or horizontal radio box. Store radio group if
 *  non-null.
 */

GtkWidget *create_radio_box (int active, gboolean hor_vert, 
  GtkSignalFunc func, GSList **group, ...)
{
  GtkWidget *bbox;
  GtkWidget *button;
  char *label;
  GSList *slist;
  va_list ap;
  int i;

  va_start (ap, group);
  if (hor_vert) {
    bbox = gtk_hbox_new (FALSE, 0);
  } else {
    bbox = gtk_vbox_new (FALSE, 0);
  }
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
  button = NULL; slist = NULL;
  for (i = 0; TRUE; i++) {
    label = va_arg (ap, char *);
    if (!label) break;
    if (button) slist = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
    button = gtk_radio_button_new_with_label (slist, label);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button),
      (active == i) ? TRUE : FALSE);
    if (func) 
      gtk_signal_connect (GTK_OBJECT (button), "toggled", func, (gpointer) i);
    gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, FALSE, 0);
  }
  va_end (ap);
  if (group) {
    *group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
  }
  return bbox;
}

/*
 * Set active index of radio group
 */

void set_radio_group_active (GSList *group, int active)
{
  group = g_slist_nth (group, g_slist_length (group) - active - 1);
  if (group) 
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (group->data), TRUE);
  }
}

/*
 * Create justified label
 */

GtkWidget *create_aligned_label (const char *str, gfloat xalign, gfloat yalign)
{
  GtkWidget *label;

  label = gtk_label_new (str);  
  gtk_misc_set_alignment (GTK_MISC (label), xalign, yalign);
  return label;
}

/* 
 * Create notify widget (misuse adjustment for now) 
 */

GuiNotify *create_notify (void)
{
  return (GuiNotify *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
}

/*
 * Change adjustment only if new value is different
 */

void change_adjustment (GtkAdjustment *adj, int value, GtkSignalFunc func)
{
  if (adj->value != value) {
    if (func) 
      gtk_signal_handler_block_by_func (GTK_OBJECT (adj), func, NULL);
    gtk_adjustment_set_value (adj, value);
    if (func) 
      gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj), func, NULL);
  }
}

/*
 * Connect show and hide signals 
 */

void set_show_hide (GtkWidget *page, GtkAdjustment *show, GtkAdjustment *hide)
{
  gtk_signal_connect_object (GTK_OBJECT (show), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));
}

/* -------- State methods -------- */

/* ---- register changed (on signal chain) ---- */

void reg_changed_cb (GtkAdjustment *adj, gpointer data)
{
  GuiRegMask *m = (GuiRegMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");

  if (m->size == sizeof(int)) {
    *((int *) m->addr) = (int) adj->value; 
  } else if (m->size == sizeof(long)) {
    *((long *) m->addr) = (long) adj->value;
  }
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void reg_low_changed_cb (GtkAdjustment *adj, gpointer data)
{
  GuiRegMask *m = (GuiRegMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  long l;

  if (m->size != sizeof(long)) {
    RAISE (MSG_ERROR, "reg_low_changed: not a long value");
    return;
  }
  l = *((long *) m->addr);
  l = (l / 100000) * 100000 + (long) adj->value;
  *((long *) m->addr) = l;
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void reg_high_changed_cb (GtkAdjustment *adj, gpointer data)
{
  GuiRegMask *m = (GuiRegMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  long l;

  if (m->size != sizeof(long)) {
    RAISE (MSG_ERROR, "reg_high_changed: not a long value");
    return;
  }
  l = *((long *) m->addr);
  l = (l % 100000) + ((long) adj->value * 100000);
  *((long *) m->addr) = l;
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

/* ---- register update (on signal chain) ---- */

void reg_update_cb (GtkAdjustment *adj, gpointer data)
{
  GuiRegMask *m = (GuiRegMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = 0.0;
  if (m->size == sizeof(int)) {
    val = (gfloat) *((int *) m->addr);
  } else if (m->size == sizeof(long)) {
    val = (gfloat) *((long *) m->addr);
  }
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_changed_cb), NULL);
}

void reg_low_update_cb (GtkAdjustment *adj, gpointer data)
{
  GuiRegMask *m = (GuiRegMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = 0.0;
  if (m->size != sizeof(long)) {
    RAISE (MSG_ERROR, "reg_low_update: not a long value");
    return;
  }
  val = (gfloat) (*((long *) m->addr) % 100000);
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_low_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_low_changed_cb), NULL);
}

void reg_high_update_cb (GtkAdjustment *adj, gpointer data)
{
  GuiRegMask *m = (GuiRegMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = 0.0;
  if (m->size != sizeof(long)) {
    RAISE (MSG_ERROR, "reg_high_update: not a long value");
    return;
  }
  val = (gfloat) (*((long *) m->addr) / 100000);
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_high_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_high_changed_cb), NULL);
}

/* ---- flag (on signal chain) ---- */

void flag_changed_cb (GtkWidget *button, gpointer data)
{
  GuiFlagMask *m = (GuiFlagMask *) 
    gtk_object_get_data (GTK_OBJECT(button), "mask");

  if (GTK_TOGGLE_BUTTON (button)->active) {
    *((int *) m->addr) |= m->mask;
  } else {
    *((int *) m->addr) &= ~m->mask;
  }
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void flag_opt_changed_cb (GtkWidget *opt, gpointer data)
{
  GuiFlagMask *m = (GuiFlagMask *) 
    gtk_object_get_data (GTK_OBJECT(opt), "mask");

  if (data) {
    *((int *) m->addr) |= m->mask;
  } else {
    *((int *) m->addr) &= ~m->mask;
  }
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void flag_update_cb (GtkWidget *button, gpointer data)
{
  GuiFlagMask *m = (GuiFlagMask *) 
    gtk_object_get_data (GTK_OBJECT(button), "mask");

  gtk_signal_handler_block_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (flag_changed_cb), NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button),
    (*((int *) m->addr) & m->mask) ? TRUE : FALSE);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (flag_changed_cb), NULL);
}

void flag_opt_update_cb (GtkWidget *opt, gpointer data)
{
  GuiFlagMask *m = (GuiFlagMask *) 
    gtk_object_get_data (GTK_OBJECT(opt), "mask");

  /* No need to block changed signal */
  gtk_option_menu_set_history (GTK_OPTION_MENU (opt), 
    (*((int *) m->addr) & m->mask) ? 1 : 0);
}

/* ---- setting (on signal chain) ---- */

void set_changed_cb (GtkAdjustment *adj, gpointer data)
{
  GuiSetMask *m = (GuiSetMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");

  RAISE (MSG_DEBUG, "set_changed %p %s", m, m->label);
  *((int *) m->addr) = (int) adj->value; 
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void set_update_cb (GtkAdjustment *adj, gpointer data)
{
  GuiSetMask *m = (GuiSetMask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = (gfloat) *((int *) m->addr);
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (set_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (set_changed_cb), NULL);
}

/* ---- port registers ---- */

void gui_port_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_port), "changed");
}

void gui_port_set (TVRegs *regs)
{
  gui_regs.devFlags = regs->devFlags;
  gui_regs.portHost = regs->portHost;
  gui_regs.portEnc  = regs->portEnc;
  RAISE (MSG_DEBUG, "gui_set_port");
  gui_port_changed ();
}

/* ---- crt registers ---- */

void gui_crt_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_crt), "changed");
}

void gui_crt_set (TVCrtcRegs *crtc)
{
  gui_regs.crtc = *crtc;
  RAISE (MSG_DEBUG, "gui_set_crt");
  gui_crt_changed ();
}

/* ---- chip registers ---- */

void gui_tv_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_chip), "changed");
}

void gui_tv_set (void)
{
  gui_tv_changed ();
}

/* ---- both registers ---- */

void gui_regs_set (TVRegs *regs, Bool setup)
{
  if (regs) gui_regs = *regs;
  if (gui_func) {
    if (setup) gui_func->setup (&gui_set, &gui_regs);
  }
  gui_crt_changed ();
  gui_tv_changed ();
  gui_port_changed ();
}

/* -------- Settings -------- */

void gui_get_settings_flags (void)
{
  gui_set.flags = 0;
  if (gui_settings.macrovis->active)      
    gui_set.flags |= TV_DESC_MACROVISION;
  if (gui_settings.monochrome->active)    
    gui_set.flags |= TV_DESC_MONOCHROME;
  if (gui_settings.noninterlaced->active) 
    gui_set.flags |= TV_DESC_NONINTERLACED;
  if (gui_settings.colorfix->active) 
    gui_set.flags |= TV_DESC_COLORFIX;
  if (gui_settings.dualview->active) {
    gui_set.flags |= TV_DESC_DUALVIEW;
    gui_regs.devFlags |= DEV_MONITOR;
  } else {
    gui_regs.devFlags &= ~DEV_MONITOR;
  }
}

void gui_set_settings_flags (void)
{
  RAISE (MSG_DEBUG, "set settings flags");
  gtk_toggle_button_set_active (gui_settings.dualview,
    (gui_set.flags & TV_DESC_DUALVIEW) ? TRUE : FALSE);
  gtk_toggle_button_set_active (gui_settings.macrovis, 
    (gui_set.flags & TV_DESC_MACROVISION) ? TRUE : FALSE);
  gtk_toggle_button_set_active (gui_settings.monochrome,
    (gui_set.flags & TV_DESC_MONOCHROME) ? TRUE : FALSE);
  gtk_toggle_button_set_active (gui_settings.noninterlaced,
    (gui_set.flags & TV_DESC_NONINTERLACED) ? TRUE : FALSE);
  gtk_toggle_button_set_active (gui_settings.colorfix,
    (gui_set.flags & TV_DESC_COLORFIX) ? TRUE : FALSE);
}

void gui_get_settings (void)
{
  RAISE (MSG_DEBUG, "get settings");
  gui_set.tv_hoffset  = gui_tv_offset.hadj->value;
  gui_set.tv_voffset  = gui_tv_offset.vadj->value;
  gui_set.mon_hoffset = gui_mon_offset.hadj->value;
  gui_set.mon_voffset = gui_mon_offset.vadj->value;
  gui_set.connector   = gui_settings.connect;
}

void settings_cb (GtkObject *obj, gpointer data);

void gui_set_settings (void)
{
  static int lock = 0;

  RAISE (MSG_DEBUG, "set settings %i", lock);
  if (lock > 0) return;
  lock++;
  change_adjustment (gui_tv_offset.hadj,  gui_set.tv_hoffset,
    GTK_SIGNAL_FUNC (settings_cb));
  change_adjustment (gui_tv_offset.vadj,  gui_set.tv_voffset,
    GTK_SIGNAL_FUNC (settings_cb));
  change_adjustment (gui_mon_offset.hadj, gui_set.mon_hoffset,
    GTK_SIGNAL_FUNC (settings_cb));
  change_adjustment (gui_mon_offset.vadj, gui_set.mon_voffset,
    GTK_SIGNAL_FUNC (settings_cb));
  RAISE (MSG_DEBUG, "set settings changed");
  gtk_signal_emit_by_name (GTK_OBJECT (update_set), "changed");
  lock--;
  RAISE (MSG_DEBUG, "set settings done %i", lock);
}

void settings_cb (GtkObject *obj, gpointer data)
{
  static int lock = 0;

  RAISE (MSG_DEBUG, "settings_cb %i", lock);
  if (lock > 0) return;
  lock++;
  if (gui_func) {
    gui_get_settings ();
    gui_func->clamp (&gui_set, &gui_regs);
    gui_set_settings ();
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
  lock--;
  RAISE (MSG_DEBUG, "settings_cb done %i", lock);
}

void settings_flags_cb (GtkObject *obj, gpointer data)
{
  static int lock = 0;

  RAISE (MSG_DEBUG, "settings_flags_cb %i", lock);
  if (lock > 0) return;
  lock++;
  if (gui_func) {
    gui_get_settings_flags ();
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
  lock--;
}

void setup_cb (GtkWidget *button, gpointer data)
{
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data), TRUE);
  gui_regs_set (NULL, TRUE);
}

void bypass_cb (GtkWidget *button, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (button)->active) {
    gui_bypass = TRUE; 
  } else {
    gui_bypass = FALSE; 
  }
  gtk_signal_emit_by_name (GTK_OBJECT (update_bypass), "changed");
}

void bypass_update_cb (GtkAdjustment *adj, GtkObject *button)
{
  gtk_signal_handler_block_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (bypass_cb), NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), gui_bypass);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (bypass_cb), NULL);
}

/* ---- actual mode ---- */

void gui_act_mode_changed (void)
{
  register int mask;

  if (gui_act_mode) {
    gui_regs_set (&gui_act_mode->regs, gui_bypass);
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
    gui_get_settings_flags ();
    /* save flags that were available and are forced now */
    mask = (gui_set_saved & ~gui_act_mode->descFlags & TV_CAP_MASK) >> 
      TV_CAP_BIT;
    gui_set_saved &= ~mask;
    gui_set_saved |= mask & gui_set.flags;
    /* restore flags that were forced and are available now */
    mask = (~gui_set_saved & gui_act_mode->descFlags & TV_CAP_MASK) >> 
      TV_CAP_BIT;
    gui_set.flags &= ~mask;
    gui_set.flags |= mask & gui_set_saved;
    /* force flags */
    mask = (~gui_act_mode->descFlags & TV_CAP_MASK) >> TV_CAP_BIT;
    gui_set.flags &= ~mask;
    gui_set.flags |= mask & gui_act_mode->descFlags;
    /* save force mask */
    gui_set_saved &= ~TV_CAP_MASK;
    gui_set_saved |= TV_CAP_MASK & gui_act_mode->descFlags;
    gui_set_settings_flags ();
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.dualview), 
      (gui_act_mode->descFlags & TV_CAP_DUALVIEW) ? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.macrovis), 
      (gui_act_mode->descFlags & TV_CAP_MACROVISION) ? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.monochrome), 
      (gui_act_mode->descFlags & TV_CAP_MONOCHROME) ? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.noninterlaced), 
      (gui_act_mode->descFlags & TV_CAP_NONINTERLACED) ? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.colorfix), 
      (gui_act_mode->descFlags & TV_CAP_COLORFIX) ? TRUE : FALSE);
  }
}

void gui_act_mode_set (TVMode *mode)
{
  char s[20];

  gui_act_mode = mode;
  if (gui_act_mode) {
    snprintf (s, 20, "%3i x %3i", 
	      gui_act_mode->spec.res_x, gui_act_mode->spec.res_y);
    gtk_label_set_text (gui_mode_mask.res, s);
    gtk_label_set_text (gui_mode_mask.size, gui_act_mode->spec.size);
    gtk_label_set_text (gui_mode_mask.aspect, gui_act_mode->spec.aspect);
    snprintf (s, 20, "%05.2f x %05.2f", 
	      gui_act_mode->spec.hoc, gui_act_mode->spec.voc);
    gtk_label_set_text (gui_mode_mask.overscan, s);
  } else {
    gtk_label_set_text (gui_mode_mask.res, "<None>");
    gtk_label_set_text (gui_mode_mask.size, "<None>");
    gtk_label_set_text (gui_mode_mask.overscan, "<None>");
  }
  gui_act_mode_changed ();
}

/* ---- list mode (current selection) ---- */

void gui_list_mode_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_mode), "changed");
}

void gui_list_mode_cb (GtkCList *clist, gint row, gint column,
  GdkEventButton *event, gpointer data)
{
  gui_list_mode = gtk_clist_get_row_data (gui_mode_mask.modes, row);
  gui_list_mode_changed ();
  gui_act_mode_set (gui_list_mode);
}

/* ---- list of all modes ---- */

/*
 * Set list of modes selected by gui_system.
 */

void gui_modes_init (void)
{
  TVMode *m;
  char sr[20], ss[20], so[20], sa[10];
  char *text [5];
  int row;
  int select;

  /* TODO FIXME: Use listModes */
  if (gui_func && gui_card) {
    m = gui_func->modes ();
  } else {
    m = NULL;
  }
  gtk_clist_freeze (gui_mode_mask.modes);
  gtk_clist_clear (gui_mode_mask.modes);
  if (m) {
    row = 0; select = 0;
    while (m->spec.system != TV_SYSTEM_NONE) {
      if (m->spec.system == gui_system) {
	if (opt_size) {
	  if (m->spec.res_x == opt_res_x && m->spec.res_y == opt_res_y && 
	      strcasecmp (m->spec.size, opt_size) == 0) 
	  {
	    select = row;
	  }
	}
	snprintf (sr, 20, "%3i x %3i", m->spec.res_x, m->spec.res_y);
	snprintf (ss, 20, "%s", m->spec.size);
	snprintf (so, 20, "%05.2f x %05.2f", m->spec.hoc, m->spec.voc);
	snprintf (sa, 10, "%s", m->spec.aspect); 
        /* FIXME relative aspect? */
	/* snprintf (sa, 10, "%7.5f", aspect); */
	text [0] = sr;
	text [1] = ss;
	text [2] = so;
	text [3] = sa;
	gtk_clist_append (gui_mode_mask.modes, text);
	gtk_clist_set_row_data (gui_mode_mask.modes, row, m);
	row++;
      }
      m++;
    }
    gtk_clist_select_row (gui_mode_mask.modes, select, 0); 
  }
  gui_mode_mask.column = -1;
  gtk_clist_thaw (gui_mode_mask.modes);
  /* FIXME: Create custom mode if opt_hoc and opt_voc are set */
}

void gui_modes_column_cb (GtkCList *clist, gint column, 
  struct mask_mode_page *m)
{
  g_return_if_fail (GTK_IS_CLIST (m->modes));
  if (m->column == column) {
    gtk_clist_set_sort_type (m->modes, GTK_SORT_DESCENDING);
    m->column = -1;
  } else {
    gtk_clist_set_sort_type (m->modes, GTK_SORT_ASCENDING);
    m->column = column;
  }
  gtk_clist_set_sort_column (m->modes, column); 
  gtk_clist_sort (m->modes);
  gtk_clist_set_sort_type (m->modes, GTK_SORT_ASCENDING);
}

/* ---- tv system ---- */

void gui_system_changed (void)
{
  gui_modes_init ();
}

void gui_system_cb (GtkWidget *widget, gpointer data)
{
  gui_system = (TVSystem) data;
  gui_system_changed ();
}

void gui_system_set (TVSystem system)
{
  gui_system = system;
  if (gui_system > TV_SYSTEM_NONE)
    gtk_option_menu_set_history (gui_mode_mask.system, gui_system);
  gui_system_changed ();
}

void gui_system_init (void)
{
  gui_system = TV_SYSTEM_PAL; /* Must have some default. */
  if (opt_system != TV_SYSTEM_NONE) {
    gui_system = opt_system;
  }
  if (gui_system > TV_SYSTEM_NONE) {
    gtk_option_menu_set_history (gui_mode_mask.system, gui_system);
  }
  /* mode will be init by caller */
}

/* -------- Heads -------- */

void head_cb (GtkWidget *widget, gpointer data);
void view_cb (GtkWidget *widget, gpointer data);
gint service_view_cb (gpointer data);

void gui_set_heads (void)
{
  gtk_signal_handler_block_by_func (GTK_OBJECT (gui_heads.main),
    GTK_SIGNAL_FUNC (head_cb), NULL);
  gtk_signal_handler_block_by_func (GTK_OBJECT (gui_heads.tv),
    GTK_SIGNAL_FUNC (head_cb), NULL);
  gtk_signal_handler_block_by_func (GTK_OBJECT (gui_heads.video),
    GTK_SIGNAL_FUNC (head_cb), NULL);
  gtk_adjustment_set_value (gui_heads.main,  gui_main_head);
  gtk_adjustment_set_value (gui_heads.tv,    gui_tv_head); 
  gtk_adjustment_set_value (gui_heads.video, gui_video_head); 
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_heads.main),
    GTK_SIGNAL_FUNC (head_cb), NULL);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_heads.tv),
    GTK_SIGNAL_FUNC (head_cb), NULL);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_heads.video),
    GTK_SIGNAL_FUNC (head_cb), NULL);
}

void gui_get_heads (void) 
{
  gui_main_head  = gui_heads.main->value;  
  gui_tv_head    = gui_heads.tv->value;    
  gui_video_head = gui_heads.video->value; 
}

void head_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "head");
  gui_get_heads ();
  back_card->setHeads (gui_main_head, gui_tv_head, gui_video_head);
  back_card->getHeads (&gui_main_head, &gui_tv_head, &gui_video_head, NULL);
  gui_set_heads ();
}

void head_update_cb (GtkWidget *widget, gpointer data)
{
  int flags;
  int i;

  for (i = 0; i < gui_max_head; i++) {
    back_card->getHeadDev (i+1, &flags);
    gtk_frame_set_shadow_type (gui_heads.disp_mon[i], 
      (flags & DEV_MONITOR)    ? GTK_SHADOW_IN : GTK_SHADOW_ETCHED_OUT);
    gtk_frame_set_shadow_type (gui_heads.disp_tv[i], 
      (flags & DEV_TELEVISION) ? GTK_SHADOW_IN : GTK_SHADOW_ETCHED_OUT);
    gtk_frame_set_shadow_type (gui_heads.disp_fp[i], 
      (flags & DEV_FLATPANEL)  ? GTK_SHADOW_IN : GTK_SHADOW_ETCHED_OUT);
  }
}

void gui_set_view (int view_x, int view_y)
{
  gtk_signal_handler_block_by_func (GTK_OBJECT (gui_view_pos.hadj),
    GTK_SIGNAL_FUNC (view_cb), NULL);
  gtk_signal_handler_block_by_func (GTK_OBJECT (gui_view_pos.vadj),
    GTK_SIGNAL_FUNC (view_cb), NULL);
  gtk_adjustment_set_value (gui_view_pos.hadj, view_x);
  gtk_adjustment_set_value (gui_view_pos.vadj, view_y);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_view_pos.hadj),
    GTK_SIGNAL_FUNC (view_cb), NULL);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_view_pos.vadj),
    GTK_SIGNAL_FUNC (view_cb), NULL);
}

void duplicate_cb (GtkWidget *widget, gpointer data)
{
  int x, y;

  RAISE (MSG_DEBUG, "duplicate");
  if (GTK_TOGGLE_BUTTON (widget)->active) {
    if (gui_main_head != gui_tv_head) {
      gtk_label_set_text (gui_heads.twin, "lost");
      back_card->initSharedView (&x, &y);
      gui_view_pos.hcenter = x;
      gui_view_pos.vcenter = y;
      gui_set_view (x, y);
      /* FIXME set active */
    }
  } else {
    /* FIXME set inactive */
  }
}

void view_cb (GtkWidget *widget, gpointer data)
{
  int view_x, view_y;

  RAISE (MSG_DEBUG, "view pos");
  view_x = (int) gui_view_pos.hadj->value;
  view_y = (int) gui_view_pos.vadj->value;
  if (back_card->adjustViewport (gui_service_flags, &view_x, &view_y))
  {
    gui_set_view (view_x, view_y);
  }
}

void service_timeout (void)
{
  char s[20];

  gui_service_timeout_id = gtk_timeout_add (
    gui_service_delays [gui_service_delay], 
    (GtkFunction) service_view_cb, NULL);
  snprintf (s, 20, "%4i ms", gui_service_delays [gui_service_delay]);
  gtk_label_set_text (gui_heads.timer, s);
}

gint service_view_cb (gpointer data)
{
  int view_x, view_y;
  Window root, child;
  int root_x, root_y;
  int win_x, win_y;
  unsigned int mask;

  if (gui_tvstate && gui_main_head != gui_tv_head && gui_service_flags != 0) {
    XQueryPointer (gdk_display, my_gdk_root_window, &root, &child, 
		   &root_x, &root_y, &win_x, &win_y, &mask);
    RAISE (MSG_DEBUG, "service %i,%i", root_x, root_y);
    view_x = (int) gui_view_pos.hadj->value;
    view_y = (int) gui_view_pos.vadj->value;
    if (back_card->serviceViewportCursor (gui_service_flags, 
         root_x, root_y, &view_x, &view_y))
    {
      gui_set_view (view_x, view_y);
      gui_service_delay = 0;
    } else {
      if (gui_service_delays [gui_service_delay + 1] != 0)
	gui_service_delay++;
    }
  } else {
    if (gui_service_delays [gui_service_delay + 1] != 0)
      gui_service_delay++;
  }
  service_timeout ();
  return FALSE; /* continue */
}

void service_flag_cb (GtkWidget *widget, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (widget)->active) {
    gui_service_flags |= (int) data;
  } else {
    gui_service_flags &= ~ (int) data;
  }
  RAISE (MSG_DEBUG, "service flag %x", gui_service_flags);
}

void service_update_cb (GtkWidget *button, gpointer data)
{
  GuiFlagMask *m = (GuiFlagMask *) 
    gtk_object_get_data (GTK_OBJECT(button), "mask");

  gtk_signal_handler_block_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (service_flag_cb), (gpointer) m->mask);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button),
    (*((int *) m->addr) & m->mask) ? TRUE : FALSE);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (service_flag_cb), (gpointer) m->mask);
}

/* ---- connector ---- */

void gui_connector_cb (GtkWidget *widget, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (widget)->active)
  {
    gui_settings.connect = (TVConnect) data;
  }
  settings_cb (NULL, 0);
}

/*
 *  Check which TV connections are available, and set radio group.
 */

void gui_connector_set (TVConnect connect)
{
  RAISE (MSG_DEBUG, "set connect %i %i", connect, opt_connect);
  if (connect == CONNECT_AUTO)
  {
    RAISE (MSG_DEBUG, "auto connect");
    connect = back_card->getConnection ();
  }
  gui_settings.connect = connect;
  set_radio_group_active (gui_settings.output, (int) connect);
}

void gui_connector_init (void)
{
  if (opt_connect == CONNECT_NONE) {
    gui_connector_set (CONNECT_AUTO);
  } else {
    gui_connector_set (opt_connect);
  }
}

void check_monitor_cb (GtkWidget *widget, gpointer data)
{
  gui_connector_set (CONNECT_AUTO);
}

/* ---- chip (I2C bus device) ---- */

void gui_device_changed (TVChip last_type)
{
  RAISE (MSG_DEBUG, "gui_device_changed %i %i", last_type, gui_tv_chip);
  if (last_type != gui_tv_chip) 
  {
    switch (last_type & TV_ENCODER) 
    {
      case TV_BROOKTREE:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_bt), "changed");
	break;
      case TV_CONEXANT:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_cx), "changed");
	break;
      case TV_CHRONTEL_MODEL1:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_ch), "changed");
	break;
      case TV_CHRONTEL_MODEL2: /* FIXME */
	gtk_signal_emit_by_name (GTK_OBJECT (hide_ch), "changed");
	break;
      case TV_PHILIPS_MODEL1:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_ph1), "changed");
	break;
      case TV_PHILIPS_MODEL2:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_ph2), "changed");
	break;
      case TV_NVIDIA:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_nx), "changed");
	break;
      default: 
	break;
    }
  }
  /* Always, because card could have changed */
  gui_func = data_func (gui_card->type, gui_tv_chip); 
  if (last_type != gui_tv_chip) 
  {
    switch (gui_tv_chip & TV_ENCODER) 
    {
      case TV_BROOKTREE:
	gtk_signal_emit_by_name (GTK_OBJECT (show_bt), "changed");
	break;
      case TV_CONEXANT:
	gtk_signal_emit_by_name (GTK_OBJECT (show_cx), "changed");
	break;
      case TV_CHRONTEL_MODEL1:
	gtk_signal_emit_by_name (GTK_OBJECT (show_ch), "changed");
	break;
      case TV_CHRONTEL_MODEL2: /* FIXME */
	gtk_signal_emit_by_name (GTK_OBJECT (show_ch), "changed");
	break;
      case TV_PHILIPS_MODEL1:
	gtk_signal_emit_by_name (GTK_OBJECT (show_ph1), "changed");
	break;
      case TV_PHILIPS_MODEL2:
	gtk_signal_emit_by_name (GTK_OBJECT (show_ph2), "changed");
	break;
      case TV_NVIDIA:
	gtk_signal_emit_by_name (GTK_OBJECT (show_nx), "changed");
	break;
      default: 
	break;
    }
    if (gui_func) gui_func->defaults (&gui_set);
    modifySet (&gui_set, opt_set_act);
    gui_func->clamp (&gui_set, &gui_regs);
    gui_set_settings ();
  }
  gui_connector_init ();
  gui_modes_init (); 
  /* FIXME: Should set head here */
  RAISE (MSG_DEBUG, "gui_device_changed end");
}

/*
 * Set selected I2C device (TV chip).
 */

void gui_device_cb (GtkWidget *widget, ChipPtr chip)
{
  TVChip last_type;

  last_type = gui_tv_chip;
  back_card->setChip (chip, TRUE);
  gui_tv_chip = chip->type;
  gui_device_changed (last_type);
}

void gui_device_set (ChipPtr chip)
{
  TVChip last_type;
  int i;
  ChipPtr c;

  last_type = gui_tv_chip;
  if (!chip) return;
  back_card->setChip (chip, TRUE);
  gui_tv_chip = chip->type;
  for (i = 0, c = gui_card->chips; c; c = c->next, i++)
  {
    if (c == chip) 
      gtk_option_menu_set_history (gui_config_mask.chip, i);
  }
  gui_device_changed (last_type);
  RAISE (MSG_DEBUG, "gui_device_set end");
}

void gui_device_init (Bool probe)
{
  GtkWidget *menu;
  GtkWidget *item;
  ChipPtr c, ca;

  RAISE (MSG_DEBUG, "gui_device_init");
  menu = gtk_menu_new ();
  if (probe) back_card->probeChips ();
  for (c = gui_card->chips; c; c = c->next)
  {
    item = gtk_menu_item_new_with_label (c->name);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC(gui_device_cb), (gpointer) c);
    gtk_menu_append (GTK_MENU (menu), item);
  }
  if (!gui_card->chips) {
    item = gtk_menu_item_new_with_label ("<No Device>");
    gtk_menu_append (GTK_MENU (menu), item);
  }
  /* unrefs and frees the old menu ... hopefully. */
  gtk_option_menu_set_menu (gui_config_mask.chip, menu);
  /* make labels visible */
  gtk_widget_show_all (menu);
  gui_device_set (findDefOptChip (gui_card));
  RAISE (MSG_DEBUG, "gui_device_init end");
}

void check_chip_cb (GtkWidget *widget, gpointer data)
{
  gui_device_init (TRUE);
}

/* ---- pci card ---- */

void gui_card_changed (CardPtr last_card)
{
  int x, y;
  char s [30];

  RAISE (MSG_DEBUG, "gui_card_changed");
  if (gui_card) {
    snprintf (s, 30, "%s", gui_card->arch);
  } else {
    strncpy (s, "(No card)", 20);
  }
  gtk_label_set_text (gui_config_mask.io_addr, s);
  if (last_card != gui_card) {
    if (last_card)
      gtk_signal_emit_by_name (GTK_OBJECT (
        hide_card [last_card->type - CARD_FIRST]), "changed");
    if (gui_card)
      gtk_signal_emit_by_name (GTK_OBJECT (
        show_card [gui_card->type - CARD_FIRST]), "changed");
  }   
  back_card->getHeads (&gui_main_head, &gui_tv_head, &gui_video_head, 
		     &gui_max_head);
  gui_heads.main->upper = 
  gui_heads.tv->upper =
  gui_heads.video->upper = gui_max_head;
  if (opt_head != -1) {
    gui_tv_head = gui_video_head = opt_head;
    back_card->setHeads (gui_main_head, gui_tv_head, gui_video_head);
  }
  gui_set_heads ();
  if (gui_max_head > 1) {
    gtk_signal_emit_by_name (GTK_OBJECT (show_head), "changed");
  } else {
    gtk_signal_emit_by_name (GTK_OBJECT (hide_head), "changed");
  }
  if (back_card->getTwinView (&x, &y)) {
    gtk_label_set_text (gui_heads.twin, "On");
    gui_set_view (x, y);
  } else {
    gtk_label_set_text (gui_heads.twin, "Off");
  }
  head_update_cb (NULL, NULL);
#ifndef DISABLE_TIMEOUT 
  if (gui_max_head > 1) {  
    gui_service_delay = 0;
    service_timeout ();
  } else {
    if (gui_service_timeout_id != -1) 
      gtk_timeout_remove (gui_service_timeout_id);
  }
#endif
  gui_device_init (FALSE);
}

void gui_card_set (CardPtr card)
{
  CardPtr last_card;

  last_card = gui_card;
  if (gui_card) {
    back_access->closeCard ();
  }
  gui_card = card;
  back_access->openCard (gui_card);
  gui_card_changed (last_card);
}

void gui_card_cb (GtkWidget *widget, gpointer data)
{
  gui_card_set ((CardPtr) data);
}

/* -------- X Video Mode -------- */

/*
 * Switch to smallest fitting video mode.
 */

void xswitch_mode_cb (GtkWidget *widget, gpointer data)
{
  if (!gui_act_mode) return;
  switch_vidmode (gdk_display, my_gdk_screen, 
		  gui_act_mode->spec.res_x, gui_act_mode->spec.res_y);
}

/* -------- X window center & resize -------- */

void gui_xresize_cb (GtkWidget *widget, gpointer data);
void gui_xmove_cb (GtkWidget *widget, gpointer data);
void gui_xwindow_changed (void);

/*
 *  Check if window size is current, and get new size & hints if not.
 */

Bool gui_xwindow_error;

int gui_xwindow_handler (Display *dpy, XErrorEvent *error)
{
  gui_xwindow_error = TRUE;
  return 0;
}

void gui_xwindow_check (void)
{
  XErrorHandler handler;
  Window root;
  int x, y;
  unsigned width, height, borderw, depth;
  static XSizeHints *hints = NULL;
  long user;
  Window junkwin;

  RAISE (MSG_DEBUG, "xwindow_check %li", gui_window);
  if (!gui_window) return;
  if (!hints) hints = XAllocSizeHints();
  gui_xwindow_error = FALSE;
  handler = XSetErrorHandler (gui_xwindow_handler);
  XGetGeometry (gdk_display, gui_window, &root, &x, &y, &width, &height, 
		&borderw, &depth);
  XTranslateCoordinates (gdk_display, gui_window, my_gdk_root_window,
			 x-borderw, y-borderw, &x, &y, &junkwin);
  XSetErrorHandler(handler);
  if (gui_xwindow_error) {
    gui_window = None;
    gui_xwindow_changed ();
    return;
  }
  if (x     != gui_window_x     || y      != gui_window_y ||
      width != gui_window_width || height != gui_window_height)
  {
    RAISE (MSG_DEBUG, "  %i,%i %ix%i", x, y, width, height);
    gui_window_x = x;
    gui_window_y = y;
    gui_window_width = width;
    gui_window_height = height;
    XGetWMNormalHints (gdk_display, gui_window, hints, &user);

    gtk_signal_handler_block_by_func (GTK_OBJECT (gui_xwin_pos.hadj),
      GTK_SIGNAL_FUNC (gui_xmove_cb), NULL);
    gtk_signal_handler_block_by_func (GTK_OBJECT (gui_xwin_pos.vadj),
      GTK_SIGNAL_FUNC (gui_xmove_cb), NULL);
    gtk_adjustment_set_value (gui_xwin_pos.hadj, x);
    gtk_adjustment_set_value (gui_xwin_pos.vadj, y);
    gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_xwin_pos.hadj),
      GTK_SIGNAL_FUNC (gui_xmove_cb), NULL);
    gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_xwin_pos.vadj),
      GTK_SIGNAL_FUNC (gui_xmove_cb), NULL);

    gtk_signal_handler_block_by_func (GTK_OBJECT (gui_xwin_size.hadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
    gtk_signal_handler_block_by_func (GTK_OBJECT (gui_xwin_size.vadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
    if (hints->flags & PResizeInc) {
      gui_xwin_size.hadj->step_increment = hints->width_inc;
      gui_xwin_size.hadj->page_increment = 10 * hints->width_inc;
      gui_xwin_size.vadj->step_increment = hints->height_inc;
      gui_xwin_size.vadj->page_increment = 10 * hints->height_inc;
    }
    gtk_adjustment_set_value (gui_xwin_size.hadj, width);
    gtk_adjustment_set_value (gui_xwin_size.vadj, height);
    gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_xwin_size.hadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
    gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_xwin_size.vadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
  }
}

/*
 *  Move the X window.
 */

void gui_xmove_cb (GtkWidget *widget, gpointer data)
{
  gui_xwindow_check ();
  gui_window_x = (int) gui_xwin_pos.hadj->value;
  gui_window_y = (int) gui_xwin_pos.vadj->value;
  if (gui_window)
  {
    XMoveWindow (gdk_display, gui_window, gui_window_x, gui_window_y);
    XSync (gdk_display, False); /* give the WM a chance */
    gui_xwindow_check ();
  }
}

/*
 *  Resize the X window.
 */

void gui_xresize_cb (GtkWidget *widget, gpointer data)
{
  gui_xwindow_check ();
  gui_window_width = (int) gui_xwin_size.hadj->value;
  gui_window_height = (int) gui_xwin_size.vadj->value;
  if (gui_window)
  {
    XResizeWindow (gdk_display, gui_window, 
		   gui_window_width, gui_window_height);
    XSync (gdk_display, False); /* give the WM a chance */
    gui_xwindow_check ();
  }
}

/*
 *  The X Window selection has changed.
 */

void gui_xwindow_changed (void)
{
  char *name;

  gui_window_width = -1;
  gui_window_height = -1;
  if (gui_window) 
  {
    gui_xwindow_check (); /* get width and height */
    XFetchName(gdk_display, gui_window, &name);
    gtk_label_set_text (gui_mode_mask.xwin_name, name);
    gui_xwin_pos.hcenter = (int) gui_window_x;
    gui_xwin_pos.vcenter = (int) gui_window_y;
    gui_xwin_size.hcenter = (int) gui_window_width;
    gui_xwin_size.vcenter = (int) gui_window_height;
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xwin_frame), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xcenter), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xadjust), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xwin_pos), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xwin_size), TRUE);
  } else {
    gtk_label_set_text (gui_mode_mask.xwin_name, "(None)");
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xwin_frame), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xcenter), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xadjust), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xwin_pos), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xwin_size), FALSE);
  }
}

/*
 *  Select an X window, and store original size.
 */

void xselect_window_cb (GtkWidget *widget, gpointer data)
{
  gui_window = Select_Window (gdk_display, my_gdk_screen);
  if (gui_window != None) 
    gui_window = XmuClientWindow (gdk_display, gui_window);
  gui_xwindow_changed ();
}

/*
 *  Center the X window in the X virtual viewport.
 */

void xcenter_window_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "xcenter_window_cb %li", gui_window);
  gui_xwindow_check ();
  if (!gui_window) return;
  if (gui_tvstate) {
    center_window (gdk_display, my_gdk_screen, gui_window, 
      gui_act_mode->spec.res_x, gui_act_mode->spec.res_y);
  } else {
    center_window (gdk_display, my_gdk_screen, gui_window, -1, -1);
  }
}

/*
 *  Adjust the X window size to the mode size.
 */

void xadjust_window_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "xadjust_window_cb %li", gui_window);
  gui_xwindow_check ();
  if (!gui_window) return;
  if (!gui_act_mode) return;
  XResizeWindow (gdk_display, gui_window, 
		 gui_act_mode->spec.res_x, gui_act_mode->spec.res_y);
  /* FIXME: If in dualhead mode, also move window ? */
  XSync (gdk_display, False); /* give the WM a chance */
  gui_xwindow_check (); /* update size */
}

/* -------- -------- */

void offset_center_cb (GtkWidget *widget, struct mask_offset *offset)
{
  /* FIXME: freeze */
  gtk_adjustment_set_value (offset->hadj, offset->hcenter);
  gtk_adjustment_set_value (offset->vadj, offset->vcenter);
  /* FIXME: thaw, signal changed */
}

void spin_dec_cb (GtkWidget *widget, gpointer data)
{
  gtk_spin_button_spin (GTK_SPIN_BUTTON(data), GTK_SPIN_STEP_BACKWARD, 0.0);
}

void spin_inc_cb (GtkWidget *widget, gpointer data)
{
  gtk_spin_button_spin (GTK_SPIN_BUTTON(data), GTK_SPIN_STEP_FORWARD, 0.0);
}

void changed_all_cb (GtkObject *obj, gpointer data)
{
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

/* -------- -------- */

void gui_orig_update (void)
{
  TVRegs temp;

  back_card->getMode (&temp);
  gui_tvstate = (temp.devFlags & DEV_TELEVISION) ? TRUE : FALSE;
  if (!gui_tvstate) {
    gui_orig_regs = &gui_orig_mode.regs;
    *gui_orig_regs = temp;
  }
}

void gui_head_init (void)
{
  int view_x, view_y;

  RAISE (MSG_DEBUG, "gui_head_init");
  if (gui_main_head != gui_tv_head && 
      !back_card->getTwinView (&view_x, &view_y)) 
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gui_heads.shared), TRUE);
    /* this calls duplicate_cb, which does the rest. */
  }
}

void tvmonitor_on_cb (GtkButton *button, gpointer data)
{
  gui_get_settings ();
  gui_get_settings_flags ();
  gui_orig_update ();
  back_card->setModeSettings (&gui_regs, gui_bypass ? NULL : &gui_set);
  gui_tvstate = TRUE;
  gui_head_init ();
  gtk_label_set_text (gui_mode_mask.tv_state, "On");
  gtk_label_set_text (gui_mode_mask.mon_state, 
    (gui_regs.devFlags & DEV_MONITOR) ? "Dual" : "Off");
  head_update_cb (NULL, NULL);
}

void tvmonitor_off_cb (GtkButton *button, gpointer data)
{
  if (!gui_orig_regs) {
    /* We don't have valid crt values, so try the X vidmode 
       current modeline. */
    if (get_vidmode (gdk_display, my_gdk_screen, NULL, NULL, 
          &gui_orig_mode.regs.crtc, data_card_func (gui_card->type)->make)) 
    {
      gui_orig_regs= &gui_orig_mode.regs;
      gui_orig_regs->devFlags = DEV_MONITOR;
      gui_orig_regs->portHost = PORT_PCLK_MASTER | PORT_SYNC_MASTER;
      gui_orig_regs->portEnc = 0;
    }
  }
  if (gui_orig_regs) {
    back_card->setModeSettings (gui_orig_regs, NULL);
  } else {
    fprintf (stderr, "No monitor crt modeline found.");
  }
  gui_tvstate = FALSE;
  gtk_label_set_text (gui_mode_mask.tv_state, "Off");
  gtk_label_set_text (gui_mode_mask.mon_state, "On");
  head_update_cb (NULL, NULL);
}

void color_bar_cb (GtkButton *button, gpointer data)
{
  back_card->setTestImage (&gui_regs.enc, gui_bypass ? NULL : &gui_set);
  gtk_label_set_text (gui_mode_mask.tv_state, "Test");
  head_update_cb (NULL, NULL);
}

void apply_cb (GtkButton *button, gpointer data)
{
  if (gui_tvstate) {
    gui_get_settings ();
    gui_get_settings_flags ();
    back_card->setModeSettings (&gui_regs, gui_bypass ? NULL : &gui_set);
  }
}

void auto_apply_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "auto_apply_cb");
  if (gui_tvstate && gui_auto) {
    gui_get_settings ();
    gui_get_settings_flags ();
    back_card->setModeSettings (&gui_regs, gui_bypass ? NULL : &gui_set);
  }
}

void auto_cb (GtkWidget *widget, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (widget)->active) {
    gui_auto = TRUE;
  } else {
    gui_auto = FALSE;
  }
}

void gui_print_cb (GtkButton *button, gpointer data)
{
  TVRegs temp;

  temp = gui_regs;
  gui_get_settings ();
  gui_get_settings_flags ();
  if (gui_func && !gui_bypass) 
    gui_func->setup (&gui_set, &temp);
  if (gui_act_mode) {
    printf ("*** Resolution %03i x %03i  Overscan %06.3f x %06.3f",
	    gui_act_mode->spec.res_x, gui_act_mode->spec.res_y,
	    gui_act_mode->spec.hoc, gui_act_mode->spec.voc);
  }
  if ((int) data & PRINT_FP_REGS) 
    print_fp_regs (&temp.crtc, gui_card->type);
  if ((int) data & PRINT_CRT_REGS) 
    print_crt_regs (&temp.crtc, gui_card->type);
  if ((int) data & PRINT_CHIP_REGS) 
    print_tv_regs (&temp.enc, gui_tv_chip);
};			      

void gui_reset_port_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_reset_port_cb ");
  if (gui_act_mode) {
    gui_port_set (&gui_act_mode->regs);
  }
}

void gui_reset_crt_cb (GtkWidget *widget, gpointer data)
{
  if (gui_act_mode) {
    gui_crt_set (&gui_act_mode->regs.crtc);
  }
}

void reset_setting_cb (GtkWidget *widget, gpointer data)
{
  if (gui_func) {
    gui_func->defaults (&gui_set);
    gui_set_settings ();
    auto_apply_cb (NULL, NULL);
  }
}

void gui_monitor_mode_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_monitor_cb ");
  if (gui_orig_regs) {
    gui_act_mode_set (&gui_orig_mode);
  }
}

void gui_fetch_cb (GtkWidget *widget, gpointer data)
{
  RAISE (MSG_DEBUG, "gui_fetch_cb");

  back_card->getMode (&gui_fetch_mode.regs);
  /* FIXME calculate spec?? ... card/encoder dependent */
  gui_act_mode_set (&gui_fetch_mode);
}

void quit_cb (GtkWidget *widget, gpointer data)
{
  gtk_main_quit ();
}

void gui_map_cb (GtkWidget *widget, GtkAccelGroup *group)
{
  gtk_window_add_accel_group (GTK_WINDOW (gui_main_window), group);
}

void gui_unmap_cb (GtkWidget *widget, GtkAccelGroup *group)
{
  gtk_window_remove_accel_group (GTK_WINDOW (gui_main_window), group);
}

/* -------- -------- */

gfloat lower_bound (int bits) 
{
  if (bits >= 0) return 0;
  bits = -bits - 1;
  return (gfloat) - (((long long)1 << bits) - 1);
}

gfloat upper_bound (int bits) 
{
  if (bits < 0) bits = -bits - 1;
  return (gfloat) (((long long)1 << bits) - 1);
}


/* -------- GUI Pages helper routines -------- */

GtkWidget *create_adjust_frame (char *title, int min, int max, 
  gboolean hor_vert, struct mask_offset *offset, GtkSignalFunc signal,
  GtkWidget **ptable)
{
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *spin;   
  GtkWidget *box;
  GtkAdjustment *adj;

  frame = gtk_frame_new (title);

  table = gtk_table_new (3, 3, FALSE);
  if (ptable) *ptable = table;
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  if (hor_vert) {
    box = gtk_hbox_new (FALSE, 0);
  } else {
    box = gtk_vbox_new (FALSE, 0);
  }
  gtk_table_attach (GTK_TABLE(table), box, 0,3, 3,4,
    GTK_FILL, GTK_FILL,  0,0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            (double) min, (double) max, 1.0, 10.0, 0.0);
  offset->hadj = adj;
  offset->hcenter = 0.0;

  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_box_pack_start(GTK_BOX (box), spin, TRUE, TRUE, 0);

  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_LEFT,  0,1, 1,2);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_dec_cb), spin);
  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_RIGHT, 2,3, 1,2);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_inc_cb), spin);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            (double) min, (double) max, 1.0, 10.0, 0.0);
  offset->vadj = adj;
  offset->vcenter = 0.0;

  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_box_pack_start(GTK_BOX (box), spin, TRUE, TRUE, 0);

  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_UP,    1,2, 0,1);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_dec_cb), spin);
  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_DOWN,  1,2, 2,3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_inc_cb), spin);

  button = create_arrow_button (GTK_TABLE(table), (GtkArrowType) -1, 1,2, 1,2);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (offset_center_cb), offset);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  if (signal) {
    gtk_signal_connect (GTK_OBJECT (offset->hadj), "value-changed", 
      signal, NULL);
    gtk_signal_connect (GTK_OBJECT (offset->vadj), "value-changed", 
      signal, NULL);
  }

  return frame;
}

void gui_mask_checkbutton (GtkWidget *table, GtkObject *changed, 
		     GuiFlagMask *m, int x1, int x2, int y)
{
  GtkWidget *button;   

  button = gtk_check_button_new_with_label (m->label);
  gtk_table_attach (GTK_TABLE(table), button, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_object_set_data (GTK_OBJECT (button), "mask", m);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      GTK_SIGNAL_FUNC (flag_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (flag_update_cb), GTK_OBJECT (button));
}

/*
 *  Create an option menu for flags
 */

GtkWidget* gui_mask_option_menu_flag (GtkWidget *table, GtkObject *changed, 
  GuiFlagMask *m, int x1, int x2, int y, char *clear, char *set)
{
  GtkWidget *opt, *menu, *item;

  opt = gtk_option_menu_new();
  gtk_object_set_data (GTK_OBJECT (opt), "mask", m);
    menu = gtk_menu_new();
  item = gtk_menu_item_new_with_label (clear);
  gtk_object_set_data (GTK_OBJECT (item), "mask", m);
  gtk_signal_connect (GTK_OBJECT (item), "activate", 
    GTK_SIGNAL_FUNC (flag_opt_changed_cb), (gpointer) 0);
  gtk_menu_append (GTK_MENU (menu), item);
  item = gtk_menu_item_new_with_label (set);
  gtk_object_set_data (GTK_OBJECT (item), "mask", m);
  gtk_signal_connect (GTK_OBJECT (item), "activate",
    GTK_SIGNAL_FUNC (flag_opt_changed_cb), (gpointer) 1);
  gtk_signal_connect_object (changed, "changed",
    GTK_SIGNAL_FUNC (flag_opt_update_cb), GTK_OBJECT (opt));
  gtk_menu_append (GTK_MENU (menu), item);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);
  if (table) {
    gtk_table_attach (GTK_TABLE(table), opt, x1,x2, y,y+1,
      GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  }
  return opt;
}

/*
 *  Create a labelled label in a frame. 
 */

GtkWidget* create_framed_label (GtkWidget *table, char *title, char *value,
  int x1, int x2, int x3, int y)
{
  GtkWidget *frame;
  GtkWidget *label;

  if (title) {
    label = gtk_label_new (title);
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
    gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1,
      GTK_FILL, GTK_FILL, 0, 0);
  }

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_table_attach (GTK_TABLE(table), frame, x2, x3, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new (value);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 0);
  gtk_container_add (GTK_CONTAINER(frame), label);

  return label;
}

/*
 * Create label with another label in a table, for status page.
 */

GtkLabel *gui_mask_label (GtkWidget *table, char *title, char *val, 
  int x1, int x2, int x3, int y)
{
  GtkWidget *label;

  label = gtk_label_new (title);
  gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1, 
		    GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new (val);
  gtk_table_attach (GTK_TABLE(table), label, x2, x3, y, y+1, 
		    GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  return (GtkLabel *) label;
}

/*
 * Create label with spinbutton, adjustment and signals, for head selection.
 */

GtkAdjustment *gui_labelled_entry (GtkWidget *table, char *title, 
  GtkSignalFunc signal, int x1, int x2, int x3, int y)
{
  GtkWidget *spin;   
  GtkWidget *label;
  GtkAdjustment *adj;

  label = gtk_label_new (title);
  gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 1.0, 2.0, 
	  1.0, 1.0, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  my_gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin),
				 GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x2, x3, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  if (signal) {
    gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			signal, NULL);
  }
  return adj;
}

/*
 * Create scale with signals, for settings mask. Return adjustment.
 */

GtkAdjustment *gui_mask_scale (GtkWidget *table, GtkObject *changed, 
  GtkSignalFunc changed_cb, GuiSetMask *m, int x1, int x2, int y)
{
  GtkWidget *scale;
  GtkAdjustment *adj;

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
           (double) m->min, (double) m->max, 1.0, 10.0, 0.0);

  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  // gtk_adjustment_set_value (adj, 0.0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);

  gtk_table_attach (GTK_TABLE(table), scale, x1,x2, y,y+1, 
      GTK_FILL | GTK_EXPAND, GTK_FILL, 0,0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (set_changed_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      changed_cb, NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (set_update_cb), GTK_OBJECT (adj));
  return adj;
}

/*
 * Create label with spinbutton, adjustment and signals, for register mask. 
 */

void gui_mask_entry (GtkWidget *table, GtkObject *changed, 
		     GuiRegMask *m, int x1, int x2, int x3, int y)
{
  GtkWidget *spin;   
  GtkWidget *label;
  GtkAdjustment *adj;

  label = gtk_label_new (m->label);
  gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
	  lower_bound (m->bits), upper_bound (m->bits),
	  m->tick, m->tick * 10, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  my_gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin),
				 GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x2, x3, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (reg_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (reg_update_cb), GTK_OBJECT (adj));
}

/*
 * Create label with two spinbuttons, adjustments and signals. 
 * This is a workaround, since the stupid spinbuttons can only hold
 * float values, which is not enough for 32 bits.
 */

void gui_mask_twin_entry (GtkWidget *table, GtkObject *changed, 
  GuiRegMask *m, int x1, int x2, int x3, int x4, int y)
{
  GtkWidget *spin;   
  GtkWidget *label;
  GtkAdjustment *adj;

  label = gtk_label_new (m->label);
  gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 99999.0,
	  m->tick, m->tick * 100, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  my_gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin), GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x2, x3, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (reg_high_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (reg_high_update_cb), GTK_OBJECT (adj));

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 99999.0,
	  m->tick, m->tick * 100, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  my_gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin), GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x3, x4, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (reg_low_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (reg_low_update_cb), GTK_OBJECT (adj));
}

/* -------- GUI Pages -------- */

GtkWidget *gui_regs_page (char *title, GtkAccelGroup *accel_group, 
  int print_mode, GtkObject *update, GtkSignalFunc reset_cb, 
  int reg_x, int reg_y, int reg_lines, GuiRegMask *mask_reg, 
  int twin_x, int twin_y, GuiRegMask *mask_twin,
  int flag_x, int flag_y, int flag_lines, GuiFlagMask *mask_flag)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  int i;

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) accel_group);

  frame = gtk_frame_new (title);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,4,0,1);
  
  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  if (mask_reg) {
    for (i = 0; TRUE; i++) {
      GuiRegMask *m = mask_reg + i;
      int x = (i / reg_lines) * 3 + reg_x;
      int y =  i % reg_lines      + reg_y;
      
      if (!(m->label)) break;
      gui_mask_entry (table, update, m, x, x+2, x+3, y);
    }
  }
  if (mask_twin) {
    for (i = 0; TRUE; i++) {
      GuiRegMask *m = mask_twin + i;
      int x =     twin_x;
      int y = i + twin_y;
      
      if (!(m->label)) break;
      gui_mask_twin_entry (table, update, m, x, x+2, x+3, x+4, y);
    }
  }
  if (mask_flag) {
    for (i = 0; TRUE; i++) {
      GuiFlagMask *m = mask_flag + i;
      int x = (i / flag_lines) * 2 + flag_x;
      int y =  i % flag_lines      + flag_y;

      if (!(m->label)) break;
      gui_mask_checkbutton (table, update, m, x, x+2, y);
    }
  }

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  /* Buttons */

  button = gtk_button_new_with_label ("Fetch");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_fetch_cb), NULL); 
  gtk_table_attach(GTK_TABLE(page), button, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_toggle_button_new_with_label ("Bypass");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (bypass_cb), NULL); 
  gtk_signal_connect (GTK_OBJECT (update_bypass), "changed",
    GTK_SIGNAL_FUNC (bypass_update_cb), GTK_OBJECT (button));
  gtk_table_attach(GTK_TABLE(page), button, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), (gpointer) print_mode);
  gtk_table_attach(GTK_TABLE(page), button, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked", reset_cb, NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

/* -------- */

GtkWidget *gui_config_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *box;
  GtkWidget *button;
  int i;
  CardPtr card;
  GtkWidget *opt, *menu, *item;

  page = gtk_table_new (1,1,FALSE);

  /* Hardware, Card Detected */

  frame = gtk_frame_new ("Hardware");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,2, 0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

  table = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("TV Encoder:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);

  opt = create_option_menu (NULL, "<No Device>", NULL);
  gui_config_mask.chip = (GtkOptionMenu *) (opt);
  gtk_table_attach (GTK_TABLE(table), opt, 1,2, 0,1, 
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

  label = gtk_label_new ("Video Card:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2, 
    GTK_FILL, GTK_FILL, 0, 0);

  opt = gtk_option_menu_new();
  menu = gtk_menu_new();
  for (card = gui_card_list; card; card = card->next) 
  {
    item = gtk_menu_item_new_with_label (card->name);  
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC(gui_card_cb), (gpointer) card);
    gtk_menu_append (GTK_MENU (menu), item);
  }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);
  gtk_table_attach (GTK_TABLE(table), opt, 1,2, 1,2, 
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

  label = gtk_label_new ("Architecture:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new ("---");
  gui_config_mask.io_addr = GTK_LABEL (label);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 1,2, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_table_set_row_spacings (GTK_TABLE(table), 3);
  gtk_table_set_col_spacings (GTK_TABLE(table), 5);

  /* Connector */  

  frame = gtk_frame_new ("Connector");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 1,2, 1,2, 
    GTK_FILL, GTK_FILL | GTK_EXPAND, 0,0);

  box = create_radio_box (2, FALSE, GTK_SIGNAL_FUNC (gui_connector_cb), 
    &gui_settings.output, "Composite", "SVideo", "Both", "Convert", NULL);
  gtk_container_set_border_width (GTK_CONTAINER (box), 5);
  gtk_container_add (GTK_CONTAINER (frame), box);

  button = gtk_button_new_with_label ("Probe");
  gtk_misc_set_padding (GTK_MISC (GTK_BIN(button)->child), 10, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (check_monitor_cb), NULL); 
  gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);

  /* Color bars */

  frame = gtk_frame_new ("Test Image");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach(GTK_TABLE(page), frame, 1,2, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Color bars");
  gtk_container_set_border_width (GTK_CONTAINER (button), 5);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (color_bar_cb), NULL); 
  gtk_container_add (GTK_CONTAINER (frame), button);

  /* Accelerators */

  frame = gtk_frame_new ("Keyboard accelerators");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,1, 1,3,
    GTK_FILL, GTK_FILL | GTK_EXPAND, 0,0);

  table = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_table_set_row_spacings (GTK_TABLE(table), 3);
  gtk_table_set_col_spacings (GTK_TABLE(table), 20);

  for (i = 0; i < ACCEL_LAST; i++) {
    label = gtk_label_new (gui_accel[i].label);
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
    gtk_table_attach (GTK_TABLE(table), label, 0,1, i,i+1, 
      GTK_FILL, GTK_FILL, 0, 0);

    label = gtk_label_new (gui_accel[i].accel);
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
    gtk_table_attach (GTK_TABLE(table), label, 1,2, i,i+1, 
      GTK_FILL, GTK_FILL, 0, 0);
  }

  return page;
}

GtkWidget *gui_mode_page (void)
{
  GtkWidget *page;
  GtkWidget *pane;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *opt;
  GtkWidget *button;
  GtkWidget *scroller;
  GtkWidget *clist;

  page = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (page), 0);

  pane = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (pane), 5);
  gtk_table_attach(GTK_TABLE(page), pane, 0,4, 0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  /* TV System */ 

  frame = gtk_frame_new ("TV System");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 0,1, 0,1, 
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0,0);

  opt = create_option_menu (GTK_SIGNAL_FUNC (gui_system_cb), 
    "NTSC", "NTSC-J", "PAL", "PAL-60", "PAL-N", "PAL-Nc", "PAL-M", 
    "PAL-M60", "PAL-X", "SECAM", NULL);
  gui_mode_mask.system = (GtkOptionMenu *) opt;
  gtk_container_set_border_width (GTK_CONTAINER (opt), 3); 
  /* FIXME resize/send event to option menu after new border width */
  gtk_container_add (GTK_CONTAINER (frame), opt);
  
  /* Resolution */

  frame = gtk_frame_new ("Resolution");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 1,2, 0,1, 
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new ("---- x ---");
  gui_mode_mask.res = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);
  
  /* Size */ 

  frame = gtk_frame_new ("Size");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 2,3, 0,1,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new ("--------");
  gui_mode_mask.size = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);
  
  /* Aspect */ 

  frame = gtk_frame_new ("Aspect");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 3,4, 0,1,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new ("4:3");
  gui_mode_mask.aspect = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);

  /* X Window */ 

  frame = gtk_frame_new ("X Window");
  gui_mode_mask.xwin_frame = frame;
  gtk_widget_set_sensitive (GTK_WIDGET (frame), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new ("(None)");
  gui_mode_mask.xwin_name = (GtkLabel *) label;
  gtk_widget_set_usize (GTK_WIDGET (label), 1, 1); /* Hack: Clip name */
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER(frame), label);

  /* Overscan */ 

  frame = gtk_frame_new ("Overscan");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new (NULL);
  gui_mode_mask.overscan = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);

  /* Monitor State */ 

  frame = gtk_frame_new ("Monitor");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new ("Off");
  gui_mode_mask.mon_state = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);

  /* TV State  */ 

  frame = gtk_frame_new ("TV");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
  gtk_table_attach (GTK_TABLE(pane), frame, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new ("Off");
  gui_mode_mask.tv_state = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);

  /* Mode list */

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_set_border_width (GTK_CONTAINER (scroller), 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_table_attach (GTK_TABLE(pane), scroller, 0,4, 2,3,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  clist = gtk_clist_new (4);
  gui_mode_mask.modes = (GtkCList *) clist;
  gui_mode_mask.column = -1;
  gtk_clist_set_column_title (GTK_CLIST (clist), 0, "Resolution");
  gtk_clist_set_column_title (GTK_CLIST (clist), 1, "Size");
  gtk_clist_set_column_title (GTK_CLIST (clist), 2, "Overscan");
  gtk_clist_set_column_title (GTK_CLIST (clist), 3, "Aspect");
  gtk_clist_column_titles_show (GTK_CLIST (clist));
  gtk_clist_column_titles_active (GTK_CLIST (clist)); /* must be after show */
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_IN); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, 70); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 1, 50); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 2, 90); 
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
    GTK_SIGNAL_FUNC (gui_list_mode_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (clist), "click_column",
    GTK_SIGNAL_FUNC (gui_modes_column_cb), (gpointer) &gui_mode_mask);
  gtk_container_add (GTK_CONTAINER(scroller), clist);

  gtk_table_set_col_spacings (GTK_TABLE(pane), 3);
  gtk_table_set_row_spacings (GTK_TABLE(pane), 3);

  /* Buttons */

  button = gtk_button_new_with_label ("X Mode");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xswitch_mode_cb), NULL); 
  gtk_table_attach(GTK_TABLE(page), button, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group,
    gui_accel[ACCEL_SWITCH].key, gui_accel[ACCEL_SWITCH].mods,
    GTK_ACCEL_VISIBLE);

  button = gtk_button_new_with_label ("X Select");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xselect_window_cb), NULL); 
  gtk_table_attach(GTK_TABLE(page), button, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("X Center");
  gui_mode_mask.xcenter = button;
  gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xcenter_window_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group, 
    gui_accel[ACCEL_CENTER].key, gui_accel[ACCEL_CENTER].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("X Resize");
  gui_mode_mask.xadjust = button;
  gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xadjust_window_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group, 
    gui_accel[ACCEL_ADJUST].key, gui_accel[ACCEL_ADJUST].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  /* return */

  return page;
}
  
GtkWidget *gui_position_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkAccelGroup *gui_setting_accel_group;

  gui_setting_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_setting_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_setting_accel_group);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);
  gtk_table_attach_defaults(GTK_TABLE(page), table, 0,5,0,1);

  /* TV Position */

  frame = create_adjust_frame ("TV Position", -100, 100, TRUE,
    &gui_tv_offset, GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), frame, 0,1, 0,1, 
    GTK_FILL, GTK_FILL, 0,0);

  /* Monitor Position */

  frame = create_adjust_frame ("Monitor Position", -100, 100, TRUE,
    &gui_mon_offset, GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), frame, 1,2, 0,1, 
    GTK_FILL, GTK_FILL, 0,0);

  /* Window Position */

  frame = create_adjust_frame ("Move Window", -1000, 10000, TRUE,
    &gui_xwin_pos, GTK_SIGNAL_FUNC (gui_xmove_cb), NULL);
  gui_mode_mask.xwin_pos = frame;
  gtk_widget_set_sensitive (GTK_WIDGET (frame), FALSE);
  gtk_table_attach (GTK_TABLE(table), frame, 0,1, 1,2, 
    GTK_FILL, GTK_FILL, 0,0);

  /* Window Resize */

  frame = create_adjust_frame ("Resize Window", 0, 10000, TRUE, 
    &gui_xwin_size, GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
  gui_mode_mask.xwin_size = frame;
  // gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_widget_set_sensitive (GTK_WIDGET (frame), FALSE);
  gtk_table_attach (GTK_TABLE(table), frame, 1,2, 1,2, 
    GTK_FILL, GTK_FILL, 0,0);

  return page;
}
  
GtkWidget *gui_settings_page (void)
{
  int i, y;
  GtkWidget *page;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *label;
  GtkWidget *widget;
  GtkAdjustment *adj;
  GtkAccelGroup *gui_setting_accel_group;

  gui_setting_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_setting_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_setting_accel_group);

  vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), vbox, 0,4, 0,1);

  /* Options */

  frame = gtk_frame_new ("TV Options");
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 0);
  gtk_container_add (GTK_CONTAINER (frame), table);

  button = gtk_check_button_new_with_label ("Dualview");
  gui_settings.dualview = GTK_TOGGLE_BUTTON (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (settings_flags_cb), (gpointer) button); 
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 0,1, 
    GTK_FILL,GTK_FILL, 0,0);

  button = gtk_check_button_new_with_label ("Colorfix");
  gui_settings.colorfix = GTK_TOGGLE_BUTTON (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (settings_flags_cb), (gpointer) button); 
  gtk_table_attach (GTK_TABLE(table), button, 0,1, 1,2, 
    GTK_FILL,GTK_FILL, 0,0);

  button = gtk_check_button_new_with_label ("Monochrome");
  gui_settings.monochrome = GTK_TOGGLE_BUTTON (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (settings_flags_cb), (gpointer) button); 
  gtk_table_attach (GTK_TABLE(table), button, 1,2, 0,1,
    GTK_FILL,GTK_FILL, 0,0);

  button = gtk_check_button_new_with_label ("Non-interlaced");
  gui_settings.noninterlaced = GTK_TOGGLE_BUTTON (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (settings_flags_cb), (gpointer) button); 
  gtk_table_attach (GTK_TABLE(table), button, 1,2, 1,2,
    GTK_FILL,GTK_FILL, 0,0);

  button = gtk_check_button_new_with_label ("Macrovision");
  gui_settings.macrovis = GTK_TOGGLE_BUTTON (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (settings_flags_cb), (gpointer) button); 
  gtk_table_attach (GTK_TABLE(table), button, 2,3, 0,1,
    GTK_FILL,GTK_FILL, 0,0);

  button = gtk_check_button_new_with_label ("Free carrier");
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (settings_flags_cb), (gpointer) button); 
 gtk_table_attach (GTK_TABLE(table), button, 2,3, 1,2,
    GTK_FILL,GTK_FILL, 0,0);

  /* TV Settings */

  frame = gtk_frame_new ("TV Settings");
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  y = -1;
  for (i = 0; TRUE; i++) 
  {
    GuiSetMask *m = gui_set_mask + i;

    if (!m->label) break;
    if (m->label [0] != '+') {
      y++;
      label = gtk_label_new (m->label);
      gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
      gtk_table_attach (GTK_TABLE(table), label, 0,1, y,y+1, 
			GTK_FILL,GTK_FILL,0,0);
    }
    adj = gui_mask_scale (table, GTK_OBJECT (update_set),
	    GTK_SIGNAL_FUNC (settings_cb), m, m->xl, m->xr, y);
    if (m->adj) *(m->adj) = adj;
  }

  /* Hack: control size of sliders */
  gtk_table_set_col_spacing (GTK_TABLE(table), 1, 100);
  gtk_table_set_col_spacing (GTK_TABLE(table), 3, 100);

  /* Buttons */

  widget = gtk_toggle_button_new_with_label ("Bypass");
  button = gtk_button_new_with_label ("Setup");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (setup_cb), (gpointer) widget); 
  gtk_table_attach(GTK_TABLE(page), button, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = widget;
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (bypass_cb), NULL); 
  gtk_signal_connect (GTK_OBJECT (update_bypass), "changed",
    GTK_SIGNAL_FUNC (bypass_update_cb), GTK_OBJECT (button));
  gtk_table_attach(GTK_TABLE(page), button, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), 
    (gpointer)(PRINT_CRT_REGS | PRINT_CHIP_REGS));
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group, 
    gui_accel[ACCEL_PRINT].key, gui_accel[ACCEL_PRINT].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (reset_setting_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_setting_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}
  
GtkWidget *gui_head_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *label;
  GtkWidget *bbox;

  GtkAccelGroup *gui_head_accel_group;

  gui_head_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_head_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_head_accel_group);

  /* Display */

  frame = gtk_frame_new ("Display");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,3, 0,1,
    GTK_FILL, GTK_FILL, 0,0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = create_framed_label (table, "Head 1:", "Monitor",   0,1,2, 0);
  gui_heads.disp_mon[0] = GTK_FRAME(label->parent);
  label = create_framed_label (table, NULL,      "TV",        0,2,3, 0);
  gui_heads.disp_tv[0] = GTK_FRAME(label->parent);
  label = create_framed_label (table, NULL,      "Flatpanel", 0,3,4, 0);
  gui_heads.disp_fp[0] = GTK_FRAME(label->parent);

  label = create_framed_label (table, "Head 2:", "Monitor",   0,1,2, 1);
  gui_heads.disp_mon[1] = GTK_FRAME(label->parent);
  label = create_framed_label (table, NULL,      "TV",        0,2,3, 1);
  gui_heads.disp_tv[1] = GTK_FRAME(label->parent);
  label = create_framed_label (table, NULL,      "Flatpanel", 0,3,4, 1);
  gui_heads.disp_fp[1] = GTK_FRAME(label->parent);

  button = gtk_button_new_with_label (" Update ");
  gtk_table_attach (GTK_TABLE(table), button, 4,5, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (head_update_cb), NULL);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_col_spacing (GTK_TABLE(table), 3, 15);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  /* Viewport Position */

  frame = create_adjust_frame ("Viewport Position", 0, 2000, TRUE,
    &gui_view_pos, GTK_SIGNAL_FUNC (view_cb), NULL);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,2, 1,3,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0,0);

  /* Adjust */

  frame = gtk_frame_new ("Adjust");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0,0);

  bbox = gtk_vbutton_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), bbox);

  button = gtk_check_button_new_with_label ("Cursor");
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
    GTK_SIGNAL_FUNC (service_flag_cb), (gpointer) BACK_SERVICE_CURSOR);
  gtk_object_set_data (GTK_OBJECT (button), "mask", service_mask_flag+0);
  gtk_signal_connect_object (GTK_OBJECT(update_service), "changed",
     GTK_SIGNAL_FUNC (service_update_cb), GTK_OBJECT (button));
  gtk_container_add (GTK_CONTAINER(bbox), button);

  /* Adjust viewport */

  frame = gtk_frame_new ("Adjust viewport");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 2,3, 2,3,
    GTK_FILL, GTK_FILL, 0,0);

  bbox = gtk_vbutton_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), bbox);

  button = gtk_check_button_new_with_label ("by Cursor");
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
    GTK_SIGNAL_FUNC (service_flag_cb), (gpointer) BACK_SERVICE_VIEW_CURSOR);
  gtk_object_set_data (GTK_OBJECT (button), "mask", service_mask_flag+1);
  gtk_signal_connect_object (GTK_OBJECT(update_service), "changed",
     GTK_SIGNAL_FUNC (service_update_cb), GTK_OBJECT (button));
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("by Monitor");
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
    GTK_SIGNAL_FUNC (service_flag_cb), (gpointer) BACK_SERVICE_VIEW_MAIN);
  gtk_object_set_data (GTK_OBJECT (button), "mask", service_mask_flag+2);
  gtk_signal_connect_object (GTK_OBJECT(update_service), "changed",
     GTK_SIGNAL_FUNC (service_update_cb), GTK_OBJECT (button));
  gtk_container_add (GTK_CONTAINER(bbox), button);

  /* Heads */

  frame = gtk_frame_new ("Heads");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,1, 3,4,
    GTK_FILL, GTK_FILL, 0,0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  gui_heads.main =
    gui_labelled_entry (table, "Monitor:", GTK_SIGNAL_FUNC(head_cb), 0,1,2, 0);
  gui_heads.tv =
    gui_labelled_entry (table, "TV:",  GTK_SIGNAL_FUNC(head_cb), 0,1,2, 1);
  gui_heads.video =
    gui_labelled_entry (table, "Video:",  GTK_SIGNAL_FUNC(head_cb), 0,1,2, 2);

  gtk_table_set_row_spacings (GTK_TABLE(table), 5);
  gtk_table_set_col_spacings (GTK_TABLE(table), 10);

  /* Status */

  frame = gtk_frame_new ("Status");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 1,3, 3,4,
    GTK_FILL, GTK_FILL, 0,0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = create_framed_label (table, "Timer:",    "Off", 0,1,2, 0);
  gui_heads.timer = GTK_LABEL (label);

  label = create_framed_label (table, "Twinview:", "Off", 0,1,2, 1);
  gui_heads.twin = GTK_LABEL (label);

  label = gtk_label_new ("Shared view:"); /* NULL */
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  
  button = gtk_toggle_button_new_with_label (" Enable ");
  gui_heads.shared = GTK_TOGGLE_BUTTON (button);
  /* gtk_container_set_border_width (GTK_CONTAINER (button), 2); */
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (duplicate_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), button, 1,2, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  return page;
}

GtkWidget *gui_port_frame (char *title, GtkObject *update, GuiFlagMask *flag)
{
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *opt;

  frame = gtk_frame_new (title);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* Attention: Label for zero value must come first */

  label = create_aligned_label ("Clock:", 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  opt = gui_mask_option_menu_flag (table, update, flag+0, 1,2, 0,
    "Master", "Slave");						 
  opt = gui_mask_option_menu_flag (table, update, flag+1, 2,3, 0,
    "Low", "High");						 
								 
  label = create_aligned_label ("Sync:", 0.0, 0.5); 
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2,		 
    GTK_FILL, GTK_FILL, 0, 0);					 
  opt = gui_mask_option_menu_flag (table, update, flag+2, 1,2, 1,
    "Master", "Slave");						 
  opt = gui_mask_option_menu_flag (table, update, flag+3, 2,3, 1,
    "Low", "High");						 
  opt = gui_mask_option_menu_flag (table, update, flag+4, 3,4, 1,
    "Low", "High");						 
								 
  label = create_aligned_label ("Blank:", 0.0, 0.5); 
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,		 
    GTK_FILL, GTK_FILL, 0, 0);					 
  opt = gui_mask_option_menu_flag (table, update, flag+5, 1,2, 2,
    "Out", "In");						 
  opt = gui_mask_option_menu_flag (table, update, flag+6, 2,3, 2,
    "Low", "High");						 
  opt = gui_mask_option_menu_flag (table, update, flag+7, 3,4, 2,
    "Region", "DotClock");

  label = create_aligned_label ("Format:", 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 3,4,
    GTK_FILL, GTK_FILL, 0, 0);
  opt = gui_mask_option_menu_flag (table, update, flag+8, 1,2, 3,
    "RGB", "YCrCb");
  opt = gui_mask_option_menu_flag (table, update, flag+9, 2,3, 3,
    "Normal", "Alternate"); 
								 
  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  return frame;
}

GtkWidget *gui_port_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *label;
  int i;

  GtkAccelGroup *gui_port_accel_group;

  gui_port_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_port_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_port_accel_group);

  /* Display */

  frame = gtk_frame_new ("Device");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,5, 0,1,
    GTK_FILL, GTK_FILL, 0,0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  for (i = 0; TRUE; i++) {
    GuiFlagMask *m = dev_port_mask_flag + i;

    if (!(m->label)) break;
    gui_mask_checkbutton (table, GTK_OBJECT (update_port), m, i, i+1, 0);
  }

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  /* Ports */

  frame = gui_port_frame ("Host port", 
    GTK_OBJECT (update_port), host_port_mask_flag);
  gtk_table_attach (GTK_TABLE(page), frame, 0,5, 1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0,0);

  frame = gui_port_frame ("Encoder port", 
    GTK_OBJECT (update_port), enc_port_mask_flag);
  gtk_table_attach (GTK_TABLE(page), frame, 0,5, 2,3,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0,0);

  /* Empty expanding widget */

  label = gtk_label_new (NULL);
  gtk_table_attach (GTK_TABLE(page), label, 0,5, 3,4,
    GTK_FILL, GTK_FILL | GTK_EXPAND, 0,0);

  /* Buttons */

  button = gtk_button_new_with_label ("Monitor");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_monitor_mode_cb), NULL);
  gtk_table_attach(GTK_TABLE(page), button, 0,1, 4,5,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked", 
    GTK_SIGNAL_FUNC(gui_reset_port_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_port_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5, 4,5,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}


GtkWidget *gui_about_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *scroller;
  GtkWidget *label;

  page = gtk_table_new (1,1,FALSE);

  frame = gtk_frame_new ("About");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,1, 0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_set_border_width (GTK_CONTAINER (scroller), 5);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_add (GTK_CONTAINER (frame), scroller);

  label = gtk_label_new (
  "TV-OUT for NVidia Cards (Version " PACKAGE_VERSION ")\n\n"
  "Written by Dirk Thierbach\n\n"
  "Visit the Sourceforge project at\n"
  "http://sourceforge.net/projects/nv-tv-out/\n\n"
  "With many thanks to the members of the Sourceforge Rivatv project. "
  "Without their help this program would not have been possible.\n\n"
  "This program is protected by the Gnu Public License (GPL). You should "
  "have received the source code of this program together with a copy of "
  "the GPL. You may freely use, distribute, or modify the program, as long "
  "as anything derived in this way from the source code is covered by the "
  "GPL as well. See the text of the GPL for details.");
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_label_set_pattern (GTK_LABEL (label), ""); /* does not work... */
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_FILL);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroller), 
    label);

  return page;
}

void gui_main_buttons (GtkWidget *window, GtkWidget *table)
{
  GtkWidget *button;

  button = gtk_button_new_with_label ("TV on");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 0,1,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (tvmonitor_on_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group,
    gui_accel[ACCEL_TV_ON].key, gui_accel[ACCEL_TV_ON].mods,
    GTK_ACCEL_VISIBLE);

  button = gtk_button_new_with_label ("TV off");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 1,2,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (tvmonitor_off_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group,
    gui_accel[ACCEL_TV_OFF].key, gui_accel[ACCEL_TV_OFF].mods,
    GTK_ACCEL_VISIBLE);

  button = gtk_button_new_with_label ("Apply");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 2,3,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (apply_cb), NULL);

  button = gtk_toggle_button_new_with_label ("AutoApply");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 3,4,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (auto_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
		      GTK_SIGNAL_FUNC (auto_apply_cb), NULL);

  button = gtk_button_new_with_label ("Quit");
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_table_attach(GTK_TABLE(table), button, 4,5,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
}

void gui_accel_parse (void)
{
  GuiAccelIndex i;

  for (i = ACCEL_FIRST; i < ACCEL_LAST; i++) {
    gtk_accelerator_parse (gui_accel[i].accel, 
    &gui_accel[i].key, &gui_accel[i].mods);
  }
}

void gui_accel_root (GtkWidget *main_window)
{
/* FIXME TODO:
register gdk_root_window as gdk window, if not already registered
XSelectInput on root window
gdk_window_set_user_data (widget->window, main_window);
... and hope for the best...
 */
}

/* -------- Main -------- */
 
void gui_main (int argc, char *argv[], CardPtr card_list)
{
  GtkWidget *table;
  GtkWidget *notebook;
  GtkWidget *page;

  RAISE (MSG_DEBUG, "gui_main");
  gtk_init (&argc, &argv);

  gui_card_list = card_list;

  /* In the meantime, misuse adjustments for update signal chains. */

  update_crt  = create_notify ();
  update_chip = create_notify ();
  update_mode = create_notify ();
  update_set  = create_notify ();
  update_port = create_notify ();
  update_service = create_notify ();
  update_bypass  = create_notify ();
  changed_all = create_notify ();
  show_bt  = create_notify ();
  show_cx  = create_notify ();
  show_ch  = create_notify ();
  show_ph1 = create_notify ();
  show_ph2 = create_notify ();
  show_nx  = create_notify ();
  hide_bt  = create_notify ();
  hide_cx  = create_notify ();
  hide_ch  = create_notify ();
  hide_ph1 = create_notify ();
  hide_ph2 = create_notify ();
  hide_nx  = create_notify ();
  {
    int i;

    for (i = 0; i < CARD_LAST; i++) {
      show_card[i] = create_notify ();
      hide_card[i] = create_notify ();
    }
  }
  show_head = create_notify ();
  hide_head = create_notify ();

  gui_bt_init ();
  gui_cx_init ();
  gui_ch_init ();
  gui_ph_init ();
  gui_nx_init ();

  gui_main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (gui_main_window), "NVidia TV-Out");
  gtk_signal_connect (GTK_OBJECT (gui_main_window), "destroy",
		      GTK_SIGNAL_FUNC (quit_cb), NULL);

  gui_main_accel_group = gtk_accel_group_new ();
  gtk_window_add_accel_group (GTK_WINDOW (gui_main_window), 
    gui_main_accel_group);
  gui_accel_parse ();

  table = gtk_table_new (1,1,FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (gui_main_window), table);
    
  notebook = gtk_notebook_new ();
  gtk_widget_show (notebook);
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_LEFT);
  gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table), notebook, 0,5,0,1);
    
  page = gui_about_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("About", 0.0, 0.5));

  page = gui_mode_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Mode", 0.0, 0.5));

  page = gui_position_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Position", 0.0, 0.5));

  page = gui_settings_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Settings", 0.0, 0.5));

  page = gui_config_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Config", 0.0, 0.5));

  page = gui_head_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Heads", 0.0, 0.5));
  set_show_hide (page, show_head, hide_head);

  page = gui_port_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Ports", 0.0, 0.5));

  page = gui_nv_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("NV Regs", 0.0, 0.5));
  set_show_hide (page, show_card[CARD_NVIDIA-CARD_FIRST], 
                       hide_card[CARD_NVIDIA-CARD_FIRST]);
  set_show_hide (page, show_card[CARD_XBOX-CARD_FIRST], 
                       hide_card[CARD_XBOX-CARD_FIRST]);

  page = gui_nv_fp_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("FP Regs", 0.0, 0.5));
  set_show_hide (page, show_card[CARD_XBOX-CARD_FIRST], 
                       hide_card[CARD_XBOX-CARD_FIRST]);

  page = gui_tdfx_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("3dfx Regs", 0.0, 0.5));
  set_show_hide (page, show_card[CARD_TDFX-CARD_FIRST], 
                       hide_card[CARD_TDFX-CARD_FIRST]);

  page = gui_i810_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("I810 Regs", 0.0, 0.5));
  set_show_hide (page, show_card[CARD_I810-CARD_FIRST], 
                       hide_card[CARD_I810-CARD_FIRST]);

  page = gui_bt_calc_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("BT/CX Calc", 0.0, 0.5));
  set_show_hide (page, show_bt, hide_bt);
  set_show_hide (page, show_cx, hide_cx);

  page = gui_bt_reg1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("BT Regs1", 0.0, 0.5));
  set_show_hide (page, show_bt, hide_bt);

  page = gui_bt_reg2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("BT Regs2", 0.0, 0.5));
  set_show_hide (page, show_bt, hide_bt);

  page = gui_bt_reg3_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("BT Regs3", 0.0, 0.5));
  set_show_hide (page, show_bt, hide_bt);

  page = gui_bt_status_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Status", 0.0, 0.5));
  set_show_hide (page, show_bt, hide_bt);

  page = gui_cx_reg1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("CX Regs1", 0.0, 0.5));
  set_show_hide (page, show_cx, hide_cx);

  page = gui_cx_reg2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("CX Regs2", 0.0, 0.5));
  set_show_hide (page, show_cx, hide_cx);

  page = gui_cx_reg3_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("CX Regs3", 0.0, 0.5));
  set_show_hide (page, show_cx, hide_cx);

  page = gui_cx_reg4_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("CX Regs4", 0.0, 0.5));
  set_show_hide (page, show_cx, hide_cx);

  page = gui_cx_status_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Status", 0.0, 0.5));
  set_show_hide (page, show_cx, hide_cx);

  page = gui_ch_reg1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("CH Regs1", 0.0, 0.5));
  set_show_hide (page, show_ch, hide_ch);

  page = gui_ch_reg2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("CH Regs2", 0.0, 0.5));
  set_show_hide (page, show_ch, hide_ch);

  page = gui_ch_status_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Status", 0.0, 0.5));
  set_show_hide (page, show_ch, hide_ch);

  page = gui_ph_calc_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("PH Calc", 0.0, 0.5));
  set_show_hide (page, show_ph1, hide_ph1);
  set_show_hide (page, show_ph2, hide_ph2);

  page = gui_ph1_reg1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("PH Regs1", 0.0, 0.5));
  set_show_hide (page, show_ph1, hide_ph1);

  page = gui_ph1_reg2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("PH Regs2", 0.0, 0.5));
  set_show_hide (page, show_ph1, hide_ph1);

  page = gui_ph2_reg1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("PH Regs1", 0.0, 0.5));
  set_show_hide (page, show_ph2, hide_ph2);

  page = gui_ph2_reg2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("PH Regs2", 0.0, 0.5));
  set_show_hide (page, show_ph2, hide_ph2);

  page = gui_ph_status_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("Status", 0.0, 0.5));
  set_show_hide (page, show_ph1, hide_ph1);
  set_show_hide (page, show_ph2, hide_ph2);

  page = gui_nx_reg1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("NX Regs1", 0.0, 0.5));
  set_show_hide (page, show_nx, hide_nx);

  page = gui_nx_reg2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("NX Regs2", 0.0, 0.5));
  set_show_hide (page, show_nx, hide_nx);

  page = gui_nx_filt1_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("NX Filt1", 0.0, 0.5));
  set_show_hide (page, show_nx, hide_nx);

  page = gui_nx_filt2_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("NX Filt2", 0.0, 0.5));
  set_show_hide (page, show_nx, hide_nx);

  page = gui_nx_filt3_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    create_aligned_label ("NX Filt3", 0.0, 0.5));
  set_show_hide (page, show_nx, hide_nx);

  gtk_notebook_set_page (GTK_NOTEBOOK (notebook), 1);
  
  gui_main_buttons (gui_main_window, table);

  /* Realize */

  /* Show, then hide all pages, so the size calculation will
     work properly. I wish one could do that without displaying them */
  
  {
    int i;

    gtk_signal_emit_by_name (GTK_OBJECT (show_bt), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (show_cx), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (show_ch), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (show_ph1), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (show_ph2), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (show_nx), "changed");
    for (i = 0; i < CARD_LAST; i++) 
      gtk_signal_emit_by_name (GTK_OBJECT (show_card[i]), "changed");
    gtk_widget_show (gui_main_window);
    gtk_signal_emit_by_name (GTK_OBJECT (hide_bt), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (hide_cx), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (hide_ch), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (hide_ph1), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (hide_ph2), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (hide_nx), "changed");
    for (i = 0; i < CARD_LAST; i++) 
      gtk_signal_emit_by_name (GTK_OBJECT (hide_card[i]), "changed");
  }

  /* Update values */

  gui_set_saved &= ~opt_mode_mask;
  gui_set_saved |= opt_mode_flags;
  gui_set_saved &= ~(opt_mode_mask << TV_CAP_BIT);

  gui_service_flags = opt_service_flags;
  gtk_signal_emit_by_name (GTK_OBJECT (update_service), "changed");

  gui_system_init (); /* Init system only once. */
  gui_card_set (card_list);
  gui_orig_update ();
  gui_head_init ();
  if (opt_window != None) {
    gui_window = opt_window;
    gui_xwindow_changed ();
  }

  color_bar_cb (NULL, NULL);

  gtk_main ();
}

/*

dev  Monitor TV Flatpanel

Port:

clock master high
sync  master high high
blank in     high region

*/
