///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/undo/UndoManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>

#include "ShowPeriodicImagesModifier.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ShowPeriodicImagesModifier, AtomsObjectModifierBase)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "ShowImageX", _showImageX)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "ShowImageY", _showImageY)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "ShowImageZ", _showImageZ)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "NumImagesX", _numImagesX)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "NumImagesY", _numImagesY)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "NumImagesZ", _numImagesZ)
DEFINE_PROPERTY_FIELD(ShowPeriodicImagesModifier, "AdjustBoxSize", _adjustBoxSize)
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _showImageX, "Show periodic images - X")
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _showImageY, "Show periodic images - Y")
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _showImageZ, "Show periodic images - Z")
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _numImagesX, "Number of periodic images - X")
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _numImagesY, "Number of periodic images - Y")
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _numImagesZ, "Number of periodic images - Z")
SET_PROPERTY_FIELD_LABEL(ShowPeriodicImagesModifier, _adjustBoxSize, "Adjust simulation box size")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
ShowPeriodicImagesModifier::ShowPeriodicImagesModifier(bool isLoading) : AtomsObjectModifierBase(isLoading),
	_showImageX(false), _showImageY(false), _showImageZ(false), _numImagesX(3), _numImagesY(3), _numImagesZ(3), _adjustBoxSize(false)
{
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _showImageX);
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _showImageY);
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _showImageZ);
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _numImagesX);
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _numImagesY);
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _numImagesZ);
	INIT_PROPERTY_FIELD(ShowPeriodicImagesModifier, _adjustBoxSize);
}

/******************************************************************************
* Modifies the atoms object. The time interval passed
* to the function is reduced to the interval where the modified object is valid/constant.
******************************************************************************/
EvaluationStatus ShowPeriodicImagesModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	// Calculate new number of atoms.
	size_t numCopies = (_showImageX ? (int)_numImagesX : 1) * (_showImageY ? (int)_numImagesY : 1) * (_showImageZ ? (int)_numImagesZ : 1);
	if(numCopies <= 1 || input()->atomsCount() == 0) return EvaluationStatus();

	// Enlarge atoms arrays.
	size_t oldAtomsCount = input()->atomsCount();
	size_t newAtomsCount = oldAtomsCount * numCopies;
	output()->setAtomsCount(newAtomsCount);

	int nPBCx = _showImageX ? max((int)_numImagesX,1) : 1;
	int nPBCy = _showImageY ? max((int)_numImagesY,1) : 1;
	int nPBCz = _showImageZ ? max((int)_numImagesZ,1) : 1;

	// Duplicate values for each data channel.
	CloneHelper cloneHelper;
	Q_FOREACH(DataChannel* channel, output()->dataChannels()) {

		size_t destinationIndex = input()->atomsCount();

		AffineTransformation simCell = output()->simulationCell()->cellMatrix();
		for(int imageX = -(nPBCx-1)/2; imageX <= nPBCx/2; imageX++) {
			for(int imageY = -(nPBCy-1)/2; imageY <= nPBCy/2; imageY++) {
				for(int imageZ = -(nPBCz-1)/2; imageZ <= nPBCz/2; imageZ++) {
					if(imageX == 0 && imageY == 0 && imageZ == 0)
						continue;

					memcpy(channel->data() + (destinationIndex * channel->perAtomSize()),
							channel->constData(), channel->perAtomSize() * input()->atomsCount());

					if(channel->id() == DataChannel::PositionChannel) {
						// Shift atomic positions by the periodicity length.
						const Vector3 imageDelta = simCell * Vector3(imageX, imageY, imageZ);

						const Point3* pend = channel->dataPoint3() + (destinationIndex + input()->atomsCount());
						for(Point3* p = channel->dataPoint3() + destinationIndex; p != pend; ++p)
							*p += imageDelta;
					}

					destinationIndex += input()->atomsCount();
				}
			}
		}
	}

	if(_adjustBoxSize) {
		AffineTransformation cell = input()->simulationCell()->cellMatrix();
		cell.column(3) -= (FloatType)((nPBCx-1)/2) * cell.column(0);
		cell.column(3) -= (FloatType)((nPBCy-1)/2) * cell.column(1);
		cell.column(3) -= (FloatType)((nPBCz-1)/2) * cell.column(2);
		cell.column(0) *= nPBCx;
		cell.column(1) *= nPBCy;
		cell.column(2) *= nPBCz;
		output()->simulationCell()->setCellMatrix(cell);
	}

	return EvaluationStatus();
}

IMPLEMENT_PLUGIN_CLASS(ShowPeriodicImagesModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void ShowPeriodicImagesModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	QWidget* panel = createRollout(tr("Show periodic images"), rolloutParams, "atomviz.modifiers.show_periodic_images");

    // Create the rollout contents.
	QGridLayout* layout = new QGridLayout(panel);
	layout->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout->setHorizontalSpacing(0);
	layout->setVerticalSpacing(2);
#endif
	layout->setColumnStretch(1, 1);

	BooleanPropertyUI* showPeriodicImageXUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _showImageX));
	layout->addWidget(showPeriodicImageXUI->checkBox(), 0, 0);
	IntegerPropertyUI* numImagesXPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _numImagesX));
	numImagesXPUI->setMinValue(1);
	layout->addLayout(numImagesXPUI->createFieldLayout(), 0, 1);

	BooleanPropertyUI* showPeriodicImageYUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _showImageY));
	layout->addWidget(showPeriodicImageYUI->checkBox(), 1, 0);
	IntegerPropertyUI* numImagesYPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _numImagesY));
	numImagesYPUI->setMinValue(1);
	layout->addLayout(numImagesYPUI->createFieldLayout(), 1, 1);

	BooleanPropertyUI* showPeriodicImageZUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _showImageZ));
	layout->addWidget(showPeriodicImageZUI->checkBox(), 2, 0);
	IntegerPropertyUI* numImagesZPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _numImagesZ));
	numImagesZPUI->setMinValue(1);
	layout->addLayout(numImagesZPUI->createFieldLayout(), 2, 1);

	BooleanPropertyUI* adjustBoxSizeUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ShowPeriodicImagesModifier, _adjustBoxSize));
	layout->addWidget(adjustBoxSizeUI->checkBox(), 3, 0, 1, 2);
}

};	// End of namespace AtomViz
