///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/utilities/ProgressIndicator.h>

#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/stream.hpp>

#include "CFGParser.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/utils/ChemicalElements.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CFGParser, MultiFileParser)

/******************************************************************************
* Opens the settings dialog for this parser.
******************************************************************************/
bool CFGParser::showSettingsDialog(QWidget* parent)
{
	return true;
}

/******************************************************************************
* Parses the header of the given file and returns the number of data columns contained in the file.
******************************************************************************/
bool CFGParser::inspectFileHeader(const QString& filepath, int& numberOfColumns, QStringList& columnNames)
{
	numberOfColumns = 8;
	columnNames << tr("Mass");
	columnNames << tr("Chem. Element");
	columnNames << tr("Position X");
	columnNames << tr("Position Y");
	columnNames << tr("Position Z");
	columnNames << tr("Velocity X");
	columnNames << tr("Velocity Y");
	columnNames << tr("Velocity Z");
	return true;
}

/******************************************************************************
* Parses the atomic data of a single time step.
******************************************************************************/
bool CFGParser::loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, qint64 byteOffset, int lineNumber, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);
	using namespace boost::iostreams;

	// Show the progress indicator.
	ProgressIndicator progress(tr("Opening CFG file '%1'").arg(filename), 0, suppressDialogs);

	// Open input file.
	stream<file_source> input_stream(filename.toStdString());
	if(!input_stream.is_open())
		throw Exception(tr("Failed to open CFG file %1.").arg(filename));

	ColumnChannelMapping standardCFGMapping;
	standardCFGMapping.defineStandardColumn(0, DataChannel::MassChannel);
	standardCFGMapping.defineStandardColumn(1, DataChannel::AtomTypeChannel);
	standardCFGMapping.defineStandardColumn(2, DataChannel::PositionChannel, 0);
	standardCFGMapping.defineStandardColumn(3, DataChannel::PositionChannel, 1);
	standardCFGMapping.defineStandardColumn(4, DataChannel::PositionChannel, 2);
	standardCFGMapping.defineStandardColumn(5, DataChannel::VelocityChannel, 0);
	standardCFGMapping.defineStandardColumn(6, DataChannel::VelocityChannel, 1);
	standardCFGMapping.defineStandardColumn(7, DataChannel::VelocityChannel, 2);

	// Prepare the mapping between input file columns and data channels.
	DataRecordParserHelper recordParser(&standardCFGMapping, destination);

	unsigned int numAtoms = 0;
	unsigned int nAtomsRead = 0;
	lineNumber--;

	Matrix3 H0(NULL_MATRIX);
	Matrix3 H(IDENTITY);
	SymmetricTensor2 strain(0);
	char line[1024];

	do {

		// Parse line
		input_stream.getline(line, sizeof(line));
		lineNumber++;

		// Ignore comments
		char* commentChar = strchr(line, '#');
		if(commentChar) *commentChar = '\0';

		// Skip empty lines.
		char* trimmedLine = line + strspn(line," \t\n\r");
		if(!trimmedLine[0]) continue;

		char* splitChar = strchr(trimmedLine, '=');
		if(splitChar) {
			*splitChar = '\0';
			splitChar++;
			char* tagname = trimmedLine + strspn(trimmedLine," \t\n\r");
			char* rightside = splitChar + strspn(splitChar," \t\n\r");

			/*
			QString value;
			QString units;
			if(splitPos < 0)
				value = rightside;
			else {
				value = rightside.left(splitPos).trimmed();
				units = rightside.mid(splitPos+1).trimmed();
			}

			if(tagname == "Number of particles") {
				numAtoms = locale.toUInt(value, &ok);
				if(!ok || numAtoms <= 0 || numAtoms > 100000000)
					throw Exception(tr("Invalid number of atoms in line %1 of CFG file: %2").arg(lineNumber).arg(line));
				progress.reset(new ProgressIndicator(tr("Loading CFG file..."), numAtoms, suppressDialogs));
				// Resize atom array.
				destination->setAtomsCount(numAtoms);
			}
			else if(tagname == "H0(1,1)") H0(0,0) = locale.toDouble(value);
			else if(tagname == "H0(1,2)") H0(1,0) = locale.toDouble(value);
			else if(tagname == "H0(1,3)") H0(2,0) = locale.toDouble(value);
			else if(tagname == "H0(2,1)") H0(0,1) = locale.toDouble(value);
			else if(tagname == "H0(2,2)") H0(1,1) = locale.toDouble(value);
			else if(tagname == "H0(2,3)") H0(2,1) = locale.toDouble(value);
			else if(tagname == "H0(3,1)") H0(0,2) = locale.toDouble(value);
			else if(tagname == "H0(3,2)") H0(1,2) = locale.toDouble(value);
			else if(tagname == "H0(3,3)") H0(2,2) = locale.toDouble(value);
			else if(tagname == "Transform(1,1)") H(0,0) = locale.toDouble(value);
			else if(tagname == "Transform(1,2)") H(1,0) = locale.toDouble(value);
			else if(tagname == "Transform(1,3)") H(2,0) = locale.toDouble(value);
			else if(tagname == "Transform(2,1)") H(0,1) = locale.toDouble(value);
			else if(tagname == "Transform(2,2)") H(1,1) = locale.toDouble(value);
			else if(tagname == "Transform(2,3)") H(2,1) = locale.toDouble(value);
			else if(tagname == "Transform(3,1)") H(0,2) = locale.toDouble(value);
			else if(tagname == "Transform(3,2)") H(1,2) = locale.toDouble(value);
			else if(tagname == "Transform(3,3)") H(2,2) = locale.toDouble(value);
			else if(tagname == "eta(1,1)") strain(0,0) = locale.toDouble(value);
			else if(tagname == "eta(1,2)") strain(0,1) = locale.toDouble(value);
			else if(tagname == "eta(1,3)") strain(0,2) = locale.toDouble(value);
			else if(tagname == "eta(2,2)") strain(1,1) = locale.toDouble(value);
			else if(tagname == "eta(2,3)") strain(1,2) = locale.toDouble(value);
			else if(tagname == "eta(3,3)") strain(2,2) = locale.toDouble(value);
			else if(tagname == "A") {}
			else if(tagname == "R") {}
			else throw Exception(tr("Invalid tagname '%1' in line %2 of CFG file.").arg(tagname).arg(lineNumber));
			*/
		}
		else {
			// Read atomic positions.

			// Update progress indicator.
			if((nAtomsRead % 2000) == 0) {
				progress.setValue(nAtomsRead);
				if(progress.isCanceled()) return false;
			}

			try {
				/*
				QStringList tokens = QString(line).split(QRegExp("\\s+"), QString::SkipEmptyParts);
				recordParser.storeAtom(nAtomsRead, tokens);
				*/
				nAtomsRead++;
			}
			catch(const Exception& ex) {
				throw SecondaryException(ex, tr("Parsing error in line %1 of CFG file.").arg(lineNumber));
			}
		}

	} while(!input_stream.eof());

	Matrix3 Hfinal = H * H0;
	destination->simulationCell()->setCellMatrix(Hfinal);

	// The CFG file contains the atomic positions in reduced coordinates.
	// Now we have to rescale them to world coordinates.
	DataChannel* posChannel = destination->getStandardDataChannel(DataChannel::PositionChannel);
	if(posChannel != NULL) {
		Point3* p = posChannel->dataPoint3();
		Point3* pend = p + destination->atomsCount();
		for(; p != pend; ++p)
			*p = Hfinal * (*p);
	}

	destination->invalidate();
	return true;
}

};
