///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ApplyModifierAction.h>
#include <core/actions/ActionProxy.h>
#include <core/plugins/Plugin.h>
#include <core/data/DataSetManager.h>
#include <core/scene/objects/Modifier.h>
#include <core/scene/ObjectNode.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/panels/CommandPanel.h>
#include <core/gui/panels/modify/ModifyCommandPage.h>
#include <core/gui/panels/modify/ModifierStack.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_PLUGIN_CLASS(ApplyModifierAction, Action)

/******************************************************************************
* Initializes the action object.
******************************************************************************/
ApplyModifierAction::ApplyModifierAction(PluginClassDescriptor* modifierClass)
	: // Derive a unique action ID from the modifier class name.
		Action(QString("Modifier.Apply.%1.%2").arg(modifierClass->plugin()->pluginId(), modifierClass->name())),
		_modifierClass(modifierClass), _needsUpdate(true)
{
	OVITO_ASSERT_MSG(modifierClass->isKindOf(PLUGINCLASSINFO(Modifier)), "ApplyModifierAction constructor", "The plugin class passed to the action constructor must be derived from Modifier.");

	// Retrieve the list of input object classes to which the modifier may be applied.
	QDomElement inputClassElement = _modifierClass->getMetaData("Input-Object-Type");
	while(inputClassElement.isElement()) {
		_inputObjectClasses.push_back(_modifierClass->plugin()->getRequiredClass(inputClassElement));
		// Continue with the next <Input-Object-Type> element.
		inputClassElement = inputClassElement.nextSiblingElement("Input-Object-Type");
	}

	connect(&DATASET_MANAGER, SIGNAL(selectionChangeComplete(SelectionSet*)), this, SLOT(onSelectionChanged(SelectionSet*)));
	connect(&selectionSetListener, SIGNAL(notificationMessage(RefTargetMessage*)), this, SLOT(onSelectionSetMessage(RefTargetMessage*)));
	connect(this, SIGNAL(updateActionStateSignal()), this, SLOT(updateActionState()), Qt::QueuedConnection);
}

/******************************************************************************
* This is called by the system after the action object has been bound to its ActionProxy.
******************************************************************************/
void ApplyModifierAction::initializeAction(ActionProxy* proxy)
{
	Action::initializeAction(proxy);

	proxy->setText(_modifierClass->schematicTitle());
	proxy->setStatusTip(tr("Apply '%1' modifier").arg(_modifierClass->schematicTitle()));

	// Extract the icon from the class descriptor metadata.
	QDomElement iconElement = _modifierClass->getMetaData("Icon");
	if(iconElement.isElement()) {
		QString iconPath = iconElement.attribute("Path");
		OVITO_ASSERT_MSG(!iconPath.isEmpty(), "ApplyModifierAction::initializeAction()", "<Icon> element in modifier class descriptor must have the Path attribute.");
		// The creation of QIcon instances is only allowed in GUI mode.
		if(APPLICATION_MANAGER.guiMode())
			proxy->setIcon(QIcon(iconPath));
	}

	connect(proxy, SIGNAL(triggered(bool)), this, SLOT(onActionTriggered(bool)));
}

/******************************************************************************
* This is called after the selection set has changed.
******************************************************************************/
void ApplyModifierAction::onSelectionChanged(SelectionSet* newSelection)
{
	selectionSetListener.setTarget(newSelection);
	_needsUpdate = true;
	updateActionState();
}

/******************************************************************************
* This is called by the RefTargetListener that listens to notification messages sent by the
* current selection set.
******************************************************************************/
void ApplyModifierAction::onSelectionSetMessage(RefTargetMessage* msg)
{
	// Update the action if one of the nodes in the current selection has changed.
	if(msg->type() == NODE_IN_SELECTION_SET_CHANGED) {
		if(!_needsUpdate) {
			_needsUpdate = true;
			Q_EMIT updateActionStateSignal();
		}
	}
}

/******************************************************************************
* Updates the enabled/disable state of the modifier action based on the
* currently selected objects.
******************************************************************************/
void ApplyModifierAction::updateActionState()
{
	if(!_needsUpdate) return;
	if(proxy() == NULL) return;
	_needsUpdate = false;

	if(DATASET_MANAGER.currentSelection()->empty()) {
		proxy()->setEnabled(false);
	}
	else {
		if(_inputObjectClasses.isEmpty()) {
			proxy()->setEnabled(true);
		}
		else {
			// Find out whether all selected objects can be converted to at least
			// one of the object types accepted by the modifier.
			bool canConvert = true;
			Q_FOREACH(SceneNode* node, DATASET_MANAGER.currentSelection()->nodes()) {
				ObjectNode* objNode = static_object_cast<ObjectNode>(node);
				if(objNode) {
					const PipelineFlowState& state = objNode->evalPipeline(ANIM_MANAGER.time());
					if(state.result()) {
						bool foundMatch = false;
						Q_FOREACH(PluginClassDescriptor* clazz, _inputObjectClasses) {
							if(state.result()->canConvertTo(clazz)) {
								foundMatch = true;
								break;
							}
						}
						if(!foundMatch) {
							canConvert = false;
							break;
						}
					}
				}
			}
			proxy()->setEnabled(canConvert);
		}
	}
}

/******************************************************************************
* Is called when the user has triggered the action's state.
******************************************************************************/
void ApplyModifierAction::onActionTriggered(bool checked)
{
	// Everything needs to be recorded for undo.
	UNDO_MANAGER.beginCompoundOperation(tr("Apply Modifier"));
	try {
		// Create an instance of the modifier.
		Modifier::SmartPtr modifier = static_object_cast<Modifier>(_modifierClass->createInstance());
		CHECK_OBJECT_POINTER(modifier);

		// If the modification panel is not visible then the modifier is always
		// inserted on top of the modifier stack.
		if(APPLICATION_MANAGER.consoleMode() || MAIN_FRAME->commandPanel()->currentPage() != CommandPanel::MODIFY_PAGE) {
			// Apply modifier to each node separately.
			Q_FOREACH(SceneNode* node, DATASET_MANAGER.currentSelection()->nodes()) {
				ObjectNode* objNode = static_object_cast<ObjectNode>(node);
				if(objNode != NULL)
					objNode->applyModifier(modifier);
			}
		}
		else {
			// If the modification panel is active then we insert the modifer above the selected entry
			// in the modifier stack.
			ModifyCommandPage* modifyPage = MAIN_FRAME->commandPanel()->modifyPage();
			modifyPage->modifierStack()->applyModifier(modifier.get());
			modifyPage->modifierStack()->invalidate();
		}
	}
	catch(const Exception& ex) {
		ex.showError();
		UNDO_MANAGER.currentCompoundOperation()->clear();
	}
	UNDO_MANAGER.endCompoundOperation();
}

};
