///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ModifiedObject.h
 * \brief Contains the definition of the Core::ModifiedObject class.
 */

#ifndef __OVITO_MODIFIED_OBJECT_H
#define __OVITO_MODIFIED_OBJECT_H

#include <core/Core.h>
#include "SceneObject.h"
#include "ModifierApplication.h"

namespace Core {

/**
 * \brief This special scene object type takes an input object
 *        and applies one or more modifiers to it.
 *
 * This class takes part in the geometry pipeline. It allows the application of one or more
 * modifiers to an input object.
 *
 * \author Alexander Stukowski
 * \sa Modifier
 */
class CORE_DLLEXPORT ModifiedObject : public SceneObject
{
public:

	/// \brief Default constructor that creates an empty object without input.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	ModifiedObject(bool isLoading = false);

	/// \brief Asks the object for the result of the geometry pipeline at the given time
	///        up to a given point in the modifier stack.
	/// \param time The animation at which the geometry pipeline should be evaluated.
	/// \param upToHere If \a upToHere is \c NULL then the complete modifier stack will be evaluated.
	///                 Otherwise only the modifiers in the pipeline before the given point will be applied to the
	///                 input object. \a upToHere must be one of the application objects returned by modifierApplications().
	/// \param including Specifies whether the last modifier given by \a upToHere will also be applied to the input object.
	/// \return The result object.
	PipelineFlowState evalObject(TimeTicks time, ModifierApplication* upToHere, bool including);

	/// \brief Returns the input object of this modified object.
	/// \sa setInputObject()
	SceneObject* inputObject() const { return _inputObject; }

	/// \brief Sets the input object for the geometry pipeline.
	/// \param inputObject The new input object to which the modifiers will be applied.
	/// \undoable
	/// \sa inputObject()
	void setInputObject(SceneObject* inputObject) { _inputObject = inputObject; }

	/// \brief Sets the input object for the modifiers.
	/// \param inputObject The new input object to which the modifiers will be applied.
	///
	/// This is the same as the above method but takes a smart pointer instead of a raw pointer.
	/// \undoable
	void setInputObject(const SceneObject::SmartPtr& inputObject) { setInputObject(inputObject.get()); }

	/// \brief Returns the list of modifier applications.
	/// \return The list of applications of modifiers that make up the geometry pipeline.
	///         The modifiers in this list are applied to the input object in asscending order.
	const QVector<ModifierApplication*>& modifierApplications() const { return apps; }

	/// \brief Inserts a modifier into the geometry pipeline.
	/// \param modifier The modifier to be inserted.
	/// \param atIndex Specifies the position in the geometry pipeline where the modifier should be applied.
	///                It must be between zero and the number of existing modifier applications as returned by
	///                modifierApplications(). Modifiers are applied in asscending order, i.e. the modifier
	///                at index 0 is applied first.
	/// \return The application object that has been created for the usage of the modifier instance in this
	///         geometry pipeline.
	/// \undoable
	/// \sa modifierApplications()
	/// \sa removeModifier()
	ModifierApplication* insertModifier(Modifier* modifier, int atIndex);

	/// \brief Inserts a modifier application into the internal list.
	/// \param modApp The modifier application to be inserted.
	/// \param atIndex Specifies the position in the geometry pipeline where the modifier should be applied.
	///                It must be between zero and the number of existing modifier applications as returned by
	///                modifierApplications(). Modifiers are applied in asscending order, i.e. the modifier
	///                at index 0 is applied first.
	/// \undoable
	/// \sa insertModifier()
	/// \sa modifierApplications()
	/// \sa removeModifier()
	void insertModifierApplication(ModifierApplication* modApp, int atIndex);

	/// \brief Removes the given modifier application from the geometry pipeline.
	/// \param app The application of a modifier instance that should be removed. This must be one from the
	///            list returned by modifierApplications().
	/// \undoable
	/// \sa insertModifier()
	void removeModifier(ModifierApplication* app);

	/////////////////////////////////////// from SceneObject /////////////////////////////////////////

	/// Asks the object for its validity interval at the given time.
	virtual TimeInterval objectValidity(TimeTicks time);

	/// Render the object into the viewport.
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// Returns the bounding box of the object in local object coordinates.
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode);

	/// Asks the object for the result of the geometry pipeline at the given time.
	virtual PipelineFlowState evalObject(TimeTicks time) { return evalObject(time, NULL, true); }

	/// Returns the number of input objects that are referenced by this scene object.
	virtual int inputObjectCount() { return 1; }

	/// Returns the input object of this scene object.
	virtual SceneObject* inputObject(int index) { return _inputObject; }

protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Is called when the value of a reference field of this RefMaker changes.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget);

	/// Is called when a reference target has been added to a list reference field of this RefMaker.
	virtual void onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex);

	/// Is called when a reference target has been removed from a list reference field of this RefMaker.
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex);

	/// Notifies all modifiers from the given index on that their input has changed.
	void notifyModifiersInputChanged(int changedIndex);

	/// This method invalidates the internal geometry pipeline cache of the ModifiedObject.
	void invalidatePipelineCache() { pipelineCache = PipelineFlowState(); pipelineCacheIndex = -1; }

private:

	/// The input object that is modified by the modifiers.
	ReferenceField<SceneObject> _inputObject;

	/// The ordered list of modifiers that are applied to the input object.
	/// The modifiers are applied to the input object in the reverse order of this list.
	VectorReferenceField<ModifierApplication> apps;

	/// The cached result from the geometry pipeline evaluation.
	PipelineFlowState pipelineCache;

	/// The modifier stack position that is represented by the internal
	/// pipeline cache. If the pipeline cache is not valid then this is -1.
	int pipelineCacheIndex;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ModifiedObject)
	DECLARE_REFERENCE_FIELD(_inputObject)
	DECLARE_VECTOR_REFERENCE_FIELD(apps)
};


};

#endif // __OVITO_MODIFIED_OBJECT_H
