/*
 *                    Copyright (C) 2015, UChicago Argonne, LLC
 *                               All Rights Reserved
 *
 *                               Generic IO (ANL-15-066)
 *                     Hal Finkel, Argonne National Laboratory
 *
 *                              OPEN SOURCE LICENSE
 *
 * Under the terms of Contract No. DE-AC02-06CH11357 with UChicago Argonne,
 * LLC, the U.S. Government retains certain rights in this software.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the names of UChicago Argonne, LLC or the Department of Energy
 *      nor the names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 * *****************************************************************************
 *
 *                                  DISCLAIMER
 * THE SOFTWARE IS SUPPLIED “AS IS” WITHOUT WARRANTY OF ANY KIND.  NEITHER THE
 * UNTED STATES GOVERNMENT, NOR THE UNITED STATES DEPARTMENT OF ENERGY, NOR
 * UCHICAGO ARGONNE, LLC, NOR ANY OF THEIR EMPLOYEES, MAKES ANY WARRANTY,
 * EXPRESS OR IMPLIED, OR ASSUMES ANY LEGAL LIABILITY OR RESPONSIBILITY FOR THE
 * ACCURACY, COMPLETENESS, OR USEFULNESS OF ANY INFORMATION, DATA, APPARATUS,
 * PRODUCT, OR PROCESS DISCLOSED, OR REPRESENTS THAT ITS USE WOULD NOT INFRINGE
 * PRIVATELY OWNED RIGHTS.
 *
 * *****************************************************************************
 */

#ifndef CRC64_H
#define CRC64_H

#ifndef __STDC_CONSTANT_MACROS
#define __STDC_CONSTANT_MACROS
#endif

#include <cstdlib>
#include <stdint.h>

#ifdef _OPENMP
#include <omp.h>
#endif

namespace lanl
{

// These functions compute the CRC-64 checksum on a block of data
// and provide a way to combine the checksums on two blocks of data.
// For more information, see:
// http://en.wikipedia.org/wiki/Computation_of_CRC
// http://checksumcrc.blogspot.com/2011/12/should-you-use-crc-or-checksum.html
// http://crcutil.googlecode.com/files/crc-doc.1.0.pdf
// http://www.ross.net/crc/download/crc_v3.txt
// This uses the ECMA-182 polynomial with -1 initialization, and computes
// the bit-reversed CRC.

// The polynomial here is the bit-reversed encoding of 0x42f0e1eba9ea3693.
static const uint64_t crc64_poly = UINT64_C(0xc96c5795d7870f42);

static const uint64_t crc64_table[4][256] = {
  { UINT64_C(0x0000000000000000), UINT64_C(0x1dee8a5e222ca1dc), UINT64_C(0x3bdd14bc445943b8),
    UINT64_C(0x26339ee26675e264), UINT64_C(0x77ba297888b28770), UINT64_C(0x6a54a326aa9e26ac),
    UINT64_C(0x4c673dc4ccebc4c8), UINT64_C(0x5189b79aeec76514), UINT64_C(0xef7452f111650ee0),
    UINT64_C(0xf29ad8af3349af3c), UINT64_C(0xd4a9464d553c4d58), UINT64_C(0xc947cc137710ec84),
    UINT64_C(0x98ce7b8999d78990), UINT64_C(0x8520f1d7bbfb284c), UINT64_C(0xa3136f35dd8eca28),
    UINT64_C(0xbefde56bffa26bf4), UINT64_C(0x4c300ac98dc40345), UINT64_C(0x51de8097afe8a299),
    UINT64_C(0x77ed1e75c99d40fd), UINT64_C(0x6a03942bebb1e121), UINT64_C(0x3b8a23b105768435),
    UINT64_C(0x2664a9ef275a25e9), UINT64_C(0x0057370d412fc78d), UINT64_C(0x1db9bd5363036651),
    UINT64_C(0xa34458389ca10da5), UINT64_C(0xbeaad266be8dac79), UINT64_C(0x98994c84d8f84e1d),
    UINT64_C(0x8577c6dafad4efc1), UINT64_C(0xd4fe714014138ad5), UINT64_C(0xc910fb1e363f2b09),
    UINT64_C(0xef2365fc504ac96d), UINT64_C(0xf2cdefa2726668b1), UINT64_C(0x986015931b88068a),
    UINT64_C(0x858e9fcd39a4a756), UINT64_C(0xa3bd012f5fd14532), UINT64_C(0xbe538b717dfde4ee),
    UINT64_C(0xefda3ceb933a81fa), UINT64_C(0xf234b6b5b1162026), UINT64_C(0xd4072857d763c242),
    UINT64_C(0xc9e9a209f54f639e), UINT64_C(0x771447620aed086a), UINT64_C(0x6afacd3c28c1a9b6),
    UINT64_C(0x4cc953de4eb44bd2), UINT64_C(0x5127d9806c98ea0e), UINT64_C(0x00ae6e1a825f8f1a),
    UINT64_C(0x1d40e444a0732ec6), UINT64_C(0x3b737aa6c606cca2), UINT64_C(0x269df0f8e42a6d7e),
    UINT64_C(0xd4501f5a964c05cf), UINT64_C(0xc9be9504b460a413), UINT64_C(0xef8d0be6d2154677),
    UINT64_C(0xf26381b8f039e7ab), UINT64_C(0xa3ea36221efe82bf), UINT64_C(0xbe04bc7c3cd22363),
    UINT64_C(0x9837229e5aa7c107), UINT64_C(0x85d9a8c0788b60db), UINT64_C(0x3b244dab87290b2f),
    UINT64_C(0x26cac7f5a505aaf3), UINT64_C(0x00f95917c3704897), UINT64_C(0x1d17d349e15ce94b),
    UINT64_C(0x4c9e64d30f9b8c5f), UINT64_C(0x5170ee8d2db72d83), UINT64_C(0x7743706f4bc2cfe7),
    UINT64_C(0x6aadfa3169ee6e3b), UINT64_C(0xa218840d981e1391), UINT64_C(0xbff60e53ba32b24d),
    UINT64_C(0x99c590b1dc475029), UINT64_C(0x842b1aeffe6bf1f5), UINT64_C(0xd5a2ad7510ac94e1),
    UINT64_C(0xc84c272b3280353d), UINT64_C(0xee7fb9c954f5d759), UINT64_C(0xf391339776d97685),
    UINT64_C(0x4d6cd6fc897b1d71), UINT64_C(0x50825ca2ab57bcad), UINT64_C(0x76b1c240cd225ec9),
    UINT64_C(0x6b5f481eef0eff15), UINT64_C(0x3ad6ff8401c99a01), UINT64_C(0x273875da23e53bdd),
    UINT64_C(0x010beb384590d9b9), UINT64_C(0x1ce5616667bc7865), UINT64_C(0xee288ec415da10d4),
    UINT64_C(0xf3c6049a37f6b108), UINT64_C(0xd5f59a785183536c), UINT64_C(0xc81b102673aff2b0),
    UINT64_C(0x9992a7bc9d6897a4), UINT64_C(0x847c2de2bf443678), UINT64_C(0xa24fb300d931d41c),
    UINT64_C(0xbfa1395efb1d75c0), UINT64_C(0x015cdc3504bf1e34), UINT64_C(0x1cb2566b2693bfe8),
    UINT64_C(0x3a81c88940e65d8c), UINT64_C(0x276f42d762cafc50), UINT64_C(0x76e6f54d8c0d9944),
    UINT64_C(0x6b087f13ae213898), UINT64_C(0x4d3be1f1c854dafc), UINT64_C(0x50d56bafea787b20),
    UINT64_C(0x3a78919e8396151b), UINT64_C(0x27961bc0a1bab4c7), UINT64_C(0x01a58522c7cf56a3),
    UINT64_C(0x1c4b0f7ce5e3f77f), UINT64_C(0x4dc2b8e60b24926b), UINT64_C(0x502c32b8290833b7),
    UINT64_C(0x761fac5a4f7dd1d3), UINT64_C(0x6bf126046d51700f), UINT64_C(0xd50cc36f92f31bfb),
    UINT64_C(0xc8e24931b0dfba27), UINT64_C(0xeed1d7d3d6aa5843), UINT64_C(0xf33f5d8df486f99f),
    UINT64_C(0xa2b6ea171a419c8b), UINT64_C(0xbf586049386d3d57), UINT64_C(0x996bfeab5e18df33),
    UINT64_C(0x848574f57c347eef), UINT64_C(0x76489b570e52165e), UINT64_C(0x6ba611092c7eb782),
    UINT64_C(0x4d958feb4a0b55e6), UINT64_C(0x507b05b56827f43a), UINT64_C(0x01f2b22f86e0912e),
    UINT64_C(0x1c1c3871a4cc30f2), UINT64_C(0x3a2fa693c2b9d296), UINT64_C(0x27c12ccde095734a),
    UINT64_C(0x993cc9a61f3718be), UINT64_C(0x84d243f83d1bb962), UINT64_C(0xa2e1dd1a5b6e5b06),
    UINT64_C(0xbf0f57447942fada), UINT64_C(0xee86e0de97859fce), UINT64_C(0xf3686a80b5a93e12),
    UINT64_C(0xd55bf462d3dcdc76), UINT64_C(0xc8b57e3cf1f07daa), UINT64_C(0xd6e9a7309f3239a7),
    UINT64_C(0xcb072d6ebd1e987b), UINT64_C(0xed34b38cdb6b7a1f), UINT64_C(0xf0da39d2f947dbc3),
    UINT64_C(0xa1538e481780bed7), UINT64_C(0xbcbd041635ac1f0b), UINT64_C(0x9a8e9af453d9fd6f),
    UINT64_C(0x876010aa71f55cb3), UINT64_C(0x399df5c18e573747), UINT64_C(0x24737f9fac7b969b),
    UINT64_C(0x0240e17dca0e74ff), UINT64_C(0x1fae6b23e822d523), UINT64_C(0x4e27dcb906e5b037),
    UINT64_C(0x53c956e724c911eb), UINT64_C(0x75fac80542bcf38f), UINT64_C(0x6814425b60905253),
    UINT64_C(0x9ad9adf912f63ae2), UINT64_C(0x873727a730da9b3e), UINT64_C(0xa104b94556af795a),
    UINT64_C(0xbcea331b7483d886), UINT64_C(0xed6384819a44bd92), UINT64_C(0xf08d0edfb8681c4e),
    UINT64_C(0xd6be903dde1dfe2a), UINT64_C(0xcb501a63fc315ff6), UINT64_C(0x75adff0803933402),
    UINT64_C(0x6843755621bf95de), UINT64_C(0x4e70ebb447ca77ba), UINT64_C(0x539e61ea65e6d666),
    UINT64_C(0x0217d6708b21b372), UINT64_C(0x1ff95c2ea90d12ae), UINT64_C(0x39cac2cccf78f0ca),
    UINT64_C(0x24244892ed545116), UINT64_C(0x4e89b2a384ba3f2d), UINT64_C(0x536738fda6969ef1),
    UINT64_C(0x7554a61fc0e37c95), UINT64_C(0x68ba2c41e2cfdd49), UINT64_C(0x39339bdb0c08b85d),
    UINT64_C(0x24dd11852e241981), UINT64_C(0x02ee8f674851fbe5), UINT64_C(0x1f0005396a7d5a39),
    UINT64_C(0xa1fde05295df31cd), UINT64_C(0xbc136a0cb7f39011), UINT64_C(0x9a20f4eed1867275),
    UINT64_C(0x87ce7eb0f3aad3a9), UINT64_C(0xd647c92a1d6db6bd), UINT64_C(0xcba943743f411761),
    UINT64_C(0xed9add965934f505), UINT64_C(0xf07457c87b1854d9), UINT64_C(0x02b9b86a097e3c68),
    UINT64_C(0x1f5732342b529db4), UINT64_C(0x3964acd64d277fd0), UINT64_C(0x248a26886f0bde0c),
    UINT64_C(0x7503911281ccbb18), UINT64_C(0x68ed1b4ca3e01ac4), UINT64_C(0x4ede85aec595f8a0),
    UINT64_C(0x53300ff0e7b9597c), UINT64_C(0xedcdea9b181b3288), UINT64_C(0xf02360c53a379354),
    UINT64_C(0xd610fe275c427130), UINT64_C(0xcbfe74797e6ed0ec), UINT64_C(0x9a77c3e390a9b5f8),
    UINT64_C(0x879949bdb2851424), UINT64_C(0xa1aad75fd4f0f640), UINT64_C(0xbc445d01f6dc579c),
    UINT64_C(0x74f1233d072c2a36), UINT64_C(0x691fa96325008bea), UINT64_C(0x4f2c37814375698e),
    UINT64_C(0x52c2bddf6159c852), UINT64_C(0x034b0a458f9ead46), UINT64_C(0x1ea5801badb20c9a),
    UINT64_C(0x38961ef9cbc7eefe), UINT64_C(0x257894a7e9eb4f22), UINT64_C(0x9b8571cc164924d6),
    UINT64_C(0x866bfb923465850a), UINT64_C(0xa05865705210676e), UINT64_C(0xbdb6ef2e703cc6b2),
    UINT64_C(0xec3f58b49efba3a6), UINT64_C(0xf1d1d2eabcd7027a), UINT64_C(0xd7e24c08daa2e01e),
    UINT64_C(0xca0cc656f88e41c2), UINT64_C(0x38c129f48ae82973), UINT64_C(0x252fa3aaa8c488af),
    UINT64_C(0x031c3d48ceb16acb), UINT64_C(0x1ef2b716ec9dcb17), UINT64_C(0x4f7b008c025aae03),
    UINT64_C(0x52958ad220760fdf), UINT64_C(0x74a614304603edbb), UINT64_C(0x69489e6e642f4c67),
    UINT64_C(0xd7b57b059b8d2793), UINT64_C(0xca5bf15bb9a1864f), UINT64_C(0xec686fb9dfd4642b),
    UINT64_C(0xf186e5e7fdf8c5f7), UINT64_C(0xa00f527d133fa0e3), UINT64_C(0xbde1d8233113013f),
    UINT64_C(0x9bd246c15766e35b), UINT64_C(0x863ccc9f754a4287), UINT64_C(0xec9136ae1ca42cbc),
    UINT64_C(0xf17fbcf03e888d60), UINT64_C(0xd74c221258fd6f04), UINT64_C(0xcaa2a84c7ad1ced8),
    UINT64_C(0x9b2b1fd69416abcc), UINT64_C(0x86c59588b63a0a10), UINT64_C(0xa0f60b6ad04fe874),
    UINT64_C(0xbd188134f26349a8), UINT64_C(0x03e5645f0dc1225c), UINT64_C(0x1e0bee012fed8380),
    UINT64_C(0x383870e3499861e4), UINT64_C(0x25d6fabd6bb4c038), UINT64_C(0x745f4d278573a52c),
    UINT64_C(0x69b1c779a75f04f0), UINT64_C(0x4f82599bc12ae694), UINT64_C(0x526cd3c5e3064748),
    UINT64_C(0xa0a13c6791602ff9), UINT64_C(0xbd4fb639b34c8e25), UINT64_C(0x9b7c28dbd5396c41),
    UINT64_C(0x8692a285f715cd9d), UINT64_C(0xd71b151f19d2a889), UINT64_C(0xcaf59f413bfe0955),
    UINT64_C(0xecc601a35d8beb31), UINT64_C(0xf1288bfd7fa74aed), UINT64_C(0x4fd56e9680052119),
    UINT64_C(0x523be4c8a22980c5), UINT64_C(0x74087a2ac45c62a1), UINT64_C(0x69e6f074e670c37d),
    UINT64_C(0x386f47ee08b7a669), UINT64_C(0x2581cdb02a9b07b5), UINT64_C(0x03b253524ceee5d1),
    UINT64_C(0x1e5cd90c6ec2440d) },
  { UINT64_C(0x0000000000000000), UINT64_C(0x3f0be14a916a6dcb), UINT64_C(0x7e17c29522d4db96),
    UINT64_C(0x411c23dfb3beb65d), UINT64_C(0xfc2f852a45a9b72c), UINT64_C(0xc3246460d4c3dae7),
    UINT64_C(0x823847bf677d6cba), UINT64_C(0xbd33a6f5f6170171), UINT64_C(0x6a87a57f245d70dd),
    UINT64_C(0x558c4435b5371d16), UINT64_C(0x149067ea0689ab4b), UINT64_C(0x2b9b86a097e3c680),
    UINT64_C(0x96a8205561f4c7f1), UINT64_C(0xa9a3c11ff09eaa3a), UINT64_C(0xe8bfe2c043201c67),
    UINT64_C(0xd7b4038ad24a71ac), UINT64_C(0xd50f4afe48bae1ba), UINT64_C(0xea04abb4d9d08c71),
    UINT64_C(0xab18886b6a6e3a2c), UINT64_C(0x94136921fb0457e7), UINT64_C(0x2920cfd40d135696),
    UINT64_C(0x162b2e9e9c793b5d), UINT64_C(0x57370d412fc78d00), UINT64_C(0x683cec0bbeade0cb),
    UINT64_C(0xbf88ef816ce79167), UINT64_C(0x80830ecbfd8dfcac), UINT64_C(0xc19f2d144e334af1),
    UINT64_C(0xfe94cc5edf59273a), UINT64_C(0x43a76aab294e264b), UINT64_C(0x7cac8be1b8244b80),
    UINT64_C(0x3db0a83e0b9afddd), UINT64_C(0x02bb49749af09016), UINT64_C(0x38c63ad73e7bddf1),
    UINT64_C(0x07cddb9daf11b03a), UINT64_C(0x46d1f8421caf0667), UINT64_C(0x79da19088dc56bac),
    UINT64_C(0xc4e9bffd7bd26add), UINT64_C(0xfbe25eb7eab80716), UINT64_C(0xbafe7d685906b14b),
    UINT64_C(0x85f59c22c86cdc80), UINT64_C(0x52419fa81a26ad2c), UINT64_C(0x6d4a7ee28b4cc0e7),
    UINT64_C(0x2c565d3d38f276ba), UINT64_C(0x135dbc77a9981b71), UINT64_C(0xae6e1a825f8f1a00),
    UINT64_C(0x9165fbc8cee577cb), UINT64_C(0xd079d8177d5bc196), UINT64_C(0xef72395dec31ac5d),
    UINT64_C(0xedc9702976c13c4b), UINT64_C(0xd2c29163e7ab5180), UINT64_C(0x93deb2bc5415e7dd),
    UINT64_C(0xacd553f6c57f8a16), UINT64_C(0x11e6f50333688b67), UINT64_C(0x2eed1449a202e6ac),
    UINT64_C(0x6ff1379611bc50f1), UINT64_C(0x50fad6dc80d63d3a), UINT64_C(0x874ed556529c4c96),
    UINT64_C(0xb845341cc3f6215d), UINT64_C(0xf95917c370489700), UINT64_C(0xc652f689e122facb),
    UINT64_C(0x7b61507c1735fbba), UINT64_C(0x446ab136865f9671), UINT64_C(0x057692e935e1202c),
    UINT64_C(0x3a7d73a3a48b4de7), UINT64_C(0x718c75ae7cf7bbe2), UINT64_C(0x4e8794e4ed9dd629),
    UINT64_C(0x0f9bb73b5e236074), UINT64_C(0x30905671cf490dbf), UINT64_C(0x8da3f084395e0cce),
    UINT64_C(0xb2a811cea8346105), UINT64_C(0xf3b432111b8ad758), UINT64_C(0xccbfd35b8ae0ba93),
    UINT64_C(0x1b0bd0d158aacb3f), UINT64_C(0x2400319bc9c0a6f4), UINT64_C(0x651c12447a7e10a9),
    UINT64_C(0x5a17f30eeb147d62), UINT64_C(0xe72455fb1d037c13), UINT64_C(0xd82fb4b18c6911d8),
    UINT64_C(0x9933976e3fd7a785), UINT64_C(0xa6387624aebdca4e), UINT64_C(0xa4833f50344d5a58),
    UINT64_C(0x9b88de1aa5273793), UINT64_C(0xda94fdc5169981ce), UINT64_C(0xe59f1c8f87f3ec05),
    UINT64_C(0x58acba7a71e4ed74), UINT64_C(0x67a75b30e08e80bf), UINT64_C(0x26bb78ef533036e2),
    UINT64_C(0x19b099a5c25a5b29), UINT64_C(0xce049a2f10102a85), UINT64_C(0xf10f7b65817a474e),
    UINT64_C(0xb01358ba32c4f113), UINT64_C(0x8f18b9f0a3ae9cd8), UINT64_C(0x322b1f0555b99da9),
    UINT64_C(0x0d20fe4fc4d3f062), UINT64_C(0x4c3cdd90776d463f), UINT64_C(0x73373cdae6072bf4),
    UINT64_C(0x494a4f79428c6613), UINT64_C(0x7641ae33d3e60bd8), UINT64_C(0x375d8dec6058bd85),
    UINT64_C(0x08566ca6f132d04e), UINT64_C(0xb565ca530725d13f), UINT64_C(0x8a6e2b19964fbcf4),
    UINT64_C(0xcb7208c625f10aa9), UINT64_C(0xf479e98cb49b6762), UINT64_C(0x23cdea0666d116ce),
    UINT64_C(0x1cc60b4cf7bb7b05), UINT64_C(0x5dda28934405cd58), UINT64_C(0x62d1c9d9d56fa093),
    UINT64_C(0xdfe26f2c2378a1e2), UINT64_C(0xe0e98e66b212cc29), UINT64_C(0xa1f5adb901ac7a74),
    UINT64_C(0x9efe4cf390c617bf), UINT64_C(0x9c4505870a3687a9), UINT64_C(0xa34ee4cd9b5cea62),
    UINT64_C(0xe252c71228e25c3f), UINT64_C(0xdd592658b98831f4), UINT64_C(0x606a80ad4f9f3085),
    UINT64_C(0x5f6161e7def55d4e), UINT64_C(0x1e7d42386d4beb13), UINT64_C(0x2176a372fc2186d8),
    UINT64_C(0xf6c2a0f82e6bf774), UINT64_C(0xc9c941b2bf019abf), UINT64_C(0x88d5626d0cbf2ce2),
    UINT64_C(0xb7de83279dd54129), UINT64_C(0x0aed25d26bc24058), UINT64_C(0x35e6c498faa82d93),
    UINT64_C(0x74fae74749169bce), UINT64_C(0x4bf1060dd87cf605), UINT64_C(0xe318eb5cf9ef77c4),
    UINT64_C(0xdc130a1668851a0f), UINT64_C(0x9d0f29c9db3bac52), UINT64_C(0xa204c8834a51c199),
    UINT64_C(0x1f376e76bc46c0e8), UINT64_C(0x203c8f3c2d2cad23), UINT64_C(0x6120ace39e921b7e),
    UINT64_C(0x5e2b4da90ff876b5), UINT64_C(0x899f4e23ddb20719), UINT64_C(0xb694af694cd86ad2),
    UINT64_C(0xf7888cb6ff66dc8f), UINT64_C(0xc8836dfc6e0cb144), UINT64_C(0x75b0cb09981bb035),
    UINT64_C(0x4abb2a430971ddfe), UINT64_C(0x0ba7099cbacf6ba3), UINT64_C(0x34ace8d62ba50668),
    UINT64_C(0x3617a1a2b155967e), UINT64_C(0x091c40e8203ffbb5), UINT64_C(0x4800633793814de8),
    UINT64_C(0x770b827d02eb2023), UINT64_C(0xca382488f4fc2152), UINT64_C(0xf533c5c265964c99),
    UINT64_C(0xb42fe61dd628fac4), UINT64_C(0x8b2407574742970f), UINT64_C(0x5c9004dd9508e6a3),
    UINT64_C(0x639be59704628b68), UINT64_C(0x2287c648b7dc3d35), UINT64_C(0x1d8c270226b650fe),
    UINT64_C(0xa0bf81f7d0a1518f), UINT64_C(0x9fb460bd41cb3c44), UINT64_C(0xdea84362f2758a19),
    UINT64_C(0xe1a3a228631fe7d2), UINT64_C(0xdbded18bc794aa35), UINT64_C(0xe4d530c156fec7fe),
    UINT64_C(0xa5c9131ee54071a3), UINT64_C(0x9ac2f254742a1c68), UINT64_C(0x27f154a1823d1d19),
    UINT64_C(0x18fab5eb135770d2), UINT64_C(0x59e69634a0e9c68f), UINT64_C(0x66ed777e3183ab44),
    UINT64_C(0xb15974f4e3c9dae8), UINT64_C(0x8e5295be72a3b723), UINT64_C(0xcf4eb661c11d017e),
    UINT64_C(0xf045572b50776cb5), UINT64_C(0x4d76f1dea6606dc4), UINT64_C(0x727d1094370a000f),
    UINT64_C(0x3361334b84b4b652), UINT64_C(0x0c6ad20115dedb99), UINT64_C(0x0ed19b758f2e4b8f),
    UINT64_C(0x31da7a3f1e442644), UINT64_C(0x70c659e0adfa9019), UINT64_C(0x4fcdb8aa3c90fdd2),
    UINT64_C(0xf2fe1e5fca87fca3), UINT64_C(0xcdf5ff155bed9168), UINT64_C(0x8ce9dccae8532735),
    UINT64_C(0xb3e23d8079394afe), UINT64_C(0x64563e0aab733b52), UINT64_C(0x5b5ddf403a195699),
    UINT64_C(0x1a41fc9f89a7e0c4), UINT64_C(0x254a1dd518cd8d0f), UINT64_C(0x9879bb20eeda8c7e),
    UINT64_C(0xa7725a6a7fb0e1b5), UINT64_C(0xe66e79b5cc0e57e8), UINT64_C(0xd96598ff5d643a23),
    UINT64_C(0x92949ef28518cc26), UINT64_C(0xad9f7fb81472a1ed), UINT64_C(0xec835c67a7cc17b0),
    UINT64_C(0xd388bd2d36a67a7b), UINT64_C(0x6ebb1bd8c0b17b0a), UINT64_C(0x51b0fa9251db16c1),
    UINT64_C(0x10acd94de265a09c), UINT64_C(0x2fa73807730fcd57), UINT64_C(0xf8133b8da145bcfb),
    UINT64_C(0xc718dac7302fd130), UINT64_C(0x8604f9188391676d), UINT64_C(0xb90f185212fb0aa6),
    UINT64_C(0x043cbea7e4ec0bd7), UINT64_C(0x3b375fed7586661c), UINT64_C(0x7a2b7c32c638d041),
    UINT64_C(0x45209d785752bd8a), UINT64_C(0x479bd40ccda22d9c), UINT64_C(0x789035465cc84057),
    UINT64_C(0x398c1699ef76f60a), UINT64_C(0x0687f7d37e1c9bc1), UINT64_C(0xbbb45126880b9ab0),
    UINT64_C(0x84bfb06c1961f77b), UINT64_C(0xc5a393b3aadf4126), UINT64_C(0xfaa872f93bb52ced),
    UINT64_C(0x2d1c7173e9ff5d41), UINT64_C(0x121790397895308a), UINT64_C(0x530bb3e6cb2b86d7),
    UINT64_C(0x6c0052ac5a41eb1c), UINT64_C(0xd133f459ac56ea6d), UINT64_C(0xee3815133d3c87a6),
    UINT64_C(0xaf2436cc8e8231fb), UINT64_C(0x902fd7861fe85c30), UINT64_C(0xaa52a425bb6311d7),
    UINT64_C(0x9559456f2a097c1c), UINT64_C(0xd44566b099b7ca41), UINT64_C(0xeb4e87fa08dda78a),
    UINT64_C(0x567d210ffecaa6fb), UINT64_C(0x6976c0456fa0cb30), UINT64_C(0x286ae39adc1e7d6d),
    UINT64_C(0x176102d04d7410a6), UINT64_C(0xc0d5015a9f3e610a), UINT64_C(0xffdee0100e540cc1),
    UINT64_C(0xbec2c3cfbdeaba9c), UINT64_C(0x81c922852c80d757), UINT64_C(0x3cfa8470da97d626),
    UINT64_C(0x03f1653a4bfdbbed), UINT64_C(0x42ed46e5f8430db0), UINT64_C(0x7de6a7af6929607b),
    UINT64_C(0x7f5deedbf3d9f06d), UINT64_C(0x40560f9162b39da6), UINT64_C(0x014a2c4ed10d2bfb),
    UINT64_C(0x3e41cd0440674630), UINT64_C(0x83726bf1b6704741), UINT64_C(0xbc798abb271a2a8a),
    UINT64_C(0xfd65a96494a49cd7), UINT64_C(0xc26e482e05cef11c), UINT64_C(0x15da4ba4d78480b0),
    UINT64_C(0x2ad1aaee46eeed7b), UINT64_C(0x6bcd8931f5505b26), UINT64_C(0x54c6687b643a36ed),
    UINT64_C(0xe9f5ce8e922d379c), UINT64_C(0xd6fe2fc403475a57), UINT64_C(0x97e20c1bb0f9ec0a),
    UINT64_C(0xa8e9ed51219381c1) },
  { UINT64_C(0x0000000000000000), UINT64_C(0x54e979925cd0f10d), UINT64_C(0xa9d2f324b9a1e21a),
    UINT64_C(0xfd3b8ab6e5711317), UINT64_C(0xc17d4962dc4ddab1), UINT64_C(0x959430f0809d2bbc),
    UINT64_C(0x68afba4665ec38ab), UINT64_C(0x3c46c3d4393cc9a6), UINT64_C(0x10223dee1795abe7),
    UINT64_C(0x44cb447c4b455aea), UINT64_C(0xb9f0cecaae3449fd), UINT64_C(0xed19b758f2e4b8f0),
    UINT64_C(0xd15f748ccbd87156), UINT64_C(0x85b60d1e9708805b), UINT64_C(0x788d87a87279934c),
    UINT64_C(0x2c64fe3a2ea96241), UINT64_C(0x20447bdc2f2b57ce), UINT64_C(0x74ad024e73fba6c3),
    UINT64_C(0x899688f8968ab5d4), UINT64_C(0xdd7ff16aca5a44d9), UINT64_C(0xe13932bef3668d7f),
    UINT64_C(0xb5d04b2cafb67c72), UINT64_C(0x48ebc19a4ac76f65), UINT64_C(0x1c02b80816179e68),
    UINT64_C(0x3066463238befc29), UINT64_C(0x648f3fa0646e0d24), UINT64_C(0x99b4b516811f1e33),
    UINT64_C(0xcd5dcc84ddcfef3e), UINT64_C(0xf11b0f50e4f32698), UINT64_C(0xa5f276c2b823d795),
    UINT64_C(0x58c9fc745d52c482), UINT64_C(0x0c2085e60182358f), UINT64_C(0x4088f7b85e56af9c),
    UINT64_C(0x14618e2a02865e91), UINT64_C(0xe95a049ce7f74d86), UINT64_C(0xbdb37d0ebb27bc8b),
    UINT64_C(0x81f5beda821b752d), UINT64_C(0xd51cc748decb8420), UINT64_C(0x28274dfe3bba9737),
    UINT64_C(0x7cce346c676a663a), UINT64_C(0x50aaca5649c3047b), UINT64_C(0x0443b3c41513f576),
    UINT64_C(0xf9783972f062e661), UINT64_C(0xad9140e0acb2176c), UINT64_C(0x91d78334958edeca),
    UINT64_C(0xc53efaa6c95e2fc7), UINT64_C(0x380570102c2f3cd0), UINT64_C(0x6cec098270ffcddd),
    UINT64_C(0x60cc8c64717df852), UINT64_C(0x3425f5f62dad095f), UINT64_C(0xc91e7f40c8dc1a48),
    UINT64_C(0x9df706d2940ceb45), UINT64_C(0xa1b1c506ad3022e3), UINT64_C(0xf558bc94f1e0d3ee),
    UINT64_C(0x086336221491c0f9), UINT64_C(0x5c8a4fb0484131f4), UINT64_C(0x70eeb18a66e853b5),
    UINT64_C(0x2407c8183a38a2b8), UINT64_C(0xd93c42aedf49b1af), UINT64_C(0x8dd53b3c839940a2),
    UINT64_C(0xb193f8e8baa58904), UINT64_C(0xe57a817ae6757809), UINT64_C(0x18410bcc03046b1e),
    UINT64_C(0x4ca8725e5fd49a13), UINT64_C(0x8111ef70bcad5f38), UINT64_C(0xd5f896e2e07dae35),
    UINT64_C(0x28c31c54050cbd22), UINT64_C(0x7c2a65c659dc4c2f), UINT64_C(0x406ca61260e08589),
    UINT64_C(0x1485df803c307484), UINT64_C(0xe9be5536d9416793), UINT64_C(0xbd572ca48591969e),
    UINT64_C(0x9133d29eab38f4df), UINT64_C(0xc5daab0cf7e805d2), UINT64_C(0x38e121ba129916c5),
    UINT64_C(0x6c0858284e49e7c8), UINT64_C(0x504e9bfc77752e6e), UINT64_C(0x04a7e26e2ba5df63),
    UINT64_C(0xf99c68d8ced4cc74), UINT64_C(0xad75114a92043d79), UINT64_C(0xa15594ac938608f6),
    UINT64_C(0xf5bced3ecf56f9fb), UINT64_C(0x088767882a27eaec), UINT64_C(0x5c6e1e1a76f71be1),
    UINT64_C(0x6028ddce4fcbd247), UINT64_C(0x34c1a45c131b234a), UINT64_C(0xc9fa2eeaf66a305d),
    UINT64_C(0x9d135778aabac150), UINT64_C(0xb177a9428413a311), UINT64_C(0xe59ed0d0d8c3521c),
    UINT64_C(0x18a55a663db2410b), UINT64_C(0x4c4c23f46162b006), UINT64_C(0x700ae020585e79a0),
    UINT64_C(0x24e399b2048e88ad), UINT64_C(0xd9d81304e1ff9bba), UINT64_C(0x8d316a96bd2f6ab7),
    UINT64_C(0xc19918c8e2fbf0a4), UINT64_C(0x9570615abe2b01a9), UINT64_C(0x684bebec5b5a12be),
    UINT64_C(0x3ca2927e078ae3b3), UINT64_C(0x00e451aa3eb62a15), UINT64_C(0x540d28386266db18),
    UINT64_C(0xa936a28e8717c80f), UINT64_C(0xfddfdb1cdbc73902), UINT64_C(0xd1bb2526f56e5b43),
    UINT64_C(0x85525cb4a9beaa4e), UINT64_C(0x7869d6024ccfb959), UINT64_C(0x2c80af90101f4854),
    UINT64_C(0x10c66c44292381f2), UINT64_C(0x442f15d675f370ff), UINT64_C(0xb9149f60908263e8),
    UINT64_C(0xedfde6f2cc5292e5), UINT64_C(0xe1dd6314cdd0a76a), UINT64_C(0xb5341a8691005667),
    UINT64_C(0x480f903074714570), UINT64_C(0x1ce6e9a228a1b47d), UINT64_C(0x20a02a76119d7ddb),
    UINT64_C(0x744953e44d4d8cd6), UINT64_C(0x8972d952a83c9fc1), UINT64_C(0xdd9ba0c0f4ec6ecc),
    UINT64_C(0xf1ff5efada450c8d), UINT64_C(0xa51627688695fd80), UINT64_C(0x582dadde63e4ee97),
    UINT64_C(0x0cc4d44c3f341f9a), UINT64_C(0x308217980608d63c), UINT64_C(0x646b6e0a5ad82731),
    UINT64_C(0x9950e4bcbfa93426), UINT64_C(0xcdb99d2ee379c52b), UINT64_C(0x90fb71cad654a0f5),
    UINT64_C(0xc41208588a8451f8), UINT64_C(0x392982ee6ff542ef), UINT64_C(0x6dc0fb7c3325b3e2),
    UINT64_C(0x518638a80a197a44), UINT64_C(0x056f413a56c98b49), UINT64_C(0xf854cb8cb3b8985e),
    UINT64_C(0xacbdb21eef686953), UINT64_C(0x80d94c24c1c10b12), UINT64_C(0xd43035b69d11fa1f),
    UINT64_C(0x290bbf007860e908), UINT64_C(0x7de2c69224b01805), UINT64_C(0x41a405461d8cd1a3),
    UINT64_C(0x154d7cd4415c20ae), UINT64_C(0xe876f662a42d33b9), UINT64_C(0xbc9f8ff0f8fdc2b4),
    UINT64_C(0xb0bf0a16f97ff73b), UINT64_C(0xe4567384a5af0636), UINT64_C(0x196df93240de1521),
    UINT64_C(0x4d8480a01c0ee42c), UINT64_C(0x71c2437425322d8a), UINT64_C(0x252b3ae679e2dc87),
    UINT64_C(0xd810b0509c93cf90), UINT64_C(0x8cf9c9c2c0433e9d), UINT64_C(0xa09d37f8eeea5cdc),
    UINT64_C(0xf4744e6ab23aadd1), UINT64_C(0x094fc4dc574bbec6), UINT64_C(0x5da6bd4e0b9b4fcb),
    UINT64_C(0x61e07e9a32a7866d), UINT64_C(0x350907086e777760), UINT64_C(0xc8328dbe8b066477),
    UINT64_C(0x9cdbf42cd7d6957a), UINT64_C(0xd073867288020f69), UINT64_C(0x849affe0d4d2fe64),
    UINT64_C(0x79a1755631a3ed73), UINT64_C(0x2d480cc46d731c7e), UINT64_C(0x110ecf10544fd5d8),
    UINT64_C(0x45e7b682089f24d5), UINT64_C(0xb8dc3c34edee37c2), UINT64_C(0xec3545a6b13ec6cf),
    UINT64_C(0xc051bb9c9f97a48e), UINT64_C(0x94b8c20ec3475583), UINT64_C(0x698348b826364694),
    UINT64_C(0x3d6a312a7ae6b799), UINT64_C(0x012cf2fe43da7e3f), UINT64_C(0x55c58b6c1f0a8f32),
    UINT64_C(0xa8fe01dafa7b9c25), UINT64_C(0xfc177848a6ab6d28), UINT64_C(0xf037fdaea72958a7),
    UINT64_C(0xa4de843cfbf9a9aa), UINT64_C(0x59e50e8a1e88babd), UINT64_C(0x0d0c771842584bb0),
    UINT64_C(0x314ab4cc7b648216), UINT64_C(0x65a3cd5e27b4731b), UINT64_C(0x989847e8c2c5600c),
    UINT64_C(0xcc713e7a9e159101), UINT64_C(0xe015c040b0bcf340), UINT64_C(0xb4fcb9d2ec6c024d),
    UINT64_C(0x49c73364091d115a), UINT64_C(0x1d2e4af655cde057), UINT64_C(0x216889226cf129f1),
    UINT64_C(0x7581f0b03021d8fc), UINT64_C(0x88ba7a06d550cbeb), UINT64_C(0xdc53039489803ae6),
    UINT64_C(0x11ea9eba6af9ffcd), UINT64_C(0x4503e72836290ec0), UINT64_C(0xb8386d9ed3581dd7),
    UINT64_C(0xecd1140c8f88ecda), UINT64_C(0xd097d7d8b6b4257c), UINT64_C(0x847eae4aea64d471),
    UINT64_C(0x794524fc0f15c766), UINT64_C(0x2dac5d6e53c5366b), UINT64_C(0x01c8a3547d6c542a),
    UINT64_C(0x5521dac621bca527), UINT64_C(0xa81a5070c4cdb630), UINT64_C(0xfcf329e2981d473d),
    UINT64_C(0xc0b5ea36a1218e9b), UINT64_C(0x945c93a4fdf17f96), UINT64_C(0x6967191218806c81),
    UINT64_C(0x3d8e608044509d8c), UINT64_C(0x31aee56645d2a803), UINT64_C(0x65479cf41902590e),
    UINT64_C(0x987c1642fc734a19), UINT64_C(0xcc956fd0a0a3bb14), UINT64_C(0xf0d3ac04999f72b2),
    UINT64_C(0xa43ad596c54f83bf), UINT64_C(0x59015f20203e90a8), UINT64_C(0x0de826b27cee61a5),
    UINT64_C(0x218cd888524703e4), UINT64_C(0x7565a11a0e97f2e9), UINT64_C(0x885e2bacebe6e1fe),
    UINT64_C(0xdcb7523eb73610f3), UINT64_C(0xe0f191ea8e0ad955), UINT64_C(0xb418e878d2da2858),
    UINT64_C(0x492362ce37ab3b4f), UINT64_C(0x1dca1b5c6b7bca42), UINT64_C(0x5162690234af5051),
    UINT64_C(0x058b1090687fa15c), UINT64_C(0xf8b09a268d0eb24b), UINT64_C(0xac59e3b4d1de4346),
    UINT64_C(0x901f2060e8e28ae0), UINT64_C(0xc4f659f2b4327bed), UINT64_C(0x39cdd344514368fa),
    UINT64_C(0x6d24aad60d9399f7), UINT64_C(0x414054ec233afbb6), UINT64_C(0x15a92d7e7fea0abb),
    UINT64_C(0xe892a7c89a9b19ac), UINT64_C(0xbc7bde5ac64be8a1), UINT64_C(0x803d1d8eff772107),
    UINT64_C(0xd4d4641ca3a7d00a), UINT64_C(0x29efeeaa46d6c31d), UINT64_C(0x7d0697381a063210),
    UINT64_C(0x712612de1b84079f), UINT64_C(0x25cf6b4c4754f692), UINT64_C(0xd8f4e1faa225e585),
    UINT64_C(0x8c1d9868fef51488), UINT64_C(0xb05b5bbcc7c9dd2e), UINT64_C(0xe4b2222e9b192c23),
    UINT64_C(0x1989a8987e683f34), UINT64_C(0x4d60d10a22b8ce39), UINT64_C(0x61042f300c11ac78),
    UINT64_C(0x35ed56a250c15d75), UINT64_C(0xc8d6dc14b5b04e62), UINT64_C(0x9c3fa586e960bf6f),
    UINT64_C(0xa0796652d05c76c9), UINT64_C(0xf4901fc08c8c87c4), UINT64_C(0x09ab957669fd94d3),
    UINT64_C(0x5d42ece4352d65de) },
  { UINT64_C(0x0000000000000000), UINT64_C(0xb32e4cbe03a75f6f), UINT64_C(0xf4843657a840a05b),
    UINT64_C(0x47aa7ae9abe7ff34), UINT64_C(0x7bd0c384ff8f5e33), UINT64_C(0xc8fe8f3afc28015c),
    UINT64_C(0x8f54f5d357cffe68), UINT64_C(0x3c7ab96d5468a107), UINT64_C(0xf7a18709ff1ebc66),
    UINT64_C(0x448fcbb7fcb9e309), UINT64_C(0x0325b15e575e1c3d), UINT64_C(0xb00bfde054f94352),
    UINT64_C(0x8c71448d0091e255), UINT64_C(0x3f5f08330336bd3a), UINT64_C(0x78f572daa8d1420e),
    UINT64_C(0xcbdb3e64ab761d61), UINT64_C(0x7d9ba13851336649), UINT64_C(0xceb5ed8652943926),
    UINT64_C(0x891f976ff973c612), UINT64_C(0x3a31dbd1fad4997d), UINT64_C(0x064b62bcaebc387a),
    UINT64_C(0xb5652e02ad1b6715), UINT64_C(0xf2cf54eb06fc9821), UINT64_C(0x41e11855055bc74e),
    UINT64_C(0x8a3a2631ae2dda2f), UINT64_C(0x39146a8fad8a8540), UINT64_C(0x7ebe1066066d7a74),
    UINT64_C(0xcd905cd805ca251b), UINT64_C(0xf1eae5b551a2841c), UINT64_C(0x42c4a90b5205db73),
    UINT64_C(0x056ed3e2f9e22447), UINT64_C(0xb6409f5cfa457b28), UINT64_C(0xfb374270a266cc92),
    UINT64_C(0x48190ecea1c193fd), UINT64_C(0x0fb374270a266cc9), UINT64_C(0xbc9d3899098133a6),
    UINT64_C(0x80e781f45de992a1), UINT64_C(0x33c9cd4a5e4ecdce), UINT64_C(0x7463b7a3f5a932fa),
    UINT64_C(0xc74dfb1df60e6d95), UINT64_C(0x0c96c5795d7870f4), UINT64_C(0xbfb889c75edf2f9b),
    UINT64_C(0xf812f32ef538d0af), UINT64_C(0x4b3cbf90f69f8fc0), UINT64_C(0x774606fda2f72ec7),
    UINT64_C(0xc4684a43a15071a8), UINT64_C(0x83c230aa0ab78e9c), UINT64_C(0x30ec7c140910d1f3),
    UINT64_C(0x86ace348f355aadb), UINT64_C(0x3582aff6f0f2f5b4), UINT64_C(0x7228d51f5b150a80),
    UINT64_C(0xc10699a158b255ef), UINT64_C(0xfd7c20cc0cdaf4e8), UINT64_C(0x4e526c720f7dab87),
    UINT64_C(0x09f8169ba49a54b3), UINT64_C(0xbad65a25a73d0bdc), UINT64_C(0x710d64410c4b16bd),
    UINT64_C(0xc22328ff0fec49d2), UINT64_C(0x85895216a40bb6e6), UINT64_C(0x36a71ea8a7ace989),
    UINT64_C(0x0adda7c5f3c4488e), UINT64_C(0xb9f3eb7bf06317e1), UINT64_C(0xfe5991925b84e8d5),
    UINT64_C(0x4d77dd2c5823b7ba), UINT64_C(0x64b62bcaebc387a1), UINT64_C(0xd7986774e864d8ce),
    UINT64_C(0x90321d9d438327fa), UINT64_C(0x231c512340247895), UINT64_C(0x1f66e84e144cd992),
    UINT64_C(0xac48a4f017eb86fd), UINT64_C(0xebe2de19bc0c79c9), UINT64_C(0x58cc92a7bfab26a6),
    UINT64_C(0x9317acc314dd3bc7), UINT64_C(0x2039e07d177a64a8), UINT64_C(0x67939a94bc9d9b9c),
    UINT64_C(0xd4bdd62abf3ac4f3), UINT64_C(0xe8c76f47eb5265f4), UINT64_C(0x5be923f9e8f53a9b),
    UINT64_C(0x1c4359104312c5af), UINT64_C(0xaf6d15ae40b59ac0), UINT64_C(0x192d8af2baf0e1e8),
    UINT64_C(0xaa03c64cb957be87), UINT64_C(0xeda9bca512b041b3), UINT64_C(0x5e87f01b11171edc),
    UINT64_C(0x62fd4976457fbfdb), UINT64_C(0xd1d305c846d8e0b4), UINT64_C(0x96797f21ed3f1f80),
    UINT64_C(0x2557339fee9840ef), UINT64_C(0xee8c0dfb45ee5d8e), UINT64_C(0x5da24145464902e1),
    UINT64_C(0x1a083bacedaefdd5), UINT64_C(0xa9267712ee09a2ba), UINT64_C(0x955cce7fba6103bd),
    UINT64_C(0x267282c1b9c65cd2), UINT64_C(0x61d8f8281221a3e6), UINT64_C(0xd2f6b4961186fc89),
    UINT64_C(0x9f8169ba49a54b33), UINT64_C(0x2caf25044a02145c), UINT64_C(0x6b055fede1e5eb68),
    UINT64_C(0xd82b1353e242b407), UINT64_C(0xe451aa3eb62a1500), UINT64_C(0x577fe680b58d4a6f),
    UINT64_C(0x10d59c691e6ab55b), UINT64_C(0xa3fbd0d71dcdea34), UINT64_C(0x6820eeb3b6bbf755),
    UINT64_C(0xdb0ea20db51ca83a), UINT64_C(0x9ca4d8e41efb570e), UINT64_C(0x2f8a945a1d5c0861),
    UINT64_C(0x13f02d374934a966), UINT64_C(0xa0de61894a93f609), UINT64_C(0xe7741b60e174093d),
    UINT64_C(0x545a57dee2d35652), UINT64_C(0xe21ac88218962d7a), UINT64_C(0x5134843c1b317215),
    UINT64_C(0x169efed5b0d68d21), UINT64_C(0xa5b0b26bb371d24e), UINT64_C(0x99ca0b06e7197349),
    UINT64_C(0x2ae447b8e4be2c26), UINT64_C(0x6d4e3d514f59d312), UINT64_C(0xde6071ef4cfe8c7d),
    UINT64_C(0x15bb4f8be788911c), UINT64_C(0xa6950335e42fce73), UINT64_C(0xe13f79dc4fc83147),
    UINT64_C(0x521135624c6f6e28), UINT64_C(0x6e6b8c0f1807cf2f), UINT64_C(0xdd45c0b11ba09040),
    UINT64_C(0x9aefba58b0476f74), UINT64_C(0x29c1f6e6b3e0301b), UINT64_C(0xc96c5795d7870f42),
    UINT64_C(0x7a421b2bd420502d), UINT64_C(0x3de861c27fc7af19), UINT64_C(0x8ec62d7c7c60f076),
    UINT64_C(0xb2bc941128085171), UINT64_C(0x0192d8af2baf0e1e), UINT64_C(0x4638a2468048f12a),
    UINT64_C(0xf516eef883efae45), UINT64_C(0x3ecdd09c2899b324), UINT64_C(0x8de39c222b3eec4b),
    UINT64_C(0xca49e6cb80d9137f), UINT64_C(0x7967aa75837e4c10), UINT64_C(0x451d1318d716ed17),
    UINT64_C(0xf6335fa6d4b1b278), UINT64_C(0xb199254f7f564d4c), UINT64_C(0x02b769f17cf11223),
    UINT64_C(0xb4f7f6ad86b4690b), UINT64_C(0x07d9ba1385133664), UINT64_C(0x4073c0fa2ef4c950),
    UINT64_C(0xf35d8c442d53963f), UINT64_C(0xcf273529793b3738), UINT64_C(0x7c0979977a9c6857),
    UINT64_C(0x3ba3037ed17b9763), UINT64_C(0x888d4fc0d2dcc80c), UINT64_C(0x435671a479aad56d),
    UINT64_C(0xf0783d1a7a0d8a02), UINT64_C(0xb7d247f3d1ea7536), UINT64_C(0x04fc0b4dd24d2a59),
    UINT64_C(0x3886b22086258b5e), UINT64_C(0x8ba8fe9e8582d431), UINT64_C(0xcc0284772e652b05),
    UINT64_C(0x7f2cc8c92dc2746a), UINT64_C(0x325b15e575e1c3d0), UINT64_C(0x8175595b76469cbf),
    UINT64_C(0xc6df23b2dda1638b), UINT64_C(0x75f16f0cde063ce4), UINT64_C(0x498bd6618a6e9de3),
    UINT64_C(0xfaa59adf89c9c28c), UINT64_C(0xbd0fe036222e3db8), UINT64_C(0x0e21ac88218962d7),
    UINT64_C(0xc5fa92ec8aff7fb6), UINT64_C(0x76d4de52895820d9), UINT64_C(0x317ea4bb22bfdfed),
    UINT64_C(0x8250e80521188082), UINT64_C(0xbe2a516875702185), UINT64_C(0x0d041dd676d77eea),
    UINT64_C(0x4aae673fdd3081de), UINT64_C(0xf9802b81de97deb1), UINT64_C(0x4fc0b4dd24d2a599),
    UINT64_C(0xfceef8632775faf6), UINT64_C(0xbb44828a8c9205c2), UINT64_C(0x086ace348f355aad),
    UINT64_C(0x34107759db5dfbaa), UINT64_C(0x873e3be7d8faa4c5), UINT64_C(0xc094410e731d5bf1),
    UINT64_C(0x73ba0db070ba049e), UINT64_C(0xb86133d4dbcc19ff), UINT64_C(0x0b4f7f6ad86b4690),
    UINT64_C(0x4ce50583738cb9a4), UINT64_C(0xffcb493d702be6cb), UINT64_C(0xc3b1f050244347cc),
    UINT64_C(0x709fbcee27e418a3), UINT64_C(0x3735c6078c03e797), UINT64_C(0x841b8ab98fa4b8f8),
    UINT64_C(0xadda7c5f3c4488e3), UINT64_C(0x1ef430e13fe3d78c), UINT64_C(0x595e4a08940428b8),
    UINT64_C(0xea7006b697a377d7), UINT64_C(0xd60abfdbc3cbd6d0), UINT64_C(0x6524f365c06c89bf),
    UINT64_C(0x228e898c6b8b768b), UINT64_C(0x91a0c532682c29e4), UINT64_C(0x5a7bfb56c35a3485),
    UINT64_C(0xe955b7e8c0fd6bea), UINT64_C(0xaeffcd016b1a94de), UINT64_C(0x1dd181bf68bdcbb1),
    UINT64_C(0x21ab38d23cd56ab6), UINT64_C(0x9285746c3f7235d9), UINT64_C(0xd52f0e859495caed),
    UINT64_C(0x6601423b97329582), UINT64_C(0xd041dd676d77eeaa), UINT64_C(0x636f91d96ed0b1c5),
    UINT64_C(0x24c5eb30c5374ef1), UINT64_C(0x97eba78ec690119e), UINT64_C(0xab911ee392f8b099),
    UINT64_C(0x18bf525d915feff6), UINT64_C(0x5f1528b43ab810c2), UINT64_C(0xec3b640a391f4fad),
    UINT64_C(0x27e05a6e926952cc), UINT64_C(0x94ce16d091ce0da3), UINT64_C(0xd3646c393a29f297),
    UINT64_C(0x604a2087398eadf8), UINT64_C(0x5c3099ea6de60cff), UINT64_C(0xef1ed5546e415390),
    UINT64_C(0xa8b4afbdc5a6aca4), UINT64_C(0x1b9ae303c601f3cb), UINT64_C(0x56ed3e2f9e224471),
    UINT64_C(0xe5c372919d851b1e), UINT64_C(0xa26908783662e42a), UINT64_C(0x114744c635c5bb45),
    UINT64_C(0x2d3dfdab61ad1a42), UINT64_C(0x9e13b115620a452d), UINT64_C(0xd9b9cbfcc9edba19),
    UINT64_C(0x6a978742ca4ae576), UINT64_C(0xa14cb926613cf817), UINT64_C(0x1262f598629ba778),
    UINT64_C(0x55c88f71c97c584c), UINT64_C(0xe6e6c3cfcadb0723), UINT64_C(0xda9c7aa29eb3a624),
    UINT64_C(0x69b2361c9d14f94b), UINT64_C(0x2e184cf536f3067f), UINT64_C(0x9d36004b35545910),
    UINT64_C(0x2b769f17cf112238), UINT64_C(0x9858d3a9ccb67d57), UINT64_C(0xdff2a94067518263),
    UINT64_C(0x6cdce5fe64f6dd0c), UINT64_C(0x50a65c93309e7c0b), UINT64_C(0xe388102d33392364),
    UINT64_C(0xa4226ac498dedc50), UINT64_C(0x170c267a9b79833f), UINT64_C(0xdcd7181e300f9e5e),
    UINT64_C(0x6ff954a033a8c131), UINT64_C(0x28532e49984f3e05), UINT64_C(0x9b7d62f79be8616a),
    UINT64_C(0xa707db9acf80c06d), UINT64_C(0x14299724cc279f02), UINT64_C(0x5383edcd67c06036),
    UINT64_C(0xe0ada17364673f59) }
};

static const uint64_t crc64_interleaved_table[4][256] = {
  { UINT64_C(0x0000000000000000), UINT64_C(0xe88a0d0c5521de3d), UINT64_C(0x43ccb533054da2ff),
    UINT64_C(0xab46b83f506c7cc2), UINT64_C(0x87996a660a9b45fe), UINT64_C(0x6f13676a5fba9bc3),
    UINT64_C(0xc455df550fd6e701), UINT64_C(0x2cdfd2595af7393c), UINT64_C(0x9dea7be7ba389579),
    UINT64_C(0x756076ebef194b44), UINT64_C(0xde26ced4bf753786), UINT64_C(0x36acc3d8ea54e9bb),
    UINT64_C(0x1a731181b0a3d087), UINT64_C(0xf2f91c8de5820eba), UINT64_C(0x59bfa4b2b5ee7278),
    UINT64_C(0xb135a9bee0cfac45), UINT64_C(0xa90c58e4db7f3477), UINT64_C(0x418655e88e5eea4a),
    UINT64_C(0xeac0edd7de329688), UINT64_C(0x024ae0db8b1348b5), UINT64_C(0x2e953282d1e47189),
    UINT64_C(0xc61f3f8e84c5afb4), UINT64_C(0x6d5987b1d4a9d376), UINT64_C(0x85d38abd81880d4b),
    UINT64_C(0x34e623036147a10e), UINT64_C(0xdc6c2e0f34667f33), UINT64_C(0x772a9630640a03f1),
    UINT64_C(0x9fa09b3c312bddcc), UINT64_C(0xb37f49656bdce4f0), UINT64_C(0x5bf544693efd3acd),
    UINT64_C(0xf0b3fc566e91460f), UINT64_C(0x1839f15a3bb09832), UINT64_C(0xc0c01ee219f0766b),
    UINT64_C(0x284a13ee4cd1a856), UINT64_C(0x830cabd11cbdd494), UINT64_C(0x6b86a6dd499c0aa9),
    UINT64_C(0x47597484136b3395), UINT64_C(0xafd37988464aeda8), UINT64_C(0x0495c1b71626916a),
    UINT64_C(0xec1fccbb43074f57), UINT64_C(0x5d2a6505a3c8e312), UINT64_C(0xb5a06809f6e93d2f),
    UINT64_C(0x1ee6d036a68541ed), UINT64_C(0xf66cdd3af3a49fd0), UINT64_C(0xdab30f63a953a6ec),
    UINT64_C(0x3239026ffc7278d1), UINT64_C(0x997fba50ac1e0413), UINT64_C(0x71f5b75cf93fda2e),
    UINT64_C(0x69cc4606c28f421c), UINT64_C(0x81464b0a97ae9c21), UINT64_C(0x2a00f335c7c2e0e3),
    UINT64_C(0xc28afe3992e33ede), UINT64_C(0xee552c60c81407e2), UINT64_C(0x06df216c9d35d9df),
    UINT64_C(0xad999953cd59a51d), UINT64_C(0x4513945f98787b20), UINT64_C(0xf4263de178b7d765),
    UINT64_C(0x1cac30ed2d960958), UINT64_C(0xb7ea88d27dfa759a), UINT64_C(0x5f6085de28dbaba7),
    UINT64_C(0x73bf5787722c929b), UINT64_C(0x9b355a8b270d4ca6), UINT64_C(0x3073e2b477613064),
    UINT64_C(0xd8f9efb82240ee59), UINT64_C(0x135892ef9ceef253), UINT64_C(0xfbd29fe3c9cf2c6e),
    UINT64_C(0x509427dc99a350ac), UINT64_C(0xb81e2ad0cc828e91), UINT64_C(0x94c1f8899675b7ad),
    UINT64_C(0x7c4bf585c3546990), UINT64_C(0xd70d4dba93381552), UINT64_C(0x3f8740b6c619cb6f),
    UINT64_C(0x8eb2e90826d6672a), UINT64_C(0x6638e40473f7b917), UINT64_C(0xcd7e5c3b239bc5d5),
    UINT64_C(0x25f4513776ba1be8), UINT64_C(0x092b836e2c4d22d4), UINT64_C(0xe1a18e62796cfce9),
    UINT64_C(0x4ae7365d2900802b), UINT64_C(0xa26d3b517c215e16), UINT64_C(0xba54ca0b4791c624),
    UINT64_C(0x52dec70712b01819), UINT64_C(0xf9987f3842dc64db), UINT64_C(0x1112723417fdbae6),
    UINT64_C(0x3dcda06d4d0a83da), UINT64_C(0xd547ad61182b5de7), UINT64_C(0x7e01155e48472125),
    UINT64_C(0x968b18521d66ff18), UINT64_C(0x27beb1ecfda9535d), UINT64_C(0xcf34bce0a8888d60),
    UINT64_C(0x647204dff8e4f1a2), UINT64_C(0x8cf809d3adc52f9f), UINT64_C(0xa027db8af73216a3),
    UINT64_C(0x48add686a213c89e), UINT64_C(0xe3eb6eb9f27fb45c), UINT64_C(0x0b6163b5a75e6a61),
    UINT64_C(0xd3988c0d851e8438), UINT64_C(0x3b128101d03f5a05), UINT64_C(0x9054393e805326c7),
    UINT64_C(0x78de3432d572f8fa), UINT64_C(0x5401e66b8f85c1c6), UINT64_C(0xbc8beb67daa41ffb),
    UINT64_C(0x17cd53588ac86339), UINT64_C(0xff475e54dfe9bd04), UINT64_C(0x4e72f7ea3f261141),
    UINT64_C(0xa6f8fae66a07cf7c), UINT64_C(0x0dbe42d93a6bb3be), UINT64_C(0xe5344fd56f4a6d83),
    UINT64_C(0xc9eb9d8c35bd54bf), UINT64_C(0x21619080609c8a82), UINT64_C(0x8a2728bf30f0f640),
    UINT64_C(0x62ad25b365d1287d), UINT64_C(0x7a94d4e95e61b04f), UINT64_C(0x921ed9e50b406e72),
    UINT64_C(0x395861da5b2c12b0), UINT64_C(0xd1d26cd60e0dcc8d), UINT64_C(0xfd0dbe8f54faf5b1),
    UINT64_C(0x1587b38301db2b8c), UINT64_C(0xbec10bbc51b7574e), UINT64_C(0x564b06b004968973),
    UINT64_C(0xe77eaf0ee4592536), UINT64_C(0x0ff4a202b178fb0b), UINT64_C(0xa4b21a3de11487c9),
    UINT64_C(0x4c381731b43559f4), UINT64_C(0x60e7c568eec260c8), UINT64_C(0x886dc864bbe3bef5),
    UINT64_C(0x232b705beb8fc237), UINT64_C(0xcba17d57beae1c0a), UINT64_C(0x26b125df39dde4a6),
    UINT64_C(0xce3b28d36cfc3a9b), UINT64_C(0x657d90ec3c904659), UINT64_C(0x8df79de069b19864),
    UINT64_C(0xa1284fb93346a158), UINT64_C(0x49a242b566677f65), UINT64_C(0xe2e4fa8a360b03a7),
    UINT64_C(0x0a6ef786632add9a), UINT64_C(0xbb5b5e3883e571df), UINT64_C(0x53d15334d6c4afe2),
    UINT64_C(0xf897eb0b86a8d320), UINT64_C(0x101de607d3890d1d), UINT64_C(0x3cc2345e897e3421),
    UINT64_C(0xd4483952dc5fea1c), UINT64_C(0x7f0e816d8c3396de), UINT64_C(0x97848c61d91248e3),
    UINT64_C(0x8fbd7d3be2a2d0d1), UINT64_C(0x67377037b7830eec), UINT64_C(0xcc71c808e7ef722e),
    UINT64_C(0x24fbc504b2ceac13), UINT64_C(0x0824175de839952f), UINT64_C(0xe0ae1a51bd184b12),
    UINT64_C(0x4be8a26eed7437d0), UINT64_C(0xa362af62b855e9ed), UINT64_C(0x125706dc589a45a8),
    UINT64_C(0xfadd0bd00dbb9b95), UINT64_C(0x519bb3ef5dd7e757), UINT64_C(0xb911bee308f6396a),
    UINT64_C(0x95ce6cba52010056), UINT64_C(0x7d4461b60720de6b), UINT64_C(0xd602d989574ca2a9),
    UINT64_C(0x3e88d485026d7c94), UINT64_C(0xe6713b3d202d92cd), UINT64_C(0x0efb3631750c4cf0),
    UINT64_C(0xa5bd8e0e25603032), UINT64_C(0x4d3783027041ee0f), UINT64_C(0x61e8515b2ab6d733),
    UINT64_C(0x89625c577f97090e), UINT64_C(0x2224e4682ffb75cc), UINT64_C(0xcaaee9647adaabf1),
    UINT64_C(0x7b9b40da9a1507b4), UINT64_C(0x93114dd6cf34d989), UINT64_C(0x3857f5e99f58a54b),
    UINT64_C(0xd0ddf8e5ca797b76), UINT64_C(0xfc022abc908e424a), UINT64_C(0x148827b0c5af9c77),
    UINT64_C(0xbfce9f8f95c3e0b5), UINT64_C(0x57449283c0e23e88), UINT64_C(0x4f7d63d9fb52a6ba),
    UINT64_C(0xa7f76ed5ae737887), UINT64_C(0x0cb1d6eafe1f0445), UINT64_C(0xe43bdbe6ab3eda78),
    UINT64_C(0xc8e409bff1c9e344), UINT64_C(0x206e04b3a4e83d79), UINT64_C(0x8b28bc8cf48441bb),
    UINT64_C(0x63a2b180a1a59f86), UINT64_C(0xd297183e416a33c3), UINT64_C(0x3a1d1532144bedfe),
    UINT64_C(0x915bad0d4427913c), UINT64_C(0x79d1a00111064f01), UINT64_C(0x550e72584bf1763d),
    UINT64_C(0xbd847f541ed0a800), UINT64_C(0x16c2c76b4ebcd4c2), UINT64_C(0xfe48ca671b9d0aff),
    UINT64_C(0x35e9b730a53316f5), UINT64_C(0xdd63ba3cf012c8c8), UINT64_C(0x76250203a07eb40a),
    UINT64_C(0x9eaf0f0ff55f6a37), UINT64_C(0xb270dd56afa8530b), UINT64_C(0x5afad05afa898d36),
    UINT64_C(0xf1bc6865aae5f1f4), UINT64_C(0x19366569ffc42fc9), UINT64_C(0xa803ccd71f0b838c),
    UINT64_C(0x4089c1db4a2a5db1), UINT64_C(0xebcf79e41a462173), UINT64_C(0x034574e84f67ff4e),
    UINT64_C(0x2f9aa6b11590c672), UINT64_C(0xc710abbd40b1184f), UINT64_C(0x6c56138210dd648d),
    UINT64_C(0x84dc1e8e45fcbab0), UINT64_C(0x9ce5efd47e4c2282), UINT64_C(0x746fe2d82b6dfcbf),
    UINT64_C(0xdf295ae77b01807d), UINT64_C(0x37a357eb2e205e40), UINT64_C(0x1b7c85b274d7677c),
    UINT64_C(0xf3f688be21f6b941), UINT64_C(0x58b03081719ac583), UINT64_C(0xb03a3d8d24bb1bbe),
    UINT64_C(0x010f9433c474b7fb), UINT64_C(0xe985993f915569c6), UINT64_C(0x42c32100c1391504),
    UINT64_C(0xaa492c0c9418cb39), UINT64_C(0x8696fe55ceeff205), UINT64_C(0x6e1cf3599bce2c38),
    UINT64_C(0xc55a4b66cba250fa), UINT64_C(0x2dd0466a9e838ec7), UINT64_C(0xf529a9d2bcc3609e),
    UINT64_C(0x1da3a4dee9e2bea3), UINT64_C(0xb6e51ce1b98ec261), UINT64_C(0x5e6f11edecaf1c5c),
    UINT64_C(0x72b0c3b4b6582560), UINT64_C(0x9a3aceb8e379fb5d), UINT64_C(0x317c7687b315879f),
    UINT64_C(0xd9f67b8be63459a2), UINT64_C(0x68c3d23506fbf5e7), UINT64_C(0x8049df3953da2bda),
    UINT64_C(0x2b0f670603b65718), UINT64_C(0xc3856a0a56978925), UINT64_C(0xef5ab8530c60b019),
    UINT64_C(0x07d0b55f59416e24), UINT64_C(0xac960d60092d12e6), UINT64_C(0x441c006c5c0cccdb),
    UINT64_C(0x5c25f13667bc54e9), UINT64_C(0xb4affc3a329d8ad4), UINT64_C(0x1fe9440562f1f616),
    UINT64_C(0xf763490937d0282b), UINT64_C(0xdbbc9b506d271117), UINT64_C(0x3336965c3806cf2a),
    UINT64_C(0x98702e63686ab3e8), UINT64_C(0x70fa236f3d4b6dd5), UINT64_C(0xc1cf8ad1dd84c190),
    UINT64_C(0x294587dd88a51fad), UINT64_C(0x82033fe2d8c9636f), UINT64_C(0x6a8932ee8de8bd52),
    UINT64_C(0x4656e0b7d71f846e), UINT64_C(0xaedcedbb823e5a53), UINT64_C(0x059a5584d2522691),
    UINT64_C(0xed1058888773f8ac) },
  { UINT64_C(0x0000000000000000), UINT64_C(0x4d624bbe73bbc94c), UINT64_C(0x9ac4977ce7779298),
    UINT64_C(0xd7a6dcc294cc5bd4), UINT64_C(0xa75181d261e13bb5), UINT64_C(0xea33ca6c125af2f9),
    UINT64_C(0x3d9516ae8696a92d), UINT64_C(0x70f75d10f52d6061), UINT64_C(0xdc7bac8f6ccc69ef),
    UINT64_C(0x9119e7311f77a0a3), UINT64_C(0x46bf3bf38bbbfb77), UINT64_C(0x0bdd704df800323b),
    UINT64_C(0x7b2a2d5d0d2d525a), UINT64_C(0x364866e37e969b16), UINT64_C(0xe1eeba21ea5ac0c2),
    UINT64_C(0xac8cf19f99e1098e), UINT64_C(0x2a2ff6357696cd5b), UINT64_C(0x674dbd8b052d0417),
    UINT64_C(0xb0eb614991e15fc3), UINT64_C(0xfd892af7e25a968f), UINT64_C(0x8d7e77e71777f6ee),
    UINT64_C(0xc01c3c5964cc3fa2), UINT64_C(0x17bae09bf0006476), UINT64_C(0x5ad8ab2583bbad3a),
    UINT64_C(0xf6545aba1a5aa4b4), UINT64_C(0xbb36110469e16df8), UINT64_C(0x6c90cdc6fd2d362c),
    UINT64_C(0x21f286788e96ff60), UINT64_C(0x5105db687bbb9f01), UINT64_C(0x1c6790d60800564d),
    UINT64_C(0xcbc14c149ccc0d99), UINT64_C(0x86a307aaef77c4d5), UINT64_C(0x545fec6aed2d9ab6),
    UINT64_C(0x193da7d49e9653fa), UINT64_C(0xce9b7b160a5a082e), UINT64_C(0x83f930a879e1c162),
    UINT64_C(0xf30e6db88ccca103), UINT64_C(0xbe6c2606ff77684f), UINT64_C(0x69cafac46bbb339b),
    UINT64_C(0x24a8b17a1800fad7), UINT64_C(0x882440e581e1f359), UINT64_C(0xc5460b5bf25a3a15),
    UINT64_C(0x12e0d799669661c1), UINT64_C(0x5f829c27152da88d), UINT64_C(0x2f75c137e000c8ec),
    UINT64_C(0x62178a8993bb01a0), UINT64_C(0xb5b1564b07775a74), UINT64_C(0xf8d31df574cc9338),
    UINT64_C(0x7e701a5f9bbb57ed), UINT64_C(0x331251e1e8009ea1), UINT64_C(0xe4b48d237cccc575),
    UINT64_C(0xa9d6c69d0f770c39), UINT64_C(0xd9219b8dfa5a6c58), UINT64_C(0x9443d03389e1a514),
    UINT64_C(0x43e50cf11d2dfec0), UINT64_C(0x0e87474f6e96378c), UINT64_C(0xa20bb6d0f7773e02),
    UINT64_C(0xef69fd6e84ccf74e), UINT64_C(0x38cf21ac1000ac9a), UINT64_C(0x75ad6a1263bb65d6),
    UINT64_C(0x055a3702969605b7), UINT64_C(0x48387cbce52dccfb), UINT64_C(0x9f9ea07e71e1972f),
    UINT64_C(0xd2fcebc0025a5e63), UINT64_C(0xa8bfd8d5da5b356c), UINT64_C(0xe5dd936ba9e0fc20),
    UINT64_C(0x327b4fa93d2ca7f4), UINT64_C(0x7f1904174e976eb8), UINT64_C(0x0fee5907bbba0ed9),
    UINT64_C(0x428c12b9c801c795), UINT64_C(0x952ace7b5ccd9c41), UINT64_C(0xd84885c52f76550d),
    UINT64_C(0x74c4745ab6975c83), UINT64_C(0x39a63fe4c52c95cf), UINT64_C(0xee00e32651e0ce1b),
    UINT64_C(0xa362a898225b0757), UINT64_C(0xd395f588d7766736), UINT64_C(0x9ef7be36a4cdae7a),
    UINT64_C(0x495162f43001f5ae), UINT64_C(0x0433294a43ba3ce2), UINT64_C(0x82902ee0accdf837),
    UINT64_C(0xcff2655edf76317b), UINT64_C(0x1854b99c4bba6aaf), UINT64_C(0x5536f2223801a3e3),
    UINT64_C(0x25c1af32cd2cc382), UINT64_C(0x68a3e48cbe970ace), UINT64_C(0xbf05384e2a5b511a),
    UINT64_C(0xf26773f059e09856), UINT64_C(0x5eeb826fc00191d8), UINT64_C(0x1389c9d1b3ba5894),
    UINT64_C(0xc42f151327760340), UINT64_C(0x894d5ead54cdca0c), UINT64_C(0xf9ba03bda1e0aa6d),
    UINT64_C(0xb4d84803d25b6321), UINT64_C(0x637e94c1469738f5), UINT64_C(0x2e1cdf7f352cf1b9),
    UINT64_C(0xfce034bf3776afda), UINT64_C(0xb1827f0144cd6696), UINT64_C(0x6624a3c3d0013d42),
    UINT64_C(0x2b46e87da3baf40e), UINT64_C(0x5bb1b56d5697946f), UINT64_C(0x16d3fed3252c5d23),
    UINT64_C(0xc1752211b1e006f7), UINT64_C(0x8c1769afc25bcfbb), UINT64_C(0x209b98305bbac635),
    UINT64_C(0x6df9d38e28010f79), UINT64_C(0xba5f0f4cbccd54ad), UINT64_C(0xf73d44f2cf769de1),
    UINT64_C(0x87ca19e23a5bfd80), UINT64_C(0xcaa8525c49e034cc), UINT64_C(0x1d0e8e9edd2c6f18),
    UINT64_C(0x506cc520ae97a654), UINT64_C(0xd6cfc28a41e06281), UINT64_C(0x9bad8934325babcd),
    UINT64_C(0x4c0b55f6a697f019), UINT64_C(0x01691e48d52c3955), UINT64_C(0x719e435820015934),
    UINT64_C(0x3cfc08e653ba9078), UINT64_C(0xeb5ad424c776cbac), UINT64_C(0xa6389f9ab4cd02e0),
    UINT64_C(0x0ab46e052d2c0b6e), UINT64_C(0x47d625bb5e97c222), UINT64_C(0x9070f979ca5b99f6),
    UINT64_C(0xdd12b2c7b9e050ba), UINT64_C(0xade5efd74ccd30db), UINT64_C(0xe087a4693f76f997),
    UINT64_C(0x372178ababbaa243), UINT64_C(0x7a433315d8016b0f), UINT64_C(0xc3a71e801bb8745d),
    UINT64_C(0x8ec5553e6803bd11), UINT64_C(0x596389fcfccfe6c5), UINT64_C(0x1401c2428f742f89),
    UINT64_C(0x64f69f527a594fe8), UINT64_C(0x2994d4ec09e286a4), UINT64_C(0xfe32082e9d2edd70),
    UINT64_C(0xb3504390ee95143c), UINT64_C(0x1fdcb20f77741db2), UINT64_C(0x52bef9b104cfd4fe),
    UINT64_C(0x8518257390038f2a), UINT64_C(0xc87a6ecde3b84666), UINT64_C(0xb88d33dd16952607),
    UINT64_C(0xf5ef7863652eef4b), UINT64_C(0x2249a4a1f1e2b49f), UINT64_C(0x6f2bef1f82597dd3),
    UINT64_C(0xe988e8b56d2eb906), UINT64_C(0xa4eaa30b1e95704a), UINT64_C(0x734c7fc98a592b9e),
    UINT64_C(0x3e2e3477f9e2e2d2), UINT64_C(0x4ed969670ccf82b3), UINT64_C(0x03bb22d97f744bff),
    UINT64_C(0xd41dfe1bebb8102b), UINT64_C(0x997fb5a59803d967), UINT64_C(0x35f3443a01e2d0e9),
    UINT64_C(0x78910f84725919a5), UINT64_C(0xaf37d346e6954271), UINT64_C(0xe25598f8952e8b3d),
    UINT64_C(0x92a2c5e86003eb5c), UINT64_C(0xdfc08e5613b82210), UINT64_C(0x08665294877479c4),
    UINT64_C(0x4504192af4cfb088), UINT64_C(0x97f8f2eaf695eeeb), UINT64_C(0xda9ab954852e27a7),
    UINT64_C(0x0d3c659611e27c73), UINT64_C(0x405e2e286259b53f), UINT64_C(0x30a973389774d55e),
    UINT64_C(0x7dcb3886e4cf1c12), UINT64_C(0xaa6de444700347c6), UINT64_C(0xe70faffa03b88e8a),
    UINT64_C(0x4b835e659a598704), UINT64_C(0x06e115dbe9e24e48), UINT64_C(0xd147c9197d2e159c),
    UINT64_C(0x9c2582a70e95dcd0), UINT64_C(0xecd2dfb7fbb8bcb1), UINT64_C(0xa1b09409880375fd),
    UINT64_C(0x761648cb1ccf2e29), UINT64_C(0x3b7403756f74e765), UINT64_C(0xbdd704df800323b0),
    UINT64_C(0xf0b54f61f3b8eafc), UINT64_C(0x271393a36774b128), UINT64_C(0x6a71d81d14cf7864),
    UINT64_C(0x1a86850de1e21805), UINT64_C(0x57e4ceb39259d149), UINT64_C(0x8042127106958a9d),
    UINT64_C(0xcd2059cf752e43d1), UINT64_C(0x61aca850eccf4a5f), UINT64_C(0x2ccee3ee9f748313),
    UINT64_C(0xfb683f2c0bb8d8c7), UINT64_C(0xb60a74927803118b), UINT64_C(0xc6fd29828d2e71ea),
    UINT64_C(0x8b9f623cfe95b8a6), UINT64_C(0x5c39befe6a59e372), UINT64_C(0x115bf54019e22a3e),
    UINT64_C(0x6b18c655c1e34131), UINT64_C(0x267a8debb258887d), UINT64_C(0xf1dc51292694d3a9),
    UINT64_C(0xbcbe1a97552f1ae5), UINT64_C(0xcc494787a0027a84), UINT64_C(0x812b0c39d3b9b3c8),
    UINT64_C(0x568dd0fb4775e81c), UINT64_C(0x1bef9b4534ce2150), UINT64_C(0xb7636adaad2f28de),
    UINT64_C(0xfa012164de94e192), UINT64_C(0x2da7fda64a58ba46), UINT64_C(0x60c5b61839e3730a),
    UINT64_C(0x1032eb08ccce136b), UINT64_C(0x5d50a0b6bf75da27), UINT64_C(0x8af67c742bb981f3),
    UINT64_C(0xc79437ca580248bf), UINT64_C(0x41373060b7758c6a), UINT64_C(0x0c557bdec4ce4526),
    UINT64_C(0xdbf3a71c50021ef2), UINT64_C(0x9691eca223b9d7be), UINT64_C(0xe666b1b2d694b7df),
    UINT64_C(0xab04fa0ca52f7e93), UINT64_C(0x7ca226ce31e32547), UINT64_C(0x31c06d704258ec0b),
    UINT64_C(0x9d4c9cefdbb9e585), UINT64_C(0xd02ed751a8022cc9), UINT64_C(0x07880b933cce771d),
    UINT64_C(0x4aea402d4f75be51), UINT64_C(0x3a1d1d3dba58de30), UINT64_C(0x777f5683c9e3177c),
    UINT64_C(0xa0d98a415d2f4ca8), UINT64_C(0xedbbc1ff2e9485e4), UINT64_C(0x3f472a3f2ccedb87),
    UINT64_C(0x722561815f7512cb), UINT64_C(0xa583bd43cbb9491f), UINT64_C(0xe8e1f6fdb8028053),
    UINT64_C(0x9816abed4d2fe032), UINT64_C(0xd574e0533e94297e), UINT64_C(0x02d23c91aa5872aa),
    UINT64_C(0x4fb0772fd9e3bbe6), UINT64_C(0xe33c86b04002b268), UINT64_C(0xae5ecd0e33b97b24),
    UINT64_C(0x79f811cca77520f0), UINT64_C(0x349a5a72d4cee9bc), UINT64_C(0x446d076221e389dd),
    UINT64_C(0x090f4cdc52584091), UINT64_C(0xdea9901ec6941b45), UINT64_C(0x93cbdba0b52fd209),
    UINT64_C(0x1568dc0a5a5816dc), UINT64_C(0x580a97b429e3df90), UINT64_C(0x8fac4b76bd2f8444),
    UINT64_C(0xc2ce00c8ce944d08), UINT64_C(0xb2395dd83bb92d69), UINT64_C(0xff5b16664802e425),
    UINT64_C(0x28fdcaa4dccebff1), UINT64_C(0x659f811aaf7576bd), UINT64_C(0xc913708536947f33),
    UINT64_C(0x84713b3b452fb67f), UINT64_C(0x53d7e7f9d1e3edab), UINT64_C(0x1eb5ac47a25824e7),
    UINT64_C(0x6e42f15757754486), UINT64_C(0x2320bae924ce8dca), UINT64_C(0xf486662bb002d61e),
    UINT64_C(0xb9e42d95c3b91f52) },
  { UINT64_C(0x0000000000000000), UINT64_C(0x1596922b987ef63f), UINT64_C(0x2b2d245730fdec7e),
    UINT64_C(0x3ebbb67ca8831a41), UINT64_C(0x565a48ae61fbd8fc), UINT64_C(0x43ccda85f9852ec3),
    UINT64_C(0x7d776cf951063482), UINT64_C(0x68e1fed2c978c2bd), UINT64_C(0xacb4915cc3f7b1f8),
    UINT64_C(0xb92203775b8947c7), UINT64_C(0x8799b50bf30a5d86), UINT64_C(0x920f27206b74abb9),
    UINT64_C(0xfaeed9f2a20c6904), UINT64_C(0xef784bd93a729f3b), UINT64_C(0xd1c3fda592f1857a),
    UINT64_C(0xc4556f8e0a8f7345), UINT64_C(0xcbb18d9228e17d75), UINT64_C(0xde271fb9b09f8b4a),
    UINT64_C(0xe09ca9c5181c910b), UINT64_C(0xf50a3bee80626734), UINT64_C(0x9debc53c491aa589),
    UINT64_C(0x887d5717d16453b6), UINT64_C(0xb6c6e16b79e749f7), UINT64_C(0xa3507340e199bfc8),
    UINT64_C(0x67051cceeb16cc8d), UINT64_C(0x72938ee573683ab2), UINT64_C(0x4c283899dbeb20f3),
    UINT64_C(0x59beaab24395d6cc), UINT64_C(0x315f54608aed1471), UINT64_C(0x24c9c64b1293e24e),
    UINT64_C(0x1a727037ba10f80f), UINT64_C(0x0fe4e21c226e0e30), UINT64_C(0x05bbb40ffecce46f),
    UINT64_C(0x102d262466b21250), UINT64_C(0x2e969058ce310811), UINT64_C(0x3b000273564ffe2e),
    UINT64_C(0x53e1fca19f373c93), UINT64_C(0x46776e8a0749caac), UINT64_C(0x78ccd8f6afcad0ed),
    UINT64_C(0x6d5a4add37b426d2), UINT64_C(0xa90f25533d3b5597), UINT64_C(0xbc99b778a545a3a8),
    UINT64_C(0x822201040dc6b9e9), UINT64_C(0x97b4932f95b84fd6), UINT64_C(0xff556dfd5cc08d6b),
    UINT64_C(0xeac3ffd6c4be7b54), UINT64_C(0xd47849aa6c3d6115), UINT64_C(0xc1eedb81f443972a),
    UINT64_C(0xce0a399dd62d991a), UINT64_C(0xdb9cabb64e536f25), UINT64_C(0xe5271dcae6d07564),
    UINT64_C(0xf0b18fe17eae835b), UINT64_C(0x98507133b7d641e6), UINT64_C(0x8dc6e3182fa8b7d9),
    UINT64_C(0xb37d5564872bad98), UINT64_C(0xa6ebc74f1f555ba7), UINT64_C(0x62bea8c115da28e2),
    UINT64_C(0x77283aea8da4dedd), UINT64_C(0x49938c962527c49c), UINT64_C(0x5c051ebdbd5932a3),
    UINT64_C(0x34e4e06f7421f01e), UINT64_C(0x21727244ec5f0621), UINT64_C(0x1fc9c43844dc1c60),
    UINT64_C(0x0a5f5613dca2ea5f), UINT64_C(0x0b77681ffd99c8de), UINT64_C(0x1ee1fa3465e73ee1),
    UINT64_C(0x205a4c48cd6424a0), UINT64_C(0x35ccde63551ad29f), UINT64_C(0x5d2d20b19c621022),
    UINT64_C(0x48bbb29a041ce61d), UINT64_C(0x760004e6ac9ffc5c), UINT64_C(0x639696cd34e10a63),
    UINT64_C(0xa7c3f9433e6e7926), UINT64_C(0xb2556b68a6108f19), UINT64_C(0x8ceedd140e939558),
    UINT64_C(0x99784f3f96ed6367), UINT64_C(0xf199b1ed5f95a1da), UINT64_C(0xe40f23c6c7eb57e5),
    UINT64_C(0xdab495ba6f684da4), UINT64_C(0xcf220791f716bb9b), UINT64_C(0xc0c6e58dd578b5ab),
    UINT64_C(0xd55077a64d064394), UINT64_C(0xebebc1dae58559d5), UINT64_C(0xfe7d53f17dfbafea),
    UINT64_C(0x969cad23b4836d57), UINT64_C(0x830a3f082cfd9b68), UINT64_C(0xbdb18974847e8129),
    UINT64_C(0xa8271b5f1c007716), UINT64_C(0x6c7274d1168f0453), UINT64_C(0x79e4e6fa8ef1f26c),
    UINT64_C(0x475f50862672e82d), UINT64_C(0x52c9c2adbe0c1e12), UINT64_C(0x3a283c7f7774dcaf),
    UINT64_C(0x2fbeae54ef0a2a90), UINT64_C(0x11051828478930d1), UINT64_C(0x04938a03dff7c6ee),
    UINT64_C(0x0eccdc1003552cb1), UINT64_C(0x1b5a4e3b9b2bda8e), UINT64_C(0x25e1f84733a8c0cf),
    UINT64_C(0x30776a6cabd636f0), UINT64_C(0x589694be62aef44d), UINT64_C(0x4d000695fad00272),
    UINT64_C(0x73bbb0e952531833), UINT64_C(0x662d22c2ca2dee0c), UINT64_C(0xa2784d4cc0a29d49),
    UINT64_C(0xb7eedf6758dc6b76), UINT64_C(0x8955691bf05f7137), UINT64_C(0x9cc3fb3068218708),
    UINT64_C(0xf42205e2a15945b5), UINT64_C(0xe1b497c93927b38a), UINT64_C(0xdf0f21b591a4a9cb),
    UINT64_C(0xca99b39e09da5ff4), UINT64_C(0xc57d51822bb451c4), UINT64_C(0xd0ebc3a9b3caa7fb),
    UINT64_C(0xee5075d51b49bdba), UINT64_C(0xfbc6e7fe83374b85), UINT64_C(0x9327192c4a4f8938),
    UINT64_C(0x86b18b07d2317f07), UINT64_C(0xb80a3d7b7ab26546), UINT64_C(0xad9caf50e2cc9379),
    UINT64_C(0x69c9c0dee843e03c), UINT64_C(0x7c5f52f5703d1603), UINT64_C(0x42e4e489d8be0c42),
    UINT64_C(0x577276a240c0fa7d), UINT64_C(0x3f93887089b838c0), UINT64_C(0x2a051a5b11c6ceff),
    UINT64_C(0x14beac27b945d4be), UINT64_C(0x01283e0c213b2281), UINT64_C(0x16eed03ffb3391bc),
    UINT64_C(0x03784214634d6783), UINT64_C(0x3dc3f468cbce7dc2), UINT64_C(0x2855664353b08bfd),
    UINT64_C(0x40b498919ac84940), UINT64_C(0x55220aba02b6bf7f), UINT64_C(0x6b99bcc6aa35a53e),
    UINT64_C(0x7e0f2eed324b5301), UINT64_C(0xba5a416338c42044), UINT64_C(0xafccd348a0bad67b),
    UINT64_C(0x917765340839cc3a), UINT64_C(0x84e1f71f90473a05), UINT64_C(0xec0009cd593ff8b8),
    UINT64_C(0xf9969be6c1410e87), UINT64_C(0xc72d2d9a69c214c6), UINT64_C(0xd2bbbfb1f1bce2f9),
    UINT64_C(0xdd5f5dadd3d2ecc9), UINT64_C(0xc8c9cf864bac1af6), UINT64_C(0xf67279fae32f00b7),
    UINT64_C(0xe3e4ebd17b51f688), UINT64_C(0x8b051503b2293435), UINT64_C(0x9e9387282a57c20a),
    UINT64_C(0xa028315482d4d84b), UINT64_C(0xb5bea37f1aaa2e74), UINT64_C(0x71ebccf110255d31),
    UINT64_C(0x647d5eda885bab0e), UINT64_C(0x5ac6e8a620d8b14f), UINT64_C(0x4f507a8db8a64770),
    UINT64_C(0x27b1845f71de85cd), UINT64_C(0x32271674e9a073f2), UINT64_C(0x0c9ca008412369b3),
    UINT64_C(0x190a3223d95d9f8c), UINT64_C(0x1355643005ff75d3), UINT64_C(0x06c3f61b9d8183ec),
    UINT64_C(0x38784067350299ad), UINT64_C(0x2deed24cad7c6f92), UINT64_C(0x450f2c9e6404ad2f),
    UINT64_C(0x5099beb5fc7a5b10), UINT64_C(0x6e2208c954f94151), UINT64_C(0x7bb49ae2cc87b76e),
    UINT64_C(0xbfe1f56cc608c42b), UINT64_C(0xaa7767475e763214), UINT64_C(0x94ccd13bf6f52855),
    UINT64_C(0x815a43106e8bde6a), UINT64_C(0xe9bbbdc2a7f31cd7), UINT64_C(0xfc2d2fe93f8deae8),
    UINT64_C(0xc2969995970ef0a9), UINT64_C(0xd7000bbe0f700696), UINT64_C(0xd8e4e9a22d1e08a6),
    UINT64_C(0xcd727b89b560fe99), UINT64_C(0xf3c9cdf51de3e4d8), UINT64_C(0xe65f5fde859d12e7),
    UINT64_C(0x8ebea10c4ce5d05a), UINT64_C(0x9b283327d49b2665), UINT64_C(0xa593855b7c183c24),
    UINT64_C(0xb0051770e466ca1b), UINT64_C(0x745078feeee9b95e), UINT64_C(0x61c6ead576974f61),
    UINT64_C(0x5f7d5ca9de145520), UINT64_C(0x4aebce82466aa31f), UINT64_C(0x220a30508f1261a2),
    UINT64_C(0x379ca27b176c979d), UINT64_C(0x09271407bfef8ddc), UINT64_C(0x1cb1862c27917be3),
    UINT64_C(0x1d99b82006aa5962), UINT64_C(0x080f2a0b9ed4af5d), UINT64_C(0x36b49c773657b51c),
    UINT64_C(0x23220e5cae294323), UINT64_C(0x4bc3f08e6751819e), UINT64_C(0x5e5562a5ff2f77a1),
    UINT64_C(0x60eed4d957ac6de0), UINT64_C(0x757846f2cfd29bdf), UINT64_C(0xb12d297cc55de89a),
    UINT64_C(0xa4bbbb575d231ea5), UINT64_C(0x9a000d2bf5a004e4), UINT64_C(0x8f969f006ddef2db),
    UINT64_C(0xe77761d2a4a63066), UINT64_C(0xf2e1f3f93cd8c659), UINT64_C(0xcc5a4585945bdc18),
    UINT64_C(0xd9ccd7ae0c252a27), UINT64_C(0xd62835b22e4b2417), UINT64_C(0xc3bea799b635d228),
    UINT64_C(0xfd0511e51eb6c869), UINT64_C(0xe89383ce86c83e56), UINT64_C(0x80727d1c4fb0fceb),
    UINT64_C(0x95e4ef37d7ce0ad4), UINT64_C(0xab5f594b7f4d1095), UINT64_C(0xbec9cb60e733e6aa),
    UINT64_C(0x7a9ca4eeedbc95ef), UINT64_C(0x6f0a36c575c263d0), UINT64_C(0x51b180b9dd417991),
    UINT64_C(0x44271292453f8fae), UINT64_C(0x2cc6ec408c474d13), UINT64_C(0x39507e6b1439bb2c),
    UINT64_C(0x07ebc817bcbaa16d), UINT64_C(0x127d5a3c24c45752), UINT64_C(0x18220c2ff866bd0d),
    UINT64_C(0x0db49e0460184b32), UINT64_C(0x330f2878c89b5173), UINT64_C(0x2699ba5350e5a74c),
    UINT64_C(0x4e784481999d65f1), UINT64_C(0x5beed6aa01e393ce), UINT64_C(0x655560d6a960898f),
    UINT64_C(0x70c3f2fd311e7fb0), UINT64_C(0xb4969d733b910cf5), UINT64_C(0xa1000f58a3effaca),
    UINT64_C(0x9fbbb9240b6ce08b), UINT64_C(0x8a2d2b0f931216b4), UINT64_C(0xe2ccd5dd5a6ad409),
    UINT64_C(0xf75a47f6c2142236), UINT64_C(0xc9e1f18a6a973877), UINT64_C(0xdc7763a1f2e9ce48),
    UINT64_C(0xd39381bdd087c078), UINT64_C(0xc605139648f93647), UINT64_C(0xf8bea5eae07a2c06),
    UINT64_C(0xed2837c17804da39), UINT64_C(0x85c9c913b17c1884), UINT64_C(0x905f5b382902eebb),
    UINT64_C(0xaee4ed448181f4fa), UINT64_C(0xbb727f6f19ff02c5), UINT64_C(0x7f2710e113707180),
    UINT64_C(0x6ab182ca8b0e87bf), UINT64_C(0x540a34b6238d9dfe), UINT64_C(0x419ca69dbbf36bc1),
    UINT64_C(0x297d584f728ba97c), UINT64_C(0x3cebca64eaf55f43), UINT64_C(0x02507c1842764502),
    UINT64_C(0x17c6ee33da08b33d) },
  { UINT64_C(0x0000000000000000), UINT64_C(0x2ddda07ff6672378), UINT64_C(0x5bbb40ffecce46f0),
    UINT64_C(0x7666e0801aa96588), UINT64_C(0xb77681ffd99c8de0), UINT64_C(0x9aab21802ffbae98),
    UINT64_C(0xeccdc1003552cb10), UINT64_C(0xc110617fc335e868), UINT64_C(0xfc35acd41c370545),
    UINT64_C(0xd1e80cabea50263d), UINT64_C(0xa78eec2bf0f943b5), UINT64_C(0x8a534c54069e60cd),
    UINT64_C(0x4b432d2bc5ab88a5), UINT64_C(0x669e8d5433ccabdd), UINT64_C(0x10f86dd42965ce55),
    UINT64_C(0x3d25cdabdf02ed2d), UINT64_C(0x6ab3f6839760140f), UINT64_C(0x476e56fc61073777),
    UINT64_C(0x3108b67c7bae52ff), UINT64_C(0x1cd516038dc97187), UINT64_C(0xddc5777c4efc99ef),
    UINT64_C(0xf018d703b89bba97), UINT64_C(0x867e3783a232df1f), UINT64_C(0xaba397fc5455fc67),
    UINT64_C(0x96865a578b57114a), UINT64_C(0xbb5bfa287d303232), UINT64_C(0xcd3d1aa8679957ba),
    UINT64_C(0xe0e0bad791fe74c2), UINT64_C(0x21f0dba852cb9caa), UINT64_C(0x0c2d7bd7a4acbfd2),
    UINT64_C(0x7a4b9b57be05da5a), UINT64_C(0x57963b284862f922), UINT64_C(0xd567ed072ec0281e),
    UINT64_C(0xf8ba4d78d8a70b66), UINT64_C(0x8edcadf8c20e6eee), UINT64_C(0xa3010d8734694d96),
    UINT64_C(0x62116cf8f75ca5fe), UINT64_C(0x4fcccc87013b8686), UINT64_C(0x39aa2c071b92e30e),
    UINT64_C(0x14778c78edf5c076), UINT64_C(0x295241d332f72d5b), UINT64_C(0x048fe1acc4900e23),
    UINT64_C(0x72e9012cde396bab), UINT64_C(0x5f34a153285e48d3), UINT64_C(0x9e24c02ceb6ba0bb),
    UINT64_C(0xb3f960531d0c83c3), UINT64_C(0xc59f80d307a5e64b), UINT64_C(0xe84220acf1c2c533),
    UINT64_C(0xbfd41b84b9a03c11), UINT64_C(0x9209bbfb4fc71f69), UINT64_C(0xe46f5b7b556e7ae1),
    UINT64_C(0xc9b2fb04a3095999), UINT64_C(0x08a29a7b603cb1f1), UINT64_C(0x257f3a04965b9289),
    UINT64_C(0x5319da848cf2f701), UINT64_C(0x7ec47afb7a95d479), UINT64_C(0x43e1b750a5973954),
    UINT64_C(0x6e3c172f53f01a2c), UINT64_C(0x185af7af49597fa4), UINT64_C(0x358757d0bf3e5cdc),
    UINT64_C(0xf49736af7c0bb4b4), UINT64_C(0xd94a96d08a6c97cc), UINT64_C(0xaf2c765090c5f244),
    UINT64_C(0x82f1d62f66a2d13c), UINT64_C(0x38177525f28e4eb9), UINT64_C(0x15cad55a04e96dc1),
    UINT64_C(0x63ac35da1e400849), UINT64_C(0x4e7195a5e8272b31), UINT64_C(0x8f61f4da2b12c359),
    UINT64_C(0xa2bc54a5dd75e021), UINT64_C(0xd4dab425c7dc85a9), UINT64_C(0xf907145a31bba6d1),
    UINT64_C(0xc422d9f1eeb94bfc), UINT64_C(0xe9ff798e18de6884), UINT64_C(0x9f99990e02770d0c),
    UINT64_C(0xb2443971f4102e74), UINT64_C(0x7354580e3725c61c), UINT64_C(0x5e89f871c142e564),
    UINT64_C(0x28ef18f1dbeb80ec), UINT64_C(0x0532b88e2d8ca394), UINT64_C(0x52a483a665ee5ab6),
    UINT64_C(0x7f7923d9938979ce), UINT64_C(0x091fc35989201c46), UINT64_C(0x24c263267f473f3e),
    UINT64_C(0xe5d20259bc72d756), UINT64_C(0xc80fa2264a15f42e), UINT64_C(0xbe6942a650bc91a6),
    UINT64_C(0x93b4e2d9a6dbb2de), UINT64_C(0xae912f7279d95ff3), UINT64_C(0x834c8f0d8fbe7c8b),
    UINT64_C(0xf52a6f8d95171903), UINT64_C(0xd8f7cff263703a7b), UINT64_C(0x19e7ae8da045d213),
    UINT64_C(0x343a0ef25622f16b), UINT64_C(0x425cee724c8b94e3), UINT64_C(0x6f814e0dbaecb79b),
    UINT64_C(0xed709822dc4e66a7), UINT64_C(0xc0ad385d2a2945df), UINT64_C(0xb6cbd8dd30802057),
    UINT64_C(0x9b1678a2c6e7032f), UINT64_C(0x5a0619dd05d2eb47), UINT64_C(0x77dbb9a2f3b5c83f),
    UINT64_C(0x01bd5922e91cadb7), UINT64_C(0x2c60f95d1f7b8ecf), UINT64_C(0x114534f6c07963e2),
    UINT64_C(0x3c989489361e409a), UINT64_C(0x4afe74092cb72512), UINT64_C(0x6723d476dad0066a),
    UINT64_C(0xa633b50919e5ee02), UINT64_C(0x8bee1576ef82cd7a), UINT64_C(0xfd88f5f6f52ba8f2),
    UINT64_C(0xd0555589034c8b8a), UINT64_C(0x87c36ea14b2e72a8), UINT64_C(0xaa1ecedebd4951d0),
    UINT64_C(0xdc782e5ea7e03458), UINT64_C(0xf1a58e2151871720), UINT64_C(0x30b5ef5e92b2ff48),
    UINT64_C(0x1d684f2164d5dc30), UINT64_C(0x6b0eafa17e7cb9b8), UINT64_C(0x46d30fde881b9ac0),
    UINT64_C(0x7bf6c275571977ed), UINT64_C(0x562b620aa17e5495), UINT64_C(0x204d828abbd7311d),
    UINT64_C(0x0d9022f54db01265), UINT64_C(0xcc80438a8e85fa0d), UINT64_C(0xe15de3f578e2d975),
    UINT64_C(0x973b0375624bbcfd), UINT64_C(0xbae6a30a942c9f85), UINT64_C(0x702eea4be51c9d72),
    UINT64_C(0x5df34a34137bbe0a), UINT64_C(0x2b95aab409d2db82), UINT64_C(0x06480acbffb5f8fa),
    UINT64_C(0xc7586bb43c801092), UINT64_C(0xea85cbcbcae733ea), UINT64_C(0x9ce32b4bd04e5662),
    UINT64_C(0xb13e8b342629751a), UINT64_C(0x8c1b469ff92b9837), UINT64_C(0xa1c6e6e00f4cbb4f),
    UINT64_C(0xd7a0066015e5dec7), UINT64_C(0xfa7da61fe382fdbf), UINT64_C(0x3b6dc76020b715d7),
    UINT64_C(0x16b0671fd6d036af), UINT64_C(0x60d6879fcc795327), UINT64_C(0x4d0b27e03a1e705f),
    UINT64_C(0x1a9d1cc8727c897d), UINT64_C(0x3740bcb7841baa05), UINT64_C(0x41265c379eb2cf8d),
    UINT64_C(0x6cfbfc4868d5ecf5), UINT64_C(0xadeb9d37abe0049d), UINT64_C(0x80363d485d8727e5),
    UINT64_C(0xf650ddc8472e426d), UINT64_C(0xdb8d7db7b1496115), UINT64_C(0xe6a8b01c6e4b8c38),
    UINT64_C(0xcb751063982caf40), UINT64_C(0xbd13f0e38285cac8), UINT64_C(0x90ce509c74e2e9b0),
    UINT64_C(0x51de31e3b7d701d8), UINT64_C(0x7c03919c41b022a0), UINT64_C(0x0a65711c5b194728),
    UINT64_C(0x27b8d163ad7e6450), UINT64_C(0xa549074ccbdcb56c), UINT64_C(0x8894a7333dbb9614),
    UINT64_C(0xfef247b32712f39c), UINT64_C(0xd32fe7ccd175d0e4), UINT64_C(0x123f86b31240388c),
    UINT64_C(0x3fe226cce4271bf4), UINT64_C(0x4984c64cfe8e7e7c), UINT64_C(0x6459663308e95d04),
    UINT64_C(0x597cab98d7ebb029), UINT64_C(0x74a10be7218c9351), UINT64_C(0x02c7eb673b25f6d9),
    UINT64_C(0x2f1a4b18cd42d5a1), UINT64_C(0xee0a2a670e773dc9), UINT64_C(0xc3d78a18f8101eb1),
    UINT64_C(0xb5b16a98e2b97b39), UINT64_C(0x986ccae714de5841), UINT64_C(0xcffaf1cf5cbca163),
    UINT64_C(0xe22751b0aadb821b), UINT64_C(0x9441b130b072e793), UINT64_C(0xb99c114f4615c4eb),
    UINT64_C(0x788c703085202c83), UINT64_C(0x5551d04f73470ffb), UINT64_C(0x233730cf69ee6a73),
    UINT64_C(0x0eea90b09f89490b), UINT64_C(0x33cf5d1b408ba426), UINT64_C(0x1e12fd64b6ec875e),
    UINT64_C(0x68741de4ac45e2d6), UINT64_C(0x45a9bd9b5a22c1ae), UINT64_C(0x84b9dce4991729c6),
    UINT64_C(0xa9647c9b6f700abe), UINT64_C(0xdf029c1b75d96f36), UINT64_C(0xf2df3c6483be4c4e),
    UINT64_C(0x48399f6e1792d3cb), UINT64_C(0x65e43f11e1f5f0b3), UINT64_C(0x1382df91fb5c953b),
    UINT64_C(0x3e5f7fee0d3bb643), UINT64_C(0xff4f1e91ce0e5e2b), UINT64_C(0xd292beee38697d53),
    UINT64_C(0xa4f45e6e22c018db), UINT64_C(0x8929fe11d4a73ba3), UINT64_C(0xb40c33ba0ba5d68e),
    UINT64_C(0x99d193c5fdc2f5f6), UINT64_C(0xefb77345e76b907e), UINT64_C(0xc26ad33a110cb306),
    UINT64_C(0x037ab245d2395b6e), UINT64_C(0x2ea7123a245e7816), UINT64_C(0x58c1f2ba3ef71d9e),
    UINT64_C(0x751c52c5c8903ee6), UINT64_C(0x228a69ed80f2c7c4), UINT64_C(0x0f57c9927695e4bc),
    UINT64_C(0x793129126c3c8134), UINT64_C(0x54ec896d9a5ba24c), UINT64_C(0x95fce812596e4a24),
    UINT64_C(0xb821486daf09695c), UINT64_C(0xce47a8edb5a00cd4), UINT64_C(0xe39a089243c72fac),
    UINT64_C(0xdebfc5399cc5c281), UINT64_C(0xf36265466aa2e1f9), UINT64_C(0x850485c6700b8471),
    UINT64_C(0xa8d925b9866ca709), UINT64_C(0x69c944c645594f61), UINT64_C(0x4414e4b9b33e6c19),
    UINT64_C(0x32720439a9970991), UINT64_C(0x1fafa4465ff02ae9), UINT64_C(0x9d5e72693952fbd5),
    UINT64_C(0xb083d216cf35d8ad), UINT64_C(0xc6e53296d59cbd25), UINT64_C(0xeb3892e923fb9e5d),
    UINT64_C(0x2a28f396e0ce7635), UINT64_C(0x07f553e916a9554d), UINT64_C(0x7193b3690c0030c5),
    UINT64_C(0x5c4e1316fa6713bd), UINT64_C(0x616bdebd2565fe90), UINT64_C(0x4cb67ec2d302dde8),
    UINT64_C(0x3ad09e42c9abb860), UINT64_C(0x170d3e3d3fcc9b18), UINT64_C(0xd61d5f42fcf97370),
    UINT64_C(0xfbc0ff3d0a9e5008), UINT64_C(0x8da61fbd10373580), UINT64_C(0xa07bbfc2e65016f8),
    UINT64_C(0xf7ed84eaae32efda), UINT64_C(0xda3024955855cca2), UINT64_C(0xac56c41542fca92a),
    UINT64_C(0x818b646ab49b8a52), UINT64_C(0x409b051577ae623a), UINT64_C(0x6d46a56a81c94142),
    UINT64_C(0x1b2045ea9b6024ca), UINT64_C(0x36fde5956d0707b2), UINT64_C(0x0bd8283eb205ea9f),
    UINT64_C(0x260588414462c9e7), UINT64_C(0x506368c15ecbac6f), UINT64_C(0x7dbec8bea8ac8f17),
    UINT64_C(0xbcaea9c16b99677f), UINT64_C(0x917309be9dfe4407), UINT64_C(0xe715e93e8757218f),
    UINT64_C(0xcac84941713002f7) }
};

static inline uint64_t crc64_slow(const void* input, size_t nbytes)
{
  const unsigned char* data = (const unsigned char*)input;
  uint64_t cs = UINT64_C(0xffffffffffffffff);

  while (nbytes--)
  {
    uint32_t idx = ((uint32_t)(cs ^ *data++)) & 0xff;
    cs = crc64_table[3][idx] ^ (cs >> 8);
  }

  return cs ^ UINT64_C(0xffffffffffffffff);
}

// Loads an input 32-bit word in little-endian order from a big-endian machine.
static inline uint32_t crc64_load_le32_(const uint32_t* p)
{
#ifdef __ppc__
  // See: http://hardwarebug.org/2008/10/25/gcc-inline-asm-annoyance/
  uint32_t v;
  asm("lwbrx %0, %y1" : "=r"(v) : "Z"(*p));
  return v;
#else
  uint32_t w = *p;
  return ((((w)&0xff000000) >> 24) | (((w)&0x00ff0000) >> 8) | (((w)&0x0000ff00) << 8) |
    (((w)&0x000000ff) << 24));
#endif
}

// A parallel multiword interleaved algorithm with a word size of 4 bytes
// and a stride factor of 5.
static inline uint64_t crc64(const void* input, size_t nbytes)
{
  const unsigned char* data = (const unsigned char*)input;
  const unsigned char* end = data + nbytes;
  uint64_t cs[5] = { UINT64_C(0xffffffffffffffff), 0, 0, 0, 0 };

  // Process byte-by-byte until proper alignment is attained.
  // In the inner loop, we process 5 4-byte words (20 bytes in total)
  // per iteration. If the amount of data remaining is small,
  // then we also use the slow algorithm.
  while (data < end && ((((size_t)data) & 3) || (end - data < 20)))
  {
    uint32_t idx = ((uint32_t)(cs[0] ^ *data++)) & 0xff;
    cs[0] = crc64_table[3][idx] ^ (cs[0] >> 8);
  }

  if (data == end)
    return cs[0] ^ UINT64_C(0xffffffffffffffff);

  const uint32_t one = 1;
  bool big_endian = !(*((char*)(&one)));

  uint64_t cry = 0;
  uint32_t in[5];

  if (!big_endian)
  {
    for (unsigned i = 0; i < 5; ++i)
      in[i] = ((const uint32_t*)data)[i];
    data += 20;

    for (; end - data >= 20; data += 20)
    {
      cs[0] ^= cry;

      in[0] ^= (uint32_t)cs[0];
      cs[1] ^= cs[0] >> 32;
      cs[0] = crc64_interleaved_table[0][in[0] & 0xff];
      in[0] >>= 8;

      in[1] ^= (uint32_t)cs[1];
      cs[2] ^= cs[1] >> 32;
      cs[1] = crc64_interleaved_table[0][in[1] & 0xff];
      in[1] >>= 8;

      in[2] ^= (uint32_t)cs[2];
      cs[3] ^= cs[2] >> 32;
      cs[2] = crc64_interleaved_table[0][in[2] & 0xff];
      in[2] >>= 8;

      in[3] ^= (uint32_t)cs[3];
      cs[4] ^= cs[3] >> 32;
      cs[3] = crc64_interleaved_table[0][in[3] & 0xff];
      in[3] >>= 8;

      in[4] ^= (uint32_t)cs[4];
      cry = cs[4] >> 32;
      cs[4] = crc64_interleaved_table[0][in[4] & 0xff];
      in[4] >>= 8;

      for (unsigned b = 1; b < 3; ++b)
      {
        cs[0] ^= crc64_interleaved_table[b][in[0] & 0xff];
        in[0] >>= 8;

        cs[1] ^= crc64_interleaved_table[b][in[1] & 0xff];
        in[1] >>= 8;

        cs[2] ^= crc64_interleaved_table[b][in[2] & 0xff];
        in[2] >>= 8;

        cs[3] ^= crc64_interleaved_table[b][in[3] & 0xff];
        in[3] >>= 8;

        cs[4] ^= crc64_interleaved_table[b][in[4] & 0xff];
        in[4] >>= 8;
      }

      cs[0] ^= crc64_interleaved_table[3][in[0] & 0xff];
      in[0] = ((const uint32_t*)data)[0];

      cs[1] ^= crc64_interleaved_table[3][in[1] & 0xff];
      in[1] = ((const uint32_t*)data)[1];

      cs[2] ^= crc64_interleaved_table[3][in[2] & 0xff];
      in[2] = ((const uint32_t*)data)[2];

      cs[3] ^= crc64_interleaved_table[3][in[3] & 0xff];
      in[3] = ((const uint32_t*)data)[3];

      cs[4] ^= crc64_interleaved_table[3][in[4] & 0xff];
      in[4] = ((const uint32_t*)data)[4];
    }
  }
  else
  {
    for (unsigned i = 0; i < 5; ++i)
    {
      in[i] = crc64_load_le32_(&((const uint32_t*)data)[i]);
    }
    data += 20;

    for (; end - data >= 20; data += 20)
    {
      cs[0] ^= cry;

      in[0] ^= (uint32_t)cs[0];
      cs[1] ^= cs[0] >> 32;
      cs[0] = crc64_interleaved_table[0][in[0] & 0xff];
      in[0] >>= 8;

      in[1] ^= (uint32_t)cs[1];
      cs[2] ^= cs[1] >> 32;
      cs[1] = crc64_interleaved_table[0][in[1] & 0xff];
      in[1] >>= 8;

      in[2] ^= (uint32_t)cs[2];
      cs[3] ^= cs[2] >> 32;
      cs[2] = crc64_interleaved_table[0][in[2] & 0xff];
      in[2] >>= 8;

      in[3] ^= (uint32_t)cs[3];
      cs[4] ^= cs[3] >> 32;
      cs[3] = crc64_interleaved_table[0][in[3] & 0xff];
      in[3] >>= 8;

      in[4] ^= (uint32_t)cs[4];
      cry = cs[4] >> 32;
      cs[4] = crc64_interleaved_table[0][in[4] & 0xff];
      in[4] >>= 8;

      for (unsigned b = 1; b < 3; ++b)
      {
        cs[0] ^= crc64_interleaved_table[b][in[0] & 0xff];
        in[0] >>= 8;

        cs[1] ^= crc64_interleaved_table[b][in[1] & 0xff];
        in[1] >>= 8;

        cs[2] ^= crc64_interleaved_table[b][in[2] & 0xff];
        in[2] >>= 8;

        cs[3] ^= crc64_interleaved_table[b][in[3] & 0xff];
        in[3] >>= 8;

        cs[4] ^= crc64_interleaved_table[b][in[4] & 0xff];
        in[4] >>= 8;
      }

      cs[0] ^= crc64_interleaved_table[3][in[0] & 0xff];
      in[0] = crc64_load_le32_(&((const uint32_t*)data)[0]);

      cs[1] ^= crc64_interleaved_table[3][in[1] & 0xff];
      in[1] = crc64_load_le32_(&((const uint32_t*)data)[1]);

      cs[2] ^= crc64_interleaved_table[3][in[2] & 0xff];
      in[2] = crc64_load_le32_(&((const uint32_t*)data)[2]);

      cs[3] ^= crc64_interleaved_table[3][in[3] & 0xff];
      in[3] = crc64_load_le32_(&((const uint32_t*)data)[3]);

      cs[4] ^= crc64_interleaved_table[3][in[4] & 0xff];
      in[4] = crc64_load_le32_(&((const uint32_t*)data)[4]);
    }
  }

  cs[0] ^= cry;

  for (unsigned i = 0; i < 5; ++i)
  {
    if (i > 0)
      cs[0] ^= cs[i];
    in[i] ^= (uint32_t)cs[0];
    cs[0] = cs[0] >> 32;

    for (unsigned b = 0; b < 3; ++b)
    {
      cs[0] ^= crc64_table[b][in[i] & 0xff];
      in[i] >>= 8;
    }

    cs[0] ^= crc64_table[3][in[i] & 0xff];
  }

  while (data < end)
  {
    uint32_t idx = ((uint32_t)(cs[0] ^ *data++)) & 0xff;
    cs[0] = crc64_table[3][idx] ^ (cs[0] >> 8);
  }

  return cs[0] ^ UINT64_C(0xffffffffffffffff);
}

// Calculate the 'check bytes' for the provided checksum. If these bytes are
// appended to the original buffer, then the new total checksum should be zero.
static inline void crc64_invert(uint64_t cs, void* buffer)
{
  unsigned char* bytes = (unsigned char*)buffer;
  cs ^= UINT64_C(0xffffffffffffffff);

  // The CRC is self-inverting (in big-endian, so the bit-reversed CRC is
  // self-inverting in little-endian).
  bytes[7] = (cs >> 56) & 0xff;
  bytes[6] = (cs >> 48) & 0xff;
  bytes[5] = (cs >> 40) & 0xff;
  bytes[4] = (cs >> 32) & 0xff;
  bytes[3] = (cs >> 24) & 0xff;
  bytes[2] = (cs >> 16) & 0xff;
  bytes[1] = (cs >> 8) & 0xff;
  bytes[0] = cs & 0xff;
}

static const uint64_t crc64_x_pow_2n[64] = { UINT64_C(0x4000000000000000),
  UINT64_C(0x2000000000000000), UINT64_C(0x0800000000000000), UINT64_C(0x0080000000000000),
  UINT64_C(0x0000800000000000), UINT64_C(0x0000000080000000), UINT64_C(0xc96c5795d7870f42),
  UINT64_C(0x6d5f4ad7e3c3afa0), UINT64_C(0xd49f7e445077d8ea), UINT64_C(0x040fb02a53c216fa),
  UINT64_C(0x6bec35957b9ef3a0), UINT64_C(0xb0e3bb0658964afe), UINT64_C(0x218578c7a2dff638),
  UINT64_C(0x6dbb920f24dd5cf2), UINT64_C(0x7a140cfcdb4d5eb5), UINT64_C(0x41b3705ecbc4057b),
  UINT64_C(0xd46ab656accac1ea), UINT64_C(0x329beda6fc34fb73), UINT64_C(0x51a4fcd4350b9797),
  UINT64_C(0x314fa85637efae9d), UINT64_C(0xacf27e9a1518d512), UINT64_C(0xffe2a3388a4d8ce7),
  UINT64_C(0x48b9697e60cc2e4e), UINT64_C(0xada73cb78dd62460), UINT64_C(0x3ea5454d8ce5c1bb),
  UINT64_C(0x5e84e3a6c70feaf1), UINT64_C(0x90fd49b66cbd81d1), UINT64_C(0xe2943e0c1db254e8),
  UINT64_C(0xecfa6adeca8834a1), UINT64_C(0xf513e212593ee321), UINT64_C(0xf36ae57331040916),
  UINT64_C(0x63fbd333b87b6717), UINT64_C(0xbd60f8e152f50b8b), UINT64_C(0xa5ce4a8299c1567d),
  UINT64_C(0x0bd445f0cbdb55ee), UINT64_C(0xfdd6824e20134285), UINT64_C(0xcead8b6ebda2227a),
  UINT64_C(0xe44b17e4f5d4fb5c), UINT64_C(0x9b29c81ad01ca7c5), UINT64_C(0x1b4366e40fea4055),
  UINT64_C(0x27bca1551aae167b), UINT64_C(0xaa57bcd1b39a5690), UINT64_C(0xd7fce83fa1234db9),
  UINT64_C(0xcce4986efea3ff8e), UINT64_C(0x3602a4d9e65341f1), UINT64_C(0x722b1da2df516145),
  UINT64_C(0xecfc3ddd3a08da83), UINT64_C(0x0fb96dcca83507e6), UINT64_C(0x125f2fe78d70f080),
  UINT64_C(0x842f50b7651aa516), UINT64_C(0x09bc34188cd9836f), UINT64_C(0xf43666c84196d909),
  UINT64_C(0xb56feb30c0df6ccb), UINT64_C(0xaa66e04ce7f30958), UINT64_C(0xb7b1187e9af29547),
  UINT64_C(0x113255f8476495de), UINT64_C(0x8fb19f783095d77e), UINT64_C(0xaec4aacc7c82b133),
  UINT64_C(0xf64e6d09218428cf), UINT64_C(0x036a72ea5ac258a0), UINT64_C(0x5235ef12eb7aaa6a),
  UINT64_C(0x2fed7b1685657853), UINT64_C(0x8ef8951d46606fb5), UINT64_C(0x9d58c1090f034d14) };

// Compute (a*b) mod P
// See: https://code.google.com/p/crcutil/source/browse/code/gf_util.h
static inline uint64_t crc64_multiply_(uint64_t a, uint64_t b)
{
  if ((a ^ (a - 1)) < (b ^ (b - 1)))
  {
    uint64_t t = a;
    a = b;
    b = t;
  }

  if (a == 0)
    return 0;

  uint64_t r = 0, h = UINT64_C(1) << 63;
  for (; a != 0; a <<= 1)
  {
    if (a & h)
    {
      r ^= b;
      a ^= h;
    }

    b = (b >> 1) ^ ((b & 1) ? crc64_poly : 0);
  }

  return r;
}

// Compute x**n mod P
static inline uint64_t crc64_x_pow_n_(uint64_t n)
{
  uint64_t r = UINT64_C(1) << 63;
  for (size_t i = 0; n != 0; n >>= 1, ++i)
  {
    if (n & 1)
      r = crc64_multiply_(r, crc64_x_pow_2n[i]);
  }

  return r;
}

static inline uint64_t crc64_combine(uint64_t cs1, uint64_t cs2, size_t nbytes2)
{
  // For M = CONCAT(M1, M2) => CRC(M, a) = CRC(M2, CRC(M1, a)) and:
  // CRC(M, b) = CRC(M, a) + ((b-a)x^|M|) mod P.
  return cs2 ^ crc64_multiply_(cs1, crc64_x_pow_n_(8 * nbytes2));
}

static const size_t crc64_min_thread_bytes = 1024;

static inline uint64_t crc64_omp(const void* input, size_t nbytes)
{
#ifdef _OPENMP
  if (nbytes > 2 * crc64_min_thread_bytes)
  {
    int nthreads = omp_get_max_threads();

    if (nbytes < nthreads * crc64_min_thread_bytes)
      nthreads = nbytes / crc64_min_thread_bytes;

    uint64_t thread_cs[nthreads];
    size_t thread_sz[nthreads];

    const unsigned char* data = (const unsigned char*)input;
#pragma omp parallel num_threads(nthreads)
    {
      int tid = omp_get_thread_num();
      size_t bpt = nbytes / nthreads;
      const unsigned char *start = data + bpt * tid, *end;
      if (tid != nthreads - 1)
        end = start + bpt;
      else
        end = data + nbytes;

      size_t sz = end - start;
      thread_sz[tid] = sz;
      thread_cs[tid] = crc64(start, sz);
    }

    uint64_t cs = thread_cs[0];
    for (int i = 1; i < nthreads; ++i)
    {
      cs = crc64_combine(cs, thread_cs[i], thread_sz[i]);
    }

    return cs;
  }
#endif

  return crc64(input, nbytes);
}
} /* END namespace lanl */
#endif // CRC64_H
