#include <petsc-private/isimpl.h>
#include <petsc-private/vecimpl.h>             /*I "petscvec.h" I*/

#undef __FUNCT__
#define __FUNCT__ "VecScatterInitializeForGPU"
/*@
   VecScatterInitializeForGPU - Initializes a generalized scatter from one vector to
 another for GPU based computation.  Effectively, this function creates all the
 necessary indexing buffers and work vectors needed to move data only those data points
 in a vector which need to be communicated across ranks. This is done at the first time
 this function is called. Thereafter, this function launches a kernel,
 VecCUSPCopySomeToContiguousBufferGPU_Public, which moves the scattered data into a
 contiguous buffer on the GPU. Currently, this only used in the context of the parallel
 SpMV call in MatMult_MPIAIJCUSP (in mpi/mpicusp/mpiaijcusp.cu) or MatMult_MPIAIJCUSPARSE
 (in mpi/mpicusparse/mpiaijcusparse.cu). This function is executed before the call to
 MatMult. This enables the memory transfers to be overlapped with the MatMult SpMV kernel
 call.

   Input Parameters:
+  inctx - scatter context generated by VecScatterCreate()
.  x - the vector from which we scatter
-  mode - the scattering mode, usually SCATTER_FORWARD.  The available modes are:
    SCATTER_FORWARD or SCATTER_REVERSE

  Level: intermediate

.seealso: VecScatterCreate(), VecScatterEnd()
@*/
PetscErrorCode  VecScatterInitializeForGPU(VecScatter inctx,Vec x,ScatterMode mode)
{
  VecScatter_MPI_General *to,*from;
  PetscScalar            *xv;
  PetscErrorCode         ierr;
  PetscInt               i,*indices,*sstartsSends,*sstartsRecvs,nrecvs,nsends,bs;

  PetscFunctionBegin;
  if (mode & SCATTER_REVERSE) {
    to     = (VecScatter_MPI_General*)inctx->fromdata;
    from   = (VecScatter_MPI_General*)inctx->todata;
  } else {
    to     = (VecScatter_MPI_General*)inctx->todata;
    from   = (VecScatter_MPI_General*)inctx->fromdata;
  }
  bs           = to->bs;
  nrecvs       = from->n;
  nsends       = to->n;
  indices      = to->indices;
  sstartsSends = to->starts;
  sstartsRecvs = from->starts;
  if (x->valid_GPU_array != PETSC_CUSP_UNALLOCATED && (nsends>0 || nrecvs>0)) {
    if (!inctx->spptr) {
      PetscInt k,*tindicesSends,*sindicesSends,*tindicesRecvs,*sindicesRecvs;
      PetscInt ns = sstartsSends[nsends],nr = sstartsRecvs[nrecvs];
      /* Here we create indices for both the senders and receivers. */
      ierr = PetscMalloc(ns*sizeof(PetscInt),&tindicesSends);CHKERRQ(ierr);
      ierr = PetscMalloc(nr*sizeof(PetscInt),&tindicesRecvs);CHKERRQ(ierr);

      ierr = PetscMemcpy(tindicesSends,indices,ns*sizeof(PetscInt));CHKERRQ(ierr);
      ierr = PetscMemcpy(tindicesRecvs,from->indices,nr*sizeof(PetscInt));CHKERRQ(ierr);

      ierr = PetscSortRemoveDupsInt(&ns,tindicesSends);CHKERRQ(ierr);
      ierr = PetscSortRemoveDupsInt(&nr,tindicesRecvs);CHKERRQ(ierr);

      ierr = PetscMalloc(bs*ns*sizeof(PetscInt),&sindicesSends);CHKERRQ(ierr);
      ierr = PetscMalloc(from->bs*nr*sizeof(PetscInt),&sindicesRecvs);CHKERRQ(ierr);

      /* sender indices */
      for (i=0; i<ns; i++) {
        for (k=0; k<bs; k++) sindicesSends[i*bs+k] = tindicesSends[i]+k;
      }
      ierr = PetscFree(tindicesSends);CHKERRQ(ierr);

      /* receiver indices */
      for (i=0; i<nr; i++) {
        for (k=0; k<from->bs; k++) sindicesRecvs[i*from->bs+k] = tindicesRecvs[i]+k;
      }
      ierr = PetscFree(tindicesRecvs);CHKERRQ(ierr);

      /* create GPU indices, work vectors, ... */
      ierr = PetscCUSPIndicesCreate(ns*bs,sindicesSends,nr*from->bs,sindicesRecvs,(PetscCUSPIndices*)&inctx->spptr);CHKERRQ(ierr);
      ierr = PetscFree(sindicesSends);CHKERRQ(ierr);
      ierr = PetscFree(sindicesRecvs);CHKERRQ(ierr);
    }
    /*
     This should be called here.
     ... basically, we launch the copy kernel that takes the scattered data and puts it in a
         a contiguous buffer. Then, this buffer is messaged after the MatMult is called.
     */
#if 0 /* Paul, why did you leave this line commented after writing the note above explaining why it should be called? */
    /* I couldn't make this version run more efficiently. In theory, I would like to do it this way
       since the amount of data transfer between GPU and CPU is reduced. However, gather kernels
       really don't perform very well on the device. Thus, what I do is message (from GPU to CPU) the
       smallest contiguous chunk of the vector containing all those elements needing to be MPI-messaged.
       I would like to leave this code in here for now ... maybe I'll figure out how to do a better
       gather kernel on GPU. */
    ierr = VecCUSPCopySomeToContiguousBufferGPU_Public(x,(PetscCUSPIndices)inctx->spptr);CHKERRQ(ierr);
#endif
    } else {
    ierr = VecGetArrayRead(x,(const PetscScalar**)&xv);CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
#undef __FUNCT__
#define __FUNCT__ "VecScatterFinalizeForGPU"
/*@
   VecScatterFinalizeForGPU - Finalizes a generalized scatter from one vector to
 another for GPU based computation. Effectively, this function resets the temporary
 buffer flags. Currently, this only used in the context of the parallel SpMV call in
 in MatMult_MPIAIJCUSP (in mpi/mpicusp/mpiaijcusp.cu) or MatMult_MPIAIJCUSPARSE
 (in mpi/mpicusparse/mpiaijcusparse.cu). Once the MatMultAdd is finished,
 the GPU temporary buffers used for messaging are no longer valid.

   Input Parameters:
+  inctx - scatter context generated by VecScatterCreate()

  Level: intermediate

@*/
PetscErrorCode  VecScatterFinalizeForGPU(VecScatter inctx)
{
  PetscFunctionBegin;
  if (inctx->spptr) {
    PetscErrorCode ierr;
    ierr = VecCUSPResetIndexBuffersFlagsGPU_Public((PetscCUSPIndices)inctx->spptr);CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

