//
// StringTest.cpp
//
// $Id: //poco/1.3/Foundation/testsuite/src/StringTest.cpp#1 $
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "StringTest.h"
#include "CppUnit/TestCaller.h"
#include "CppUnit/TestSuite.h"
#include "Poco/String.h"


using Poco::trimLeft;
using Poco::trimLeftInPlace;
using Poco::trimRight;
using Poco::trimRightInPlace;
using Poco::trim;
using Poco::trimInPlace;
using Poco::toUpper;
using Poco::toUpperInPlace;
using Poco::toLower;
using Poco::toLowerInPlace;
using Poco::icompare;
using Poco::translate;
using Poco::translateInPlace;
using Poco::replace;
using Poco::replaceInPlace;
using Poco::cat;


StringTest::StringTest(const std::string& name): CppUnit::TestCase(name)
{
}


StringTest::~StringTest()
{
}


void StringTest::testTrimLeft()
{
	{
		std::string s = "abc";
		assert (trimLeft(s) == "abc");
	}
		std::string s = " abc ";
		assert (trimLeft(s) == "abc ");
	{
	std::string s = "  ab c ";
	assert (trimLeft(s) == "ab c ");
	}
}


void StringTest::testTrimLeftInPlace()
{
	{
		std::string s = "abc";
		assert (trimLeftInPlace(s) == "abc");
	}
	{
		std::string s = " abc ";
		assert (trimLeftInPlace(s) == "abc ");
	}
	{
		std::string s = "  ab c ";
		assert (trimLeftInPlace(s) == "ab c ");
	}
}


void StringTest::testTrimRight()
{
	{
		std::string s = "abc";
		assert (trimRight(s) == "abc");
	}
	{
		std::string s = " abc ";
		assert (trimRight(s) == " abc");
	}
	{
		std::string s = "  ab c  ";
		assert (trimRight(s) == "  ab c");
	}
}


void StringTest::testTrimRightInPlace()
{
	{
		std::string s = "abc";
		assert (trimRightInPlace(s) == "abc");
	}
	{
		std::string s = " abc ";
		assert (trimRightInPlace(s) == " abc");
	}
	{
		std::string s = "  ab c  ";
		assert (trimRightInPlace(s) == "  ab c");
	}
}


void StringTest::testTrim()
{
	{
		std::string s = "abc";
		assert (trim(s) == "abc");
	}
	{
		std::string s = "abc ";
		assert (trim(s) == "abc");
	}
	{
		std::string s = "  ab c  ";
		assert (trim(s) == "ab c");
	}
}


void StringTest::testTrimInPlace()
{
	{
		std::string s = "abc";
		assert (trimInPlace(s) == "abc");
	}
	{
		std::string s = " abc ";
		assert (trimInPlace(s) == "abc");
	}
	{
		std::string s = "  ab c  ";
		assert (trimInPlace(s) == "ab c");
	}
}


void StringTest::testToUpper()
{
	{
		std::string s = "abc";
		assert (toUpper(s) == "ABC");
	}
	{
		std::string s = "Abc";
		assert (toUpper(s) == "ABC");
	}
	{
		std::string s = "abc";
		assert (toUpperInPlace(s) == "ABC");
	}
	{
		std::string s = "Abc";
		assert (toUpperInPlace(s) == "ABC");
	}
}


void StringTest::testToLower()
{
	{
		std::string s = "ABC";
		assert (toLower(s) == "abc");
	}
	{
		std::string s = "aBC";
		assert (toLower(s) == "abc");
	}
	{
		std::string s = "ABC";
		assert (toLowerInPlace(s) == "abc");
	}
	{
		std::string s = "aBC";
		assert (toLowerInPlace(s) == "abc");
	}
}


void StringTest::testIcompare()
{
	std::string s1 = "AAA";
	std::string s2 = "aaa";
	std::string s3 = "bbb";
	std::string s4 = "cCcCc";
	std::string s5;
	assert (icompare(s1, s2) == 0);
	assert (icompare(s1, s3) < 0);
	assert (icompare(s1, s4) < 0);
	assert (icompare(s3, s1) > 0);
	assert (icompare(s4, s2) > 0);
	assert (icompare(s2, s4) < 0);
	assert (icompare(s1, s5) > 0);
	assert (icompare(s5, s4) < 0);

	std::string ss1 = "xxAAAzz";
	std::string ss2 = "YaaaX";
	std::string ss3 = "YbbbX";
	assert (icompare(ss1, 2, 3, ss2, 1, 3) == 0);
	assert (icompare(ss1, 2, 3, ss3, 1, 3) < 0);
	assert (icompare(ss1, 2, 3, ss2, 1) == 0);
	assert (icompare(ss1, 2, 3, ss3, 1) < 0);
	assert (icompare(ss1, 2, 2, ss2, 1, 3) < 0);
	assert (icompare(ss1, 2, 2, ss2, 1, 2) == 0);
	assert (icompare(ss3, 1, 3, ss1, 2, 3) > 0);
	
	assert (icompare(s1, s2.c_str()) == 0);
	assert (icompare(s1, s3.c_str()) < 0);
	assert (icompare(s1, s4.c_str()) < 0);
	assert (icompare(s3, s1.c_str()) > 0);
	assert (icompare(s4, s2.c_str()) > 0);
	assert (icompare(s2, s4.c_str()) < 0);
	assert (icompare(s1, s5.c_str()) > 0);
	assert (icompare(s5, s4.c_str()) < 0);
	
	assert (icompare(ss1, 2, 3, "aaa") == 0);
	assert (icompare(ss1, 2, 2, "aaa") < 0);
	assert (icompare(ss1, 2, 3, "AAA") == 0);
	assert (icompare(ss1, 2, 2, "bb") < 0);
	
	assert (icompare(ss1, 2, "aaa") > 0);
}


void StringTest::testTranslate()
{
	std::string s = "aabbccdd";
	assert (translate(s, "abc", "ABC") == "AABBCCdd");
	assert (translate(s, "abc", "AB") == "AABBdd");
	assert (translate(s, "abc", "") == "dd");
	assert (translate(s, "cba", "CB") == "BBCCdd");
	assert (translate(s, "", "CB") == "aabbccdd");
}


void StringTest::testTranslateInPlace()
{
	std::string s = "aabbccdd";
	translateInPlace(s, "abc", "ABC");
	assert (s == "AABBCCdd");
}


void StringTest::testReplace()
{
	std::string s("aabbccdd");
	
	assert (replace(s, std::string("aa"), std::string("xx")) == "xxbbccdd");
	assert (replace(s, std::string("bb"), std::string("xx")) == "aaxxccdd");
	assert (replace(s, std::string("dd"), std::string("xx")) == "aabbccxx");
	assert (replace(s, std::string("bbcc"), std::string("xx")) == "aaxxdd");
	assert (replace(s, std::string("b"), std::string("xx")) == "aaxxxxccdd");
	assert (replace(s, std::string("bb"), std::string("")) == "aaccdd");
	assert (replace(s, std::string("b"), std::string("")) == "aaccdd");
	assert (replace(s, std::string("ee"), std::string("xx")) == "aabbccdd");
	assert (replace(s, std::string("dd"), std::string("")) == "aabbcc");

	assert (replace(s, "aa", "xx") == "xxbbccdd");
	assert (replace(s, "bb", "xx") == "aaxxccdd");
	assert (replace(s, "dd", "xx") == "aabbccxx");
	assert (replace(s, "bbcc", "xx") == "aaxxdd");
	assert (replace(s, "bb", "") == "aaccdd");
	assert (replace(s, "b", "") == "aaccdd");
	assert (replace(s, "ee", "xx") == "aabbccdd");
	assert (replace(s, "dd", "") == "aabbcc");
	
	s = "aabbaabb";
	assert (replace(s, std::string("aa"), std::string("")) == "bbbb");
	assert (replace(s, std::string("a"), std::string("")) == "bbbb");
	assert (replace(s, std::string("a"), std::string("x")) == "xxbbxxbb");
	assert (replace(s, std::string("a"), std::string("xx")) == "xxxxbbxxxxbb");
	assert (replace(s, std::string("aa"), std::string("xxx")) == "xxxbbxxxbb");

	assert (replace(s, std::string("aa"), std::string("xx"), 2) == "aabbxxbb");

	assert (replace(s, "aa", "") == "bbbb");
	assert (replace(s, "a", "") == "bbbb");
	assert (replace(s, "a", "x") == "xxbbxxbb");
	assert (replace(s, "a", "xx") == "xxxxbbxxxxbb");
	assert (replace(s, "aa", "xxx") == "xxxbbxxxbb");
	
	assert (replace(s, "aa", "xx", 2) == "aabbxxbb");	
}


void StringTest::testReplaceInPlace()
{
	std::string s("aabbccdd");

	assert (replaceInPlace(s, std::string("aa"), std::string("xx")) == "xxbbccdd");
}


void StringTest::testCat()
{
	std::string s1("one");
	std::string s2("two");
	std::string s3("three");
	std::string s4("four");
	std::string s5("five");
	std::string s6("six");

	assert (cat(s1, s2) == "onetwo");
	assert (cat(s1, s2, s3) == "onetwothree");
	assert (cat(s1, s2, s3, s4) == "onetwothreefour");
	assert (cat(s1, s2, s3, s4, s5) == "onetwothreefourfive");
	assert (cat(s1, s2, s3, s4, s5, s6) == "onetwothreefourfivesix");
	
	std::vector<std::string> vec;
	assert (cat(std::string(), vec.begin(), vec.end()) == "");
	assert (cat(std::string(","), vec.begin(), vec.end()) == "");
	vec.push_back(s1);
	assert (cat(std::string(","), vec.begin(), vec.end()) == "one");
	vec.push_back(s2);
	assert (cat(std::string(","), vec.begin(), vec.end()) == "one,two");
	vec.push_back(s3);
	assert (cat(std::string(","), vec.begin(), vec.end()) == "one,two,three");
}


void StringTest::setUp()
{
}


void StringTest::tearDown()
{
}


CppUnit::Test* StringTest::suite()
{
	CppUnit::TestSuite* pSuite = new CppUnit::TestSuite("StringTest");

	CppUnit_addTest(pSuite, StringTest, testTrimLeft);
	CppUnit_addTest(pSuite, StringTest, testTrimLeftInPlace);
	CppUnit_addTest(pSuite, StringTest, testTrimRight);
	CppUnit_addTest(pSuite, StringTest, testTrimInPlace);
	CppUnit_addTest(pSuite, StringTest, testTrim);
	CppUnit_addTest(pSuite, StringTest, testTrimRightInPlace);
	CppUnit_addTest(pSuite, StringTest, testToUpper);
	CppUnit_addTest(pSuite, StringTest, testToLower);
	CppUnit_addTest(pSuite, StringTest, testIcompare);
	CppUnit_addTest(pSuite, StringTest, testTranslate);
	CppUnit_addTest(pSuite, StringTest, testTranslateInPlace);
	CppUnit_addTest(pSuite, StringTest, testReplace);
	CppUnit_addTest(pSuite, StringTest, testReplaceInPlace);
	CppUnit_addTest(pSuite, StringTest, testCat);

	return pSuite;
}
