//
// HTMLForm.h
//
// $Id: //poco/1.3/Net/include/Poco/Net/HTMLForm.h#1 $
//
// Library: Net
// Package: HTML
// Module:  HTMLForm
//
// Definition of the HTMLForm class.
//
// Copyright (c) 2005-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Net_HTMLForm_INCLUDED
#define Net_HTMLForm_INCLUDED


#include "Poco/Net/Net.h"
#include "Poco/Net/NameValueCollection.h"
#include <ostream>
#include <istream>
#include <vector>


namespace Poco {
namespace Net {


class HTTPRequest;
class PartHandler;
class PartSource;


class Net_API HTMLForm: public NameValueCollection
	/// HTMLForm is a helper class for working with HTML forms,
	/// both on the client and on the server side.
{
public:
	HTMLForm();
		/// Creates an empty HTMLForm and sets the
		/// encoding to "application/x-www-form-urlencoded".
		
	explicit HTMLForm(const std::string& encoding);
		/// Creates an empty HTMLForm that uses
		/// the given encoding.
		///
		/// Encoding must be either "application/x-www-form-urlencoded"
		/// (which is the default) or "multipart/form-data".
		
	HTMLForm(const HTTPRequest& request, std::istream& requestBody, PartHandler& handler);
		/// Creates a HTMLForm from the given HTTP request.
		///
		/// Uploaded files are passed to the given PartHandler.

	HTMLForm(const HTTPRequest& request, std::istream& requestBody);
		/// Creates a HTMLForm from the given HTTP request.
		///
		/// Uploaded files are silently discarded.

	explicit HTMLForm(const HTTPRequest& request);
		/// Creates a HTMLForm from the given HTTP request.
		///
		/// The request must be a GET request and the form data
		/// must be in the query string (URL encoded).
		///
		/// For POST requests, you must use one of the constructors
		/// taking an additional input stream for the request body.
		
	~HTMLForm();
		/// Destroys the HTMLForm.

	void setEncoding(const std::string& encoding);
		/// Sets the encoding used for posting the form.
		///
		/// Encoding must be either "application/x-www-form-urlencoded"
		/// (which is the default) or "multipart/form-data".
		
	const std::string& getEncoding() const;
		/// Returns the encoding used for posting the form.

	void addPart(const std::string& name, PartSource* pSource);
		/// Adds an part/attachment (file upload) to the form.
		///
		/// The form takes ownership of the PartSource and deletes it
		/// when it is no longer needed.
		///
		/// The part will only be sent if the encoding
		/// set for the form is "multipart/form-data"

	void load(const HTTPRequest& request, std::istream& requestBody, PartHandler& handler);
		/// Reads the form data from the given HTTP request.
		///
		/// Uploaded files are passed to the given PartHandler.

	void load(const HTTPRequest& request, std::istream& requestBody);
		/// Reads the form data from the given HTTP request.
		///
		/// Uploaded files are silently discarded.

	void load(const HTTPRequest& request);
		/// Reads the form data from the given HTTP request.
		///
		/// The request must be a GET request and the form data
		/// must be in the query string (URL encoded).
		///
		/// For POST requests, you must use one of the overloads
		/// taking an additional input stream for the request body.

	void read(std::istream& istr, PartHandler& handler);
		/// Reads the form data from the given input stream.
		///
		/// The form data read from the stream must be
		/// in the encoding specified for the form.
		
	void prepareSubmit(HTTPRequest& request);
		/// Fills out the request object for submitting the form.
		///
		/// If the request method is GET, the encoded form is appended to the
		/// request URI as query string. Otherwise (the method is
		/// POST), the form's content type is set to the form's encoding.
		/// The form's parameters must be written to the
		/// request body separately, with a call to write.
		/// If the request's HTTP version is HTTP/1.0:
		///    - persistent connections are disabled
		///    - the content transfer encoding is set to identity encoding
		/// Otherwise, if the request's HTTP version is HTTP/1.1:
		///    - the request's persistent connection state is left unchanged
		///    - the content transfer encoding is set to chunked
		
	void write(std::ostream& ostr, const std::string& boundary);
		/// Writes the form data to the given output stream,
		/// using the specified encoding.

	void write(std::ostream& ostr);
		/// Writes the form data to the given output stream,
		/// using the specified encoding.

	const std::string& boundary() const;
		/// Returns the MIME boundary used for writing
		/// multipart form data.

	static const std::string ENCODING_URL;       /// "application/x-www-form-urlencoded"
	static const std::string ENCODING_MULTIPART; /// "multipart/form-data"

protected:
	void readUrl(std::istream& istr);
	void readMultipart(std::istream& istr, PartHandler& handler);
	void writeUrl(std::ostream& ostr);
	void writeMultipart(std::ostream& ostr);

private:
	HTMLForm(const HTMLForm&);
	HTMLForm& operator = (const HTMLForm&);

	struct Part
	{
		std::string name;
		PartSource* pSource;
	};
	
	typedef std::vector<Part> PartVec;
	
	std::string _encoding;
	std::string _boundary;
	PartVec     _parts;
};


//
// inlines
//
inline const std::string& HTMLForm::getEncoding() const
{
	return _encoding;
}


inline const std::string& HTMLForm::boundary() const
{
	return _boundary;
}


} } // namespace Poco::Net


#endif // Net_HTMLForm_INCLUDED
