/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "syncprocessor.h"
#include "global.h"
#include "configparams.h"
#include <math.h>
#include "dispatcher.h"
#ifndef QT_NO_DEBUG
#include "scopeview.h"
#endif
#include "rxwidget.h"
#include "filter.h"
#include "filterparam.h"


#define SCOPE_ENABLED

//consecSyncs volumeTreshold syncOnTreshold syncOffTreshold maxVolTreshold missingSyncs

// the start of the sync pulse is conditioned by syncVolume is above syncOnTreshold*volume
// the end of the sync pulse is conditioned by syncVolume is below syncOffTreshold*volume
// a supplementary condition is that the max volume off the syncpulse must be above maxVol*volume

ssquelch squelchTbl[SQUELCHLEVELS]=
{
  { 2, 400,  1.6,   0.90, 2.0, 26},
  { 2, 350,  1.6,   0.90, 1.9, 23},
  { 2, 300,  1.6,   0.90, 1.8, 20}
};





/*!
  \class syncProcessor Filters the incoming signal using a narrow bandpassfilter centered around 1200 Hz

	Event generated: verticalRetraceEvent
	<br> isInSync: test for sync status \sa getSyncPosition() 
	

*/

/*!
	constructor

	\param len The stripeLength of the incoming data (e.g; len=1024, one block of data contains 1024 short int values)
*/
syncProcessor::syncProcessor(int len)
{
	// precalculate constants
// 	int	k;
  stripeLength=len;
  syncStateBuffer=new DSPFLOAT[stripeLength];
  tAvgBuffer=     new DSPFLOAT[stripeLength];

  // syncFilter=new filter(len,sharp1200BP,RXNUMTAPS,1200,FALSE);
  syncFilter=new filter(len,wide1200BP,RXNUMTAPS,1200,FALSE);
  volumeFilter=new filter(len,veryWideVolume,RXNUMTAPS,1750,FALSE);
  syncBufferPtr=syncFilter->filteredDataPtr();
  volumeBufferPtr=volumeFilter->volumePtr();
  volumeFreqPtr=volumeFilter->filteredDataPtr();
  init(0);

}


syncProcessor::~syncProcessor()
{
  delete [] tAvgBuffer;
  delete [] syncStateBuffer;
  delete volumeFilter;
  delete syncFilter;
}

/*!
	init() must be called before the start a new capture (i.e with every new image acquisition)
*/
void syncProcessor::init(unsigned int modeComboIdx)
{
	syncStarted=FALSE;
	syncArrayIndex=0;
	syncArray[0].clear();
	sampleCounter=0;
	syncFound=FALSE;
	newClock=FALSE;
	modifiedClock=rxClock;
	slantAdjustLine=6;
	prevTmp=0;
  volumeFiltered=700.;
  volumeSync=500.;
	mode=NOTVALID;
  syncWd=MINSYNCWIDTH*rxClock;
  noSyncInStripe=0;
  prevSyncRejected=TRUE;
  modeComboIndex=modeComboIdx;
  initSignalQuality();
  addToLog("syncProcessor: init() called",DBSYNC1);
}

/*!
  process() this method is called from rxFunctions.
  \param buf Pointer to raw data
  \param filter Pointer to the rxFilter used (to access the demodulated data and the volume)

  - syncBufferPtr points to the data from the 1200Hz bandpass filter
  - volumeBufferPtr points to the wideband filter volume detection
  - volumeFreqPtr points to the FM demodulated data to get to the frequency around 1200Hz (the standaard rx-filter is to narrow for that)

  When all the filtering is done, then the synpulses are evaluated by the ananlyser
*/


void syncProcessor::process(short int *buf)
{

  displaySyncEvent* ce;
  addToLog(QString("Process sync %1").arg(sampleCounter),DBPERFORM);
  volumeFilter->processFM(buf);
  syncFilter->processBP(buf);
  for(unsigned int i=0;i<stripeLength;i++)
    {
      // topdetector
      if(prevTmp<syncBufferPtr[i]) prevTmp=prevTmp+0.9*(syncBufferPtr[i]-prevTmp);
      else prevTmp=prevTmp+0.1*(syncBufferPtr[i]-prevTmp);
      syncBufferPtr[i]=prevTmp;
      if(volumeFiltered<volumeBufferPtr[i]) volumeFiltered=volumeFiltered+0.001*(volumeBufferPtr[i]-volumeFiltered);
      else volumeFiltered=volumeFiltered+0.0005*(volumeBufferPtr[i]-volumeFiltered);
      tAvgBuffer[i]=volumeFiltered;
      avgVolume=volumeFiltered+0.8*(avgVolume-volumeFiltered);
    }

  double x=log10(volumeFiltered);
  if(x>=2.8) x-=2.8; // lowest level =1000
  else x=0;
  if ((mode<AVT24) || (mode > AVT94))
    {

      ce = new displaySyncEvent(signalQuality*2.5,x*7);
    }
  else
    {
      ce = new displaySyncEvent(0,x*7);
    }
  QApplication::postEvent(dispatchPtr, ce);
  analyze();

#ifndef QT_NO_DEBUG
#ifdef SCOPE_ENABLED
  scopeViewer->addData1(syncBufferPtr,sampleCounter,stripeLength);
  scopeViewer->addData2(syncStateBuffer,sampleCounter,stripeLength);
  scopeViewer->addData3(tAvgBuffer,sampleCounter,stripeLength);
  scopeViewer->addData4(volumeFreqPtr,sampleCounter,stripeLength);
#endif
#endif
  sampleCounter+=stripeLength;
}


/*!
Analyze is run after every block of data has been filtered. It's looking for the start and end of the sync pulse and makes an entry
in the syncArrayArray filling in the start and end position of the sync. It also measures the length of the sync pulse to determine
if we have a vertical retrace.
*/


void syncProcessor::analyze()
{
  qApp->processEvents();
  unsigned int i;
  for(i=0;i<stripeLength;i++)
		{
			if(tAvgBuffer[i]<squelchTbl[rxMW->squelch-1].volumeTreshold) // we have not enough audio
				{
					syncStarted=FALSE;
					syncStateBuffer[i]=0.;
//					if(isInSync())
//						{
//							addToLog("Synclost low volume",DBSYNC1);
//							syncLost();
//							return;
//						}
					continue;
				 }
			if(!syncStarted)
				{
          if(prevSyncRejected==TRUE) syncStateBuffer[i]=0.;
          else syncStateBuffer[i]=0.5;
          syncAvgFreq=1200;
          syncMaxVol=0;
          syncAvgCount=0;
         if(syncBufferPtr[i]>tAvgBuffer[i]*squelchTbl[rxMW->squelch-1].syncOnTreshold)
          {
							syncStarted=TRUE;
              visCode=0;
              visCounter=0;
              visAvg=0.;
							syncArray[syncArrayIndex].startPosition=sampleCounter+i;
						}
				}
			else
				{
					noSyncInStripe++;
					syncStateBuffer[i]=1.;
          visCounter++;
          syncAvgCount++;
          if((syncAvgCount>10) && (syncAvgCount<50))
            {
              syncAvgFreq=syncAvgFreq+0.1*(volumeFreqPtr[i]-syncAvgFreq);
            }
          if(syncMaxVol<syncBufferPtr[i]) syncMaxVol=syncBufferPtr[i];
          if(visCounter>100)
            {
              visAvg+=volumeFreqPtr[i];
            }
          if(visCounter==200)
            {
              visAvg/=100.;
              visCode=visCode>>1;
              if(visAvg<=1200.) visCode=visCode | 0X100;
              addToLog(QString("visfreq %1, counter %2, code=%3").arg(visAvg).arg(visCounter).arg(visCode),DBSYNC2);
             }
          if(visCounter > (unsigned int)(0.030*rxClock)) //reset visCounter per bit
            {
              visCounter=0;
              visAvg=0.;
            }
          if(syncBufferPtr[i]<(tAvgBuffer[i]*squelchTbl[rxMW->squelch-1].syncOffTreshold))
            {
              if(addSync(i))
                {
                  noSyncInStripe=0;
                }
              else
                {
                  prevSyncRejected=TRUE;
                  rejectCounter++;
                }
             }
        }
    }
  if(noSyncStripes>noSyncStripes)
    {
      addToLog(QString("no sync in stripe stripes=%1, stripes=%2 reject counter %3").arg(noSyncStripes).arg(noSyncStripes).arg(rejectCounter),DBSYNC1);
      rejectCounter+=20;
      noSyncInStripe=0;

    }
  if(!signalQualityOK())
    {
      addToLog("Signal Quality too low",DBSYNC1);
      syncLost();
    }

}

/**
 addSync is called at the detection of an end of sync.
 It checks several sync parameters to define if we have a valid sync
 - max volme of the sync
 - minimum length
 - retrace or not
 - accuracy of the retrace to validate the VIS code

 At the end it calls validate to see if we should have enough info to start an image capture
*/

bool syncProcessor::addSync(int pos)
{
  prevSyncRejected=FALSE;
  syncArray[syncArrayIndex].retrace=FALSE;
  syncStarted=FALSE;
  syncArray[syncArrayIndex].maxVol=syncMaxVol;
  syncArray[syncArrayIndex].freq= syncAvgFreq;
  syncArray[syncArrayIndex].endPosition=sampleCounter+pos;
  syncArray[syncArrayIndex].length=syncArray[syncArrayIndex].endPosition-syncArray[syncArrayIndex].startPosition;
  if(syncMaxVol<tAvgBuffer[pos]*squelchTbl[rxMW->squelch-1].maxVolTreshold)
    {
      addToLog(QString("syncprocessor:sync rejected on maxVol=%1 at syncend %2, volume=%3, signalQuality=%4")
               .arg(syncMaxVol).arg(syncArray[syncArrayIndex].endPosition).arg(tAvgBuffer[pos]*squelchTbl[rxMW->squelch-1].maxVolTreshold).arg(signalQuality),DBSYNC2);
      return FALSE;
    }
  if((syncArray[syncArrayIndex].freq<1000) || (syncArray[syncArrayIndex].freq>1400))
    {
      addToLog(QString("syncprocessor:sync rejected on freq:= %1 at syncend %2, volume=%3, signalQuality=%4")
               .arg(syncArray[syncArrayIndex].freq).arg(syncArray[syncArrayIndex].endPosition).arg(tAvgBuffer[pos]*squelchTbl[rxMW->squelch-1].maxVolTreshold).arg(signalQuality),DBSYNC2);

      return FALSE;
    }
  addToLog(QString("sync: syncIndex:%1 syncstart=%2  syncend=%3 synclength=%4 volume=%5, signalQuality=%6")
                    .arg(syncArrayIndex).arg(syncArray[syncArrayIndex].startPosition).arg(syncArray[syncArrayIndex].endPosition)
                   .arg(syncArray[syncArrayIndex].endPosition-syncArray[syncArrayIndex].startPosition).arg(tAvgBuffer[pos]).arg(signalQuality),DBSYNC2);

  if(syncArray[syncArrayIndex].length>((unsigned int)(RETRACEWIDTH*0.7*rxClock)))
    {
      syncArray[syncArrayIndex].retrace=TRUE;
      visCode=visCode&0xFF;
      if(syncArray[syncArrayIndex].length<((unsigned int)(RETRACEWIDTH*0.95*rxClock)))
        {
          // we only accept the viscode when the retrace length is accurate enough
          visCode=0;
        }
      addToLog(QString("syncprocessor:retrace detected: viscode=%1").arg(QString::number(visCode,16)),DBSYNC1);
    }
  else if(syncArray[syncArrayIndex].length>MAXSYNCWIDTH*1.1*rxClock)
  {
    addToLog(QString("syncprocessor:syncwidth to long %1").arg(syncArray[syncArrayIndex].length),DBSYNC1);
    return FALSE;
  }
  else if(syncArray[syncArrayIndex].length<MINSYNCWIDTH*0.9*rxClock)
  {
    addToLog(QString("syncprocessor:syncwidth to short %1").arg(syncArray[syncArrayIndex].length),DBSYNC1);
    return FALSE;
  }
  validate();
  return TRUE;
}



/*!
  Adds a sync record to the syncArray. The syncArray is reset when a retrace pulse has been detetected and a retrace event is posted.
 The modeDetect is called when we are still hunting for an image. The program keeps track of the syncs during the the image acquisition.
*/

void syncProcessor::validate()
{
  bool done=FALSE;
  currentClosestLine=1;

  if(syncArray[syncArrayIndex].retrace==TRUE)
    {
      syncArray[0]=syncArray[syncArrayIndex];
      syncArrayIndex=0;
      if(mode!=NOTVALID) // only send if we are capturing an image
        {
          verticalRetraceEvent* ce = new verticalRetraceEvent();
          ce->waitFor(&done);
          QApplication::postEvent(dispatchPtr, ce);
          while(!done) { usleep(10);}
          return;
        }
      if(visCode==0)
      {
        return; // we can't use this retrace to synchronize on
      }
      modeDetect(visCode);
    }

  else if(!syncFound)
    {
       if (!rxMW->useVIS)  modeDetect(0); //check via lineLength only if useVIS is not required
    }
  else // we are capturing an image
    {
      if(validateSyncEntry())
        {
          if(!slantAdjust())
            {
              addToLog("Clock jump",DBSYNC1);
              syncLost();
            }
         }
    }
  if((!isInSync())&&(syncArrayIndex>=30)) // get rid off old sync info
    {
       memmove(syncArray,syncArray+1,sizeof(ssyncArray)*(syncArrayIndex));
       return;
    }

  if(syncArrayIndex==(SYNCARRAYLEN-1)) // do not overflow the syncArrayTable
    {
      memmove(syncArray,syncArray+1,sizeof(ssyncArray)*(SYNCARRAYLEN-1));
    }
  else
    {
      addToLog(QString("Increment syncArrayIndex %1").arg( syncArrayIndex+1),DBSYNC2);

      syncArrayIndex++;
      syncArray[syncArrayIndex].clear();
    }
}

/**
  The modeDetect tries to find a valid image capturing mode
  - in case of a retrace it will try to use the VIS code if it is valid
  - else it will try to find it via the line length of the mode
  */

void syncProcessor::modeDetect(unsigned int visCde)
{
   if((mode=lookupVIS(visCde))!=NOTVALID)
    {
      syncFound=TRUE;
      initSignalQuality();
//      if((mode>=S1) && (mode<=SDX)) moveToTop(1); // take next sync pulse as reference
     syncPosition=syncArray[0].endPosition;
      addToLog(QString("modeDetect: after retrace syncpos:=%1").arg(syncPosition),DBSYNC1);
      samplesPerLine=lineLength(mode,modifiedClock);
      syncWd=syncWidth(mode,modifiedClock);
      noSyncStripes=squelchTbl[rxMW->squelch-1].missingSyncs*samplesPerLine/stripeLength;
      addToLog(QString("syncProcessor:modeDetect Viscode used: %1,mode=%2").arg(QString::number(visCde,16)).arg(mode),DBSYNC1);
      return;
    }
  else if(visCde!=0)
    {
      addToLog(QString("syncProcessor:modeDetect Viscode rejected: %1").arg(QString::number(visCde,16)),DBSYNC1);
    }
  modeDetectByLine();
 // addToLog("modeDetect: stop",DBPERFORM);
}



/**
  We have a syncArray with at least a minimal number of syncs
  We take the first one and see if the rest is matching a certain mode.
  If it does not
*/

void syncProcessor::modeDetectByLine()
{
  unsigned int i=0;
  unsigned int highest=0;
  unsigned int lowestLineCount=1000;
  unsigned int idxStart=0;
  unsigned int idxEnd=(unsigned int)AVT24;
  if(modeComboIndex!=0)
    {
      idxStart=modeComboIndex-1;
      idxEnd=modeComboIndex;
    }

  syncTimeSpan=longestLine(modifiedClock)*2.1;
  if((syncArray[syncArrayIndex].endPosition-syncArray[0].endPosition)>syncTimeSpan)
    {
      addToLog(QString("Deleting first syncArray entry, syncTimeSpan exceeded %1").arg(syncTimeSpan),DBSYNC2);
      //moveToTop(1);
      cleanNotInUse();
    }
  esstvMode selectedMode=NOTVALID;
  if(!(syncArrayIndex>=squelchTbl[rxMW->squelch-1].consecSyncs)) return;

  for(i=idxStart;i<idxEnd;i++)
    {
      getMatchingSyncPulse(i);
    }




  for(i=idxStart;i<idxEnd;i++)
    {
      if((modeArray[i].match> highest)&&(modeArray[i].consecutiveSyncs>2))
        {
//          if(modeArray[i].lineNumber<20)
            {
              highest=modeArray[i].match;
              selectedMode=(esstvMode)i;
            }
        }
      if(modeArray[i].lineNumber<lowestLineCount)
      {
        lowestLineCount=modeArray[i].lineNumber;
      }
    }
  if(selectedMode==NOTVALID) return;
  addToLog(QString("Mode Candidate: mode=%1, match=%2, consecSyncs=%3, firstSync=%4, ratio=%5 lineLength=%6")
           .arg(getSSTVModeNameShort(selectedMode)).arg(modeArray[selectedMode].match).arg(modeArray[selectedMode].consecutiveSyncs)
            .arg(modeArray[selectedMode].firstSyncIndex).arg(modeArray[selectedMode].ratio)
            .arg(lineLength(selectedMode,modifiedClock)),DBSYNC1);
//  if(lowestLineCount<2) return;
//  if(modeArray[(int)selectedMode].lineNumber>20)
//  {
//    addToLog(QString("Linenumber to high %1").arg(modeArray[(int)selectedMode].lineNumber),DBSYNC2);
//   // moveToTop(1);
//  }
//  if(((modeArray[selectedMode].consecutiveSyncs>(squelchTbl[rxMW->squelch-1].consecSyncs))&&highest>2 )||
//     ((highest>7)))
//  if(((modeArray[selectedMode].ratio<1.9)&&(highest>2) ) || ((modeArray[selectedMode].match>4)&&(modeArray[selectedMode].consecutiveSyncs>2)))
//  if((modeArray[selectedMode].ratio<1.9)&&(highest>3))
  if(highest>3)
    {
      syncFound=TRUE;
      initSignalQuality();
      mode=selectedMode;
      samplesPerLine=lineLength(mode,modifiedClock);
      syncWd=syncWidth(mode,modifiedClock);
      //find first occurence
      syncPosition=syncArray[modeArray[(int)selectedMode].firstSyncIndex].endPosition;
      dumpSyncArray();
      cleanupSyncArray(selectedMode);
      noSyncStripes=(squelchTbl[rxMW->squelch-1].missingSyncs)*samplesPerLine/stripeLength;
//      if(modeArray[selectedMode].ratio<1.2) syncQuality=MAXSYNCQUALITY;
//      else syncQuality=MAXSYNCQUALITY/2;
      addToLog(QString("syncFound:  mode=%1").arg(selectedMode),DBSYNC1);

    }
}




/**

  We do a backward search for every new syncpulse added to the array.
  */




int syncProcessor::matchingSync(int start,unsigned int pos,DSPFLOAT syncWidth)
{
  int i;

  for(i=start;i>=0;i--)
  {
     DSPFLOAT sw=(DSPFLOAT)(syncArray[i].endPosition-syncArray[i].startPosition);
    if(syncArray[i].retrace==TRUE)
      {
        if((pos>syncArray[i].endPosition-MAXSYNCWIDTH*modifiedClock) && (pos<syncArray[i].endPosition)) return i;
      }
    else if((syncWidth>(1.2*sw))||(syncWidth<(0.8*sw)))
    {
  //     addToLog(QString("Match rejected on syncWidth:%1 out of bound with %2").arg(sw).arg(syncWidth),DBSYNC2);
      continue;
    }
    else if((pos>syncArray[i].startPosition) && (pos<syncArray[i].endPosition)) return i;
  }
  return -1;
}

void syncProcessor::getMatchingSyncPulse(int modeIndex)
{
  bool adjecent=TRUE;
  DSPFLOAT ratio;
  unsigned int pos;
  int result;
  double length=lineLength((esstvMode)modeIndex,modifiedClock);
  double syncW=syncWidth((esstvMode) modeIndex,modifiedClock);
  unsigned int offset=syncArray[syncArrayIndex].endPosition-(int)(syncW/2);
  modeArray[modeIndex].clear();
  ratio=1000;
  if(offset<=length) return;
  pos=(offset-length);
  int start=(syncArrayIndex-1);
  while(1)
    {
      if((result=matchingSync(start,pos,syncW))>=0)
        {
          pos=syncArray[result].endPosition-(int)(syncW/2);
          start=result-1;
          modeArray[modeIndex].match++;
          modeArray[modeIndex].firstSyncIndex=result;
          syncArray[result].inUse=1;
          if(adjecent==TRUE) modeArray[modeIndex].consecutiveSyncs++;
         }
      else
        {
          adjecent=FALSE;
        }
      modeArray[modeIndex].lineNumber++;
      if(pos<length) break;
      if(pos<syncArray[0].startPosition)break;
      pos-=length;
    }
  if( modeArray[modeIndex].consecutiveSyncs>0)
    {
      ratio=((double)(syncArray[syncArrayIndex].endPosition-syncArray[modeArray[modeIndex].firstSyncIndex].endPosition))/(length*(double) modeArray[modeIndex].match);
      ratio*=(double) modeArray[modeIndex].match/(double) modeArray[modeIndex].consecutiveSyncs;
    }
  modeArray[modeIndex].ratio=ratio;

}


void syncProcessor::cleanupSyncArray(int modeIndex)
{
  unsigned int i;
  unsigned int pos;
  int result;
  double length=lineLength((esstvMode)modeIndex,modifiedClock);
  double syncW=syncWidth((esstvMode) modeIndex,modifiedClock);
  unsigned int offset=syncArray[syncArrayIndex].endPosition-(int)(syncW/2);
  modeArray[modeIndex].clear();
  if(offset<=length) return;
  pos=(offset-length);
  int start=(syncArrayIndex-1);
  while(1)
    {
      if((result=matchingSync(start,pos,syncW))>=0)
        {
          pos=syncArray[result].endPosition-(int)(syncW/2);
          start=result-1;
          syncArray[result].keep=TRUE;
         }
      if(pos<length) break;
      if(pos<syncArray[0].startPosition)break;
      pos-=length;
    }
  syncArray[syncArrayIndex].keep=TRUE; // always keep last syncEntry
  for(i=0;i<=syncArrayIndex;)
  {
    if(!syncArray[i].keep)
    {
      deleteSyncEntry(i);
    }
    else
    {
      syncArray[i].lineNumber=rint((syncArray[i].endPosition-syncArray[0].endPosition+samplesPerLine/2)/samplesPerLine);
      i++;
    }
  }
  addToLog("After cleaning",DBSYNC1);
 dumpSyncArray();
}

void syncProcessor::cleanNotInUse()
{
 unsigned int i;
//  addToLog("Before cleaning not in use",DBSYNC1);
//  dumpSyncArray();
  for(i=0;i<syncArrayIndex;)
  {
    if((syncArray[i].inUse==0) && ((syncArray[syncArrayIndex].endPosition-syncArray[i].endPosition)>syncTimeSpan))
      {
        deleteSyncEntry(i);
      }
    else i++;
  }
//  addToLog("After cleaning not in use",DBSYNC1);
//  dumpSyncArray();

}

void syncProcessor::initSignalQuality()
{
  noSyncStripes=0;
  noSyncInStripe=0;
  //syncQuality=0;
 // unsigned int successiveSyncs;
  rejectCounter=0;
}

bool syncProcessor::signalQualityOK()
{
  unsigned int i;
  unsigned int missingLines=0;
  signalQuality=4;
  if(avgVolume<600) signalQuality--;
  if(avgVolume<300) signalQuality--;
  if(avgVolume<200) signalQuality=0;
  if(rejectCounter>60) signalQuality--;
  if(rejectCounter>125) signalQuality--;
  if(rejectCounter>150) signalQuality=0;
  if(syncArrayIndex>3)
  {
    for(i=syncArrayIndex-3;i<=syncArrayIndex;i++)
    {
      missingLines+=syncArray[i].closestLine;
    }
  }
  if(missingLines>squelchTbl[rxMW->squelch-1].missingSyncs)
  {
    signalQuality=0;
  }

  addToLog(QString("SQ=%1,Volume=%2,RejectCounter=%3, missingLines=%5").arg(signalQuality).arg(avgVolume)
           .arg(rejectCounter).arg(missingLines),DBSYNC2);
  return ((signalQuality>0) || !(isInSync()));
}



/*!
Check if this is a valid entry in the syncArray. This function is only called when we are receiving an image.
So we will only accept syncpulses that are close to the calculated position.
\return TRUE if OK
*/
bool syncProcessor::validateSyncEntry()
{
  unsigned int closestLine;
	DSPFLOAT diff;
	DSPFLOAT syncError;
	if(syncArrayIndex==0) return TRUE;
	if((syncArrayIndex==1) &&(syncArray[0].retrace==TRUE))
     {
    // do not calculate diff after retrace to avoid problems with scottie modes
//    syncQuality+=2;
//    if(syncQuality>MAXSYNCQUALITY) syncQuality=MAXSYNCQUALITY;
    return TRUE;
    }
  diff=(DSPFLOAT)(syncArray[syncArrayIndex].endPosition-syncArray[syncArrayIndex-1].endPosition);
  syncArray[syncArrayIndex].lineNumber=rint((syncArray[syncArrayIndex].endPosition-syncArray[0].endPosition+samplesPerLine/2)/samplesPerLine);
  closestLine=(unsigned int)((diff+samplesPerLine/2))/samplesPerLine;
  syncArray[syncArrayIndex].closestLine=closestLine;
  currentClosestLine=closestLine;
  syncError=fabs((diff-samplesPerLine*closestLine)/samplesPerLine);
  if ((syncError<0.05)&&(closestLine>0))
  {
    //accept
    syncArray[syncArrayIndex].lineNumber=rint((syncArray[syncArrayIndex].endPosition-syncArray[0].endPosition+samplesPerLine/2)/samplesPerLine);
    addToLog(QString("sync accepted index=%1 syncError%2 syncWidth =%3 diff=%4 index=%5 syncEnd1=%6 syncEnd2=%7, closestline %8, signalQuality: %9")
                .arg(syncArrayIndex).arg(syncError).arg(syncArray[syncArrayIndex].endPosition-syncArray[syncArrayIndex].startPosition)
                .arg(diff).arg(syncArrayIndex).arg(syncArray[syncArrayIndex].endPosition).arg(syncArray[syncArrayIndex-1].endPosition).arg(closestLine).arg(signalQuality),DBSYNC2);
    rejectCounter=0;
    return TRUE;
  }
  // the syncpulse does not seem to be correct
 addToLog(QString("sync not acceptedindex=%1 syncError%2 syncWidth =%3 diff=%4 index=%5 syncEnd1=%6 syncEnd2=%7, closestline %8, signalQuality: %9")
             .arg(syncArrayIndex).arg(syncError).arg(syncArray[syncArrayIndex].endPosition-syncArray[syncArrayIndex].startPosition)
             .arg(diff).arg(syncArrayIndex).arg(syncArray[syncArrayIndex].endPosition).arg(syncArray[syncArrayIndex-1].endPosition).arg(closestLine).arg(signalQuality),DBSYNC2);
 //try to find a better match
 rejectCounter++;
 deleteSyncEntry(syncArrayIndex);
 return FALSE;
}







/*!  shift up the syncArray so that syncArray[index] is at the top.
*/
void syncProcessor::moveToTop(int index)
{
	memmove(syncArray,syncArray+index,sizeof(ssyncArray)*(syncArrayIndex+1-index));
	syncArrayIndex-=index;
}

void syncProcessor::deleteSyncEntry(int index)
{
	memmove(syncArray+index,syncArray+index+1,sizeof(ssyncArray)*(syncArrayIndex-index));
	syncArrayIndex--;
//  if(syncQuality>0) syncQuality--;
}


/*!
returns 0 if not in Sync
returns 1 if in Sync without a retrace
returns 2 if in Sync and retrace -- the syncPosition is at the end of the retrace
*/
int syncProcessor::isInSync()
{
  int result=0;
  if(syncFound)
  {
    result++;
  }
  if(syncArray[0].retrace==TRUE)
  {
    result++;
   }

  return result;
}


/*!
 The modeDetect tries to find the sstv mode based on the line length. If there are a number (based on the squelch) of consecutive correct lines the syncFound and the mode are set. The syncArray is the searched for the first valid line length and the pointer is set at the end of the  sync pulse.
*/


void syncProcessor::dumpSyncArray()
{
  addToLog(QString("syncProcessor: syncArrayDump index=%1 start=%2 end=%3 width=%4 linelength=n/a closestLine=n/a frequency=%5")
    .arg(0).arg(syncArray[0].startPosition).arg(syncArray[0].endPosition)
    .arg(syncArray[0].endPosition-syncArray[0].startPosition).arg(syncArray[0].freq),DBSYNC1);
  for(uint i=1;i<=syncArrayIndex;i++)
		{
      addToLog(QString("syncProcessor: syncArrayDump index=%1 start=%2 end=%3 width=%4 linelength=%5 closestLine=%6 frequency=%7")
        .arg(i).arg(syncArray[i].startPosition).arg(syncArray[i].endPosition)
        .arg(syncArray[i].endPosition-syncArray[i].startPosition).arg(syncArray[i].endPosition-syncArray[i-1].endPosition)
                                                                      .arg(syncArray[i].closestLine).arg(syncArray[i].freq),DBSYNC1);
		}	
}

void syncProcessor::regression(DSPFLOAT &a,DSPFLOAT &b,int start, int end)
{
  /* calculate linear regression
    formula x=a+by
    b=sum((x[i]-xm)*(y[i]-ym))/sum((y[i]-ym)*(y[i]-ym))
    a=xm-b*ym
  */
  int i,j;

  DSPFLOAT sum_x,sum_y,sum_xx,sum_xy;
  int count=end-start+1;
  sum_x=sum_y=sum_xx=sum_xy=a=b=0;
  for(j=1,i=start;i<=end;i++,j++)
    {
      slantArray[j].y= (DSPFLOAT)(syncArray[i].endPosition-syncArray[start-1].endPosition);
      slantArray[j].x= ((unsigned int)((((slantArray[j].y+samplesPerLine/2)/samplesPerLine)))*samplesPerLine);
      sum_x+=slantArray[j].x;
      sum_y+=slantArray[j].y;
      sum_xx+=slantArray[j].x*slantArray[j].x;
      sum_xy+=slantArray[j].x*slantArray[j].y;
    }
  b=((count)*sum_xy-(sum_x*sum_y))/((count)*sum_xx-(sum_x*sum_x));
  a=sum_y/(count)-(b*sum_x)/(count);
}



bool syncProcessor::slantAdjust()
{
  DSPFLOAT a,b;
  //DSPFLOAT a1,b1;
  if ((mode>=AVT24) && (mode <= AVT94)) return TRUE;
  if(mode==NOTVALID) return TRUE;
  if(syncArrayIndex==0)
    {
      logfile->addToAux(QString("Index\tStart\tEnd\tClosestLine\tRetrace\t%1").arg(samplesPerLine));
    }
  logfile->addToAux(QString("%1\t%2\t%3\t%4\t%5")
                    .arg(syncArrayIndex)
                    .arg(syncArray[syncArrayIndex].startPosition)
                    .arg(syncArray[syncArrayIndex].endPosition)
                    .arg(syncArray[syncArrayIndex].closestLine)
                    .arg(syncArray[syncArrayIndex].retrace)
                    );



  if(syncArrayIndex<5) return TRUE;
  if (((mode==S1)||(mode==S2)||(mode==SDX))&&(syncArray[0].retrace==TRUE))
    {
      regression(a,b,2,syncArrayIndex);
    }
  else regression(a,b,1,syncArrayIndex);
  addToLog(QString("step:%1 a=%2 b=%3, modified rxclock=%4").arg(syncArrayIndex).arg(a).arg(b).arg(modifiedClock*b),DBSLANT);

//  if(syncArrayIndex>10)
//    {
//      regression(a1,b1,syncArrayIndex-7,syncArrayIndex);
//      addToLog(QString("step:%1 a1=%2 b1=%3, modified rxclock=%4").arg(syncArrayIndex).arg(a1).arg(b1).arg(modifiedClock*b1),DBSLANT);
//      if(fabs(modifiedClock*b-modifiedClock*b1)>50)
//      {
//        dumpSyncArray();
//        addToLog(QString("slantAdjust clockDiff=%1 too high").arg(modifiedClock*b-modifiedClock*b1),DBSLANT);
//        //return FALSE; // clock jump
//       return TRUE; // debug joma
//      }
//    }

	if((rxMW->autoSlantAdjust) && (syncArray[syncArrayIndex].lineNumber>(int)slantAdjustLine))
	{
		if(syncArray[syncArrayIndex].closestLine>=8)
			{
				return TRUE;
			}
//		dumpSyncArray();
		slantAdjustLine+=15;
		if(fabs(1.-b)>0.00001)
			{
				//dumpSyncArray();
				newClock=TRUE;
				modifiedClock=modifiedClock*b;
				samplesPerLine=lineLength(mode,modifiedClock); //recalculate the samples per line
				 addToLog("new clock accepted",DBSLANT);
				 syncPosition=syncArray[0].endPosition+(long)rint(a);
				 addToLog(QString("slantAdjust: modified  syncpos:=%1").arg(syncPosition),DBSYNC1);
			 }
		}
  return TRUE;

}

void syncProcessor::syncLost()
{
  if(!isInSync()) return;
  bool done;
  syncFound=FALSE;
  if(syncArrayIndex==0) return;
  rewindPosition=syncArray[syncArrayIndex-1].startPosition;
  syncLostEvent* ce = new syncLostEvent();
  ce->waitFor(&done);
  QApplication::postEvent(dispatchPtr, ce);
  addToLog("syncLost send",DBSYNC1 );
  while(!done) { usleep(10);}
  addToLog(QString("syncLost done: rewindPosition=%1, syncArrayIndex-1=%2").arg(rewindPosition).arg(syncArrayIndex-1),DBSYNC1 );
}



void syncProcessor::logStatus()
{
  if(syncFound)
  {
    addToLog("syncFound=TRUE",DBDISPAT);
   }
  else
  {
    addToLog("syncFound=FALSE",DBDISPAT);
  }
  addToLog(QString("syncArrayIndex=%1, signalQuality=%3 ")
               .arg(syncArrayIndex).arg(signalQuality),DBDISPAT);
}

void syncProcessor::copySyncArray()
{
  uint i;
  for(i=0;i<=syncArrayIndex;i++)
    {
      syncArrayCopy[i]=syncArray[i];
    }
  syncArrayCopyIndex=syncArrayIndex;
}

void syncProcessor::restoreSyncArray()
{
  uint i;
  for(i=0;i<=syncArrayCopyIndex;i++)
    {
      syncArray[i]=syncArrayCopy[i];
      syncArray[i].closestLine=0;
    }
  syncArrayIndex=syncArrayCopyIndex;
}
