% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rstanarm_tidiers.R
\name{rstanarm_tidiers}
\alias{rstanarm_tidiers}
\alias{tidy.stanreg}
\alias{glance.stanreg}
\title{Tidying methods for an rstanarm model}
\usage{
\method{tidy}{stanreg}(
  x,
  effects = c("fixed", "ran_pars"),
  conf.int = FALSE,
  conf.level = 0.9,
  conf.method = c("quantile", "HPDinterval"),
  exponentiate = FALSE,
  ...
)

\method{glance}{stanreg}(x, looic = FALSE, ...)
}
\arguments{
\item{x}{Fitted model object from the \pkg{rstanarm} package. See
\code{\link[rstanarm]{stanreg-objects}}.}

\item{effects}{A character vector including one or more of \code{"fixed"},
\code{"ran_vals"}, or \code{"ran_pars"}.
See the Value section for details.}

\item{conf.int}{If \code{TRUE} columns for the lower (\code{conf.low}) and upper (\code{conf.high}) bounds of the
\code{100*prob}\% posterior uncertainty intervals are included. See
\code{\link[rstantools]{posterior_interval}} for details.}

\item{conf.level}{See \code{\link[rstantools]{posterior_interval}}.}

\item{conf.method}{method for computing confidence intervals
("quantile" or "HPDinterval")}

\item{exponentiate}{whether to exponentiate the fixed-effect coefficient estimates and confidence intervals (common for logistic regression); if \code{TRUE}, also scales the standard errors by the exponentiated coefficient, transforming them to the new scale}

\item{...}{For \code{glance}, if \code{looic=TRUE}, optional arguments to
\code{\link[rstan]{loo.stanfit}}.}

\item{looic}{Should the LOO Information Criterion (and related info) be
included? See \code{\link[rstan]{loo.stanfit}} for details. (This
can be slow for models fit to large datasets.)}
}
\value{
All tidying methods return a \code{data.frame} without rownames.
The structure depends on the method chosen.

When \code{effects="fixed"} (the default), \code{tidy.stanreg} returns
one row for each coefficient, with three columns:
\item{term}{The name of the corresponding term in the model.}
\item{estimate}{A point estimate of the coefficient (posterior median).}
\item{std.error}{A standard error for the point estimate based on
\code{\link[stats]{mad}}. See the \emph{Uncertainty estimates} section in
\code{\link[rstanarm]{print.stanreg}} for more details.}

For models with group-specific parameters (e.g., models fit with
\code{\link[rstanarm]{stan_glmer}}), setting \code{effects="ran_vals"}
selects the group-level parameters instead of the non-varying regression
coefficients. Addtional columns are added indicating the \code{level} and
\code{group}. Specifying \code{effects="ran_pars"} selects the
standard deviations and (for certain models) correlations of the group-level
parameters.

Setting \code{effects="auxiliary"} will select parameters other than those
included by the other options. The particular parameters depend on which
\pkg{rstanarm} modeling function was used to fit the model. For example, for
models fit using \code{\link[rstanarm]{stan_glm}} the overdispersion
parameter is included if \code{effects="aux"}, for
\code{\link[rstanarm]{stan_lm}} the auxiliary parameters include the residual
SD, R^2, and log(fit_ratio), etc.

\code{glance} returns one row with the columns
  \item{algorithm}{The algorithm used to fit the model.}
  \item{pss}{The posterior sample size (except for models fit using
  optimization).}
  \item{nobs}{The number of observations used to fit the model.}
  \item{sigma}{The square root of the estimated residual variance, if
  applicable. If not applicable (e.g., for binomial GLMs), \code{sigma} will
  be given the value \code{1} in the returned object.}

  If \code{looic=TRUE}, then the following additional columns are also
  included:
  \item{looic}{The LOO Information Criterion.}
  \item{elpd_loo}{The expected log predictive density (\code{elpd_loo = -2 *
  looic}).}
  \item{p_loo}{The effective number of parameters.}
}
\description{
These methods tidy the estimates from \code{rstanarm} fits
(\code{stan_glm}, \code{stan_glmer}, etc.)
into a summary.
}
\examples{

if (require("rstanarm")) {
\dontrun{
#'     ## original models
  fit <- stan_glmer(mpg ~ wt + (1|cyl) + (1+wt|gear), data = mtcars,
                      iter = 500, chains = 2)
  fit2 <- stan_glmer((mpg>20) ~ wt + (1 | cyl) + (1 + wt | gear),
                    data = mtcars,
                    family = binomial,
                    iter = 500, chains = 2
  }
## load example data
  load(system.file("extdata", "rstanarm_example.rda", package="broom.mixed"))

  # non-varying ("population") parameters
  tidy(fit, conf.int = TRUE, conf.level = 0.5)
  tidy(fit, conf.int = TRUE, conf.method = "HPDinterval", conf.level = 0.5)

  #  exponentiating (in this case, from log-odds to odds ratios)
  (tidy(fit2, conf.int = TRUE, conf.level = 0.5)
          |> dplyr::filter(term != "(Intercept)")
  )
  (tidy(fit2, conf.int = TRUE, conf.level = 0.5, exponentiate = TRUE)
          |> dplyr::filter(term != "(Intercept)")
  )

  # hierarchical sd & correlation parameters
  tidy(fit, effects = "ran_pars")

  # group-specific deviations from "population" parameters
  tidy(fit, effects = "ran_vals")

  # glance method
   glance(fit)
  \dontrun{
     glance(fit, looic = TRUE, cores = 1)
  }
} ## if require("rstanarm")
}
\seealso{
\code{\link[rstan]{summary,stanfit-method}}
}
