\name{geom_raster}
\alias{geom_raster}
\title{High-performance rectangular tiling.}
\usage{
  geom_raster(mapping = NULL, data = NULL,
    stat = "identity", position = "identity", hjust = 0.5,
    vjust = 0.5, interpolate = FALSE, ...)
}
\arguments{
  \item{hjust,vjust}{horizontal and vertical justification
  of the grob.  Each justification value should be a number
  between 0 and 1.  Defaults to 0.5 for both, centering
  each pixel over its data location.}

  \item{interpolate}{If \code{TRUE} interpolate linearly,
  if \code{FALSE} (the default) don't interpolate.}

  \item{mapping}{The aesthetic mapping, usually constructed
  with \code{\link{aes}} or \code{\link{aes_string}}. Only
  needs to be set at the layer level if you are overriding
  the plot defaults.}

  \item{data}{A layer specific dataset - only needed if you
  want to override the plot defaults.}

  \item{stat}{The statistical transformation to use on the
  data for this layer.}

  \item{position}{The position adjustment to use for
  overlappling points on this layer}

  \item{...}{other arguments passed on to
  \code{\link{layer}}. This can include aesthetics whose
  values you want to set, not map. See \code{\link{layer}}
  for more details.}
}
\description{
  This is a special case of \code{\link{geom_tile}} where
  all tiles are the same size.  It is implemented highly
  efficiently using the internal \code{rasterGrob}
  function.
}
\details{
  By default, \code{geom_raster} add a vertical and
  horizontal padding. The size of padding depends on the
  resolution of data. If you want to manually set the
  padding (e.g. want zero-padding), you can change the
  behavior by setting \code{hpad} and \code{vpad}.
}
\section{Aesthetics}{
  \Sexpr[results=rd,stage=build]{ggplot2:::rd_aesthetics("geom",
  "raster")}
}
\examples{
\donttest{
# Generate data
pp <- function (n,r=4) {
 x <- seq(-r*pi, r*pi, len=n)
 df <- expand.grid(x=x, y=x)
 df$r <- sqrt(df$x^2 + df$y^2)
 df$z <- cos(df$r^2)*exp(-df$r/6)
 df
}
qplot(x, y, data = pp(20), fill = z, geom = "raster")
# Interpolation worsens the apperance of this plot, but can help when
# rendering images.
qplot(x, y, data = pp(20), fill = z, geom = "raster", interpolate = TRUE)

# For the special cases where it is applicable, geom_raster is much
# faster than geom_tile:
pp200 <- pp(200)
base <- ggplot(pp200, aes(x, y, fill = z))
benchplot(base + geom_raster())
benchplot(base + geom_tile())

# justification
df <- expand.grid(x = 0:5, y = 0:5)
df$z <- runif(nrow(df))
# default is compatible with geom_tile()
ggplot(df, aes(x, y, fill = z)) + geom_raster()
# zero padding
ggplot(df, aes(x, y, fill = z)) + geom_raster(hjust = 0, vjust = 0)
}
}

