/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2004, 2005  Clifford Wolf <clifford@clifford.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  mod_encode_xml.c: A simple XML encoding function
 */

/**
 * A simple XML encoding function
 */

#include <stdlib.h>
#include "spl.h"
#include "compat.h"

extern void SPL_ABI(spl_mod_encode_xml_init)(struct spl_vm *vm, struct spl_module *mod, int restore);
extern void SPL_ABI(spl_mod_encode_xml_done)(struct spl_vm *vm, struct spl_module *mod);

static char *xml_encode(const char *source)
{
	int source_i, target_i;

	for (source_i = target_i = 0; source[source_i]; source_i++)
		switch (source[source_i]) {
			case '&':
				/* &amp; */
				target_i += 5;
				break;
			case '<':
				/* &lt; */
				target_i += 4;
				break;
			case '>':
				/* &gt; */
				target_i += 4;
				break;
			case '"':
				/* &quot; */
				target_i += 6;
				break;
			case '\'':
				/* &apos; */
				target_i += 6;
				break;
			default:
				target_i++;
		}

	char *target = malloc(target_i+1);

	for (source_i = target_i = 0; source[source_i]; source_i++)
		switch (source[source_i]) {
			case '&':
				/* &amp; */
				target[target_i++] = '&';
				target[target_i++] = 'a';
				target[target_i++] = 'm';
				target[target_i++] = 'p';
				target[target_i++] = ';';
				break;
			case '<':
				/* &lt; */
				target[target_i++] = '&';
				target[target_i++] = 'l';
				target[target_i++] = 't';
				target[target_i++] = ';';
				break;
			case '>':
				/* &gt; */
				target[target_i++] = '&';
				target[target_i++] = 'g';
				target[target_i++] = 't';
				target[target_i++] = ';';
				break;
			case '"':
				/* &quot; */
				target[target_i++] = '&';
				target[target_i++] = 'q';
				target[target_i++] = 'u';
				target[target_i++] = 'o';
				target[target_i++] = 't';
				target[target_i++] = ';';
				break;
			case '\'':
				/* &apos; */
				target[target_i++] = '&';
				target[target_i++] = 'a';
				target[target_i++] = 'p';
				target[target_i++] = 'o';
				target[target_i++] = 's';
				target[target_i++] = ';';
				break;
			default:
				target[target_i++] = source[source_i];
		}

	target[target_i] = 0;
	return target;
}

/**
 * This function ecodes a text in XML. I.e. the characters &, <, >, " and '
 * are translated to &amp;, &lt;, &gt;, &quot;, and &apos;.
 *
 * This function is designed to be used with the encoding/quoting operator (::).
 */
// builtin encode_xml(text)
static struct spl_node *handler_encode_xml(struct spl_task *task, void *data UNUSED)
{
	char *source = spl_clib_get_string(task);
	return SPL_NEW_STRING(xml_encode(source));
}

void SPL_ABI(spl_mod_encode_xml_init)(struct spl_vm *vm, struct spl_module *mod UNUSED, int restore UNUSED)
{
	spl_clib_reg(vm, "encode_xml", handler_encode_xml, 0);
}

void SPL_ABI(spl_mod_encode_xml_done)(struct spl_vm *vm UNUSED, struct spl_module *mod UNUSED)
{
	return;
}

