/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2004, 2005  Clifford Wolf <clifford@clifford.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  mod_prime.c: A simple example for CLIB handled namespaces
 */

/**
 * Example Module for hosted namespaces
 */

#include <stdlib.h>
#include <string.h>

#include "spl.h"
#include "compat.h"

/**
 * This is an example hosted namespace. Numeric keys which are prime do point
 * to true values, other numerical keys point to false values and non-numerical
 * keys are undeclared. Example:
 *
 *	debug "declared prime['foo'] = " ~ declared prime['foo'];
 *	debug "prime[5] = " ~ prime[5];
 *	debug "prime[6] = " ~ prime[6];
 *
 * This is just an example for writing modules which provide hosted namespaces.
 * It is not really meant to be used in any applications.
 */
// namespace prime;

extern void SPL_ABI(spl_mod_prime_init)(struct spl_vm *vm, struct spl_module *mod, int restore);
extern void SPL_ABI(spl_mod_prime_done)(struct spl_vm *vm, struct spl_module *mod);

static void handler_primenode(struct spl_task *task UNUSED, struct spl_vm *vm UNUSED,
		struct spl_node *node UNUSED, struct spl_hnode_args *args, void *data UNUSED)
{
	if (args->action == SPL_HNODE_ACTION_LOOKUP)
	{
		const char *key = args->key;
		while ( *key == '?' ) key++;
		if (*key < '0' || *key > '9') return;

		int keyval = atoi(key);

		if (keyval < 2) {
			args->value = SPL_NEW_INT(0);
			return;
		}

		for (int i=2; i<keyval; i++)
			if ( keyval%i == 0 ) {
				args->value = SPL_NEW_INT(0);
				return;
			}

		args->value = SPL_NEW_INT(1);
		return;
	}

	if (args->action == SPL_HNODE_ACTION_CREATE)
	{
		const char *key = args->key;
		while ( *key == '?' ) key++;

		printf("You tried 'prime.[%s] = %d'. Please do not define your own prime numbers!\n", key, spl_get_int(args->value));
		fflush(stdout);
		return;
	}

	if (args->action == SPL_HNODE_ACTION_DELETE)
	{
		printf("You called method 'delete' in prime namespace. Primes are read-only!\n");
		fflush(stdout);
	}

	// printf("You called unhandled action '%d' in prime namespace.\n", args->action);
	// fflush(stdout);
}

void SPL_ABI(spl_mod_prime_init)(struct spl_vm *vm, struct spl_module *mod, int restore)
{
	spl_hnode_reg(vm, "prime", handler_primenode, 0);

	if ( !restore )
		spl_hnode(vm, vm->root, "prime", "prime", mod);
}

void SPL_ABI(spl_mod_prime_done)(struct spl_vm *vm UNUSED, struct spl_module *mod UNUSED)
{
	return;
}

