/*
	Copyright (C) 2011 - 2022
	by Sergey Popov <loonycyborg@gmail.com>
	Part of the Battle for Wesnoth Project https://www.wesnoth.org/

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY.

	See the COPYING file for more details.
*/

#pragma once

#include "exceptions.hpp"
#include "lua_jailbreak_exception.hpp"

#include <boost/system/error_code.hpp>

/** An error occurred during when trying to communicate with the wesnothd server. */
struct wesnothd_error : public game::error
{
	wesnothd_error(const std::string& error)
		: game::error(error)
	{
	}
};

/**
 * Error used when the client is rejected by the MP server.
 * Technically, this is not an error but the only way to handle the condition is as if it were an error.
 */
struct wesnothd_rejected_client_error : public game::error
{
	wesnothd_rejected_client_error(const std::string& msg)
		: game::error(msg)
	{
	}
};

/**
 * We received invalid data from wesnothd during a game
 * This means we cannot continue with the game but we can stay connected to wesnothd and start a new game.
 * TODO: find a short name
 */
struct ingame_wesnothd_error : public wesnothd_error, public lua_jailbreak_exception
{
	ingame_wesnothd_error(const std::string& error)
		: wesnothd_error(error)
	{
	}

	IMPLEMENT_LUA_JAILBREAK_EXCEPTION(ingame_wesnothd_error)
};

struct leavegame_wesnothd_error : ingame_wesnothd_error
{
	leavegame_wesnothd_error(const std::string& error)
		: ingame_wesnothd_error(error)
	{
	}
};

/**
 * An error occurred inside the underlying network communication code (boost asio)
 * TODO: find a short name
 */
struct wesnothd_connection_error : public wesnothd_error, public lua_jailbreak_exception
{
	wesnothd_connection_error(const boost::system::error_code& error, const std::string& msg = "")
		: wesnothd_error(error.message())
		, user_message(msg)
	{
	}

	/** User-friendly and potentially translated message for use in the UI. */
	std::string user_message;

	IMPLEMENT_LUA_JAILBREAK_EXCEPTION(wesnothd_connection_error)
};
